unit Checkeditgroup;

{$I Misc.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla 
Public License Version 1.1 (the "License"); you may not use this file except 
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for 
the specific language governing rights and limitations under the License.

The Original Code is: Checkeditgroup.pas, released 12 September 2002.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 email: mat.ballard@chemware.hypermart.net.
                Robert Ross                 email: rross@sigmaofficesolutions.com

Last Modified: 05/25/2002
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.


Purpose:
Multi-select radio group.


Known Issues:
-----------------------------------------------------------------------------}

{$I Misc.inc}

interface

uses
  Classes, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Graphics, Messages, Stdctrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Graphics, Messages, Stdctrls,
{$ENDIF}
{$IFDEF LINUX}
  QT,
  QGraphics, QStdctrls,
{$ENDIF}
  Misc, Checkeditframe;

const
  TCHECKEDITGROUP_VERSION = 300;

type
  TOnCheckBoxClick = procedure(Sender: TObject; Box: TCheckBox) of object;
  TOnEditClick = procedure(Sender: TObject; Edit: TEdit) of object;
  TOnEditChange = TOnEditClick;
  TOnEditEnter = TOnEditClick;
  TOnEditExit = TOnEditClick;
  TOnEditKeyDown = procedure(Sender: TObject; Edit: TEdit; var Key: Word; Shift: TShiftState) of object;
  TOnEditKeyPress = procedure(Sender: TObject; Edit: TEdit; var Key: Char) of object;
  TOnEditKeyUp = procedure(Sender: TObject; Edit: TEdit; var Key: Word; Shift: TShiftState) of object;

  TCheckEditGroup = class(TGroupBox)
  private
 { Private fields of TCheckEditGroup }
 { Storage for property Items }
    FCaptions : TStrings;
    FColumns: Byte;
    FItems: TStrings;
    FGap: Byte;
    //FNo_EditBoxes: Integer;

{events:}
    FOnCheckBoxClick: TOnCheckBoxClick;
    FOnEditClick: TOnEditClick;
    FOnEditChange: TOnEditChange;
    FOnEditEnter: TOnEditEnter;
    FOnEditExit: TOnEditExit;
    FOnEditKeyDown: TOnEditKeyDown;
    FOnEditKeyPress: TOnEditKeyPress;
    FOnEditKeyUp: TOnEditKeyUp;

    {procedure CMFontChanged(var Message: TMessage);}
    {procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;}
{$IFDEF MSWINDOWS}
    procedure WMSize(var Message: TMessage); message WM_SIZE;
{$ENDIF}
{$IFDEF LINUX}
{$ENDIF}
  protected
    function GetCheckEdit(Index: Integer): TCheckEdit;
    function GetChecked(Index: Integer): Boolean;
    function GetCount: Word;
    function GetEditLeft: Integer;
    function GetRatio: TPercent;
    {procedure WMPaint(var Message: TWMPaint); message WM_PAINT;)}

    procedure DoCheckBoxClick(Sender: TObject); virtual;
    procedure DoEditClick(Sender: TObject); virtual;
    procedure DoEditChange(Sender: TObject); virtual;
    procedure DoEditEnter(Sender: TObject); virtual;
    procedure DoEditExit(Sender: TObject); virtual;
    procedure DoEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;
    procedure DoEditKeyPress(Sender: TObject; var Key: Char); virtual;
    procedure DoEditKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;

    procedure Loaded; override;
    procedure SetCaptions(Value : TStrings);
    procedure SetChecked(Index: Integer; Value: Boolean);
    procedure SetColumns(Value : Byte);
    procedure SetCount(Value : Word);
    procedure SetEditLeft(Value: Integer);
    procedure SetRatio(Value: TPercent);
    procedure SetGap(Value : Byte);
    //procedure SetFontCaption(Value : TFont);
    procedure SetItems(Value : TStrings);
    procedure CreateAllCheckEdits;
  public
 { The CheckEdits }
    property CheckEdits[Index: Integer]: TCheckEdit read GetCheckEdit;
    property Checked[Index: Integer]: Boolean read GetChecked write SetChecked;

 { Public methods of TCheckEditGroup }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ArrangeCheckEdits;
    {procedure SetText(Text: String; Index: Integer);
    function GetText(Index: Integer);}

  published
    property Captions : TStrings read FCaptions write SetCaptions;
    property Columns: Byte read FColumns write SetColumns default 1;
    property Count: Word read GetCount write SetCount stored FALSE;
    property EditLeft: Integer read GetEditLeft write SetEditLeft;
    property Ratio: TPercent read GetRatio write SetRatio;
    property Gap: Byte read FGap write SetGap default 4;
    //property FontCaption: TFont read FFontCaption write SetFontCaption;
    property Items : TStrings read FItems write SetItems;
    property OnCheckBoxClick: TOnCheckBoxClick read FOnCheckBoxClick write FOnCheckBoxClick;
    property OnEditClick: TOnEditClick read FOnEditClick write FOnEditClick;
    property OnEditChange: TOnEditChange read FOnEditChange  write FOnEditChange;
    property OnEditEnter: TOnEditEnter read FOnEditEnter write FOnEditEnter;
    property OnEditExit: TOnEditExit read FOnEditExit write FOnEditExit;
    property OnEditKeyDown: TOnEditKeyDown read FOnEditKeyDown write FOnEditKeyDown;
    property OnEditKeyPress: TOnEditKeyPress read FOnEditKeyPress write FOnEditKeyPress;
    property OnEditKeyUp: TOnEditKeyUp read FOnEditKeyUp write FOnEditKeyUp;
  end;

implementation

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.Create
  Description: standard constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: initializes Properties
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TCheckEditGroup.Create(AOwner: TComponent);
begin
{ Call the Create method of the container's parent class     }
   inherited Create(AOwner);

   Width := 200;
   Height := 200;
   //Font.Style := [fsBold];

   FColumns := 1;
   FItems := TStringList.Create;
   FCaptions := TStringList.Create;
   FGap := 4;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TCheckEditGroup.Destroy;
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    TCheckEdit(FItems.Objects[i]).Free;
  FItems.Free;
  FCaptions.Free;
  //FFontCaption.Free;

{ Last, free the component by calling the Destroy method of the  }
{ parent class.                          }
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.DoCheckBoxClick et al
  Description: responds to a click event of a single CheckEdit
       Author: Mat Ballard
 Suggested by: Robert Ross
 Date created: 11/27/2000
Date modified: 11/27/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.DoCheckBoxClick(Sender: TObject);
begin
  if Assigned(FOnCheckBoxClick) then
    OnCheckBoxClick(Self, TCheckBox(Sender));
end;
procedure TCheckEditGroup.DoEditClick(Sender: TObject);
begin
  if Assigned(FOnEditClick) then
    OnEditClick(Self, TEdit(Sender));
end;
procedure TCheckEditGroup.DoEditChange(Sender: TObject);
begin
  if Assigned(FOnEditChange) then
    OnEditChange(Self, TEdit(Sender));
end;
procedure TCheckEditGroup.DoEditEnter(Sender: TObject);
begin
  if Assigned(FOnEditEnter) then
    OnEditEnter(Self, TEdit(Sender));
end;
procedure TCheckEditGroup.DoEditExit(Sender: TObject);
begin
  if Assigned(FOnEditExit) then
    OnEditExit(Self, TEdit(Sender));
end;
procedure TCheckEditGroup.DoEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOnEditKeyDown) then
    OnEditKeyDown(Self, TEdit(Sender), Key, Shift);
end;
procedure TCheckEditGroup.DoEditKeyPress(Sender: TObject; var Key: Char);
begin
  if Assigned(FOnEditKeyPress) then
    OnEditKeyPress(Self, TEdit(Sender), Key);
end;
procedure TCheckEditGroup.DoEditKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOnEditKeyUp) then
    OnEditKeyUp(Self, TEdit(Sender), Key, Shift);
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.Loaded
  Description: responds to a load event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.Loaded;
begin
  inherited Loaded;
  //SetNoEditBoxes(FItems.Count);
  CreateAllCheckEdits;
  ArrangeCheckEdits;
end;

{------------------------------------------------------------------------------
     Function: TCheckEditGroup.GetCheckEdit
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual CheckEdits
 Known Issues:
 Return Value: TCheckEditGroup
 ------------------------------------------------------------------------------}
function TCheckEditGroup.GetCheckEdit(Index: Integer): TCheckEdit;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TCheckEdit(FItems.Objects[Index])
   else
    Result := nil;
end;

function TCheckEditGroup.GetChecked(Index: Integer): Boolean;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TCheckEdit(FItems.Objects[Index]).Checkbox.Checked
   else
    Result := FALSE;
end;

procedure TCheckEditGroup.SetChecked(Index: Integer; Value: Boolean);
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
  begin
    TCheckEdit(FItems.Objects[Index]).Checkbox.Checked := Value;
  end;
end;

{------------------------------------------------------------------------------
     Function: TCheckEditGroup.GetCount
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual CheckEdits
 Known Issues:
 Return Value: TCheckEditGroup
 ------------------------------------------------------------------------------}
function TCheckEditGroup.GetCount: Word;
begin
  Result := FItems.Count;
end;

function TCheckEditGroup.GetEditLeft: Integer;
begin
  if (FItems.Count > 0) then
    Result := TCheckEdit(FItems.Objects[0]).Edit.Left
   else
    Result := 100;
end;

procedure TCheckEditGroup.SetEditLeft(Value: Integer);
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
  begin
    TCheckEdit(FItems.Objects[i]).Edit.Left := Value;
    TCheckEdit(FItems.Objects[i]).Edit.Width :=
      TCheckEdit(FItems.Objects[i]).Width -
        TCheckEdit(FItems.Objects[i]).Edit.Left - 4;
  end;
end;

function TCheckEditGroup.GetRatio: TPercent;
begin
  if (FItems.Count > 0) then
    Result := TCheckEdit(FItems.Objects[0]).Ratio
   else
    Result := 50;
end;

procedure TCheckEditGroup.SetRatio(Value: TPercent);
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    TCheckEdit(FItems.Objects[i]).Ratio := Value;
end;

procedure TCheckEditGroup.SetGap(Value : Byte);
begin
  FGap := Value;
  ArrangeCheckEdits;
end;
{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.SetCaptions
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Captions Property
 Known Issues: The stream system does not call this procedure !
               Instead, streamed values are loaded directly into FCaptions
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.SetCaptions(Value : TStrings);
var
  i: Integer;
begin
  if (csLoading in ComponentState) then
  begin
    FCaptions.Assign(Value);
  end
  else
  begin
    SetCount(Value.Count);
    for i := 0 to FCaptions.Count-1 do
    begin
      FCaptions.Strings[i] := Value.Strings[i];
      TCheckEdit(FItems.Objects[i]).Checkbox.Caption := Value.Strings[i];
    end;
    Invalidate;
    ArrangeCheckEdits;
  end;
end;

{procedure TCheckEditGroup.SetFontCaption(Value : TFont);
var
  i: Integer;
begin
  FFontCaption.Assign(Value);
  for i := 0 to FItems.Count-1 do
  begin
    TCheckEdit(FItems.Objects[i]).CheckFont.Assign(FFontCaption);
  end;
end;}

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.SetItems
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: The stream system does not call this procedure !
               Instead, streamed values are loaded directly into FItems
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.SetItems(Value : TStrings);
var
  i: Integer;
begin
  if (csLoading in ComponentState) then
  begin
    FItems.Assign(Value);
  end
  else
  begin
    SetCount(Value.Count);
    for i := 0 to FItems.Count-1 do
    begin
      FItems.Strings[i] := Value.Strings[i];
      TCheckEdit(FItems.Objects[i]).Edit.Text := Value.Strings[i];
    end;
    Invalidate;
    ArrangeCheckEdits;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.CreateAllCheckEdits
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: we only create CheckEdits once loading is complete
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.CreateAllCheckEdits;
var
  i: Integer;
  ACheckEdit: TCheckEdit;
begin
  for i := 0 to FItems.Count-1 do
  begin
    ACheckEdit := TCheckEdit.Create(Self);
    ACheckEdit.Name := Format('a%d', [i]);
    ACheckEdit.Parent := Self;
    ACheckEdit.Edit.Text := FItems.Strings[i];
    //ACheckEdit.Caption := FCaptions.Strings[i]; 
    ACheckEdit.Tag := i;
    ACheckEdit.Font.Assign(Font);
    //ACheckEdit.CheckFont.Assign(FFontCaption);
    ACheckEdit.CheckBox .OnClick := DoCheckBoxClick;
    ACheckEdit.Edit.OnClick := DoEditClick;
    ACheckEdit.Edit.OnChange := DoEditChange;
    ACheckEdit.Edit.OnEnter := DoEditEnter;
    ACheckEdit.Edit.OnExit := DoEditExit;
    ACheckEdit.Edit.OnKeyDown := DoEditKeyDown;
    ACheckEdit.Edit.OnKeyPress := DoEditKeyPress;
    ACheckEdit.Edit.OnKeyUp := DoEditKeyUp;
    FItems.Objects[i] := ACheckEdit;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.SetCount
  Description: semi-standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the number of elements to the FItems StringList
 Known Issues: Count is not streamable
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.SetCount(Value : Word);
var
  ACheckEdit: TCheckEdit;
begin
  if (csLoading in ComponentState) then
    EComponentError.Create('SetCount called while loading !');

  if (Value > FItems.Count) then
  begin
    while (FItems.Count < Value) do
    begin
      ACheckEdit := TCheckEdit.Create(Self);
      ACheckEdit.Name := Format('CheckEdit%d', [FItems.Count+1]);
      ACheckEdit.Parent := Self;
      ACheckEdit.Tag := FItems.Count;
      ACheckEdit.CheckBox.OnClick := DoCheckBoxClick;
      ACheckEdit.Edit.OnClick := DoEditClick;
      ACheckEdit.Edit.OnChange := DoEditChange;
      ACheckEdit.Edit.OnEnter := DoEditEnter;
      ACheckEdit.Edit.OnExit := DoEditExit;
      ACheckEdit.Edit.OnKeyDown := DoEditKeyDown;
      ACheckEdit.Edit.OnKeyPress := DoEditKeyPress;
      ACheckEdit.Edit.OnKeyUp := DoEditKeyUp;
      ACheckEdit.Font.Assign(Font);
      //ACheckEdit.CheckFont.Assign(FFontCaption);
      FItems.AddObject('', ACheckEdit);
    end;
  end
  else if (Value < FItems.Count) then
  begin
    while (FItems.Count > Value) do
    begin
      ACheckEdit := TCheckEdit(FItems.Objects[FItems.Count-1]);
      ACheckEdit.Free;
      FItems.Delete(FItems.Count-1);
    end;
  end;
  while (FCaptions.Count < Value) do
    FCaptions.Add('');
  while (FCaptions.Count > Value) do
    FCaptions.Delete(FCaptions.Count-1);
  ArrangeCheckEdits;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.SetColumns
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Columns Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.SetColumns(Value : Byte);
begin
  if (Value = 0) then Value := 1;
  FColumns := Value;
  ArrangeCheckEdits;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.ArrangeCheckEdits
  Description: This arranges the EditBoxes on the panel
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Display management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckEditGroup.ArrangeCheckEdits;
var
  i,
  NoRows,
  TheTop,
  TheWidth,
  DeltaHeight: Integer;
  ACheckEdit: TCheckEdit;
begin
  if (FItems.Count > 0) then
  begin
{position the EditBoxes:}
    TheTop := Abs(Font.Height) + FGap;
    TheWidth := (Width - (FColumns+1)*FGap - 2) div FColumns;
    NoRows := FItems.Count div FColumns;
    if ((NoRows * FColumns) < FItems.Count) then
      Inc(NoRows);
    DeltaHeight := (Height - TheTop - FGap) div NoRows;

    for i := 0 to FItems.Count-1 do
    begin
      ACheckEdit := TCheckEdit(FItems.Objects[i]);
      if (ACheckEdit <> nil) then
      begin
        ACheckEdit.Top := TheTop + (i div FColumns)*DeltaHeight;
        ACheckEdit.Left := FGap + (TheWidth + FGap) * (i Mod FColumns);
        ACheckEdit.Width := TheWidth;
        ACheckEdit.Visible := TRUE;
        ACheckEdit.Invalidate;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckEditGroup.WMSize
  Description: standard ReSize message handler
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Re-Arranges the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF MSWINDOWS}
procedure TCheckEditGroup.WMSize(var Message: TMessage);
begin
  inherited;
  ArrangeCheckEdits;
end;
{$ENDIF}

{procedure TCheckEditGroup.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ArrangeCheckEdits;
end;}

{procedure TCheckEditGroup.WMPaint(var Message: TWMPaint);
var
  i: Integer;
begin
  inherited;
  Broadcast(Message);
end;}

end.

