unit Checkgrp;

{$I Misc.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: CheckGroup.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 email: mat.ballard@chemware.hypermart.net.
                Robert Ross                 email: rross@sigmaofficesolutions.com

Last Modified: 05/25/2000
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 


Purpose:
Multi-select radio group


Known Issues:
-----------------------------------------------------------------------------}

{$I Misc.inc}

interface

uses
  Classes, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Graphics, Messages, Stdctrls
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Graphics, Messages, Stdctrls
{$ENDIF}
{$IFDEF LINUX}
  QT,
  QGraphics, QStdctrls
{$ENDIF}
  ;

type
  TOnBoxClick = procedure(Sender: TObject; Box: TObject) of object;

  TCheckGroup = class(TGroupBox)
  private
 { Private fields of TCheckGroup }
 { Storage for property Items }
    FItems : TStrings;
    {FNo_EditBoxes: Integer;}
    FColumns: Byte;
    FOnBoxClick: TOnBoxClick;

    {procedure CMFontChanged(var Message: TMessage);}
    {procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;}
{$IFDEF MSWINDOWS}
    procedure WMSize(var Message: TMessage); message WM_SIZE;
{$ENDIF}
{$IFDEF LINUX}
{$ENDIF}
  protected
    function GetCheckBox(Index: Integer): TCheckBox;
    function GetCount: Word;
    function GetItemIndex: Integer;
    {procedure WMPaint(var Message: TWMPaint); message WM_PAINT;)}
    procedure DoBoxClick(Sender: TObject); virtual;
    procedure Loaded; override;
    procedure SetColumns(Value : Byte);
    procedure SetCount(Value : Word);
    procedure SetItems(Value : TStrings);
    procedure SetItemIndex(Value: Integer);
    procedure CreateAllCheckBoxes;
  public
 { The CheckBoxes }
    property CheckBoxes[Index: Integer]: TCheckBox read GetCheckBox;

 { Public methods of TCheckGroup }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ArrangeCheckBoxes;

  published
    property Columns: Byte read FColumns write SetColumns;
    property Count: Word read GetCount write SetCount stored FALSE;
    property Items : TStrings read FItems write SetItems;
    property ItemIndex : Integer read GetItemIndex write SetItemIndex;
    property OnBoxClick: TOnBoxClick read FOnBoxClick write FOnBoxClick;
  end;

implementation

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.Create
  Description: standard constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: initializes Properties
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TCheckGroup.Create(AOwner: TComponent);
begin
{ Call the Create method of the container's parent class     }
   inherited Create(AOwner);

   Width := 81;
   Height := 217;
   Font.Style := [fsBold];

   FColumns := 1;
   FItems := TStringList.Create;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TCheckGroup.Destroy;
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    TCheckBox(FItems.Objects[i]).Free;
  FItems.Free;

{ Last, free the component by calling the Destroy method of the  }
{ parent class.                          }
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.DoBoxClick
  Description: responds to a click event of a single checkbox
       Author: Mat Ballard
 Suggested by: Robert Ross
 Date created: 11/27/2000
Date modified: 11/27/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckGroup.DoBoxClick(Sender: TObject);
begin
  if Assigned(FOnBoxClick) then
    OnBoxClick(Self, Sender);
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.Loaded
  Description: responds to a load event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckGroup.Loaded;
begin
  inherited Loaded;
  {SetNoEditBoxes(FItems.Count);}
  CreateAllCheckBoxes;
  ArrangeCheckBoxes;
end;

{------------------------------------------------------------------------------
     Function: TCheckGroup.GetCheckBox
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual CheckBoxes
 Known Issues:
 Return Value: TCheckGroup
 ------------------------------------------------------------------------------}
function TCheckGroup.GetCheckBox(Index: Integer): TCheckBox;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TCheckBox(FItems.Objects[Index])
   else
    Result := nil;
end;

{------------------------------------------------------------------------------
     Function: TCheckGroup.GetCount
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual CheckBoxes
 Known Issues:
 Return Value: TCheckGroup
 ------------------------------------------------------------------------------}
function TCheckGroup.GetCount: Word;
begin
  Result := FItems.Count;
end;

function TCheckGroup.GetItemIndex: Integer;
var
  i: Integer;
begin
  Result := -1;
  for i := 0 to FItems.Count-1 do
  begin
    if (TCheckBox(FItems.Objects[i]).Checked) then
    begin
      Result := i;
      break;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.SetItems
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: The stream system does not call this procedure !
               Instead, streamed values are loaded directly into FItems
 ------------------------------------------------------------------------------}
procedure TCheckGroup.SetItems(Value : TStrings);
var
  i: Integer;
begin
  if (csLoading in ComponentState) then
  begin
    FItems.Assign(Value);
  end
  else
  begin
    SetCount(Value.Count);
    for i := 0 to FItems.Count-1 do
    begin
      FItems.Strings[i] := Value.Strings[i];
      TCheckBox(FItems.Objects[i]).Caption := Value.Strings[i];
    end;
    Invalidate;
    ArrangeCheckBoxes;
  end;
end;

procedure TCheckGroup.SetItemIndex(Value: Integer);
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    if Assigned(FItems.Objects[i]) then
      TCheckBox(FItems.Objects[i]).Checked := (Value = i);
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.CreateAllCheckBoxes
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: we only create checkboxes once loading is complete
 ------------------------------------------------------------------------------}
procedure TCheckGroup.CreateAllCheckBoxes;
var
  i: Integer;
  ACheckBox: TCheckBox;
begin
  for i := 0 to FItems.Count-1 do
  begin
    ACheckBox := TCheckBox.Create(Self);
    ACheckBox.Parent := Self;
    ACheckBox.Caption := FItems.Strings[i];
    ACheckBox.Tag := i;
    ACheckBox.OnClick := DoBoxClick;
    FItems.Objects[i] := ACheckBox;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.SetCount
  Description: semi-standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the number of elements to the FItems StringList
 Known Issues: Count is not streamable
 ------------------------------------------------------------------------------}
procedure TCheckGroup.SetCount(Value : Word);
var
  ACheckBox: TCheckBox;
begin
  if (csLoading in ComponentState) then
    EComponentError.Create('SetCount called while loading !');

  if (Value > FItems.Count) then
  begin
    while (FItems.Count < Value) do
    begin
      ACheckBox := TCheckBox.Create(Self);
      ACheckBox.Parent := Self;
      ACheckBox.Tag := FItems.Count;
      ACheckBox.OnClick := DoBoxClick;
      FItems.AddObject('', ACheckBox);
    end;
  end
  else if (Value < FItems.Count) then
  begin
    while (FItems.Count > Value) do
    begin
      ACheckBox := TCheckBox(FItems.Objects[FItems.Count-1]);
      ACheckBox.Free;
      FItems.Delete(FItems.Count-1);
    end;
  end;
  ArrangeCheckBoxes;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.SetColumns
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Columns Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckGroup.SetColumns(Value : Byte);
begin
  if (Value = 0) then Value := 1;
  FColumns := Value;
  ArrangeCheckBoxes;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.ArrangeCheckBoxes
  Description: This arranges the EditBoxes on the panel
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Display management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCheckGroup.ArrangeCheckBoxes;
var
  i,
  NoRows,
  TheTop,
  TheLeft,
  TheWidth,
  TheGap,
  DeltaHeight: Integer;
  ACheckBox: TCheckBox;
begin
  if (FItems.Count > 0) then
  begin
{position the EditBoxes:}
    TheTop := 24;
    TheLeft := 8;
    TheGap := 12;
    TheWidth := Width - (FColumns+1)*TheLeft - 2;
    TheWidth := TheWidth div FColumns;
    NoRows := FItems.Count div FColumns;
    if ((NoRows * FColumns) < FItems.Count) then
      Inc(NoRows);
    DeltaHeight := (Height - TheTop) div NoRows;

    for i := 0 to FItems.Count-1 do
    begin
      ACheckBox := TCheckBox(FItems.Objects[i]);
      if (ACheckBox <> nil) then
      begin
        ACheckBox.Top := TheTop + (i div FColumns)*DeltaHeight;
        ACheckBox.Left := TheLeft + (TheWidth + TheGap) * (i Mod FColumns);
        ACheckBox.Width := TheWidth;
        ACheckBox.Visible := TRUE;
        ACheckBox.Invalidate;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCheckGroup.WMSize
  Description: standard ReSize message handler
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Re-Arranges the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF MSWINDOWS}
procedure TCheckGroup.WMSize(var Message: TMessage);
begin
  inherited;
  ArrangeCheckBoxes;
end;
{$ENDIF}

{procedure TCheckGroup.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ArrangeCheckBoxes;
end;}

{procedure TCheckGroup.WMPaint(var Message: TWMPaint);
var
  i: Integer;
begin
  inherited;
  Broadcast(Message);
end;}

end.

