unit DblCheckEditgroup;

{$I Misc.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla 
Public License Version 1.1 (the "License"); you may not use this file except 
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for 
the specific language governing rights and limitations under the License.

The Original Code is: DblCheckEditgroup.pas, released 12 September 2002.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 email: mat.ballard@chemware.hypermart.net.
                Robert Ross                 email: rross@sigmaofficesolutions.com

Last Modified: 05/25/2002
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.


Purpose:
Multi-select radio group.


Known Issues:
-----------------------------------------------------------------------------}

{$I Misc.inc}

interface

uses
  Classes, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Graphics, Messages, Stdctrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Graphics, Messages, Stdctrls,
{$ENDIF}
{$IFDEF LINUX}
  QT,
  QGraphics, QStdctrls,
{$ENDIF}
  Misc, DblCheckEditframe;

const
  TDblCheckEditGROUP_VERSION = 300;

type
  TOnCheckBoxClick = procedure(Sender: TObject; Box: TCheckBox) of object;
  TOnEditClick = procedure(Sender: TObject; Edit: TEdit) of object;
  TOnEditChange = TOnEditClick;
  TOnEditEnter = TOnEditClick;
  TOnEditExit = TOnEditClick;
  TOnEditKeyDown = procedure(Sender: TObject; Edit: TEdit; var Key: Word; Shift: TShiftState) of object;
  TOnEditKeyPress = procedure(Sender: TObject; Edit: TEdit; var Key: Char) of object;
  TOnEditKeyUp = procedure(Sender: TObject; Edit: TEdit; var Key: Word; Shift: TShiftState) of object;

  TDblCheckEditGroup = class(TGroupBox)
  private
 { Private fields of TDblCheckEditGroup }
 { Storage for property Items }
    FCaptions : TStrings;
    FColumns: Byte;
    FItems: TStrings;
    FGap: Byte;
    //FNo_EditBoxes: Integer;

{events:}
    FOnCheckBoxClick: TOnCheckBoxClick;
    FOnEditClick: TOnEditClick;
    FOnEditChange: TOnEditChange;
    FOnEditEnter: TOnEditEnter;
    FOnEditExit: TOnEditExit;
    FOnEditKeyDown: TOnEditKeyDown;
    FOnEditKeyPress: TOnEditKeyPress;
    FOnEditKeyUp: TOnEditKeyUp;

    {procedure CMFontChanged(var Message: TMessage);}
    {procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;}
{$IFDEF MSWINDOWS}
    procedure WMSize(var Message: TMessage); message WM_SIZE;
{$ENDIF}
{$IFDEF LINUX}
{$ENDIF}
  protected
    function GetDblCheckEdit(Index: Integer): TDblCheckEdit;
    function GetChecked0(Index: Integer): Boolean;
    function GetChecked1(Index: Integer): Boolean;
    function GetCount: Word;
    function GetEditLeft: Integer;
    function GetRatio: TPercent;
    {procedure WMPaint(var Message: TWMPaint); message WM_PAINT;)}

    procedure DoCheckBoxClick(Sender: TObject); virtual;
    procedure DoEditClick(Sender: TObject); virtual;
    procedure DoEditChange(Sender: TObject); virtual;
    procedure DoEditEnter(Sender: TObject); virtual;
    procedure DoEditExit(Sender: TObject); virtual;
    procedure DoEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;
    procedure DoEditKeyPress(Sender: TObject; var Key: Char); virtual;
    procedure DoEditKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;

    procedure Loaded; override;
    procedure SetCaptions(Value : TStrings);
    procedure SetChecked0(Index: Integer; Value: Boolean);
    procedure SetChecked1(Index: Integer; Value: Boolean);
    procedure SetColumns(Value : Byte);
    procedure SetCount(Value : Word);
    procedure SetEditLeft(Value: Integer);
    procedure SetRatio(Value: TPercent);
    procedure SetGap(Value : Byte);
    //procedure SetFontCaption(Value : TFont);
    procedure SetItems(Value : TStrings);
    procedure CreateAllDblCheckEdits;
  public
 { The DblCheckEdits }
    property DblCheckEdits[Index: Integer]: TDblCheckEdit read GetDblCheckEdit;
    property Checked0[Index: Integer]: Boolean read GetChecked0 write SetChecked0;
    property Checked1[Index: Integer]: Boolean read GetChecked1 write SetChecked1;

 { Public methods of TDblCheckEditGroup }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ArrangeDblCheckEdits;
    {procedure SetText(Text: String; Index: Integer);
    function GetText(Index: Integer);}

  published
    property Captions : TStrings read FCaptions write SetCaptions;
    property Columns: Byte read FColumns write SetColumns default 1;
    property Count: Word read GetCount write SetCount stored FALSE;
    property EditLeft: Integer read GetEditLeft write SetEditLeft;
    property Ratio: TPercent read GetRatio write SetRatio;
    property Gap: Byte read FGap write SetGap default 4;
    //property FontCaption: TFont read FFontCaption write SetFontCaption;
    property Items : TStrings read FItems write SetItems;
    property OnCheckBoxClick: TOnCheckBoxClick read FOnCheckBoxClick write FOnCheckBoxClick;
    property OnEditClick: TOnEditClick read FOnEditClick write FOnEditClick;
    property OnEditChange: TOnEditChange read FOnEditChange  write FOnEditChange;
    property OnEditEnter: TOnEditEnter read FOnEditEnter write FOnEditEnter;
    property OnEditExit: TOnEditExit read FOnEditExit write FOnEditExit;
    property OnEditKeyDown: TOnEditKeyDown read FOnEditKeyDown write FOnEditKeyDown;
    property OnEditKeyPress: TOnEditKeyPress read FOnEditKeyPress write FOnEditKeyPress;
    property OnEditKeyUp: TOnEditKeyUp read FOnEditKeyUp write FOnEditKeyUp;
  end;

implementation

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.Create
  Description: standard constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: initializes Properties
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TDblCheckEditGroup.Create(AOwner: TComponent);
begin
{ Call the Create method of the container's parent class     }
   inherited Create(AOwner);

   Width := 200;
   Height := 200;
   //Font.Style := [fsBold];

   FColumns := 1;
   FItems := TStringList.Create;
   FCaptions := TStringList.Create;
   FGap := 4;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TDblCheckEditGroup.Destroy;
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    TDblCheckEdit(FItems.Objects[i]).Free;
  FItems.Free;
  FCaptions.Free;
  //FFontCaption.Free;

{ Last, free the component by calling the Destroy method of the  }
{ parent class.                          }
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.DoCheckBoxClick et al
  Description: responds to a click event of a single DblCheckEdit
       Author: Mat Ballard
 Suggested by: Robert Ross
 Date created: 11/27/2000
Date modified: 11/27/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.DoCheckBoxClick(Sender: TObject);
begin
  if Assigned(FOnCheckBoxClick) then
    OnCheckBoxClick(Self, TCheckBox(Sender));
end;
procedure TDblCheckEditGroup.DoEditClick(Sender: TObject);
begin
  if Assigned(FOnEditClick) then
    OnEditClick(Self, TEdit(Sender));
end;
procedure TDblCheckEditGroup.DoEditChange(Sender: TObject);
begin
  if Assigned(FOnEditChange) then
    OnEditChange(Self, TEdit(Sender));
end;
procedure TDblCheckEditGroup.DoEditEnter(Sender: TObject);
begin
  if Assigned(FOnEditEnter) then
    OnEditEnter(Self, TEdit(Sender));
end;
procedure TDblCheckEditGroup.DoEditExit(Sender: TObject);
begin
  if Assigned(FOnEditExit) then
    OnEditExit(Self, TEdit(Sender));
end;
procedure TDblCheckEditGroup.DoEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOnEditKeyDown) then
    OnEditKeyDown(Self, TEdit(Sender), Key, Shift);
end;
procedure TDblCheckEditGroup.DoEditKeyPress(Sender: TObject; var Key: Char);
begin
  if Assigned(FOnEditKeyPress) then
    OnEditKeyPress(Self, TEdit(Sender), Key);
end;
procedure TDblCheckEditGroup.DoEditKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOnEditKeyUp) then
    OnEditKeyUp(Self, TEdit(Sender), Key, Shift);
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.Loaded
  Description: responds to a load event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.Loaded;
begin
  inherited Loaded;
  //SetNoEditBoxes(FItems.Count);
  CreateAllDblCheckEdits;
  ArrangeDblCheckEdits;
end;

{------------------------------------------------------------------------------
     Function: TDblCheckEditGroup.GetDblCheckEdit
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual DblCheckEdits
 Known Issues:
 Return Value: TDblCheckEditGroup
 ------------------------------------------------------------------------------}
function TDblCheckEditGroup.GetDblCheckEdit(Index: Integer): TDblCheckEdit;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TDblCheckEdit(FItems.Objects[Index])
   else
    Result := nil;
end;

function TDblCheckEditGroup.GetChecked0(Index: Integer): Boolean;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TDblCheckEdit(FItems.Objects[Index]).Checkbox0.Checked
   else
    Result := FALSE;
end;

function TDblCheckEditGroup.GetChecked1(Index: Integer): Boolean;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TDblCheckEdit(FItems.Objects[Index]).Checkbox1.Checked
   else
    Result := FALSE;
end;

procedure TDblCheckEditGroup.SetChecked0(Index: Integer; Value: Boolean);
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
  begin
    TDblCheckEdit(FItems.Objects[Index]).Checkbox0.Checked := Value;
  end;
end;

procedure TDblCheckEditGroup.SetChecked1(Index: Integer; Value: Boolean);
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
  begin
    TDblCheckEdit(FItems.Objects[Index]).Checkbox1.Checked := Value;
  end;
end;

{------------------------------------------------------------------------------
     Function: TDblCheckEditGroup.GetCount
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual DblCheckEdits
 Known Issues:
 Return Value: TDblCheckEditGroup
 ------------------------------------------------------------------------------}
function TDblCheckEditGroup.GetCount: Word;
begin
  Result := FItems.Count;
end;

function TDblCheckEditGroup.GetEditLeft: Integer;
begin
  if (FItems.Count > 0) then
    Result := TDblCheckEdit(FItems.Objects[0]).Edit.Left
   else
    Result := 100;
end;

procedure TDblCheckEditGroup.SetEditLeft(Value: Integer);
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
  begin
    TDblCheckEdit(FItems.Objects[i]).Edit.Left := Value;
    TDblCheckEdit(FItems.Objects[i]).Edit.Width :=
      TDblCheckEdit(FItems.Objects[i]).Width -
        TDblCheckEdit(FItems.Objects[i]).Edit.Left - 4;
  end;
end;

function TDblCheckEditGroup.GetRatio: TPercent;
begin
  if (FItems.Count > 0) then
    Result := TDblCheckEdit(FItems.Objects[0]).Ratio
   else
    Result := 50;
end;

procedure TDblCheckEditGroup.SetRatio(Value: TPercent);
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    TDblCheckEdit(FItems.Objects[i]).Ratio := Value;
end;

procedure TDblCheckEditGroup.SetGap(Value : Byte);
begin
  FGap := Value;
  ArrangeDblCheckEdits;
end;
{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.SetCaptions
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Captions Property
 Known Issues: The stream system does not call this procedure !
               Instead, streamed values are loaded directly into FCaptions
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.SetCaptions(Value : TStrings);
var
  i: Integer;
begin
  if (csLoading in ComponentState) then
  begin
    FCaptions.Assign(Value);
  end
  else
  begin
    SetCount(Value.Count);
    for i := 0 to FCaptions.Count-1 do
    begin
      FCaptions.Strings[i] := Value.Strings[i];
      TDblCheckEdit(FItems.Objects[i]).Checkbox0.Caption := Value.Strings[i];
    end;
    Invalidate;
    ArrangeDblCheckEdits;
  end;
end;

{procedure TDblCheckEditGroup.SetFontCaption(Value : TFont);
var
  i: Integer;
begin
  FFontCaption.Assign(Value);
  for i := 0 to FItems.Count-1 do
  begin
    TDblCheckEdit(FItems.Objects[i]).CheckFont.Assign(FFontCaption);
  end;
end;}

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.SetItems
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: The stream system does not call this procedure !
               Instead, streamed values are loaded directly into FItems
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.SetItems(Value : TStrings);
var
  i: Integer;
begin
  if (csLoading in ComponentState) then
  begin
    FItems.Assign(Value);
  end
  else
  begin
    SetCount(Value.Count);
    for i := 0 to FItems.Count-1 do
    begin
      FItems.Strings[i] := Value.Strings[i];
      TDblCheckEdit(FItems.Objects[i]).Edit.Text := Value.Strings[i];
    end;
    Invalidate;
    ArrangeDblCheckEdits;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.CreateAllDblCheckEdits
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: we only create DblCheckEdits once loading is complete
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.CreateAllDblCheckEdits;
var
  i: Integer;
  ADblCheckEdit: TDblCheckEdit;
begin
  for i := 0 to FItems.Count-1 do
  begin
    ADblCheckEdit := TDblCheckEdit.Create(Self);
    ADblCheckEdit.Name := Format('a%d', [i]);
    ADblCheckEdit.Parent := Self;
    ADblCheckEdit.Edit.Text := FItems.Strings[i];
    //ADblCheckEdit.Caption := FCaptions.Strings[i]; 
    ADblCheckEdit.Tag := i;
    ADblCheckEdit.Font.Assign(Font);
    //ADblCheckEdit.CheckFont.Assign(FFontCaption);
    ADblCheckEdit.CheckBox0.OnClick := DoCheckBoxClick;
    ADblCheckEdit.CheckBox1.OnClick := DoCheckBoxClick;
    ADblCheckEdit.Edit.OnClick := DoEditClick;
    ADblCheckEdit.Edit.OnChange := DoEditChange;
    ADblCheckEdit.Edit.OnEnter := DoEditEnter;
    ADblCheckEdit.Edit.OnExit := DoEditExit;
    ADblCheckEdit.Edit.OnKeyDown := DoEditKeyDown;
    ADblCheckEdit.Edit.OnKeyPress := DoEditKeyPress;
    ADblCheckEdit.Edit.OnKeyUp := DoEditKeyUp;
    FItems.Objects[i] := ADblCheckEdit;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.SetCount
  Description: semi-standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the number of elements to the FItems StringList
 Known Issues: Count is not streamable
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.SetCount(Value : Word);
var
  ADblCheckEdit: TDblCheckEdit;
begin
  if (csLoading in ComponentState) then
    EComponentError.Create('SetCount called while loading !');

  if (Value > FItems.Count) then
  begin
    while (FItems.Count < Value) do
    begin
      ADblCheckEdit := TDblCheckEdit.Create(Self);
      ADblCheckEdit.Name := Format('DblCheckEdit%d', [FItems.Count+1]);
      ADblCheckEdit.Parent := Self;
      ADblCheckEdit.Tag := FItems.Count;
      ADblCheckEdit.CheckBox0.OnClick := DoCheckBoxClick;
      ADblCheckEdit.CheckBox1.OnClick := DoCheckBoxClick;
      ADblCheckEdit.Edit.OnClick := DoEditClick;
      ADblCheckEdit.Edit.OnChange := DoEditChange;
      ADblCheckEdit.Edit.OnEnter := DoEditEnter;
      ADblCheckEdit.Edit.OnExit := DoEditExit;
      ADblCheckEdit.Edit.OnKeyDown := DoEditKeyDown;
      ADblCheckEdit.Edit.OnKeyPress := DoEditKeyPress;
      ADblCheckEdit.Edit.OnKeyUp := DoEditKeyUp;
      ADblCheckEdit.Font.Assign(Font);
      //ADblCheckEdit.CheckFont.Assign(FFontCaption);
      FItems.AddObject('', ADblCheckEdit);
    end;
  end
  else if (Value < FItems.Count) then
  begin
    while (FItems.Count > Value) do
    begin
      ADblCheckEdit := TDblCheckEdit(FItems.Objects[FItems.Count-1]);
      ADblCheckEdit.Free;
      FItems.Delete(FItems.Count-1);
    end;
  end;
  while (FCaptions.Count < Value) do
    FCaptions.Add('');
  while (FCaptions.Count > Value) do
    FCaptions.Delete(FCaptions.Count-1);
  ArrangeDblCheckEdits;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.SetColumns
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Columns Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.SetColumns(Value : Byte);
begin
  if (Value = 0) then Value := 1;
  FColumns := Value;
  ArrangeDblCheckEdits;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.ArrangeDblCheckEdits
  Description: This arranges the EditBoxes on the panel
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Display management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TDblCheckEditGroup.ArrangeDblCheckEdits;
var
  i,
  NoRows,
  TheTop,
  TheWidth,
  DeltaHeight: Integer;
  ADblCheckEdit: TDblCheckEdit;
begin
  if (FItems.Count > 0) then
  begin
{position the EditBoxes:}
    TheTop := Abs(Font.Height) + FGap;
    TheWidth := (Width - (FColumns+1)*FGap - 2) div FColumns;
    NoRows := FItems.Count div FColumns;
    if ((NoRows * FColumns) < FItems.Count) then
      Inc(NoRows);
    DeltaHeight := (Height - TheTop - FGap) div NoRows;

    for i := 0 to FItems.Count-1 do
    begin
      ADblCheckEdit := TDblCheckEdit(FItems.Objects[i]);
      if (ADblCheckEdit <> nil) then
      begin
        ADblCheckEdit.Top := TheTop + (i div FColumns)*DeltaHeight;
        ADblCheckEdit.Left := FGap + (TheWidth + FGap) * (i Mod FColumns);
        ADblCheckEdit.Width := TheWidth;
        ADblCheckEdit.Visible := TRUE;
        ADblCheckEdit.Invalidate;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TDblCheckEditGroup.WMSize
  Description: standard ReSize message handler
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Re-Arranges the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF MSWINDOWS}
procedure TDblCheckEditGroup.WMSize(var Message: TMessage);
begin
  inherited;
  ArrangeDblCheckEdits;
end;
{$ENDIF}

{procedure TDblCheckEditGroup.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ArrangeDblCheckEdits;
end;}

{procedure TDblCheckEditGroup.WMPaint(var Message: TWMPaint);
var
  i: Integer;
begin
  inherited;
  Broadcast(Message);
end;}

end.

