unit Editgrp;

{$I Misc.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla 
Public License Version 1.1 (the "License"); you may not use this file except 
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for 
the specific language governing rights and limitations under the License.
The Original Code is: EditGroup.pas, released 12 May 2002.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 email: mat.ballard@chemware.hypermart.net.
                Robert Ross                 email: rross@sigmaofficesolutions.com

Last Modified: 05/25/2000
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 


Purpose:
Multi-select radio group


Known Issues:
-----------------------------------------------------------------------------}

{$I Misc.inc}

interface

uses
  Classes, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Graphics, Messages, Stdctrls
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Graphics, Messages, Stdctrls
{$ENDIF}
{$IFDEF LINUX}
  QT,
  QGraphics, QStdctrls
{$ENDIF}
  ;

const
  TEDITGROUP_VERSION = 300;

type
  TOnBoxClick = procedure(Sender: TObject; Box: TObject) of object;
  TOnBoxChange = TOnBoxClick;
  TOnBoxEnter = TOnBoxClick;
  TOnBoxExit = TOnBoxClick;
  TOnBoxKeyDown = procedure(Sender: TObject; Box: TObject; var Key: Word; Shift: TShiftState) of object;
  TOnBoxKeyPress = procedure(Sender: TObject; Box: TObject; var Key: Char) of object;
  TOnBoxKeyUp = TOnBoxKeyDown;

  {TEditType = (etNormal, etMasked, etNumeric, etValue);}

  TEditGroup = class(TGroupBox)
  private
 { Private fields of TEditGroup }
 { Storage for property Items }
    FItems : TStrings;
    FItemIndex: Integer;
    {FNo_EditBoxes: Integer;}
    FColumns: Byte;
    {FEditType: TEditType;  }
    FOnBoxClick: TOnBoxClick;
    FOnBoxChange: TOnBoxChange;
    FOnBoxEnter: TOnBoxEnter;
    FOnBoxExit: TOnBoxExit;
    FOnBoxKeyDown: TOnBoxKeyDown;
    FOnBoxKeyPress: TOnBoxKeyPress;
    FOnBoxKeyUp: TOnBoxKeyUp;

    {procedure CMFontChanged(var Message: TMessage);}
    {procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;}
{$IFDEF MSWINDOWS}
    procedure WMSize(var Message: TMessage); message WM_SIZE;
{$ENDIF}
{$IFDEF LINUX}
{$ENDIF}
  protected
    function GetEditBox(Index: Integer): TEdit;
    function GetCount: Word;
    {procedure WMPaint(var Message: TWMPaint); message WM_PAINT;)}
    procedure Loaded; override;
    procedure SetColumns(Value : Byte);
    procedure SetCount(Value : Word);
    procedure SetItems(Value : TStrings);
    procedure CreateAllEditBoxes;

    procedure DoBoxClick(Sender: TObject); virtual;
    procedure DoBoxChange(Sender: TObject); virtual;
    procedure DoBoxEnter(Sender: TObject); virtual;
    procedure DoBoxExit(Sender: TObject); virtual;
    procedure DoBoxKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;
    procedure DoBoxKeyPress(Sender: TObject; var Key: Char); virtual;
    procedure DoBoxKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;

  public
 { The EditBoxes }
    property EditBoxes[Index: Integer]: TEdit read GetEditBox;

 { Public methods of TEditGroup }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AddBox(Value: String);
    procedure ArrangeEditBoxes;

  published
    property Columns: Byte read FColumns write SetColumns;
    property Count: Word read GetCount write SetCount stored FALSE;
    property Items: TStrings read FItems write SetItems;
    property ItemIndex: Integer read FItemIndex;

    property OnBoxClick: TOnBoxClick read FOnBoxClick write FOnBoxClick;
    property OnBoxChange: TOnBoxChange read FOnBoxChange write FOnBoxChange;
    property OnBoxEnter: TOnBoxEnter read FOnBoxEnter write FOnBoxEnter;
    property OnBoxExit: TOnBoxExit read FOnBoxExit write FOnBoxExit;
    property OnBoxKeyDown: TOnBoxKeyDown read FOnBoxKeyDown write FOnBoxKeyDown;
    property OnBoxKeyPress: TOnBoxKeyPress read FOnBoxKeyPress write FOnBoxKeyPress;
    property OnBoxKeyUp: TOnBoxKeyUp read FOnBoxKeyUp write FOnBoxKeyUp;
  end;

implementation

{------------------------------------------------------------------------------
    Procedure: TEditGroup.Create
  Description: standard constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: initializes Properties
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TEditGroup.Create(AOwner: TComponent);
begin
{ Call the Create method of the container's parent class     }
   inherited Create(AOwner);

   Width := 100;
   Height := 200;

   FColumns := 1;
   FItems := TStringList.Create;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TEditGroup.Destroy;
var
  i: Integer;
begin
  for i := 0 to FItems.Count-1 do
    TEdit(FItems.Objects[i]).Free;
  FItems.Free;

{ Last, free the component by calling the Destroy method of the  }
{ parent class.                          }
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.DoBoxClick
  Description: responds to a click event of a single editbox
       Author: Mat Ballard
 Suggested by: Robert Ross
 Date created: 11/27/2000
Date modified: 11/27/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.DoBoxClick(Sender: TObject);
begin
  if Assigned(FOnBoxClick) then
    OnBoxClick(Self, Sender);
end;

procedure TEditGroup.DoBoxChange(Sender: TObject);
begin
  FItems.Strings[TEdit(Sender).Tag] := TEdit(Sender).Text;
  FItemIndex := TEdit(Sender).Tag;
  if Assigned(FOnBoxChange) then
    OnBoxChange(Self, Sender);
end;

procedure TEditGroup.DoBoxEnter(Sender: TObject);
begin
  if Assigned(FOnBoxEnter) then
    OnBoxEnter(Self, Sender);
end;

procedure TEditGroup.DoBoxExit(Sender: TObject);
begin
  if Assigned(FOnBoxExit) then
    OnBoxExit(Self, Sender);
end;

procedure TEditGroup.DoBoxKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOnBoxKeyDown) then
    OnBoxKeyDown(Self, Sender, Key, Shift);
end;

procedure TEditGroup.DoBoxKeyPress(Sender: TObject; var Key: Char);
begin
  if Assigned(FOnBoxKeyPress) then
    OnBoxKeyPress(Self, Sender, Key);
end;

procedure TEditGroup.DoBoxKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOnBoxKeyUp) then
    OnBoxKeyUp(Self, Sender, Key, Shift);
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.Loaded
  Description: responds to a load event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.Loaded;
begin
  inherited Loaded;
  {SetNoEditBoxes(FItems.Count);}
  CreateAllEditBoxes;
  ArrangeEditBoxes;
end;

{------------------------------------------------------------------------------
     Function: TEditGroup.GetEditBox
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual EditBoxes
 Known Issues:
 Return Value: TEditGroup
 ------------------------------------------------------------------------------}
function TEditGroup.GetEditBox(Index: Integer): TEdit;
begin
  if ((0 <= Index) and (Index < FItems.Count)) then
    Result := TEdit(FItems.Objects[Index])
   else
    Result := nil;
end;

{------------------------------------------------------------------------------
     Function: TEditGroup.GetCount
  Description: standard array property Get function
       Author: Mat Ballard
 Date created: 04/25/2002
Date modified: 04/25/2002 by Mat Ballard
      Purpose: gives access to the individual EditBoxes
 Known Issues:
 Return Value: TEditGroup
 ------------------------------------------------------------------------------}
function TEditGroup.GetCount: Word;
begin
  Result := FItems.Count;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.SetItems
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: The stream system does not call this procedure !
               Instead, streamed values are loaded directly into FItems
 ------------------------------------------------------------------------------}
procedure TEditGroup.SetItems(Value : TStrings);
var
  i: Integer;
begin
  if (csLoading in ComponentState) then
  begin
    FItems.Assign(Value);
  end
  else
  begin
    SetCount(Value.Count);
    for i := 0 to FItems.Count-1 do
    begin
      FItems.Strings[i] := Value.Strings[i];
      TEdit(FItems.Objects[i]).Text := Value.Strings[i];
    end;
    Invalidate;
    ArrangeEditBoxes;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.CreateAllEditBoxes
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: we only create Editboxes once loading is complete
 ------------------------------------------------------------------------------}
procedure TEditGroup.CreateAllEditBoxes;
var
  i: Integer;
  AEdit: TEdit;
begin
  for i := 0 to FItems.Count-1 do
  begin
    AEdit := TEdit.Create(Self);
    AEdit.Parent := Self;
    AEdit.Text := FItems.Strings[i];
    AEdit.Tag := i;
    AEdit.OnChange := DoBoxChange;
    AEdit.OnClick := DoBoxClick;
    AEdit.OnEnter := DoBoxEnter;
    AEdit.OnExit := DoBoxExit;
    AEdit.OnKeyDown := DoBoxKeyDown;
    AEdit.OnKeyPress := DoBoxKeyPress;
    AEdit.OnKeyUp := DoBoxKeyUp;
    FItems.Objects[i] := AEdit;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.SetCount
  Description: semi-standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the number of elements to the FItems StringList
 Known Issues: Count is not streamable
 ------------------------------------------------------------------------------}
procedure TEditGroup.SetCount(Value : Word);
var
  AEdit: TEdit;
begin
{$IFDEF DELPHI3_UP}
  Assert(not (csLoading in ComponentState), 'SetCount called while loading !');
{$ENDIF}

  if (Value > FItems.Count) then
  begin
    while (FItems.Count < Value) do
    begin
      AEdit := TEdit.Create(Self);
      AEdit.Parent := Self;
      AEdit.Tag := FItems.Count;
      AEdit.OnChange := DoBoxChange;
      AEdit.OnClick := DoBoxClick;
      AEdit.OnEnter := DoBoxEnter;
      AEdit.OnExit := DoBoxExit;
      AEdit.OnKeyDown := DoBoxKeyDown;
      AEdit.OnKeyPress := DoBoxKeyPress;
      AEdit.OnKeyUp := DoBoxKeyUp;
      FItems.AddObject('', AEdit);
    end;
  end
  else if (Value < FItems.Count) then
  begin
    while (FItems.Count > Value) do
    begin
      AEdit := TEdit(FItems.Objects[FItems.Count-1]);
      AEdit.Free;
      FItems.Delete(FItems.Count-1);
    end;
  end;
  ArrangeEditBoxes;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.SetColumns
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Columns Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.SetColumns(Value : Byte);
begin
  if (Value = 0) then Value := 1;
  FColumns := Value;
  ArrangeEditBoxes;
end;

procedure TEditGroup.AddBox(Value: String);
begin
  SetCount(Items.Count + 1);
  Items.Strings[Items.Count-1] := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.ArrangeEditBoxes
  Description: This arranges the EditBoxes on the panel
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Display management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.ArrangeEditBoxes;
var
  i,
  BoxenHeight,
  NoRows,
  TheTop,
  TheLeft,
  TheWidth,
  TheGap,
  DeltaHeight: Integer;
  AEditBox: TEdit;
begin
  if (FItems.Count > 0) then
  begin
    AEditBox := TEdit(FItems.Objects[0]);
    NoRows := FItems.Count div FColumns;
    if (FColumns * NoRows < FItems.Count) then
      Inc(NoRows);
    BoxenHeight := NoRows * AEditBox.Height;
    TheTop := (Height - BoxenHeight) div (NoRows + 1);

{position the EditBoxes:}
    DeltaHeight := TheTop + AEditBox.Height;
    TheLeft := 8;
    TheGap := 12;
    TheWidth := Width - (FColumns+1)*TheLeft - 2;
    TheWidth := TheWidth div FColumns;

    for i := 0 to FItems.Count-1 do
    begin
      AEditBox := TEdit(FItems.Objects[i]);
      if (AEditBox <> nil) then
      begin
        AEditBox.Top := TheTop + (i div FColumns)*DeltaHeight;
        AEditBox.Left := TheLeft + (TheWidth + TheGap) * (i Mod FColumns);
        AEditBox.Width := TheWidth;
        AEditBox.Visible := TRUE;
        AEditBox.Invalidate;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.WMSize
  Description: standard ReSize message handler
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Re-Arranges the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF MSWINDOWS}
procedure TEditGroup.WMSize(var Message: TMessage);
begin
  inherited;
  ArrangeEditBoxes;
end;
{$ENDIF}

{procedure TEditGroup.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ArrangeEditBoxes;
end;}

{procedure TEditGroup.WMPaint(var Message: TWMPaint);
var
  i: Integer;
begin
  inherited;
  Broadcast(Message);
end;}

end.

