unit Misc;

{$I Misc.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: PlotMisc.pas, released 1 July 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 09/25/2002
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 


Purpose:
Collection of miscellaneous routines and type definitions, that really don't
belong anywhere else.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, TypInfo,
{$IFDEF NO_MATH}
  Nomath,
{$ELSE}
  Math,
{$ENDIF}
{$IFDEF WINDOWS}
  WinTypes, WinProcs, Winsock, 
  Buttons, Controls, Dialogs, Forms, Graphics, ShellApi
{$ENDIF}
{$IFDEF WIN32}
  Windows, Winsock,
  Buttons, Clipbrd, Controls, Dialogs, Forms, Graphics, ShellApi, stdctrls
{$ENDIF}
{$IFDEF LINUX}
  Libc, Types, Qt,
  QButtons, QControls, QDialogs, QForms, QGraphics, QStdctrls
{$ENDIF}
  ;

{Misc and TPlot now only work for Compiler 3 up !}
{$IFDEF COMPILER3_UP}
{$ELSE}
  { Misc and TPlot now only work for Compiler 3 up !}
{$ENDIF}

type
  TOnMessageEvent = procedure(Sender: TObject; Msg: String) of object;
{This event is used to signal Messages.}
  TOnErrorEvent = TOnMessageEvent;
{This event is used to signal Errors.}
  TOnWarningEvent = TOnMessageEvent;
{This event is used to signal Warnings.}

  pSingle = ^Single;
{Pointer to single.}
  pDouble = ^Double;
{Pointer to Double.}

{dynamic matrix definitions:}
{$IFDEF DELPHI1}
  TIntegerArray = array[0..MaxInt - 1] of Integer;
  TSingleArray = array[0..MaxInt div 2 - 1] of Single;
  TDoubleArray = array[0..MaxInt div 4 - 1] of Double;
  TBooleanArray = array[0..MaxInt - 1] of Boolean;
{$ELSE}
  TIntegerArray = array[0..MaxInt div SizeOf(Integer) - 1] of Integer;
{A dynamic array of Integers.}
  TSingleArray = array[0..MaxInt div SizeOf(Single) -1] of Single;
{A dynamic array of Single.}
  TDoubleArray = array[0..MaxInt div SizeOf(Double) - 1] of Double;
{A dynamic array of Double.}
  TBooleanArray = array[0..MaxInt div SizeOf(Boolean) - 1] of Boolean;
{A dynamic array of Boolean.}
{$ENDIF}

{dynamic matrix definitions - pointers thereto:}
  pIntegerArray = ^TIntegerArray;
{A pointer to dynamic array of Integers.}
  pSingleArray = ^TSingleArray;
{A pointer to dynamic array of Single.}
  pDoubleArray = ^TDoubleArray;
{A pointer to dynamic array of Double.}
  pBooleanArray = ^TBooleanArray;
{A pointer to dynamic array of Boolean.}

  TPercent = 0..100;
{Integer numbers between 0 and 100: ie: percent.}

  TXYPoint = record
    X: Single;
    Y: Single;
  end;
  pXYPoint = ^TXYPoint;
{A real (X, Y) point, and a pointer thereto.}

  TXYArray = array of TXYPoint;
{A dynamic array of real (X, Y) points.}
  pXYArray = ^TXYArray;
{A pointer to dynamic array of real (X, Y) points.}

  {TIdentMapEntry = record
    Value: TColor;
    Name: String;
  end;}
{Definition from Delphi source file classes.pas not included in project; for color management.}

{$IFDEF LINUX}
  hFont = QFontH; {for TextOutAnglePersist}
  
  TRGBTriple = packed record
    rgbtBlue: Byte;
    rgbtGreen: Byte;
    rgbtRed: Byte;
  end;
{$ENDIF}
  TRGBArray    = array[0..20000] OF TRGBTriple;
  pRGBArray    = ^TRGBArray;
{Another 24 bit color definition for color management.}

  TRainbowColor = record
    R: Integer;
    G: Integer;
    B: Integer;
  end;
{Another 24 bit color definition for color management.}

  TFileParts = record
    Drive: char;
    Machine: String;
    Dir: String;
    Root: String;
    Index: Integer;
  end;

{TFileList ********************************************************************}
  TFileList = class(TStringList)
    private
    protected
    public
      procedure AppendToFile(const FileName: string); virtual;
{Sometimes it's nice to be able to Append to a text file, rather than re-write the whole thing.}
    published
  end;

{TMemoryStreamEx **************************************************************}
  TMemoryStreamEx = class(TMemoryStream)
    private
    protected
    public
      procedure AppendToFile(const FileName: string); virtual;
{Sometimes it's nice to be able to Append to a memory stream, rather than re-write the whole thing.}
  end;

{Sundry functions *************************************************************}
  function GetLineLengthFromStream(AStream: TMemoryStream): Integer;
{This function gets the length of the line (of text) at AStream.Position.}
  function ReadLine(AStream: TMemoryStream): String;
{This function gets line (of text) at AStream.Position.}
  function FindStringInStream(TheString: String; AStream: TMemoryStream): Boolean;
{This function finds the first occurrence of TheString in AStream from AStream.Position onwards}
  function GetNumberProperty(AStr: String; AStream: TMemoryStream): Extended;
{This function finds the value of first occurrence of AStr in AStream from
 AStream.Position onwards, and returns it as a number.}
  function GetStringProperty(AStr: String; AStream: TMemoryStream): String;
{This function finds the value of first occurrence of AStr in AStream from
 AStream.Position onwards, and returns it as a string.}
  function GetBooleanProperty(AStr: String; AStream: TMemoryStream): Boolean;
{This function finds the value of first occurrence of AStr in AStream from
 AStream.Position onwards, and returns it as a Boolean.}

  function AbsPos(SubStr, Str: String; Start: Integer): Boolean;
{Is SubStr at position Start in Str ?}
  function BackPos(SubStr, Str: String; Start: Integer): Integer;
{This function finds the previous occurrence of SubStr in Str BEFORE Start.}
  function ForwardPos(SubStr, Str: String; Start: Integer): Integer;
{This function finds the first occurrence of SubStr in Str AFTER Start.}
  function ExtractFileRoot(FileName: String): String;
{This function returns the base file name with no extension.}
  function ExtractHtmlValue(Html, Tag: String): String;
{This function extracts the tag value from a HTML string}
  procedure ExtractHtmlFontSize(Html: String; var Size: Integer);
{This function extracts the font size alone from a HTML string}
  function ExtractHtmlFontInfo(Html: String; var Face: String; var Size: Integer; var Color: TColor; var Style: TFontStyles; var Text: String): Boolean;
{This function extracts all the font properties from a HTML string}
  function CleanString(AString: String; TheChar: Char): String;
{This function removes offending characters from a string.}
  function StrRev(TheStr: String): String;
{This function reverses a string.}
  function GetAngle(Xi, Yi: Integer): Extended;
{This returns the (clockwise) angle of a point from the vertical, in radians.}
  function GetAngleDeg(Xi, Yi: Integer): Extended;
{This returns the (clockwise) angle of a point from the vertical, in degrees.}

  function GetWord(var ALine: String; Delimiter: String): String;
{The GetWord function returns all the characters up to Delimiter in ALine,
 and removes all characters up to and including Delimiter from ThisLine.}
{}
{This is very useful for extracting comma or tab-seperated strings (numbers)
 from text data.}
{$IFDEF DELPHI_1}
  function TrimLeft(Const Str: String): String;
{$ENDIF}
  function GetNextToken(var ALine: String): String;
{This function gets the next whitespace-delimited cluster of characters.}
  function GetNextTokenAsym(var ALine: String; var Complete: Boolean; Delim1, Delim2: Char): String;
{This function gets the next delimited cluster of characters.}
  function StripTags(ALine: String): String;
{This function strips out HTML tags from a string.}
  function TrimEx(ALine: String): String;
{This function removes all except numbers and letters.}

  function IndexOfColorValue(Value: TColor): Integer;
{This function gets the index of a color in the MyColorValues array.}
  function IndexOfColorName(Name: String): Integer;
{This function gets the index of a color name in the MyColorValues array.}
  function GetColorMix(Color1, Color2: TColor; Fraction: Single): TColor;
{This function blends two colors to make a third; good for gradient effects.}
  function GetDarkerColor(Value: TColor; Brightness: Integer): TColor;
{This function gets a darker shade of the input color.}
  function GetInverseColor(Value: TColor): TColor;
{This function gets the inverse of the input color.}
  function GetPalerColor(Value: TColor; Brightness: Integer): TColor;
{This function gets a paler shade of the input color.}
  function Rainbow(Fraction, Gamma, LambdaMin, LambdaSpan: Single): TColor;
{This function returns a spectral color, depending on the Fraction, and the minimum and
 maximum wavelengths.}
  function RainbowOld(Fraction: Single): TColor;
{This function returns a rainbow color, depending on the Fraction.
 Superceded by Rainbow.}
  function Spectrum(Lambda, Gamma: Single): TColor;
{This function returns a spectral color, depending on the wavelength Lambda.}
  function InputColor(var AColor: TColor): Boolean;
{This function prompts the user for a color.}
  function ColorToHTML(AColor: TColor): String;
{This function converts a TColor to a HTML color, eg: #FF0088}
  procedure MessageState(AState: TComponentState);

  function BinToInt(Value: String): {$IFDEF DELPHI1}Longint{$ELSE}{$IFDEF BCB}Longint{$ELSE}Longword{$ENDIF}{$ENDIF};
{This function converts a binary string to an integer number.}
  function IntToBin(Value: {$IFDEF DELPHI1}Longint{$ELSE}{$IFDEF BCB}Longint{$ELSE}Longword{$ENDIF}{$ENDIF}): string;
{This function converts an integer number to a binary string.}
  function LongintToRoman(const N : longint) : string; { from WR-ROMAN.PAS } 
  function RomanToLongint(const S : string) : longint;

  function IsNAN(s:single): boolean;
{This function tests s for NAN (not a number).}
  procedure MakeNAN(var s:single);
{This function makes s into NAN (not a number).}
  function IsInteger(Value: String): Boolean;
{This function tests for the string Value being an integer.}
  function IsFixed(Value: String): Boolean;
{This function tests for the string Value being a fixed point real number.}
  function IsReal(Value: String): Boolean;
{This function tests for the string Value being a real number.}

  procedure LoadResourceFile(aFile:string; var MemoryStream:TMemoryStream);
{This function loads a binary or text file from resources into a memory stream.}

{$IFDEF LINUX}
  function GetBrowser: String;
{This function gets the user's prefered browser in Linux.}
  function GetMailer: String;
{This function gets the user's prefered Mailer in Linux.}
  function CheckForRPM(AnRPM: String): String;
{This function checks for the existence / installation of the AnRPM program in Linux.}
{$ENDIF}

{$IFDEF DELPHI1}
  function GetCurrentDir: String;
{This function gets the current directory.}
  procedure SetLength(var Str: String; len: Integer);
{$ENDIF}

{Sundry procedures ************************************************************}
  procedure DeSci(ExtNumber: Extended; var Mantissa: Extended; var Exponent: Integer);
{This method breaks a number down into its mantissa and exponent.
 Eg: 0.00579 has a mantissa of 5.79, and an exponent of -3.}

  procedure Wait(mSeconds: Integer; ProcessMessages: Boolean);
{This method waits mSeconds ms, and ProcessMessages if required.}

  procedure SetDialogGeometry(AForm: TForm; AButton: TControl; BorderWidth: Integer);
{This method sets the dialog Geometry under Windows and Linux.}

  function TextOutAnglePersist(
    ACanvas: TCanvas;
    var OldFontHandle: hFont;
    Angle, X, Y: Integer;
    TheText: String): hFont;
{This method draws text on the input canvas, at an angle, and leaves the rotated font intact.}
  procedure TextOutAnglePersistCleanUp(
    ACanvas: TCanvas;
    NewFontHandle, OldFontHandle: hFont);
{This method cleans up a previously rotated persistent font}
  procedure TextOutAngle(
    ACanvas: TCanvas;
    Angle, Left, Top: Integer;
    TheText: String);
{This function draws angled text on the input canvas under both Windows and Linux.}
  procedure ShellExec(Cmd: String);
{This method is wrapper for the windows "ShellExecute" API call, extended to Linux.}
  procedure ShowHTML(Cmd: String);
{This function shows a html file under both Windows and Linux.}
  procedure DoMail(Cmd: String);
{This function drops an email address to the users mail program under both Windows and Linux.}
{$IFDEF LINUX}
  procedure DoHTMLHelp(
    HelpType: THelpType;
    HelpContext: Integer;
    HelpKeyword: string;
    HelpFile: string;
    var Handled: Boolean);
{This function displays a topic from a HTML-based help website.}
{$ENDIF}
{$IFDEF WIN32}
  procedure RunDosInMemo(DosApp:String; AMemo: TMemo);
{This function runs a DOS program and outputs the results to a Memo.
 I no longer have the details of the original author, and it has a few issues (lockups) anyway.}
  function GetDNSName(var DNSName, IP, Msg: String): Boolean;
{Returns the DNS name and IP address of the local PC}
{$ENDIF}
  function CreatePathFromRelativePath(BasePath, RelPath: string): string;
{This function is the opposite of ExtractRelativePath.}
  function StripDir(APath: string): string;
{This function removes path information up to trailing '\'.}
  procedure DbgPrint(Msg: String);
{Debug Print}

const
{$IFDEF MSWINDOWS}
  DIR_SEPERATOR = '\';
{$ENDIF}
{$IFDEF LINUX}
  DIR_SEPERATOR = '/';
{$ENDIF}

  PI_ON_2 =       1.57079632679489; {66192313216916398}
  THREE_PI_ON_2 = 4.71238898038468; {98576939650749193}
  TWO_PI =        6.28318530717958; {6476925286766559}
  DEGS_PER_RAD =  57.2957795130823; {20876798154814105}

{Time intervals used by TDateTime:}
  AN_HOUR = 1.0 / 24.0;
  A_MINUTE = AN_HOUR / 60.0;
  A_SECOND = A_MINUTE / 60.0;

  CRLF = #13#10;
  MY_COLORS_MAX = 15;
{The number of MyColors runs from 0..15.}

  MyColorValues: array[0..15] of TColor = (
    clBlack,
    clRed,
    clBlue,
    clGreen,
    clPurple,
    clFuchsia,
    clAqua,
    clMaroon,
    clOlive,
    clNavy,
    clTeal,
    clGray,
    clSilver,
    clLime,
    clYellow,
    clWhite);
{MyColorValues is based on the Colors definition in Graphics.pas,
 restricted the the basic 16 colors, and in a different order
 more suitable for graphs.}

  MAX_RAINBOW_COLORS = 5;
  RainbowColors: array[0..MAX_RAINBOW_COLORS, 0..2] of Integer =
    ({(0, 0, 0),          //black}
     (255, 0, 0),        {red}
     (255, 255, 0),      {yellow}
     (0, 255, 0),        {green}
     (0, 255, 255),      {aqua}
     (0, 0, 255),        {blue}
     (255, 0, 255));      {purple}
     {(255, 255, 255));   //white}
{Note: Black and white have been removed to avoid confusion with the background.}

  LAMBDA_MIN = 380;
{The minimum wavelength in a rainbow, which is blue in color.}
  LAMBDA_MAX = 780;
{The maximum wavelength in a rainbow, which is red in color.}

implementation

uses
  Coloredt, Nedit, Options;

{$IFDEF LINUX}
resourcestring
  sFileName = '/tmp/delete-me.txt';
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TFileList.AppendToFile
  Description: appends this stringlist to an existing file
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saving data to disk
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TFileList.AppendToFile(const FileName: string);
var
  Stream: TStream;
begin
  if (FileExists(FileName)) then
  begin
    Stream := TFileStream.Create(FileName, fmOpenReadWrite);
    Stream.Seek(0, soFromEnd);
  end
  else
  begin
    Stream := TFileStream.Create(FileName, fmCreate);
  end;

  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;
{end TFileList ----------------------------------------------------------------}

{Begin TMemoryStreamEx ----------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TMemoryStreamEx.AppendToFile
  Description: appends this MemoryStream to an existing file
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saving data to disk
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TMemoryStreamEx.AppendToFile(const FileName: string);
var
  Stream: TStream;
begin
  if (FileExists(FileName)) then
  begin
    Stream := TFileStream.Create(FileName, fmOpenReadWrite);
    Stream.Seek(0, soFromEnd);
  end
  else
  begin
    Stream := TFileStream.Create(FileName, fmCreate);
  end;

  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;
{end TMemoryStreamEx ----------------------------------------------------------}

{------------------------------------------------------------------------------
     Function: GetLineLengthFromStream
  Description: gets the length of the line (of text) at AStream.Position
       Author: Mat Ballard
 Date created: 08/09/2000
Date modified: 08/09/2000 by Mat Ballard
      Purpose: Stream manipulation
 Return Value: the length of the line, up to CRLF
 Known Issues:
 ------------------------------------------------------------------------------}
function GetLineLengthFromStream(AStream: TMemoryStream): Integer;
var
  pCR,
  pLF: PChar;
  i: Longint;
begin
  pCR := AStream.Memory;
  Inc(pCR, AStream.Position);
{default is the entire stream:}
  GetLineLengthFromStream := AStream.Size - AStream.Position;
  for i := AStream.Position to AStream.Size-1 do
  begin
    if (pCR^ = #13) then
    begin
      pLF := pCR;
      Inc(pLF);
      if (pLF^ = #10) then
      begin
        GetLineLengthFromStream := i - AStream.Position;
        break;
      end;
    end;
    Inc(pCR);
  end;
end;

{------------------------------------------------------------------------------
     Function: ReadLine
  Description: gets line (of text) at AStream.Position
       Author: Mat Ballard
 Date created: 08/09/2000
Date modified: 04/28/2001 by Mat Ballard
      Purpose: Stream manipulation
 Return Value: the line as a string
 Known Issues: does not work against TBlobStream
 ------------------------------------------------------------------------------}
function ReadLine(AStream: TMemoryStream): String;
var
  LineLength: Integer;
  pLine: array [0..1023] of char;
begin
  LineLength := GetLineLengthFromStream(AStream);
{get the line of text:}
{$IFDEF DELPHI1}
  AStream.Read(pLine, LineLength);
  Result := StrPas(pLine);
{$ELSE}
  SetString(Result, PChar(nil), LineLength);
  AStream.Read(Pointer(Result)^, LineLength);
{$ENDIF}
{get the CRLF:}
  AStream.Read(pLine, 2);
end;

{------------------------------------------------------------------------------
     Function: FindStringInStream
  Description: Finds the first occurrence of TheString in AStream from
               AStream.Position onwards
       Author: Mat Ballard
 Date created: 08/09/2000
Date modified: 08/09/2000 by Mat Ballard
      Purpose:
 Return Value: TRUE if successful, FALSE otherwise
 Known Issues:
 ------------------------------------------------------------------------------}
function FindStringInStream(TheString: String; AStream: TMemoryStream): Boolean;
var
  pStart,
  pTheChar: PChar;
  i,
  j: Longint;
  FoundIt: Boolean;
begin
  pStart := AStream.Memory;
  Inc(pStart, AStream.Position);
{default is the entire stream:}
  FindStringInStream := FALSE;
  for i := AStream.Position to AStream.Size-1 do
  begin
    pTheChar := pStart;
    FoundIt := TRUE;
    for j := 1 to Length(TheString) do
    begin
      if (pTheChar^ <> TheString[j]) then
      begin
        FoundIt := FALSE;
        break;
      end;
      Inc(pTheChar);
    end;

    if (FoundIt) then
    begin
      AStream.Position := i;
      FindStringInStream := TRUE;
      break;
    end;

    Inc(pStart);
  end;
end;

{------------------------------------------------------------------------------
     Function: GetNumberProperty
  Description: Finds the value of first occurrence of AStr in AStream from
               AStream.Position onwards, and returns it as a number
       Author: Mat Ballard
 Date created: 06/09/2002
Date modified: 06/09/2002 by Mat Ballard
      Purpose:
 Return Value: The property value
 Known Issues:
 ------------------------------------------------------------------------------}
function GetNumberProperty(AStr: String; AStream: TMemoryStream): Extended;
var
  OldPos: Integer;
  pStr: pChar;
  Ptr: Pointer;
  pValueType: ^TValueType;
  pShortint: ^Shortint;
  pSmallInt: ^Smallint;
  pInteger: ^Integer;
{$IFDEF COMPILER4_UP}
  pSingle: ^Single;
{$ENDIF}
  pExt: ^Extended;
begin
  Result := 0;
  OldPos := AStream.Position;
  if (Misc.FindStringInStream(AStr, AStream)) then
  begin
    pStr := AStream.Memory;
    Inc(pStr, AStream.Position + Length(AStr));
    Ptr := pStr;
    pValueType := Ptr;
    Inc(pStr, SizeOf(TValueType));
    Ptr := pStr;
    case pValueType^ of
      vaInt8:
        begin
          pShortint := Ptr;
          Result := pShortint^;
        end;
      vaInt16:
        begin
          pSmallInt := Ptr;
          Result := pSmallint^;
        end;
      vaInt32:
        begin
          pInteger := Ptr;
          Result := pInteger^;
        end;
      vaExtended:
        begin
          pExt := Ptr;
          Result := pExt^;
        end;
{$IFDEF COMPILER4_UP}
      vaSingle:
        begin
          pSingle := Ptr;
          Result := pSingle^;
        end;
{$ENDIF}
    else
      EComponentError.CreateFmt('Unknown type %d !', [Ord(pValueType^)]);
    end;
  end
  else
  begin
    AStream.Position := OldPos;
    {Write(Format('%s = 0' + #13#10, [AStr]));}
  end;
end;

{------------------------------------------------------------------------------
     Function: GetStringProperty
  Description: Finds the value of first occurrence of AStr in AStream from
               AStream.Position onwards
       Author: Mat Ballard
 Date created: 06/09/2002
Date modified: 06/09/2002 by Mat Ballard
      Purpose:
 Return Value: The property value
 Known Issues:
 ------------------------------------------------------------------------------}
function GetStringProperty(AStr: String; AStream: TMemoryStream): String;
var
  i, NoChars, OldPos: Integer;
  pStr: pChar;
  Ptr: Pointer;
  pValueType: ^TValueType;
  pShortint: ^Shortint;
  pSmallInt: ^Smallint;
begin
  Result := '';
  OldPos := AStream.Position;
  if (Misc.FindStringInStream(AStr, AStream)) then
  begin
    NoChars := 0;
    pStr := AStream.Memory;
    Inc(pStr, AStream.Position + Length(AStr));
    Ptr := pStr;
    pValueType := Ptr;
    Inc(pStr, SizeOf(TValueType));
    Ptr := pStr;
    case pValueType^ of
      vaIdent, vaString:
        begin
          pShortint := Ptr;
          NoChars := pShortint^;
          Inc(pStr, SizeOf(Shortint));
        end;
{$IFDEF COMPILER2_UP}
      vaLString:
        begin
          pSmallInt := Ptr;
          NoChars := pSmallInt^;
          Inc(pStr, SizeOf(SmallInt));
        end;
{$ENDIF}
    else
      EComponentError.CreateFmt('Unknown type %d !', [Ord(pValueType^)]);
    end;
    for i := 1 to NoChars do
    begin
      Result := Result + pStr^;
      Inc(pStr);
    end;
  end
  else
  begin
    AStream.Position := OldPos;
  end;
end;

{------------------------------------------------------------------------------
     Function: GetBooleanProperty
  Description: Finds the value of first occurrence of AStr in AStream from
               AStream.Position onwards
       Author: Mat Ballard
 Date created: 06/09/2002
Date modified: 06/09/2002 by Mat Ballard
      Purpose:
 Return Value: The property value
 Known Issues:
 ------------------------------------------------------------------------------}
function GetBooleanProperty(AStr: String; AStream: TMemoryStream): Boolean;
var
  OldPos: Integer;
  pStr: pChar;
  Ptr: Pointer;
  pValueType: ^TValueType;
begin
  OldPos := AStream.Position;
  Result := FALSE;
  if (Misc.FindStringInStream(AStr, AStream)) then
  begin
    pStr := AStream.Memory;
    Inc(pStr, AStream.Position + Length(AStr));
    Ptr := pStr;
    pValueType := Ptr;
    Result := pValueType^ = vaTrue;
  end
  else
  begin
    AStream.Position := OldPos;
  end;
end;

function AbsPos(SubStr, Str: String; Start: Integer): Boolean;
var
  i: Integer;
begin
  Result := FALSE;
  if (Length(SubStr) + Start -1 <= Length(Str)) then
  begin
    for i := 1 to Length(SubStr) do
    begin
      if (SubStr[i] <> Str[Start + i - 1]) then exit;
    end;
    Result := TRUE;
  end;
end;

{------------------------------------------------------------------------------
     Function: BackPos
  Description: finds the previous occurrence of SubStr in Str BEFORE Start
       Author: Mat Ballard
 Date created: 01/12/2001
Date modified: 01/12/2001 by Mat Ballard
      Purpose: string manipulation
 Return Value: the position of the previous SubStr
 Known Issues:
 ------------------------------------------------------------------------------}
{    Example:   the location of "\strike" prior to "\[link=" in:
               \tx2085\plain\f3\fs20\strike New\plain\f3\fs20 \{link=AHLG7T\}
function BackPos(SubStr, Str: String; Start: Integer): Integer;
var
  i, j: Integer;
  Matched: Boolean;
begin
  Result := 0;
  if (Start < 1) then exit;
  if (Start > Length(Str)) then
    Start := Length(Str);
  for i := Start downto 1 do
  begin
    Matched := TRUE;
    for j := 1 to Length(SubStr) do
    begin
     if (SubStr[j] <> Str[i+j-1]) then
     begin
       Matched := FALSE;
       break;
     end;
    end;
    if (Matched) then
    begin
     Result := i;
     break;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: ForwardPos
  Description: finds the first occurrence of SubStr in Str AFTER Start
       Author: Mat Ballard
 Date created: 01/12/2001
Date modified: 01/12/2001 by Mat Ballard
      Purpose: string manipulation
 Return Value: the position of the next SubStr
 Known Issues:
 ------------------------------------------------------------------------------}
function ForwardPos(SubStr, Str: String; Start: Integer): Integer;
var
  i, j: Integer;
  Matched: Boolean;
begin
  Result := 0;
  if ((Start < 1) or (Start > Length(Str))) then exit;
  for i := Start to (Length(Str) - Length(SubStr)+1) do
  begin
    Matched := TRUE;
    for j := 1 to Length(SubStr) do
    begin
     if (SubStr[j] <> Str[i+j-1]) then
     begin
       Matched := FALSE;
       break;
     end;
    end;
    if (Matched) then
    begin
     Result := i;
     break;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: ExtractFileRoot
  Description: returns the base file name with no extension
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: menu manipulation
 Return Value: the cleaned string
 Known Issues:
 ------------------------------------------------------------------------------}
function ExtractFileRoot(FileName: String): String;
begin
  FileName := ExtractFileName(FileName);
  Result := Copy(FileName, 1, Pos(ExtractFileExt(FileName), FileName)-1);
end;

{------------------------------------------------------------------------------
     Function: ExtractHtmlValue
  Description: This function extracts the font tag value from a HTML string
       Author: Mat Ballard
 Date created: 05/10/2003
Date modified: 05/10/2003 by Mat Ballard
      Purpose: fonts for list- and combo boxes
 Return Value: the string value
 Known Issues: initialize the variables first with defaults !
 ------------------------------------------------------------------------------}
function ExtractHtmlValue(Html, Tag: String): String;
var
  iPos: Integer;
begin
  Result := '';
  iPos := Pos(Tag, Html);
  if (iPos > 0) then
  begin
    iPos := iPos + Length(Tag);
    if (Html[iPos] = '"') then Inc(iPos);
    while iPos < Length(Html) do
    begin
      if ((Html[iPos] = ' ') or (Html[iPos] = '"')) then
        break;
      Result := Result + Html[iPos];
      Inc(iPos);
    end;
  end
end;

{------------------------------------------------------------------------------
    Procedure: ExtractHtmlFontSize
  Description: This function extracts the font size alone from a HTML string
       Author: Mat Ballard
 Date created: 05/10/2003
Date modified: 05/10/2003 by Mat Ballard
      Purpose: fonts for list- and combo boxes
 Known Issues: initialize the variables first with defaults !
 ------------------------------------------------------------------------------}
procedure ExtractHtmlFontSize(Html: String; var Size: Integer);
var
  Str: String;
  i: Integer;
begin
  Html := Lowercase(Html);
  if (Pos('<font', Html) > 0) then
  begin
    i := Pos('>', Html);
    if (i > 0) then
    begin
      SetLength(Html, i);
      Str := ExtractHtmlValue(Html,'size=');
      if (Length(Str) > 0) then
      begin
        Size := StrToInt(Str);
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: ExtractHtmlFontInfo
  Description: This function extracts all the font information from a HTML string
       Author: Mat Ballard
 Date created: 05/10/2003
Date modified: 05/10/2003 by Mat Ballard
      Purpose: fonts for list- and combo boxes
 Return Value: tag-free text      
 Known Issues: initialize the variables first with defaults !
 ------------------------------------------------------------------------------}
function ExtractHtmlFontInfo(Html: String;
  var Face: String;
  var Size: Integer;
  var Color: TColor;
  var Style: TFontStyles;
  var Text: String): Boolean;
var
  MyHtml, Str: String;
  i: Longint;
  AStyle: TFontStyles;
  InTag: Boolean;
begin
  Result := FALSE;

  MyHtml := Lowercase(Html);
  AStyle := [];
  if (Pos('<b>', MyHtml) > 0) then AStyle := [fsBold];
  if (Pos('<i>', MyHtml) > 0) then AStyle := AStyle + [fsItalic];
  if (Pos('<u>', MyHtml) > 0) then
    AStyle := AStyle + [fsUnderline];
  if (Style <> AStyle) then
  begin
    Style := AStyle;
    Result := TRUE;
  end;

  if (Pos('<font', MyHtml) > 0) then
  begin
    i := Pos('>', MyHtml);
    if (i > 0) then
    begin
      SetLength(MyHtml, i);
      Str := ExtractHtmlValue(MyHtml,'size=');
      if (Length(Str) > 0) then
      begin
        Size := StrToInt(Str);
        Result := TRUE;
      end;
      Str := ExtractHtmlValue(MyHtml,'face=');
      if (Length(Str) > 0) then
      begin
        Face := Str;
        Result := TRUE;
      end;
      Str := ExtractHtmlValue(MyHtml,'color=');
      if (Length(Str) > 0) then
      begin
        if (Str[1] = '#') then
        begin
          Str[1] := '$';
          Color := TColor(StrToInt(Str));
          Result := TRUE;
        end
        else
        begin
          if (IdentToColor(Str, i)) then
          begin
            Color := TColor(i);
            Result := TRUE;
          end
          else if (IdentToColor('cl' + Str, i)) then
          begin
            Color := TColor(i);
            Result := TRUE;
          end;
        end;
      end;
    end;
  end; {<font}

  Text := '';
  if (Result) then
  begin // have to extract plain text
    InTag := FALSE;
    for i := 1 to Length(Html) do
    begin
      if InTag then
      begin
        if (Html[i] = '>') then // tag ends
          InTag := FALSE;
      end
      else
      begin
        if (Html[i] = '<') then
          InTag := TRUE
         else // not in tag, not start of one:
          Text := Text + Html[i];
      end;
    end;
  end
   else // just plain text
    Text := Html;
end;


{------------------------------------------------------------------------------
     Function: CleanString
  Description: removes offending characters from a string
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: menu manipulation
 Return Value: the cleaned string
 Known Issues:
 ------------------------------------------------------------------------------}
function CleanString(AString: String; TheChar: Char): String;
var
  i: Integer;
begin
  Result := '';
  for i := 1 to Length(AString) do
  begin
    if (AString[i] <> TheChar) then
    begin
      Result := Result + AString[i];
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: StrRev
  Description: reverses a string
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: string manipulation
 Return Value: the reverse of a string
 Known Issues:
 ------------------------------------------------------------------------------}
function StrRev(TheStr: String): String;
var
  i,
  l: Integer;
  RevStr: String;
begin
  l := Length(TheStr);
  SetLength(RevStr, l);

  for i := 1 to l do
  begin
    RevStr[i] := TheStr[l-i+1];
  end;
  StrRev := RevStr;
end;

{------------------------------------------------------------------------------
    Procedure: DeSci
  Description: breaks a number up into its Mantissa and Exponent
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Tick and Label scaling
 Known Issues: Why not use Math.Frexp() - because that works on POWERS OF TWO !
 ------------------------------------------------------------------------------}
procedure DeSci(ExtNumber: Extended; var Mantissa: Extended; var Exponent: Integer);
var
  TheLog: Extended;
  TheSign: Extended;
begin
  TheSign := 1;

  if (ExtNumber = 0) then
  begin
    Mantissa := 0;
    Exponent := 0;
    exit;
  end;

  if (ExtNumber < 0) then
  begin
    TheSign := -1;
    ExtNumber := -ExtNumber;
  end;

  TheLog := Log10(ExtNumber);
  Exponent := Floor(TheLog);
  Mantissa := TheLog - Exponent;
  Mantissa := Power(10.0, Mantissa);
  if (TheSign < 0) then Mantissa := -Mantissa;
end;

procedure Wait(mSeconds: Integer; ProcessMessages: Boolean);
var
  StartTime: TDateTime;
begin
  StartTime := Now;
  Screen.Cursor := crHourGlass;
  while (Now < (StartTime + mSeconds / (1000 * 3600 * 24))) do
  begin
    if (ProcessMessages) then
      Application.ProcessMessages;
  end;
  Screen.Cursor := crDefault;
end;

function GetAngleDeg(Xi, Yi: Integer): Extended;
begin
  Result := DEGS_PER_RAD * GetAngle(Xi, Yi);
end;

function GetAngle(Xi, Yi: Integer): Extended;
begin
  if (Yi = 0) then
  begin
    if (Xi > 0) then
      Result := PI_ON_2
     else
      Result := THREE_PI_ON_2;
  end
  else
  begin
    if (Xi > 0) then
    begin
      if (Yi < 0) then {top-right quadrant}
        Result := ArcTan(-Xi/Yi)
      else {bottom-right}
        Result := Pi - ArcTan(Xi/Yi);
    end
    else
    begin {X < 0}
      if (Yi > 0) then {bottom-left}
        Result := Pi + ArcTan(-Xi/Yi)
      else {top-left}
        Result := TWO_PI - ArcTan(Xi/Yi);
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: GetWord
  Description: splits a phrase into two at the delimiter
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: string manipulation
 Return Value: the left side
 Known Issues:
 ------------------------------------------------------------------------------}
Function GetWord(var ALine: String; Delimiter: String): String;
var
  Delimiter_Position: Integer;
begin
  Delimiter_Position := Pos(Delimiter, ALine);
  if (Delimiter_Position > 0) Then
  begin
    Result := Copy(ALine, 1, Delimiter_Position-1);
    ALine := Copy(ALine, Delimiter_Position + Length(Delimiter), Length(ALine));
  end
  else
  begin
    Result := ALine;
    ALine := '';
  end;
end;

{$IFDEF DELPHI1}
function TrimLeft(Const Str: String): String;
var
  len: Byte absolute Str;
  i: Integer;
begin
  i := 1;
  while (i <= len) and (Str[i] = ' ') do Inc(i);
  Result := Copy(Str,i,len)
end {TrimLeft};

procedure SetLength(var Str: String; len: Integer);
begin
  Str := Copy(Str, 1, len);
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: GetNextToken
  Description: gets the next whitespace-delimited cluster of characters
       Author: Mat Ballard
 Date created: 10/25/2001
Date modified: 10/25/2001 by Mat Ballard
      Purpose: string manipulation
 Return Value: the next token, plus the residual
 Known Issues:
 ------------------------------------------------------------------------------}
function GetNextToken(var ALine: String): String;
var
  i,
  TokenStart,
  TokenFinish: Integer;
begin
  TokenStart := 0;
  for i := 1 to Length(ALine) do
  begin
    if ((ALine[i] = ' ') or
        (ALine[i] = #9)) then { TAB }
      Continue;
    TokenStart := i;
    break;
  end;

  if (TokenStart > 0) then
  begin
    TokenFinish := Length(ALine);
    for i := TokenStart to Length(ALine) do
    begin
      if ((ALine[i] = ' ') or
          (ALine[i] = #9)) then { TAB }
      begin
        TokenFinish := i-1;
        break;
      end;
    end;
    Result := Copy(ALine, TokenStart, TokenFinish-TokenStart+1);
    if (TokenFinish < Length(ALine)) then
      ALine := TrimLeft(Copy(ALine, TokenFinish+1, Length(ALine)))
     else
      ALine := '';
  end
  else
  begin
    Result := ALine;
    ALine := '';
  end;
end;

{------------------------------------------------------------------------------
     Function: GetNextTokenAsym
  Description: gets the next delimited cluster of characters
       Author: Mat Ballard
 Date created: 10/25/2001
Date modified: 10/25/2001 by Mat Ballard
      Purpose: string manipulation
 Return Value: the next token, plus the residual
 Known Issues:
 ------------------------------------------------------------------------------}
function GetNextTokenAsym(var ALine: String; var Complete: Boolean; Delim1, Delim2: Char): String;
var
  i,
  DelimCount,
  TokenStart,
  TokenFinish: Integer;
begin
  DelimCount := 0;
  Complete := FALSE;
  TokenStart := 0;
  TokenFinish := 0;

  for i := 1 to Length(ALine) do
  begin
    if (ALine[i] = Delim1) then
    begin
      Inc(DelimCount);
      if (TokenStart = 0) then
        TokenStart := i;
    end;
    if (DelimCount > 0) then
    begin
      if (ALine[i] = Delim2) then
      begin
        Dec(DelimCount);
        if (DelimCount = 0) then
        begin
          TokenFinish := i;
          Complete := TRUE;
          break;
        end;
      end;
    end;
  end;

  if (TokenStart > 0) then
  begin
    if (Complete) then
    begin
{we strip off the delimiters:}    
      Result := Copy(ALine, TokenStart+1, TokenFinish-TokenStart-1);
      if (TokenFinish < Length(ALine)) then
        ALine := Copy(ALine, TokenFinish+1, Length(ALine))
       else
        ALine := '';
    end
    else
    begin
      Result := Copy(ALine, TokenStart, Length(ALine));
      ALine := '';
    end;
  end;
end;

function StripTags(ALine: String): String;
var
  i: Integer;
  InTag: Boolean;
begin
  Result := '';
  if (Length(ALine) > 0) then
  begin
    InTag := (ALine[1] = '<');
    for i := 1 to Length(ALine) do
    begin
      if (ALine[i] = '<') then InTag := TRUE
      else if (ALine[i] = '>') then InTag := FALSE
      else if (not InTag) then Result := Result + ALine[i];
    end;
  end;
end;

function TrimEx(ALine: String): String;
var
  i, j: Integer;
begin
  j := 0;
  for i := 1 to Length(ALine) do
  begin
    case Ord(ALine[i]) of
      0..47, 58..64, 91..96, 123..255: Inc(j); { delete all except numbers and letters}
    else
      break;
    end;
  end;
  if (j > 0) then
    System.Delete(ALine, 1, j);
  j := 0;
  for i := Length(ALine) downto 1 do
  begin
    case Ord(ALine[i]) of
      0..47, 58..64, 91..96, 123..255: Inc(j);
    else
      break;
    end;
  end;
  if (j > 0) then
    SetLength(ALine, Length(ALine)-j);
  Result := ALine;
end;

{------------------------------------------------------------------------------
     Function: IndexOfColorValue
  Description: gets the index of a color
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: color manipulation
 Return Value: Index of a color
 Known Issues:
 ------------------------------------------------------------------------------}
function IndexOfColorValue(Value: TColor): Integer;
var
  i: Integer;
begin
  IndexOfColorValue := -1;
  for i := 0 to MY_COLORS_MAX do
  begin
    if (MyColorValues[i] = Value) then
    begin
      IndexOfColorValue := i;
      break;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: IndexOfColorName
  Description: gets the name of a color
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: color manipulation
 Return Value: string containing the color name
 Known Issues:
 ------------------------------------------------------------------------------}
function IndexOfColorName(Name: String): Integer;
var
  i: Integer;
begin
  IndexOfColorName := -1;
  for i := 0 to MY_COLORS_MAX do
  begin
    if (ColorToString(MyColorValues[i]) = Name) then
    begin
      IndexOfColorName := i;
      break;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: GetPalerColor
  Description: gets a paler shade of the input color
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: color manipulation
 Return Value: TColor
 Known Issues:
 ------------------------------------------------------------------------------}
function GetPalerColor(Value: TColor; Brightness: Integer): TColor;
var
  iColor,
  iRed,
  iBlue,
  iGreen: Longint;
begin
  iColor := ColorToRGB(Value);

  iRed := (iColor and $000000FF);
  iRed := iRed + Brightness * ($FF-iRed) div 100;

  iGreen := (iColor and $0000FF00) shr 8;
  iGreen := iGreen + Brightness * ($FF-iGreen) div 100;

  iBlue := (iColor and $00FF0000) shr 16;
  iBlue := iBlue + Brightness * ($FF-iBlue) div 100;

  GetPalerColor := TColor(iRed  or (iGreen shl 8) or (iBlue shl 16));
end;

{------------------------------------------------------------------------------
     Function: GetColorMix
  Description: blends two colors to make a third; good for gradient effects
       Author: Mat Ballard
 Date created: 07/25/2002
Date modified: 07/25/2002 by Mat Ballard
      Purpose: color manipulation
 Return Value: TColor
 Known Issues:
 ------------------------------------------------------------------------------}
function GetColorMix(Color1, Color2: TColor; Fraction: Single): TColor;
var
  iRed1, iRed2,
  iBlue1, iBlue2,
  iGreen1, iGreen2: Longint;
begin
  iRed1 := (Color1 and $000000FF);
  iRed2 := (Color2 and $000000FF);
  iRed1 := Round((1-Fraction)*iRed1 + Fraction*iRed2);
  iGreen1 := (Color1 and $0000FF00) shr 8;
  iGreen2 := (Color2 and $0000FF00) shr 8;
  iGreen1 := Round((1-Fraction)*iGreen1 + Fraction*iGreen2);
  iBlue1 := (Color1 and $00FF0000) shr 16;
  iBlue2 := (Color2 and $00FF0000) shr 16;
  iBlue1 := Round((1-Fraction)*iBlue1 + Fraction*iBlue2);
  Result := TColor(iRed1 or (iGreen1 shl 8) or (iBlue1 shl 16));
end;

{------------------------------------------------------------------------------
     Function: GetDarkerColor
  Description: gets a darker shade of the input color
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: color manipulation
 Return Value: TColor
 Known Issues:
 ------------------------------------------------------------------------------}
function GetDarkerColor(Value: TColor; Brightness: Integer): TColor;
var
  iColor,
  iRed,
  iBlue,
  iGreen: Longint;
begin
  iColor := ColorToRGB(Value);

  iRed := (iColor and $000000FF);
  iRed := iRed * Brightness div 100;

  iGreen := (iColor and $0000FF00) shr 8;
  iGreen := iGreen * Brightness div 100;

  iBlue := (iColor and $00FF0000) shr 16;
  iBlue := iBlue * Brightness div 100;

  Result := TColor(iRed  or (iGreen shl 8) or (iBlue shl 16));
end;

{------------------------------------------------------------------------------
     Function: GetInverseColor
  Description: gets the inverse of the input color
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: color manipulation
 Return Value: TColor
 Known Issues: does not return an inverse if Value is close to grey, because the
               inverse of gray is gray !
 ------------------------------------------------------------------------------}
function GetInverseColor(Value: TColor): TColor;
var
  iColor,
  iRed,
  iBlue,
  iGreen,
  Difference: Longint;
begin
  iColor := ColorToRGB(Value);

  iRed := (iColor and $000000FF);
  iRed := 255 - iRed;

  iGreen := (iColor and $0000FF00) shr 8;
  iGreen := 255 - iGreen;

  iBlue := (iColor and $00FF0000) shr 16;
  iBlue := 255 - iBlue;

  Difference := Abs(255 - (2*iRed + 2*iGreen + 2*iBlue) div 3);

  if (Difference > 26) then
    Result := TColor(iRed  or (iGreen shl 8) or (iBlue shl 16))
   else
    Result := clBlack;
end;

{------------------------------------------------------------------------------
     Function: Rainbow
  Description: returns a rainbow color, depending on the Fraction
       Author: Mat Ballard
 Date created: 02/15/2001
Date modified: 02/15/2001 by Mat Ballard
      Purpose: color manipulation for contour graphs
 Return Value: TColor
 Known Issues:
 ------------------------------------------------------------------------------}
function RainbowOld(Fraction: Single): TColor;
var
  i,
  LowIndex,
  HighIndex: Integer;
  RainbowColor: array [0..2] of Integer;
  HighFraction,
  LowFraction,
  CellWidth: Single;
begin
  CellWidth := 1 / MAX_RAINBOW_COLORS;
  LowIndex := Trunc(Fraction / CellWidth);
  HighIndex := LowIndex + 1;
  HighFraction := (Fraction - LowIndex * CellWidth) / CellWidth;
  LowFraction := 1.0 - HighFraction;

  if (LowIndex = MAX_RAINBOW_COLORS) then
  begin
    for i := 0 to 2 do
      RainbowColor[i] := 255;
  end
  else
  begin
    for i := 0 to 2 do
    RainbowColor[i] := Round(
      LowFraction * RainbowColors[LowIndex, i] +
      HighFraction * RainbowColors[HighIndex, i]);
  end;
  Result := TColor(
    RainbowColor[0] +
    RainbowColor[1] shl 8 +
    RainbowColor[2] shl 16);
end;

{------------------------------------------------------------------------------
     Function: Rainbow
  Description: returns a spectral color, depending on the Fraction
       Author: Mat Ballard
 Date created: 01/07/2002
Date modified: 01/07/2001 by Mat Ballard
      Purpose: color manipulation for contour graphs
 Return Value: TColor
 Known Issues:
 ------------------------------------------------------------------------------}
function Rainbow(Fraction, Gamma, LambdaMin, LambdaSpan: Single): TColor;
begin
  Result := Spectrum(LambdaMin + LambdaSpan * Fraction, Gamma);
end;

{------------------------------------------------------------------------------
     Function: Spectrum
  Description: returns a spectral color, depending on the wavelength Lambda
       Author: Mat Ballard
 Date created: 01/07/2002
Date modified: 01/07/2002 by Mat Ballard
      Purpose: color manipulation for contour graphs
 Return Value: TColor
 Known Issues:
     Comments: adapted from Dan Bruton's fortran code at:
               http://www.physics.sfasu.edu/astro/color.html
 ------------------------------------------------------------------------------}
function Spectrum(Lambda, Gamma: Single): TColor;
const
  {GAMMA = 0.80;}
  INTENSITY_MAX = 255;
var
  Red: Single;
  Green: Single;
  Blue: Single;
  factor: Single;

  function Adjust(Color, Factor:  Single):  Integer;
  begin
   if (Color = 0.0) then
     Result := 0
    else
     Result := Round(INTENSITY_MAX * Power(Color * Factor, Gamma))
  end;

begin
  case Trunc(Lambda) of
    380..439:
      begin
        Red   := -(Lambda - 440) / (440 - 380);
        Green := 0.0;
        Blue  := 1.0
      end;
    440..489:
      begin
       Red   := 0.0;
       Green := (Lambda - 440) / (490 - 440);
       Blue  := 1.0
      end;
    490..509:
      begin
       Red   := 0.0;
       Green := 1.0;
       Blue  := -(Lambda - 510) / (510 - 490)
      end;
    510..579:
      begin
       Red   := (Lambda - 510) / (580 - 510);
       Green := 1.0;
       Blue  := 0.0
      end;
    580..644:
      begin
       Red   := 1.0;
       Green := -(Lambda - 645) / (645 - 580);
       Blue  := 0.0
      end;
    645..780:
      begin
       Red   := 1.0;
       Green := 0.0;
       Blue  := 0.0
      end;
  else
    Red   := 0.0;
    Green := 0.0;
    Blue  := 0.0
  end;

{reduce intensity near the vision limits:}
  case Trunc(Lambda) of
    380..419: factor := 0.3 + 0.7*(Lambda - 380) / (420 - 380);
    420..700: factor := 1.0;
    701..780: factor := 0.3 + 0.7*(780 - Lambda) / (780 - 700)
    else factor := 0.0;
  end;

  {R := Adjust(Red,   Factor);
  G := Adjust(Green, Factor);
  B := Adjust(Blue,  Factor);}

  Result := Adjust(Red,   Factor) +
    (Adjust(Green, Factor) shl 8) +
    (Adjust(Blue,  Factor) shl 16);
end;

{------------------------------------------------------------------------------
     Function: InputColor
  Description: prompts the user for a color
       Author: Mat Ballard
 Date created: 01/15/2001
Date modified: 01/15/2001 by Mat Ballard
      Purpose: color management
 Return Value: Boolean
 Known Issues:
 ------------------------------------------------------------------------------}
function InputColor(var AColor: TColor): Boolean;
var
  ColorDialog: TColorDialog;
begin
  InputColor := FALSE;
  ColorDialog := TColorDialog.Create(nil);
{$IFDEF MSWINDOWS}
  ColorDialog.Options := [cdFullOpen];
{$ENDIF}
  ColorDialog.Color := AColor;
  ColorDialog.CustomColors.Add('Current=' + IntToHex(ColorToRGB(AColor), 6));

  if (ColorDialog.Execute) then
  begin
    AColor := ColorDialog.Color;
    InputColor := TRUE;
  end;
  ColorDialog.Free;
end;

function ColorToHTML(AColor: TColor): String;
var
  R, G, B: Integer;
  Color: Longint;
begin
  Color := ColorToRGB(AColor);
  R := Color and $FF;
  G := (Color and $FF00) shr 8;
  B := (Color and $FF0000) shr 16;
  Result := Format('#%2.2x%2.2x%2.2x', [R, G, B]);
end;

procedure MessageState(AState: TComponentState);
var
  Msg: String;
begin
  Msg := '';
  if (csAncestor in AState) then Msg := Msg + 'csAncestor, ';
  if (csDesigning in AState) then Msg := Msg + 'csDesigning, ';
  if (csDestroying in AState) then Msg := Msg + 'csDestroying, ';
  if (csFixups in AState) then Msg := Msg + 'csFixups, ';
  if (csLoading in AState) then Msg := Msg + 'csLoading, ';
  if (csReading in AState) then Msg := Msg + 'csReading, ';
  if (csUpdating in AState) then Msg := Msg + 'csUpdating, ';
  if (csWriting in AState) then Msg := Msg + 'csWriting.';
{$IFDEF WINDOWS}
  MessageBox(0, PChar(Msg), 'ComponentState', MB_OK);
{$ENDIF}
{$IFDEF LINUX}
  ShowMessage(Msg);
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: SetDialogGeometry
  Description: sets the dialog Geometry under Windows and Linux
      Authors: Mat Ballard
 Date created: 04/03/2001
Date modified: 04/03/2001 by Mat Ballard
      Purpose: Dialog Geometry control
 Known Issues: an alternative approach is:

  Scaled := FALSE;
  AutoScroll := FALSE;
// DESIGNSCREENWIDTHPIX is a constant depending on the width at design time, eg: 1024
  ScaleBy(Screen.Width, DESIGNSCREENWIDTHPIX);
 ------------------------------------------------------------------------------}
procedure SetDialogGeometry(AForm: TForm; AButton: TControl; BorderWidth: Integer);
var
  i: Integer;
begin
{$IFDEF MSWINDOWS}
  {AForm.PixelsPerInch := 96;}
  AForm.BorderStyle := bsDialog;
{$ENDIF}
{$IFDEF LINUX}
  {AForm.PixelsPerInch := 75;}
  AForm.BorderStyle := fbsDialog;
{$ENDIF}
  AForm.Scaled := FALSE;
  AForm.HorzScrollBar.Visible := FALSE;
  AForm.VertScrollBar.Visible := FALSE;

  AForm.Left := 10;
  AForm.Top := 10;
  AForm.ClientHeight := AButton.Top + 3 * AButton.Height div 2;
  AForm.ClientWidth := AButton.Left + AButton.Width + BorderWidth;
{set combo and edit box widths:}
  for i := 0 to AForm.ComponentCount - 1 do
    if ((AForm.Components[i] is TNEdit) or
        (AForm.Components[i] is TColorEdit) {$IFDEF DELPHI2_UP} or
        (AForm.Components[i] is TComboBox) {$ENDIF} ) then
      TControl(AForm.Components[i]).Width := 72; {88}
  for i := 0 to AForm.ComponentCount - 1 do
    if (AForm.Components[i] is TBitBtn) then
    begin
      TControl(AForm.Components[i]).Width := 80; {97}
      TControl(AForm.Components[i]).Height := 24;
    end;
end;


{------------------------------------------------------------------------------}
function BinToInt(Value: String): {$IFDEF DELPHI1}Longint{$ELSE}{$IFDEF BCB}Longint{$ELSE}Longword{$ENDIF}{$ENDIF};
var
  i: Integer;
  Pow2,
  TheResult: {$IFDEF DELPHI1}Longint{$ELSE}{$IFDEF BCB}Longint{$ELSE}Longword{$ENDIF}{$ENDIF};
begin
  Pow2 := 1;
  TheResult := 0;
  for i := 1 to Length(Value) do
  begin
    if (Value[i] = '1') then
      TheResult := TheResult + Pow2;
    Pow2 := Pow2 shl 1;
  end;
  BinToInt := TheResult;
end;

function IntToBin(Value: {$IFDEF DELPHI1}Longint{$ELSE}{$IFDEF BCB}Longint{$ELSE}Longword{$ENDIF}{$ENDIF}): string;
var
  i: Integer;
  StrResult: String;

{$IFDEF DELPHI1}
  function LTrim(Const Str: String): String;
  var
    len: Byte absolute Str;
    i: Integer;
  begin
    i := 1;
    while (i <= len) and (Str[i] = ' ') do Inc(i);
    LTrim := Copy(Str,i,len)
  end ;
{$ENDIF}

begin
  i := 1;
  SetLength(StrResult, 32);
  repeat
    if ((Value and 1) > 0) then
      StrResult[i] := '1'
     else
      StrResult[i] := '0';
    Value := Value shr 1;
    Inc(i);
  until (Value = 0);
  SetLength(StrResult, i-1);
  StrResult := StrRev(StrResult);

  IntToBin := StrResult;
end;

{------------------------------------------------------------------------------
     Function: LongintToRoman
  Description: converts from decimal to Roman numbers
       Author: Dr J R Stockton, from WR-ROMAN.PAS
 Date created:
Date modified: 5/01/2003 by Mat Ballard
      Purpose: coping with RTF
 Return Value: string in Roman numeral format
 Known Issues:
 ------------------------------------------------------------------------------}
function LongintToRoman(const N : longint) : string;
const
  Digits = 7;
  //Four = 4;
  Pattern : array ['0'..'9'] of string [4] =
    ('', 'x', 'xx', 'xxx', 'xy', 'y', 'yx', 'yxx', 'yxxx', 'xz');
  DecXlatn : array [0 .. Digits-1] of array ['x'..'z'] of char =
    ('IVX', 'XLC', 'CDM', 'Mvx', 'xlc', 'cdm', 'm?!' {, ...});
var
  DecDigit, Ch : byte; S : string [Succ(Digits)];
  PartRomn : string [4];
begin
  Result := '';
  if (N < 0) then
  begin
    Result := 'Negative';
  end
  else
  begin
    Str(N, S);
    if (Length(S) > Digits) then
    begin
      Result := 'Too Big';
    end
    else
    begin
      for DecDigit := 1 to Length(S) do
      begin
        PartRomn := Pattern[S[DecDigit]];
        for Ch := 1 to Length(PartRomn) do
          PartRomn[Ch] := DecXlatn[Length(S)-DecDigit][PartRomn[Ch]];
        Result := Result + PartRomn;
      end; {DecDigit}
    end;
  end;
end; {LongintToRoman}

{------------------------------------------------------------------------------
     Function: RomanToLongint
  Description: converts from decimal to Roman numbers
       Author: Dr J R Stockton, from WR-ROMAN.PAS
 Date created:
Date modified: 5/01/2003 by Mat Ballard
      Purpose: coping with RTF
 Return Value: Longint
 Known Issues:
 ------------------------------------------------------------------------------}
function RomanToLongint(const S : string) : longint;
const
  Chars = 'IVXLCDMvxlcdm?!';
  Valu : array [0..Length(Chars)] of longint =
    (0, 1, 5, 10, 50, 100, 500, 1000, 5000, 10000, 50000, 100000,
     500000, 1000000, 5000000, 10000000);
var
  Sum, NewV, OldV : longint;
  K : byte;
begin
  Sum := 0;
  OldV := 0;
  for K := Length(S) downto 1 do
  begin
    NewV := Valu[Pos(S[K], Chars)];
    if NewV=0 then
      EMathError.CreateFmt('%s is not a valid Roman Numeral !', [S]);
    if (NewV < OldV) then
      NewV := - NewV;
    Inc(Sum, NewV);
    OldV := NewV;
  end;
  Result := Sum;
end; {RomanToLongint}

function IsInteger(Value: String): Boolean;
var
  i: Integer;
  TheStart: Integer;
begin
  Result := FALSE;
  if (Length(Value) = 0) then exit;

  TheStart := 1;
  if (Value[1] = '-') then
  begin
    if (Length(Value) = 1) then exit;
    TheStart := 2;
  end;

  for i := TheStart to Length(Value) do
  begin
    if ((Value[i] < '0') or
        (Value[i] > '9')) then
      exit;
  end;
  Result := TRUE;
end;

{From http://homepages.borland.com/efg2lab/Mathematics/NaN.htm#QuietAndSignaling
    Dr. Chris Rorden
    Department of Psychology
    University of Nottingham
    Nottingham NG7 2RD, UK
    http://www.psychology.nottingham.ac.uk/staff/cr1/}
function IsNAN(s:single): boolean;
{ returns true if s is Infinity, NAN or Indeterminate
  4byte IEEE: msb[31] = signbit, bits[23-30] exponent,
  bits[0..22] mantissa
  exponent of all 1s = Infinity, NAN or Indeterminate }
const
  kSpecialExponent = 255 shl 23;
var
  Ptr: Pointer;
{$IFDEF COMPILER4_UP}
  iPtr: ^LongWord;
{$ELSE}
  iPtr: ^LongInt;
{$ENDIF}
begin
  Ptr := @s;
  iPtr := Ptr;
  if ((iPtr^ and kSpecialExponent) = kSpecialExponent) then
    RESULT := TRUE
   else
    RESULT := FALSE;
end;

procedure MakeNAN(var s:single);
{ returns true if s is Infinity, NAN or Indeterminate
  4byte IEEE: msb[31] = signbit, bits[23-30] exponent,
  bits[0..22] mantissa
  exponent of all 1s = Infinity, NAN or Indeterminate}
const
  kSpecialExponent = 255 shl 23;
var
  Ptr: Pointer;
{$IFDEF COMPILER4_UP}
  iPtr: ^LongWord;
{$ELSE}
  iPtr: ^LongInt;
{$ENDIF}
begin
  Ptr := @s;
  iPtr := Ptr;
  iPtr^ := kSpecialExponent;
end;

function IsFixed(Value: String): Boolean;
var
  i: Integer;
begin
  Result := FALSE;

  for i := 1 to Length(Value) do
  begin
    if ((Value[i] < '0') or
        (Value[i] > '9')) then
    begin
      if ((Value[i] = '.') or (Value[i] = '-') or (Value[i] = ' ')) then
        Continue;
      Exit;
    end;
  end;
  Result := IsReal(Value);
end;

function IsReal(Value: String): Boolean;
var
  TheResult: Boolean;
begin
  TheResult := FALSE;
  try
    StrToFloat(Value);
    TheResult := TRUE;
  finally
    Result := TheResult;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: LoadResourceFile
  Description: Loads a binary or text file from resources into a memory stream
      Authors: Mat Ballard
 Date created: 04/01/2002
Date modified: 04/01/2002 by Mat Ballard
      Purpose: Resource Management
 Known Issues:
     Comments: after work by Jan Verhoeven at:
               http://www.jansfreeware.com/articles/delphiresource.html
 ------------------------------------------------------------------------------}
procedure LoadResourceFile(aFile:string; var MemoryStream:TMemoryStream);
var
  HResInfo: {$IFDEF DELPHI1}Integer{$ELSE}HRSRC{$ENDIF};
  HGlobal: THandle;
  Buffer: pchar;
  Ext:string;
{$IFDEF DELPHI1}
  pFile, pExt: array [0..255] of char;
{$ENDIF}
begin
  if (MemoryStream = nil) then
    MemoryStream := TMemoryStream.Create;
  ext:=Uppercase(ExtractFileExt(aFile));
  ext:=Copy(ext, 2, 99);
  aFile := ExtractFileRoot(aFile);
{$IFDEF DELPHI1}
  StrPCopy(pFile, aFile);
  StrPCopy(pExt, Ext);
  HResInfo := FindResource(HInstance, pFile, pExt);
{$ELSE}
  HResInfo := FindResource(HInstance, PChar(aFile), PChar(ext));
{$ENDIF}
  HGlobal := LoadResource(HInstance, HResInfo);
  if HGlobal = 0 then
    raise EResNotFound.Create('Cannot load resource: ' + aFile);
  Buffer := LockResource(HGlobal);
  MemoryStream.Clear;
  MemoryStream.WriteBuffer(Buffer[0], SizeOfResource(HInstance, HResInfo));
  MemoryStream.Seek(0,0);
  UnlockResource(HGlobal);
  FreeResource(HGlobal);
end;

{$IFDEF DELPHI1}
function GetCurrentDir: String;
var
  ThisDir: String;
begin
  GetDir(0, ThisDir);
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TextOutAnglePersist
  Description: draws text on the input canvas, at an angle, and leaves the rotated font intact
 Date created: 02/15/2001
Date modified: 10/30/2002 by Mat Ballard
      Purpose: Vertical and angular fonts
 Known Issues: ACanvas.Font remains rotated until cleaned up
               Requires a subsequent call to TextOutAnglePersistCleanUp
 ------------------------------------------------------------------------------}
function TextOutAnglePersist(
  ACanvas: TCanvas;
  var OldFontHandle: hFont;
  Angle, X, Y: Integer;
  TheText: String): hFont;
{$IFDEF MSWINDOWS}
var
  LogRec: TLogFont;
{$ENDIF}
begin
{$IFDEF MSWINDOWS}
{Gotta use Windows GDI functions to rotate the font:}
  GetObject(ACanvas.Font.Handle, SizeOf(LogRec), Addr(LogRec));
  LogRec.lfEscapement := 10*Angle;
  LogRec.lfOrientation := LogRec.lfEscapement;
  Result := Windows.CreateFontIndirect(LogRec); // NewFontHandle
{select the new font:}
  OldFontHandle := Windows.SelectObject(ACanvas.Handle, Result);
{Print the text:}
  ACanvas.TextOut(X, Y, TheText);
(*{when you are finished with this rotated font, you must go back to original font:}
  NewFontHandle := Windows.SelectObject(ACanvas.Handle, OldFontHandle);
{and delete the new one:}
  DeleteObject(NewFontHandle);*)
{$ENDIF}

{$IFDEF LINUX}
{this code is courtesy of Jon Shemitz <jon@midnightbeach.com>}
{Outside of a Paint handler, bracket QPainter_ calls with a Start/Stop}
  ACanvas.Start;
  try
    OldFontHandle := ACanvas.Font.Handle;
    Qt.QPainter_save(ACanvas.Handle);
{Move 0,0 to the center of the form}
    Qt.QPainter_translate(ACanvas.Handle, X, Y);
{Rotate; note negative angle:}
    QPainter_rotate(ACanvas.Handle, -Angle);
    ACanvas.TextOut(0, 0, TheText);
  finally
(*{when you are finished with this rotated font, you must
    Qt.QPainter_restore(ACanvas.Handle);*)
    Result := ACanvas.Font.Handle;
    ACanvas.Stop;
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TextOutAnglePersistCleanUp
  Description: cleans up after a persistent rotated
 Date created: 10/30/2002
Date modified: 10/30/2002 by Mat Ballard
      Purpose: Vertical and angular font management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TextOutAnglePersistCleanUp(
  ACanvas: TCanvas;
  NewFontHandle, OldFontHandle: hFont);
begin
{$IFDEF GUI}
  {$IFDEF MSWINDOWS}
{go back to original font:}
  NewFontHandle := Windows.SelectObject(ACanvas.Handle, OldFontHandle);
{and delete the old one:}
  Windows.DeleteObject(NewFontHandle);
  {$ENDIF}
  {$IFDEF LINUX}
  ACanvas.Start;
  try
    Qt.QPainter_restore(ACanvas.Handle);
  finally
    ACanvas.Stop;
  end;
  {$ENDIF}
{$ENDIF} {GUI}
end;

{------------------------------------------------------------------------------
    Procedure: TextOutAngle
  Description: draws angled text on the input canvas
      Authors: Mat Ballard
 Date created: 04/15/2000
Date modified: 04/15/2000 by Mat Ballard
      Purpose: Vertical fonts
 Known Issues: derived from the very early GPC work;
               ACanvas.Font does not remain rotated
               Note: Angle of rotation is Anti-Clockwise in Win32,
               Clockwise in Qt/Linux
 ------------------------------------------------------------------------------}
procedure TextOutAngle(
  ACanvas: TCanvas;
  Angle, Left, Top: Integer;
  TheText: String);
{$IFDEF MSWINDOWS}
var
  LogRec: TLogFont;
  OldFontHandle, NewFontHandle: hFont;
{$ENDIF}
begin
{$IFDEF MSWINDOWS}
{Gotta use Windows GDI functions to rotate the font:}
  GetObject(ACanvas.Font.Handle, SizeOf(LogRec), Addr(LogRec));
  LogRec.lfEscapement := 10*Angle;
  LogRec.lfOrientation := LogRec.lfEscapement;
  NewFontHandle := {Windows.}CreateFontIndirect(LogRec);
{select the new font:}
  OldFontHandle := {Windows.}SelectObject(ACanvas.Handle, NewFontHandle);
{Print the text:}
  ACanvas.TextOut(Left, Top, TheText);
{go back to original font:}
  NewFontHandle := {Windows.}SelectObject(ACanvas.Handle, OldFontHandle);
{and delete the old one:}
  DeleteObject(NewFontHandle);
{$ENDIF}

{$IFDEF LINUX}
{this code is courtesy of Jon Shemitz <jon@midnightbeach.com>}
{Outside of a Paint handler, bracket QPainter_ calls with a Start/Stop}
  ACanvas.Start;
  try
    Qt.QPainter_save(ACanvas.Handle);
{Move 0,0 to the center of the form}
    Qt.QPainter_translate(ACanvas.Handle, Left, Top);
{Rotate; note negative angle:}
    QPainter_rotate(ACanvas.Handle, -Angle);
    ACanvas.TextOut(0, 0, TheText);
  finally
    Qt.QPainter_restore(ACanvas.Handle);
    ACanvas.Stop;
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: ShellExec
  Description: wrapper for the windows "ShellExecute" API call, extended to Linux
      Authors: Mat Ballard
 Date created: 04/15/2000
Date modified: 03/28/2001 by Mat Ballard
      Purpose: Execute an external program with arguments
 Known Issues: does not cope properly with spaces in arguments (eg: "My File.txt")
 ------------------------------------------------------------------------------}
procedure ShellExec(Cmd: String);
{$IFDEF WINDOWS} {Delphi 1}
var
  sObjectPath: array[0..1023] of Char;
{$ENDIF}
begin
{$IFDEF WINDOWS} {Delphi 1}
  StrPCopy(sObjectPath, Cmd);
  ShellExecute(0, Nil, sObjectPath, Nil, Nil, 3);  {?SW_SHOW ?}
{$ENDIF}
{$IFDEF WIN32}
  ShellExecute(0, Nil, PChar(Cmd), Nil, Nil, SW_NORMAL);
{$ENDIF}
{$IFDEF LINUX}
{Fire command; add a ' &' to continue immediately:}
  Libc.system(PChar(Cmd));
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: ShowHTML
  Description: shows a html file
      Authors: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure ShowHTML(Cmd: String);
{$IFDEF WINDOWS} {Delphi 1}
var
  sObjectPath: array[0..1023] of Char;
{$ENDIF}
{$IFDEF LINUX}
var
  TheBrowser: String;
{$ENDIF}
begin
{$IFDEF WINDOWS} {Delphi 1}
  StrPCopy(sObjectPath, Cmd);
  ShellExecute(0, Nil, sObjectPath, Nil, Nil, 3);  {?SW_SHOW ?}
{$ENDIF}
{$IFDEF WIN32}
  ShellExecute(0, Nil, PChar(Cmd), Nil, Nil, SW_NORMAL);
{$ENDIF}
{$IFDEF LINUX}
  TheBrowser := GetBrowser;
{the ' &' means immediately continue:}
  if (Length(TheBrowser) > 0) then
    Libc.system(PChar(TheBrowser + ' ' + Cmd + ' &'));
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: DoMail
  Description: drops an email address to the users mail program
      Authors: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure DoMail(Cmd: String);
{$IFDEF WINDOWS} {Delphi 1}
var
  sObjectPath: array[0..1023] of Char;
{$ENDIF}
{$IFDEF LINUX}
var
  TheMailer: String;
{$ENDIF}
begin
{$IFDEF WINDOWS} {Delphi 1}
  StrPCopy(sObjectPath, Cmd);
  ShellExecute(0, Nil, sObjectPath, Nil, Nil, 3);  {?SW_SHOW ?}
{$ENDIF}
{$IFDEF WIN32}
  ShellExecute(0, Nil, PChar(Cmd), Nil, Nil, SW_NORMAL);
{$ENDIF}
{$IFDEF LINUX}
  TheMailer := GetMailer;
  if (Length(TheMailer) > 0) then
{the ' &' means immediately continue:}
    Libc.system(PChar(TheMailer + ' ' + Cmd + ' &'));
{$ENDIF}
end;

{------------------------------------------------------------------------------
     Function: CheckForRPM
  Description: checks for the existence of the AnRPM program
      Authors: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: help, mail and html management
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF LINUX}
function CheckForRPM(AnRPM: String): String;
var
  TmpFile: TStringList;
begin
  Result := '';
  TmpFile := TStringList.Create;
  Libc.system(PChar('rpm -ql ' + AnRPM + ' > ' + sFileName));
  TmpFile.LoadFromFile(sFileName);
  if (Length(TmpFile.Strings[0]) > 0) then
    if (Pos('not installed', TmpFile.Strings[0]) = 0) then
      Result := TmpFile.Strings[0];
  DeleteFile(sFileName);
  TmpFile.Free;
end;

{$ENDIF}


{------------------------------------------------------------------------------
     Function: GetBrowser
  Description: gets the user's prefered browser in Linux
      Authors: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: help and html management
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF LINUX}
function GetBrowser: String;
var
  Index: Integer;
  AProgram,
  ExeName: String;
  OptionsDlg: TOptionsDlg;
begin
{Get the $BROWSER environment variable:}
  ExeName := getenv('BROWSER');
  if (Length(ExeName) = 0) then
  begin
{Get the various possible browsers:}
    OptionsDlg := TOptionsDlg.Create(nil);
    OptionsDlg.FormTitle := 'Browser Selection';
    OptionsDlg.Question := 'Which Web Browser Program To Use ?';
    if (FileExists('/usr/bin/konqueror')) then
    begin
      OptionsDlg.OptionList.Add('/usr/bin/konqueror');
    end;
    AProgram := CheckForRPM('mozilla');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('netscape-common');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('opera');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('lynx');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('links');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);

    Index := OptionsDlg.Execute - 1;

    if (Index >= 0) then
    begin
      ExeName := OptionsDlg.OptionList.Strings[Index];
      Libc.putenv(PChar('BROWSER=' + ExeName));
    end;

    OptionsDlg.Free;
  end;
  Result := ExeName;
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: GetMailer
  Description: gets the user's prefered Mailer in Linux
      Authors: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: help and html management
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF LINUX}
function GetMailer: String;
var
  Index: Integer;
  AProgram,
  ExeName: String;
  OptionsDlg: TOptionsDlg;
begin
{Get the $MAILER environment variable:}
  ExeName := getenv('MAILER');
  if (Length(ExeName) = 0) then
  begin
{Get the various possible browsers:}
    OptionsDlg := TOptionsDlg.Create(nil);
    OptionsDlg.FormTitle := 'Mailer Selection';
    OptionsDlg.Question := 'Which Email Program To Use ?';
    AProgram := CheckForRPM('mozilla');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('netscape-common');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('mailx');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);
    AProgram := CheckForRPM('pine');
    if (Length(AProgram) > 0) then
      OptionsDlg.OptionList.Add(AProgram);

    Index := OptionsDlg.Execute - 1;

    if (Index >= 0) then
    begin
      ExeName := OptionsDlg.OptionList.Strings[Index];
      Libc.putenv(PChar('MAILER=' + ExeName));
    end;

    OptionsDlg.Free;
  end;
  Result := ExeName;
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: DoHTMLHelp
  Description: displays a topic from a HTML-based help website
       Author: Mat Ballard
 Date created: 05/10/2001
Date modified: 05/10/2001 by Mat Ballard
      Purpose: help management
 Return Value: Boolean
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF LINUX}
procedure DoHTMLHelp(
  HelpType: THelpType;
  HelpContext: Integer;
  HelpKeyword: string;
  HelpFile: string;
  var Handled: Boolean);
var
  MyHTMLHelpTopicFile: String;
  HelpPath: String;
  TheBrowser: String;
begin
  Handled := FALSE;
  HelpPath := ExtractFilePath(HelpFile);
  MyHTMLHelpTopicFile := HelpPath + 'hs' + IntToStr(HelpContext) + '.htm';
  if FileExists(MyHTMLHelpTopicFile) then
  begin
    TheBrowser := GetBrowser;
    if (Length(TheBrowser) > 0) then
    begin
{the ' &' means immediately continue:}
      ShellExec(TheBrowser + ' ' + MyHTMLHelpTopicFile + ' &');
      Handled := TRUE;
    end;
  end;
end;
{$ENDIF}

{$IFDEF WIN32}
procedure RunDosInMemo(DosApp:String; AMemo:TMemo);
const
  ReadBuffer = 2400;
var
  Security            : TSecurityAttributes;
  ReadPipe,WritePipe  : THandle;
  start               : TStartUpInfo;
  ProcessInfo         : TProcessInformation;
  Buffer              : Pchar;
  BytesRead           : DWord;
  Apprunning          : DWord;
  //ErrorCode           : DWord;
  lpMsgBuf            : array [0..1023] of Char;
begin
  With Security do begin
    nlength              := SizeOf(TSecurityAttributes);
    binherithandle       := true;
    lpsecuritydescriptor := nil;
  end;

  if Createpipe (ReadPipe, WritePipe, @Security, 0) then
  begin
    Buffer  := AllocMem(ReadBuffer + 1);
    FillChar(Start,Sizeof(Start),#0);
    start.cb          := SizeOf(start);
    start.hStdOutput  := WritePipe;
    start.hStdInput   := ReadPipe;
    start.dwFlags     := STARTF_USESTDHANDLES +
                         STARTF_USESHOWWINDOW;
    start.wShowWindow := SW_HIDE;

    if CreateProcess(nil,
          PChar(DosApp),
          @Security,
          @Security,
          true,
          NORMAL_PRIORITY_CLASS, //IDLE_
          nil,
          nil,
          start,
          ProcessInfo) then
    begin
      repeat
        Apprunning := WaitForSingleObject(ProcessInfo.hProcess,100);
        Application.ProcessMessages;
      until (Apprunning <> WAIT_TIMEOUT);
      repeat
        BytesRead := 0;
        ReadFile(ReadPipe,Buffer[0], ReadBuffer,BytesRead,nil);
        Buffer[BytesRead]:= #0;
        OemToAnsi(Buffer,Buffer);
        AMemo.Text := AMemo.text + String(Buffer);
      until (BytesRead < ReadBuffer);
    end
    else
    begin
      FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
        nil,
        GetLastError(),
        0, // Default language
        lpMsgBuf,
        1023,
        nil);
    end;
    FreeMem(Buffer);
  end;
  CloseHandle(ProcessInfo.hProcess);
  CloseHandle(ProcessInfo.hThread);
  CloseHandle(ReadPipe);
  CloseHandle(WritePipe);
end;
{$ENDIF}

{the opposite of ExtractRelativePath}
function CreatePathFromRelativePath(BasePath, RelPath: string): string;
var
  i: Integer;
begin
{remove trailing '\':}
  while (Pos('..' + DIR_SEPERATOR, RelPath) = 1) do
  begin
    RelPath := Copy(RelPath, 4, 9999);
{there might be a trailing '\':}
    i := Length(BasePath) - 1;
    while ((i > 0) and (BasePath[i] <> DIR_SEPERATOR)) do
    begin
      Dec(i);
    end;
    SetLength(BasePath, i);
  end; {while}
{Resolve seperators:}
  if (BasePath[Length(BasePath)] = DIR_SEPERATOR) then
  begin
    if (RelPath[1] = DIR_SEPERATOR) then
      SetLength(BasePath, Length(BasePath)-1);
  end
  else
  begin
    if (RelPath[1] <> DIR_SEPERATOR) then
      BasePath := BasePath + DIR_SEPERATOR;
  end;
  Result := BasePath + RelPath;
end;

function StripDir(APath: string): string;
var
  i: Integer;
begin
{remove up to trailing '\'}
  if (APath[Length(APath)] = DIR_SEPERATOR) then
  begin
    SetLength(APath, Length(APath)-1);
  end
  else if (Pos('.', ExtractFileName(APath)) > 0) then
  begin { is file}
    APath := ExtractFilePath(APath);
    SetLength(APath, Length(APath)-1);
  end;
  i := Length(APath);
  while (APath[i] <> DIR_SEPERATOR) do
    Dec(i);
  SetLength(APath, i);
  Result := APath;
end;

procedure DbgPrint(Msg: String);
const
  FILENAME = 'c:\temp\misc.dbg.log';
var
  F: TextFile;
begin
  AssignFile(F, FILENAME);
  Append(F);
  Writeln(F, Msg);
  Flush(F);  { ensures that the text was actually written to file }
  CloseFile(F);
end;

{$IFDEF WIN32}
function GetDNSName(var DNSName, IP, Msg: String): Boolean;
var
  i, NameLen: Integer;
  Name: array[0..255] of Char;
  WSAData: TWSAData;
  wVersionRequested: WORD;
  ptrHostEnt: PHostEnt;
{type
  PWSAData = ^TWSAData;
  TWSAData = packed record
    wVersion: Word;
    wHighVersion: Word;
    szDescription: array[0..WSADESCRIPTION_LEN] of Char;
    szSystemStatus: array[0..WSASYS_STATUS_LEN] of Char;
    iMaxSockets: Word;
    iMaxUdpDg: Word;
    lpVendorInfo: PChar;
  end;

type
  PHostEnt = ^THostEnt;
  THostEnt = packed record
    h_name: PChar;
    h_aliases: ^PChar;
    h_addrtype: Smallint;
    h_length: Smallint;
    case Byte of
      0: (h_addr_list: ^PChar);
      1: (h_addr: ^PChar)
  end;
}
begin
  Result := FALSE;
  wVersionRequested := 2 + 256*0;
  if (WSAStartup(wVersionRequested, WSAData) = 0) then
  begin
    NameLen := 256;
    if (gethostname(Name, NameLen) = 0) then
    begin
      ptrHostEnt := gethostbyname (Name);
      if (ptrHostEnt = nil) then
      begin
        Msg := 'The Host Name is ' + StrPas(Name);
        Msg := Msg + #10 + ' but I could not get the Host Entry because:' + #10 + #10;
        case WSAGetLastError() of
          WSANOTINITIALISED: Msg := Msg + 'A successful WSAStartup must occur before using this function.';
          WSAENETDOWN: Msg := Msg + 'The Windows Sockets implementation has detected that the network subsystem has failed.';
          WSAHOST_NOT_FOUND: Msg := Msg + 'Authoritative Answer Host not found.';
          WSATRY_AGAIN: Msg := Msg + 'Non-Authoritative Host not found, or SERVERFAIL.';
          WSANO_RECOVERY: Msg := Msg + 'Nonrecoverable errors: FORMERR, REFUSED, NOTIMP.';
          WSANO_DATA: Msg := Msg + 'Valid name, no data record of requested type.';
          WSAEINPROGRESS: Msg := Msg + 'A blocking Windows Sockets operation is in progress.';
          WSAEINTR: Msg := Msg + 'The (blocking) call was canceled using WSACancelBlockingCall.';
        end;
      end
      else
      begin
{it actually worked:}
        Result := TRUE;
        DNSName := StrPas(ptrHostEnt^.h_name);
        for i := 0 to ptrHostEnt^.h_length - 1 do
         IP := Concat(IP, IntToStr(Ord(ptrHostEnt^.h_addr_list^[i])) + '.');
        SetLength(IP, Length(IP) - 1);
        Msg := 'The Host Name is: ' + StrPas(Name);
        Msg := Msg + #10 + ' and the full DNS name is: ' + StrPas(ptrHostEnt^.h_name);
        Msg := Msg + #10 + #10 + 'The alias are: ' + StrPas(ptrHostEnt^.h_aliases^);
      end;
    end
    else
    begin
      case WSAGetLastError() of
        WSAEFAULT: Msg := 'The namelen parameter is too small.';
        WSANOTINITIALISED: Msg := 'A successful WSAStartup must occur before using this function.';
        WSAENETDOWN: Msg := 'The Windows Sockets implementation has detected that the network subsystem has failed.';
        WSAEINPROGRESS: Msg := 'A blocking Windows Sockets operation is in progress.';
      end;
    end;
  end
  else
  begin
    case WSAGetLastError() of
      WSASYSNOTREADY: Msg := 'Indicates that the underlying network subsystem is not ready for network communication.';
      WSAVERNOTSUPPORTED: Msg := 'The version of Windows Sockets support requested is not provided by this particular Windows Sockets implementation.';
      WSAEINVAL: Msg := 'The Windows Sockets version specified by the application is not supported by this DLL.';
    end;
  end;
  WSACleanup();
  {ShowMessage(Msg);}
end;
{$ENDIF}


end.
