unit Titlestringgrid;

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Titlestringgrid.pas, released 1 Sept 2002.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 09/25/2002
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.


Purpose:
A Stringgrid with a Title row and column that can be set in the IDE

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  SysUtils, Classes,
{$IFDEF LINUX}
  Types, Qt,
  QGraphics, QControls, QForms, QDialogs, QGrids, QClipbrd,
{$ENDIF}
{$IFDEF WINDOWS}
  Wintypes, WinProcs, Messages, Graphics, Controls, Forms, Dialogs, Grids, Clipbrd,
{$ENDIF}
{$IFDEF WIN32}
  StdCtrls, Windows, Messages, Graphics, Controls, Forms, Dialogs, Grids, Clipbrd,
{$ENDIF}
  Misc;


type
  TAutoColWidth = (asNo, asProportional, asLeft, asRight);

  TTitleStringGrid = class(TStringGrid)
  private
    //FColTitles: TStringList;
    FAutoColWidth: TAutoColWidth;
    FColFont: TFont;
    //FRowTitles: TStringList;
    FRowFont: TFont;
    FMouseSelect: TGridRect;
    FOnTitleChange: TNotifyEvent;
  protected
    function GetColTitles: TStrings;
    function GetRowTitles: TStrings;

    procedure AssignTo(Dest: TPersistent); override;

    procedure SetAutoColWidth(Value: TAutoColWidth);
    procedure SetColTitles(Value: TStrings);
    procedure SetColFont(Value: TFont);
    procedure SetRowTitles(Value: TStrings);
    procedure SetRowFont(Value: TFont);
    procedure DoHandleChange;

    procedure Loaded; override;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect;
      AState: TGridDrawState); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure Resize; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure CopyToClipBoard;
  published
    property AutoColWidth: TAutoColWidth read FAutoColWidth write SetAutoColWidth default asNo;
    property ColTitles: TStrings read GetColTitles write SetColTitles;
{This is the Column Titles across the first row. The first Title is Cells[0,0].}
    property ColFont: TFont read FColFont write SetColFont;
    property RowTitles: TStrings read GetRowTitles write SetRowTitles;
{This is the Row Titles down the first column. The first Title is Cells[0,1].}
    property RowFont: TFont read FRowFont write SetRowFont;
    property OnTitleChange: TNotifyEvent read FOnTitleChange write FOnTitleChange;
  end;

implementation

constructor TTitleStringGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FColFont := TFont.Create;
  FRowFont := TFont.Create;
  FColFont.Style := [fsBold];
  FRowFont.Style := [fsBold];
  FMouseSelect.Left := -2;
  FMouseSelect.Top := -2;
end;

destructor TTitleStringGrid.Destroy;
begin
  FColFont.Free;
  FRowFont.Free;
  inherited Destroy;
end;

procedure TTitleStringGrid.Loaded;
begin
  inherited Loaded;
end;

procedure TTitleStringGrid.Resize;
var
  i, AllWidth, CellWidth: Integer;
  Ratio: Single;
begin
  case FAutoColWidth of
    asNo: ;
    asProportional:
      begin
        CellWidth := Self.ClientWidth - 1 - Self.ColCount;
        Self.DefaultColWidth := CellWidth div Self.ColCount - 1;
        Self.ColWidths[0] := Self.ClientWidth - (Self.ColCount-1) * (Self.DefaultColWidth+1) - 1;
      end;
    asLeft:
      begin
        CellWidth := Self.ClientWidth - 2;
        for i := Self.ColCount-1 downto 1 do
        begin
          CellWidth := CellWidth - Self.ColWidths[i] - 1;
        end;
        if (CellWidth > Self.DefaultColWidth) then
        begin
          Self.ColWidths[0] := CellWidth;
        end
        else
        begin
          AllWidth := Self.ClientWidth + Self.DefaultColWidth - CellWidth;
          Ratio := Self.ClientWidth / AllWidth;
          CellWidth := Self.ClientWidth - 2;
          for i := Self.ColCount-1 downto 1 do
          begin
            Self.ColWidths[i] := Round(Self.ColWidths[i] * Ratio);
            CellWidth := CellWidth - Self.ColWidths[i] - 1;
          end;
          Self.ColWidths[0] := CellWidth;
        end;
      end;
    asRight:
      begin
        CellWidth := Self.ClientWidth - 2;
        for i := 0 to Self.ColCount-2 do
        begin
          CellWidth := CellWidth - Self.ColWidths[i] - 1;
        end;
        if (CellWidth > Self.DefaultColWidth) then
        begin
          Self.ColWidths[Self.ColCount-1] := CellWidth;
        end
        else
        begin
          AllWidth := Self.ClientWidth + Self.DefaultColWidth - CellWidth;
          Ratio := Self.ClientWidth / AllWidth;
          CellWidth := Self.ClientWidth - 2;
          for i := 0 to Self.ColCount-2 do
          begin
            Self.ColWidths[i] := Round(Self.ColWidths[i] * Ratio);
            CellWidth := CellWidth - Self.ColWidths[i] - 1;
          end;
          Self.ColWidths[Self.ColCount-1] := CellWidth;
        end;
      end;
  end; {case}
  if Assigned(OnResize) then OnResize(Self);
end;

procedure TTitleStringGrid.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TTitleStringGrid(Dest).ColFont.Assign(FColFont);
  TTitleStringGrid(Dest).RowFont.Assign(FRowFont);
end;

function TTitleStringGrid.GetColTitles: TStrings;
begin
  Result := TStrings(Self.Rows[0]);
end;

procedure TTitleStringGrid.SetAutoColWidth(Value: TAutoColWidth);
begin
  FAutoColWidth := Value;
  Resize;
end;

procedure TTitleStringGrid.SetColTitles(Value: TStrings);
var
  i: Integer;
begin
  if (Value.Count > Self.ColCount) then
    Self.ColCount := Value.Count;
  for i := 0 to Value.Count-1 do
    Rows[0].Strings[i] := Value.Strings[i];
  for i := Value.Count to Self.ColCount-1 do
    Rows[0].Strings[i] := '';
  DoHandleChange;
end;

function TTitleStringGrid.GetRowTitles: TStrings;
begin
  Result := TStringList.Create;
  Result.Assign(Self.Cols[0]);
  Result.Delete(0);
end;

procedure TTitleStringGrid.SetRowTitles(Value: TStrings);
var
  i: Integer;
begin
{  if (Self.FixedCols > 0) then
  begin}
  if (Value.Count > Self.RowCount) then
    Self.RowCount := Value.Count+1;
  for i := 1 to Value.Count do
    Cols[0].Strings[i] := Value.Strings[i-1];
  for i := Value.Count+1 to Self.RowCount-1 do
    Cols[0].Strings[i] := '';
  DoHandleChange;
end;

procedure TTitleStringGrid.SetColFont(Value: TFont);
begin
  FColFont.Assign(Value);
  DoHandleChange;
end;

procedure TTitleStringGrid.SetRowFont(Value: TFont);
begin
  FRowFont.Assign(Value);
  DoHandleChange;
end;

procedure TTitleStringGrid.DoHandleChange;
begin
  if Assigned(FOnTitleChange) then
    OnTitleChange(Self);
  Refresh;  
end;

procedure TTitleStringGrid.DrawCell(ACol, ARow: Longint; ARect: TRect;
  AState: TGridDrawState);
begin
  if (ARow < FixedRows) then
    Self.Canvas.Font.Assign(FColFont)
   else if (ACol < FixedCols) then
    Self.Canvas.Font.Assign(FRowFont);
  inherited DrawCell(ACol, ARow, ARect, AState);
end;

procedure TTitleStringGrid.CopyToClipBoard;
var
  iCol, jRow: Integer;
  Str: String;
begin
  Str := '';
  for jRow := 0 to Self.RowCount-1 do
  begin
    for iCol := 0 to Self.ColCount-1 do
      Str := Str + Self.Cells[iCol, jRow] + #9;
    SetLength(Str, Length(Str)-1);
    Str := Str + #13#10;
  end;
  SetLength(Str, Length(Str)-2);
  Clipboard.AsText := Str;
end;

procedure TTitleStringGrid.KeyDown(var Key: Word; Shift: TShiftState);

  procedure DoDelete;
  var
    iCol, jRow: Integer;
  begin
    for iCol := Self.Selection.Left to Self.Selection.Right do
    begin
      for jRow := Self.Selection.Top to Self.Selection.Bottom do
        Self.Cells[iCol, jRow] := '';
    end;
  end;

  procedure DoCopy;
  var
    iCol, jRow: Integer;
    Str: String;
  begin
    Str := '';
    for jRow := Self.Selection.Top to Self.Selection.Bottom do
    begin
      for iCol := Self.Selection.Left to Self.Selection.Right do
        Str := Str + Self.Cells[iCol, jRow] + #9;
      SetLength(Str, Length(Str)-1);
      Str := Str + #13#10;
    end;
    SetLength(Str, Length(Str)-2);
    Clipboard.AsText := Str;
  end;

  procedure DoPaste;
  var
    iCol, jRow, TabPos, CRLFPos: Integer;
    Str: String;

    procedure CheckSize(x, y: Integer);
    begin
      if (y > Self.RowCount-1) then
        Self.RowCount := Self.RowCount + 1;
      if (x > Self.ColCount-1) then
        Self.ColCount := Self.ColCount + 1;
    end;

  begin
    Str := Clipboard.AsText;
    iCol := Self.Col;
    jRow := Self.Row;
    while (Length(Str) > 0) do
    begin
      TabPos := Pos(#9, Str);
      CRLFPos := Pos(#13#10, Str);
      if ((0 < TabPos) and (TabPos < CRLFPos)) then
      begin
        CheckSize(iCol, jRow);
        Self.Cells[iCol, jRow] := GetWord(Str, #9);
        Inc(iCol);
      end
      else
      begin
        CheckSize(iCol, jRow);
        Self.Cells[iCol, jRow] := GetWord(Str, #13#10);
        iCol := Self.Col;
        Inc(jRow);
        //CheckSize(iCol, jRow);
        //Self.Cells[iCol, jRow] := GetWord(Str, #9);
      end; {Tab or new row ?}
    end; {while}
  end;

begin
  if (Key = VK_DELETE) then
  begin
  end
  else if (ssCtrl in Shift) then
  begin
    if (Key = Ord('A')) then
    begin // select all
      {Self.Selection.Left := Self.FixedCols;
      Self.Selection.Top := Self.FixedRows;
      Self.Selection.Right := Self.ColCount-1;
      Self.Selection.Bottom := Self.RowCount-1;
      Key := 0;}
    end
    else if (Key = Ord('C')) then
    begin // copy
      DoCopy;
      Key := 0;
    end
    else if (Key = Ord('V')) then
    begin // paste
      DoPaste;
      Key := 0;
    end
    else if (Key = Ord('X')) then
    begin // cut
      DoCopy;
      DoDelete;
      Key := 0;
    end;
  end;

  inherited KeyDown(Key, Shift);
end;

procedure TTitleStringGrid.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  FMouseSelect.TopLeft := Self.MouseCoord(X, Y);
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TTitleStringGrid.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  GridCoord: TGridCoord;
begin
  GridCoord := Self.MouseCoord(X, Y);
  if ((goRangeSelect in Self.Options) and
      (ssLeft in Shift) and
      (X <> FMouseSelect.Left) and
      (Y <> FMouseSelect.Top) and
      (FMouseSelect.Left >= 0) and
      (FMouseSelect.Top >= 0)) then
  begin
    FMouseSelect.BottomRight := GridCoord;
    Self.Selection := FMouseSelect;
  end;
  inherited MouseMove(Shift, X, Y);
end;

procedure TTitleStringGrid.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  FMouseSelect.Left := -2;
  FMouseSelect.Top := -2;
  inherited MouseUp(Button, Shift, X, Y);
  Self.Refresh;
end;



end.
