unit Vector;

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Vector.pas, released 3 August 2001.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 08/03/2001
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.


Purpose:
Basic 3D vector class and methods.

Known Issues: Completely and utterly untested. Caveat Emptor !
-----------------------------------------------------------------------------}

interface

uses
  Classes, Math;

type  
  TVector = class;

  TVector = class(TPersistent) // use TObject if you don't want streaming
    private
      FX: Single;
      FY: Single;
      FZ: Single;
      FOnChange: TNotifyEvent;
    protected
      procedure SetX(Value: Single);
      procedure SetY(Value: Single);
      procedure SetZ(Value: Single);
    public
      constructor Create;
      destructor Destroy; override;
{All the meaty bits:}
      procedure DoAdd(AVector: TVector);
      function ReturnAdd(AVector: TVector): TVector;
      function DotProduct(AVector: TVector): Single;
      procedure DoCrossProduct(AVector: TVector);
      function ReturnCrossProduct(AVector: TVector): TVector;
      function GetAngle(AVector: TVector): Single;
    published
      property X: Single read FX write SetX;
      property Y: Single read FY write SetY;
      property Z: Single read FZ write SetZ;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

implementation

{begin TVector ----------------------------------------------------------------}
constructor TVector.Create;
begin
  inherited Create;
  FX := 0;
  FY := 0;
  FZ := 0;
  FOnChange := nil;
end;

destructor TVector.Destroy; // set FOnChange := nil
begin
  FOnChange := nil;
  inherited Destroy;
end;

procedure TVector.SetX(Value: Single); // fire FOnChange if assigned
begin
  FX := Value;
  if Assigned(FOnChange) then OnChange(Self);
end;

procedure TVector.SetY(Value: Single);
begin
  FY := Value;
  if Assigned(FOnChange) then OnChange(Self);
end;

procedure TVector.SetZ(Value: Single);
begin
  FZ := Value;
  if Assigned(FOnChange) then OnChange(Self);
end;

procedure TVector.DoAdd(AVector: TVector);
begin
  FX := FX + AVector.X;
  FY := FY + AVector.Y;
  FZ := FZ + AVector.Z;
  if Assigned(FOnChange) then OnChange(Self);
end;

function TVector.ReturnAdd(AVector: TVector): TVector;
begin
  Result.X := FX + AVector.X;
  Result.Y := FY + AVector.Y;
  Result.Z := FZ + AVector.Z;
end;

function TVector.DotProduct(AVector: TVector): Single;
begin
  Result := FX * AVector.X + FY * AVector.Y + FZ * AVector.Z;
end;

procedure TVector.DoCrossProduct(AVector: TVector);
begin
  FX := FY * AVector.Z - FZ * AVector.Y;
  FY := FZ * AVector.X - FX * AVector.Z;
  FZ := FX * AVector.Y - FY * AVector.X;
  if Assigned(FOnChange) then OnChange(Self);
end;

function TVector.ReturnCrossProduct(AVector: TVector): TVector;
begin
  Result.X := FY * AVector.Z - FZ * AVector.Y;
  Result.Y := FZ * AVector.X - FX * AVector.Z;
  Result.Z := FX * AVector.Y - FY * AVector.X;
end;

function TVector.GetAngle(AVector: TVector): Single;
var
  CosTheta: Single;
begin
  CosTheta := DotProduct(AVector) /
    (Sqrt(Sqr(FX) + Sqr(FY) + Sqr(FZ)) *
     Sqrt(Sqr(AVector.X) + Sqr(AVector.Y) + Sqr(AVector.Z)));
  Result := ArcCos(CosTheta);   
end;

{end TVector ----------------------------------------------------------------}

end.
