{-----------------------------------------------------------------------------
 Unit Name: uDelphiPackage
 Author:    herzogs2
 Purpose:   cool functions to install/uninstall and recompile delphi projects.
 History: 19.11.2002 - added error message in case the user has no administrator rights
          03.12.2002 - removed some unused code.
-----------------------------------------------------------------------------}
{$DEFINE MJB}

unit uDelphiPackage;

interface
uses StdCtrls,
     Classes,
     Windows,
     Grids,
     Registry;
const
  cDelphiKey='SOFTWARE\BORLAND\DELPHI';

procedure InstallPackage(_PackageName,_PackageDirectory:String;_DelphiVersion:Integer);
procedure UninstallPackage(_PackageName,_PackageDirectory:String;_DelphiVersion:Integer);
procedure DeleteBPLFile(_Filename:String);  // delete the bpl file.
procedure ReadPackageListfromBPGFile(_filename:string;var _stg:TStringGrid); // read the package names from the delphi project group file <.bpg>
function  CompileProject(_Compiler,_CompilerSwitches,_ProjectName,_WorkPath:string;Var Output:String):boolean; // compile the package
function  CanInstallPackage(_PackageName:string):boolean; // checks if it is a designtime package or not
function  GetPackageDescription(_PackageName:string):String; // get the description text
function  WinExecAndWait32V2(FileName,CommandLine,WorkPath: string; Visibility: Integer;Var Output:String): LongWord;
procedure Trace(_msg:String);
procedure AddPackageToRegistry(_RootKey:HKey;_Key,_PackageName,_PackageDescription:string); // delete a package entry from the registery
procedure RemovePackageFromRegistry(_RootKey:HKey;_Key,_PackageName:string); // delete a package entry from the registery
function  isDelphiStarted(_DelphiVersion:Integer): Boolean;
procedure ShutDownDelphi(_DelphiVersion:Integer;_Blocking : Boolean);
procedure StartUpDelphi(_path:String);
function  GetCompilerSwitches(_cfgFilename:String;var Conditions:string;var SearchPath:String):boolean; // get informations from the cfg-file.


implementation
uses SysUtils,
     ShellApi,
     Messages,
{$IFNDEF MJB}     MainFrm, {$ENDIF}
     Dialogs;

//*****************************************************
// Method:  GetField
// Programmer: S.Herzog
// Description: get a field from a string <_s> seperated by chars <_ch>.
// Last changes: 23.04.02
//*****************************************************
function GetField(_ch:char;var _s:string):string;
var
_pos:integer;
begin
  Result:='';
  _pos:=Pos(_ch,_s);
  if _pos=0 then begin
    Result:=_s;
    _s:='';
    exit;
  end;
  Result:=Copy(_s,1,_pos-1);
  Delete(_s,1,_pos);
end;


{-----------------------------------------------------------------------------
  Procedure: GetCompilerSwitches
  Author:    herzogs2
  Date:      27-Sep-2002
  Arguments: _cfgFilename:String;var _Conditions:String
  Result:    boolean
  Purpose:   get information from the cfg-file.
  History:
-----------------------------------------------------------------------------}
function GetCompilerSwitches(_cfgFilename:String;var Conditions:string;var SearchPath:String):boolean; // get informations from the cfg-file.
var
  _CFGFile:TStrings;
  _Pos:Integer;
  _Text:String;
begin
  Result:=false;
  if not fileExists(_cfgFilename) then begin
    Trace(format('GetCompilerSwitches: Could not find the file <%s>.',[_cfgFilename]));
    exit;
  end;
  _CFGFile:=TStringList.Create;
  _CFGFile.LoadFromFile(_cfgFilename);
  _Text:=_CFGFile.Text;
  _Pos:=Pos('-D',_Text);
  if _pos>0 then begin
    Delete(_text,1,_pos+1);
    _pos:=Pos(#$D+#$A,_Text);
    Conditions:=Copy(_Text,1,_Pos-1);
    Conditions:='-D"'+Conditions+'"';
    Trace('GetCompilerSwitches: Conditions are <'+Conditions+'>.');
  end;

  _Text:=_CFGFile.Text;
  _Pos:=Pos('-U"',_Text);
  if _pos>0 then begin
    Delete(_text,1,_pos+2);
    _pos:=Pos('"',_Text);
    SearchPath:=Copy(_Text,1,_Pos);
    Trace('GetCompilerSwitches: SearchPath are <'+SearchPath+'>.');
  end;

  _CFGFile.free;
end;

{-----------------------------------------------------------------------------
  Procedure: CloseDelphi
  Author:    herzogs2
  Date:      30-Aug-2002
  Arguments: None
  Result:    None
  Description:
-----------------------------------------------------------------------------}
procedure CloseDelphi;
var
W : HWnd;
begin
  W := FindWindow(NIL,'delphi32.exe');
  if not(W=0) then begin
    PostMessage(W, wm_close, 1,1);
  end;
end;

{-----------------------------------------------------------------------------
  Procedure: UninstallPackage
  Author:    herzogs2
  Date:      30-Aug-2002
  Arguments: _PackageName,_PackageDirectory:String;_DelphiVersion:Integer
  Result:    None
  Description: uninstall the package <_PackageName>.
-----------------------------------------------------------------------------}
procedure UnInstallPackage(_PackageName,_PackageDirectory:String;_DelphiVersion:Integer);
var
_sDelphiVersion:String;
_PackageKey:String;
begin
  if ExtractFileExt(_PackageName)<>'.dpk' then exit;
  _sDelphiVersion:=inttostr(_DelphiVersion)+'.0';
  _PackageKey:=cDelphiKey+'\'+_sDelphiVersion+'\Known Packages';
  RemovePackageFromRegistry(HKEY_CURRENT_USER,_PackageKey,_PackageDirectory+ChangeFileExt(ExtractFileName(_PackageName),'.bpl'));
  RemovePackageFromRegistry(HKEY_LOCAL_MACHINE,_PackageKey,_PackageDirectory+ChangeFileExt(ExtractFileName(_PackageName),'.bpl'));
  _PackageKey:=cDelphiKey+'\'+_sDelphiVersion+'\Disabled Packages';
  RemovePackageFromRegistry(HKEY_CURRENT_USER,_PackageKey,_PackageDirectory+ChangeFileExt(ExtractFileName(_PackageName),'.bpl'));
  RemovePackageFromRegistry(HKEY_LOCAL_MACHINE,_PackageKey,_PackageDirectory+ChangeFileExt(ExtractFileName(_PackageName),'.bpl'));
end;

{-----------------------------------------------------------------------------
  Procedure: InstallPackage
  Author:    herzogs2
  Date:      30-Aug-2002
  Arguments: _PackageName,_PackageDirectory:String;_DelphiVersion:Integer
  Result:    None
  Description: install the delphi package <_PackageName> which is located in the
               directory <_PackageDirectory> for delphi version <_DelphiVersion>.
               The parameter <_PackageDirectory> points to the location of the <.bpl> file.
-----------------------------------------------------------------------------}
procedure InstallPackage(_PackageName,_PackageDirectory:String;_DelphiVersion:Integer);
var
_PackageDescription:string;
_sDelphiVersion:String;
_PackageKey:String;
begin
  if ExtractFileExt(_PackageName)<>'.dpk' then exit;
  _sDelphiVersion:=inttostr(_DelphiVersion)+'.0';
  _PackageKey:=cDelphiKey+'\'+_sDelphiVersion+'\Known Packages';
  _PackageDescription:=GetPackageDescription(_PackageName);
  if CanInstallPackage(_PackageName) then begin
    AddPackageToRegistry(HKEY_CURRENT_USER,_PackageKey,_PackageDirectory+ChangeFileExt(ExtractFilename(_PackageName),'.bpl'),_PackageDescription);
    AddPackageToRegistry(HKEY_LOCAL_MACHINE,_PackageKey,_PackageDirectory+ChangeFileExt(ExtractFilename(_PackageName),'.bpl'),_PackageDescription);
  end
  else Trace(Format('InstallPackage: The package <%s> is a runtime only package. It will not be installed into the IDE.',[_PackageName]));
end;


{-----------------------------------------------------------------------------
  Procedure: GetPackageDescription
  Author:    herzogs2
  Date:      22-Aug-2002
  Arguments: _PackageName:string
  Result:    String
  Description: get the package description text out of the dpk-file.
-----------------------------------------------------------------------------}
function GetPackageDescription(_PackageName:string):String; // get the description text
var
  _DPKFile:TStrings;
  _Pos:Integer;
  _Text:String;
begin
  Result:='';
  if not fileExists(_PackageName) then begin
    Trace(format('IsDesignPackage: Could not find the file <%s>.',[_PackageName]));
    exit;
  end;
  _DPKFile:=TStringList.Create;
  _DPKFile.LoadFromFile(_PackageName);
  _Text:=_DPKFile.Text;
  _Pos:=Pos('$DESCRIPTION',_Text);
  Delete(_Text,1,_pos+13);
  _pos:=Pos('''',_Text);
  Result:=Copy(_Text,1,_Pos-1);
  _DPKFile.free;
end;

{-----------------------------------------------------------------------------
  Procedure: WinExecAndWait32V2
  Author:    P.Below
             Added Output parameter by S.Herzog
  Date:      22-Aug-2002
  Arguments: FileName: string; Visibility: Integer
  Result:    DWORD
  Description:
-----------------------------------------------------------------------------}
function WinExecAndWait32V2(FileName,CommandLine,WorkPath: string; Visibility: Integer;Var Output:String): LongWord;
  procedure WaitFor(processHandle: THandle);
  var
    Msg: TMsg;
    ret: DWORD;
  begin
    repeat
      ret := MsgWaitForMultipleObjects(1, { 1 handle to wait on }
        processHandle, { the handle }
        False, { wake on any event }
        INFINITE, { wait without timeout }
        QS_PAINT or { wake on paint messages }
        QS_SENDMESSAGE { or messages from other threads }
        );
      if ret = WAIT_FAILED then Exit; { can do little here }
      if ret = (WAIT_OBJECT_0 + 1) then
      begin
          { Woke on a message, process paint messages only. Calling
            PeekMessage gets messages send from other threads processed. }
        while PeekMessage(Msg, 0, WM_PAINT, WM_PAINT, PM_REMOVE) do
          DispatchMessage(Msg);
      end;
    until ret = WAIT_OBJECT_0;
  end; { Waitfor }
var { V1 by Pat Ritchey, V2 by P.Below }
  zAppName: array[0..2048] of char;
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
  SA: TSecurityAttributes;
  StdOutPipeRead,
  StdOutPipeWrite: THandle;
  Buffer: array[0..255] of Char;
  BytesRead: Cardinal;
  _Line: String;
  _tmp:String;
begin { WinExecAndWait32V2 }
//  Trace(Filename);
  _Line:='';
   with SA do
   begin
     nLength := SizeOf(SA);
     bInheritHandle := True;
     lpSecurityDescriptor := nil;
   end;
   CreatePipe(StdOutPipeRead,  // read handle
              StdOutPipeWrite, // write handle
              @SA,             // security attributes
              65535                // number of bytes reserved for pipe - 0 default
              );

//  StrPCopy(zAppName, FileName);
  FillChar(StartupInfo, SizeOf(StartupInfo), #0);
  StartupInfo.cb          := SizeOf(StartupInfo);
  StartupInfo.dwFlags     := STARTF_USESHOWWINDOW  or  STARTF_USESTDHANDLES;
  StartupInfo.wShowWindow := Visibility;
  StartupInfo.hStdInput  := GetStdHandle(STD_INPUT_HANDLE); // don't redirect std input
  StartupInfo.hStdOutput := StdOutPipeWrite;
  StartupInfo.hStdError  := StdOutPipeWrite;

  if not CreateProcess(
    //zAppName, { pointer to command line string }
    PChar(FileName),    //appplication to be executed
    PChar(CommandLine), // command line.
    nil,
    nil, { pointer to thread security attributes }
    true, { handle inheritance flag }
    CREATE_NEW_CONSOLE or { creation flags }
    NORMAL_PRIORITY_CLASS,
    nil, { pointer to new environment block }
    PChar(WorkPath), { pointer to current directory name }
    StartupInfo, { pointer to STARTUPINFO }
    ProcessInfo) { pointer to PROCESS_INF } then
    Result := DWORD(-1) { failed, GetLastError has error code }
  else
  begin
    Waitfor(ProcessInfo.hProcess);
    GetExitCodeProcess(ProcessInfo.hProcess, Result);
    CloseHandle(ProcessInfo.hProcess);
    CloseHandle(ProcessInfo.hThread);
    CloseHandle(StdOutPipeWrite);
  end; { Else }
  ReadFile(StdOutPipeRead, Buffer, 255, BytesRead, nil);
  while BytesRead>0 do begin
    Buffer[BytesRead] := #0;
    // combine the buffer with the rest of the last run
    _tmp:=string(trim(Buffer));
    _Line := _Line + _tmp;
    ReadFile(StdOutPipeRead, Buffer, 255, BytesRead, nil);
  end;
  if length(_line)>0 then Output:=_Line;
  CloseHandle(StdOutPipeRead);
end; { WinExecAndWait32V2 }


{-----------------------------------------------------------------------------
  Procedure: CanInstallPackage
  Author:    herzogs2
  Date:      22-Aug-2002
  Arguments: _PackageName:string
  Result:    boolean
  Description: runtime only packages can not be installed in the delphi IDE.
-----------------------------------------------------------------------------}
function CanInstallPackage(_PackageName:string):boolean; // checks if it is a designtime package or not
var
  _DPKFile:TStrings;
begin
  Result:=False;
  if not fileExists(_PackageName) then begin
    Trace(format('IsDesignPackage: Could not find the file <%s>.',[_PackageName]));
    exit;
  end;
  _DPKFile:=TStringList.Create;
  _DPKFile.LoadFromFile(_PackageName);
  if Pos('$RUNONLY',_DPKFile.Text)=0 then Result:=True;
  _DPKFile.free;
end;


{-----------------------------------------------------------------------------
  Procedure: AddPackageToRegistry
  Author:    herzogs2
  Date:      30-Aug-2002
  Arguments: _RootKey:HKey;_Key,_PackageName,_PackageDescription:string
  Result:    None
  Description:
               19.11.2002 - added error message in case the user has no administrator rights
-----------------------------------------------------------------------------}
procedure AddPackageToRegistry(_RootKey:HKey;_Key,_PackageName,_PackageDescription:string); // delete a package entry from the registery
var
  _Reg: TRegistry;
begin
  _Reg := TRegistry.Create;
  try
    _Reg.RootKey := _RootKey;
    if not _Reg.ValueExists(_PackageName) then begin
      _Reg.OpenKey(_Key,false);
      try
        _Reg.WriteString(_PackageName,_PackageDescription);
        Trace(format('AddPackageToRegistry: Successfully installed the  Packages <%s>.',[_PackageName]));
      except
        Trace(format('Problem in AddPackageToRegistry: Could not add the package <%s> to the registry.You need to have Admin rights for this computer.',[_PackageName]));
      end;
      _Reg.CloseKey;
    end else Trace(format('AddPackageToRegistry: The Packages <%s> is already registered.',[_PackageName]));
  finally
    _Reg.Free;
  end;
end;

{-----------------------------------------------------------------------------
  Procedure: CompileProject
  Author:    herzogs2
  Date:      29-Aug-2002
  Arguments:
  _Compiler   --> full name and path to bcc32.exe
  _CompilerSwitches, --> compiler switches like -W -H -B for example
  _ProjectName --> the name of the project to be compiled (can be .dpr or .dpk )
  _TargetPath,  --> output target path - removed by MJB because unused
  Var Output:String   --> output text of the compiler.
  Result:    boolean
  Description:
-----------------------------------------------------------------------------}
function CompileProject(_Compiler,_CompilerSwitches,_ProjectName,_WorkPath:string;Var Output:String):boolean; // compile the package
var
_commandLine:string;
begin
  Result:=false;
  if fileexists(_Compiler) then begin
    if FileExists(_ProjectName) then begin
      Trace('');
      Trace('*************************************************************************************');
      Trace(format('Compile Project <%s>.',[_ProjectName]));
      _commandLine:=Trim(_CompilerSwitches+' '+_ProjectName);
      WinExecAndWait32V2(_compiler,
                         _commandLine,
                         _WorkPath,
                         SW_HIDE,
                         Output);

      if (pos('seconds,',Output)>0) or
         (pos('Sekunden',Output)>0) then begin
        Trace(format('CompileProject: Successfully build Project file <%s>.',[_ProjectName]));
        Result:=True;
      end;
    end else Trace(format('CompileProject: Problem, could not find the Project file <%s>.',[_ProjectName]));
  end else Trace(format('CompileProject: Problem, could not find the Delphi Compiler file <%s>.',[_Compiler]));
end;


{-----------------------------------------------------------------------------
  Procedure: ReadPackageListfromBPGFile
  Author:    herzogs2
  Date:      22-Aug-2002
  Arguments: _filename:string;var _lbx:TListbox
  Result:    None
  Description:
-----------------------------------------------------------------------------}
procedure ReadPackageListfromBPGFile(_filename:string;var _stg:TStringGrid); // read the package names from the delphi project group file <.bpg>
var
  _BPGFile:TStrings;
  i:integer;
  _line:string;
  _Projects:String;
  _Project:String;
  _idx:Integer;
  _row:integer;

function RemoveSlash(_Project:String):String;
var
_pos:Integer;
begin
  Result:=_Project;
  _pos:=Pos('\',_Project);
  if _pos>0 then begin
    System.Delete(_Project,_pos,1);
    Result:=trim(_project);
  end;
end;

begin
  _row:=1;
  if not fileExists(_filename) then begin
    Trace(format('ReadPackageListfromBPGFile: Could not find the file <%s>.',[_filename]));
    exit;
  end;
  _BPGFile:=TStringList.Create;
  _BPGFile.LoadFromFile(_filename);
  _stg.rowcount:=2;
  _idx:=0;
  _Projects:='';
  for i:=0 to _BPGFile.Count-1 do begin
    _line:=trim(_BPGFile.Strings[i]);
    if Pos('PROJECTS',_line)=1 then _Projects:=_line
    else begin
      if _Projects<>'' then begin
        if pos('#--------------',_line)=1 then begin
          _idx:=i;
          break;
        end
        else _Projects:=_Projects+_line;
      end;
    end;
  end;
  System.Delete(_Projects,1,10);
  _Projects:=trim(_Projects);
  _Project:=GetField(' ',_Projects);
  _Project:=RemoveSlash(_Project);
  while _Project<>'' do begin
    for i:=_idx to _BPGFile.Count-1 do begin
      _line:=trim(_BPGFile.Strings[i]);
      if Pos(_Project,_line)=1 then begin
        System.Delete(_line,1,Pos(':',_line));
        _stg.cells[0,_row]:=inttostr(_row);
        _stg.cells[1,_row]:=trim(_line);
        _stg.cells[2,_row]:='';
        _stg.cells[3,_row]:='';
        inc(_row);
        _stg.RowCount:=_row;
        break;
      end;
    end;
    _Project:=GetField(' ',_Projects);
    _Project:=RemoveSlash(_Project);
  end;
  _BPGFile.free;
end;


{-----------------------------------------------------------------------------
  Procedure: DeleteBPLFile
  Author:    herzogs2
  Date:      22-Aug-2002
  Arguments: _Filename:String
  Result:    None
  Description:
-----------------------------------------------------------------------------}
procedure DeleteBPLFile(_Filename:String);  // delete the bpl file.
begin
  if fileexists(_Filename) then begin
    DeleteFile(_Filename);
    Trace(format('DeleteBPLFile: Deleted file <%s>.',[_Filename]));
  end;

  _Filename:=ChangeFileExt(_Filename,'.dcp');
  if fileexists(_Filename) then begin
    DeleteFile(_Filename);
    Trace(format('DeleteBPLFile: Deleted file <%s>.',[_Filename]));
  end;

end;


{-----------------------------------------------------------------------------
  Procedure: RemovePackageFromRegistry
  Author:    herzogs2
  Date:      30-Aug-2002
  Arguments: _RootKey:HKey;_Key,_PackageName:string
  Result:    None
  Description:
-----------------------------------------------------------------------------}
procedure RemovePackageFromRegistry(_RootKey:HKey;_Key,_PackageName:string);
var
  _Reg: TRegistry;
begin
  _Reg := TRegistry.Create;
  try
    _Reg.RootKey := _RootKey;
    _Reg.OpenKey(_Key,false);
    if _Reg.ValueExists(_PackageName) then begin
      if _Reg.DeleteValue(_PackageName) then Trace(format('RemovePackageFromRegistry: Successfully deleted Key Value <%s> from <%s>.',[_PackageName,_Key]))
                                        else Trace(format('RemovePackageFromRegistry: Could not find Key Value <%s> in <%s>.',[_PackageName,_key]));
    end;
    _Reg.CloseKey;
  finally
    _Reg.Free;
  end;
end;

procedure Trace(_msg:String);
begin
  OutputDebugString(PChar(_msg));
{$IFDEF MJB}
{$ELSE}
  Form1.mmoLogFile.lines.add(_msg);
{$ENDIF}
end;

{-----------------------------------------------------------------------------
  Procedure: GetDelphiHandle
  Author:    m.birbaumer
  Date:      30-Aug-2002
  Arguments: None
  Result:    THandle
  Description:
-----------------------------------------------------------------------------}
function GetDelphiHandle(_DelphiVersion:Integer): THandle;
begin
  result := FindWindow('TApplication', PChar('Delphi '+inttostr(_DelphiVersion)));
end;

{-----------------------------------------------------------------------------
  Procedure: DelphiStarted
  Author:    m.birbaumer
  Date:      30-Aug-2002
  Arguments: None
  Result:    Boolean
  Description:
-----------------------------------------------------------------------------}
function isDelphiStarted(_DelphiVersion:Integer): Boolean;
begin
  result := GetDelphiHandle(_DelphiVersion) <> 0;
end;

{-----------------------------------------------------------------------------
  Procedure: ShutDownDelphi
  Author:    m.birbaumer
  Date:      30-Aug-2002
  Arguments: Blocking : Boolean
  Result:    None
  Description:
-----------------------------------------------------------------------------}
procedure ShutDownDelphi(_DelphiVersion:Integer;_Blocking : Boolean);
var
  _hWnd : THandle;
begin
  _hWnd := GetDelphiHandle(_DelphiVersion);
  if _hWnd <> 0 then begin
    if _Blocking then SendMessage(_hWnd, WM_CLOSE, 0, 0)
                 else PostMessage(_hWnd, WM_CLOSE, 0, 0);
  end;
end;

{-----------------------------------------------------------------------------
  Procedure: StartUpDelphi
  Author:    m.birbaumer
  Date:      30-Aug-2002
  Arguments: None
  Result:    None
  Description:
-----------------------------------------------------------------------------}
procedure StartUpDelphi(_path:String);
var
  _FileName : String;
begin
  _FileName := _Path+'delphi32.exe';
  if FileExists(_FileName) then WinExec(PChar(_FileName), SW_SHOWNORMAL);
end;

end.
