program cleandfm;

{$APPTYPE CONSOLE}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License Version 1.1 (the "License"); you may not use this file except in compliance with the License. You may obtain a copy of the License at

http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for the specific language governing rights and limitations under the License.

The Original Code is: cleandfm.dpr, released 22 June 2001.

Distantly Derived from Peter N Roth's
  D2T = Dfm To Text.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 2001 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 01/22/2003
Current Version: 2.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.


Purpose:
To "clean" a Delphi form file (dfm) to remove or change properties that
cause problems in Delphi or Kylix.
 

Known Issues:
  Caveat Emptor !
  This can possibly destroy your source code !
  Never apply to your original source !
  Make several backups in several locations first !
  Check the output forms carefully before discarding your originals !
  Use at your own risk !
  Wear lead-plated underwear !
  Untested under Kylix, but may work.

Todo:
  We really want to convert 16 and 24 bit images, rather than nuke all images.
-----------------------------------------------------------------------------}


uses
  Classes,
  SysUtils,
  FileCtrl,
  Misc;

type
  TTargetOS = (osBoth, osWindows, osLinux);
  TBadProperty = record
    Level: Integer;
    Text: String;
  end;

var
  DownToVersion: Integer;
  CurrentDir,
  Log,
  pasName,
  Str: string;
  TheMemory: TMemoryStream;
  TheText: TStringList;

{Win32 file and time structures:}
  FileData: TSearchRec;
  i: Integer;
  AsText,
  DeleteImages,
  IsBinary,
  Recursive: Boolean;
  TargetOS: TTargetOS;
  NewProperties: array [2..9] of TStringList;

const
{$IFDEF WIN32}
  DELIMITER = '\';
{$ENDIF}
{$IFDEF LINUX}
  DELIMITER = '/';
{$ENDIF}

{You may want to strip all non-Delphi-1 properties from forms:}
{const
  NEW_PROPERTIES_MAX = 14;
  NewProperties: array[0..NEW_PROPERTIES_MAX] of string =
     ('Action =', 'Align =', 'Anchors =', 'BiDiMode =', 'BorderWidth =',
      'Constraints =', 'DefaultMonitor =', 'Docksite =', 'DragKind =', 'DragMode =',
      'HelpFile =', 'PixelsPerInch =', 'ParentBiDiMode =', 'ParentFont =', 'UseDockManager =');
}

resourcestring
{What is new in ...}
  sNewD2 = '';
  sNewD3 = 'FullRepaint,OnStartDrag,HotTrack,ScrollOpposite,TabPosition {TPageControl},AllowActiveDoc,DesktopFont';
  sNewD4 = 'Action,Anchors,AutoSize,BiDiMode,Constraints,Docksite,DragKind,DragMode,ParentBiDiMode,UseDockManager,' +
    'OnCanResize,OnConstrainedResize,OnContextPopup,OnDockDrop,OnDockOver,OnEndDock,OnGetSiteInfo,OnStartDock,OnUnDock';
  sNewD5 = 'HoverTime,OnColumnRightClick,OnSectionDrag,ItemEnabled,OnAdvancedCustomDraw,OnAdvancedCustomDrawItem,OnInfotip';
  sNewD6 = 'AlphaBlend,AlphaBlendValue,TransparentColor,TransparentColorValue,BevelEdges,BevelKind,' +
    'AutoComplete,HeaderColor,HeaderBackgroundColor,CaseSensitive,FatalException,OnAddition,' +
    'Format,HotTrack,Menu,NullValueKey,AutoComplete,AutoDropDown,OptionsEx,Proportional,' +
    'OnSettingChange,AutoCheck,OnCloseUp,OnSelect,OnData,OnDataFind,OnDataObject,' +
    'clMoneyGreen,clSkyBlue,clCream,clMedGray';
  sNewD7 = 'ScreenSnap,SnapBuffer,AutoCompleteOptions';
  sNewD8 = '';
  sNewD9 = '';

  procedure WriteLog(Str: String);
  begin
    Log := Log + Str;
    Write(Str);
  end;

  procedure ProcessDir(Dir: String);
  var
    j, k, l,
    hSearch,
    ThePosition: Integer;
    WriteError,
    Bugout, Changed: Boolean;
    backupName,
    StrValue: string;
    TheDirs: TStringList;
    dfm: TFileStream;
  begin
    WriteLog(#13#10 + '  Entering ' + Dir + #13#10);
    SetCurrentDir(Dir);
    TheDirs := TStringList.Create;
{We use the Delphi version of file finding:}
    hSearch := Sysutils.FindFirst(GetCurrentDir + DELIMITER + '*.*', faAnyFile, FileData);
    while (hSearch = 0) do
    begin
      if (Lowercase(ExtractFileExt(FileData.Name)) = '.dfm') then
      begin
{Initialize booleans:}
        IsBinary := FALSE;
        Changed := FALSE;
{process filename:}
        backupName := FileData.Name + '.bak';
        pasName := ChangeFileExt(FileData.Name, '.pas');
{Let the user know:}
        WriteLog('    processing ' + FileData.Name + #13+#10);
{Load the dfm file:}
        TheText.LoadFromFile(FileData.Name);
        if (TheText.Strings[0] = '') then
        begin
{It's binary ! Reload as such:}
          TheText.Clear;
          IsBinary := TRUE;
{stream it in}
          dfm := TFileStream.Create(FileData.Name, fmOpenRead); //  may throw
          try
{do the conversion from binary to text:}
            ObjectResourceToText(dfm,TheMemory);
            TheMemory.Seek(0, soFromBeginning);
{stick it in the stringlist}
            TheText.LoadFromStream(TheMemory);
            if (AsText) then
              TheText.SaveToFile(FileData.Name + '.txt');
            TheMemory.Clear;
          except
{give up on this file; find next one and continue:}
            WriteLog('*** Something went wrong when reading ***' + FileData.Name +#13+#10);
            dfm.free;
            hSearch := Sysutils.FindNext(FileData);
            Continue;
          end;
          dfm.free;
        end; {IsBinary}

{Initialize line counter:}
        j := 0;
        repeat
          ThePosition := Pos('BorderStyle = ', TheText.Strings[j]);
{Kylix sticks an 'f' in front of the Delphic styles:}
          if (ThePosition > 0) then
          begin
            StrValue := Copy(TheText.Strings[j],
              ThePosition + Length('BorderStyle = '), 999);
            if (TargetOS = osWindows) then
              if (StrValue[1] = 'f') then
                System.Delete(StrValue, 1, 1);
            if (TargetOS = osLinux) then
              if (StrValue[1] = 'b') then
                StrValue := 'f' + StrValue;
            if (TargetOS = osWindows) or (TargetOS = osLinux) then
              TheText.Strings[j] := Copy(TheText.Strings[j],
                1, ThePosition + Length('BorderStyle = ') -1) + StrValue;
            if (TargetOS = osBoth) then
            begin
              TheText.Delete(j);
              Continue;
            end;
            Changed := TRUE;
          end;

{The default form color in Delphi is clBtnFace.
In Kylix, it's clBackground. However, Kylix translates clBtnface to clButton -
which does not exist under Windows.
However, clBackground in Windows is usually a peculaiar shade of green.
Thus Kylix understands clBtnFace, so that's what we make it.
Ditto with clText == clWindowText
Apologies to all you clBackground fans out there !}
          ThePosition := Pos('Color = clBackground', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Strings[j] := Copy(TheText.Strings[j],
              1, ThePosition + Length('Color = ') -1) + 'clBtnFace';
            Changed := TRUE;
            Continue;
          end;
          ThePosition := Pos('Color = clButton', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Strings[j] := Copy(TheText.Strings[j],
              1, ThePosition + Length('Color = ') -1) + 'clBtnFace';
            Changed := TRUE;
            Continue;
          end;
          ThePosition := Pos('Color = clText', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Strings[j] := Copy(TheText.Strings[j],
              1, ThePosition + Length('Color = ') -1) + 'clWindowText';
            Changed := TRUE;
            Continue;
          end;

{Look for the additional Kylix stuff that irritates Delphi:}
          ThePosition := Pos('Helptype', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Delete(j);
            Changed := TRUE;
            Continue;
          end;

          ThePosition := Pos('Font.Weight', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Delete(j);
            Changed := TRUE;
            Continue;
          end;

{Now look for Delphic stuff that irritates Kylix:}
          ThePosition := Pos('Charset', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Delete(j);
            Changed := TRUE;
            Continue;
          end;

          ThePosition := Pos('OldCreateOrder', TheText.Strings[j]);
          if (ThePosition > 0) then
          begin
            TheText.Delete(j);
            Changed := TRUE;
            Continue;
          end;

{Strip out bitmaps and icons if required:}
          if (DeleteImages) then
          begin
            ThePosition := Pos('.Data = {', TheText.Strings[j]);
            if (ThePosition > 0) then
            begin
              repeat
                TheText.Delete(j);
              until (Pos('}', TheText.Strings[j]) > 0);
              TheText.Delete(j);
              Changed := TRUE;
              Continue;
            end;
          end;

{Strip out every property except those in D1:}
          for k := 9 downto DownToVersion+1 do
          begin
            Bugout := FALSE;
            for l := 0 to NewProperties[k].Count-1 do
            begin
              ThePosition := Pos(NewProperties[k].Strings[l], TheText.Strings[j]);
              if (ThePosition > 0) then
              begin
                TheText.Delete(j);
                Changed := TRUE;
                Bugout := TRUE;
                Break;
              end;
            end;
            if (Bugout) then Break;
          end;

          Inc(j);
        until (j >= TheText.Count); {end main text loop}

{Do we need to backup and save this file ?}
        if (Changed) then
        begin
          WriteLog('    backing up then saving modified ' + FileData.Name +#13+#10);
{backup if one does not exist:}
          if (not FileExists(backupName)) then
          begin
            TheMemory.Clear;
            TheMemory.LoadFromFile(FileData.Name);
            TheMemory.SaveToFile(backupName);
            TheMemory.Clear;
          end;
          if (IsBinary) then
          begin
            WriteError := FALSE;
            TheMemory.Clear;
            dfm := TFileStream.Create(FileData.Name, fmCreate); //  may throw
            try
              dfm.Seek(0, soFromBeginning);
{stick it in a stringlist}
              TheText.SaveToStream(TheMemory);
              TheMemory.Seek(0, soFromBeginning);
{do the conversion from binary to text:}
              ObjectTextToResource(TheMemory, dfm);
            except
{give up on this file; find next one and continue:}
              on E: EParserError do
              begin
                WriteLog('*** ' + E.Message + ' ***' +#13+#10);
                TheText.Insert(0, '');
                TheText.Insert(0, E.Message);
                TheText.SaveToFile(FileData.Name + '.err');
                WriteError := TRUE;
              end
              else
              begin
                WriteLog('*** Something went wrong when writing ' + FileData.Name + ' ***' +#13+#10);
                WriteError := TRUE;
              end;
            end;
            dfm.free;
            if (WriteError) then
            begin
              hSearch := Sysutils.FindNext(FileData);
              Continue;
            end;
          end
          else
          begin {is not binary, is text dfm:}
            TheText.SaveToFile(FileData.Name);
          end;
        end; {Changed}

{Clean up working components:}
        TheMemory.Clear;
        TheText.Clear;

{Check that "$R *.DFM" is now lower case, to be consistent with the IDE saved file:}
        if (FileExists(pasName)) then
        begin  {NB: C++ Builder uses lowercase anyway: #pragma resource "*.dfm"}
          Changed := FALSE;
          TheText.LoadFromFile(pasName);
          for j := 1 to TheText.Count do
          begin
            if (TheText.Strings[j] = '{$R *.dfm}') then
              break;
            if (TheText.Strings[j] = '{$R *.DFM}') then
            begin
              TheText.Strings[j] := '{$R *.dfm}';
              Changed := TRUE;
              WriteLog('    Modifying {$R *.DFM} in ' + pasName + #13+#10);
              break;
            end;
          end;
        end;
        if (Changed) then
          TheText.SaveToFile(pasName);
      end {that was a .dfm file !}
      else if (Recursive and
              ((faDirectory and FileData.Attr) > 0) and
              (FileData.Name <> '.') and
              (FileData.Name <> '..')) then
      begin {is a subdirectory:}
        TheDirs.Add(Dir + DELIMITER + FileData.Name);
        {WriteLog('  Entering ' + FileData.Name + #13#10);
        ProcessDir(FileData.Name);
        SetCurrentDir('..' + DELIMITER);}
      end;
      hSearch := Sysutils.FindNext(FileData);
    end; {found something}
    Sysutils.FindClose(FileData);
    for j := 0 to TheDirs.Count-1 do
    begin
      ProcessDir(TheDirs.Strings[j]);
    end;
    TheDirs.Free;
  end;

  procedure ProcessProperties(Props: TStringList; Dn: String);
  begin
    while (Length(Dn) > 0) do
    begin
      Props.Add(Misc.Getword(Dn, ','))
    end;
  end;

{Program begins ***************************************************************}
begin
{The default is to clean everything out:}
  TargetOS := osBoth;
{leave images in:}
  DeleteImages := FALSE;
{Recurse subdirectories:}
  Recursive := FALSE;
{Save binary .dfm as text .dfm.txt:}
  AsText := FALSE;
{Remove properties down to:}
  DownToVersion := 7;

  for i := 2 to 9 do
    NewProperties[i] := TStringList.Create;
  ProcessProperties(NewProperties[2], sNewD2);
  ProcessProperties(NewProperties[3], sNewD3);
  ProcessProperties(NewProperties[4], sNewD4);
  ProcessProperties(NewProperties[5], sNewD5);
  ProcessProperties(NewProperties[6], sNewD6);
  ProcessProperties(NewProperties[7], sNewD7);
  ProcessProperties(NewProperties[8], sNewD8);
  ProcessProperties(NewProperties[9], sNewD9);

  writeln('cleandfm: removes "difficult" properties from Delphi .dfm files');
  writeln('  selected options are:');

{Process user input on command line:}
  for i := 1 to ParamCount do
  begin
    pasName := ParamStr(i);
    if (LowerCase(pasName) = '-delimages') then
    begin
      DeleteImages := TRUE;
      writeln('   - delete images');
    end
    else if (LowerCase(pasName) = '-win') then
    begin
      TargetOS := osWindows;
      writeln('   - delete Windows-isms');
    end
    else if (LowerCase(pasName) = '-lin') then
    begin
      TargetOS := osLinux;
      writeln('   - delete Linux-isms');
    end
    else if (LowerCase(pasName) = '-r') then
    begin
      Recursive := TRUE;
      writeln('   - recurse subdirectories');
    end
    else if (LowerCase(pasName) = '-text') then
    begin
      AsText := TRUE;
      writeln('   - produce text .dfms');
    end
    else if (LowerCase(pasName)[2] = 'd') then
    begin
      Str := LowerCase(pasName)[3];
      if (Str[1] in ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9']) then
      begin
        DownToVersion := StrToInt(Str);
        writeln(Format('   - down to Delphi version %d', [DownToVersion]));
      end;
    end
    else if ((Pos('help', pasName) > 0) or
      (Pos('?', pasName) > 0)) then
    begin
      write('usage: cleandfm [-r] [-delimages] [-#n] [-OS] [directory path]' + #13+#10);
      write('    -r         = recurse subdirectories' + #13+#10);
      write('    -delimages = delete all images from dfm files' + #13+#10);
      write('    -text      = save binary dfms as text .dfm.txt' + #13+#10);
      write('    -Dn        = strip out properties down to Compiler (Delphi) n (caution !)' + #13+#10);
      write('    -OS        = win or lin' + #13+#10);
      write('      if OS is absent, it means clean out both Windows and Linux problem properties' + #13+#10);
      write('  Example: recursively strip down to Delphi 4, from ..\..\Source:' + #13+#10);
      write('    cleandfm -r -D4 ..\..\Source' + #13+#10);
      exit;
    end
    else
    begin
      if (not SetCurrentDir(pasName)) then exit;
    end;
  end;

  writeln('  in directory:');
  writeln('    ' + GetCurrentDir);
  writeln('');
  writeln('This program, "cleandfm", can destroy your precious source code !');
  writeln('You have made a secure backup, have you not ?');
  writeln('Shall I proceed (y or n) ?');
  readln(Str);

  if (Lowercase(Str) = 'y') then
  begin
{Create working memory and text components:}
    TheMemory := TMemoryStream.Create;
    TheText := TStringList.Create;
    CurrentDir := GetCurrentDir;

    Log := Format('CleanDFM on %s:' + #13#10, [DateTimeToStr(Now)]);
    ProcessDir(CurrentDir);

    SetCurrentDir(CurrentDir);
    TheText.Text := Log;
    TheText.SaveToFile('Cleandfm.log');

{Final cleanup:}
    TheMemory.Free;
    TheText.Free;
  end;
  for i := 2 to 9 do
    NewProperties[i].Free;
end.
