unit Editgrp;

{$I Misc.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License Version 1.1 (the "License"); you may not use this file except in compliance with the License. You may obtain a copy of the License at

http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for the specific language governing rights and limitations under the License.

The Original Code is: CheckGroup.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 email: mat.ballard@chemware.hypermart.net.
                Robert Ross                 email: rross@sigmaofficesolutions.com

Last Modified: 05/25/2000
Current Version: 2.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 


Purpose:
Multi-select radio group


Known Issues:
-----------------------------------------------------------------------------}

{$I Misc.inc}

interface

uses
  Classes, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Graphics, Messages, Stdctrls
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Graphics, Messages, Stdctrls
{$ENDIF}
{$IFDEF LINUX}
  QT,
  QGraphics, QStdctrls
{$ENDIF}
  ;

const
  TEditGroup_VERSION = 100;

type
  TEditArray = array [0..1023] of TEdit;
  pEditArray = ^TEditArray;

  TOnBoxClick = procedure(Sender: TObject; Box: TEdit) of object;

  TEditGroup = class(TGroupBox)
  private
 { Private fields of TEditGroup }
 { Storage for property Items }
    FItems : TStringList;
    FNo_EditBoxes: Integer;
    FColumns: Byte;
    FOnBoxClick: TOnBoxClick;

 { Private methods of TEditGroup }
    procedure SetColumns(Value : Byte);

 { Write method for property Items }
    procedure SetItems(Value : TStringList);
    {procedure CMFontChanged(var Message: TMessage);}
    {procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;}
{$IFDEF MSWINDOWS}
    procedure WMSize(var Message: TMessage); message WM_SIZE;
{$ENDIF}
{$IFDEF LINUX}
{$ENDIF}
  protected
    {procedure WMPaint(var Message: TWMPaint); message WM_PAINT;)}
    procedure DoBoxClick(Sender: TObject); virtual;
    procedure Loaded; override;
    procedure SetNoEditBoxes(Value: Integer);
 { This arranges the EditBoxes on the panel }
    Procedure OnItemsChange(Sender: TObject); virtual;
  public
 { A list of the EditBoxes }
    EditBoxes: pEditArray;

 { Public methods of TEditGroup }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
 { This sets the size of the EditBoxes array, and creates the EditBoxes }
    procedure ArrangeEditBoxes;

  published
 { List of radio CheckBox names. The number of items sets the number o }
    property Columns: Byte read FColumns write SetColumns;
    property Items : TStringList read FItems write SetItems;
    property OnBoxClick: TOnBoxClick read FOnBoxClick write FOnBoxClick;
  end;

implementation

{------------------------------------------------------------------------------
    Procedure: TEditGroup.Create
  Description: standard constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: initializes Properties
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TEditGroup.Create(AOwner: TComponent);
begin
{ Call the Create method of the container's parent class     }
   inherited Create(AOwner);

   Width := 81;
   Height := 217;
   Font.Style := [fsBold];

   FColumns := 1;
   FItems := TStringList.Create;
   FItems.OnChange := OnItemsChange;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TEditGroup.Destroy;
begin
   FItems.Free;
   SetNoEditBoxes(0);

{ Last, free the component by calling the Destroy method of the  }
{ parent class.                          }
   inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.DoBoxClick
  Description: responds to a click event of a single checkbox
       Author: Mat Ballard
 Suggested by: Robert Ross
 Date created: 11/27/2000
Date modified: 11/27/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.DoBoxClick(Sender: TObject);
begin
  if Assigned(FOnBoxClick) then
    OnBoxClick(Self, TEdit(Sender));
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.Loaded
  Description: responds to a load event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: overrides the ancestor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.Loaded;
begin
  SetNoEditBoxes(FItems.Count);
  ArrangeEditBoxes;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.SetItems
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Items Property
 Known Issues: if the Items are changed programmatically, then
               the above routine is bypassed.
 ------------------------------------------------------------------------------}
procedure TEditGroup.SetItems(Value : TStringList);
begin
{ Use Assign method because TStrings is an object type }
  FItems.Assign(Value);

  SetNoEditBoxes(FItems.Count);
{ If changing this property affects the appearance of
     the component, call Invalidate here so the image will be
     updated. }
  Invalidate;
  ArrangeEditBoxes;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.OnItemsChange
  Description: standard change event handler for Items
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: rearranges the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
Procedure TEditGroup.OnItemsChange(Sender: TObject);
begin
  SetNoEditBoxes(FItems.Count);
  Invalidate;
  ArrangeEditBoxes;
end;

procedure TEditGroup.SetColumns(Value : Byte);
begin
  if (Value = 0) then Value := 1;
  FColumns := Value;
  ArrangeEditBoxes;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.SetNoEditBoxes
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: This sets the size of the EditBoxes array, and creates the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.SetNoEditBoxes(Value: Integer);
var
  i: Integer;
begin
  for i := 0 to FNo_EditBoxes-1 do
  begin
    EditBoxes^[i].Free;
  end;

  if (Value > 0) then
  begin
    EditBoxes := AllocMem(Value*SizeOf(TEdit));
  end
  else
  begin
    FreeMem(EditBoxes,
            FNo_EditBoxes*SizeOf(TEdit));
    EditBoxes := nil;
  end;

  FNo_EditBoxes := Value;

{create the EditBoxes:}
  for i := 0 to FNo_EditBoxes-1 do
  begin
    EditBoxes^[i] := TEdit.Create(Self);
    EditBoxes^[i].Parent := Self;
    EditBoxes^[i].Caption := FItems.Strings[i];
{Robert Ross suggests:}
    EditBoxes^[i].Tag := i;
    EditBoxes^[i].OnClick := DoBoxClick;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.ArrangeEditBoxes
  Description: This arranges the EditBoxes on the panel
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Display management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TEditGroup.ArrangeEditBoxes;
var
  i,
  NoRows,
  TheTop,
  TheLeft,
  TheWidth,
  TheGap,
  DeltaHeight: Integer;
begin
  if (FNo_EditBoxes > 0) then
  begin
{position the EditBoxes:}
    TheTop := 24;
    TheLeft := 8;
    TheGap := 12;
    TheWidth := Width - (FColumns+1)*TheLeft - 2;
    TheWidth := TheWidth div FColumns;
    NoRows := FNo_EditBoxes div FColumns;
    if ((NoRows * FColumns) < FNo_EditBoxes) then
      Inc(NoRows);
    DeltaHeight := (Height - TheTop) div NoRows;

    for i := 0 to FNo_EditBoxes-1 do
    begin
      EditBoxes^[i].Top := TheTop + (i div FColumns)*DeltaHeight;
      EditBoxes^[i].Left := TheLeft + (TheWidth + TheGap) * (i Mod FColumns);
      EditBoxes^[i].Width := TheWidth;
      EditBoxes^[i].Visible := TRUE;
      EditBoxes^[i].Invalidate;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TEditGroup.WMSize
  Description: standard ReSize message handler
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Re-Arranges the EditBoxes
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF MSWINDOWS}
procedure TEditGroup.WMSize(var Message: TMessage);
begin
  inherited;
  ArrangeEditBoxes;
end;
{$ENDIF}

{procedure TEditGroup.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ArrangeEditBoxes;
end;}

{procedure TEditGroup.WMPaint(var Message: TWMPaint);
var
  i: Integer;
begin
  inherited;
  Broadcast(Message);
end;}

end.

