unit Speed;

{$I Version.inc}

{$IFDEF WIN32}
  {$DEFINE TCHART}
{$ELSE}
  {.$ DEFINE TCHART} // not many people have TChart for Linux
{$ENDIF}

interface

uses
  SysUtils, Classes, TypInfo,
{$IFDEF WIN32}                                               
  Windows, Messages, Graphics, Controls, Forms, Dialogs,
  ImgList, Menus, Grids, ExtCtrls, StdCtrls,
  Buttons, ToolWin, ComCtrls,
{$ENDIF}
{$IFDEF LINUX}
  Types, Libc, 
  Qt, QTypes,
  QGraphics, QControls, QForms, QDialogs,
  QImgList, QMenus, QGrids, QExtCtrls, QStdCtrls,
  QButtons, QComCtrls,
{$ENDIF}

{$IFDEF TCHART}
  TeeProcs, TeEngine, Chart, Series,
{$ENDIF}
  Plot, Plotdefs, Plotmenu, Plotimagelist, Data, Plottoolbar, Nedit, Misc;

type
  TMainForm = class(TForm)
    Panel1: TPanel;
    MinNEdit: TNEdit;
    Label1: TLabel;
    Label2: TLabel;
    MaxNEdit: TNEdit;
    Label3: TLabel;
    Label4: TLabel;
    MeanNEdit: TNEdit;
    Label5: TLabel;
    StdDevNEdit: TNEdit;
    StatusBar1: TStatusBar;
    SpeedTestBitBtn: TBitBtn;
    ExitBitBtn: TBitBtn;
    StatsMemo: TMemo;
    NoPtsNEdit: TNEdit;
    ChartTimer: TTimer;
    procedure ClearAllBitBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure PlotMenu1ExitMenuItemClick(Sender: TObject);
    procedure SpeedTestBitBtnClick(Sender: TObject);
    procedure MyPlotFileOpen(Sender: TObject; TheFile: String);
    procedure Plot1AfterPaint(Sender: TObject; ACanvas: TCanvas);
    procedure Chart1AfterDraw(Sender: TObject);
    procedure DoNextFrame(Sender: TObject);
  private
    MyPlot: TPlot;
{$IFDEF TCHART}
    MyChart: TChart;
{$ENDIF}
    Revolutions,
    StartWidth,
    StartHeight: Integer;
    Angle,
    AngleInc: Single;
    StartTime: TDateTime;
    TypeClicked: Boolean;
    TPlotRevs, TChartRevs: Single;
    StartMem, TPlotMem, MiddleMem, TChartMem, EndMem: Integer;
    TPlotStartUpTime, TPlotRunTime, TChartStartUpTime, TChartRunTime: TDateTime;
  public
    procedure CreateTPlot;
{$IFDEF TCHART}
    procedure CreateTChart;
{$ENDIF}

  end;

var
  MainForm: TMainForm;

implementation

{$R *.dfm}

{Kylix seems to have a versioninfo resource already.
 Unfortunately, it is empty.}
{.$ R version.res}

const
  RADIUS = 50.0;
{$IFDEF WIN32}
  MYCAPTION = 'TPlot Speed Demo for Delphi';
{$ENDIF}
{$IFDEF LINUX}
  MYCAPTION = 'TPlot Speed Demo for Kylix';
{$ENDIF}

procedure TMainForm.FormCreate(Sender: TObject);
begin
  TypeClicked := FALSE;
  Height := 516;
  Width := 712;

  Self.Caption := MYCAPTION;
  StatsMemo.Align := alClient;
  //StatsMemo.Lines.Text := 'No Pts        init       fps      bytes     leak      init       fps      bytes     leak';
  StatsMemo.Font.Name := 'Courier New';
  StatsMemo.Font.Size := 10;
  StatsMemo.ScrollBars := ssHorizontal;
end;

procedure TMainForm.ClearAllBitBtnClick(Sender: TObject);
begin
  MyPlot.Clear(FALSE);
end;

procedure TMainForm.CreateTPlot;
var
  NoPts: Longword;
  X, Y: Single;
  Mean: Single;
  StdDev: Single;
  Min: Single;
  Max: Single;
  StepSize: Single;
begin
  Screen.Cursor := crHourGlass;
  Mean := MeanNEdit.AsReal;
  StdDev := StdDevNEdit.AsReal;
  Min := MinNEdit.AsReal;
  Max := MaxNEdit.AsReal;
  NoPts := NoPtsNEdit.AsInteger;
  StepSize := (Max - Min) / NoPts;

  StartTime := Now;
  MyPlot := TPlot.Create(Self);
  MyPlot.Parent := Self;
  MyPlot.Align := alClient;
  MyPlot.PlotType := ptXY;
  MyPlot.SaveOptions := [soProperties]; // soAsText
  MyPlot.OnFileOpen := MyPlotFileOpen;
  MyPlot.NoSeries := 1;

{Set the axes:}
  MyPlot.XAxis.Max := Max;
  MyPlot.XAxis.Min := Min;
  MyPlot.YAxis.Min := 0;
  MyPlot.XAxis.Intercept := 0;

  X := Min;
  while (X <= Max) do
  begin
    Y := Exp(-Sqr((X-Mean)/(2*StdDev))) /
      Sqrt(2*Pi*StdDev);
{Don't fire any events, and don't adjust axes:}
    MyPlot.Series[0].AddPoint(X, Y, FALSE, FALSE);
    X := X + StepSize;
  end;
  //MyPlot.Series[0].Visible := TRUE;

  MyPlot.YAxis.Max := MyPlot.Series[0].YMax;
  TPlotStartUpTime := Now - StartTime;
  MyPlot.Instructions.Clear;

  Self.Caption := Format('TPlot - %d points', [NoPts]);
  Screen.Cursor := crDefault;
end;

{$IFDEF TCHART}
procedure TMainForm.CreateTChart;
var
  NoPts: Longword;
  ASeries: TFastLineSeries;
  X, Y: Single;
  Mean: Single;
  StdDev: Single;
  Min: Single;
  Max: Single;
  StepSize: Single;
begin
  Screen.Cursor := crHourGlass;
  Mean := MeanNEdit.AsReal;
  StdDev := StdDevNEdit.AsReal;
  Min := MinNEdit.AsReal;
  Max := MaxNEdit.AsReal;
  NoPts := NoPtsNEdit.AsInteger;
  StepSize := (Max - Min) / NoPts;

  StartTime := Now;
  MyChart := TChart.Create(Self);
  MyChart.Parent := Self;
  MyChart.Align := alClient;

  ASeries := TFastLineSeries.Create(Self);
  ASeries.ParentChart := MyChart;
{Set the axes:}
  MyChart.BottomAxis.Maximum := Max;
  MyChart.BottomAxis.Minimum := Min;
  MyChart.LeftAxis.Minimum := 0;
  //MyChart.LeftAxis.Intercept := 0;

  X := Min;
  while (X <= Max) do
  begin
    Y := Exp(-Sqr((X-Mean)/(2*StdDev))) /
      Sqrt(2*Pi*StdDev);
    ASeries.AddXY(X, Y);
    X := X + StepSize;
  end;

  {MyChart.YAxis.Min := MyChart.Series[0].YMin;}
  MyChart.LeftAxis.Maximum := MyChart.Series[0].MaxYValue;
  TChartStartUpTime := Now - StartTime;


  Self.Caption := Format('TChart - %d points', [NoPts]);
  Screen.Cursor := crDefault;
end;
{$ENDIF}

procedure TMainForm.PlotMenu1ExitMenuItemClick(Sender: TObject);
begin
  Close;
end;

{0 - initial, nothing happening
 1 - Create and start TPlot running
 2 - Stop and Free TPlot, Create and Start TChart}
procedure TMainForm.SpeedTestBitBtnClick(Sender: TObject);
var
  Hour, Min, Sec, MSec: Word;
  TPlotInit, TChartInit: Single;

  function GetMemUsage: Integer;
  var
{$IFDEF WIN32}
    HeapStatus: THeapStatus;
  begin
    HeapStatus := GetHeapStatus; // HeapStatus.TotalAddrSpace or TotalAllocated ?
    Result := HeapStatus.TotalAllocated;
{$ENDIF}
{$IFDEF LINUX}
    usage: TRUsage;
  begin
    Result := 0;
{NB: getrusage returns 0 for all the memory members; how does TOP then work ?}
{NNB: use "top -p nnnn", where nnnn is the pid of this process.}
    if (0 = Libc.getrusage (RUSAGE_SELF, usage)) then
    begin
      Result := usage.ru_ixrss;
    end;
{$ENDIF}
  end;

begin
  if (SpeedTestBitBtn.Tag = 0) then
  begin
    SpeedTestBitBtn.Tag := 1;
    Height := 516;
    Width := 712;
    StatsMemo.Visible := FALSE;
    Application.ProcessMessages;
    StartMem := GetMemUsage;
    CreateTPlot;
    Revolutions := 0;
    StartWidth := Width;
    StartHeight := Height;
    Angle := 0;
    AngleInc := 4 * Pi / 180;
    SpeedTestBitBtn.Caption := 'Stop TPlot !';
    MyPlot.OnAfterPaint := Plot1AfterPaint;
    StartTime := Now;
    MyPlot.Invalidate;
  end
  else if (SpeedTestBitBtn.Tag = 1) then
  begin
    TPlotRunTime := Now - StartTime;
    TPlotMem := GetMemUsage;
    TPlotRevs := Revolutions;
    MyPlot.Free;
    MiddleMem := GetMemUsage;
    SpeedTestBitBtn.Tag := 2;
    Height := 516;
    Width := 712;
    Application.ProcessMessages;
    TChartRunTime := Now - StartTime;
{$IFDEF TCHART}
    CreateTChart;
    Revolutions := 0;
    StartWidth := Width;
    StartHeight := Height;
    Angle := 0;
    AngleInc := 4 * Pi / 180;
    SpeedTestBitBtn.Caption := 'Stop TChart !';
    //MyChart.OnAfterDraw := Chart1AfterDraw;
    StartTime := Now;
    ChartTimer.Enabled := TRUE;
  end
  else if (SpeedTestBitBtn.Tag = 2) then
  begin
    ChartTimer.Enabled := FALSE;
    TChartRunTime := Now - StartTime;
    TChartMem := GetMemUsage;
    TChartRevs := Revolutions;
    MyChart.Free;
{$ENDIF}
    EndMem := GetMemUsage;
    SpeedTestBitBtn.Tag := 0;
    Height := 516;
    Width := 712;
    SpeedTestBitBtn.Caption := 'Speed Test';
{Calculate statistics:}
    DecodeTime(TPlotRunTime, Hour, Min, Sec, MSec);
    TPlotRevs := TPlotRevs / (3600 * Hour + 60 * Min + Sec + MSec / 1000);
    DecodeTime(TPlotStartUpTime, Hour, Min, Sec, MSec);
    TPlotInit := 3600 * Hour + 60 * Min + Sec + MSec / 1000;
    DecodeTime(TChartRunTime, Hour, Min, Sec, MSec);
    TChartRevs := TChartRevs / (3600 * Hour + 60 * Min + Sec + MSec / 1000);
    DecodeTime(TPlotStartUpTime, Hour, Min, Sec, MSec);
    TChartInit := 3600 * Hour + 60 * Min + Sec + MSec / 1000;
{Display:}
    StatsMemo.Lines.Add(Format('%8d  %8.2f  %8.2f  %8d  %8d  %8.2f  %8.2f  %8d  %8d',
      [NoPtsNEdit.AsInteger,
       TPlotInit, TPlotRevs, TPlotMem - StartMem, MiddleMem - StartMem,
       TChartInit, TChartRevs, TChartMem - MiddleMem, EndMem - MiddleMem]));
    StatsMemo.Visible := TRUE;
  end;
end;

procedure TMainForm.DoNextFrame(Sender: TObject);
var
  fps: Single;
  ElapsedTime: TDateTime;
begin
  Width := StartWidth + Round(RADIUS * Sin(Angle));
  Height := StartHeight + Round(RADIUS * Cos(Angle));
  Angle := Angle + AngleInc;
  Inc(Revolutions);
  ElapsedTime := Now - StartTime;
  if (ElapsedTime > 0) then
    fps := Revolutions / ((24 * 3600)*ElapsedTime)
   else
    fps := 0;
  StatusBar1.SimpleText := Format(
    '%d frames, %8.2f frames per second',
    [Revolutions, fps]);
{the following line causes the most bizzare behaviour under Wk2:
 the program freezes after 29 frames !
  Application.ProcessMessages;}
end;

procedure TMainForm.MyPlotFileOpen(Sender: TObject; TheFile: String);
var
  TheTitle: String;
begin
  TheTitle := ExtractFileName(Application.ExeName);
  TheTitle := Copy(TheTitle, 1, Length(TheTitle)-4);
  TheTitle := TheTitle + ' - ' + ExtractFileName(TheFile);
  Application.Title := TheTitle;
  MainForm.Caption := TheTitle;
end;

procedure TMainForm.Plot1AfterPaint(Sender: TObject; ACanvas: TCanvas);
begin
  Application.ProcessMessages;
  DoNextFrame(Sender);
end;

procedure TMainForm.Chart1AfterDraw(Sender: TObject);
begin
  Application.ProcessMessages;
  DoNextFrame(Sender);
end;

end.
