unit Axis;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Axis.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 04/09/2001
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Purpose:
To implement an Axis component for use by the main TPlot graphing component.

Known Issues:

History:
 1.01 21 September 2000: fix mFont.cx bug in TAxis.Draw
        add LabelText property to TAxis (for columns)
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils,
{$IFDEF GUI}
  {$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Graphics,
  {$ENDIF}
  {$IFDEF WIN32}
  Windows,
  Graphics,
  {$ENDIF}
  {$IFDEF LINUX}
  Types,
  QGraphics,
  {$ENDIF}

{$ELSE}
  gd, gd_io, gdWindows, gdGraphics, {gdWinapi,}
{$ENDIF}
  {$IFDEF NO_MATH}NoMath,{$ELSE}Math,{$ENDIF}
  Misc, Plotdefs, Titles;

type
  TOnOutOfBounds = procedure(Sender: TObject; iX, iY: Integer) of object;

  TAxisType = (atNone, atPrimaryX, atZ, atPrimaryY, atSecondaryY, atTertiaryY);

  TLabelFormat = (
    lfGeneral, lfExponent, lfFixed, lfNumber, lfCurrency,
    lfSI, lfPercent,
    lfTime, lfDate, lfDateTime, lfElapsedTime, lfElapsedDate);
{lfGeneral ... lfCurrency are just TFloatFormat.}
{}
{We then add SI and Percentage, then the rest are so that we can display times in various formats.}
{}
{NOTE: SI means the standard SI postfixes: p, n u, m, -, K, M, G, T}
{}
{NOTE 2: we have removed the old lfSeconds, etc time units and replaced them with
 more general time units based on interpreting the StartTime, and the XData
 being (Now - StartTime).}

{  TOnPositionChangeEvent = procedure(
    Sender: TObject;
    bIntercept: Boolean; did the Intercept change ? or the screen position ?
    var TheIntercept: Single;
    var ThePosition: Integer) of object;}

  TAxis = class;

{Begin TAxisLabel declarations ------------------------------------------------}
  TAxisLabel = class(TText)
  private
    FAxis: TAxis;
    FDefaultSide: Boolean;
    FDigits: Byte;
    FPrecision: Byte;
    FNumberFormat: TLabelFormat;

  protected
    procedure SetDefaultSide(Value: Boolean);
    procedure SetDigits(Value: Byte);
    procedure SetPrecision(Value: Byte);
    procedure SetNumberFormat(Value: TLabelFormat);

  public
    Property Axis: TAxis read FAxis;
{What Axis is this label linked to ? If any (could be global title).}
    Constructor Create(Axis: TAxis); reintroduce;
{The standard constructor, where standard properties are set.}
    Destructor Destroy; override;
{The standard destructor, where the OnChange event is "freed".}

    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;

  published
    property DefaultSide: Boolean read FDefaultSide write SetDefaultSide default TRUE;
{Is the caption on the Default Side ?}
{}
{Note that the "Default Side" of a Plot Title it the Top, and that of an Axis
 depends on the Angle and the screen location of the Axis.}
    Property Digits: Byte read FDigits write SetDigits default 1;
{This (and Precision) control the numeric format of the Axis Labels.
 See the Borland documentation on FloatToStrF for the precise meaning of
 this property, or simply experiment in the IDE designer.}
    Property Precision: Byte read FPrecision write SetPrecision default 3;
{This (and Digits) control the numeric format of the Axis Labels.
 See the Borland documentation on FloatToStrF for the precise meaning of
 this property, or simply experiment in the IDE designer.}
    Property NumberFormat: TLabelFormat read FNumberFormat write SetNumberFormat default lfGeneral;
{This property controls how the numbers of the Axis labels are displayed.}
  end;

{Begin TAxis declarations ---------------------------------------------------}
  TAxis = class(TAngleRect)
  private
    FArrowSize: Byte;
    FAutoScale: Boolean;
    FAutoTick: Boolean;
    FAutoZero: Boolean;
    FAxisType: TAxisType;
    FdTick: TPoint;
    FIntercept: Single;
    FLabels: TAxisLabel;
    FLabelSeries: TPersistent;
    FLimitLower: Single;
    FLimitUpper: Single;
    FLimitsVisible: Boolean;
    FLogScale: Boolean;
    FMin: Single;
    FMax: Single;
    FNewMin: Single;
    FNewMax: Single;
    FPen: TPen;
    FStepSize: Single;
    FStepStart: Single;
    FTickMinor: Byte;
    //FTickSign: Integer;
    FTickSize: Byte;
    //FTickDirection: TOrientation;
    FTitle: TTitle;
    //FTitleVector: TPoint;

    FZoomIntercept: Single;
    FZoomMin: Single;
    FZoomMax: Single;
    FZInterceptY: Single;

    FOnOutOfBounds: TOnOutOfBounds;

    mSinM30,
    mCosM30,
    mSinP30,
    mCosP30: Extended;
    mSlope, mIntercept_b: Single; {y := mSlopex + mIntercept_b}

    mLogSpan: Single;
    mPrecisionAdded: Integer;
    mSpan: Single;
    mTickNum: Byte;

{Calculated in Draw, but used in other places:}
    mChar: TSize;
    mFont: TSize;

  protected
    function GetTick(ADefaultSide: Boolean): TPoint;
{Set procedures:}
    procedure SetAngle(Value: TDegrees); override;
    procedure SetArrowSize(Value: Byte);
    procedure SetAutoScale(Value: Boolean);
    procedure SetAutoTick(Value: Boolean);
    procedure SetAutoZero(Value: Boolean);
    //procedure SetDirection(Value: TDirection);
    procedure SetIntercept(Value: Single); virtual;
    procedure SetLimitLower(Value: Single);
    procedure SetLimitUpper(Value: Single);
    procedure SetLimitsVisible(Value: Boolean);
    procedure SetLogScale(Value: Boolean);
    procedure SetMin(Value: Single);
    procedure SetMax(Value: Single);
    procedure SetPen(Value: TPen);
    procedure SetStepSize(Value: Single);
    procedure SetStepStart(Value: Single);
    procedure SetTickMinor(Value: Byte);
    {procedure SetTickNum(Value: Byte);}
    procedure SetTickSize(Value: Byte); virtual;
    //procedure SetOrientation(Value: TOrientation);
    procedure SetAxisType(Value: TAxisType);
    procedure SetZInterceptY(Value: Single);

    procedure StyleChange(Sender: TObject); virtual;
    procedure TitleChange(Sender: TObject); virtual;

  public
    procedure DoGeometry; override;
{This calculates the slope and intercept, which are needed for Title and Label Management.}
    procedure DoTicks;
{This methods calculate the default tick vector (direction).}
    procedure ReScale;
{This method recalculates the Stepsize and Stepstart, etc}
    function SameSideAs(X, Y: Integer; ACentre: TPoint): Boolean;
{Are (X, Y) and ACentre on the same side of this Axis ?}
    function IsDefaultSide(X, Y: Integer): Boolean;
{Is (X, Y) on the default side of this Axis ?}
    function Reflect(ACentre: TPoint): TPoint;
{This reflects the point ACentre across the axis.}
    procedure SetMinMax(AMin, AMax: Single);
{This sets the Min and Max simultaneously.}
    procedure ClearNewMinMax;
    procedure StoreNewMinMax(AMin, AMax: Single);
{This stores new values for the Min and Max, if they exceed current ones.}
    procedure UseNewMinMax;
    property Height;
{The standard Height property.}
    property Width;
{The standard Width property.}
    property dTick: TPoint read FdTick;
{dTick is a vector that gives the default direction of the ticks on the Axis.}
    //Property TitleVector: TPoint read FTitleVector;
    Property ZoomIntercept: Single read FZoomIntercept write FZoomIntercept;
{The (old) ZOOMED OUT Intercept in data co-ordinates.}
    Property ZoomMin: Single read FZoomMin write FZoomMin;
{The (old) ZOOMED OUT minimum, Left or Bottom of the Axis, in data co-ordinates.}
    Property ZoomMax: Single read FZoomMax write FZoomMax;
{The (old) ZOOMED OUT maximum, Right or Top of the Axis, in data co-ordinates.}

    Constructor Create(AOwner: TCollection); override;
{The standard constructor, where sub-components are created, and standard
 properties are set.}

    Destructor Destroy; override;
{The standard destructor, where sub-components and the OnChange event is "freed".}

    procedure Draw(ACanvas: TCanvas; Vector: TPoint); reintroduce;
{This draws the Axis on the given Canvas.}
    function GetTitleCentre(APosition: Integer; ADefaultSide: Boolean): TPoint;
{This returns the position of the title, given the TitleVector and Position.}
    function EndClickedOn(iX, iY, Radius: Integer): Boolean;
{Was the end of this Axis clicked on ?}
    function GetNextXValue(XValue: Single): Single;
{This calculates the next tick point. Used externally by TCustomPlot.DrawGrid}
    function LabelToStrF(Value: Single): String;
{This method converts a number to a string, given the current Labels' NumberFormat.}
    function StrToLabel(Value: String): Single;
{This method converts a string to a number, given the current Labels' NumberFormat.}
    function dFofZ(Z: Single): TPoint;
{This returns the change in pixel position on screen as a function of the real data ordinate Z.}
    function FofX(X: Single): Integer;
{This converts an X data value to a screen X co-ordinate.}
    function FofY(Y: Single): Integer;
{This converts a Y data value to a screen Y co-ordinate.}
    function XofF(F: Integer): Single;
{This converts a screen X co-ordinate to a X data value.}
    function YofF(F: Integer): Single;
{This converts a screen Y co-ordinate to a Y data value.}
    procedure SetLabelSeries(Value: TPersistent);
{This is called by a series to set the X data as strings.}
    procedure SetMinFromSeries(Value: Single);
{This sets the Min property of the Axis. It is used exclusively by TSeries.}
    procedure SetMaxFromSeries(Value: Single);
{This sets the Max property of the Axis. It is used exclusively by TSeries.
 Exactly how it affects the Axis depends on TPlot.DisplayMode.}
    procedure SetMinMaxFromSeries(AMin, AMax: Single);
{This sets the Min and Max properties of the Axis. It is used exclusively by TSeries.
 Exactly how it affects the Axis depends on TPlot.DisplayMode.}

    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;

  published
    Property ArrowSize: Byte read FArrowSize write SetArrowSize default 0;
{This is the size (in pixels) of the arrowhead on the Axis.}
    Property AutoScale: Boolean read FAutoScale write SetAutoScale default TRUE;
{Do we use the StepSize property or does TPlot work them out ?}
    Property AutoTick: Boolean read FAutoTick write SetAutoTick default TRUE;
{Do we use the StepSize property or does TPlot work them out ?}
    Property AutoZero: Boolean read FAutoZero write SetAutoZero default TRUE;
{Do we automatically have an intercept of zero ?}
    Property AxisType: TAxisType read FAxisType write SetAxisType;
{What sort of axis is this ?}
    Property Title: TTitle read FTitle write FTitle;
{The Title on and of the Axis. Note that the Title can be clicked and dragged
 around the Axis.}
    //Property Direction: TDirection read FDirection write SetDirection;
{Is the Axis Horizontal (X) or Vertical (Y or Y2).}
    Property Intercept: Single read FIntercept write SetIntercept;
{The intercept of this Axis on the complementary Axis.}
    Property Labels: TAxisLabel read FLabels write FLabels;
{The numerals on the Axis.}
    property LimitLower: Single read FLimitLower write SetLimitLower;
{The lower limit, drawn perpendicular to the axis with a dashed line, if LimitsVisible.}
    property LimitUpper: Single read FLimitUpper write SetLimitUpper;
{The upper limit, drawn perpendicular to the axis with a dashed line, if LimitsVisible.}
    property LimitsVisible: Boolean read FLimitsVisible write SetLimitsVisible default FALSE;
{Determines if Limits perpendicular to the Axis are drawn.}
    Property LogScale: Boolean read FLogScale write SetLogScale default FALSE;
{Is this Axis on a logarithmic scale ?}
    Property Min: Single read FMin write SetMin;
{The minimum, Left or Bottom of the Axis, in data co-ordinates.}
    Property Max: Single read FMax write SetMax;
{The maximum, Right or Top of the Axis, in data co-ordinates.}
    Property Pen: TPen read FPen write SetPen;
{The Pen that the Axis is drawn with.}
    Property StepSize: Single read FStepSize write SetStepSize;
{The interval between tick (and labels) on the Axis.}
{}
{If the axis is a Log Scale, then this is the multiple, not the interval !}
    Property StepStart: Single read FStepStart write SetStepStart;
{The position of the first tick (and labels) on the Axis.}
    Property TickMinor: Byte read FTickMinor write SetTickMinor default 1;
{Sets the number of minor ticks between labels.}
    Property TickSize: Byte read FTickSize write SetTickSize default 10;
{The Length of the Ticks, in screen pixels.}
    //Property TickDirection: TOrientation read FTickDirection write SetOrientation;
{Are the Ticks to the left or right of the Axis ?}
    property ZInterceptY: Single read FZInterceptY write SetZInterceptY;

    property OnOutOfBounds: TOnOutOfBounds read FOnOutOfBounds write FOnOutOfBounds;

  end;


implementation

uses
  Data, Plot;

const
  ONE_SIXTH = 1/6;
    
{TAxislabel methods ---------------------------------------------------------}
{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TAxisLabel.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Precision and Digits Properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TAxisLabel.Create(Axis: TAxis);
begin
{First call the ancestor, but with nil: AxisLabels do not belong to a Collection.}
  inherited Create(nil);

  FAxis := Axis;
{Put your own initialisation (memory allocation, etc) here:}

{we insert the default values that cannot be "defaulted":}
  FDefaultSide := TRUE;
  FDigits := 1;
  FPrecision := 3;
end;

{------------------------------------------------------------------------------
  Destructor: TAxisLabel.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TAxisLabel.Destroy;
begin
  OnChange := nil;
{Put your de-allocation, etc, here:}

{then call ancestor:}
 inherited Destroy;
end;

{End Constructor and Destructor:---------------------------------------------}

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.AssignTo
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TAxisLabel(Dest).DefaultSide := FDefaultSide;
  TAxisLabel(Dest).Digits := FDigits;
  TAxisLabel(Dest).NumberFormat := FNumberFormat;
  TAxisLabel(Dest).Precision := FPrecision;
end;

{Begin Set Procedures --------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetDefaultSide
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/25/2002
Date modified: 08/25/2002 by Mat Ballard
      Purpose: sets the Side (of the axis) on which text is drawn
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetDefaultSide(Value: Boolean);
begin
  if (Value <> FDefaultSide) then
  begin
    FDefaultSide := Value;
    FAxis.DoTicks;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetDigits
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Digits Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetDigits(Value: Byte);
begin
  if ((FDigits <> Value) and (Value <= 18)) then
  begin
    case FNumberFormat of
      lfGeneral:  if (FDigits > 4) then exit;
      lfExponent: if (FDigits > 4) then exit;
    end;
    FDigits := Value;

    if Assigned(OnChange) then OnChange(Self);
  end;  
end;

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetPrecision
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Precision Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetPrecision(Value: Byte);
begin
  if ((FPrecision <> Value) and (Value <= 7)) then
  begin
    FPrecision := Value;
    if Assigned(OnChange) then OnChange(Self);
  end;  
end;

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetNumberFormat
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the NumberFormat Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetNumberFormat(Value: TLabelFormat);
begin
  if (FNumberFormat <> Value) then
  begin
    FNumberFormat := Value;
    case FNumberFormat of
      lfGeneral:  if (FDigits > 4) then FDigits := 4;
      lfExponent: if (FDigits > 4) then FDigits := 4;
    end;

    if Assigned(OnChange) then OnChange(Self);
  end;
end;

{TAxis methods --------------------------------------------------------------}
{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TAxis.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates the subcomponents and sets various Properties
 Known Issues: we DO NOT call the inherited constructor first;
               if we do, then dTick is calculated before FTickSize is set;
               so we set the simple properties first !
 ------------------------------------------------------------------------------}
Constructor TAxis.Create(AOwner: TCollection);
begin
{make arrows invisible:}
  FArrowSize := 0;
  FAutoScale := TRUE;
  FAutoTick := TRUE;
  FAutoZero := TRUE;
  FIntercept := 0;
  FMin := 0;
  FMax := 10;
  FLimitLower := 3;
  FLimitUpper := 7;
  FTickMinor := 1;
  FTickSize := 10;
  mTickNum := 5;

{Last call the ancestor:}
  inherited Create(AOwner);

{Create Pen:}
  FPen := TPen.Create;
  FPen.Color := clRed;
  FPen.OnChange := StyleChange;

  FLabels := TAxisLabel.Create(Self);
  FLabels.OnChange := StyleChange;
  FLabelSeries := nil;

{create the Title geometry manager:}
  FTitle := TTitle.Create(Self);
  FTitle.OnChange := StyleChange;
  FTitle.OnCaptionChange := TitleChange;
  //FTitle.Caption := 'X-Axis';
  FTitle.Font.Size := MEDIUM_FONT_SIZE;
  FTitle.FireEvents := TRUE;
  //FTitle.DoGeometry;

  Visible := TRUE;
  ReScale;
end;

{------------------------------------------------------------------------------
  Destructor: TAxis.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the subcomponents and the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TAxis.Destroy;
begin
{$IFDEF COMPILER5_UP}
{TODO 3 -o Delphi 4 Expert -c why is csDestroying _NOT_ in ComponentState when TPlot is deleted from a form ?}
  if not ((FAxisType = atZ) or
      (FAxisType > atPrimaryY) or
      (csDestroying in TAxisList(Collection).Plot.ComponentState) or
      (csReading in TAxisList(Collection).Plot.ComponentState)) then
    raise EComponentError.Create('You cannot destroy a Primary Axis, you Fiend !');
{$ENDIF}

{is a valid destruction of this axis:}
  OnChange := nil;
{Put your de-allocation, etc, here:}
  FLabels.Free;
  FPen.Free;
  FTitle.Free;

{then call ancestor:}
  inherited Destroy;
end;
{End Constructor and Destructor:---------------------------------------------}

{------------------------------------------------------------------------------
    Procedure: TAxis.TitleChange
  Description: sets the Name and Label's Name
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to a change in the Title
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.TitleChange(Sender: TObject);
begin
  if (Pos('xis', FTitle.Caption) > 0) then
  begin
    Name := FTitle.Caption;
    FLabels.Name := FTitle.Caption + ' Labels';
  end
  else
  begin
//Stick Axis in in the names:
    Name := FTitle.Caption + ' Axis';
    FLabels.Name := FTitle.Caption + ' Axis Labels';
  end;
end;

{Begin normal Set Procedures -------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAxis.SetAngle
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Angle Property: clockwise angle between vertical and long axis
 Known Issues: Windows defines angles for fonts ANTICLOCKWISE in 0.1 increments from X Axis;
 ------------------------------------------------------------------------------}
procedure TAxis.SetAngle(Value: TDegrees);
begin
  if (Angle <> Value) then
  begin
    inherited SetAngle(Value);
{look back along the axis, then 30 degrees less, for the arrow:}
    SinCos(mAngleRadians + Pi*(0.5 - ONE_SIXTH), mSinM30, mCosM30);
{look back along the axis, then 30 degrees more:}
    SinCos(mAngleRadians + Pi*(0.5 + ONE_SIXTH), mSinP30, mCosP30);
    if (Assigned(FLabels)) then
      FLabels.Angle := Value;
    if (Assigned(FTitle)) then
    begin
      if (Value < 180) then
        FTitle.Angle := Value
       else
        FTitle.Angle := Value - 180;
    end;
    DoTicks;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.DoGeometry
  Description: standard geometry calculator
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Angle Property: clockwise angle between vertical and long axis
 Known Issues: Windows defines angles for fonts ANTICLOCKWISE in 0.1 increments from X Axis;
 ------------------------------------------------------------------------------}
procedure TAxis.DoGeometry;
begin
  inherited DoGeometry;
  if (Origin.x = EndX) then
  begin
    mSlope := 1.0e38;
    mIntercept_b := 0;
  end
  else
  begin
    mSlope := (Origin.y - EndY) / (Origin.x - EndX);
    mIntercept_b := Origin.y - mSlope * Origin.x;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.DoTicks
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Angle Property: clockwise angle between vertical and long axis
 Known Issues: Windows defines angles for fonts ANTICLOCKWISE in 0.1 increments from X Axis;
 ------------------------------------------------------------------------------}
procedure TAxis.DoTicks;
begin
{Ticks on the axis:}
  FdTick.x := 0;
  FdTick.y := 0;
  case Angle of
    0: // Y Axis
      begin
        if (MidX = TAxisList(Collection).Plot.Border.Right) then // Y Axis on Right
          FdTick.x := FTickSize
         else
        FdTick.x := -FTickSize;
      end;
    1 .. 60: FdTick.x := FTickSize;
    61 .. 120:
      begin
        if ((Angle = 90) and  // X Axis
            (MidY = TAxisList(Collection).Plot.Border.Top)) then // X Axis on Top
          FdTick.y := -FTickSize
         else
          FdTick.y := FTickSize;
      end;
    121 .. 180: FdTick.x := FTickSize;
    181 .. 240: FdTick.x := -FTickSize;
    241 .. 300: FdTick.y := FTickSize;
    301 .. 359: FdTick.x := -FTickSize;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetArrowSize
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the ArrowSize Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetArrowSize(Value: Byte);
begin
  if (Value <> FArrowSize) then
  begin
    FArrowSize := Value;
    StyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetAutoScale
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 07/25/2002 by Mat Ballard
      Purpose: sets the AutoScale Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetAutoScale(Value: Boolean);
begin
  if (Value <> FAutoScale) then
  begin
    FAutoScale := Value;
    ReScale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetAutoTick
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 07/25/2002
Date modified: 07/25/2002 by Mat Ballard
      Purpose: sets the AutoTick Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetAutoTick(Value: Boolean);
begin
  if (Value <> FAutoTick) then
  begin
    FAutoTick := Value;
    Rescale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetAutoZero
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the AutoZero Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetAutoZero(Value: Boolean);
begin
  if (Value <> FAutoZero) then
  begin
    FAutoZero := Value;
    StyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetIntercept
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Intercept virtual Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetIntercept(Value: Single);
begin
  if (FIntercept <> Value) then
  begin
    if not (csLoading in TAxisList(Collection).Plot.ComponentState) then
      FAutoZero := FALSE;
    FIntercept := Value;
    StyleChange(Self);
  end;
end;

procedure TAxis.SetLimitLower(Value: Single);
begin
  if (FLimitLower <> Value) then
  begin
    FLimitLower := Value;
    if (FLimitLower > FLimitUpper) then
      FLimitUpper := Value;
    StyleChange(Self);
  end;
end;

procedure TAxis.SetLimitUpper(Value: Single);
begin
  if (FLimitUpper <> Value) then
  begin
    FLimitUpper := Value;
    if (FLimitUpper < FLimitLower) then
      FLimitLower := Value;
    StyleChange(Self);
  end;
end;

procedure TAxis.SetLimitsVisible(Value: Boolean);
begin
  if (FLimitsVisible <> Value) then
  begin
    FLimitsVisible := Value;
    StyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetLogScale
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 05/25/2002 by Mat Ballard
      Purpose: sets the LogScale Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetLogScale(Value: Boolean);
begin
  if (Value <> FLogScale) then
  begin
    if (Value) then
    begin {we are going to a log scale:}
      if (FMax <= 0) then exit;
      if (FMin <= 0) then
        FMin := FMax / 1000;
    end;

    FLogScale := Value;
    ReScale;
  end;
end;

procedure TAxis.ClearNewMinMax;
begin
  FNewMin := 0.0;
  FNewMax := 0.0;
end;

procedure TAxis.StoreNewMinMax(AMin, AMax: Single);
begin
  if (FNewMin > AMin) then
    FNewMin := AMin;
  if (FNewMax < AMax) then
    FNewMax := AMax;
end;
{This stores new values for the Min and Max, if they exceed current ones.}

procedure TAxis.UseNewMinMax;
begin
  if (FNewMin = FNewMax) then
    FNewMax := FNewMax + 10;
  SetMinMax(FNewMin, FNewMax);  
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMinMax
  Description: This sets the Min and Max simultaneously
       Author: Mat Ballard
 Date created: 10/25/2003
Date modified: 10/25/2003 by Mat Ballard
      Purpose: sets the Min and Max Properties
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMinMax(AMin, AMax: Single);
begin
  if (AMin < AMax) then
  begin
    if ((AMin <= 0) and (FLogScale)) then
    begin
      AMin := FMin / 1000;
      if ((AMax <= 0) and (FLogScale)) then
        AMax := FMax / 1000;
    end;
    FMin := AMin;
    FMax := AMax;
    ReScale;
  end
  else if (AMin = AMax) then
  begin
    SetMinMax(AMin-1, AMax+1);
  end
   else
    SetMinMax(AMax, AMin);
end;
{------------------------------------------------------------------------------
    Procedure: TAxis.SetMin
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Min Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMin(Value: Single);
begin
  if ((Value <> FMin) and (Value < FMax)) then
  begin
    if not (csLoading in TAxisList(Collection).Plot.ComponentState) then
      FAutoScale := FALSE;
    if ((Value <= 0) and (FLogScale)) then
      Value := FMax / 1000;

    FMin := Value;
    ReScale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMax
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Max Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMax(Value: Single);
begin
  if ((Value <> FMax) and (Value > FMin)) then
  begin
    if not (csLoading in TAxisList(Collection).Plot.ComponentState) then
      FAutoScale := FALSE;
    FMax := Value;
    ReScale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMinFromSeries
  Description: property Setting procedure for calling by a Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Min Property when new data is added to a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMinFromSeries(Value: Single);
begin
  if (Value < FMin) then
  begin
    if ((Value > 0) or (not FLogScale)) then
    begin
      FMin := Value;
      Rescale;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMaxFromSeries
  Description: property Setting procedure for calling by a Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Max Property when new data is added to a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMaxFromSeries(Value: Single);
begin
  if (Value > FMax) then
  begin
    FMax := Value;
    if ((TPlot(TAxisList(GetOwner).Plot).DisplayMode = dmRun)) then
    begin
  {We are in a "run", and so we can expect more data with increasing X values.
   Rather than force a complete screen re-draw every time a data point is
   added, we extend the X Axis by 100%:}
      FMax := 2.0 * FMax;
    end;
    Rescale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMinMaxFromSeries
  Description: multiple property Setting procedure for calling by a Series
       Author: Mat Ballard
 Date created: 05/29/2001
Date modified: 05/29/2001 by Mat Ballard
      Purpose: sets the Min Property when new data is added to a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMinMaxFromSeries(AMin, AMax: Single);
var
  DoRescale: Boolean;
begin
  if (AMin < AMax) then
  begin
    DoRescale := FALSE;
    if (AMin < FMin) then
    begin
      if ((AMin > 0) or (not FLogScale)) then
      begin
        FMin := AMin;
        DoRescale := TRUE;
      end;
    end;
    if (AMax > FMax) then
    begin
      if ((AMax > 0) or (not FLogScale)) then
      begin
        FMax := AMax;
        DoRescale := TRUE;
      end;
    end;
    if (DoReScale) then
      Rescale;
  end;
end;


{------------------------------------------------------------------------------
    Procedure: TAxis.SetPen
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Pen Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetAxisType
  Description: unusual property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2003 by Mat Ballard
      Purpose: sets the AxisType Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetAxisType(Value: TAxisType);
begin
  FAxisType := Value;
  case FAxisType of
    atPrimaryX:
      begin
        Tag := Ord(soXAxis);
        Name := 'X-Axis';
        FLabels.Tag := Ord(soXAxisLabel);
        FTitle.Tag := Ord(soXAxisTitle);
      end;
    atZ:
      begin
        SetAngle(TAxisList(Self.Collection).Plot.ZAngle);
        Tag := Ord(soZAxis);
        Name := 'Z-Axis';
        FLabels.Tag := Ord(soZAxisLabel);
        FTitle.Tag := Ord(soZAxisTitle);
      end;
  else {All Y axes are 0 degrees:}
    SetAngle(0);
    Tag := Ord(soYAxis);
    Name := 'Y-Axis';
    FLabels.Tag := Ord(soYAxisLabel);
    FTitle.Tag := Ord(soYAxisTitle);
    if (FAxisType = atSecondaryY) then
      Name := 'Secondary ' + Name
    else if (FAxisType = atTertiaryY) then
      Name := 'Tertiary ' + Name;
  end;
  FLabels.Name := Name + ' Labels';
  FTitle.Name := Name + ' Title';
  Breadth := TAxisList(Collection).Plot.OutlineWidth;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetStepSize
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the StepSize (distance between ticks) Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetStepSize(Value: Single);
begin
  if (Value <> FStepSize) then
  begin
    FStepSize := Value;
    Rescale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetStepStart
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the StepStart (where ticks start) Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetStepStart(Value: Single);
begin
  if (Value <> FStepStart) then
  begin
    FStepStart := Value;
    Rescale;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetTickMinor
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the TickMinor (number of minor ticks) Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetTickMinor(Value: Byte);
begin
  if (Value <> FTickMinor) then
  begin
{limit the number of minors:}
    if (Value > 9) then
      Value := 9;
    if not (csLoading in TAxisList(Collection).Plot.ComponentState) then
      FAutoTick := FALSE;

    FTickMinor := Value;
    StyleChange(Self);
  end;  
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetTickSize
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the TickSize Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetTickSize(Value: Byte);
begin
  if ((Value > 0) and (Value <> FTickSize)) then
  begin
    if not (csLoading in TAxisList(Collection).Plot.ComponentState) then
      FAutoTick := FALSE;
    FTickSize := Value;
    DoTicks;
    StyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetZInterceptY
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the ZInterceptY Property: the intercept of the Z Axis with the Y Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetZInterceptY(Value: Single);
begin
  if (FZInterceptY <> Value) then
  begin
    FZInterceptY := Value;
    //DoGeometry;
    StyleChange(Self);
  end;
end;

{Various other Functions and Procedures--------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAxis.StyleChange
  Description: event firing proedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.StyleChange(Sender: TObject);
begin
  //DoTicks;
  if (Assigned(OnChange) and Visible) then OnChange(Sender);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.ReScale
  Description: geometry manager
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 07/25/2002 by Mat Ballard
      Purpose: This method determines the Axis geometry (StepStart and StepSize).
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.ReScale;
var
  Exponent: Integer;
  RoughStepSize: Single;
  Mantissa: Extended;
begin
  mPrecisionAdded := 0;

  if (FLogScale) then
  begin
    if (FMin <= 0) then
      FMin := FMax / 1000;
    mLogSpan := Log10(FMax / FMin);

    if (FAutoTick) then
    begin
      DeSci(FMin, Mantissa, Exponent);
{work out a starting point, 1 x 10^Exponent:}
      FStepStart := IntPower(10.0, Exponent);

      if (mLogSpan >= 2) then
      begin {many decades of data:}
        //if (not FAutoScale) then
        FStepSize := 10;
      end
      else
      begin
        RoughStepSize := mLogSpan / (mTickNum+1);
        RoughStepSize := Power(10.0, RoughStepSize);
        if (RoughStepSize > 1.5) then
        begin
{get the Mantissa and Exponent:}
          DeSci(RoughStepSize, Mantissa, Exponent);
          FStepSize := Round(Mantissa) * IntPower(10.0, Exponent);
        end
        else
        begin
          FStepSize := RoughStepSize;
        end;
{$IFDEF DELPHI3_UP}
        Assert(FStepSize > 1.0,
          'TAxis.ReScale Error: The calculated StepSize on a Log scale is ' +
            FloatToStr(FStepSize));
{$ENDIF}
      end; {if mLogSpan ?}
    end; {if AutoTick}
{go to next multiple of FStepSize if neccessary:}
    if (FAutoScale or FAutoTick) then
      while (FStepStart <= FMin) do
        FStepStart := FStepSize * FStepStart;
  end {LogScale}
  else
  begin {normal linear scale:}
    mSpan := FMax - FMin;
    if (FAutoTick) then
    begin
      RoughStepSize := mSpan / (mTickNum+1);
{get the Mantissa and Exponent:}
      DeSci(RoughStepSize, Mantissa, Exponent);

      FStepSize := Round(Mantissa) * IntPower(10.0, Exponent);
      {mTickNum := Trunc(mSpan / FStepSize);}
      FStepStart := FStepSize * Int((FMin / FStepSize) + 0.999);
    end; {if FAutoTick}
{increase FStepStart by FStepSize:}
    if (FAutoScale or FAutoTick) then
      while (FStepStart <= FMin) do
        FStepStart := FStepStart + FStepSize;

{mPrecisionAdded is the added precision needed to display numerical labels with
 sufficient precision to be distinguishable:}
    if (Exponent <= -FLabels.FDigits) then
      mPrecisionAdded := 1 - FLabels.FDigits - Exponent;
  end;

  StyleChange(Self);
end; {ReScale}

{------------------------------------------------------------------------------
     Function: TAxis.SameSideAs
  Description: Are (X, Y) and ACentre on ths same side of this Axis ?
       Author: Mat Ballard
 Date created: 09/16/2002
Date modified: 09/16/2002 by Mat Ballard
      Purpose: Managing geometry of Titles and Labels
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.SameSideAs(X, Y: Integer; ACentre: TPoint): Boolean;
var
  Side1, Side2: Boolean;
begin
  case Angle of
    0, 180: Result := ((X > Self.MidX) = (ACentre.x > Self.MidX));
    90, 270: Result := ((Y > Self.MidY) = (ACentre.y > Self.MidY));
  else
    Side1 := (Y > mSlope*X + mIntercept_b);
    Side2 := (ACentre.y > mSlope*ACentre.x + mIntercept_b);
    Result := (Side1 = Side2);
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.IsDefaultSide
  Description: Are (X, Y) and ACentre on ths same side of this Axis ?
       Author: Mat Ballard
 Date created: 09/16/2002
Date modified: 09/16/2002 by Mat Ballard
      Purpose: Managing geometry of Titles and Labels
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.IsDefaultSide(X, Y: Integer): Boolean;
var
  ACenter: TPoint;
begin
  ACenter.x := Self.Centre.x + FdTick.x;
  ACenter.y := Self.Centre.y + FdTick.y;
  Result := SameSideAs(X, Y, ACenter);
end;

{------------------------------------------------------------------------------
     Function: TAxis.Reflect
  Description: Reflects a point across this Axis
       Author: Mat Ballard
 Date created: 09/16/2002
Date modified: 09/16/2002 by Mat Ballard
      Purpose: Managing geometry of Titles and Labels
 Known Issues:
 ------------------------------------------------------------------------------
  C = (x0, y0)            y = mx + b; x = (y - b) / m
           L2           /
   C-------------------P2 = (x2, y2)    x2 = (y0-b)/m
   | --__        L3|  /|                y1 = m*x0 + b
   |    --__      | / |
   |     L1  --__ |/  |                h = Sqrt((x2-x0)^2 + (y1-y0)^2)
   |             --/__ |
   |              /   -|__              cos = L1/rise = L1/(y0-y1)
   |             /     |  --__          L1 = (y0-y1)*cos = (y0-y1)*run/h = (y0-y1)*(x2-x0)/h
   |            /      |      --__
   |           /       |          -P    sin = L2/L1
   |          /        |                L2 = L1*sin = L1*rise/h = L1(y0-y1)/h
   |       h /         | rise
   |        /          |                cos = L3/L1
   |       /           |                L3 = L1cos = L1*run/h = L1(x2-x0)/h
   |      /            |
   |     /             |
   |    /              |
   |   /               |
   |  /                |
   | /                |
   P1__________________|
   /      run
  /
 /    P1 = (x1, y1)

}
function TAxis.Reflect(ACentre: TPoint): TPoint;
var
  Gap, L1, L2, L3: Integer;
  h: Single;
  P1, P2: TPoint;
begin
  case Angle of
    0, 180:
      begin
        Gap := ACentre.x - MidX;
        Result.x := MidX - Gap;
        Result.y := ACentre.y;
      end;
    90, 270:
      begin
        Gap := ACentre.y - MidY;
        Result.x := ACentre.x;
        Result.y := MidY - Gap;
      end;
  else  // y = mx + b; x = (y - b) / m
    P1 := Point(ACentre.x, Round(mSlope * ACentre.x + mIntercept_b));
    P2 := Point(Round((ACentre.y - mIntercept_b) / mSlope), ACentre.y);
    //P3 := Point(P2.x, P1.y);
    h := Sqrt(Sqr(P2.x-P1.x) + Sqr(P2.y-P1.y));
    L1 := Round((ACentre.y-P1.y) * (P2.x-ACentre.x) / h);
    L2 := Round(l1 * (ACentre.y-P1.y) / h);
    L3 := Round(l1 * (P2.x-ACentre.x) / h);
    Result := ACentre;
    Inc(Result.x, 2*L2);
{This math was worked out for normal axes, but screen Y values are in the
 opposite direction, so we Dec:}
    Dec(Result.y, 2*L3);
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.GetNextXValue
  Description: auxilary procedure for  Drawing
       Author: Mat Ballard
 Date created: 02/28/2001
Date modified: 02/28/2001 by Mat Ballard
      Purpose: calculates the next tick point
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.GetNextXValue(XValue: Single): Single;
begin
  if (FLogScale) then
    GetNextXValue := XValue * FStepSize
   else
    GetNextXValue := XValue + FStepSize;
end;

{------------------------------------------------------------------------------
     Function: TAxis.GetNextXValue
  Description: Was the end of this Axis clicked on ?
       Author: Mat Ballard
 Date created: 09/12/2002
Date modified: 09/12/2002 by Mat Ballard
      Purpose: screen management
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.EndClickedOn(iX, iY, Radius: Integer): Boolean;
begin
  Result := (Sqr(iX - EndX) + Sqr(iY - EndY) <= Sqr(Radius));
end;


{------------------------------------------------------------------------------
    Procedure: TAxis.Draw
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 0/25/2002 by Mat Ballard
      Purpose: draws the Axis on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.Draw(ACanvas: TCanvas; Vector: TPoint);
{Comments:
 This method is quite complex, in a tedious way.
 It has to account for the following variations:
    1. Visible or not;
    2. Arrows visible or not;
    3. Axis direction (Horizontal or vertical);
    4. Tick (and Label and Title) direction);
    5. Title Alignment Direction and Orientation;
    6. Tick, Label and Title visibility.
    7. Angle !
 An added complication is that we must generate a vertical font for the Title
 of vertical axes. Note that this only works with TrueType fonts - NOT fonts
 that are purely screen or printer.}
var
  SaveLabelGeometry, OldFireEvents: Boolean;
  i,
  iX,
  iY,
  //FontDescent,
  //MinorTickSize,
  iFontWidth: Integer;
  MinorStepSize,
  MinorStepStart,
  {NewAngle,}
  ZValue: Single;
  Line2,
  TheText: String;
  TheTickStart, TheTickVector, TheVector: TPoint;

{begin internal functions:}
  function GetNextXValue(XValue: Single): Single;
  begin
    if (FLogScale) then
      GetNextXValue := XValue * FStepSize
     else
      GetNextXValue := XValue + FStepSize;
  end;

  function GetNextMinorXValue(XValue: Single): Single;
  begin
    if (FLogScale) then
      GetNextMinorXValue := XValue * MinorStepSize
     else
      GetNextMinorXValue := XValue + MinorStepSize;
  end;

  procedure DoLimit(TheLimit: Single);
  begin
    TheTickStart := Self.dFofZ(TheLimit);
    Inc(TheTickStart.x, Origin.x);
    Inc(TheTickStart.y, Origin.y);
    ACanvas.MoveTo(TheTickStart.x, TheTickStart.y);
    Inc(TheTickStart.x, Vector.x);
    Inc(TheTickStart.y, Vector.y);
    ACanvas.LineTo(TheTickStart.x, TheTickStart.y);
  end;

  procedure DrawTicks(AValue: Single);
  begin
      TheTickStart := dFofZ(AValue);
      Inc(TheTickStart.x, Origin.x);
      Inc(TheTickStart.y, Origin.y);
      if (FTickSize > 1) then
        ACanvas.MoveTo(TheTickStart.x, TheTickStart.y);
      Inc(TheTickStart.x, TheTickVector.x);
      Inc(TheTickStart.y, TheTickVector.y);
      if (FTickSize > 1) then
        ACanvas.LineTo(TheTickStart.x, TheTickStart.y);
  end;

  procedure DrawLabels;
  var
    LinePos: Integer;
  begin
    LinePos := Pos(#10, TheText);
    if (LinePos > 0) then
    begin
      Line2 := Copy(TheText, LinePos+1, 9999);
      System.SetLength(TheText, LinePos-1);
      if (System.Length(TheText) > System.Length(Line2)) then
        iFontWidth := ACanvas.TextWidth(TheText)
       else
        iFontWidth := ACanvas.TextWidth(Line2);
    end
     else
      iFontWidth := ACanvas.TextWidth(TheText);
    if (iFontWidth > mFont.cx) then
      mFont.cx := iFontWidth;
    if (TheTickVector.x < 0) then // Tick points Left
      Dec(TheTickStart.x, iFontWidth);
    if (TheTickVector.y > 0) then
    begin //Tick points down
      Inc(TheTickStart.y, mFont.cy);
      Dec(TheTickStart.x, iFontWidth div 2);
    end;
    if (SaveLabelGeometry) then
    begin
      SaveLabelGeometry := FALSE;
      FLabels.FireEvents := FALSE;
      TheVector := TheTickStart;
    end;
{Note: I'd like a bit more space above the axis under Linux.}
{$IFDEF MSWINDOWS}
    ACanvas.TextOut(
      TheTickStart.x,
      TheTickStart.y - Abs(ACanvas.Font.Height),
      TheText);
{$ENDIF}
{$IFDEF LINUX}
{Note: Qt outputs text left and below the (X,Y) coordinates:}
    ACanvas.TextOut(
      TheTickStart.x,
      TheTickStart.y - 3*Abs(ACanvas.Font.Height) div 2,
      TheText);
{$ENDIF}
    if (LinePos > 0) then
    begin
{$IFDEF MSWINDOWS}
      ACanvas.TextOut(
        TheTickStart.x,
        TheTickStart.y,
        Line2);
{$ENDIF}
{$IFDEF LINUX}
      ACanvas.TextOut(
        TheTickStart.x,
        TheTickStart.y - Abs(ACanvas.Font.Height) div 2,
        Line2);
{$ENDIF}
    end;
  end;

begin
{the most common reason for exit:}
  if (not Visible) then exit;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TAxis.Draw: ACanvas is nil !');
{$ENDIF}

  ACanvas.Pen.Assign(FPen);

{Do the geometry:}
  DoGeometry;
  if ((EndX < 0) or (EndY > TAxisList(Collection).Plot.Height)) then
    if Assigned(FOnOutOfBounds) then
    begin
      OnOutOfBounds(Self, EndX, EndY);
      exit;
    end;

{first, squelch any "OnChange" events:}
  OldFireEvents := FireEvents;
  FireEvents := FALSE;

{Draw the axis:}
  ACanvas.MoveTo(Origin.x, Origin.y);
  ACanvas.LineTo(EndX, EndY);
{Draw the arrows on the axis:}
  if (FArrowSize > 0) then
  begin
    (*if (Alignment = taRightJustify) then*)
    begin
      ACanvas.MoveTo(EndX, EndY);
      iX := EndX + Round(FArrowSize * mCosM30);
      iY := EndY + Round(FArrowSize * mSinM30);
      ACanvas.LineTo(iX, iY);
      ACanvas.MoveTo(EndX, EndY);
{look back along the axis, then 30 degrees less:}
      iX := EndX + Round(FArrowSize * mCosP30);
      iY := EndY + Round(FArrowSize * mSinP30);
      ACanvas.LineTo(iX, iY);
    end; {taLeftJustify and taCenter therefore means no arrows !}
  end;

  if (Self.LimitsVisible) then
  begin
    ACanvas.Pen.Style := psDot;
    DoLimit(FLimitLower);
    DoLimit(FLimitUpper);
    ACanvas.Pen.Style := Self.Pen.Style;
  end;

{Prepare fonts for Labels on the axis:}
  SaveLabelGeometry := TRUE;
  ACanvas.Font.Assign(FLabels.Font);
  mChar := ACanvas.TextExtent('W');
  mFont := mChar;

{We get the vector describing the ticks:}
  TheTickVector := GetTick(FLabels.DefaultSide);

{We label the axis:}
  if ((Self.FAxisType = atZ) and (TPlot(TAxisList(GetOwner).Plot).ZSeriesNames)) then
  begin // we are a Z axis and we want to label with series names:
{Loop over every tick:}
    for i := 0 to TPlot(TAxisList(GetOwner).Plot).Series.Count-1 do
    begin
{Ticks on the Axis:}
      DrawTicks(TPlot(TAxisList(GetOwner).Plot).Series[i].ZData);
{Draw (numeric) Labels:}
      if (FLabels.Visible) then
      begin
        TheText := TPlot(TAxisList(GetOwner).Plot).Series[i].Name;
        DrawLabels;
      end;
    end; {for}
  end
  else if ((FLabelSeries = nil) or (TSeries(FLabelSeries).XStringData = nil)) then
  begin
{Initialize:}
    Line2 := '';
    ZValue := FStepStart;
{Loop over every tick:}
    while (ZValue < FMax) do
    begin
{Ticks on the Axis:}
      DrawTicks(ZValue);
{Draw (numeric) Labels:}
      if (FLabels.Visible) then
      begin
        TheText := LabelToStrF(ZValue);
        DrawLabels;
      end;
      ZValue := GetNextXValue(ZValue);
    end; {while ZValue < FMax}
  end
  else
  begin // string labels:
    for i := 0 to TSeries(FLabelSeries).XStringData.Count-1 do
    begin
      if (i >= TSeries(FLabelSeries).NoPts) then
        break;
{Ticks on the Axis:}
      DrawTicks(TSeries(FLabelSeries).XData^[i]);
      if (FLabels.Visible) then
      begin
        TheText := TSeries(FLabelSeries).XStringData.Strings[i];
        DrawLabels;
      end;
    end;
  end;

{TODO [1] [-o Matb] [-c Screen Drawing] : Remove setting of Label geometry from TAxis.Draw method and put it into (?) StyleChange}
{Work out label geometry:}
  FLabels.Breadth := Abs(ACanvas.Font.Height);
  if (Angle = 90) then
  begin
    Dec(TheVector.y, Abs(ACanvas.Font.Height) div 2);
    FLabels.Length := mChar.cx + (TheTickStart.x - TheVector.x)
  end
   else if (Angle = 0) then
    FLabels.Length := mChar.cx - (TheTickStart.y - TheVector.y)
   else
    FLabels.Length := mChar.cx + Round(Sqrt(Sqr(TheTickStart.x - TheVector.x) + Sqr(TheTickStart.y - TheVector.y)));
  FLabels.Origin := TheVector;
  FLabels.DoGeometry;
  FLabels.FireEvents := TRUE;

{Minor Ticks on the axis:}
  if ((FTickSize > 1) and (FTickMinor > 0)) then
  begin
{find out where the minors start:}
    MinorStepSize := FStepSize / (FTickMinor+1);
    MinorStepStart := FStepStart;
    while ((MinorStepStart - MinorStepSize) >= FMin) do
      MinorStepStart := MinorStepStart - MinorStepSize;
    //iY := MidY;
    TheTickVector.x := TheTickVector.x div 2;
    TheTickVector.y := TheTickVector.y div 2;

    ZValue := MinorStepStart;
    //i := 0;
    while (ZValue < FMax) do
    begin
      TheTickStart := dFofZ(ZValue);
      Inc(TheTickStart.x, Origin.x);
      Inc(TheTickStart.y, Origin.y);
      if (FTickSize > 1) then
        ACanvas.MoveTo(TheTickStart.x, TheTickStart.y);
      Inc(TheTickStart.x, TheTickVector.x);
      Inc(TheTickStart.y, TheTickVector.y);
      if (FTickSize > 1) then
        ACanvas.LineTo(TheTickStart.x, TheTickStart.y);
      ZValue := GetNextMinorXValue(ZValue);
    end;
  end; {minor ticks}

{Setup Axis Title and Draw:}
  if (FTitle.Visible) then
  begin
    FTitle.FireEvents := FALSE;
{Need to do the text geometry BEFORE setting the centre:}
    FTitle.DoCanvasGeometry(ACanvas);
    FTitle.Centre := GetTitleCentre(FTitle.Position, FTitle.DefaultSide);
{Draw calls DoGeometry:}
    FTitle.Draw(ACanvas);
    FTitle.FireEvents := TRUE;
  end;

  FireEvents := OldFireEvents;
end;

{------------------------------------------------------------------------------
     Function: TAxis.GetTick
  Description: this returns the actual tick vector, depending on the default side
       Author: Mat Ballard
 Date created: 09/12/2002
Date modified: 09/12/2002 by Mat Ballard
      Purpose: screen management
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.GetTick(ADefaultSide: Boolean): TPoint;
begin
  Result := FdTick;
  if (not ADefaultSide) then
  begin
    Result.x := -Result.x;
    Result.y := -Result.y;
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.GetTitleCentre
  Description: this returns the position of the title, given the Position and Side.
       Author: Mat Ballard
 Date created: 09/12/2002
Date modified: 09/12/2002 by Mat Ballard
      Purpose: screen management
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.GetTitleCentre(APosition: Integer; ADefaultSide: Boolean): TPoint;
var
  TheVector: TPoint;
begin
  TheVector := GetTick(ADefaultSide);
  if (TheVector.x < 0) then // Tick points Left
  begin
    Dec(TheVector.x, mFont.cx);
    Dec(TheVector.x, mChar.cy);
  end
  else if (TheVector.y < 0) then // Tick points Up
  begin
    Dec(TheVector.y, mChar.cy);
  end
  else if (TheVector.x > 0) then // Tick points Right
  begin
    Inc(TheVector.x, mFont.cx);
    Inc(TheVector.x, mChar.cx);
  end
  else if (TheVector.y > 0) then //Tick points down
  begin
    Inc(TheVector.y, mFont.cy);
    Inc(TheVector.y, mChar.cy);
  end;
{Work out the location of the Title:}
  Result := Self.Centre;
  Inc(Result.x, TheVector.x);
  Inc(Result.y, TheVector.y);
{Is the title in the middle, or has it been moved ?}
  if (APosition <> 50) then
  begin
    TheVector := Self.Vector;
    TheVector.x := (APosition - 50) * TheVector.x div 100;
    TheVector.y := (APosition - 50) * TheVector.y div 100;
    Inc(Result.x, TheVector.x);
    Inc(Result.y, TheVector.y);
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.dFofZ
  Description: standard Z transform
       Author: Mat Ballard
 Date created: 01/18/2001
Date modified: 01/18/2001 by Mat Ballard
      Purpose: returns the change in pixel position on screen as a function of the real data ordinate Z
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.dFofZ(Z: Single): TPoint;
begin
  if (FLogScale) then
  begin
    Result.x := Round(mSin * Length * ((Log10(Z / FMin)) / mLogSpan));
    Result.y := -Round(mCos * Length * ((Log10(Z / FMin)) / mLogSpan));
  end
  else
  begin
    Result.x := Round(mSin * Length * ((Z - FMin) / (mSpan)));
    Result.y := -Round(mCos * Length * ((Z - FMin) / (mSpan)));
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.FofX
  Description: standard X transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the pixel position on screen as a function of the real data ordinate X
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.FofX(X: Single): Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(FAxisType = atPrimaryX, 'Only a Primary X Axis can return F(X) !');
{$ENDIF}
{$IFDEF CATCH_EXCEPTIONS}try{$ENDIF}
    if (FLogScale) then
    begin
      Result := Round(Left + Width * ((Log10(X / FMin)) / mLogSpan))
    end
     else
      Result := Round(Left + Width * ((X - FMin) / (mSpan)));
{$IFDEF CATCH_EXCEPTIONS}
  except
    Result := -999999;
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
     Function: TAxis.FofY
  Description: standard Y transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the pixel position on screen as a function of the real data co-ordinate Y
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.FofY(Y: Single): Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(FAxisType >= atPrimaryY, 'Only a Primary Y Axis can return F(Y) !');
{$ENDIF}
  {$IFDEF CATCH_EXCEPTIONS}try{$ENDIF}
    if (FLogScale) then
    begin
      Result := Round(Bottom - Height * ((Log10(Y / FMin)) / mLogSpan))
    end
     else
      Result := Round(Bottom - Height * ((Y - FMin) / (mSpan)));
{$IFDEF CATCH_EXCEPTIONS}
  except
    Result := -999999;
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
     Function: TAxis.XofF
  Description: inverse X transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the real data ordinate X as a function of the pixel position on screen
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.XofF(F: Integer): Single;
{this function returns the real data ordinate X
 as a function of the pixel position F on screen:}
begin
{$IFDEF DELPHI3_UP}
  Assert(FAxisType = atPrimaryX, 'Only a Primary X Axis can return X(F) !');
{$ENDIF}

  if (FLogScale) then
    Result := FMin * Power(10.0, (mLogSpan * (F-Left) / Width))
   else
    Result := mSpan * ((F-Left) / Width) + FMin;
end;

{------------------------------------------------------------------------------
     Function: TAxis.YofF
  Description: inverse Y transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the real data ordinate Y as a function of the pixel position on screen
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.YofF(F: Integer): Single;
{this function returns the real data ordinate X
 as a function of the pixel position F on screen:}
begin
{$IFDEF DELPHI3_UP}
  Assert(FAxisType >= atPrimaryY, 'Only a Primary Y Axis can return Y(F) !');
{$ENDIF}

  if (FLogScale) then
    Result := FMin * Power(10.0, (mLogSpan * (Bottom-F) / Height))
   else
    Result := mSpan * ((Bottom-F) / Height) + FMin;
end;

{------------------------------------------------------------------------------
     Function: TAxis.StrToLabel
  Description: converts a string to a number, depending on the NumberFormat
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user IO
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.StrToLabel(Value: String): Single;
begin
  case (FLabels.NumberFormat) of
    lfGeneral .. lfCurrency:
      StrToLabel := StrToFloat(Value);
    lfPercent:
      StrToLabel := StrToFloat(Value) / 100;
    lfTime:
      StrToLabel := StrToTime(Value);
    lfDate:
      StrToLabel := 60 * StrToFloat(Value);
    lfDateTime:
      StrToLabel := 3600 * StrToFloat(Value);
    else
      StrToLabel := 0.0;
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.LabelToStrF
  Description: converts a number to a string, depending on the NumberFormat
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user IO
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.LabelToStrF(Value: Single): String;
var
  Mantissa: Extended;
  Exponent: Integer;
  Hour, Min, Sec, MSec: Word;
  //Year, Month, Day: Word;
begin
  {if (Abs(Value) < High(Integer)) then
    Value := Round(Value);}
  case (FLabels.NumberFormat) of
    lfGeneral .. lfCurrency:
{See Rescale for definition of mPrecisionAdded}
      Result := FloatToStrF(Value, TFloatFormat(FLabels.NumberFormat),
        FLabels.Precision + mPrecisionAdded, FLabels.Digits);
    lfSI:
      begin
        DeSci(Value, Mantissa, Exponent);
        case Exponent of {p, n u, m, -, K, M, G, T}
          -12 .. -10: Result := 'p';
          -9 .. -7: Result := 'n';
          -6 .. -4: Result := 'u';
          -3 .. -1: Result := 'm';
          3 .. 5: Result := 'K';
          6 .. 8: Result := 'M';
          9 .. 11: Result := 'G';
          12 .. 14: Result := 'T';
        else Result := '';
        end;
        if (System.Length(Result) > 0) then
        begin
          Exponent := (Exponent + 99) mod 3;
          Mantissa := Mantissa * IntPower(10, Exponent);
          Result := FloatToStrF(Mantissa, TFloatFormat(lfFixed),
            FLabels.Precision, FLabels.Digits) + Result;
        end
        else
          Result := FloatToStrF(Value, TFloatFormat(lfGeneral),
            FLabels.Precision, FLabels.Digits);
      end;
    lfPercent:
      Result := FloatToStrF(100 * Value, TFloatFormat(FLabels.NumberFormat),
        FLabels.Precision, FLabels.Digits);
    lfTime:
      Result := TimeToStr(Value + TAxisList(Self.Collection).Plot.StartTime);
    lfDate:
      Result := DateToStr(Value + TAxisList(Self.Collection).Plot.StartTime);
    lfDateTime:
      Result := TimeToStr(Value + TAxisList(Self.Collection).Plot.StartTime) + #10 +
        DateToStr(Value + TAxisList(Self.Collection).Plot.StartTime);
    lfElapsedTime:
      begin
        Result := '';
        if (Value > 1) then
        begin // days or longer
          Result := FloatToStrF(Value, TFloatFormat(FLabels.NumberFormat),
            FLabels.Precision + mPrecisionAdded, FLabels.Digits);
        end
        else
        begin // hh:mm:ss.sss
          DecodeTime(Value, Hour, Min, Sec, MSec);
          if (Hour > 0) then Result := Format('%d:', [Hour]);
          if ((Hour > 0) or (Min > 0)) then Result := Result + Format('%d:', [Min]);
          Result := Result + Format('%d.%3.3d', [Hour, MSec]);
        end;
      end;
    lfElapsedDate:
      Result := DateToStr(Value + TAxisList(Self.Collection).Plot.StartTime);
(*    lfSeconds:
      Result := FloatToStrF(Value, ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfMinutes:
      Result := FloatToStrF((Value / 60), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfHours:
      Result := FloatToStrF((Value / 3600), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfDays:
      Result := FloatToStrF((Value / 86400), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfShortTime:
      begin
        TheDateTime := Value;
        Result := FormatDateTime('t', TheDateTime);
      end;
    lfShortDate:
      begin
        TheDateTime := Value;
        Result := FormatDateTime('ddddd', TheDateTime);
      end;*)
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TAxis(Dest).ArrowSize := FArrowSize;
  TAxis(Dest).AutoScale := FAutoScale;
  TAxis(Dest).AutoTick := FAutoTick;
  TAxis(Dest).AutoZero := FAutoZero;
  TAxis(Dest).AxisType := FAxisType;
  TAxis(Dest).Intercept := FIntercept;
  TAxis(Dest).LimitLower := FLimitLower;
  TAxis(Dest).LimitUpper := FLimitUpper;
  TAxis(Dest).LimitsVisible := FLimitsVisible;
  TAxis(Dest).Logscale := FLogscale;
  TAxis(Dest).Max := FMax;
  TAxis(Dest).Min := FMin;
  TAxis(Dest).StepSize := FStepSize;
  TAxis(Dest).StepStart := FStepStart;
  TAxis(Dest).TickMinor := FTickMinor;
  TAxis(Dest).TickSize := FTickSize;
  TAxis(Dest).ZInterceptY := FZInterceptY;

  TAxis(Dest).Labels.Assign(FLabels);
  TAxis(Dest).Pen.Assign(FPen);
  TAxis(Dest).Title.Assign(FTitle);
end;

procedure TAxis.SetLabelSeries(Value: TPersistent);
begin
{Note: Labeltext is maintained within the TSeries, NOT in TAxis !}
  FLabelSeries := Value;
  StyleChange(Self);
end;


end.
