unit Axisedit;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: AxisEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 03/11/2001
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Purpose:
To facilitate user manipluation of Axis properties.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, TypInfo,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Buttons, Controls, Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Buttons, Controls, Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF LINUX}
  QButtons, QControls, QForms, QGraphics, QStdCtrls,
{$ENDIF}

  Axis, Coloredt, Misc, Nedit, Plotdefs, Titles, Slidebar;

type
  TAxisProperty = record
    ArrowDirection: TAlignment;
    ArrowSize: Byte;
    DefaultSide: Boolean; // applies to Labels and Ticks
    TitleDefaultSide: Boolean;
    LabelFormat: TLabelFormat;
    LabelDigits: Byte;
    LabelPrecision: Byte;
    LabelDirection: TDirection;
    LimitLower: Single;
    LimitUpper: Single;
    LimitsVisible: Boolean;
    PenColor: TColor;
    PenStyleIndex: Integer;
    PenWidthIndex: Integer;
    ScaleAuto: Boolean;
    ScaleLog: Boolean;
    ScaleMin: Single;
    ScaleMax: Single;
    ScaleIntercept: Single;
    TickAuto: Boolean;
    //TickDirection: TOrientation;
    TickMinors: Byte;
    TickSize: Byte;
    TickStepSize: Single;
    TickStepStart: Single;
    TitlePosition: TPercent;
    Visible: Boolean;
    ZAngle: Word;
    ZInterceptY: Single;
    ZLength: Word;
    ZeroAuto: Boolean;
  end;

  TAxisEditorForm = class(TForm)
    HelpBitBtn: TBitBtn;
    CancelBitBtn: TBitBtn;
    OKBitBtn: TBitBtn;
    VisibleCheckBox: TCheckBox;
    PenGroupBox: TGroupBox;
    PenColorLabel: TLabel;
    PenWidthLabel: TLabel;
    PenWidthComboBox: TComboBox;
    PenStyleComboBox: TComboBox;
    PenStyleLabel: TLabel;
    LabelsGroupBox: TGroupBox;
    DigitsLabel: TLabel;
    PrecisionLabel: TLabel;
    FormatLabel: TLabel;
    DigitsComboBox: TComboBox;
    PrecisionComboBox: TComboBox;
    FormatComboBox: TComboBox;
    TickGroupBox: TGroupBox;
    TickSizeLabel: TLabel;
    StepSizeLabel: TLabel;
    ScaleGroupBox: TGroupBox;
    InterceptLabel: TLabel;
    LogScaleCheckBox: TCheckBox;
    MinLabel: TLabel;
    MaxLabel: TLabel;
    TickMinorsLabel: TLabel;
    MinorsComboBox: TComboBox;
    NoComboBox: TComboBox;
    AxisNumberLabel: TLabel;
    TickSizeNEdit: TNEdit;
    StepSizeNEdit: TNEdit;
    MinNEdit: TNEdit;
    MaxNEdit: TNEdit;
    InterceptNEdit: TNEdit;
    PenColorEdit: TColorEdit;
    ApplyBitBtn: TBitBtn;
    ZAxisGroupBox: TGroupBox;
    ZAngleLabel: TLabel;
    ZLengthLabel: TLabel;
    ZAngleNEdit: TNEdit;
    ZLengthNEdit: TNEdit;
    ZInterceptLabel: TLabel;
    ZInterceptNEdit: TNEdit;
    LimitsGroupBox: TGroupBox;
    LowerLimitLabel: TLabel;
    UpperLimitLabel: TLabel;
    LowerLimitNEdit: TNEdit;
    UpperLimitNEdit: TNEdit;
    LimitsVisibleCheckBox: TCheckBox;
    AutoScaleCheckBox: TCheckBox;
    AutoTickCheckBox: TCheckBox;
    StepStartLabel: TLabel;
    StepStartNEdit: TNEdit;
    AutoZeroCheckBox: TCheckBox;
    ApplyAllBitBtn: TBitBtn;
    DefaultSideCheckBox: TCheckBox;
    GroupBox1: TGroupBox;
    TitleEdit: TEdit;
    PositionSlideBar: TSlideBar;
    TitleDefaultSideCheckBox: TCheckBox;
    ArrowSizeLabel: TLabel;
    ArrowSizeNEdit: TNEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormatComboBoxClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure PenStyleComboBoxClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NoComboBoxClick(Sender: TObject);
    procedure TitleEditChange(Sender: TObject);
    procedure DigitsComboBoxClick(Sender: TObject);
    procedure PrecisionComboBoxClick(Sender: TObject);
    procedure TickSizeNEditChange(Sender: TObject);
    procedure StepSizeNEditChange(Sender: TObject);
    procedure MinNEditChange(Sender: TObject);
    procedure MaxNEditChange(Sender: TObject);
    procedure InterceptNEditChange(Sender: TObject);
    procedure MinorsComboBoxClick(Sender: TObject);
    procedure ArrowSizeNEditChange(Sender: TObject);
    procedure VisibleCheckBoxClick(Sender: TObject);
    procedure AutoScaleCheckBoxClick(Sender: TObject);
    procedure LogScaleCheckBoxClick(Sender: TObject);
    procedure PenWidthComboBoxClick(Sender: TObject);
    procedure PenColorEditChange(Sender: TObject);
    procedure ApplyBitBtnClick(Sender: TObject);
    procedure ZAngleNEditChange(Sender: TObject);
    procedure ZLengthNEditChange(Sender: TObject);
    procedure ZInterceptNEditChange(Sender: TObject);
    procedure LowerLimitNEditChange(Sender: TObject);
    procedure UpperLimitNEditChange(Sender: TObject);
    procedure LimitsVisibleCheckBoxClick(Sender: TObject);
    procedure AutoTickCheckBoxClick(Sender: TObject);
    procedure AutoZeroCheckBoxClick(Sender: TObject);
    procedure ApplyAllBitBtnClick(Sender: TObject);
    procedure DefaultSideCheckBoxClick(Sender: TObject);
    procedure PositionSlideBarChange(Sender: TObject);
    procedure TitleDefaultSideCheckBoxClick(Sender: TObject);
  private
    pAXP: ^TAxisProperty;
    CurrentIndex: Integer;

  public
    AxisPropertyList: TList;
    AxisNames: TStringList;
    HistoryMode: Boolean;
    ThePlot: TObject;

    function AddAxis(AName: String; AnAxisProperty: TAxisProperty): Integer;
    procedure SelectAxis(Index: Integer);
    procedure DoHintsFromResource;
  end;

var
  AxisEditorForm: TAxisEditorForm;

implementation

{$R *.dfm}

uses
  Plot;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position and populates the Color combo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormCreate(Sender: TObject);
var
  i: Integer;
begin
  DoHintsFromResource;

{set combo and edit box widths:}
  SetDialogGeometry(Self, ApplyBitBtn, CancelBitBtn.Left);
{the exception:}
  NoComboBox.Width := 50;

{Populate the combo boxes:}
  for i := 0 to 18 do
    DigitsComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TLabelFormat)) do
    FormatComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TLabelFormat), i), 3, 99));
  for i := 0 to 9 do
    MinorsComboBox.Items.Add(IntToStr(i));
  for i := 0 to 20 do
    PenWidthComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TPenStyle)) do
    PenStyleComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TPenStyle), i), 3, 99));
  for i := 0 to 7 do
    PrecisionComboBox.Items.Add(IntToStr(i));

  AxisPropertyList := TList.Create;
  AxisNames := TStringList.Create;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.DoHintsFromResource
  Description: standard loading of labels from resources
       Author: Mat Ballard
 Date created: 06/25/2001
Date modified: 06/25/2001 by Mat Ballard
      Purpose: display in different languages
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.DoHintsFromResource;
begin
  TitleEdit.Hint := 'The display title of the Axis (units in brackets)';
  PositionSlideBar.Hint := 'Position of the Title along the Axis';
  NoComboBox.Hint := 'The index of the axis to edit (0=X, 1=Y, etc)';
  AutoScaleCheckBox.Hint := 'Automatically set Min and Max of the Axis';
  AutoTickCheckBox.Hint := 'Automatically set number of Ticks, Step Size and Start, and Minors';
  AutoZeroCheckBox.Hint := 'Automatically set the Intercept to Zero on the corresponding Axis';
  VisibleCheckBox.Hint := 'Make the Axis (in)visible';
  MinNEdit.Hint := 'Minimum value of the axis (real units)';
  MaxNEdit.Hint := 'Maximum value of the axis (real units)';
  InterceptNEdit.Hint := 'Intercept of this axis on the complementary axis (real units)';
  LogScaleCheckBox.Hint := 'Make the scale of this axis logarithmic';
  DefaultSideCheckBox.Hint := 'On what side of the axis do the ticks and numeric Labels go ?';
  TitleDefaultSideCheckBox.Hint := 'On what side of the axis does the Title go ?';
  MinorsComboBox.Hint := 'The number of minor ticks between each major tick';
  TickSizeNEdit.Hint := 'The size of the ticks (screen pixels)';
  StepSizeNEdit.Hint := 'The distance (real units) between adjacent ticks';
  StepStartNEdit.Hint := 'The first point to place a tick on the Axis';
  FormatComboBox.Hint := 'How do we display the numeric labels ?';
  DigitsComboBox.Hint := 'How many digits do we display in the numeric labels ?';
  PrecisionComboBox.Hint := 'With what precision do we display the numeric labels ?';
  //LabelDirectionComboBox.Hint := 'Are the numeric labels horizontal or vertical ?';
  ArrowSizeNEdit.Hint := 'The size of the arrows (screen pixels; 0 means invisible)';
  //ArrowDirectionComboBox.Hint := 'Which direction does the arrow point towards ?';
  LimitsGroupBox.Hint := 'This allows drawing of limits as lines perpendicular to the axis.';
  LowerLimitNEdit.Hint := '(in real units)';
  UpperLimitNEdit.Hint := '(in real units)';
  LimitsVisibleCheckBox.Hint := 'Do we draw the limits on screen ?';
  ZAngleNEdit.Hint := 'The angle between the Z Axis and the vertical (clockwise, degrees)';
  ZLengthNEdit.Hint := 'The length of the Z Axis (screen pixels)';
  ZInterceptNEdit.Hint := 'The intercept of the Z Axis on the Y-Axis (real units)';
  ApplyBitBtn.Hint := 'Apply these properties now !';
  ApplyAllBitBtn.Hint := 'Apply these properties to ALL Axes (where appropriate)';
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormClose
  Description: standard FormClose procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: checks for valid user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
{Check that floating point values are valid:
  Action := caHide;
  try
    if (not AutoScaleCheckBox.Checked) then
    begin
      MinEdit.SetFocus;
      Test := StrToFloat(MinEdit.Text)
      MaxEdit.SetFocus;
      TestMax := StrToFloat(MaxEdit.Text)
      if (TestMax < Test) then
        ERangeError.CreateFmt('The Min (%g) MUST be less than the Max (%g)',
          [Test, TestMax]);
      InterceptEdit.SetFocus;
      Test := StrToFloat(InterceptEdit.Text)
      StepSizeEdit.SetFocus;
      Test := Test + StrToFloat(StepSizeEdit.Text);
  {We do the above and below the squelch the warning messages.
      Tag := Trunc(Test);
    end;
  except
{Prevent closure:
    Action := caNone;
  end;}
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormatComboBoxClick
  Description: responds to NumberFormat selection
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: loads the Digits combo depending on selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormatComboBoxClick(Sender: TObject);
var
  i, iMax, OldIndex: Integer;
begin
  pAXP^.LabelFormat := TLabelFormat(FormatComboBox.ItemIndex);
  OldIndex := DigitsComboBox.ItemIndex;
  DigitsComboBox.Clear;
  iMax := 18;
  if ((FormatComboBox.ItemIndex = Ord(ffGeneral)) or
      (FormatComboBox.ItemIndex = Ord(ffExponent))) then
    iMax := 4;
  for i := 0 to iMax do
  begin
    DigitsComboBox.Items.Add(IntToStr(i));
  end;
  if (OldIndex > iMax) then OldIndex := iMax;
  DigitsComboBox.ItemIndex := OldIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.PenStyleComboBoxClick
  Description: responds to user selection
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Width to 1 if neccessary
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.PenStyleComboBoxClick(Sender: TObject);
begin
  pAXP^.PenStyleIndex := PenStyleComboBox.ItemIndex;
  if (PenStyleComboBox.ItemIndex > Ord(psSolid)) then
    PenWidthComboBox.ItemIndex := 1;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormActivate
  Description: standard FormActivate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Visibility of the Max Label
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormActivate(Sender: TObject);
begin
  if (MaxNEdit.Enabled = FALSE) then
    MaxLabel.Enabled := FALSE;
  FormatComboBoxClick(Self);
{$IFDEF LINUX}
  HelpBitBtn.Kind := bkCustom;
{$ENDIF}
end;

procedure TAxisEditorForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to AxisPropertyList.Count-1 do
    FreeMem(AxisPropertyList.Items[i], SizeOf(TAxisProperty));

  AxisPropertyList.Free;
  AxisNames.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.SelectAxis
  Description: responds to selection of a new Axis
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.SelectAxis(Index: Integer);
begin
  pAXP := AxisPropertyList.Items[Index];
  CurrentIndex := Index;

  if (Index = 0) then {X Axis:}
  begin
    if (HistoryMode) then
      MaxNEdit.Enabled := FALSE;
  end
  else
  begin
    MaxNEdit.Enabled := TRUE;
  end;

  TitleEdit.Text := AxisNames.Strings[Index];
  PositionSlideBar.Position := pAXP^.TitlePosition;
  FormatComboBox.ItemIndex := Ord(pAXP^.LabelFormat);
  DefaultSideCheckBox.Checked := pAXP^.DefaultSide;
  TitleDefaultSideCheckBox.Checked := pAXP^.TitleDefaultSide;
  DigitsComboBox.ItemIndex := pAXP^.LabelDigits;
  PrecisionComboBox.ItemIndex := pAXP^.LabelPrecision;
  //LabelDirectionComboBox.ItemIndex := Ord(pAXP^.LabelDirection);
  PenColorEdit.SetColour(pAXP^.PenColor);
  PenWidthComboBox.ItemIndex := pAXP^.PenWidthIndex;
  PenStyleComboBox.ItemIndex := pAXP^.PenStyleIndex;
  TickSizeNEdit.AsInteger := pAXP^.TickSize;
  //TickDirectionComboBox.ItemIndex := Ord(pAXP^.TickDirection);
  StepSizeNEdit.AsReal := pAXP^.TickStepSize;
  MinorsComboBox.ItemIndex := pAXP^.TickMinors;
  MinNEdit.AsReal := pAXP^.ScaleMin;
  MaxNEdit.AsReal :=  pAXP^.ScaleMax;
  InterceptNEdit.AsReal := pAXP^.ScaleIntercept;
  ZInterceptNEdit.AsReal := pAXP^.ZInterceptY;
  AutoScaleCheckBox.Checked :=  pAXP^.ScaleAuto;
  LogScaleCheckBox.Checked :=  pAXP^.ScaleLog;
  ArrowSizeNEdit.AsInteger := pAXP^.ArrowSize;
  //ArrowDirectionComboBox.ItemIndex := Ord(pAXP^.ArrowDirection);
  LowerLimitNEdit.AsReal := pAXP^.LimitLower;
  UpperLimitNEdit.AsReal := pAXP^.LimitUpper;
  LimitsVisibleCheckBox.Checked := pAXP^.LimitsVisible;
  if (pAXP^.ZLength > 0) then
  begin
    ZAngleNEdit.AsInteger := pAXP^.ZAngle;
    ZInterceptNEdit.AsReal := pAXP^.ZInterceptY;
    ZLengthNEdit.AsInteger := pAXP^.ZLength;
    ZAxisGroupBox.Enabled := TRUE;
  end
  else
  begin
    ZAxisGroupBox.Enabled := FALSE;
  end;
  ZAngleLabel.Enabled := ZAxisGroupBox.Enabled;
  ZLengthLabel.Enabled := ZAxisGroupBox.Enabled;
  ZInterceptLabel.Enabled := ZAxisGroupBox.Enabled;

  AutoScaleCheckBoxClick(Self);

  VisibleCheckBox.Checked := pAXP^.Visible;
end;

procedure TAxisEditorForm.NoComboBoxClick(Sender: TObject);
var
  Test,
  TestMax: Single;
begin
  try
    if (not AutoScaleCheckBox.Checked) then
    begin
      MinNEdit.SetFocus;
      Test := MinNEdit.AsReal;
      MaxNEdit.SetFocus;
      TestMax := MaxNEdit.AsReal;
      if (TestMax < Test) then
        ERangeError.CreateFmt('The Min (%g) MUST be less than the Max (%g)', [Test, TestMax]);
      InterceptNEdit.SetFocus;
      Test := InterceptNEdit.AsReal;
      StepSizeNEdit.SetFocus;
      Test := Test + StepSizeNEdit.AsReal;
{We do the above and below the squelch the warning messages.}
      Tag := Trunc(Test);
    end;
    SelectAxis(NoComboBox.ItemIndex);
  except
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxisEditorForm.AddAxis
  Description: adds a new Axis to the list
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: stores the Axis properties
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxisEditorForm.AddAxis(AName: String; AnAxisProperty: TAxisProperty): Integer;
var
  pDestination: Pointer;
  pDest, pSource: PChar;
  i: Integer;
begin
  NoComboBox.Items.Add(IntToStr(AxisNames.Count));

  AxisNames.Add(AName);

  GetMem(pDestination, SizeOf(AnAxisProperty));
  pSource := @AnAxisProperty;
{NB: this causes terminal access violations:
  System.Move(pSource, pDestination, SizeOf(AnAxisProperty));}

  pDest := pDestination;
  for i := 1 to SizeOf(AnAxisProperty) do
  begin
    pDest^ := pSource^;
    Inc(pSource);
    Inc(pDest);
  end;

  AxisPropertyList.Add(pDestination);
  AddAxis := AxisPropertyList.Count;
end;

procedure TAxisEditorForm.TitleEditChange(Sender: TObject);
begin
  AxisNames.Strings[CurrentIndex] := TitleEdit.Text;
end;

procedure TAxisEditorForm.DigitsComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelDigits := DigitsComboBox.ItemIndex;
end;

procedure TAxisEditorForm.PrecisionComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelPrecision := PrecisionComboBox.ItemIndex;
end;

procedure TAxisEditorForm.TickSizeNEditChange(Sender: TObject);
begin
  pAXP^.TickSize := TickSizeNEdit.AsInteger;
end;

procedure TAxisEditorForm.StepSizeNEditChange(Sender: TObject);
begin
  pAXP^.TickStepSize := StepSizeNEdit.AsReal;
end;

procedure TAxisEditorForm.MinNEditChange(Sender: TObject);
begin
  pAXP^.ScaleMin := MinNEdit.AsReal;
end;

procedure TAxisEditorForm.MaxNEditChange(Sender: TObject);
begin
  pAXP^.ScaleMax := MaxNEdit.AsReal;
end;

procedure TAxisEditorForm.InterceptNEditChange(Sender: TObject);
begin
  pAXP^.ScaleIntercept := InterceptNEdit.AsReal;
end;

procedure TAxisEditorForm.MinorsComboBoxClick(Sender: TObject);
begin
  pAXP^.TickMinors := MinorsComboBox.ItemIndex;
end;

procedure TAxisEditorForm.ArrowSizeNEditChange(Sender: TObject);
begin
  pAXP^.ArrowSize := ArrowSizeNEdit.AsInteger;
end;

procedure TAxisEditorForm.VisibleCheckBoxClick(Sender: TObject);
begin
  pAXP^.Visible := VisibleCheckBox.Checked;
end;

procedure TAxisEditorForm.AutoScaleCheckBoxClick(Sender: TObject);
begin
  pAXP^.ScaleAuto := AutoScaleCheckBox.Checked;

  MinNEdit.Enabled := not AutoScaleCheckBox.Checked;
  MaxNEdit.Enabled := not AutoScaleCheckBox.Checked;
  MinLabel.Enabled := not AutoScaleCheckBox.Checked;
  MaxLabel.Enabled := not AutoScaleCheckBox.Checked;
end;

procedure TAxisEditorForm.AutoZeroCheckBoxClick(Sender: TObject);
begin
  pAXP^.ZeroAuto := AutoZeroCheckBox.Checked;

  InterceptNEdit.Enabled := not AutoZeroCheckBox.Checked;
  if (AutoZeroCheckBox.Checked) then
    InterceptNEdit.AsReal := 0;
  InterceptLabel.Enabled := InterceptNEdit.Enabled;
end;

procedure TAxisEditorForm.AutoTickCheckBoxClick(Sender: TObject);
begin
  pAXP^.TickAuto := AutoTickCheckBox.Checked;

  MinorsComboBox.Enabled := not AutoTickCheckBox.Checked;
  TickMinorsLabel.Enabled := not AutoTickCheckBox.Checked;
  TickSizeNEdit.Enabled := not AutoTickCheckBox.Checked;
  TickSizeLabel.Enabled := not AutoTickCheckBox.Checked;
  StepSizeNEdit.Enabled := not AutoTickCheckBox.Checked;
  StepSizeLabel.Enabled := not AutoTickCheckBox.Checked;
  StepStartNEdit.Enabled := not AutoTickCheckBox.Checked;
  StepStartLabel.Enabled := not AutoTickCheckBox.Checked;
  if (AutoTickCheckBox.Checked) then
  begin
    MinorsComboBox.ItemIndex := 0;
    TickSizeNEdit.AsInteger := 10;
  end;
end;

procedure TAxisEditorForm.LogScaleCheckBoxClick(Sender: TObject);
begin
  pAXP^.ScaleLog := LogScaleCheckBox.Checked;
end;

procedure TAxisEditorForm.PenWidthComboBoxClick(Sender: TObject);
begin
  pAXP^.PenWidthIndex := PenWidthComboBox.ItemIndex;
end;

procedure TAxisEditorForm.PenColorEditChange(Sender: TObject);
begin
  pAXP^.PenColor := PenColorEdit.Color;
end;

procedure TAxisEditorForm.ApplyBitBtnClick(Sender: TObject);
begin
{$IFDEF GUI}
  TPlot(ThePlot).ApplyAxisChange(Self);
{$ENDIF}
end;


procedure TAxisEditorForm.ZAngleNEditChange(Sender: TObject);
begin
  pAXP^.ZAngle := ZAngleNEdit.AsInteger;
end;

procedure TAxisEditorForm.ZLengthNEditChange(Sender: TObject);
begin
  pAXP^.ZLength := ZLengthNEdit.AsInteger;
end;

procedure TAxisEditorForm.ZInterceptNEditChange(Sender: TObject);
begin
  pAXP^.ZInterceptY := ZInterceptNEdit.AsReal;
end;

procedure TAxisEditorForm.LowerLimitNEditChange(Sender: TObject);
begin
  pAXP^.LimitLower := LowerLimitNEdit.AsReal;
end;

procedure TAxisEditorForm.UpperLimitNEditChange(Sender: TObject);
begin
  pAXP^.LimitUpper := UpperLimitNEdit.AsReal;
end;

procedure TAxisEditorForm.LimitsVisibleCheckBoxClick(Sender: TObject);
begin
  pAXP^.LimitsVisible := LimitsVisibleCheckBox.Checked;
end;

procedure TAxisEditorForm.ApplyAllBitBtnClick(Sender: TObject);
var
  i: Integer;
  pAXPi: ^TAxisProperty;
begin
  for i := 0 to AxisPropertyList.Count-1 do
  begin
    if (i = CurrentIndex) then Continue;
    pAXPi := AxisPropertyList.Items[i];
    pAXPi^.TitlePosition := pAXP^.TitlePosition;
    pAXPi^.ArrowDirection := pAXP^.ArrowDirection;
    pAXPi^.ArrowSize := pAXP^.ArrowSize;
    pAXPi^.LabelFormat := pAXP^.LabelFormat;
    pAXPi^.LabelDigits := pAXP^.LabelDigits;
    pAXPi^.LabelPrecision := pAXP^.LabelPrecision;
    pAXPi^.DefaultSide := pAXP^.DefaultSide;
    pAXPi^.TitleDefaultSide := pAXP^.TitleDefaultSide;
    pAXPi^.PenColor := pAXP^.PenColor;
    pAXPi^.PenStyleIndex := pAXP^.PenStyleIndex;
    pAXPi^.PenWidthIndex := pAXP^.PenWidthIndex;
    pAXPi^.ScaleAuto := pAXP^.ScaleAuto;
    pAXPi^.ScaleLog := pAXP^.ScaleLog;
    pAXPi^.TickAuto := pAXP^.TickAuto;
    //pAXPi^.TickDirection := pAXP^.TickDirection;
    pAXPi^.TickMinors := pAXP^.TickMinors;
    pAXPi^.TickSize := pAXP^.TickSize;
    pAXPi^.Visible := pAXP^.Visible;
    pAXPi^.ZeroAuto := pAXP^.ZeroAuto;
  end;

  ApplyBitBtnClick(Self);
end;

procedure TAxisEditorForm.DefaultSideCheckBoxClick(Sender: TObject);
begin
  pAXP^.DefaultSide := DefaultSideCheckBox.Checked;
end;

procedure TAxisEditorForm.PositionSlideBarChange(Sender: TObject);
begin
  pAXP^.TitlePosition := PositionSlideBar.Position;
end;

procedure TAxisEditorForm.TitleDefaultSideCheckBoxClick(Sender: TObject);
begin
  pAXP^.TitleDefaultSide := TitleDefaultSideCheckBox.Checked;
end;

end.
