unit Data;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: pSeries.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 08/09/2001
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Purpose:
This unit contains the TSeries sub-component - that manages the data for a single series.

Known Issues:
      - This would normally be called Series, but TeeChart already uses that unit name.

History:
 1.20 15 Jan 2001: change name from pSeries to Data: TChart uses Series, but

                   'Data' is what this unit manages.
                   many changes to accomodate new plot types.
 1.01 21 September 2000: add Brush property for columns
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils,
{$IFDEF GUI}
  {$IFDEF WINDOWS}
  Wintypes,
  Clipbrd, Controls, Dialogs, Forms, Graphics,
  {$ENDIF}
  {$IFDEF WIN32}
  Windows,
  Clipbrd, Controls, Dialogs, Forms, Graphics,
  {$ENDIF}
  {$IFDEF LINUX}
  Types,
  QClipbrd, QControls, QDialogs, QForms, QGraphics,
  {$ENDIF}

{$ELSE}
  gd, gd_io, gdWindows, gdGraphics, {gdWinapi,}
{$ENDIF}

{$IFDEF POLYNOMIAL_FIT}
  CurveFit,
{$ENDIF}

  U_FTG2, FFourier,

  {$IFDEF NO_MATH}NoMath,{$ELSE}Math,{$ENDIF}
  Axis, Dataedit, Displace, PlotDefs, Ptedit, Misc, Titles;

const
  OUTLINE_DENSITY = 20;
{This is the number of points in a branch of an Outline.}
  MAX_MULTIPLICITY = 4;
{This is the maximum number of ExtraData arrays in SeriesType = stMultiple,
 and hence the dimension of the FExtraData pointer array.}

type
  //TOnMinMaxChangeEvent = procedure(Sender: TObject; Value: Single) of object;

  THighLow = (hlLow, hlHigh);
  TSetHighLow = set of THighLow;

  TDataStatus = (dsNone, dsInternal, dsInternalString, dsExternal);
{These are the data storage states:}
{}
{    dsNone - no data as yet;}
{    dsInternal - there is internal memory;}
{    dsInternalString - there is internal memory, with string X values;}
{    dsExternal - both the X and Y data are stored elsewhere (not in this component).}
  TExtraData = array [0..MAX_MULTIPLICITY-1] of pSingleArray;

  TSeries = class(TCollectionItem)
  private
    //FAxisList: TCollection;
    FBrush: TBrush;
    FBubbleSize: TPercent;
    //FDataChanged: Boolean;
    FDefSize: Word;
    FDeltaX: Integer;
    FDeltaY: Integer;
    FExpression: String;
    FOnExpressionChange: TNotifyEvent;
    FName: String;
    FNoPts: Integer;
    FPen: TPen;
    FHighCapacity: Integer;
    FHighCount: Integer;
    FHighLow: TSetHighLow;
    FHighs: pIntegerArray;
     FLowCount: Integer;
     FLows: pIntegerArray;
    FIsLine: Boolean;
    FMultiplicity: Byte;
    FPlot: TComponent;
    FSeriesType: TSeriesType;
    FShadeLimits: Boolean;
    FShowFit: Boolean;
    //FStartTime: TDateTime;
    FSymbol: TSymbol;
    FSymbolSize: Integer;
    FSymbolCheck: Boolean;
    FTag: Integer;
    FVisible: Boolean;
    FXMin: Single;
    FXMax: Single;
    FXData: pSingleArray;
{Is the X data maintained in a different series ?}
    FXDataSeries: TSeries;
{This is the Data Series in which the External X data, if any, is stored.}
    FXStringData: TStringList;
    FYAxisIndex: Byte;
    //FXAxis, FYAxis: TAxis;
    FYData: pSingleArray;
    FExtraData: TExtraData;
{We "cheat": each series is one plane in the Z direction, so we only need to store one datum:}
    FZData: Single;
    Fd2Y_dX2: pSingleArray;
      //Size2ndDeriv: Integer;
    FYMin: Single;
    FYMax: Single;
{The bounding rectangle for a Pie, which is stored:}
    //PieLeft, PieTop, Width, Height: Longint;
{The sum  of Y values: used in Pie graphs}
    YSum: Single;

    FOnStyleChange: TNotifyEvent;
    FOnDataChange: TNotifyEvent;
     {FOnDataChangeList: TList;}
    FOnDestroy: TNotifyEvent;
    FOnWarning: TOnWarningEvent;
    {FOnAddPoint: TNotifyEvent;}
{Currently superceded by direct calls to TAxis.SetMin[Max]FromSeries:}
    {FOnXMinChange: TOnMinMaxChangeEvent;
    FOnXMaxChange: TOnMinMaxChangeEvent;
    FOnYMinChange: TOnMinMaxChangeEvent;
    FOnYMaxChange: TOnMinMaxChangeEvent;}
{may be re-implemented later for other needs.}

    mDependentSeries: TList;
{The list of series that use this series' X-Data.}

    mDataStatus: TDataStatus;
{Was this data generated externally ? If it was, then we do not manage it,
 nor manipulate it.}
    mNoAllocPts: Integer;
{The current number of points allocated in memory.}
    mOutline: array [0..OUTLINE_DENSITY+1] of TPoint;
{An array of Outline points. These points are in screen co-ordinates (pixels).}
{}
{The Outline is used for clicking and dragging operations.}
{}
{Note that for XY-type series, the Outline is a series of points,
 but for Pie series, the first two are (Top, Left), (Right, Bottom).}
    mNoOutlinePts: Integer;
{The number of Outline points.}

    procedure CheckBounds(ThePointNo: Integer; AdjustAxis: Boolean);
{Check the Min and Max properties against this point.}
    procedure IncMemSize;
{Allocate memory for the data.}

  protected
{The one and only property getting function:}
    function GetXDataRefCount: Word;
    function GetXAxis: TAxis;
    function GetYAxis: TAxis;

{The property-setting routines:}
    //procedure SetAxisList(Value: TCollection);
    procedure SetBrush(Value: TBrush);
    procedure SetBubbleSize(Value: TPercent);
    procedure SetDeltaX(Value: Integer);
    procedure SetDeltaY(Value: Integer);
    procedure SetExpression(Value: String);
    procedure SetIsLine(Value: Boolean);
    procedure SetMultiplicity(Value: Byte);
    procedure SetName(Value: String);
    procedure SetNoPts(Value: Integer);
    procedure SetPen(Value: TPen);
    procedure SetSeriesType(Value: TSeriesType);
    procedure SetExtraDataMemSize(NewNo: Integer);
    procedure SetShadeLimits(Value: Boolean);
    procedure SetShowFit(Value: Boolean);
    //procedure SetStartTime(Value: TDateTime);
    procedure SetSymbol(Value: TSymbol);
    procedure SetSymbolSize(Value: Integer);
    procedure SetSymbolCheck(Value: Boolean);
    procedure SetVisible(Value: Boolean);
    procedure SetXStringData(Value: TStringList);
    //procedure SetYAxis(Value: TAxis);
    procedure SetYAxisIndex(Value: Byte);
    procedure SetZData(Value: Single);

    procedure DoStyleChange;
    procedure DoDataChange;

  public
    Ai: array [0..POLYORDER] of Single;
{The coefficients of a Polynomial (or other) fit to the data.
 Set to nil until fit is performed. After a linear fit,
   Ai[0] = intercept,
   Ai[1] = slope
 If POLYNOMIAL_FIT is defined,  POLYORDER = 9, otherwise we can only do straight-line fits.}
    R_Square: Single;
{The residual of a linear of polynomial fit.}
    nTerms: Byte;
{The degree of the polynomial fit.}

    //property AxisList: TCollection read FAxisList write SetAxisList;
{Make all these properties UnStored so that they are not read from the property stream:}
    //property DataChanged: Boolean read FDataChanged write FDataChanged;
{Has the data in this series changed ?}
    //property ExternalXSeries: Boolean read FExternalXSeries stored FALSE;
{Is the X data maintained in a different series ?}
    property HighCount: Integer read FHighCount stored FALSE;
{The number of Highs (Peaks)}
    property Highs: pIntegerArray read FHighs stored FALSE;
{This is a list of the Highs (Peaks) in the plot. See Lows.}
    property LowCount: Integer read FLowCount stored FALSE;
{The number of Lows (Troughs)}
    property Lows: pIntegerArray read FLows stored FALSE;
{This is a list of the Lows (Troughs) in the plot. See Highs.}
    property NoPts: Integer read FNoPts write SetNoPts stored FALSE;
{The number of points in the series.}
    property XDataRefCount: Word read GetXDataRefCount stored FALSE;
{This is the number of series that use this series as an X Data holder.}

    property XAxis: TAxis read GetXAxis;
{The X Axis to which this series is bound - needed for scaling purposes.}

    property YAxis: TAxis read GetYAxis;
{The Y Axis to which this series IS bound - can be any of the Y Axes - needed for scaling purposes.}

    property XData: pSingleArray read FXData stored FALSE;
{This is the dynamic X data array.
 It can be set by the user, or memory for the data
 can be allocated and managed by this component.}
{}
{The user can access the data points either through the GetPoint / GetXYPoint /
 ReplacePoint methods, or directly by:}
{}
{    ASeries.FXData^[i] := NewValue;}
{}
{Note that the POINTER XData is read-only, but that the array elements are
 read/write.}

    property ExtraData: TExtraData read FExtraData stored FALSE;
{The "Extra" Data; only applicable when the plot is ptXY.}
    property XError: pSingleArray read FExtraData[0] stored FALSE;
{The error in the X Data; only applicable when the seriestype is stError.}
    property XDataSeries: TSeries read FXDataSeries stored FALSE;
{If the X data is maintained in a different series, this is the series,
 otherwise it is nil.}

    property XStringData: TStringlist read FXStringData write SetXStringData stored FALSE;
{This is the X data in string format.}

    property YData: pSingleArray read FYData stored FALSE;
{This is the dynamic Y data array.
 It can be set by the user, or memory for the data
 can be allocated and managed by this component.}
{}
{The user can access the data points either through the GetPoint / GetXYPoint /
 ReplacePoint methods, or directly by:}
{}
{    ASeries.FYData^[i] := NewValue;}
{}
{Note that the POINTER YData is read-only, but that the array elements are
 read/write.}
    property YBubble: pSingleArray read FExtraData[0] stored FALSE;
{The Bubble radius data for the Y Data; only applicable when the plot is ptBubble.}
    property YError: pSingleArray read FExtraData[1] stored FALSE;
{The error in the Y Data; only applicable when the plot is ptError.}
    property YComplex: pSingleArray read FExtraData[0] stored FALSE;
{The imaginary Y Data; only applicable when the Series has been Fourier transformed.}

    property d2Y_dX2: pSingleArray read Fd2Y_dX2;
{The array of second derivatives - used in cubic splines.}
    property XMin: Single read FXMin;
{The minimum X value, determined by GetBounds.}
    property XMax: Single read FXMax;
{The maximum X value, determined by GetBounds.}
    property YMin: Single read FYMin;
{The minimum Y value, determined by GetBounds.}
    property YMax: Single read FYMax;
{The maximum Y value, determined by GetBounds.}

    Constructor Create(AOwner: TCollection); override;
{Each series needs to know a few things:}
{}
{    1. What axes it can relate to;}
{    2. Does it use another (previous) series X Values ?}
    Destructor Destroy; override;
{This is a standard Destructor.}

    procedure Loaded;
{CollectionItems do not have a Loaded method to override - which is a real pain !
 So we have to implement a public kludge here and in TCollection}

    procedure AssignTo(Dest: TPersistent); override;
{TSeries' implementation of the standard Assign(To) method.}

{The following Addxxx methods are now overloaded to add error and 3D functionality.}
    procedure AddData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer);
    procedure AddDataEx(XPointer, YPointer: pSingleArray; ExtraPointers: array of pSingleArray; NumberOfPoints: Integer); 
{This adds an entire Internal data set of an X array, a Y array,
 and the new number of points: Success returns TRUE.}
{}
{Internal means that TSeries allocates and manages the memory for this data,
 and makes a copy of the data located at XPointer and YPointer into this
 internally managed memory.}
{}
{It can therefore add, remove or edit any points.}

    function AddDrawPoint(X, Y: Single; ACanvas: TCanvas): Integer;
    function AddDrawPointEx(X, Y: Single; Extra: array of Single; ACanvas: TCanvas): Integer; 
{This adds a single point to the xy data (using AddPoint),
 draws the line segment and new point, and returns the number
 of points: -1 indicates failure.}
    procedure DrawLatestPoint(ACanvas: TCanvas);
{This draw just the very last point in the series.}
    function AddPoint(X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
    function AddPointEx(X, Y: Single; Extra: array of Single; FireEvent, AdjustAxes: Boolean): Integer; 
{This adds a single point to the xy data and returns the number of points:
 -1 indicates failure. If no memory has been allocated for the data yet, then
 IncMemSize is called automatically.}
    function AddStringPoint(XString: String; X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
    function AddStringPointEx(XString: String; X, Y: Single; Extra: array of Single; FireEvent, AdjustAxes: Boolean): Integer; 
{This adds a single point, which has a string X value, to the data and returns
 the number of points: -1 indicates failure. If no memory has been allocated
 for the data yet, then IncMemSize is called automatically.}
    function InsertPoint(X, Y: Single): Integer;
    function InsertPointEx(X, Y: Single; Extra: array of Single): Integer; 
{This inserts a single point in the xy data and returns the location of the point:
 -1 indicates failure. The point is inserted at the appropriate X value.}
    function PointToData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean;
{This adds an entire External data set of an X array, a Y array,
 and the new number of points: Success returns TRUE.}
{}
{External means that TSeries does not manage the memory for this data,
 nor can it add, remove or edit any points.}
    procedure ReplacePoint(N: Integer; NewX, NewY: Single);
    procedure ReplacePointEx(N: Integer; NewX, NewY: Single; Extra: array of Single); 
{This replaces the Nth point's values with X and Y.}

{These are the overloaded Addxxx methods to add error and 3D functionality.}
    {function AddData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean; overload;
    function AddDrawPoint(X, Y: Single; ACanvas: TCanvas): Integer; overload;
    function AddPoint(X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer; overload;
    function AddStringPoint(XString: String; X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer; overload;
    function InsertPoint(X, Y: Single): Integer; overload;
    function PointToData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean; overload;
    procedure ReplacePoint(N: Integer; NewX, NewY: Single); overload;}

    procedure AllocateNoPts(Value: Integer);
{Directly allocates memory for a fixed number of points.}
{}
{If AllocateNoPts cannot allocate memory for the requested number of points,
 it allocates what it can and returns FALSE.}

    procedure Compress(CompressRatio: Integer);
{This averages every N points in a row and so reduces the size of the
 data set by a factor of N.}
    procedure Contract(TheStart, TheFinish: Integer);
{This throws away all points before TheStart and after TheFinish.}
{$IFDEF GUI}
    procedure CopyToClipBoard;
{Does what it says.}

    procedure Displace(TheHelpFile: String);
{Runs the dialog box to set the displacement (DeltaX) of the Series.}
    procedure ApplyDisplacementChange(Sender: TObject);
{This applies changes from the Displacement Dialog.}
{$ENDIF}

    function AddDependentSeries(ASeries: TSeries): Boolean;
{This function ADDS a series that depends on this series' X-Data from the list of dependent series.}
    function RemoveDependentSeries(ASeries: TSeries): Boolean;
{This function REMOVES a series that depends on this series' X-Data from the list of dependent series.}
    function AssumeMasterSeries(XPts: Integer; OldMaster: TSeries; AList: TList): Boolean;
{This function makes this series' X-Data the MASTER for the given list of dependent series.}
    function ResetXDataSeries(OldSeries, NewSeries: TSeries): Boolean;
    function MakeXDataIndependent: Boolean;
    function MakeXDataDependent(MasterSeries: TSeries): Boolean;
    procedure MakeHighLowOpenClose;
{This procedure makes this Series a "High-Low-Open-Close" type in one simple command.}
    procedure MakeXDataString;
{This procedure readies the Series to accept String X Data.}
    function DelPoint(X, Y: Single; Confirm: Boolean): Integer;
{This deletes a single point, the closest one, from the xy data.}

    function DelPointNumber(ThePoint: Integer; Confirm: Boolean): Integer;
{This deletes a single point, the Nth one, from the xy data.}

    procedure Clear;
    function DelData: Boolean;
{This deletes an entire data set. It only works on internal data sets.}

    procedure DoFFT(Forwards: Boolean);
    procedure DoGlassmanFFT(Forwards: Boolean);

    procedure DrawHistory(ACanvas: TCanvas; HistoryX: Single);
{This draws the series on the given canvas, in History mode.
 That is, from the latest point backwards a distance HistoryX}
    procedure Draw(ACanvas: TCanvas; XYFastAt: Integer);
{This calls the correct Drawing method for the SeriesType.}
    procedure DrawXY(ACanvas: TCanvas; TheYData: pSingleArray; XYFastAt: Integer);
{This draws the series in XY fashion on the given canvas.}
    procedure DrawBubble(ACanvas: TCanvas);
{This draws the Bubbles of the Series on the given canvas.}
    procedure DrawError(ACanvas: TCanvas);
{This draws the Bubbles of the Series on the given canvas.}
    procedure DrawMultiple(ACanvas: TCanvas);
{This draws the Multiple symbols of the Series on the given canvas.}
    procedure DrawFourier(ACanvas: TCanvas);
{This draws the Fourier Transformed series on the given canvas.}
    procedure DrawShades(ACanvas: TCanvas; XYFastAt: Integer);
{This shades the series in XY fashion on the given canvas, if the Axis Limits are exceeded.}
    procedure DrawPie(ACanvas: TCanvas; PieLeft, PieTop, PieWidth, PieHeight: Integer);
{This draws the series on the given canvas as a Pie.}
    procedure DrawPolar(ACanvas: TCanvas; PolarRange: Single);
{This draws the series in Polar fashion on the given canvas.}
    procedure DrawLSFit(ACanvas: TCanvas);
{This draws the least squares fit and its caption on the given canvas.}
    procedure DrawSymbol(ACanvas: TCanvas; iX, iY: Integer);
{This draws one of the symbols on the given canvas.}
    procedure Trace(ACanvas: TCanvas);
{This draws the series on the given canvas in an erasable mode.
 The first call draws, the second call erases.}

{$IFDEF GUI}
    procedure EditData(TheHelpFile: String);
{This runs the Data Editor dialog box.}
    procedure ApplyDataChange(Sender: TObject);
{This applies changes from the DataEditor Dialog.}

    procedure EditPoint(ThePointNumber: Integer; TheHelpFile: String);
{This runs the Point Editor dialog box.}
    procedure ApplyPointChange(Sender: TObject; TheResult: TModalResult);
{This applies changes from the PointEditor Dialog.}
{$ENDIF}

    procedure GetBounds;
{Determines the Min and Max properties for the whole series.}
    procedure GetExtraBounds(Index: Integer; var sMin, sMax: Single);
{Determines the Min and Max properties for the whole series.}
{Data manipulation:}
    procedure ResetBounds;
{Reset the Min and Max properties.}

    function GetNearestPointToX(X: Single): Integer;
{This returns the point that has an X value closest to X.}
    function GetNearestPointToFX(FX: Integer): Integer;
{This returns the point that has an F(X) / Screen value closest to FX.}
    function Interpolate(X: Single): Single;

    function GetNearestPieSlice(
      iX, iY,
      PieLeft, PieTop, PieWidth, PieHeight: Integer;
      var MinDistance: Single): Integer;
{This returns the Index of the nearest point, and sets its XValue and YValue.}

    function GetNearestXYPoint(iX, iY: Integer;
      StartPt, EndPt: Integer;
      var MinDistance: Single): Integer;
{This returns the Index of the nearest point, and sets its XValue and YValue.
 It is guaranteed to find the nearest point.}

    function GetNearestXYPointFast(
      iX, iY: Integer;
      var MinDistance: Single): Integer;
{This returns the Index of the nearest point, and sets its XValue and YValue.
 This is much quicker than GetNearestXYPoint, especially for big data sets,
 but MAY NOT return the closest point.}

    procedure GetPoint(N: Integer; var X, Y: Single);
{This returns the Nth point's X and Y values.}

    function GetXYPoint(N: Integer): TXYPoint;
{This returns the Nth point's X and Y values in a TXYPoint record.}

    procedure Smooth(SmoothOrder: Integer);
{This smooths the xy data using a midpoint method.}

    procedure Sort;
{This sorts the xy data in ascending X order.}

    procedure GeneratePieOutline(
      PieLeft,
      PieTop,
      PieWidth,
      PieHeight: Integer;
      TheNearestPoint: Integer);
{This generates an pIE Outline from the data, for the specified point/rectangle.}
    procedure GenerateColumnOutline(X1, Y1, X2, Y2: Integer);
{This generates an Column Outline from the data, for the specified point/rectangle.}
    procedure GenerateXYOutline;
{This generates an XY Outline from the data. An Outline contains
 the screen coordinates for (OUTLINE_DENSITY +1) points.}
{}
{Note that the memory for the Outline is allocated in the constructor and
 freed in the destructor.}
    procedure Outline(ACanvas: TCanvas; ThePlotType: TPlotType; TheOutlineWidth: Integer);
{This draws (or erases) the Outline on the canvas.}

    procedure MoveBy(ACanvas: TCanvas; ThePlotType: TPlotType; DX, DY, TheOutlineWidth: Integer);
{This erases the old Outline from the canvas, then redraws it
 at (DX, DY) from its current position.}

    procedure MoveTo(
      ACanvas: TCanvas;
      ThePlotType: TPlotType;
      TheOutlineWidth,
      X, Y: Integer);                  {by how much}
{This erases the old Outline from the canvas, then redraws it
 at the new location (X, Y).}

    function LineBestFit(Left1, Right1, Left2, Right2: Integer): String;
{This performs a linear least-squares fit of TheSeries over the two regions
 1 (Left1 - Right1) and 2 (Left2 - Right2), and returns the Slope, Intercept and R-Square value.}
{$IFDEF POLYNOMIAL_FIT}
    function PolyBestFit(Left1, Right1, Left2, Right2: Integer): String;
{This performs a polynomial least-squares fit of TheSeries over the two regions 1 and 2,
 and returns the Polynomial Coefficicnet and R-Square value.}
{$ENDIF}
    function GetFitAsString: String;
{This returns the result of a linear or polynomial fit in a suitable format, and
 also copies it to the clipboard.}

    procedure Differentiate;
{This replaces the series by its differential.}
    procedure Integrate;
{This replaces the series by its integral.}
    function Integral(TheLeft, TheRight: Single): Single;
{This calculates the integral of a series by X co-ordinate.}
    function IntegralByPoint(Start, Finish: Integer): Single;
{This calculates the integral of a series by points.}

    procedure DoSpline(Density: Integer; pSplineSeries: TSeries);
{This calculates the cubic spline interpolation of the data (XSpline, YSpline),
 which resides in another Series.}
    procedure SecondDerivative;
{This calculates the second derivate for a cubic spline interpolation by SplineValue.}
    function SplineValue(X: Single): Single;
{This calculates the cubic spline interpolation of the data at a given point X.}
    procedure ClearSpline;

    function FindHighsLows(Start, Finish, HeightSensitivity: Integer): Integer;
    procedure MovingAverage(Span: Longint);
    function Average(TheLeft, TheRight: Single): Single;
    procedure Linearize(TheLeft, TheRight: Single);
    procedure Zero(TheLeft, TheRight: Single);
    procedure ClearHighsLows;
    procedure DrawHighs(ACanvas: TCanvas);
    {procedure SetOnDataChange(OldValue, NewValue: TNotifyEvent);}

  published
    property Brush: TBrush read FBrush write SetBrush;
{The Brush (color, width, etc) with which the series is drawn on the Canvas.}
    property BubbleSize: TPercent read FBubbleSize write SetBubbleSize;
{This is the percentage size of Bubbles, compared to the Y Axis, in the ptBubble PlotType.}
    property DefSize: Word read FDefSize write FDefSize {default 256};
{The default memory allocation block size. Allocated memory grows in blocks of
 this number of points.}
    property DeltaX: Integer read FDeltaX write SetDeltaX {default 0};
{The displacement of the series on the screen from its X origin.}
    property DeltaY: Integer read FDeltaY write SetDeltaY {default 0};
{The displacement of the series on the screen from its Y origin.}
    property Expression: String read FExpression write SetExpression;
{The function that is used to calculate this series.}
    property HighLow: TSetHighLow read FHighLow write FHighLow {default []};
{Do we show any Highs ? any Lows ? Both ? or None ?}
    property IsLine: Boolean read FIsLine write SetIsLine;
{When the PlotType is ptColumn or pt3DColumn, do we draw this series as a line anyway ?}    
    property Multiplicity: Byte read FMultiplicity write SetMultiplicity;
{When the SeriesType is stMultiple, Multiplicity determines the number of ExtraData arrays.
 Otherwise, it just reports the number of ExtraData arrays.
 1 <= Multiplicity <= MAX_MULTIPLICITY.}
    property Name: String read FName write SetName;
{The name of the data set.}
    property Pen: TPen read FPen write SetPen;
{The Pen (color, width, etc) with which the series is drawn on the Canvas.}
    property SeriesType: TSeriesType read FSeriesType write SetSeriesType;
{Is this a normal X-Y series, or does it contain additional data: sepcifically,
 Error in X and Y, or Imaginary (Complex) Y Data ?}    
    property ShadeLimits: Boolean read FShadeLimits write SetShadeLimits {default FALSE};
{Do we shade this series above and below the Y-axis limits ?}
    property ShowFit: Boolean read FShowFit write SetShowFit;
{Do we show the results of a least-squares fit ?}    
    property Symbol: TSymbol read FSymbol write SetSymbol {default syNone};
{The symbol (square, circle, etc) with which each data point is drawn.}
    property SymbolCheck: Boolean read FSymbolCheck write SetSymbolCheck {default FALSE};
{Do we place checks at the limits of stError and stMultiple SeriesTypes ?}
    property SymbolSize: Integer read FSymbolSize write SetSymbolSize {default 5};
{How big is the Symbol (0 means invisible).}
    property Tag: Integer read FTag write FTag default 0;
{The usual Tag property.}
    property Visible: Boolean read FVisible write SetVisible {default TRUE};
{Is this series visible ?}
    property YAxisIndex: Byte read FYAxisIndex write SetYAxisIndex default 1;
{The Y Axis Index to which this series IS bound - can be any of the Y Axes - needed for scaling purposes.
 We define YAxisIndex to run from 1 to FAxisList.Count-1:
    1 => The primary Y Axis,
    2 => The secondary Y Axis,
    etc.}
    property ZData: Single read FZData write SetZData;
{This is the Z-value for this series.
 It is set by the user.}
{}
{Note that unlike the read-only POINTERS XData and YData,
 ZData is a single read/write value.}



    property OnDataChange: TNotifyEvent read FOnDataChange write FOnDataChange;
{This notifies the owner (usually TSeriesList) of a change in the data of this series.}
    property OnDestroy: TNotifyEvent read FOnDestroy write FOnDestroy;
{This notifies the owner - usually a TSeriesList in another TPlot - of impending doom.}
    property OnExpressionChange: TNotifyEvent read FOnExpressionChange write FOnExpressionChange;
{When the Expression used to calculate this Series is changed, this alerts the TSeriesList,
 which then recalculates this Series.}
    property OnStyleChange: TNotifyEvent read FOnStyleChange write FOnStyleChange;
{This notifies the owner (usually TSeriesList) of a change in style of this series.}
    property OnWarning: TOnWarningEvent read FOnWarning write FOnWarning;
{This notifies the owner (usually TSeriesList) of a change in style of this series.}

    {property OnXMinChange: TOnMinMaxChangeEvent read FOnXMinChange write FOnXMinChange;
    property OnXMaxChange: TOnMinMaxChangeEvent read FOnXMaxChange write FOnXMaxChange;
    property OnYMinChange: TOnMinMaxChangeEvent read FOnYMinChange write FOnYMinChange;
    property OnYMaxChange: TOnMinMaxChangeEvent read FOnYMaxChange write FOnYMaxChange;}

    {property OnAddPoint: TNotifyEvent read FOnAddPoint write FOnAddPoint;}
  end;

  function Compare(Item1, Item2: Pointer): Integer;

implementation

uses
  Datalist, Plot;

{TSeries Constructor and Destructor:-------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TSeries.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates Pen and initializes many things
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TSeries.Create(AOwner: TCollection);
var
  i: Integer;
begin
{First call the ancestor:}
  inherited Create(AOwner);

  FPlot := TPlot(TSeriesList(AOwner).Plot);

{create sub-components:}
  FBrush := TBrush.Create;
  //FBrush.Bitmap := nil;
  FPen := TPen.Create;

{we insert the default values that cannot be "defaulted":}
{set names and color:}
  FName := Format('Series %d', [Index]);
  FPen.Color := MyColorValues[Index mod 16];
  FPen.Width := 1;
{make the brush color paler by 70%:}
  FBrush.Color := Misc.GetPalerColor(FPen.Color, 70);

  mDataStatus := dsNone;
  FDefSize := 256;
  FDeltaX := 0;
  FDeltaY := 0;
  FNoPts := 0;
  FYAxisIndex := 1;
  //FXAxis := TPlot(TSeriesList(Self.Collection).Plot).Axes[0];
  //FYAxis := TPlot(TSeriesList(Self.Collection).Plot).Axes[1];
{may be overwritten by Loaded.}

  //FAxisList := nil;

  FSymbolSize := 5;
{10% of Y Axis maximum bubble size:}
  FBubbleSize := 10;

  mDependentSeries := TList.Create;
  FXDataSeries := nil;
  FXData := nil;
  FXStringData := nil;
  FYData := nil;
  for i := 0 to MAX_MULTIPLICITY-1 do
  begin
    FExtraData[i] := nil;
  end;
  FMultiplicity := 0;

  mNoAllocPts := 0;

  Fd2Y_dX2 := nil;

  FHighs := nil;
  FLows := nil;
  FHighCount := 0;
  FLowCount := 0;
  FHighCapacity := 0;
  FVisible := TRUE;

  nTerms := 0;
end;

{------------------------------------------------------------------------------
   Destructor: TSeries.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Frees Pen and events
 Known Issues: would like a better solution to FXDataRefCount
 ------------------------------------------------------------------------------}
Destructor TSeries.Destroy;
begin
  if Assigned(FOnDestroy) then OnDestroy(Self);

  FOnDestroy := nil;
  FOnStyleChange := nil;
  FOnDataChange := nil;
  FOnExpressionChange := nil;
  FVisible := FALSE;
  ClearSpline;
  ClearHighsLows;
  FBrush.Free;
  FPen.Free;

  DelData;

  if (Assigned(FXDataSeries)) then
    FXDataSeries.RemoveDependentSeries(Self);
  if (Assigned(FXStringData)) then
    XStringData.Free;

  mDependentSeries.Free;

{then call ancestor:}
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Loaded
  Description: standard Loaded method
       Author: Mat Ballard
 Date created: 07/06/2003
Date modified: 07/06/2003 by Mat Ballard
      Purpose: implements Loaded
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Loaded;
begin
  //FXAxis := TPlot(TSeriesList(Self.Collection).Plot).Axes[0];
  //FYAxis := TPlot(TSeriesList(Self.Collection).Plot).Axes[1];
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.AssignTo(Dest: TPersistent);
{var
  i: Integer;}
begin
{we DON'T call the ancestor, because TPersistent.AssignTo simply throws an
 exception:
  inherited AssignTo(Dest);}
{Published variables:}
  TSeries(Dest).Brush.Assign(FBrush);
  TSeries(Dest).DefSize := FDefSize;
  TSeries(Dest).DeltaX := FDeltaX;
  TSeries(Dest).DeltaY := FDeltaY;
  TSeries(Dest).HighLow := FHighLow;
  TSeries(Dest).DeltaX := FDeltaX;
  TSeries(Dest).Name := FName;
  TSeries(Dest).Pen.Assign(FPen);
  TSeries(Dest).ShadeLimits := FShadeLimits;
  TSeries(Dest).Symbol := FSymbol;
  TSeries(Dest).SymbolSize := FSymbolSize;
  TSeries(Dest).Visible := Visible;

{Data:}
  if (FXStringData <> nil) then
    TSeries(Dest).XStringData.Assign(FXStringData);
  TSeries(Dest).AddData(FXData, FYData, FNoPts);
  TSeries(Dest).ZData := FZData;
end;


{Begin Set and Get Functions and Procedures----------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.SetBrush
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: sets the Brush Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetBrush(Value: TBrush);
begin
  FBrush.Assign(Value);
  DoStyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.TSeries
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 10/05/2002 by Mat Ballard
      Purpose: sets the BubbleSize in stBubble mode
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetBubbleSize(Value: TPercent);
begin
  if (FBubbleSize <> Value) then
  begin
    FBubbleSize := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetDeltaX
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the DeltaX displacement Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetDeltaX(Value: Integer);
begin
  if (FDeltaX <> Value) then
  begin
    FDeltaX := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetDeltaY
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the DeltaY displacement Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetDeltaY(Value: Integer);
begin
  if (FDeltaY <> Value) then
  begin
    FDeltaY := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetExpression
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Name Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetExpression(Value: String);
begin
  if (FName <> Value) then
  begin
    FExpression := Value;
    if Assigned(FOnExpressionChange) then
      OnExpressionChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetIsLine
  Description: standard property Setting procedure
       Author: Mat Ballard
 Date created: 08/25/2002
Date modified: 08/25/2002 by Mat Ballard
      Purpose: sets the IsLine Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetIsLine(Value: Boolean);
begin
  if (FIsLine <> Value) then
  begin
    FIsLine := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetMultiplicity
  Description: standard property Setting procedure
       Author: Mat Ballard
 Date created: 10/02/2002
Date modified: 10/02/2002 by Mat Ballard
      Purpose: sets the Multiplicity Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetMultiplicity(Value: Byte);
begin
  if (FSeriesType = stMultiple) then
    SetExtraDataMemSize(Value);
  DoDataChange;  
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetName
  Description: standard property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Name Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetName(Value: String);
begin
  if (FName <> Value) then
  begin
    FName := Value;
    DoDataChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetNoPts
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 08/10/2001
Date modified: 08/10/2001 by Mat Ballard
      Purpose: sets the NoPts Property for Externally-maintained data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetNoPts(Value: Integer);
begin
  if (mDataStatus = dsExternal) then
  begin
    if (FNoPts <> Value) then
    begin
      FNoPts := Value;
      DoDataChange;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetPen
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Pen Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
  DoStyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetSeriesType
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 10/01/2002
Date modified: 10/01/2002 by Mat Ballard
      Purpose: sets the SeriesType Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetSeriesType(Value: TSeriesType);
var
  NewNo: Integer;
begin
  if (FSeriesType <> Value) then
  begin

    case Value of
      stComplex, stBubble: NewNo := 1;
      stError: NewNo := 2;
      stMultiple: NewNo := Max(FMultiplicity, 1);
    else
      NewNo := 0;
    end;
    SetExtraDataMemSize(NewNo);
    FSeriesType := Value;
  end;
  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetExtraDataMemSize
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 10/01/2002
Date modified: 10/01/2002 by Mat Ballard
      Purpose: sets the ShadeLimits Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetExtraDataMemSize(NewNo: Integer);
var
  i: Integer;
begin
  if (NewNo > FMultiplicity) then
  begin // create some ExtraData arrays:
    if (mNoAllocPts > 0) then
      for i := FMultiplicity to NewNo-1 do
        ReAllocMem(FExtraData[i], mNoAllocPts);
  end
  else if (NewNo < FMultiplicity) then
  begin // delete some ExtraData arrays:
    for i := FMultiplicity-1 downto NewNo do
    begin
      ReAllocMem(FExtraData[i], 0);
      FExtraData[i] := nil;
    end;
  end;
  FMultiplicity := NewNo;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetShadeLimits
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the ShadeLimits Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetShadeLimits(Value: Boolean);
begin
  if (FShadeLimits <> Value) then
  begin
    FShadeLimits := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetShowFit
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 09/25/2002
Date modified: 09/25/2002 by Mat Ballard
      Purpose: sets the ShowFit: do we show the least-squares fit ?
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetShowFit(Value: Boolean);
begin
  if ((FShowFit <> Value) and (nTerms > 0)) then
  begin
    FShowFit := Value;
    DoStyleChange;
  end;  
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetXStringData
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the XStringData: the X data as text strings
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetXStringData(Value: TStringList);

  procedure NukeStringList;
  begin
    if (Assigned(FXStringData)) then
    begin
      FXStringData.Free;
      FXStringData := nil;
    end;
    //TPlot(FPlot).Axes[0].SetLabelSeries(nil);
    TPlot(FPlot).Axes[0].SetLabelSeries(nil);
    mDataStatus := dsInternal;
    exit;
  end;

begin
  if ((mDataStatus = dsNone) or (mDataStatus = dsExternal)) then
    EComponentError.Create('TSeries.SetXStringData: You cannot set the X String Data if the data is null or external !');

  if (Value = nil) then
  begin
    NukeStringList;
    exit;
  end;

  if (Value.Count = 0) then
  begin
    NukeStringList;
    exit;
  end;

  if (FXStringData = nil) then
    FXStringData := TStringList.Create;
  if (mDataStatus = dsInternal) then
    mDataStatus := dsInternalString;
{$IFDEF GUI}
  if (Integer(Value.Count) <> FNoPts) then
    ShowMessage(Format('Warning: there are %d points, but you have just added %d text X data values !',
      [FNoPts, Value.Count]));
{$ENDIF}
  FXStringData.Clear;
  FXStringData.Assign(Value);
  TPlot(FPlot).Axes[0].SetLabelSeries(Self);
  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetSymbol
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Symbol Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetSymbol(Value: TSymbol);
begin
  if (FSymbol <> Value) then
  begin
    FSymbol := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetSymbolSize
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the SymbolSize Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetSymbolSize(Value: Integer);
begin
  if (FSymbolSize <> Value) then
  begin
    FSymbolSize := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetSymbolCheck
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 10/03/2002
Date modified: 10/03/2002 by Mat Ballard
      Purpose: sets the SymbolCheck Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetSymbolCheck(Value: Boolean);
begin
  if (FSymbolCheck <> Value) then
  begin
    FSymbolCheck := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetVisible
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Visible Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetVisible(Value: Boolean);
begin
  if (FVisible <> Value) then
  begin
{Can't become visible if Axes or Data have not been set:
    if ((TPlot(FPlot).Axes[0] = nil) or (TPlot(FPlot).Axes[FYAxisIndex] = nil)) then raise
      EInvalidPointer.CreateFmt('TSeries.SetVisible: the X and Y Axis pointers are invalid !' +
        #13#10+ '(X: %p; Y: %p)', [TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[FYAxisIndex]]);}

    FVisible := Value;
    DoStyleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetYAxis
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 10/25/2003
Date modified: 10/25/2003 by Mat Ballard
      Purpose: sets the YAxis Property
 Known Issues:
 ------------------------------------------------------------------------------
procedure TSeries.SetYAxis(Value: TAxis);
begin
  if (Value.AxisType >= atPrimaryY) then
    TPlot(FPlot).Axes[FYAxisIndex] := Value;
end;}

{------------------------------------------------------------------------------
     Function: TSeries.GetYAxis
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2003
Date modified: 10/25/2003 by Mat Ballard
      Purpose: gets the YAxisIndex Property
 Known Issues: We define YAxisIndex to run from 1 to FAxisList.Count-1
 ------------------------------------------------------------------------------}
function TSeries.GetYAxis: TAxis;
begin
  Result := TPlot(FPlot).Axes[FYAxisIndex];
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetYAxisIndex
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the YAxisIndex Property
 Known Issues: We define YAxisIndex to run from 1 to FAxisList.Count-1
 ------------------------------------------------------------------------------}
procedure TSeries.SetYAxisIndex(Value: Byte);
begin
  if ((Value < 1) or
      (Value >= TPlot(FPlot).Axes.Count)) then raise
    ERangeError.CreateFmt('TSeries.SetYAxisIndex: the new Y-Axis Index (%d) must be between 1 and %d !', [TPlot(FPlot).Axes.Count-1]);

  FYAxisIndex := Value;
  DoStyleChange;
end;

procedure TSeries.SetZData(Value: Single);
begin
  if (FZData <> Value) then
  begin
    FZData := Value;
    DoDataChange;
  end;
end;

{end Set procedures, begin general procedures ---------------------------------}
{------------------------------------------------------------------------------
     Function: TSeries.AllocateNoPts
  Description: allocates memory for data points
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: memory management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.AllocateNoPts(Value: Integer);
var
  i: Integer;
begin
  try
    if (Assigned(FXDataSeries)) then
    begin
{we don't allocate memory for X data that is held in a different series:}
      FXData := Self.FXDataSeries.XData
{doesn't hurt, but should not be neccessary.}
    end
    else
    begin
      ReAllocMem(FXData, Value * SizeOf(Single));
    end;
    ReAllocMem(FYData, Value * SizeOf(Single));
    for i := 0 to FMultiplicity-1 do
      ReAllocMem(FExtraData[i], Value * SizeOf(Single));
  except
{$IFDEF GUI}
    ShowMessageFmt('Problem with the %s data series ! + #10 + Requested No Pts = %d, requested memory = %d bytes',
      [FName, Value, Value * SizeOf(Single)]);
{$ENDIF}
    raise;
  end;
  mNoAllocPts := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DoStyleChange
  Description: Fires the OnStyleChange event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: event handling
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DoStyleChange;
begin
  if (Assigned(FOnStyleChange)) then OnStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DoDataChange
  Description: Fires the OnDataChange event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 08/10/2001 by Mat Ballard
      Purpose: event handling
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DoDataChange;
begin
  if (Assigned(FOnDataChange)) then OnDataChange(Self);
end;

{Data manipulation Functions and Procedures----------------------------------}
{------------------------------------------------------------------------------
     Function: TSeries.AddData
  Description: adds data from an externally-managed array
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 10/02/2002 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues: Overlaoded
 ------------------------------------------------------------------------------}
procedure TSeries.AddData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer);
var
  i: Integer;
begin
{clear any existing data:}
  if (FNoPts > 0) then DelData;

{Allocate memory:}
  AllocateNoPts(NumberOfPoints + FDefSize);

{NB: this causes terminal access violations:
    System.Move(XPointer, FXData, NumberOfPoints * SizeOf(Single));}
  if (FXDataSeries = nil) then
  begin
    for i := 0 to NumberOfPoints-1 do
      FXData^[i] := XPointer^[i];
  end;
  for i := 0 to NumberOfPoints-1 do
    if (IsNAN(YPointer^[i])) then
      MakeNAN(FYData^[i])
     else
      FYData^[i] := YPointer^[i];

  mDataStatus := dsInternal;
  FNoPts := NumberOfPoints;

{find the new min and max:}
  GetBounds; {which calls ResetBounds}

  DoDataChange;
end;

procedure TSeries.AddDataEx(XPointer, YPointer: pSingleArray; ExtraPointers: array of pSingleArray; NumberOfPoints: Integer);
var
  i, j: Integer;
begin
  AddDataEx(XPointer, YPointer, ExtraPointers, NumberOfPoints);
  for j := 0 to FMultiplicity-1 do
    for i := 0 to NumberOfPoints-1 do
      if (IsNAN(ExtraPointers[j]^[i])) then
        MakeNAN(ExtraPointers[j]^[i])
       else
        FExtraData[j]^[i] := ExtraPointers[j]^[i];
  DoDataChange;
end;
{------------------------------------------------------------------------------
    Procedure: TSeries.MakeXDataIndependent
  Description: This procedure makes an internal copy of external X Data
       Author: Mat Ballard
 Date created: 08/31/2000
Date modified: 08/31/2000 by Mat Ballard
      Purpose: series management
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.MakeXDataIndependent: Boolean;
var
  i: Integer;
  pOldXData: pSingleArray;
{$IFDEF GUI}
  Msg: String;
{$ENDIF}  
begin
  Result := FALSE;
  if (FXDataSeries = nil) then exit;

  pOldXData := FXData;
  try
    ReAllocMem(FXData, mNoAllocPts * SizeOf(Single));
{NB: the following generates gross access violations:
    System.Move(pOldXData, FXData, FNoPts * SizeOf(Single));}
    for i := 0 to FNoPts-1 do
      FXData^[i] := pOldXData^[i];
    FXDataSeries.RemoveDependentSeries(Self);
    FXDataSeries := nil;
    Result := TRUE;
  except
{$IFDEF GUI}
    Msg := Format('Problem with the %s data series ! + #10 + Requested No Pts = %d, requested memory = %d bytes',
      [FName, mNoAllocPts, mNoAllocPts * SizeOf(Single)]);
    ShowMessage(Msg);
{$ENDIF}
    raise;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.MakeXDataIndependent
  Description: This procedure destroys the internal X Data and makes this Series
               depend on a Master series.
       Author: Mat Ballard
 Date created: 08/31/2002
Date modified: 08/31/2002 by Mat Ballard
      Purpose: series management
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.MakeXDataDependent(MasterSeries: TSeries): Boolean;
begin
  Result := FALSE;
  if (MasterSeries.NoPts >= Self.NoPts) then
  begin
    ReAllocMem(FXData, 0);
    FXData := MasterSeries.XData;
    FXDataSeries := MasterSeries;
    MasterSeries.AddDependentSeries(Self);
    Result := TRUE;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MakeHighLowOpenClose
  Description: This procedure makes this Series a "High-Low-Open-Close" type in one simple command.
       Author: Mat Ballard
 Date created: 08/31/2002
Date modified: 08/31/2002 by Mat Ballard
      Purpose: series (financial) management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MakeHighLowOpenClose;
begin
  FMultiplicity := 4;
  SetSeriesType(stMultiple);
  if (Length(FName) = 0) then
    FName := 'High-Low-Open-Close'
   else
    FName := FName + ' - HLOC';
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MakeXDataString
  Description: This procedure creates storage space for string data
       Author: Mat Ballard
 Date created: 08/31/2002
Date modified: 08/31/2002 by Mat Ballard
      Purpose: series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MakeXDataString;
begin
  if (mDataStatus = dsExternal) then
    EComponentError.Create('You cannot make X Data in another Series Text !');
  if (mDataStatus <> dsInternalString) then
  begin
    mDataStatus := dsInternalString;
    IncMemSize;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddDependentSeries
  Description: This function ADDS a series that depends on this series' X-Data from the list of dependent series.
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AddDependentSeries(ASeries: TSeries): Boolean;
{var
  pASeries: Pointer;}
begin
  if (mDependentSeries.IndexOf(ASeries) < 0) then
  begin
    mDependentSeries.Add(ASeries);
    AddDependentSeries := TRUE;
  end
  else
    AddDependentSeries := FALSE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.RemoveDependentSeries
  Description: This function REMOVES a series that depends on this series' X-Data from the list of dependent series.
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.RemoveDependentSeries(ASeries: TSeries): Boolean;
{var
  pASeries: Pointer;}
begin
  if (mDependentSeries.IndexOf(ASeries) >= 0) then
  begin
    mDependentSeries.Remove(ASeries);
    RemoveDependentSeries := TRUE;
  end
  else
    RemoveDependentSeries := FALSE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetXDataRefCount
  Description: This function returns the number of dependent series
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Word
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetXDataRefCount: Word;
begin
  GetXDataRefCount := mDependentSeries.Count;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetXAxis
  Description: This function returns the X Axis to which this series is bound
       Author: Mat Ballard
 Date created: 09/10/2002
Date modified: 09/10/2003 by Mat Ballard
      Purpose: data management
 Return Value: Word
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetXAxis: TAxis;
begin
  Result := TPlot(FPlot).Axes[0];
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetYAxis
  Description: This function returns the Y Axis to which this series is bound
       Author: Mat Ballard
 Date created: 09/10/2002
Date modified: 09/10/2003 by Mat Ballard
      Purpose: data management
 Return Value: Word
 Known Issues:
 ------------------------------------------------------------------------------
function TSeries.GetYAxis: TAxis;
begin
  if (TPlot(FPlot).Axes[FYAxisIndex] = nil) then
    TPlot(FPlot).Axes[FYAxisIndex] := TPlot(TSeriesList(Self.Collection).Plot).Axes[1];
  Result := TPlot(FPlot).Axes[FYAxisIndex];
end;}

{------------------------------------------------------------------------------
     Function: TSeries.AssumeMasterSeries
  Description: This function makes this series' X-Data the MASTER for the given list of dependent series.
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AssumeMasterSeries(
  XPts: Integer;
  OldMaster: TSeries;
  AList: TList): Boolean;
var
  i: Integer;
begin
{There are many reasons why this might be a bad idea:}
  if (OldMaster = nil) then raise
    EComponentError.Create(Self.Name +
      ' cannot Assume Mastery from a NIL Series !');

  if (OldMaster <> FXDataSeries) then raise
    EComponentError.Create(Self.Name +
      ' cannot Assume Mastery from ' + FXDataSeries.Name +
      ' Because ' + FXDataSeries.Name + ' is NOT its X Data Master !');

  if (XPts <> FNoPts) then raise
    EComponentError.CreateFmt(Self.Name +
      ' (%d Points) cannot Assume a Master (X) Series  With %d Points !',
      [FNoPts, XPts]);

{this last is probably redundant because of test #2:}
  if (mDependentSeries.Count > 0) then raise
    EComponentError.Create(Self.Name +
      ' cannot Assume a Master (X) Series  Because it already IS a Master Series !');

  for i := 0 to AList.Count-1 do
  begin
    if (AList.Items[i] <> Self) then
    begin
{add these dependent series to our own list:}
      mDependentSeries.Add(AList.Items[i]);
{tell them that this series is now the Master:}
      TSeries(AList.Items[i]).ResetXDataSeries(OldMaster, Self);
    end;
  end;

{the X Data is now internal to this series:}
  FXDataSeries := nil;
{note that we already KNOW the location of the X Data: FXData !}

  AssumeMasterSeries := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.ResetXDataSeries
  Description: When a new series Assumes X Data Master status, it has to tell
               all the dependent series
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.ResetXDataSeries(OldSeries, NewSeries: TSeries): Boolean;
begin
  if (FXDataSeries = OldSeries) then
  begin
    FXDataSeries := NewSeries;
    Result := TRUE;
  end
  else
    Result := FALSE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.PointToData
  Description: uses data from an externally-managed array
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.PointToData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean;
begin
  PointToData := FALSE;
  if (mDataStatus = dsNone) then
  begin
    mDataStatus := dsExternal;
    FXData := XPointer;
    FYData := YPointer;
    FNoPts := NumberOfPoints;
    GetBounds; {which calls ResetBounds}
    DoDataChange;
    PointToData := TRUE;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddDrawPoint
  Description: adds a point then draws it
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 10/02/2002 by Mat Ballard
      Purpose: data management and screen display
 Return Value: the number of data points
 Known Issues: 
 ------------------------------------------------------------------------------}
function TSeries.AddDrawPoint(X, Y: Single; ACanvas: TCanvas): Integer;
begin
{Add the point; we don't fire any events, but we do adjust axes if required:
 this may trigger a re-draw if Min/Max are exceeded:}
  Result := AddPoint(X, Y, FALSE, TRUE);
  if (FVisible) then
    DrawLatestPoint(ACanvas);
end;

function TSeries.AddDrawPointEx(X, Y: Single; Extra: array of Single; ACanvas: TCanvas): Integer;
var
  i: Integer;
begin
{Add the point; we don't fire any events, but we do adjust axes if required:
 this may trigger a re-draw if Min/Max are exceeded:}
  Result := AddPointEx(X, Y, Extra, FALSE, TRUE);
  for i := 0 to FMultiplicity-1 do
    FExtraData[i]^[FNoPts-1] := Extra[i];
  if (FVisible) then
    DrawLatestPoint(ACanvas);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawLatestPoint
  Description: draws the latest data point
       Author: Mat Ballard
 Date created: 10/02/2002
Date modified: 10/02/2002 by Mat Ballard
      Purpose: data management
 Return Value: the number of data points
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawLatestPoint(ACanvas: TCanvas);
var
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.AddDrawPoint: ACanvas is nil !');
{$ENDIF}
{  if (TPlot(FPlot).Axes[0] = nil) then
    TPlot(FPlot).Axes[0] := TPlot(TSeriesList(Self.Collection).Plot).Axes[0];
  if (TPlot(FPlot).Axes[FYAxisIndex] = nil) then
    TPlot(FPlot).Axes[FYAxisIndex] := TPlot(TSeriesList(Self.Collection).Plot).Axes[1];}
{Draw from last to this point:}
  ACanvas.Pen.Assign(FPen);
  if (FNoPts > 1) then
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[FNoPts-2])+ FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[FNoPts-2]) + FDeltaY;
    ACanvas.MoveTo(iX, iY);
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[FNoPts-1]) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[FNoPts-1]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end
  else
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[FNoPts-1]) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[FNoPts-1]) + FDeltaY;
  end;
  if ((FSymbol <> syNone) and (FSymbolSize > 0)) then
  begin
    ACanvas.Brush.Assign(FBrush);
    DrawSymbol(ACanvas, iX, iY);
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddPoint
  Description: adds a data point, increasing memory if required
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 10/02/2002 by Mat Ballard
      Purpose: data management
 Return Value: the number of data points
 Known Issues: 
 ------------------------------------------------------------------------------}
function TSeries.AddPoint(X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(mDataStatus <> dsExternal,
    'TSeries.AddPoint: I cannot add data points to the ' + Name + ' Series' +
      #13#10+ 'because it is externally managed !');
{$ENDIF}

  if (mDataStatus = dsNone) then
  begin
    mDataStatus := dsInternal;
    IncMemSize;
    ResetBounds;
    {if (FStartTime = 0.0) then
      FStartTime := Now;}
  end;

{Check memory available:}
  if (FNoPts >= mNoAllocPts-2) then
    IncMemSize;

{If the X data is in another series, then we do not add it:}
  if (Assigned(FXDataSeries)) then
  begin
{check validity of the External X data:}
    if (FXDataSeries = nil) then raise
      EAccessViolation.Create('TSeries.AddPoint: I cannot add Y values to the ' + Name +
        ' Series Because the FXDataSeries is undefined !');
    if (FXDataSeries.NoPts < FNoPts) then raise
      ERangeError.CreateFmt('TSeries.AddPoint: the External X data series contains %d points, and I contain %d',
        [FXDataSeries.NoPts, FNoPts]);
    {if (FXDataSeries.XData = nil) then  raise
      EAccessViolation.Create('TSeries.AddPoint: I cannot add Y values to the ' + Name +
        ' Series Because '  + 'the FXDataSeries X Data pointer is undefined !');}
  end
  else
  begin
{save the X data:}
    FXData^[FNoPts] := X;
  end;

{save the Y data:}
  FYData^[FNoPts] := Y;

{Check the min and max X and Y properties of the series,
 and adjust axes as required:}
  CheckBounds(FNoPts, AdjustAxes);
  Inc(FNoPts);
  Result := FNoPts;
  if (FireEvent) then
    DoDataChange;
end;

function TSeries.AddPointEx(X, Y: Single; Extra: array of Single; FireEvent, AdjustAxes: Boolean): Integer;
var
  i: Integer;
begin
  Result := AddPoint(X, Y, FALSE, AdjustAxes);
  for i := 0 to FMultiplicity-1 do
    FExtraData[i]^[FNoPts-1] := Extra[i];
  if (FireEvent) then
    DoDataChange;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddStringPoint
  Description: adds a data point with a String X value, increasing memory if required
       Author: Mat Ballard
 Date created: 11/16/2000
Date modified: 10/02/2002 by Mat Ballard
      Purpose: data management
 Return Value: the number of data points
 Known Issues: 
 ------------------------------------------------------------------------------}
function TSeries.AddStringPoint(XString: String; X, Y: Single;
  FireEvent, AdjustAxes: Boolean): Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(mDataStatus <> dsExternal,
    'TSeries.AddStringPoint: I cannot add data points to the ' + Name + ' Series' +
      #13#10+ 'because it is externally managed !');
{$ENDIF}

  if (mDataStatus = dsNone) then
  begin
    mDataStatus := dsInternalString;
    IncMemSize;
    ResetBounds;
  end;

{If the X data is in another series, then we do not add it:}
  if (FXDataSeries = nil) then
  begin
    if (mDataStatus <> dsInternalString) then
      MakeXDataString;
{save the X string data:}
    FXStringData.Add(XString);
  end;
  Result := AddPoint(X, Y, FireEvent, AdjustAxes);
end;

function TSeries.AddStringPointEx(XString: String; X, Y: Single;
  Extra: array of Single; FireEvent, AdjustAxes: Boolean): Integer;
var
  i: Integer;
begin
  Result := AddStringPoint(XString, X, Y, FALSE, AdjustAxes);
  for i := 0 to FMultiplicity-1 do
    FExtraData[i]^[FNoPts-1] := Extra[i];
  if (FireEvent) then
    DoDataChange;
end;
{------------------------------------------------------------------------------
     Function: TSeries.DelPoint
  Description: deletes the point nearest to X and Y
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: the new number of points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.DelPoint(X, Y: Single; Confirm: Boolean): Integer;
{This deletes a single point, the closest one, from the xy data.}
var
  i, ThePoint: Integer;
  Distance, MinDistance: Single;
begin
  Result := 0;
  if (FNoPts <= 0) then raise
    ERangeError.CreateFmt('TSeries.DelPoint: this series (%s) contains no data points, so I cannot delete any !', [FName]);

  MinDistance := 3.4e38;
  ThePoint := -1;
  for i := 0 to FNoPts-1 do
  begin
    Distance := Abs(X - FXData^[i]) + Abs(Y - FYData^[i]);
    if (MinDistance > Distance) then
    begin
      ThePoint := i;
    end;
  end;

  if (ThePoint >= 0) then
    Result := DelPointNumber(ThePoint, Confirm);
end;

{------------------------------------------------------------------------------
     Function: TSeries.DelPointNumber
  Description: deletes ThePoint by its index
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: the new number of points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.DelPointNumber(ThePoint: Integer; Confirm: Boolean): Integer;
{This deletes a single point, the Nth one, from the xy data.}
{}
{Note: this DOES NOT delete the X Value of externally-maintained X data
 values, so it shifts the upper half of the series one point to the left.}
var
  i, j: Integer;
{$IFDEF GUI}
  TheMessage: String;
{$ENDIF}  
begin
  if (FNoPts <= 0) then raise
    ERangeError.CreateFmt('TSeries.DelPointNumber: this series (%s) contains no data points, so I cannot delete any !',
      [FName]);
  if (ThePoint >= FNoPts) then raise
    ERangeError.CreateFmt('TSeries.DelPointNumber: this series (%s) contains %d points, so I cannot delete point %d !',
      [FName, FNoPts, ThePoint]);
  if (mDependentSeries.Count > 0) then
  begin
    ERangeError.CreateFmt(
      'I cannot delete any points when %d other series use my X Data !',
      [mDependentSeries.Count]);
  end;

{$IFDEF GUI}
  Result := 0;
  if (Confirm) then
  begin
    TheMessage := Format('Delete Point' +' %d: (%e.3, %e.3) ?',
      [ThePoint, FXData^[ThePoint], FYData^[ThePoint]]);
    if (mrNo = MessageDlg(
  {$IFDEF LINUX}
    'Delete Point',
  {$ENDIF}
      TheMessage,
      mtWarning,
      [mbYes, mbNo],
      0)) then
        exit;
  end;
{$ENDIF}

{we now use the slower method to be more consistent with the
 dynamic array approach:}
  if (FXDataSeries = nil) then
  begin
    for i := ThePoint to FNoPts-2 do
    begin
      FXData^[i] := FXData^[i+1];
    end;
    if ((Assigned(FXStringData)) and
        (Integer(FXStringData.Count) > ThePoint)) then
      FXStringData.Delete(ThePoint);
  end;

  for i := ThePoint to FNoPts-1 do
  begin
    FYData^[i] := FYData^[i+1];
    for j := 0 to FMultiplicity-1 do
      FExtraData[j]^[i] := FExtraData[j]^[i+1];
  end;

  Dec(FNoPts);

  DoDataChange;
  Result := FNoPts;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Clear
  Description: deletes all the data, leaving the series intact
       Author: Mat Ballard
 Date created: 04/25/2003
Date modified: 04/25/2003 by Mat Ballard
      Purpose: Series analysis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Clear;
begin
  if (not Self.DelData) then
    EComponentError.CreateFmt('TSeries.Clear: could not delete data from series "%s"', [Self.FName]);
end;

{------------------------------------------------------------------------------
     Function: TSeries.DelData
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: deletes an entire data set. It only works on internal data sets.
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.DelData: Boolean;
var
  i: Integer;
begin
  Result := FALSE;
  if (mDataStatus = dsNone) then exit;

  if (mDependentSeries.Count > 0) then
  begin
{Merde ! this series is being destroyed, but other series depend on it !
 we therefore need to "pass the buck": pass the X Data that we've created, and
 the list of dependent series to another series:}
    TSeries(mDependentSeries.Items[0]).AssumeMasterSeries(FNoPts, Self, mDependentSeries);
{and now, the X Data is managed by an external series:}
    //FExternalXSeries := TRUE;
    mDependentSeries.Clear;
  end;

  if (mDataStatus in [dsInternal, dsInternalString]) then
  begin
    if (FXDataSeries = nil) then
    begin
      ReAllocMem(FXData, 0);
      if (Assigned(FXStringData)) then
      begin
        FXStringData.Free;
        FXStringData := nil;
        TPlot(FPlot).Axes[0].SetLabelSeries(nil);
      end;
    end;
    ReAllocMem(FYData, 0);
    FXData := nil;
    FYData := nil;
    for i := 0 to FMultiplicity-1 do
    begin
      ReAllocMem(FExtraData[i], 0);
    end;
  end;

  FNoPts := 0;
  mDataStatus := dsNone;
  mNoAllocPts := 0;
  ResetBounds;

  Result := TRUE;
  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ClearHighsLows
  Description: frees the Highs and Lows, and their Counts and Capacities
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series analysis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ClearHighsLows;
begin
  if (Assigned(FHighs)) then
  begin
    FreeMem(FHighs, FHighCapacity * SizeOf(Integer));
    FHighs := nil;
  end;
  if (Assigned(FLows)) then
  begin
    FreeMem(FLows, FHighCapacity * SizeOf(Integer));
    FLows := nil;
  end;
  
  FHighLow := [];
  FHighCapacity := 10;
  FHighCount := 0;
  FLowCount := 0;
end;

{------------------------------------------------------------------------------
     Function: TSeries.Average
  Description: calculates the average of a series over a range
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: numerical calculation
 Return Value: the average: single
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.Average(TheLeft, TheRight: Single): Single;
var
  i,
  Start,
  Finish,
  Number: Integer;
  Sum: Single;
begin
  if (TheLeft > TheRight) then
  begin
{swap TheLeft and TheRight}
    Sum := TheLeft;
    TheLeft := TheRight;
    TheRight := Sum;
  end;

{get the TheLeft and TheRight points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

{adjust TheLeft and TheRight:}
  if (FXData^[Start] < TheLeft) then
    Inc(Start);
  if (FXData^[Finish] > TheRight) then
    Dec(Finish);

{initialize:}
  Number := 0;
  Sum := 0;
  for i := Start to Finish do
  begin
    Sum := Sum + FYData^[i];
    Inc(Number);
  end;

  Average := Sum / Number;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Linearize
  Description: Linearizes (turns into a straight line) the data of a series over a range
       Author: Mat Ballard
 Date created: 05/30/2001
Date modified: 05/30/2001 by Mat Ballard
      Purpose: numerical calculation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Linearize(TheLeft, TheRight: Single);
var
  i,
  Start,
  Finish: Integer;
  Slope, Intercept: Single;
begin
  if (TheLeft > TheRight) then
  begin
{swap TheLeft and TheRight}
    Slope := TheLeft;
    TheLeft := TheRight;
    TheRight := Slope;
  end;

{get the TheLeft and TheRight points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

{adjust TheLeft and TheRight:}
  if (FXData^[Start] < TheLeft) then
    Inc(Start);
  if (FXData^[Finish] > TheRight) then
    Dec(Finish);

{initialize:}
  Slope := (FYData^[Finish] - FYData^[Start]) /
    (FXData^[Finish] - FXData^[Start]);
  Intercept := FYData^[Finish] - Slope * FXData^[Finish];
  for i := Start+1 to Finish-1 do
    FYData^[i] := Slope * FXData^[i] + Intercept;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Zero
  Description: zeros the data of a series over a range
       Author: Mat Ballard
 Date created: 05/30/2001
Date modified: 05/30/2001 by Mat Ballard
      Purpose: numerical calculation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Zero(TheLeft, TheRight: Single);
var
  i,
  Start,
  Finish: Integer;
  TheTemp: Single;
begin
  if (TheLeft > TheRight) then
  begin
{swap TheLeft and TheRight}
    TheTemp := TheLeft;
    TheLeft := TheRight;
    TheRight := TheTemp;
  end;

{get the TheLeft and TheRight points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

{adjust TheLeft and TheRight:}
  if (FXData^[Start] < TheLeft) then
    Inc(Start);
  if (FXData^[Finish] > TheRight) then
    Dec(Finish);

{initialize:}
  for i := Start to Finish do
    FYData^[i] := 0;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MovingAverage
  Description: Calculates the movong average
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Smoothing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MovingAverage(Span: Longint);
var
  i, j: Integer;
  Left, Right: Longint;
  AverageData: pSingleArray;
begin
{allocate memory for arrays:}
  GetMem(AverageData, FNoPts * SizeOf(Single));

  for i := 0 to FNoPts-1 do
  begin
    AverageData^[i] := 0;
    Left := Longint(i) - Span;
    Right := Longint(i) + Span;

    if (Left < 0) then
    begin
      Right := 2*i;
      Left := 0;
    end;
    if (Right >= Longint(FNoPts)) then
    begin
      Left := i - (FNoPts-1 - i);
      Right := FNoPts-1;
    end;

    for j := Left to Right do
    begin
      AverageData^[i] := AverageData^[i] + FYData^[j];
    end;
    AverageData^[i] := AverageData^[i] / (1 + Right - Left);
  end;

{NB: the following generates gross access violations:
  System.Move(AverageData, FYData, FNoPts * SizeOf(Single));}
  for i := 0 to FNoPts-1 do
    FYData^[i] := AverageData^[i];

  FreeMem(AverageData, FNoPts * SizeOf(Single));
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DoSpline
  Description: Does the cubic spline of the data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Places the cubic spline interpolation into X and Y
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DoSpline(Density: Integer; pSplineSeries: TSeries);
var
  i, j: Integer;
  dX,
  X: Single;
begin
{calculate the ...}
  SecondDerivative;

{Index of the new spline points:}
  for i := 0 to FNoPts-2 do
  begin
    pSplineSeries.AddPoint(FXData^[i], FYData^[i], FALSE, FALSE);
    dX := (FXData^[i+1] - FXData^[i]) / (Density+1);
    X := FXData^[i];
    for j := 0 to Density-1 do
    begin
      X := X + dX;
      pSplineSeries.AddPoint(X, SplineValue(X), FALSE, FALSE);
    end;
  end;
  pSplineSeries.AddPoint(FXData^[FNoPts-1], FYData^[FNoPts-1], FALSE, FALSE);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SplineValue
  Description: Calculates the Y co-ordinate from the cubic spline
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data manipulation
 Known Issues: yet to be done
 ------------------------------------------------------------------------------}
function TSeries.SplineValue(X: Single): Single;
var
  iLeft,
  iRight,
  i: integer;
  dX,
  LeftX,
  RightX: Single;
begin
{in initialize left and right indices:}
  iLeft := 0;
  iRight := FNoPts-1;

{bracket the X value using binary search:}
  while (iRight - iLeft > 1) do
  begin
    i := (iRight+iLeft) div 2;
    if (FXData^[i] > X) then
      iRight := i
     else
      iLeft := i;
  end;
{width of bracketing interval is:}
  dX := FXData^[iRight] - FXData^[iLeft];

{should we chuck a loopy ?}
  if (dX = 0.0) then raise
    ERangeError.CreateFmt('TSeries.SplineValue: bad input data (dX = 0) !' + CRLF+
      'XData[%d] = %g, XData[%d] = %g', [iRight, FXData^[iRight], iLeft, FXData^[iLeft]]);

{the right and left portions are:}
  RightX := (FXData^[iRight]-X) / dX;
  LeftX := (X-FXData^[iLeft]) / dX;

{so the cubic spline estimate is:}
  SplineValue := RightX * FYData^[iLeft] + LeftX * FYData^[iRight] +
    ((IntPower(RightX, 3) - RightX) * Fd2Y_dX2^[iLeft] +
      (IntPower(LeftX, 3) - LeftX) * Fd2Y_dX2^[iRight]) *
        Sqr(dX) / 6.0;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SecondDerivative
  Description: Does the cubic spline of the data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Calculates the second derivatives for use by SplineValue
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SecondDerivative;
var
  i: Integer;
  TempVar,
  LeftXFraction: Single;
  UpperTriangle: pSingleArray;
begin
  ClearSpline;

{allocate memory for the second derivatives:}
  //Size2ndDeriv := FNoPts * SizeOf(Single);
  GetMem(Fd2Y_dX2, FNoPts * SizeOf(Single));

  GetMem(UpperTriangle, FNoPts * SizeOf(Single));

{handle the first point: we use "natural" boundary condition of
 zero second derivative:}
  Fd2Y_dX2^[0] := 0;
  UpperTriangle^[0] := 0;

{do the loop over middle points:}
  for i := 1 to FNoPts-2 do begin
    LeftXFraction := (FXData^[i] - FXData^[i-1]) /
      (FXData^[i+1] - FXData^[i-1]);
    TempVar := LeftXFraction * Fd2Y_dX2^[i-1] + 2.0;
    Fd2Y_dX2^[i] := (LeftXFraction - 1.0) / TempVar;
    UpperTriangle^[i] := (FYData^[i+1] - FYData^[i]) / (FXData^[i+1] - FXData^[i]) -
      (FYData^[i] - FYData^[i-1]) / (FXData^[i] - FXData^[i-1]);
    UpperTriangle^[i] := (6.0 * UpperTriangle^[i] / (FXData^[i+1] - FXData^[i-1]) -
      LeftXFraction * UpperTriangle^[i-1]) / TempVar;
  end;

{handle the last point: we use "natural" boundary condition of
 zero second derivative:}
  Fd2Y_dX2^[FNoPts-1] := 0;

  for i := FNoPts-2 downto 0 do
  begin
    Fd2Y_dX2^[i] := Fd2Y_dX2^[i] * Fd2Y_dX2^[i+1] + UpperTriangle^[i];
  end;
  FreeMem(UpperTriangle, FNoPts * SizeOf(Single));
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ClearSpline
  Description: frees the second derivative memory
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Spline memory management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ClearSpline;
begin
  if (Assigned(Fd2Y_dX2)) then
  begin
    FreeMem(Fd2Y_dX2, FNoPts * SizeOf(Single));
    Fd2Y_dX2 := nil;
    //Size2ndDeriv := 0;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Differentiate
  Description: Replaces the Series Y data with its differential
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Differentiate;
var
  i: Integer;
  Differential,
  YOld: Single;
begin
{we save the first data point:}
  YOld := FYData^[0];

{now do the first point by difference (1st order):}
  FYData^[0] :=
    (FYData^[1] - FYData^[0]) / (FXData^[1] - FXData^[0]);

  for i := 1 to FNoPts-2 do
  begin
{we calculate a mid-point (2nd order) differential}
    Differential :=
      (((FYData^[i] - YOld) / (FXData^[i] - FXData^[i-1])) +
       ((FYData^[i+1] - FYData^[i]) / (FXData^[i+1] - FXData^[i])))
       / 2;
    YOld := FYData^[i];
    FYData^[i] := Differential;
  end;

{now do the last point by difference (1st order):}
  FYData^[FNoPts-1] :=
    (FYData^[FNoPts-1] - YOld) / (FXData^[FNoPts-1] - FXData^[FNoPts-2]);

{re-scale:}    
  FDeltaX := 0;
  FDeltaY := 0;
  ResetBounds;
  GetBounds;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Integrate
  Description: Replaces the Series Y data with its integral
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Integrate;
var
  i: Integer;
  Sum,
  YOld: Single;
begin
  Sum := 0;
  YOld := FYData^[0];
  for i := 1 to FNoPts-1 do
  begin
    Sum := Sum +
      (FYData^[i] + YOld) * (FXData^[i] - FXData^[i-1]) / 2;
    YOld := FYData^[i];
    FYData^[i] := Sum;
  end;
{we set the first data point:}
  FYData^[0] := 0;

{re-scale:}
  FDeltaX := 0;
  FDeltaY := 0;
  ResetBounds;
  GetBounds;
end;

{------------------------------------------------------------------------------
     Function: TSeries.Integral
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the integral of the Series from Start to Finish
 Return Value: the real area
 Known Issues: see IntegralByPoint below
 ------------------------------------------------------------------------------}
function TSeries.Integral(TheLeft, TheRight: Single): Single;
var
  Start,
  Finish: Integer;
  Sum,
  YEst: Single;
begin
  if (TheLeft > TheRight) then
  begin
{swap TheLeft and TheRight}
    Sum := TheLeft;
    TheLeft := TheRight;
    TheRight := Sum;
  end;

{get the TheLeft and TheRight points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

{adjust TheLeft and TheRight:}
  if (FXData^[Start] < TheLeft) then
    Inc(Start);
  if (FXData^[Finish] > TheRight) then
    Dec(Finish);

{Integrate the bulk:}
  Sum := IntegralByPoint(Start, Finish);

{Add the end bits:}
  if ((Start > 0) and
      (FXData^[Start] <> TheLeft)) then
  begin
    YEst := FYData^[Start-1] +
      (FYData^[Start] - FYData^[Start-1]) *
      (TheLeft - FXData^[Start-1]) / (FXData^[Start] - FXData^[Start-1]);
    Sum := Sum +
      (FXData^[Start] - TheLeft) *
      (FYData^[Start] + YEst) / 2;
  end;
  if ((Finish < FNoPts-1) and
      (FXData^[Finish] <> TheRight)) then
  begin
    YEst := FYData^[Finish] +
      (FYData^[Finish+1] - FYData^[Finish]) *
      (TheRight - FXData^[Finish]) / (FXData^[Finish+1] - FXData^[Finish]);
    Sum := Sum +
      (TheRight - FXData^[Finish]) *
      (FYData^[Finish] + YEst) / 2;
  end;

  Integral := Sum;
end;

{------------------------------------------------------------------------------
     Function: TSeries.IntegralByPoint
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the integral of the Series from iStart to iFinish
 Return Value: the real area
 Known Issues: see Integral above
 ------------------------------------------------------------------------------}
function TSeries.IntegralByPoint(Start, Finish: Integer): Single;
var
  i: Integer;
begin
  if (Start > Finish) then
  begin
{swap Start and Finish}
    i := Start;
    Start := Finish;
    Finish := i;
  end;

  Result := 0;
  for i := Start+1 to Finish do
  begin
    Result := Result +
      (FYData^[i] + FYData^[i-1]) * (FXData^[i] - FXData^[i-1]) / 2;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.IncMemSize
  Description: increases the available memory for data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data and memory management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.IncMemSize;
begin
  AllocateNoPts(mNoAllocPts + FDefSize);
  if ((mDataStatus = dsInternalString) and (FXStringData = nil)) then
  begin
    FXStringData := TStringList.Create;
    TPlot(FPlot).Axes[0].SetLabelSeries(Self);
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.InsertPoint
  Description: inserts a data point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ??? Property
 Return Value: new number of data points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.InsertPoint(X, Y: Single): Integer;
var
  i, ThePoint: Integer;
begin
  if ((mDataStatus = dsNone) or (FNoPts = 0))then
  begin
{we add the point, firing events and adjusting axes as neccessary:}
    Result := AddPoint(X, Y, TRUE, TRUE);
    exit;
  end;

{Find out where to insert this point:}
  ThePoint := 0;
  for i := 0 to FNoPts-1 do
  begin
    if (FXData^[i] > X) then
    begin
      ThePoint := i;
    end
    else if (ThePoint > 0) then
    begin
      break;
    end;
  end;

  if (ThePoint = FNoPts-1) then
  begin
{we add the point, firing events and adjusting axes as neccessary:}
    InsertPoint := AddPoint(X, Y, TRUE, TRUE);
    exit;
  end;

{Check memory available:}
  if (FNoPts >= mNoAllocPts-2) then
    IncMemSize;

  if (FXDataSeries = nil) then
  begin
    for i := FNoPts downto ThePoint+1 do
    begin
      FXData^[i] := FXData^[i-1];
    end;
    FXData^[ThePoint] := X;
  end;

  for i := FNoPts downto ThePoint+1 do
  begin
    FYData^[i] := FYData^[i-1];
  end;
  FYData^[ThePoint] := Y;

  Inc(FNoPts);

  DoDataChange;
  Result := ThePoint;
end;

function TSeries.InsertPointEx(X, Y: Single; Extra: array of Single): Integer;
var
  i, j, ThePoint: Integer;
begin
  ThePoint := InsertPoint(X, Y);
  for i := FNoPts-1 downto ThePoint+1 do
  begin
    for j := 0 to FMultiplicity-1 do
      FExtraData[j]^[i] := FExtraData[j]^[i-1];
  end;
  for j := 0 to FMultiplicity-1 do
    FExtraData[j]^[ThePoint] := Extra[j];
  DoDataChange;
  Result := ThePoint;
end;
{------------------------------------------------------------------------------
    Procedure: TSeries.Smooth
  Description: smoothes the data using a modified Savitsky-Golay method 
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Smooth(SmoothOrder: Integer);
var
  i, j, K: Integer;
  Start, Finish: Integer;
  IntSum: Integer;
  Sum, SumStart, SumFinish: Single;
  pSmoothData: pSingleArray;
  SCMatrix, pSCMatrix: pInteger; {Integer ?}
{define pSCMatrix(i, j) == pSCMatrix(i + (FNoPts+1) * j)}
  pSCSum: pIntegerArray;    {Integer ?}
  {Msg: String;}

{NOTE: Multidimensional dynamic arrays DON'T WORK !}

  procedure SetSCMatrixPointer(i, j: Integer);
  begin
    if (SmoothOrder < 2) then raise
      ERangeError.CreateFmt('SetSCMatrixPointer: SCMatrix(%d, %d) does not exist !',
        [i, j]);
    pSCMatrix := SCMatrix;
    Inc(pSCMatrix, i + (SmoothOrder+1) * j);
  end;

  {procedure DisplayMatrix;
  var
    ii, jj: Integer;
    DMsg: String;
  begin
  //display the matrix:
    DMsg := Format('Smooth Order = %d, Start = %d, Finish = %d',
                  [SmoothOrder, Start, Finish]) + CRLF;
    DMsg := DMsg + #13#10+ 'The smoothing Matrix is:';
    For ii := 0 To SmoothOrder do
    begin
      DMsg := DMsg + CRLF;
      For jj := 0 To SmoothOrder do
      begin
        SetSCMatrixPointer(ii, jj);
        DMsg := DMsg + IntToStr(pSCMatrix^) + ', '
      end;
    end;

    DMsg := DMsg + #13#10+ CRLF+ 'The smoothing Sums are:' + #13+#10;
    pSCSum := SCSum;
    For ii := 0 To SmoothOrder do
    begin
      DMsg := DMsg + IntToStr(pSCSum^) + ', ';
      Inc(pSCSum);
    end;

    ShowMessage(DMsg);
  end;}

begin
  if ((SmoothOrder < 2) or (SmoothOrder > 20)) then raise
    ERangeError.CreateFmt('TSeries.Smooth: the Smoothing Order %d must be in the range: 2...20',
      [SmoothOrder]);

  if (FNoPts <= SmoothOrder+1) then raise
    ERangeError.CreateFmt('TSeries.Smooth: the Smoothing Order (%d) must be less than the number of points (%d) !',
      [SmoothOrder, FNoPts]);

{allocate memory for arrays:}
  GetMem(pSmoothData, FNoPts * SizeOf(Single));
  GetMem(SCMatrix, (SmoothOrder+1) * (SmoothOrder+1) * SizeOf(Integer));
  GetMem(pSCSum, (SmoothOrder+1) * SizeOf(Integer));

{Zero the matrix:}
  For i := 0 To SmoothOrder do {i <=> Rows}
  begin
    For j := 0 to SmoothOrder do {j <=> Column}
    begin
      SetSCMatrixPointer(i, j);
      pSCMatrix^ := 0;
    end;
  end;

{set the first column and the diagonals to 1:}
  For i := 0 To SmoothOrder do
  begin
    SetSCMatrixPointer(i, 0);
    pSCMatrix^ := 1;
    SetSCMatrixPointer(i, i);
    pSCMatrix^ := 1;
  end;

{Calculate the Smoothing Coefficients:
 now columns 1, 2, ... SmoothOrder:}
  For i := 2 To SmoothOrder do {i <=> Rows}
  begin
    For j := 1 to i-1 do {j <=> Column}
    begin
      SetSCMatrixPointer(i - 1, j - 1);
      IntSum := pSCMatrix^;
      SetSCMatrixPointer(i - 1, j);
      IntSum := IntSum + pSCMatrix^;
      SetSCMatrixPointer(i, j);
      pSCMatrix^ := IntSum;
    end;
  end;
{   For j% = 1 To SmoothOrder%
        For i% = j% To SmoothOrder%
            Sum! = 0
            For K% = 0 To i% - 1
                Sum! = Sum! + SC(K%, j% - 1)
            Next K%
            SC(i%, j%) = Sum!
        Next i%
    Next j%}

{Calculate the sums:}
  For i := 0 To SmoothOrder do {i <=> Rows}
  begin
    pSCSum^[i] := 0;
    For j := 0 To i do {j <=> Columns}
    begin
      SetSCMatrixPointer(i, j);
      pSCSum^[i] := pSCSum^[i] + pSCMatrix^;
    end;
  end;
{    For i% = 0 To SmoothOrder%
        SCSum(i%) = 0
        For j% = 0 To i%
            SCSum(i%) = SCSum(i%) + SC(i%, j%)
        Next j%
    Next i%}

{Calculate the starting and ending points:}
  Start := SmoothOrder div 2;
  Finish := FNoPts - Start;
{    Start% = Int(SmoothOrder% / 2)
    Finish% = Runs.No_Pts - Start%}

  {DisplayMatrix;}

{these first and last points don't change:}
  pSmoothData^[0] := FYData^[0];
  pSmoothData^[FNoPts-1] := FYData^[FNoPts-1];
{    Smooth_Data(0) = Y_Data(0)
    Smooth_Data(Runs.No_Pts) = Y_Data(Runs.No_Pts)}

{Do the messy points in between:}
  For K := 1 To (SmoothOrder - 2) div 2 do
{ For i% = 2 To (SmoothOrder% - 2) Step 2}
  begin
    i := 2*K;
    SumStart := 0;
    SumFinish := 0;
    For j := 0 To i do
    begin
      SetSCMatrixPointer(i, j);
      SumStart := SumStart + FYData^[j] * pSCMatrix^;
{     SumStart& = SumStart& + CLng(Y_Data(j%)) * CLng(SC(i%, j%))}
      SumFinish := SumFinish + FYData^[FNoPts-1-j] * pSCMatrix^;
{     SumFinish& = SumFinish& + CLng(Y_Data(Runs.No_Pts - j%)) * CLng(SC(i%, j%))}
    end;
    pSmoothData^[K] := SumStart / pSCSum^[i];
{   Smooth_Data(i% / 2) = SumStart& / SCSum(i%)}
    pSmoothData^[FNoPts-1-K] := SumFinish / pSCSum^[i];
{   Smooth_Data(Runs.No_Pts - i% / 2) = SumFinish& / SCSum(i%)}
  end;

{    For i% = 2 To (SmoothOrder% - 2) Step 2
        SumStart& = 0
        SumFinish& = 0
        For j% = 0 To i%
            SumStart& = SumStart& + CLng(Y_Data(j%)) * CLng(SC(i%, j%))
            SumFinish& = SumFinish& + CLng(Y_Data(Runs.No_Pts - j%)) * CLng(SC(i%, j%))
        Next j%
        Smooth_Data(i% / 2) = SumStart& / SCSum(i%)
        Smooth_Data(Runs.No_Pts - i% / 2) = SumFinish& / SCSum(i%)
    Next i%}

{loop over the fully-smoothed points:}
  For K := Start To Finish-1 do
  begin
    Sum := 0;
    For j := 0 To SmoothOrder do
    begin
      SetSCMatrixPointer(SmoothOrder, j);
      Sum := Sum + FYData^[K+j-Start] * pSCMatrix^;
{     Sum! = Sum! + Y_Data(K% + j% - Start%) * CSng(SC(SmoothOrder%, j%))}
    end;
    pSmoothData^[K] := Sum / pSCSum^[SmoothOrder];
{   Smooth_Data(K%) = Sum! / SCSum(SmoothOrder%)}
  end;

{finally, update the Y data:}
  For i := 0 To FNoPts-1 do
    FYData^[i] := pSmoothData^[i];
{NB: this causes terminal access violations:
  System.Move(pSmoothData, FYData, FNoPts * SizeOf(Single));}


  FreeMem(pSmoothData);
  FreeMem(SCMatrix);
  FreeMem(pSCSum);

  DoDataChange;
end;

{Sub Smooth (SmoothOrder%, X_Data() As Single, Y_Data() As Single)

'   This function smooths the data using a midpoint method
'   Keywords:
'       smooth
'   Input:
'
'   Modifies:
'       nothing
'   Output:
'       none
'   Returns:
'
'   Called From:
'
'   Calls:
'

Dim i%, j%, K%
Dim Start%, Finish%
Dim SumStart&, SumFinish&
Dim Sum!
Dim Msg$
ReDim Smooth_Data(0 To Runs.Array_Size) As Single

    On Error GoTo Smooth_ErrorHandler

'   declare the matrix of coefficients for smoothing:
    ReDim SC(0 To SmoothOrder%, 0 To SmoothOrder%) As Long
    ReDim SCSum(0 To SmoothOrder%) As Long

'   set the first column to 1:
    For i% = 0 To SmoothOrder%
        SC(i%, 0) = 1
    Next i%

'   Calculate the Smoothing Coefficients:
'   now columns 1, 2, ... SmoothOrder%:
    For j% = 1 To SmoothOrder%
        For i% = j% To SmoothOrder%
            Sum! = 0
            For K% = 0 To i% - 1
                Sum! = Sum! + SC(K%, j% - 1)
            Next K%
            SC(i%, j%) = Sum!
        Next i%
    Next j%

'   Calculate the sums:
    For i% = 0 To SmoothOrder%
        SCSum(i%) = 0
        For j% = 0 To i%
            SCSum(i%) = SCSum(i%) + SC(i%, j%)
        Next j%
    Next i%

'    Msg$ = "Smoothing Matrix:"
'    For i% = 0 To SmoothOrder%
'        Msg$ = Msg$ & LF
'        For j% = 0 To SmoothOrder%
'            Msg$ = Msg$ & Str$(SC(i%, j%)) & ", "
'        Next j%
'    Next i%
'    Msg$ = Msg$ & LF & LF & "Smoothing Sums:"
'    For i% = 0 To SmoothOrder%
'        Msg$ = Msg$ & Str$(SCSum(i%)) & ", "
'    Next i%
'    MsgBox Msg$, MB_OK, "Smoothing"

'   Calculate the starting and ending points:
    Start% = Int(SmoothOrder% / 2)
    Finish% = Runs.No_Pts - Start%

'   Do the smooth; end points are not affected:
    Smooth_Data(0) = Y_Data(0)
    Smooth_Data(Runs.No_Pts) = Y_Data(Runs.No_Pts)
'   Do the messy points in between:
    For i% = 2 To (SmoothOrder% - 2) Step 2
        SumStart& = 0
        SumFinish& = 0
        For j% = 0 To i%
            SumStart& = SumStart& + CLng(Y_Data(j%)) * CLng(SC(i%, j%))
            SumFinish& = SumFinish& + CLng(Y_Data(Runs.No_Pts - j%)) * CLng(SC(i%, j%))
        Next j%
        Smooth_Data(i% / 2) = SumStart& / SCSum(i%)
        Smooth_Data(Runs.No_Pts - i% / 2) = SumFinish& / SCSum(i%)
    Next i%

'   loop over the fully-smoothed points:
    For K% = Start% To Finish%
        Sum! = 0
        For j% = 0 To SmoothOrder%
            Sum! = Sum! + Y_Data(K% + j% - Start%) * CSng(SC(SmoothOrder%, j%))
        Next j%
        Smooth_Data(K%) = Sum! / SCSum(SmoothOrder%)
    Next K%

'   finally, update the RI data:
    For i% = 0 To Runs.No_Pts
        Y_Data(i%) = Smooth_Data(i%)
    Next i%


Smooth_FINISHED:
    Refresh

Exit Sub

Smooth_ErrorHandler:   ' Error handler line label.

    Msg$ = "Panic in " & "Smooth_ErrorHandler !"
    Msg$ = Msg$ & LF & LF & "Error No. " & Str$(Err) & ": " & Error$
    Response% = Message(Msg$, MB_OK + MB_ICONEXCLAMATION, "Error !", NO, H_PANIC)

    Resume Smooth_FINISHED

End Sub
}

{------------------------------------------------------------------------------
    Procedure: TSeries.Sort
  Description: Sorts the data using the HeapSort method
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Sort;
var
  i: Integer;
  pMem: Pointer;
  pPoint: pXYPoint;
  TheList: TList;
begin
{create and initialize the list of points:}
  TheList := TList.Create;
  TheList.Capacity := FNoPts;
{allocate one big block of memory:}
  GetMem(pMem, FNoPts * SizeOf(TXYPoint));
{point at the beginning:}
  pPoint := pMem;

{loop over all points:}
  for i := 0 to FNoPts-1 do
  begin
    pPoint^.X := FXData^[i];
    pPoint^.Y := FYData^[i];
    TheList.Add(pPoint);
    Inc(pPoint);
  end;

{do the dirty deed:}
  TheList.Sort(Compare);

{point at the beginning:}
  pPoint := pMem;
{loop over all points to save results:}
  for i := 0 to FNoPts-1 do
  begin
    FXData^[i] := pPoint^.X;
    FYData^[i] := pPoint^.Y;
    Inc(pPoint);
  end;

  TheList.Free;
  FreeMem(pMem, FNoPts * SizeOf(TXYPoint));
end;

{------------------------------------------------------------------------------
     Function: Compare
  Description: comparison function for sorting
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: compares the X ordinate of point for a TList quicksort
 Return Value: -1, 0 or 1
 Known Issues:
 ------------------------------------------------------------------------------}
function Compare(Item1, Item2: Pointer): Integer;
begin
  if (pXYPoint(Item1)^.X < pXYPoint(Item2)^.X) then
  begin
    Compare := -1;
  end
  else if (pXYPoint(Item1)^.X = pXYPoint(Item2)^.X) then
  begin
    Compare := 0;
  end
  else
  begin
    Compare := 1;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GetPoint
  Description: returns the Nth (0..NoPts-1) point's X and Y values.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.GetPoint(N: Integer; var X, Y: Single);
begin
  if (N >= FNoPts) then raise
    ERangeError.CreateFmt('TSeries.GetPoint: the Point number %d is not within the valid range of 0..%d',
      [N, FNoPts]);

  X := FXData^[N];
  Y := FYData^[N];
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetXYPoint
  Description: returns the Nth (0..NoPts-1) point's X and Y values.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: XY
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetXYPoint(N: Integer): TXYPoint;
{This returns the Nth (0..NoPts-1) point's X and Y values.}
var
  XY: TXYPoint;
begin
  if (N >= FNoPts) then raise
    ERangeError.CreateFmt('TSeries.GetXYPoint: the Point number %d is not within the valid range of 0..%d',
      [N, FNoPts]);

  XY.X := FXData^[N];
  XY.Y := FYData^[N];
  GetXYPoint := XY;
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TSeries.Displace
  Description: Runs the "Displace" dialog box
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user management of Series displacement
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Displace(TheHelpFile: String);
var
  DisplacementForm: TDisplacementForm;
begin
  DisplacementForm := TDisplacementForm.Create(nil);
  DisplacementForm.TheSeries := TObject(Self);
  
  DisplacementForm.SeriesLabel.Caption := FName;
  DisplacementForm.DeltaXNEdit.AsInteger := FDeltaX;
  DisplacementForm.DeltaYNEdit.AsInteger := FDeltaY;

  DisplacementForm.HelpFile := TheHelpFile;

  if (DisplacementForm.ShowModal = mrOK) then
    ApplyDisplacementChange(DisplacementForm);

  DisplacementForm.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ApplyDisplacementChange
  Description: This applies changes from the Displace Dialog.
       Author: Mat Ballard
 Date created: 03/28/2001
Date modified: 03/28/2001 by Mat Ballard
      Purpose: User interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ApplyDisplacementChange(Sender: TObject);
begin
  with TDisplacementForm(Sender) do
  begin
    FDeltaX := DeltaXNEdit.AsInteger;
    FDeltaY := DeltaYNEdit.AsInteger;
  end;
  DoStyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.EditData
  Description: Runs the Data Editor for the selected Series
       Author: Mat Ballard
 Date created: 03/13/2001
Date modified: 03/13/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.EditData(TheHelpFile: String);
var
  i: Integer;
  DataEditor: TDataEditorForm;
begin
  if (FNoPts > 1024) then
  begin
    ShowMessageFmt('There are too many data points (%d) to edit !', [FNoPts]);
    exit;
  end;

  DataEditor := TDataEditorForm.Create(nil);
  DataEditor.HelpFile := TheHelpFile;
  DataEditor.ExternalXSeries := ((Assigned(FXDataSeries)) or (mDataStatus = dsExternal));
  DataEditor.DependentXSeries := (mDependentSeries.Count > 0);
  DataEditor.TheSeries := TObject(Self);
  //DataEditor.ExternalXSeries := Self.FExternalXSeries;

  DataEditor.SeriesnameLabel.Caption := Self.Name;
  {StatusBar1.SimpleText := Format('%d points', [FNoPts]);}
  DataEditor.ZDataNEdit.AsReal := ZData;
  DataEditor.DataStringGrid.RowCount := FNoPts + 1;
  for i := 0 to FNoPts - 1 do
  begin
    DataEditor.DataStringGrid.Cells[0, i+1] := IntToStr(i);
    DataEditor.DataStringGrid.Cells[1, i+1] := FloatToStr(FXData^[i]);
    DataEditor.DataStringGrid.Cells[2, i+1] := FloatToStr(FYData^[i]);
  end;
  if (Assigned(XStringData)) then
  begin
    for i := 0 to XStringData.Count-1 do
      DataEditor.DataStringGrid.Cells[3, i+1] := XStringData.Strings[i];
  end;

  if (DataEditor.ShowModal = mrOK) then
    ApplyDataChange(DataEditor);

  DataEditor.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ApplyDataChange
  Description: This applies changes from the DataDialog.
       Author: Mat Ballard
 Date created: 03/28/2001
Date modified: 03/28/2001 by Mat Ballard
      Purpose: User interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ApplyDataChange(Sender: TObject);
var
  i,
  TotalLength: Integer;
  StringData: TStringList;
begin
  with TDataEditorForm(Sender) do
  begin
    if (RowCountChanged) then
    begin
      if (Integer(DataStringGrid.RowCount) >= mNoAllocPts) then
        IncMemSize;
      NumericDataChanged := TRUE;
      StringDataChanged := TRUE;
      FNoPts := DataStringGrid.RowCount-1;
    end;
    if (NumericDataChanged) then
    begin
      for i := 1 to DataStringGrid.RowCount - 1 do
      begin
        try
          FXData^[i-1] := StrToFloat(DataStringGrid.Cells[1, i]);
          FYData^[i-1] := StrToFloat(DataStringGrid.Cells[2, i]);
        except
          on EConvertError do
          begin
            MakeNAN(FYData^[i-1]);
            if Assigned(FOnWarning) then
              OnWarning(Self, 'Non-numeric data entered');
          end;
        end;
      end;
    end;
    if (StringDataChanged) then
    begin
      StringData := TStringList.Create;
      StringData.Assign(DataStringGrid.Cols[3]);
      StringData.Delete(0);
      TotalLength := 0;
      for i := 0 to StringData.Count-1 do
        TotalLength := TotalLength + Length(StringData[i]);
      if (TotalLength > 0) then
{This assignment isn't what you think it is:
exercise: trace into it:}
        XStringData.Assign(StringData);
      StringData.Free;
    end;
{Find out the new X and Y Min and Max values:}    
    ResetBounds;
    GetBounds;
  end;
  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.EditPoint
  Description: Runs the "EditPoint" dialog box
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user management of Series displacement
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.EditPoint(ThePointNumber: Integer; TheHelpFile: String);
var
  PointEditorForm: TPointEditorForm;
  TheResult: TModalResult;
begin
  PointEditorForm := TPointEditorForm.Create(nil);
  PointEditorForm.TheSeries := Self;

  PointEditorForm.Init(FXData, FYData, FXStringData, TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[FYAxisIndex]);
  PointEditorForm.PointSlideBar.Max := FNoPts-1;
  PointEditorForm.PointSlideBar.Frequency := Round(FNoPts/10);
  PointEditorForm.PointSlideBar.PageSize := PointEditorForm.PointSlideBar.Frequency;

{$IFDEF BCB}
  PointEditorForm.PointUpDown.Max := FNoPts-1;
  PointEditorForm.PointNEdit.Max := FNoPts-1;
{$ELSE}
{$IFDEF MSWINDOWS}
  PointEditorForm.PointSpinEdit.MaxValue := FNoPts-1;
{$ENDIF}
{$IFDEF LINUX}
  PointEditorForm.PointSpinEdit.Max := FNoPts-1;
{$ENDIF}
{$ENDIF}
  PointEditorForm.PointSlideBar.Position := ThePointNumber;
  PointEditorForm.FillData(ThePointNumber);
  PointEditorForm.DetailsLabel.Caption := FName;

  if (Assigned(FXDataSeries)) then
  begin
    PointEditorForm.XDataNEdit.Enabled := FALSE;
    PointEditorForm.XScreenNEdit.Enabled := FALSE;
  end;

  PointEditorForm.HelpFile := TheHelpFile;

  TheResult := PointEditorForm.ShowModal;
  ApplyPointChange(PointEditorForm, TheResult);

  PointEditorForm.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ApplyPointChange
  Description: This applies changes from the PointEditor Dialog.
       Author: Mat Ballard
 Date created: 03/28/2001
Date modified: 03/28/2001 by Mat Ballard
      Purpose: User interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ApplyPointChange(Sender: TObject; TheResult: TModalResult);
var
  ThePointNumber: Integer;
  XNew, YNew: Single;
begin
  with TPointEditorForm(Sender) do
  begin
    if (TheResult <> mrCancel) then
    begin
      ThePointNumber := PointSlideBar.Position;
      if (DataGroupBox.Enabled) then
      begin
        XNew := XDataNEdit.AsReal;
        YNew := YDataNEdit.AsReal;
      end
      else
      begin {base on screen co-ords:}
        XNew := TPlot(FPlot).Axes[0].XofF(XScreenNEdit.AsInteger);
        YNew := TPlot(FPlot).Axes[FYAxisIndex].YofF(YScreenNEdit.AsInteger);
      end;

      case TheResult of
        mrOK:
          begin
            ReplacePoint(ThePointNumber, XNew, YNew);
            if ((Assigned(FXStringData)) and
                (FXStringData.Count > ThePointNumber)) then
              FXStringData.Strings[ThePointNumber] := XStringDataEdit.Text;
            CheckBounds(ThePointNumber, TRUE);
          end;
        mrYes:
          begin
            if (Assigned(FXStringData)) then
              AddStringPoint(XStringDataEdit.Text, XNew, YNew, TRUE, TRUE)
            else
              AddPoint(XNew, YNew, TRUE, TRUE);
            CheckBounds(FNoPts, TRUE);
          end;
        mrNo:
          DelPointNumber(ThePointNumber, TRUE);
      end;
    end; {if}
  end;
  DoDataChange;
end;
{$ENDIF} {GUI Displace}

{------------------------------------------------------------------------------
    Procedure: TSeries.ReplacePoint
  Description: Replaces the Nth point with new values
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ReplacePoint(N: Integer; NewX, NewY: Single);
begin
  if (mDataStatus <> dsInternal) then exit;

  if (N >= FNoPts) then raise
    ERangeError.CreateFmt('TSeries.ReplacePoint: the Point number %d is not within the valid range of 0..%d',
      [N, FNoPts]);

  if (FXDataSeries = nil) then
    FXData^[N] := NewX;
  FYData^[N] := NewY;

  DoDataChange;
end;

procedure TSeries.ReplacePointEx(N: Integer; NewX, NewY: Single; Extra: array of Single);
var
  i: Integer;
begin
  if (mDataStatus <> dsInternal) then exit;

  if (N >= FNoPts) then raise
    ERangeError.CreateFmt('TSeries.ReplacePoint: the Point number %d is not within the valid range of 0..%d',
      [N, FNoPts]);

  if (FXDataSeries = nil) then
    FXData^[N] := NewX;
  FYData^[N] := NewY;
  for i := 0 to FMultiplicity-1 do
    FExtraData[i]^[N] := Extra[i];

  DoDataChange;
end;

{Odds and sods --------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.Compress
  Description: reduces the size of the data set by local averaging
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 10/15/2000 by Mat Ballard
      Purpose: data manipulation and management
 Known Issues: Tidied up: Contract originally compressed; now it really does contract.
 ------------------------------------------------------------------------------}
procedure TSeries.Compress(CompressRatio: Integer);
var
  i, j, k: Integer;
  XSum, YSum: Single;
begin
  if ((CompressRatio < 2) or (FNoPts div CompressRatio < 10 )) then
  begin
{$IFDEF GUI}
{we used to throw an exception here, but this roots CompressAllSeries}
    ShowMessage(Format('TSeries.Compress: cannot Compress %s (%d points) by a Ratio of %d !',
      [FName, FNoPts, CompressRatio]));
{$ENDIF}
    exit;
  end;

  j := 0;
  k := 0;
  XSum := 0;
  YSum := 0;
  for i := 0 to FNoPts-1 do
  begin
    XSum := XSum + FXData^[i];
    YSum := YSum + FYData^[i];
    Inc(j);
    if (j = CompressRatio) then
    begin
      if (FXDataSeries = nil) then
        FXData^[k] := XSum / j;
      FYData^[k] := YSum / j;
      j := 0;
      XSum := 0;
      YSum := 0;
      Inc(k);
    end;
  end; {for}
  if (j > 0) then
  begin
    if (FXDataSeries = nil) then
      FXData^[k] := XSum / j;
    FYData^[k] := YSum / j;
    Inc(k);
  end;
  FNoPts := k;

  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Contract
  Description: reduces the size of the data set by throwing away the ends of the data set
       Author: Mat Ballard
 Date created: 10/15/2000
Date modified: 10/15/2000 by Mat Ballard
      Purpose: data manipulation and management
 Known Issues: Tidied up: Contract originally compressed; now it really does contract.
 ------------------------------------------------------------------------------}
procedure TSeries.Contract(TheStart, TheFinish: Integer);
var
  i: Integer;
begin
  if (TheStart > TheFinish) then
  begin
    i := TheStart;
    TheStart := TheFinish;
    TheFinish := i;
  end;

  if (TheFinish > FNoPts) then
  begin
{$IFDEF GUI}
{we used to throw an exception here, but this roots ContractAllSeries}
    ShowMessage(Format('TSeries.Contract: cannot contract %s (%d points) from %d to %d !',
      [FName, TheStart, TheFinish]));
{$ENDIF}
    exit;
  end;

  if (TheStart > 0) then
  begin
    for i := TheStart to TheFinish do
      FYData^[i-TheStart] := FYData^[i];
    if (FXDataSeries = nil) then
      for i := TheStart to TheFinish do
        FXData^[i-TheStart] := FXData^[i];
  end;

  FNoPts := TheFinish - TheStart +1;
  Self.ResetBounds;
  Self.GetBounds;

  DoDataChange;
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TSeries.CopyToClipBoard
  Description: Copies this Series to the clipboard as tab-delimited text
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: moving data in and out of the application
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.CopyToClipBoard;
var
{Q: which is more efficient: a String or a TStringList ?}
  TheData: String;
  i: Integer;
begin
  TheData := FName;
  TheData := TheData + #13#10+ TPlot(FPlot).Axes[0].Title.Caption + #9 + TPlot(FPlot).Axes[FYAxisIndex].Title.Caption;
  TheData := TheData + #13#10+ TPlot(FPlot).Axes[0].Title.Units + #9 + TPlot(FPlot).Axes[FYAxisIndex].Title.Units;
  for i := 0 to FNoPts-1 do
  begin
    TheData := TheData + #13#10+ FloatToStr(FXData^[i]) + #9 + FloatToStr(FYData^[i]);
  end;
  Clipboard.AsText := TheData;
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TSeries.CheckBounds
  Description: Checks if ThePointNo exceeds the Series Mins and Maxes
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: many: data and screen management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.CheckBounds(ThePointNo: Integer; AdjustAxis: Boolean);
begin
  if (Assigned(FXData)) then
  begin // this series contains X Data
    if (FXMin > FXData^[ThePointNo]) then
    begin
      FXMin := FXData^[ThePointNo];
      if (AdjustAxis) then
        TPlot(FPlot).Axes[0].SetMinFromSeries(FXMin);
    end;
    if (FXMax < FXData^[ThePointNo]) then
    begin
      FXMax := FXData^[ThePointNo];
      if (AdjustAxis) then
        TPlot(FPlot).Axes[0].SetMaxFromSeries(FXMax);
    end;
  end;
  if (FYMin > FYData^[ThePointNo]) then
  begin
    FYMin := FYData^[ThePointNo];
    if (AdjustAxis) then
      TPlot(FPlot).Axes[FYAxisIndex].SetMinFromSeries(FYMin);
  end;
  if (FYMax < FYData^[ThePointNo]) then
  begin
    FYMax := FYData^[ThePointNo];
    if (AdjustAxis) then
      TPlot(FPlot).Axes[FYAxisIndex].SetMaxFromSeries(FYMax);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GetBounds
  Description: Determines the Mins and Maxes of this Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the XMin, XMax, YMin and YMax  Properties
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.GetBounds;
var
  i: Integer;
  XChanged, YChanged: Boolean;
begin
  if (FNoPts > 0) then
  begin
    XChanged := FALSE;
    YChanged := FALSE;
    for i := 0 to FNoPts-1 do
    begin
      if (FXMin > FXData^[i]) then
      begin
        FXMin := FXData^[i];
        XChanged := TRUE;
      end;
      if (FXMax < FXData^[i]) then
      begin
        FXMax := FXData^[i];
        XChanged := TRUE;
      end;
      if (FYMin > FYData^[i]) then
      begin
        FYMin := FYData^[i];
        YChanged := TRUE;
      end;
      if (FYMax < FYData^[i]) then
      begin
        FYMax := FYData^[i];
        YChanged := TRUE;
      end;
    end;
    if (XChanged) then
      TPlot(FPlot).Axes[0].SetMinMaxFromSeries(FXMin, FXMax);
    if (YChanged) then
      TPlot(FPlot).Axes[FYAxisIndex].SetMinMaxFromSeries(FYMin, FYMax);
  end;    
end;

procedure TSeries.GetExtraBounds(Index: Integer; var sMin, sMax: Single);
var
  i: Integer;
begin
  sMin := 1.0e38;
  sMax := -1.0e38;
  for i := 0 to FNoPts-1 do
  begin
    if (sMin > FExtraData[Index]^[i]) then sMin := FExtraData[Index]^[i];
    if (sMax < FExtraData[Index]^[i]) then sMax := FExtraData[Index]^[i];
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ResetBounds
  Description: Resets the Mins and Maxes
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ResetBounds;
begin
  FXMin := 3.4e38;
  FXMax := -3.4e38;
  FYMin := 3.4e38;
  FYMax := -3.4e38;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetNearestPointToFX
  Description: This uses a binary search method to find the point with an X value closest to X.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetNearestPointToFX(FX: Integer): Integer;
begin
  GetNearestPointToFX :=
    GetNearestPointToX(TPlot(FPlot).Axes[0].XofF(FX));
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetNearestPointToX
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetNearestPointToX(X: Single): Integer;
{This uses a binary search method to find the point with an X value closest to X.}
var
  iEst, iLow, iHigh: Integer;
begin
{Is X outside the range of X Values ?}
  if (X >= FXMax) then
  begin
    if (FNoPts > 0) then
      Result := FNoPts-1
     else
      Result := 0;
    exit;
  end;
  if (X <= FXMin) then
  begin
    Result := 0;
    exit;
  end;

{The lowest and highest possible points are:}
  iLow := 0;
  iHigh := FNoPts - 1;
{Estimate a starting point:}
  iEst := Round(FNoPts * (X-FXMin)/(FXMax - FXMin));

  repeat
    if (X < FXData^[iEst]) then
    begin
{The point is lower:}
      iHigh := iEst;
      iEst := (iEst + iLow) div 2;
    end
    else
    begin
{The point is higher:}
      iLow := iEst;
      iEst := (iEst + iHigh) div 2;
    end;
  until ((iEst-iLow) <= 1) and ((iHigh-iEst) <= 1);
{find the X values just below and just above:}
  if ((X < FXData^[iLow]) or (X > FXData^[iHigh])) then
  begin
    raise EComponentError.CreateFmt('Failed attempt to find point closest to X' + #13#10
      + 'X Low / Estimate / High = %g/%g/%g',
        [X, FXData^[iLow], FXData^[iEst], FXData^[iHigh]]);
  end
  else if (X < FXData^[iEst]) then
  begin
{FXData^[iLow] < X < FXData^[iEst]}
    if ((X-FXData^[iLow]) < (FXData^[iEst]-X)) then
    begin
      iEst := iLow;
    end;
  end
  else
  begin
{FXData^[iEst] < X < FXData^[iHigh]}
    if ((FXData^[iEst]-X) > (FXData^[iHigh]-X)) then
    begin
      iEst := iHigh;
    end;
  end;
  Result := iEst;
end;

{------------------------------------------------------------------------------
     Function: TSeries.Interpolate
  Description: performs a linear interpolation to estimate a Y value from a given X
       Author: Mat Ballard
 Date created: 09/25/2003
Date modified: 09/25/2003 by Mat Ballard
      Purpose: data management
 Return Value: estimated Y value at X
 Known Issues: Use the Spline methods for a (slower) non-linear interpolation.
 ------------------------------------------------------------------------------}
function TSeries.Interpolate(X: Single): Single;
var
  iX, iX2: Integer;
  b, m: Single;
begin
  iX := GetNearestPointToX(X);

  if (X = FXData[iX]) then
    Result := FYData[iX]
  else
  begin
    if (FNoPts > 1) then
    begin
      if (X < FXData[iX]) then
        iX2 := iX-1
       else
        iX2 := iX+1;
      m := (FYData[iX2] - FYData[iX]) / (FXData[iX2] - FXData[iX]);
      b := FYData[iX] - m * FXData[iX];
      Result := m * X + b;
    end else begin
      Result := FYData[0];
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetNearestPieSlice
  Description: does what it says
       Author: Mat Ballard
 Date created: 01/25/2001
Date modified: 01/25/2001 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetNearestPieSlice(
  iX, iY,
  PieLeft, PieTop, PieWidth, PieHeight: Integer;
  var MinDistance: Single): Integer;
var
  i, NearestN: Integer;
  Xi, Yi, Ri: Integer;
  Angle,
  AngleSum,
  TheAngle,
  Ratio,
  Sum: Single;

begin
  Result := 0;
{adjust for displacement:}
  Dec(iX, FDeltaX);
  Dec(iY, FDeltaY);
{et al:}
  if (MinDistance = 0) then
    MinDistance := 1.0e38;
  NearestN := 0;

  if ((iX < PieLeft) or
      (iX > (PieLeft + PieWidth)) or
      (iY < PieTop)  or
      (iY > (PieTop + PieHeight))) then
    exit;

{X and Y distances from centre of ellipse:}
  Xi := iX - (PieLeft + PieWidth div 2);
  Yi := iY - (PieTop + PieHeight div 2);
  MinDistance := Sqrt(Sqr(Xi) + Sqr(Yi));
  Ratio := PieWidth / PieHeight;
  Sum := Sqr(Xi) + Sqr(Ratio)*Sqr(Yi);
  Ri := Sqr(PieWidth div 2);
  if (Round(Sum) <= Ri) then
  begin
    TheAngle := GetAngle(Xi, Yi);
    AngleSum := 0;
    for i := 0 to FNoPts-1 do
    begin
  {angles are in radians, of course:}
      Angle := TWO_PI * FYData^[i] / YSum;
      AngleSum := AngleSum + Angle;
      if (AngleSum >= TheAngle) then
      begin
        NearestN := i;
        MinDistance := 0;
        break;
      end;
    end;
  end;

  Result := NearestN;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetNearestXYPoint
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 01/25/2001 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetNearestXYPoint(
  iX, iY: Integer;
  StartPt, EndPt: Integer;
  var MinDistance: Single): Integer;
{The data may not be sorted, so we check every point:}
var
  Distance: Single;
  i, NearestN: Integer;
  Xi, Yi: Integer;
begin
  if (FNoPts > 0) then
  begin
{adjust for displacement:}
    Dec(iX, FDeltaX);
    Dec(iY, FDeltaY);
{check the incoming value:}
    if (MinDistance = 0) then
      MinDistance := 1.0e38;
    NearestN := 0;
    if (StartPt = EndPt) then
    begin
      StartPt := 0;
      EndPt := FNoPts-1;
    end;

{loop over points in each series:}
    for i := StartPt to EndPt do
    begin
      Xi := TPlot(FPlot).Axes[0].FofX(FXData^[i]);
      Yi := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]);
      Distance := Sqrt(Sqr(Int(Xi-iX)) + Sqr(Int(Yi-iY)));
      if (MinDistance > Distance) then
      begin
        MinDistance := Distance;
        NearestN := i;
      end;
    end; {loop over points}
    //MinDistance := Sqrt(MinDistance);

    Result := NearestN;
  end
  else
    Result := 0;
end;


{------------------------------------------------------------------------------
     Function: TSeries.GetNearestXYPointFast
  Description: does what it says - a quick and dirty method
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues: will not work on very spiky data
 ------------------------------------------------------------------------------}
function TSeries.GetNearestXYPointFast(
  iX, iY: Integer;
  var MinDistance: Single): Integer;
var
  X: Single;
  N, StartPt, EndPt: Integer;
begin
  X := TPlot(FPlot).Axes[0].XofF(iX);
  N := GetNearestPointToX(X);

  StartPt := N - FNoPts div 20;
  EndPt := N + FNoPts div 20;
  if (EndPt > FNoPts) then EndPt := FNoPts;

  GetNearestXYPointFast := GetNearestXYPoint(
    iX, iY,
    StartPt, EndPt,
    MinDistance);
end;

{TSeries the movie ! ----------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.Draw
  Description: Drawing routing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 10/06/2002 by Mat Ballard
      Purpose: calls the appropriate drawing method
 Known Issues: previously the major XY draing method; most of the old code is in DrawXY
 ------------------------------------------------------------------------------}
procedure TSeries.Draw(ACanvas: TCanvas; XYFastAt: Integer);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.Draw: ACanvas is nil !');
{$ENDIF}
  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Brush.Assign(FBrush);
  {TPlot(FPlot).Axes[0] := TPlot(FPlot).Axes[0];
  TPlot(FPlot).Axes[FYAxisIndex] := TPlot(FPlot).Axes[FYAxisIndex];}

  case FSeriesType of
    stNormal: DrawXY(ACanvas, FYData, XYFastAt);
    stComplex:
      begin
        DrawXY(ACanvas, FYData, XYFastAt);
        i := Ord(ACanvas.Pen.Style);
        ACanvas.Pen.Style := TPenStyle((i+1) mod Ord(High(TPenStyle)));
        ACanvas.Pen.Color := Misc.GetPalerColor(ACanvas.Pen.Color, 50);
        DrawXY(ACanvas, FExtraData[0], XYFastAt);
      end;
    stBubble:
      begin
        DrawBubble(ACanvas);
      end;
    stError:
      begin
        DrawError(ACanvas);
        DrawXY(ACanvas, FYData, XYFastAt);
      end;
    stMultiple:
      begin
        DrawMultiple(ACanvas);
        DrawXY(ACanvas, FYData, XYFastAt);
      end;
  end;

end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawXY
  Description: standard XY Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000 / 10/06/2002
Date modified: 10/06/2002 by Mat Ballard
      Purpose: draws the Series on a given canvas
 Known Issues: PERFORMANCE stuff now removed - I know TPlot is quick !
 ------------------------------------------------------------------------------}
procedure TSeries.DrawXY(ACanvas: TCanvas; TheYData: pSingleArray; XYFastAt: Integer);
var
  i: Integer;
  iX, iXOld, iY, iYMin, iYMax: Integer;
  TheYMin, TheYMax: Single;
begin
  if (FNoPts < XYFastAt) then
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[0])+ FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(TheYData^[0]) + FDeltaY;
    ACanvas.MoveTo(iX, iY);
    for i := 1 to FNoPts-1 do
    begin
      iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(TheYData^[i]) + FDeltaY;
      ACanvas.LineTo(iX, iY);
    end; {loop over points}
  end
  else
  begin
{There is a huge number of points (> 10000).
We therefore adopt a new drawing procedure:
                     TPlot                      TChart
            Fast     Slow     Memory        Time     Memory
            (ms)     (ms)      (K)          (ms)       (K)
 9990 pts:  108      ----     2902          204      2668
10001 pts:  13.9     16.2     2976
100000 pts:  23.4     67.7     3628         2226      6000
1000000 pts: 123 ms   592      10736        19271     41040

{This is the more accurate but slower algorithm:}
    i := 0;
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[0])+ FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(TheYData^[0]) + FDeltaY;
    ACanvas.MoveTo(iX, iY);
    while i < FNoPts do
    begin
      iXOld := iX;
      TheYMin := TheYData^[i];
      TheYMax := TheYMin;
      repeat
        iX := TPlot(FPlot).Axes[0].FofX(FXData^[i])+ FDeltaX;
        if (iX > iXOld) then break;
        if (TheYMin > TheYData^[i]) then
          TheYMin := TheYData^[i];
        if (TheYMax < TheYData^[i]) then
          TheYMax := TheYData^[i];
        Inc(i);
      until (i = FNoPts-1);
      iYMin := TPlot(FPlot).Axes[FYAxisIndex].FofY(TheYMin)+ FDeltaY;
      iYMax := TPlot(FPlot).Axes[FYAxisIndex].FofY(TheYMax)+ FDeltaY;
      ACanvas.LineTo(iX, iYMax);
      ACanvas.LineTo(iX, iYMin);
      Inc(i);
    end;

  end; {if FNoPts < XYFastAt}

{do symbols:}
  if ((FSymbol <> syNone) and (FSymbolSize > 0) and (FNoPts < XYFastAt)) then
  begin
    for i := 0 to FNoPts-1 do
    begin
      iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(TheYData^[i]) + FDeltaY;
      DrawSymbol(ACanvas, iX, iY);
    end;
  end;

{Highs and Lows:}
  if (FHighCount > 0) then
    if (FHighLow <> []) then
      DrawHighs(ACanvas);
{Linear or Polynomial fit with the Ai[]:}
  if (FShowFit) then
    DrawLSFit(ACanvas);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawError
  Description: Error Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000 / 10/06/2002
Date modified: 10/06/2002 by Mat Ballard
      Purpose: draws the Error as a cross on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawError(ACanvas: TCanvas);
var
  i: Integer;
  iX, iY: Integer;

  procedure DoCheck(idx, idy: Integer);
  begin
    ACanvas.MoveTo(iX - idx, iY - idy);
    ACanvas.LineTo(iX + idx, iY + idy);
  end;

begin
  for i := 0 to FNoPts-1 do
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i] - FExtraData[0]^[i])+ FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
    if (FSymbolCheck) then DoCheck(0, FSymbolSize);
    ACanvas.MoveTo(iX, iY);
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i] + FExtraData[0]^[i]) + FDeltaX;
    ACanvas.LineTo(iX, iY);
    if (FSymbolCheck) then DoCheck(0, FSymbolSize);
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i] - FExtraData[1]^[i]) + FDeltaY;
    if (FSymbolCheck) then DoCheck(FSymbolSize, 0);
    ACanvas.MoveTo(iX, iY);
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i] + FExtraData[1]^[i]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
    if (FSymbolCheck) then DoCheck(FSymbolSize, 0);
  end; {loop over points}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawMultiple
  Description: Multiple Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000 / 10/06/2002
Date modified: 10/06/2002 by Mat Ballard
      Purpose: draws the Multiple as a cross on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawMultiple(ACanvas: TCanvas);
var
  i, j, k: Integer;
  iX, iY: Integer;
  s: Single;
  Sort: array [0..MAX_MULTIPLICITY] of Single;

  procedure DoCheck;
  begin
    ACanvas.MoveTo(iX - FSymbolSize, iY);
    ACanvas.LineTo(iX + FSymbolSize, iY);
  end;

begin
  for i := 0 to FNoPts-1 do
  begin
{Sort the Y data into order:}
    Sort[0] := FYData^[i];
    for j := 1 to FMultiplicity do
     Sort[j] := FExtraData[j-1]^[i];
    for j := 0 to FMultiplicity do
    begin
      for k := j+1 to FMultiplicity do
      begin
        if (Sort[j] < Sort[k]) then
        begin
          s := Sort[j];
          Sort[j] := Sort[k];
          Sort[k] := s;
        end;
      end;
    end;

    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(Sort[FMultiplicity]) + FDeltaY;
    if (FSymbolCheck) then DoCheck;
    ACanvas.MoveTo(iX, iY);
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(Sort[0]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
    if (FSymbolCheck) then DoCheck;
    if (FMultiplicity >= 3) then
    begin // we draw a filled rectangle:
      iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) - FSymbolSize + FDeltaX;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(Sort[FMultiplicity-1]) + FDeltaY;
      ACanvas.Rectangle(iX, iY, iX + 2*FSymbolSize,
        TPlot(FPlot).Axes[FYAxisIndex].FofY(Sort[1]) + FDeltaY);
    end;
  end; {loop over points}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawBubble
  Description: standard Bubble Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000 / 10/06/2002
Date modified: 10/06/2002 by Mat Ballard
      Purpose: draws the Series on a given canvas as Bubbles
 Known Issues: 
 ------------------------------------------------------------------------------}
procedure TSeries.DrawBubble(ACanvas: TCanvas);
var
  i,
  iX,
  iY,
  YRadius: Integer;
  BubbleMin, BubbleMax: Single;
begin
  GetExtraBounds(0, BubbleMin, BubbleMax);
  BubbleMax := 100 * BubbleMax / (FBubbleSize * TPlot(FPlot).Axes[FYAxisIndex].Height);
  for i := 0 to FNoPts-1 do
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
    YRadius := Round(FExtraData[0]^[i] / BubbleMax);
    if (YRadius >= 0) then
      ACanvas.Brush.Style := bsSolid
     else
      ACanvas.Brush.Style := bsCross;
    ACanvas.Ellipse(iX - YRadius, iY - YRadius, iX + YRadius, iY + YRadius)
  end; {for}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawFourier
  Description: Drawing procedure for Fourier Transformed series
       Author: Mat Ballard
 Date created: 09/25/2002
Date modified: 09/25/2002 by Mat Ballard
      Purpose: draws the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawFourier(ACanvas: TCanvas);
var
  i: Integer;
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.DrawFourier: ACanvas is nil !');
{$ENDIF}
  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Brush.Assign(FBrush);

  {TPlot(FPlot).Axes[0] := TPlot(FPlot).Axes[0];
  TPlot(FPlot).Axes[FYAxisIndex] := TPlot(FPlot).Axes[FYAxisIndex];}

  iX := TPlot(FPlot).Axes[0].Left;
  iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[0]) + FDeltaY;
  ACanvas.MoveTo(iX, iY);
  for i := 1 to FNoPts-1 do
  begin
    iX := TPlot(FPlot).Axes[0].Left + Integer(i * Integer(TPlot(FPlot).Axes[0].Width) div (FNoPts-1)) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end; {loop over points}

{do symbols:}
  if ((FSymbol <> syNone) and (FSymbolSize > 0)) then
  begin
    for i := 0 to FNoPts-1 do
    begin
      iX := TPlot(FPlot).Axes[0].Left + Integer(i * Integer(TPlot(FPlot).Axes[0].Width) div (FNoPts-1)) + FDeltaX;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
      DrawSymbol(ACanvas, iX, iY);
    end;
  end;

  if (FHighCount > 0) then
    if (FHighLow <> []) then
      DrawHighs(ACanvas);
  if (FShowFit) then
    DrawLSFit(ACanvas);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawShades
  Description: Drawing procedure exyension
       Author: Mat Ballard
 Date created: 07/23/2001
Date modified: 07/23/2001 by Mat Ballard
      Purpose: draws the portion of a Series that exceeds the axis limits on a given canvas
 Known Issues: 
 ------------------------------------------------------------------------------}
procedure TSeries.DrawShades(ACanvas: TCanvas; XYFastAt: Integer);
var
  i: Integer;
  iX, iXOld, iXCalc, iY, iYOld: Integer;
  LowerLimit, UpperLimit: Integer;
  Slope, Intercept: Single;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.DrawShades: ACanvas is nil !');
{$ENDIF}
  if ((not FVisible) or
      (FNoPts = 0)) then exit;
      
  if (FNoPts >= XYFastAt) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Brush.Assign(FBrush);

  if (FShadeLimits) then
  begin
    ACanvas.Pen.Style := psClear;
    LowerLimit := TPlot(FPlot).Axes[FYAxisIndex].FofY(TPlot(FPlot).Axes[FYAxisIndex].LimitLower);
    UpperLimit := TPlot(FPlot).Axes[FYAxisIndex].FofY(TPlot(FPlot).Axes[FYAxisIndex].LimitUpper);
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[0])+ FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[0]) + FDeltaY;
    for i := 1 to FNoPts-1 do
    begin
      iXOld := iX;
      iYOld := iY;
      iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
      if (iY > LowerLimit) then
      begin
        if (iYOld >= LowerLimit) then
          ACanvas.Polygon([
            Point(iXOld, iYOld),
            Point(iXOld, LowerLimit),
            Point(iX, LowerLimit),
            Point(iX, iY)])
        else
        begin
          Slope := (iY-iYOld) / (iX-iXOld);
          Intercept := iY - Slope * iX;
          iXCalc := Round((LowerLimit - Intercept) / Slope);
          ACanvas.Polygon([
            Point(iXCalc, LowerLimit),
            Point(iX, LowerLimit),
            Point(iX, iY)])
        end;
      end
      else if (iY < UpperLimit) then
      begin
        if (iYOld <= UpperLimit) then
          ACanvas.Polygon([
            Point(iXOld, iYOld),
            Point(iXOld, UpperLimit),
            Point(iX, UpperLimit),
            Point(iX, iY)])
        else
        begin
          Slope := (iY-iYOld) / (iX-iXOld);
          Intercept := iY - Slope * iX;
          iXCalc := Round((UpperLimit - Intercept) / Slope);
          ACanvas.Polygon([
            Point(iXCalc, UpperLimit),
            Point(iX, UpperLimit),
            Point(iX, iY)])
        end
      end  
      else if (iYOld > LowerLimit) then
      begin
        Slope := (iY-iYOld) / (iX-iXOld);
        Intercept := iY - Slope * iX;
        iXCalc := Round((LowerLimit - Intercept) / Slope);
        ACanvas.Polygon([
          Point(iXCalc, LowerLimit),
          Point(iXOld, LowerLimit),
          Point(iXOld, iYOld)])
      end
      else if (iYOld < UpperLimit) then
      begin
          Slope := (iY-iYOld) / (iX-iXOld);
          Intercept := iY - Slope * iX;
          iXCalc := Round((UpperLimit - Intercept) / Slope);
          ACanvas.Polygon([
            Point(iXCalc, UpperLimit),
            Point(iXOld, UpperLimit),
            Point(iXOld, iYOld)])
      end; {iY or iYOld in Limits}
    end; {loop over points}
    ACanvas.Pen.Style := FPen.Style;
  end; {Shaded}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawPie
  Description: standard Pie Drawing procedure
       Author: Mat Ballard
 Date created: 12/21/2000
Date modified: 12/21/2000 by Mat Ballard
      Purpose: draws the Series on a given canvas as a Pie
 Known Issues:
 ------------------------------------------------------------------------------}

{ Equation of an ellipse, origin at (h, k), with x-radius a and y-radius b is:

    (x - h)^2     (y - k)^2
    ----------  +  ----------   =   1
       a^2            b^2

  The polar version of this equation is:

  r  =  1 / Sqrt(Cos^2/a^2 + Sin^2/b^2)

  where:
  a^2  = b^2 + c^2         c is the focus
  x  =  r Cos 
  y  =  r Sin 
}

procedure TSeries.DrawPie(
  ACanvas: TCanvas;
  PieLeft,
  PieTop,
  PieWidth,
  PieHeight: Integer);
var
  a, b, d,
  i, j, Index, TextIndex, WallIndex,
  iAngleSum, iOldAngleSum,
  NoTopPts,
  StartSolid, EndSolid,
  StringWidth: Integer;
  TextAngle,
  TheSin, TheCos: Extended;
  Angle, AngleSum, OldAngleSum,
  PolarAngle,
  Radius, Ratio: Single;
  IsWall, DoAmount: Boolean;
  Points: TPoints;
  pPoints: pTPoints;
  AngleSumPos, TextPos, DeltaPos: TPoint;
  TheText: String;

  {Note: this function only works for values of Angle between 0 and 360.}
  function PolarRadians(AnAngle: Extended): Extended;
  var
    TheResult: Extended;
  begin
    TheResult := 90.0 - AnAngle;
    if (TheResult < 0) then
      TheResult := TheResult + 360.0;
    PolarRadians := TWO_PI * TheResult / 360;
  end;

begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.Draw: ACanvas is nil !');
{$ENDIF}
  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Brush.Assign(FBrush);
  ACanvas.Font.Assign(TPlot(FPlot).Axes[0].Labels.Font);

  if (Assigned(FXDataSeries)) then
    Self.FXStringData := Self.FXDataSeries.XStringData;

{Get the total; note Abs() - negative sectors are bad news:}
  YSum := 0;
  for i := 0 to FNoPts-1 do
    YSum := YSum + Abs(FYData^[i]);

{Points[0] is the centre of the ellipse:}
  Points[0].x := PieLeft + PieWidth div 2 + FDeltaX;
  Points[0].y := PieTop + PieHeight div 2 + FDeltaY;
{a is the horizontal major axis length}
  a := PieWidth div 2;
{b is the vertical minor axis length}
  b := PieHeight div 2;
{c is the distance of the focus from the centre:}
  d := a - b;
  if (d > PieHeight div 5) then
    d := PieHeight div 5;
  IsWall := FALSE;

{This is the angle, in degrees, from 12 o'clock, clockwise:}
  AngleSum := 0;
  OldAngleSum := 0;
  iOldAngleSum := 0;
  AngleSumPos.x := Points[0].x;
  AngleSumPos.y := Points[0].y - b;

  for i := 0 to FNoPts-1 do
  begin
    Index := 1;
    StartSolid := -1;
    EndSolid := -1;
    if (iOldAngleSum < OldAngleSum) then
    begin
      Points[Index].x := AngleSumPos.x;
      Points[Index].y := AngleSumPos.y;
{only angles between 90 and 270 - the lower side of the ellipse -
 can have a "wall":}
      if ((90 <= OldAngleSum) and (OldAngleSum <= 270)) then
        StartSolid := Index;
      Inc(Index);
    end;
    ACanvas.Brush.Color := MyColorValues[1 + i mod 15];
    Angle := 360.0 * Abs(FYData^[i]) / YSum;
    AngleSum := AngleSum + Angle;
    iAngleSum := Trunc(AngleSum);
    for j := iOldAngleSum to iAngleSum do
    begin
{we look for start of the "wall":}
      if ((StartSolid < 0) and (90 <= j) and (j <= 270)) then
        StartSolid := Index;
{gotta find the start before the end:}
      if ((StartSolid > 0) and (j <= 270)) then
        EndSolid := Index;
      PolarAngle := PolarRadians(j);
      SinCos(PolarAngle, TheSin, TheCos);
      Radius :=  1.0 / Sqrt(Sqr(TheCos/a) + Sqr(TheSin/b));
      AngleSumPos.x := Points[0].x + Round(Radius * TheCos);
      AngleSumPos.y := Points[0].y - Round(Radius * TheSin);
      Points[Index].x := AngleSumPos.x;
      Points[Index].y := AngleSumPos.y;
      Inc(Index);
    end;
    if (iAngleSum < AngleSum) then
    begin
      if ((StartSolid > 0) and (AngleSum <= 270)) then
        EndSolid := Index;
      PolarAngle := PolarRadians(AngleSum);
      SinCos(PolarAngle, TheSin, TheCos);
      Radius :=  1.0 / Sqrt(Sqr(TheCos/a) + Sqr(TheSin/b));
      AngleSumPos.x := Points[0].x + Round(Radius * TheCos);
      AngleSumPos.y := Points[0].y - Round(Radius * TheSin);
      Points[Index].x := AngleSumPos.x;
      Points[Index].y := AngleSumPos.y;
      Inc(Index);
    end;
{Draw the pie slice:}
    ACanvas.Polygon(Slice(Points, Index));
    TextAngle := OldAngleSum + Angle/2;

{Should we put the amounts in ?}
    j := Round(Sqrt(
      Sqr(Points[1].x - Points[Index-1].x) +
      Sqr(Points[1].y - Points[Index-1].y)));
    TheText := FloatToStrF(100 * FYData^[i] / YSum, ffFixed, 0, 0) + '%';
    StringWidth := ACanvas.TextWidth(TheText);
    DoAmount := (j > StringWidth);
{NB: we do this before the wall section because the latter changes the Points array,
 however, we do the textout after the wall because of brush and color issues.}

{Draw the bottom wall:}
    if ((d > 0) and (StartSolid > 0) and (EndSolid > 0)) then
    begin
      IsWall := TRUE;
      ACanvas.Brush.Color := Misc.GetDarkerColor(MyColorValues[1 + i mod 15], 50);
      pPoints := @Points[StartSolid];
      NoTopPts := EndSolid - StartSolid + 1;
      WallIndex := NoTopPts;
      for j := NoTopPts-1 downto 0 do
      begin
        pPoints^[WallIndex].x := pPoints^[j].x;
        pPoints^[WallIndex].y := pPoints^[j].y + d;
        Inc(WallIndex);
      end;
{Draw the pie wall:}
      ACanvas.Polygon(Slice(pPoints^, 2 * NoTopPts));
    end;

{Set brush up for text mode:}
    ACanvas.Brush.Style := bsClear;

{Should we put the amounts in ?
 See above}
    if (DoAmount) then
    begin
      ACanvas.Font.Color := GetInverseColor(MyColorValues[1 + i mod 15]);
      TextPos := Points[Index div 2];
      DeltaPos.x := Points[0].x - TextPos.x;
      DeltaPos.y := Points[0].y - TextPos.y;
      j := Round(Sqrt(Sqr(DeltaPos.x) + Sqr(DeltaPos.y)));
      Ratio := StringWidth / j;
      DeltaPos.x := Round(Ratio * DeltaPos.x);
      DeltaPos.y := Round(Ratio * DeltaPos.y);
      TextPos.x := TextPos.x + DeltaPos.x - StringWidth div 2;
      TextPos.y := TextPos.y + DeltaPos.y - ACanvas.TextHeight(TheText) div 2;
      ACanvas.TextOut(TextPos.x, TextPos.y, TheText);
    end;

{Is the X Data is in the form of a string ?}
    if (Assigned(FXStringData)) then
    begin
      if (i < FXStringData.Count) then
      begin
        if (Angle > 6.0) then {6 degrees}
        begin
{There is a large enough amount to denote:}
          ACanvas.Font.Color := MyColorValues[1 + i mod 15];
          TextPos := Points[Index div 2];
{put the text outside the circle:}
          if (TextAngle > 180) then
            Dec(TextPos.x, ACanvas.TextWidth(FXStringData.Strings[i]));
          if ((TextAngle < 90) or (TextAngle > 270)) then
            Dec(TextPos.y, ACanvas.TextHeight('Ap'))
          else if (IsWall) then
            Inc(TextPos.y, d);
          ACanvas.TextOut(TextPos.x, TextPos.y, FXStringData.Strings[i]);
{restore brush:}
          ACanvas.Brush.Style := FBrush.Style;
        end; {Angle > 0.1}
      end; {there is a string}
    end; {stringdata}

    iOldAngleSum := iAngleSum;
    OldAngleSum := AngleSum;
    //Application.ProcessMessages;
  end; {for i}
{restore the string pointer:}
  if (Assigned(FXDataSeries)) then
    Self.FXStringData := nil;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawPolar
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 01/25/2001
Date modified: 01/25/2001 by Mat Ballard
      Purpose: draws the Series on a given canvas as a Polar graph
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawPolar(ACanvas: TCanvas; PolarRange: Single);
var
  i: Integer;
  iX, iY: Integer;
  Angle,
  X,
  Y: Single;
  SinTheta, CosTheta: Extended;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.Draw: ACanvas is nil !');
{$ENDIF}
  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Brush.Assign(FBrush);
  if (ACanvas.Pen.Width > 0) then
  begin
    Angle := TWO_PI * FXData^[0] / PolarRange;
    SinCos(Angle, SinTheta, CosTheta);
    X := SinTheta * FYData^[0];
    Y := CosTheta * FYData^[0];
    iX := TPlot(FPlot).Axes[0].FofX(X);
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(Y);
    ACanvas.MoveTo(iX, iY);

    for i := 1 to FNoPts-1 do
    begin
      Angle := TWO_PI * FXData^[i] / PolarRange;
      SinCos(Angle, SinTheta, CosTheta);
      X := SinTheta * FYData^[i];
      Y := CosTheta * FYData^[i];
      iX := TPlot(FPlot).Axes[0].FofX(X);
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(Y);
      ACanvas.LineTo(iX, iY);
      if ((FSymbol <> syNone) and (FSymbolSize > 0)) then
        DrawSymbol(ACanvas, iX, iY);
    end; {loop over points}
  end;

  {if (FHighCount > 0) then
    if (FHighLow <> []) then
      DrawHighs(ACanvas);}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Trace
  Description: Draws the series in an erasable mode
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: rapidly changing screen display
 Known Issues:
 ------------------------------------------------------------------------------}
{TODO 3 -o Mat B -c Drawing Graph : Completely revise Trace mode}
procedure TSeries.Trace(ACanvas: TCanvas);
var
  i: Integer;
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.Trace: ACanvas is nil !');
{$ENDIF}

  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Pen.Mode := pmNotXOR;
  iX := TPlot(FPlot).Axes[0].FofX(FXData^[0])+ FDeltaX;
  iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[0]) + FDeltaY;
  ACanvas.MoveTo(iX, iY);
  for i := 1 to FNoPts-1 do
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i]) + FDeltaX;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end; {loop over points}

  if ((FSymbol <> syNone) and (FSymbolSize > 0)) then
  begin
    ACanvas.Brush.Assign(FBrush);
    for i := 0 to FNoPts-1 do
    begin
      iX := TPlot(FPlot).Axes[0].FofX(FXData^[i])+ FDeltaX;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
      DrawSymbol(ACanvas, iX, iY);
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawHighs
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Highs of the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawHighs(ACanvas: TCanvas);
var
  i,
  iX,
  iY: Integer;
{$IFDEF GUI}
  OldFontHandle, NewFontHandle: hFont;
{$ENDIF}

  procedure DrawHiLo(iHiLo: Integer);
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[iHiLo]);
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[iHiLo]);
    ACanvas.MoveTo(iX, iY-2);
    ACanvas.LineTo(iX, iY + ACanvas.Font.Height);
{$IFDEF GUI}
  {$IFDEF MSWINDOWS}
    ACanvas.TextOut(
      iX + ACanvas.Font.Height div 2,
      iY + ACanvas.Font.Height,
      TPlot(FPlot).Axes[0].LabelToStrF(FXData^[iHiLo]));
  {$ENDIF}
  {$IFDEF LINUX}
    ACanvas.TextOut(
      iX + ACanvas.Font.Height div 2,
      iY + ACanvas.Font.Height + Abs(ACanvas.Font.Height),
      TPlot(FPlot).Axes[0].LabelToStrF(FXData^[iHiLo]));
  {$ENDIF}
{$ELSE} // use gd:
    ACanvas.TextOutAngle(90,
      iX + ACanvas.Font.Height div 2,
      iY + ACanvas.Font.Height + Abs(ACanvas.Font.Height),
      TPlot(FPlot).Axes[0].LabelToStrF(FXData^[iHiLo]));
{$ENDIF}
  end;

begin
  ACanvas.Font.Color := ACanvas.Pen.Color;
{Note: the Brush is currently set to the color of the Series.}

{get a persistent, rotated font:}
{$IFDEF GUI}
  iX := 0;
  iY := 0;
  NewFontHandle := TextOutAnglePersist(ACanvas, OldFontHandle, 90, iX, iY, '');
{$ENDIF}

{Loop over all Highs:}
  if (hlHigh in FHighLow) then
  begin
    for i := 0 to FHighCount-1 do
    begin
      DrawHiLo(FHighs^[i]);
    end;
  end;

{Loop over all Lows:}
  if (hlLow in FHighLow) then
  begin
    for i := 0 to FLowCount-1 do
    begin
      DrawHiLo(FLows^[i]);
    end;
  end;
{$IFDEF GUI}
  TextOutAnglePersistCleanUp(ACanvas, NewFontHandle, OldFontHandle);
{$ENDIF}

{Delphi 5 bug: if we do not change the canvas font color, then the font cache
 manager (?) does not change the font. Hence, on the next drawing cycle, the
 Instructions are drawn with the current ROTATED font, despite the
 TextOutAnglePersistCleanUp call above.}
  ACanvas.Font.Color := clBlack;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawHistory
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Series on a given canvas IN HISTORY MODE
 Known Issues:
 ------------------------------------------------------------------------------}
{TODO 3 -o Mat B -c Drawing Graph : Completely revise History mode}
procedure TSeries.DrawHistory(ACanvas: TCanvas; HistoryX: Single);
var
  i: Integer;
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.DrawHistory: ACanvas is nil !');
{$ENDIF}

  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
{we set the pen mode so that a second call to DrawHistory
erases the curve on screen:}
  ACanvas.Pen.Mode := pmNotXOR;
  iX := TPlot(FPlot).Axes[0].FofX(0) + FDeltaX;
  iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[FNoPts-1]) + FDeltaY;
  ACanvas.MoveTo(iX, iY);
  for i := FNoPts-2 downto 0 do
  begin
    iX := TPlot(FPlot).Axes[0].FofX(FXData^[i] - FXData^[FNoPts-1]) + FDeltaX;
{we leave this loop if this is the last point:}
    if (iX < TPlot(FPlot).Axes[0].Left) then break;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end; {loop over points}
  if ((FSymbol <> syNone) and (FSymbolSize > 0)) then
  begin
    ACanvas.Brush.Assign(FBrush);
    for i := FNoPts-2 downto 0 do
    begin
      iX := TPlot(FPlot).Axes[0].FofX(FXData^[i] - FXData^[FNoPts-1])+ FDeltaX;
{we leave this loop if this is the last point:}
      if (iX < TPlot(FPlot).Axes[0].Left) then break;
      iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]) + FDeltaY;
      DrawSymbol(ACanvas, iX, iY);
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawLSFit
  Description: Draws the selected Symbol at each point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 03/01/2001 by Mat Ballard
      Purpose: draws the Symbols of the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawLSFit(ACanvas: TCanvas);
var
  iX, iY: Integer;
  AStep: Integer;
  x1, y1: Single;
  TheBorder: TBorder;

  function Poly: Single;
  var
    i: Integer;
    x: Single;
  begin
    x := x1;
    Result := Ai[0];
    for i := 2 to nTerms do
    begin
      Result := Result + Ai[i-1] * x;
      x := x1 * x;
    end;
  end;

begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.DrawHistory: ACanvas is nil !');
{$ENDIF}

{  TPlot(FPlot).Axes[0] := TPlot(FPlot).Axes[0];
  TPlot(FPlot).Axes[FYAxisIndex] := TPlot(FPlot).Axes[FYAxisIndex];}

{where does the curve start ?}
  iX := TPlot(FPlot).Axes[0].Left;

  if (nTerms = 2) then
  begin // straight line fit:
    x1 := TPlot(FPlot).Axes[0].Min;
    y1 := Poly;
    iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(y1);
    if ((y1 < TPlot(FPlot).Axes[FYAxisIndex].Min) or (TPlot(FPlot).Axes[FYAxisIndex].Max < y1)) then
    begin // find the line start:
      y1 := TPlot(FPlot).Axes[FYAxisIndex].Min;
      iY := TPlot(FPlot).Axes[FYAxisIndex].Bottom;
      x1 := (y1 - Ai[0]) / Ai[1];
      iX := TPlot(FPlot).Axes[0].FofX(x1);
    end;
    ACanvas.MoveTo(iX, iY);
{where does the line end ?}
    iX := TPlot(FPlot).Axes[0].Right;
    x1 := TPlot(FPlot).Axes[0].Max;
    y1 := Poly;
    if ((y1 < TPlot(FPlot).Axes[FYAxisIndex].Min) or (TPlot(FPlot).Axes[FYAxisIndex].Max < y1)) then
    begin // find the line end:
      y1 := TPlot(FPlot).Axes[FYAxisIndex].Max;
      iY := TPlot(FPlot).Axes[FYAxisIndex].Top;
      x1 := (y1 - Ai[0]) / Ai[1];
      iX := TPlot(FPlot).Axes[0].FofX(x1);
    end;
    ACanvas.Pen.Style := psDot;
    ACanvas.LineTo(iX, iY);
  end
  else if (nTerms > 2) then
  begin // polynomial
    TheBorder := TPlot(TSeriesList(Collection).Plot).Border;
    AStep := TheBorder.Width div 100;
    if (AStep < 3) then AStep := 3;

    while (iX < TPlot(FPlot).Axes[0].Right) do
    begin
      x1 := TPlot(FPlot).Axes[0].XofF(iX);
      y1 := Poly;
      if ((TPlot(FPlot).Axes[FYAxisIndex].Min <= y1) and (y1 <= TPlot(FPlot).Axes[FYAxisIndex].Max)) then
      begin
        iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(y1);
        ACanvas.MoveTo(iX, iY);
        Inc(iX, AStep);
        x1 := TPlot(FPlot).Axes[0].XofF(iX);
        y1 := Poly;
        if ((TPlot(FPlot).Axes[FYAxisIndex].Min <= y1) and (y1 <= TPlot(FPlot).Axes[FYAxisIndex].Max)) then
        begin
          iY := TPlot(FPlot).Axes[FYAxisIndex].FofY(y1);
          ACanvas.LineTo(iX, iY);
        end;
      end
      else
      begin
        Inc(iX, AStep);
      end;
      Inc(iX, AStep);
    end;
  end;
end;
{------------------------------------------------------------------------------
    Procedure: TSeries.DrawSymbol
  Description: Draws the selected Symbol at each point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 03/01/2001 by Mat Ballard
      Purpose: draws the Symbols of the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawSymbol(ACanvas: TCanvas; iX, iY: Integer);
begin
  case FSymbol of
    syDash:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
      end;
    syVertDash:
      begin
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
      end;
    syLeftDash:
      begin
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX, iY);
        ACanvas.LineTo(iX - FSymbolSize, iY);
      end;
    syRightDash:
      begin
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
      end;
    syPlus:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
      end;
    syCross:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX + FSymbolSize+1, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX + FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX - FSymbolSize-1, iY + FSymbolSize+1);
      end;
    syStar:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX - FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX + FSymbolSize+1, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX + FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX - FSymbolSize-1, iY + FSymbolSize+1);
      end;
    sySquare:
      begin
        ACanvas.Rectangle(iX - FSymbolSize, iY - FSymbolSize,
          iX + FSymbolSize+1, iY + FSymbolSize+1)
      end;
    syCircle:
      begin
        ACanvas.Ellipse(iX - FSymbolSize, iY - FSymbolSize,
          iX + FSymbolSize+1, iY + FSymbolSize+1)
      end;
    syUpTriangle:
      begin
        ACanvas.Polygon([
          Point(iX - FSymbolSize, iY + FSymbolSize+1),
          Point(iX, iY - FSymbolSize),
          Point(iX + FSymbolSize, iY + FSymbolSize+1)]);
      end;
    syDownTriangle:
      begin
        ACanvas.Polygon([
          Point(iX - FSymbolSize, iY - FSymbolSize),
          Point(iX, iY + FSymbolSize+1),
          Point(iX + FSymbolSize, iY - FSymbolSize)]);
      end;
  end;
  ACanvas.MoveTo(iX, iY);
end;

{Moving TSeries on the screen -------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.GenerateColumnOutline
  Description: calculates an Outline of the Series
       Author: Mat Ballard
 Date created: 02/26/2001
Date modified: 02/26/2001 by Mat Ballard
      Purpose: Screen display - highlighting a Series
 Known Issues: This records the position of a single "point"
 ------------------------------------------------------------------------------}
procedure TSeries.GenerateColumnOutline(X1, Y1, X2, Y2: Integer);
begin
  mOutline[0].x := X1;
  mOutline[0].y := Y1;
  mOutline[1].x := X2;
  mOutline[1].y := Y2;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GeneratePieOutline
  Description: calculates an Outline of the Series
       Author: Mat Ballard
 Date created: 02/26/2001
Date modified: 02/26/2001 by Mat Ballard
      Purpose: Screen display - highlighting a Series
 Known Issues: This records the position of the entire ellipse
 ------------------------------------------------------------------------------}
procedure TSeries.GeneratePieOutline(
  PieLeft,
  PieTop,
  PieWidth,
  PieHeight: Integer;
  TheNearestPoint: Integer);
var
  i: Integer;
  Radius: Single;
  Angle,
  AngleSum,
  TheSin, TheCos: Extended;
  Centre: TPoint;
begin
  mOutline[0].x := PieLeft;
  mOutline[0].y := PieTop;
  mOutline[1].x := PieLeft + PieWidth;
  mOutline[1].y := PieTop + PieHeight;

  Centre.x := PieLeft + PieWidth div 2;
  Centre.y := PieTop + PieHeight div 2;
  Radius := PieWidth / 2.0;

  mOutline[2].x := Centre.x;
  mOutline[2].y := PieTop;
  AngleSum := 0;
{work our way around the circle:}
  for i := 0 to TheNearestPoint do
  begin
    mOutline[3].x := mOutline[2].x;
    mOutline[3].y := mOutline[2].y;
{angles are in radians, of course:}
    Angle := TWO_PI * FYData^[i] / YSum;
    AngleSum := AngleSum + Angle;
    SinCos(AngleSum, TheSin, TheCos);
    mOutline[2].x := Centre.x + Round(Radius * TheSin);
    mOutline[2].y := Centre.y - Round(Radius * TheCos);
    {ACanvas.Pie(
      PieLeft + FDeltaX, PieTop + FDeltaY,
      PieRight + FDeltaX, PieBottom + FDeltaY,
      mOutline[2].x + FDeltaX, mOutline[2].y + FDeltaY,
      mOutline[3].x + FDeltaX, mOutline[3].y + FDeltaY);}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GenerateXYOutline
  Description: calculates an Outline of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Screen display - highlighting a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.GenerateXYOutline;
var
  i: Integer;
begin
  if (FNoPts > OUTLINE_DENSITY) then
  begin
{initialize:}
    mNoOutlinePts := OUTLINE_DENSITY+1; { = 21}
    //StepSize := FNoPts div OUTLINE_DENSITY;
{loop over data points:}
    for i := 0 to mNoOutlinePts-2 do    {0..19}
    begin
      mOutline[i].x := TPlot(FPlot).Axes[0].FofX(FXData^[i * FNoPts div OUTLINE_DENSITY]);
      mOutline[i].y := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i * FNoPts div OUTLINE_DENSITY]);
    end;
{do the end point:}
    mOutline[OUTLINE_DENSITY].x := TPlot(FPlot).Axes[0].FofX(FXData^[FNoPts-1]);
    mOutline[OUTLINE_DENSITY].y := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[FNoPts-1]);
  end
  else
  begin
{not many points:}
    mNoOutlinePts := FNoPts;
    for i := 0 to mNoOutlinePts-1 do
    begin
      mOutline[i].x := TPlot(FPlot).Axes[0].FofX(FXData^[i]);
      mOutline[i].y := TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[i]);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Outline
  Description: draws an Outline of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Screen display - highlighting a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Outline(
  ACanvas: TCanvas;
  ThePlotType: TPlotType;
  TheOutlineWidth: Integer);
var
  i: Integer;
  {pOutlinePt: pPoint;}
begin
  ACanvas.Pen.Color := clLime;
  ACanvas.Pen.Width := TheOutlineWidth;
  ACanvas.Pen.Mode := pmNotXOR;
  {ACanvas.Pen.Style := psDash;}

  case ThePlotType of
    ptXY: //, ptError, ptMultiple, ptBubble:
      begin
        if (mNoOutlinePts = 0) then exit;

        ACanvas.MoveTo(mOutline[0].x + FDeltaX, mOutline[0].y + FDeltaY);
        for i := 0 to mNoOutlinePts-1 do
        begin
          ACanvas.LineTo(mOutline[i].x + FDeltaX, mOutline[i].y + FDeltaY);
        end;
      end;
    ptColumn, ptStack, ptNormStack:
      ACanvas.Rectangle(
        mOutline[0].x,
        mOutline[0].y,
        mOutline[1].x,
        mOutline[1].y);
    ptPie:
      begin
        ACanvas.Ellipse(
          mOutline[0].x + FDeltaX,
          mOutline[0].y + FDeltaY,
          mOutline[1].x + FDeltaX,
          mOutline[1].y + FDeltaY);
        ACanvas.Pen.Width := TheOutlineWidth div 2;  
        ACanvas.Pie(
          mOutline[0].x + FDeltaX, mOutline[0].y + FDeltaY,
          mOutline[1].x + FDeltaX, mOutline[1].y + FDeltaY,
          mOutline[2].x + FDeltaX, mOutline[2].y + FDeltaY,
          mOutline[3].x + FDeltaX, mOutline[3].y + FDeltaY);
      end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MoveBy
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: moves the clicked object Outline by (DX, DY) from its current point.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MoveBy(ACanvas: TCanvas; ThePlotType: TPlotType; DX, DY, TheOutlineWidth: Integer);
begin
{erase the old Outline:}
  Outline(ACanvas, ThePlotType, TheOutlineWidth);
{save the new displacements:}
  Inc(FDeltaX, DX);
  Inc(FDeltaY, DY);

{create the new Outline:}
  Outline(ACanvas, ThePlotType, TheOutlineWidth);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MoveTo
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: moves the clicked object Outline TO (X, Y).
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MoveTo(
  ACanvas: TCanvas;
  ThePlotType: TPlotType;
  TheOutlineWidth,
  X, Y: Integer);                  {by how much}
begin
{erase the old Outline:}
  Outline(ACanvas, ThePlotType, TheOutlineWidth);

{save the new displacements:}
  FDeltaX := X - TPlot(FPlot).Axes[0].FofX(FXData^[0]);
  FDeltaY := Y - TPlot(FPlot).Axes[FYAxisIndex].FofY(FYData^[0]);

{create the new Outline:}
  Outline(ACanvas, ThePlotType, TheOutlineWidth);
end;

{$IFDEF POLYNOMIAL_FIT}
{------------------------------------------------------------------------------
    Procedure: TSeries.PolyBestFit
  Description: Does what it says
       Author: Mat Ballard
 Date created: 08/05/2001
Date modified: 08/05/2000 by Mat Ballard
      Purpose: calculates the Polynomial of best fit from Left1 to Right1, plus Left2 to Right2.
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.PolyBestFit(Left1, Right1, Left2, Right2: Integer): String;
var
  i, j, Start1, Finish1, Start2, Finish2: Integer;
  nPoints: Integer;
  Degree: String;
  DoDual: Boolean;
  XArray, YArray: pSingleArray;

  procedure ProcessPoint;
  begin
    XArray^[i] := FXData^[j];
    YArray^[i] := FYData^[j];
    Inc(i);
  end;

begin
  Result := '';
  Degree := '1';

{$IFNDEF BCB3}
  if (InputQuery('Polynomial Fit',
    'Please enter the degree of the polynomial to fit (1..9)',
    Degree)) then
{$ENDIF}
  begin
    nTerms := StrToInt(Degree) + 1;
    if (nTerms = 2) then // just a straight line anyway
      Result := LineBestFit(Left1, Right1, Left2, Right2)
    else if (nTerms < 2) or (nTerms > 10) then
    begin
      EMathError.CreateFmt('The degree of the polynomial must be in the range 1..9 - not %d !', [nTerms]);
    end
    else
    begin
{Determine the starting and ending points:}
      Start1 := GetNearestPointToFX(Left1);
      Finish1 := GetNearestPointToFX(Right1);
      DoDual := FALSE;
      if (Left2 < Right2) then
      begin
        DoDual := TRUE;
        Start2 := GetNearestPointToFX(Left2);
        Finish2 := GetNearestPointToFX(Right2);
      end
      else
      begin
        Start2 := 0;
        Finish2 := -1;
      end;
      nPoints := Finish1 - Start1 + 1;
      if (DoDual) then
        nPoints := nPoints + Finish2 - Start2 + 1;
{OK, now we run the fit:}
      try
        XArray := AllocMem(nPoints * SizeOf(Single));
        YArray := AllocMem(nPoints * SizeOf(Single));
        i := 0;
        for j := Start1 to Finish1 do
          ProcessPoint;
        for j := Start2 to Finish2 do
          ProcessPoint;
{Hold tight ! we do the curve fit:}
        CurveFit.PolyFit(XArray^, YArray^, Ai, R_Square, nPoints, nTerms);
        Result := GetFitAsString;
        FShowFit := TRUE;
      finally
        FreeMem(XArray);
        FreeMem(YArray);
      end;
    end; {valid degree}
  end; {InputQuery}
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TSeries.LineBestFit
  Description: Does what it says
       Author: Mat Ballard
 Date created: 08/05/2001
Date modified: 08/05/2000 by Mat Ballard
      Purpose: calculates the line of best fit from Left1 to Right1, plus Left2 to Right2.
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.LineBestFit(Left1, Right1, Left2, Right2: Integer): String;
var
  i: Integer;
  Start1, Finish1, Start2, Finish2: Integer;
{Variables used in least-squares fitting:}
  NoLeastSquarePts: Integer;
  SumX, SumY, SumXsq, SumXY, SumYsq: Double;
  XDatum, YDatum: Single;
  DoDual: Boolean;
{  Slope, Intercept: Single; - are globals to allow drawing of line}

  procedure InitializeFit;
  begin
{Initialize the fit parameters:}
    NoLeastSquarePts := 0;
    SumX := 0;
    SumY := 0;
    SumXsq := 0;
    SumXY := 0;
    SumYsq := 0;
  end;

  procedure ProcessPoint;
  begin
    Inc(NoLeastSquarePts);
    if (TPlot(FPlot).Axes[0].LogScale) then
      XDatum := Ln(FXData^[i])
    else
      XDatum := FXData^[i];
    if (TPlot(FPlot).Axes[FYAxisIndex].LogScale) then
      YDatum := Ln(FYData^[i])
    else
      YDatum := FYData^[i];
    SumX := SumX + XDatum;
    SumY := SumY + YDatum;
    SumXsq := SumXsq + Sqr(XDatum);
    SumXY := SumXY + XDatum * YDatum;
    SumYsq := SumYsq + Sqr(YDatum);
  end;

begin
  Result := '';
  InitializeFit;

{Determine the starting and ending points:}
  Start1 := GetNearestPointToFX(Left1);
  Finish1 := GetNearestPointToFX(Right1);
  DoDual := FALSE;
  if (Left2 < Right2) then
  begin
    DoDual := TRUE;
    Start2 := GetNearestPointToFX(Left2);
    Finish2 := GetNearestPointToFX(Right2);
  end
  else
  begin
    Start2 := 0;
    Finish2 := -1;
  end;

{Allocate memory:}
  nTerms := 2;

  for i := Start1 to Finish1 do
    ProcessPoint;

  if (DoDual) then
    for i := Start2 to Finish2 do
      ProcessPoint;

{so the slope and intercept are:}
  try
    Ai[1] := (NoLeastSquarePts * SumXY - SumX * SumY) /
             (NoLeastSquarePts * SumXsq - Sqr(SumX));
    Ai[0] := (SumY / NoLeastSquarePts) - Ai[1] * (SumX / NoLeastSquarePts);
    R_Square := Sqr(NoLeastSquarePts * SumXY - SumX * SumY) /
           ((NoLeastSquarePts * SumXsq - Sqr(SumX)) * (NoLeastSquarePts * SumYsq - Sqr(SumY)));
    Result := GetFitAsString;
    FShowFit := TRUE;
  except
    EMathError.CreateFmt('TSeries.LineBestFit: Error:' + #13#10+
      'NoLeastSquarePts = %d' + #13#10+
      'SumX = %g' + #13#10+
      'SumY = %g' + #13#10+
      'SumXsq = %g' + #13#10+
      'SumXY = %g' + #13#10+
      'SumYsq = %g.',
      [NoLeastSquarePts, SumX, SumY, SumXsq, SumXY, SumYsq]);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: GetFitAsString
  Description: Performs calculations on Screen Data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Finishes off Line of Best Fit determinations
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetFitAsString: String;
var
  i: Integer;
  Str: String;
begin
{  TPlot(FPlot).Axes[0] := Self.XAxis;
  TPlot(FPlot).Axes[FYAxisIndex] := Self.YAxis;}
  if (TPlot(FPlot).Axes[FYAxisIndex].LogScale) then
    Result := Format('Ln(%s) = ', [TPlot(FPlot).Axes[FYAxisIndex].Title.Caption])
   else
    Result := TPlot(FPlot).Axes[FYAxisIndex].Title.Caption + '  =  ';
  Result := Result + FloatToStrF(Ai[0], ffGeneral, 5, 3);
  Str := 'A[0]' + #9 + FloatToStrF(Ai[0], ffGeneral, 5, 3);
  if (TPlot(FPlot).Axes[0].LogScale) then
    Result := Result + Format(' + %s  Ln(%s)', [FloatToStrF(Ai[1], ffGeneral, 5, 3), TPlot(FPlot).Axes[0].Title.Caption])
   else
    Result := Result + Format(' + %s  %s', [FloatToStrF(Ai[1], ffGeneral, 5, 3), TPlot(FPlot).Axes[0].Title.Caption]);
  Str := Str + #13#10 + 'A[1]' + #9 + FloatToStrF(Ai[1], ffGeneral, 5, 3);
  for i := 3 to nTerms do
  begin
    if (TPlot(FPlot).Axes[0].LogScale) then
      Result := Result + #10 + Format(' + %s  Ln(%s)^%d)', [FloatToStrF(Ai[i-1], ffGeneral, 5, 3), TPlot(FPlot).Axes[0].Title.Caption, i-1])
     else
      Result := Result + #10 +  Format(' + %s  (%s)^%d', [FloatToStrF(Ai[i-1], ffGeneral, 5, 3), TPlot(FPlot).Axes[0].Title.Caption, i-1]);
    Str := Str + #13#10 + Format('A[%d]', [i-1]) + #9 + FloatToStrF(Ai[i-1], ffGeneral, 5, 3);
  end; {for}
  Result := Result + #10 + 'Rsq = ' + FloatToStrF(R_Square, ffGeneral, 5, 3);

{$IFDEF GUI}
  ClipBoard.AsText := Result + #13#10 + Str;
{$ENDIF}
end;

{Sub BestFit (iStart%, iFinish%, X_Data() As Single, Y_Data() As Single, Clear_Regs%, Slope!, Intercept!, RSQ!)
Dim i%
Dim Msg$
Static SumX!
Static SumY!
Static SumXsq!
Static SumXY!
Static SumYsq!
Static No_Pts%

    On Error GoTo BestFit_ErrorHandler

'   we initialise the sums for a least-squares fit:
    If (Clear_Regs% = True) Then
        No_Pts% = 0
        SumX! = 0
        SumY! = 0
        SumXsq! = 0
        SumXY! = 0
        SumYsq! = 0
    End If

    Select Case LogCase()
    Case 0      'neither axis is logged:
'   Do the summation:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + X_Data(i%)
            SumY! = SumY! + Y_Data(i%)
            SumXsq! = SumXsq! + X_Data(i%) ^ 2
            SumXY! = SumXY! + X_Data(i%) * Y_Data(i%)
            SumYsq! = SumYsq! + Y_Data(i%) ^ 2
        Next i%
    Case 1      'only the X-axis is logged:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + Log(X_Data(i%))
            SumY! = SumY! + Y_Data(i%)
            SumXsq! = SumXsq! + Log(X_Data(i%)) ^ 2
            SumXY! = SumXY! + Log(X_Data(i%)) * Y_Data(i%)
            SumYsq! = SumYsq! + Y_Data(i%) ^ 2
        Next i%
    Case 2      'only the Y-axis is logged:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + X_Data(i%)
            SumY! = SumY! + Log(Y_Data(i%))
            SumXsq! = SumXsq! + X_Data(i%) ^ 2
            SumXY! = SumXY! + X_Data(i%) * Log(Y_Data(i%))
            SumYsq! = SumYsq! + Log(Y_Data(i%)) ^ 2
        Next i%
    Case 3      'both axes are logged:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + Log(X_Data(i%))
            SumY! = SumY! + Log(Y_Data(i%))
            SumXsq! = SumXsq! + Log(X_Data(i%)) ^ 2
            SumXY! = SumXY! + Log(X_Data(i%)) * Log(Y_Data(i%))
            SumYsq! = SumYsq! + Log(Y_Data(i%)) ^ 2
        Next i%
    Case 4      'X axis is Log10'ed
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + LOG10_E * Log(X_Data(i%))
            SumY! = SumY! + Y_Data(i%)
            SumXsq! = SumXsq! + (LOG10_E * Log(X_Data(i%))) ^ 2
            SumXY! = SumXY! + LOG10_E * Log(X_Data(i%)) * Y_Data(i%)
            SumYsq! = SumYsq! + Y_Data(i%) ^ 2
        Next i%
    Case 6      'X axis is Log10'ed, Y axis is ln'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + LOG10_E * Log(X_Data(i%))
            SumY! = SumY! + Log(Y_Data(i%))
            SumXsq! = SumXsq! + (LOG10_E * Log(X_Data(i%))) ^ 2
            SumXY! = SumXY! + LOG10_E * Log(X_Data(i%)) * Log(Y_Data(i%))
            SumYsq! = SumYsq! + Log(Y_Data(i%)) ^ 2
        Next i%
    Case 8      'Y axis is Log10'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + X_Data(i%)
            SumY! = SumY! + LOG10_E * Log(Y_Data(i%))
            SumXsq! = SumXsq! + X_Data(i%) ^ 2
            SumXY! = SumXY! + X_Data(i%) * LOG10_E * Log(Y_Data(i%))
            SumYsq! = SumYsq! + (LOG10_E * Log(Y_Data(i%))) ^ 2
        Next i%
    Case 9      'X axis is ln'ed, Y axis is Log10'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + Log(X_Data(i%))
            SumY! = SumY! + LOG10_E * Log(Y_Data(i%))
            SumXsq! = SumXsq! + Log(X_Data(i%)) ^ 2
            SumXY! = SumXY! + Log(X_Data(i%)) * LOG10_E * Log(Y_Data(i%))
            SumYsq! = SumYsq! + (LOG10_E * Log(Y_Data(i%))) ^ 2
        Next i%
    Case 12      'both axes are Log10'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + LOG10_E * Log(X_Data(i%))
            SumY! = SumY! + LOG10_E * Log(Y_Data(i%))
            SumXsq! = SumXsq! + (LOG10_E * Log(X_Data(i%))) ^ 2
            SumXY! = SumXY! + LOG10_E * Log(X_Data(i%)) * LOG10_E * Log(Y_Data(i%))
            SumYsq! = SumYsq! + (LOG10_E * Log(Y_Data(i%))) ^ 2
        Next i%
    End Select

'   so the slope and intercept are:
    Slope! = (No_Pts% * SumXY! - SumX! * SumY!) / (No_Pts% * SumXsq! - (SumX! ^ 2))
    Intercept! = (SumY! / No_Pts%) - Slope! * (SumX! / No_Pts%)
    RSQ! = (No_Pts% * SumXY! - SumX! * SumY!) ^ 2 / ((No_Pts% * SumXsq! - (SumX! ^ 2)) * (No_Pts% * SumYsq! - (SumY! ^ 2)))

BestFit_FINISHED:


Exit Sub

BestFit_ErrorHandler:   ' Error handler line label.

    Select Case Err
    Case 5
        Resume Next
    Case Else
        Msg$ = "Panic in " & "BestFit_ErrorHandler !"
        Msg$ = Msg$ & LF & LF & "Error No. " & Str$(Err) & ": " & Error$
        Response% = Message(Msg$, MB_OK + MB_ICONEXCLAMATION, "Error !", NO, H_PANIC)
    End Select

    Resume BestFit_FINISHED


End Sub
}

{------------------------------------------------------------------------------
     Function: TSeries.FindHighsLows
  Description: This function finds all the Highs (and troughs) in a region
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ??? Property
 Return Value: the number of Highs
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.FindHighsLows(Start, Finish, HeightSensitivity: Integer): Integer;
var
  i,
  LastHigh,
  LastLow: Integer;
  Highseek: Boolean;
  Delta: Single;
begin
{this routine finds all the major Highs in a region;
 See "FindAHigh" for a single High finding routine.}

{set the sensitivity:}
  Delta := (HeightSensitivity / 100.0) * (FYMax - FYMin);
  ClearHighsLows;

{initialise variables:}
  LastHigh := Start;
  LastLow := Start;
  Highseek := TRUE;

{allocate memory for results:}
  GetMem(FHighs, FHighCapacity * SizeOf(Integer));
  GetMem(FLows, FHighCapacity * SizeOf(Integer));
{we set the first point to a low}
  {Lows^[FLowCount] := LastLow;
  Inc(FLowCount);}

  for i := Start to Finish do
  begin
    if (Highseek = TRUE) then
    begin
      if (FYData^[i] > FYData^[LastHigh]) then
        LastHigh := i;
      if (FYData^[i] < (FYData^[LastHigh] - Delta)) then
      begin
{The Last High was a real maximum:}
        Highs^[FHighCount] := LastHigh;
        Inc(FHighCount);
        if (FHighCount >= FHighCapacity-2) then
        begin
{add 10 more points:}
          Inc(FHighCapacity, 10);
          ReAllocMem(FHighs, FHighCapacity * SizeOf(Integer));
          ReAllocMem(FLows, FHighCapacity * SizeOf(Integer));
        end;
        Highseek := FALSE;
        LastLow := i;
      end;
    end
    else
    begin
      if (FYData^[i] < FYData^[LastLow]) then
        LastLow := i;
      if (FYData^[i] > (FYData^[LastLow] + Delta)) then
      begin
{The Last Low was a real minimum:}
        Lows^[FLowCount] := LastLow;
        Inc(FLowCount);
        Highseek := TRUE;
        LastHigh := i;
      end; {comparison}
    end; {seeking high or low}
  end; {for}
  Lows^[FLowCount] := LastLow;

  Result := FHighCount;
end;

{ from ..\3rd\Fourier\fft.html:
 As a concrete example, if you are using a sampling rate of 44100 samples/second,
 and the length of your recording is 1024 samples, the amount of time represented
 by the recording is 1024 / 44100 = 0.02322 seconds, so the base frequency f0
 will be 1 / 0.02322 = 43.07 Hz. If you process these 1024 samples with the FFT,
 the output will be the sine and cosine coefficients ak and bk for the frequencies
 43.07Hz, 2*43.07Hz, 3*43.07Hz, etc. To verify that the transform is functioning
 correctly, you could then generate all the sines and cosines at these frequencies,
 multiply them by their respective ak and bk coefficients, add these all together,
 and you will get your original recording back! It's a bit spooky that this actually works! }
{------------------------------------------------------------------------------
    Procedure: TSeries.DoFFT
  Description: Fast Fourier [inverse] Transforms this series.
       Author: Mat Ballard
 Date created: 08/27/2001
Date modified: 09/27/2002 by Mat Ballard
      Purpose: data manipulation
 Known Issues: Neither Getmem, Allocmem, or Reallocmem work; all crash on entry
               to Fourier.fft - for no apparent reason. Only SetLength works
 ------------------------------------------------------------------------------}
procedure TSeries.DoFFT(Forwards: Boolean);
{$IFDEF COMPILER4_UP}
var
  i: Integer;
  RealSum, ImagSum: Single;
  RealIn, ImagIn, RealOut, ImagOut: array of TPrecision;
{$ENDIF}
begin
  if (FNoPts = 0) then exit;
{$IFDEF COMPILER4_UP}
  if (FFourier.IsPowerOfTwo(FNoPts)) then
  begin
    try
      SetLength(RealIn, FNoPts);
      SetLength(ImagIn, FNoPts);
      SetLength(RealOut, FNoPts);
      SetLength(ImagOut, FNoPts);
      for i := 0 to FNoPts-1 do
        RealIn[i] := FYData[i];
      if (Self.SeriesType = stComplex) then
      begin
        for i := 0 to FNoPts-1 do
          ImagIn[i] := FExtraData[0][i];
      end
      else
      begin
        for i := 0 to FNoPts-1 do
          ImagIn[i] := 0;
      end;

      RealSum := 0;
      ImagSum := 0;
      if (Forwards) then
      begin
        FFourier.fft(FNoPts, RealIn, ImagIn, RealOut, ImagOut);
        SetSeriesType(stComplex);
        for i := 0 to FNoPts-1 do
        begin
          FYData[i] := RealOut[i];
          FExtraData[0][i] := ImagOut[i];
          RealSum := RealSum + Abs(RealOut[i]);
          ImagSum := ImagSum + Abs(ImagOut[i]);
        end;
      end
      else
      begin // we inverse transform
        FFourier.ifft(FNoPts, RealIn, ImagIn, RealOut, ImagOut);
        for i := 0 to FNoPts-1 do
        begin
          FYData[i] := RealOut[i];
          RealSum := RealSum + Abs(RealOut[i]);
          ImagSum := ImagSum + Abs(ImagOut[i]);
        end;
      end;
{Is the imaginary component essentially zero ?}
      if (RealSum / ImagSum < 10000) then
      begin
        SetSeriesType(stComplex);
        for i := 0 to FNoPts-1 do
          FExtraData[0][i] := ImagOut[i];
      end
       else
        SetSeriesType(stNormal);
    finally {Cleanup}
      SetLength(RealIn, 0);
      SetLength(ImagIn, 0);
      SetLength(RealOut, 0);
      SetLength(ImagOut, 0);
    end;
  end
   else
    //DoGlassmanFFT(Forwards);
    raise EComponentError.Create('Glassman FFT does not work yet !' + #13#10 +
      'You can only transform Series with 2^n points !');
{$ENDIF}      
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DoGlassmanFFT
  Description: Glassman Fast Fourier [inverse] Transforms this series.
       Author: Mat Ballard
 Date created: 08/27/2001
Date modified: 09/27/2002 by Mat Ballard
      Purpose: FFT for non-2^n series
 Known Issues: does not work ! keeps throwing access violation errors !
               Have to debug U_FTG2 !
 ------------------------------------------------------------------------------}
{TODO 2 -o anyone -c Fourier : Glassman FFT needs to be debugged; keeps throwing access violations}
procedure TSeries.DoGlassmanFFT(Forwards: Boolean);
var
  i: Integer;
  CInput, COutput, CScratch: PComplexArray;
begin
  if (FNoPts = 0) then exit;

  CInput := AllocMem(FNoPts * SizeOf(TComplex));
  COutput := AllocMem(FNoPts * SizeOf(TComplex));
  CScratch := AllocMem(FNoPts * SizeOf(TComplex));
  try
    {SetLength(CInput, FNoPts);
    SetLength(COutput, FNoPts);
    SetLength(CScratch, FNoPts);}
    for i := 0 to FNoPts-1 do
      CInput[i].r := FYData[i];
    if (Self.SeriesType = stComplex) then
    begin
      for i := 0 to FNoPts-1 do
        CInput[i].i := FExtraData[0][i];
    end
    else
    begin
      for i := 0 to FNoPts-1 do
        CInput[i].i := 0;
    end;

    if (Forwards) then
    begin
      U_FTG2.CFTG(CInput^, COutput^, CScratch^, FNoPts);
      SetSeriesType(stComplex);
      for i := 0 to FNoPts-1 do
      begin
        FYData^[i] := COutput[i].r;
        FExtraData[0]^[i] := COutput[i].i;
      end;
    end
    else
    begin // we inverse transform
      U_FTG2.CFTG(CInput^, COutput^, CScratch^, -FNoPts);
      SetSeriesType(stNormal);
      for i := 0 to FNoPts-1 do
      begin
        FYData^[i] := COutput[i].r;
      end;
    end;
  finally {Cleanup}
    FreeMem(CInput);
    FreeMem(COutput);
    FreeMem(CScratch);
    {SetLength(CInput, 0);
    SetLength(COutput, 0);
    SetLength(CScratch, 0);}
  end;
end;


end.

