unit Datalist;

{$I Plot.inc}

{$IFDEF DELPHI4}
  {$DEFINE DELPHI4_DOWN}
{$ENDIF}
{$IFDEF BCB3}
  {$DEFINE DELPHI4_DOWN}
{$ENDIF}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: PlotList.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 04/18/2001
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Purpose:
This unit contains the TSeriesList sub-component - that manages the data for
ALL Series for TPlot.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils,
{$IFDEF GUI}
  {$IFDEF WINDOWS}
  Controls, Dialogs, Graphics, Windows, Clipbrd,
  {$ENDIF}
  {$IFDEF WIN32}
  Controls, Dialogs, Graphics, Windows, Clipbrd,
  {$ENDIF}
  {$IFDEF LINUX}
  Types,
  QControls, QDialogs, QGraphics, QClipbrd,
  {$ENDIF}
{$ELSE}
  gd, gd_io, gdWindows, gdGraphics, {gdWinapi,}
{$ENDIF}

  Parser10, Functons,
  {$IFDEF NO_MATH}NoMath,{$ELSE}Math,{$ENDIF}
  Axis, Data, Import, Plotdefs, Misc, Titles;

type
{TSeriesList is a TCollection of Series that manages it's items.}
  TSeriesList = class(TCollection)
  private
{The AxisList is created and managed in the Plot unit and TCustomPlot component.
 The specific axes are:
   0 .. X Axis
   1 .. Primary Y Axis
   2 .. Secondary Y Axis
   3 .. Tertiary Y Axis
   4 .. etc.}
    //FAxisList: TCollection;
    FDataChanged: Boolean;
    FIgnoreChanges: Boolean;
    FPlot: TComponent;

    FOnStyleChange: TNotifyEvent;
    FOnDataChange: TNotifyEvent;

    mLastSavedPoint: Integer;
    mPlotBorder: TRect;
    mNoPieRows: Integer;

    TheParser: TParser;
{This is the FUNCTION Parser. It does deep, black magic to create a new series
 from existing data.}
    TheParserData: PParserFloatArray;

    //function GetNullValue: Single;
    function GetItem(Index: Integer): TSeries;
    procedure SetItem(Index: Integer; Value: TSeries);
  protected
{Required for the Object inspector:}
    function GetOwner: TPersistent; override;
    function GetXmin: Single;
    function GetXmax: Single;
    function GetYmin: Single;
    function GetYmax: Single;
    function GetZmin: Single;
    function GetZmax: Single;
    {function GetYErrorMin: Single;
    function GetYErrorMax: Single;}

    function GetDisplaced: Boolean;
{Have any of the Series been displaced ?}
    function GetMaxNoPts: Integer;
    function GetMinNoPts: Integer;
    function GetTotalNoPts: Integer;
    procedure StyleChange(Sender: TObject);
    procedure DataChange(Sender: TObject);
    procedure DoStyleChange; virtual;
    procedure DoDataChange; virtual;
    //procedure SetAxisList(Value: TCollection);
    procedure Update(Item: TCollectionItem); override;
    procedure SortTriangleVertices(var Pt1, Pt2, Pt3: T3DZPoint; var p1, p2, p3: p3DZPoint);
    procedure SortTriangleVerticesOnY(var Pt1, Pt2, Pt3: T3DRealPoint; var p1, p2, p3: p3DRealPoint);

  public
    //property AxisList: TCollection read FAxisList write SetAxisList;
{The Collection of axes in TPlot.}
    property DataChanged: Boolean read FDataChanged write FDataChanged stored FALSE;
{Has the data any series changed ?}
    property Displaced: Boolean read GetDisplaced;
{Has the any series been moved on screen ?}
    property IgnoreChanges: Boolean read FIgnoreChanges write FIgnoreChanges;
{Shall we ignore Change and DataChange events ?}
    property TotalNoPts: Integer read GetTotalNoPts stored FALSE;
{The total number of points in all series.}
    property MaxNoPts: Integer read GetMaxNoPts stored FALSE;
{The number of points in the largest series.}
    property MinNoPts: Integer read GetMinNoPts stored FALSE;
{The number of points in the smallest series.}
    property Items[Index: Integer]: TSeries read GetItem write SetItem; default;
{This provides direct access to the series maintained by SeriesList.}
    property Plot: TComponent read FPlot;
{The plot to which this SeriesList belongs.}    
    property Xmin: Single read GetXmin stored FALSE;
{The minimum X value of ALL Series.}
    property Xmax: Single read GetXmax stored FALSE;
{The maximum X value of ALL Series.}

    property Ymin: Single read GetYmin;
{The minimum Y value of ALL Series connected to the PRIMARY Y Axis.}
    property Ymax: Single read GetYmax;
{The maximum Y value of ALL Series connected to the PRIMARY Y Axis.}
    property Zmin: Single read GetZmin;
{The minimum Z value of ALL Series.}
    property Zmax: Single read GetZmax;
{The maximum Z value of ALL Series.}
    //property YErrorMin: Single read GetYErrorMin;
{The minimum Y value of ALL Series plus their error.}
    //property YErrorMax: Single read GetYErrorMax;
{The maximum Y value of ALL Series plus their error.}

    property OnStyleChange: TNotifyEvent read FOnStyleChange write FOnStyleChange;
{This notifies the owner (usually TPlot) of a change in style of this series.}
    property OnDataChange: TNotifyEvent read FOnDataChange write FOnDataChange;
{This notifies the owner (usually TPlot) of a change in the data of this series.}
{}
{NOTE: D1 does not allow published properties for TList descendants.}

    Constructor Create(Plot: TComponent); virtual;
{Saves the Axes List and initializes mLastSavedPoint.}
    destructor Destroy; override;
{This is a standard Destructor.}

    procedure Loaded;
{Collections do not have a Loaded method to override - which is a real pain !
 So we have to implement a public kludge here and in TCollectionItem}

{$IFDEF DELPHI4_DOWN}
    procedure Delete(Index: Integer);
{$ENDIF}

    function Add: TSeries;
{This adds a new, empty series that depends on series XSeriesIndex to the list.}

    function AddDependent(XSeriesIndex: Integer): TSeries;
{This adds a new, empty series that depends on series XSeriesIndex to the list.}
{}
{Like its ancestor function, and its relatives AddInternal and AddExternal,
 Add returns the index of the new item, where the first item in the list has
 an index of 0.}
    function AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): TSeries;
{This adds a new, empty series to the list, and sets its data to point to the
 external XPointer, YPointer data.}
{}
{Like its ancestor function, and its relatives AddInternal and Add,
 AddExternal returns the index of the new item, where the first item in the list has
 an index of 0.}
    function AddInternal(XPointer, YPointer: pSingleArray;
      NumberOfPoints: Integer): TSeries;
    function AddInternalEx(XPointer, YPointer: pSingleArray;
      ExtraPointers: array of pSingleArray; NumberOfPoints: Integer): TSeries;
{This adds a new, empty series to the list, and copies the data from the
 XPointer, YPointer data.}
{}
{Like its ancestor function, and its relatives Add and AddExternal,
 AddInternal returns the index of the new item, where the first item in the list has
 an index of 0.}

    procedure ClearSeries;
{This deletes all Series from the list, freeing the series.}
{}
{Note that the Clear does not exist in the D1-3 version, because TList.Clear
 is a static method and cannot be overridden. This is why we created ClearSeries.}

    function CloneSeries(TheSeries: Integer): TSeries;
{This adds a new, empty series to the list, copies the data and properties from
 TheSeries into the new clone, and changes the color and Y Displacement.}
{}
{CloneSeries returns TRUE if successful.}
    procedure DeleteSeries(Index: Integer; Ask: Boolean);
{This deletes TheSeries from the list.}
{$IFDEF GUI}
    function ParseData(TheData: TStringList; TheHelpFile: String): Boolean;
{This parses imported or pasted data and adds it to the SeriesList.}
{$ENDIF}
    function ConvertTextData(TheData: TStringList; InitialSeriesCount: Integer; Delimiter: String): Boolean;
{This takes a parsed stringlist converts it to numerical data.}
{$IFDEF VER200_COMPATIBLE}
    function ConvertTextData_Ver200(ColCount, SeriesCount, FirstLine: Integer;
      Delimiter: String; TheData: TStringList; SeriesInfo: pSeriesInfoArray): Boolean;
{$ENDIF}
    function ConvertXYZData(ImportForm: TImportForm; TheData: TStringList): Boolean;
{This takes an UNparsed stringlist with XYZ values and converts it to numerical data.}
    function ConvertBinaryData(TheStream: TMemoryStream; InitialSeriesCount: Integer): Boolean;
{This takes a parsed stringlist converts it to numerical data.}
{$IFDEF VER200_COMPATIBLE}
    function ConvertBinaryData_Ver200(ColCount, SeriesCount: Integer;
      TheStream: TMemoryStream; SeriesInfo: pSeriesInfoArray): Boolean;
{$ENDIF}

    procedure FourierAllSeries(Forwards: Boolean);
{Fast Fourier [inverse] Transforms ALL series.}

    function CalcFunction(NewSeries: TSeries; iStart: Integer): Boolean;
    function CreateFunctionSeries: TSeries;
{This USER-called creates a new series which is a function of the existing series.}
    function AddFunctionSeries(AnExpression: String): TSeries;
    //procedure SetupVariables(AnExpression: String);
    procedure ExpressionChange(Sender: TObject);
{This PROGRAMMER-called creates a new series which is a function of the existing series.}

    procedure DataAsHTMLTable(var TheData: TStringList);
{This returns all the data in HTML format as a StringList.}

    procedure SaveToStream(var TheStream: TMemoryStream; AsText: Boolean; Delimiter: Char);
{This returns all the data as a MemoryStream.}
    procedure SaveSubHeaderStream(TheStream: TMemoryStream; Delimiter: Char);
{This returns all the SubHeader data as a MemoryStream.}
    procedure SaveBinaryStream(TheStream: TMemoryStream; Start, Finish: Integer);
    procedure SaveTextStream(TheStream: TMemoryStream; Delimiter: Char; Start, Finish: Integer);
    procedure AppendStream(AsText: Boolean; Delimiter: Char; TheStream: TMemoryStreamEx);
{This returns the data collected since mLastSavedPoint as a MemoryStream.}
    function LoadFromStream(AStream: TMemoryStream; Delimiter: String; InitialSeriesCount: Integer; var AsText: Boolean): Boolean;
{Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
 or decides to run it through ParseData instead}
{$IFDEF VER200_COMPATIBLE}
    function LoadFromStream_Ver200(AStream: TMemoryStream; var AsText: Boolean): Boolean;
{$ENDIF}
    procedure Draw(ACanvas: TCanvas; XYFastAt: Integer);
{This draws all the series on the given canvas.}
    //procedure DrawError(ACanvas: TCanvas);
{Extended Drawing procedure for series with errorbars.}
    //procedure DrawBubble(ACanvas: TCanvas; BubbleSize: Integer);
{Extended Drawing procedure for Bubble plots.}
    //procedure DrawMultiple(ACanvas: TCanvas; Link: TLink);
{Extended Drawing procedure for linking multiple series.}
    procedure DrawColumns(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas in columns.}
    procedure DrawStack(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas in stacked (summed) columns.}
    procedure DrawNormStack(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas in normalized columns.}
    procedure DrawPie(ACanvas: TCanvas; Border: TBorder; NoRows: Integer);
{This draws all the series on the given canvas as Pie graphs.}
    procedure DrawPolar(ACanvas: TCanvas; PolarRange: Single);
{This draws all the series on the given canvas as a Polar graph.}
    procedure Draw3DWire(ACanvas: TCanvas; ZLink: Boolean);
{This draws all the series on the given canvas as a 3D WireFrame.}
    procedure Draw3DColumn(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas as a 3D Columns.}
    procedure Draw3DContour(ACanvas: TCanvas; Contour: TContour);
{This draws all the series on the given canvas.}
    procedure DrawContour(ACanvas: TCanvas; Contour: TContour);
{This draws all the series on the given canvas as a solid colour contour map.}
    procedure DrawLineContour(ACanvas: TCanvas;
      Contour: TContour);
{This draws all the series on the given canvas as a line contour map.}
    procedure DrawColorScale(ACanvas: TCanvas; TheMin, Span: Single;
      Contour: TContour);

    procedure DrawHistory(ACanvas: TCanvas; HistoryX: Single);
{This draws all the series on the given canvas, in a History mode.}
    //procedure DrawHistoryMultiple(ACanvas: TCanvas; Link: TLink);
{Extended Drawing procedure for linking multiple series in History mode.}
    function GetNearestPoint(
      ThePlotType: TPlotType;
      ColumnGap,
      iX, iY: Integer;
      var TheSeries: Integer;
      var MinDistance: Single;
      var pSeries: TSeries): Integer;
{This returns the Index of the nearest point, and sets TheSeries it belongs to}
  {published}
    function GetSeriesOfZ(ZValue: Single): TSeries;
  end;

implementation

uses
  Plot;

const
  PIE_SIZE = 0.8;

{TSeriesList Constructor and Destructor:---------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TSeriesList.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saves the Axes List and initializes mLastSavedPoint
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TSeriesList.Create(Plot: TComponent);
begin
  inherited Create(TSeries);
  FPlot := Plot;
  mLastSavedPoint := 0;
  //FAxisList := nil;
  TheParser := nil;
end;

{------------------------------------------------------------------------------
   Destructor: TSeriesList.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Frees the Axes and events
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TSeriesList.Destroy;
begin
  FOnStyleChange := nil;
  FOnDataChange := nil;
{NOTE: in D1-D3, Clear is static and cannot be overridden, so we had to
 add a ClearSeries for them:}
  ClearSeries;
  if (Assigned(TheParser)) then
  begin
    TheParser.Free;
    ReallocMem(TheParserData, 0);
  end;
  inherited Destroy;
end;

procedure TSeriesList.Loaded;
var
  i: Integer;
begin
  for i := 0 to Self.Count-1 do
    Self.Items[i].Loaded;
end;

{$IFDEF DELPHI4_DOWN}
procedure TSeriesList.Delete(Index: Integer);
begin
  TCollectionItem(Self.Items[Index]).Free;
end;
{$ENDIF}

{TSeriesList Set procedures ---------------------------------------------------}

{TSeriesList Get functions ----------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.GetOwner
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 08/25/2002
Date modified: 08/25/2002 by Mat Ballard
      Purpose: allows TSeriesList to appear in the Object Inspector
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetOwner: TPersistent;
begin
  Result := FPlot;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetXmin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the XMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetXmin: Single;
var
  i: Integer;
begin
  Result := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Items[i].Visible) then
      if (Result > Items[i].XMin) then
        Result := Items[i].XMin;
  end; {loop over series}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetXMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the XMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetXmax: Single;
var
  i: Integer;
begin
  Result := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Items[i].Visible) then
      if (Result < Items[i].XMax) then
        Result := Items[i].XMax;
  end; {loop over series}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetYMin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the YMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetYMin: Single;
var
  i: Integer;
begin
  Result := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Items[i].Visible) then
      if (Result > Items[i].YMin) then
        Result := Items[i].YMin;
  end; {loop over series}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetYMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the YMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetYMax: Single;
var
  i: Integer;
begin
  Result := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Items[i].Visible) then
      if (Result < Items[i].YMax) then
        Result := Items[i].YMax;
  end; {loop over series}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetZMin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ZMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetZMin: Single;
var
  i: Integer;
begin
  Result := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Items[i].Visible) then
      if (Result > Items[i].ZData) then
        Result := Items[i].ZData;
  end; {loop over series}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetZMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ZMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetZMax: Single;
var
  i: Integer;
begin
  Result := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Items[i].Visible) then
      if (Result < Items[i].ZData) then
        Result := Items[i].ZData;
  end; {loop over series}
end;

{TSeriesList Memory and list management ---------------------------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.Add
  Description: Adds a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2002 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.Add: TSeries;
begin
  Result := TSeries(inherited Add);

  //Result.AxisList := Self.FAxisList;
  Result.OnStyleChange := Self.StyleChange;
  Result.OnDataChange := Self.DataChange;
  Result.ZData := Self.Count-1;

  DoDataChange;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddDependent
  Description: Adds a new Series that depends on the X Axis of an existing series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2002 by Mat Ballard
      Purpose: creates, initializes and adds a new series that depends on an existing series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddDependent(XSeriesIndex: Integer): TSeries;
var
  MasterSeries: TSeries;
begin
  Result := TSeries(Add);
  if ((0 <= XSeriesIndex) and (XSeriesIndex < Count)) then
  begin
    MasterSeries := TSeries(Self.Items[XSeriesIndex]);
    if (MasterSeries <> Result) then
    begin // a series should not depend on itself
      if (not Result.MakeXDataDependent(MasterSeries)) then
      begin
        Delete(Result.Index);
        Result := nil;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddExternal
  Description: Adds a new Series that is externally maintained
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 01/25/2001 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): TSeries;
begin
  Result := TSeries(Add);

  if (not Result.PointToData(XPointer, YPointer, NumberOfPoints)) then
  begin
    Delete(Result.Index);
    Result := nil;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddInternal
  Description: Adds a new Series from external data (copies)
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 01/25/2001 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddInternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): TSeries;
begin
  Result := TSeries(Add);
  Result.AddData(XPointer, YPointer, NumberOfPoints);
end;

function TSeriesList.AddInternalEx(XPointer, YPointer: pSingleArray;
  ExtraPointers: array of pSingleArray; NumberOfPoints: Integer): TSeries;
begin
  Result := TSeries(Add);
  Result.AddDataEx(XPointer, YPointer, ExtraPointers, NumberOfPoints);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.ClearSeries
  Description: frees and deletes all the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.ClearSeries;
begin
  if (Count = 0) then exit;

  Clear;
  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DeleteSeries
  Description: frees and deletes one particular Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DeleteSeries(Index: Integer; Ask: Boolean);
begin
{$IFDEF GUI}
  if (Ask) then
    Ask := not (MessageDlg(
{$IFDEF LINUX}
    'Delete Series',
{$ENDIF}
    'Do you really want to delete ' + Items[Index].Name + ' ?',
    mtWarning, [mbYes,mbNo], 0) = mrYes);

  if (not Ask) then
  begin
    Delete(Index);
    DoDataChange;
  end;
{$ELSE}
  Delete(Index);
  DoDataChange;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.FourierAllSeries
  Description: Fast Fourier [inverse] Transforms ALL series.
       Author: Mat Ballard
 Date created: 08/27/2001
Date modified: 08/27/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.FourierAllSeries(Forwards: Boolean);
var
  i: Integer;
begin
  if (Self.Count = 0) then exit;

  for i := 1 to Self.Count-1 do
  begin
    if (Items[i].NoPts <> TSeries(Self.Items[0]).NoPts) then
      EComponentError.CreateFmt(
        'All series MUST have the same number of point to do a Fourier Transform !' + #13#10 +
        '  Series[0] has %d points, but Series[%d] has %d !',
        [TSeries(Self.Items[0]).NoPts, i, Items[i].NoPts]);
  end;

  for i := 0 to Self.Count-1 do
    Items[i].DoFFT(Forwards);
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.CreateFunctionSeries
  Description: Creates a new series which is a function of existing series
       Author: Mat Ballard
 Date created: 04/03/2001
Date modified: 04/03/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.CreateFunctionSeries: TSeries;
var
  i: Integer;
  FunctionsForm: TFunctionsForm;
  TheExpression: String;
begin
  Result := nil;
  if (Self.Count > 0) then
  begin
    FunctionsForm := TFunctionsForm.Create(nil);
    for i := 0 to Self.Count-2 do
      FunctionsForm.FunctionMemo.Lines.Add(Format('Series%d', [i]) + ' + ');
    FunctionsForm.FunctionMemo.Lines.Add(Format('Series%d', [Self.Count-1]));
    FunctionsForm.SeriesLabel.Caption := Format('Series%d :=', [Self.Count]);
    FunctionsForm.SeriesCount := Self.Count;

    if (mrOK = FunctionsForm.ShowModal) then
    begin
      TheExpression := '';
{read the equation:}
      for i := 0 to FunctionsForm.FunctionMemo.Lines.Count-1 do
        TheExpression := TheExpression + FunctionsForm.FunctionMemo.Lines[i];
      TheExpression := Misc.CleanString(TheExpression, ' ');
{actually create the series:}
      Result := AddFunctionSeries(TheExpression);
    end;
    FunctionsForm.Free;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.AddFunctionSeries
  Description: Creates and then Adds a new series which is a function of existing series
       Author: Mat Ballard
 Date created: 04/03/2001
Date modified: 04/03/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues: Programmers had better call this with an Expression that is known
               to work, otherwise it will go KA-BOOM !
 ------------------------------------------------------------------------------}
function TSeriesList.AddFunctionSeries(AnExpression: String): TSeries;
var
  i: Integer;
begin
{Does TheParser exist yet ?}
  if (TheParser = nil) then
  begin
    TheParser := TParser.Create(nil);
    ReallocMem(TheParserData, 99 * SizeOf(PParserFloat));
  end;
{Remember, we nuke it in the destructor !}

{Create the new series and set it up:}
  Result := Self.Add;
{Note: Setting Expression before the event means the event will not fire}
  Result.Expression := AnExpression;
  Result.OnExpressionChange := ExpressionChange;

{Set up the variables:}
  for i := 0 to Self.Count-1 do
  begin
    TheParserData^[i] := TheParser.SetVariable(Format(UpperCase('Series') + '%d', [i]), 0);
    TheParserData[Self.Count + i] := TheParser.SetVariable(Format('DSERIES%d', [i]), 0);
    TheParserData[2 * Self.Count + i] := TheParser.SetVariable(Format('ISERIES%d', [i]), 0);
  end;

  if (not CalcFunction(Result, 0)) then
  begin
    Delete(Result.Index);
    Result := nil;
  end;
end;

(*{------------------------------------------------------------------------------
    Procedure: TSeriesList.SetupVariables
  Description: Sets up then re-calculates a series which is a function of existing series, whose Expression changes
       Author: Mat Ballard
 Date created: 05/30/2002
Date modified: 05/30/2002 by Mat Ballard
      Purpose: data manipulation
 Known Issues: kinda hairy
 ------------------------------------------------------------------------------}
procedure TSeriesList.SetupVariables(AnExpression: String);
var
  iPos, nPos: Integer;
  Str: String;
  Handled: Boolean;

  function GetValue: Integer;
  var
    Value: String;
  begin
    Value := '';
    while ((nPos <= Length(Str) and (Str[nPos] in ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9']) do
    begin
      Value := Value + Str[nPos];
      Inc(nPos);
    end;
    Result := StrToInt(Value);
  end;

begin
  iPos := ForwardPos('SERIES', Str, 1);
  Str := UpperCase(AnExpression);
  while (iPos > 0) do
  begin
    Handled := FALSE;
    nPos := iPos + 6;
    if (iPos > 1) then
    begin
      if (Str[iPos-1] = 'I') then
      begin
        Handled := TRUE;
      end
      else if (Str[iPos-1] = 'D') then
      begin
        Handled := TRUE;
      end;
    end;
    if (not Handled) then
    begin

    end;
    iPos := ForwardPos('SERIES', Str, iPos);
  end;
end;*)

{------------------------------------------------------------------------------
    Procedure: TSeriesList.ExpressionChange
  Description: Sets up then re-calculates a series which is a function of existing series, whose Expression changes
       Author: Mat Ballard
 Date created: 05/30/2002
Date modified: 05/30/2002 by Mat Ballard
      Purpose: data manipulation
 Known Issues: kinda hairy
 ------------------------------------------------------------------------------}
procedure TSeriesList.ExpressionChange(Sender: TObject);
var
  i: Integer;
begin
{Set up the variables:}
  for i := 0 to Self.Count-1 do
  begin
    TheParserData[i] := TheParser.SetVariable(Format('SERIES%d', [i]), 0);
    TheParserData[Self.Count + i] := TheParser.SetVariable(Format('DSERIES%d', [i]), 0);
    TheParserData[2*Self.Count + i] := TheParser.SetVariable(Format('ISERIES%d', [i]), 0);
  end;

  CalcFunction(TSeries(Sender), 0);
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.CalcFunction
  Description: (Re)Calculates a series which is a function of existing series
       Author: Mat Ballard
 Date created: 05/30/2002
Date modified: 05/30/2002 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.CalcFunction(NewSeries: TSeries; iStart: Integer): Boolean;
var
  i, j: Integer;
  Str: String;
  DoDifferential, DoIntegral: Boolean;
begin
  Result := FALSE;
  if (Length(NewSeries.Expression) > 0) then
  begin
{try to run the Parser:}
    try
{Set up equation:}
      TheParser.Expression := NewSeries.Expression;
      Str := Uppercase(TheParser.Expression);
      DoDifferential := (Pos('DSERIES', Str) > 0);
      DoIntegral := (Pos('ISERIES', Str) > 0);
{Loop over all points in Series[0]:}
      for i := iStart to TSeries(Self.Items[0]).NoPts-1 do
      begin
{add the Series variables to the Parser, or reset their values:}
        for j := 0 to Self.Count-1 do
          if (i < TSeries(Self.Items[j]).NoPts) then
            TheParserData[j]^ := TSeries(Self.Items[j]).YData^[i];
        if (DoDifferential) then
        begin
          if (i < TSeries(Self.Items[0]).NoPts-1) then
          begin
            for j := 0 to Self.Count-1 do
              if (i < TSeries(Self.Items[j]).NoPts) then
                TheParserData[Integer(Self.Count) + j]^ := (TSeries(Self.Items[j]).YData^[i+1] - TSeries(Self.Items[j]).YData^[i]) /
                  (TSeries(Self.Items[j]).XData^[i+1] - TSeries(Self.Items[j]).XData^[i]);
          end
          else
          begin
            for j := 0 to Self.Count-1 do
              if (i < TSeries(Self.Items[j]).NoPts) then
                TheParserData[Integer(Self.Count) + j]^ := (TSeries(Self.Items[j]).YData^[i] - TSeries(Self.Items[j]).YData^[i-1]) /
                  (TSeries(Self.Items[j]).XData^[i] - TSeries(Self.Items[j]).XData^[i-1]);
          end;
        end;
        if (DoIntegral) then
        begin
          if (i = 0) then
          begin
            for j := 0 to Self.Count-1 do
              if (i < TSeries(Self.Items[j]).NoPts) then
                TheParserData[2*Integer(Self.Count) + j]^ := 0;
          end
          else
          begin
            for j := 0 to Self.Count-1 do
              if (i < TSeries(Self.Items[j]).NoPts) then
                TheParserData[2*Integer(Self.Count) + j]^ := TheParserData[2*Integer(Self.Count) + j]^ +
                  (TSeries(Self.Items[j]).YData^[i] + TSeries(Self.Items[j]).YData^[i-1]) *
                    (TSeries(Self.Items[j]).XData^[i] - TSeries(Self.Items[j]).XData^[i-1]) / 2;
          end;
        end;
        NewSeries.AddPoint(-1, TheParser.Value, FALSE, TRUE);
      end;
      Result := TRUE;
    except
      ShowMessage('Calculation of new series Failed !');
    end;
  end;
end;

{TSeriesList the movie ! ------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas
 Known Issues: where do we put the drawing and the "GetNearestPoint" methods ?

    ptXY         needs TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1] - done in Data
    ptError      needs TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1], AND Index - done in Datalist
    ptMultiple   needs TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1] - done in Data
    ptColumn     needs ColumnGap, Index, TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1] - done in Datalist
    ptStack      needs ALL Series, ColumnGap, Index, TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1] - done in Datalist
    ptNormStack  needs ALL Series, ColumnGap, Index, TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1] - done in Datalist
    ptPie        needs bounding rectangle - done in Data
    ptPolar      unknown as yet
    pt3D         needs ALL Series, TPlot(FPlot).Axes[0], TPlot(FPlot).Axes[1], AND FZAxis - done in Serlist

    There seems to be no simple solution to this: some methods have to be, or
    are best in this unit; others, according to the principles:
        a. bury as deep as possible;
        b. each series _SHOULD_ know how to draw itself;
    would seem better to be in Data.
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw(ACanvas: TCanvas; XYFastAt: Integer);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw: ACanvas is nil !');
{$ENDIF}

  for i := 0 to Count-1 do
  begin
    if (Items[i].ShadeLimits) then
      Items[i].DrawShades(ACanvas, XYFastAt);
  end; {loop over series}

  for i := 0 to Count-1 do
  begin
    Items[i].Draw(ACanvas, XYFastAt);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawColumns
  Description: standard Drawing procedure for Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2002 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawColumns(ACanvas: TCanvas; ColumnGap: TPercent);
var
  i,
  j,
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXStart,
  iXEnd,
  iY: Integer;
  dX: Single;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit;

{all columns are plotted against the X Axis and Data of the first series,
 otherwise chaos can result.}

{loop over every series:}
  for i := 0 to Count-1 do
  begin
    //Items[i] := Items[i];
    if (Items[i].IsLine) then // draw it as a line anyway
      Items[i].Draw(ACanvas, 10000)
    else
    begin
      ACanvas.Pen.Assign(Items[i].Pen);
      ACanvas.Brush.Assign(Items[i].Brush);
      if ((Items[i].NoPts > 0) and
          (Items[i].Visible)) then
      begin
        iXp1 := Items[0].XAxis.FofX(Items[0].XData^[0]);
    {loop over every point in each series:}
        for j := 0 to Items[i].NoPts-2 do
        begin
          iX := iXp1;
          iXp1 := Items[0].XAxis.FofX(Items[0].XData^[j+1]);
          dX := (iXp1-iX) * ((100 - ColumnGap) / (100 * Count));
          iXStart := iX + Round(i*dX);
          iXEnd := iXStart + Round(dX);
          iY := Items[i].YAxis.FofY(Items[i].YData^[j]);
          ACanvas.Rectangle(iXStart, iY, iXEnd, Items[i].YAxis.Bottom);
        end; {for}
      end; {NoPts > 0}
    end;
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawStack
  Description: standard Drawing procedure for Stacked Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Stacked Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawStack(ACanvas: TCanvas; ColumnGap: TPercent);
var
  i, j: Integer;
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXEnd,
  iY,
  iYNeg,
  iYTop,
  iYNegBottom: Integer;
  NegSum, Sum: Single;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit;

{all columns are plotted against the X and Y Axis and X Data of the first series,
 otherwise chaos can result.}
  if (Items[0].NoPts = 0) then exit;

  iXp1 := Items[0].XAxis.FofX(Items[0].XData^[0]);
{loop over every point:}
  for j := 0 to Items[0].NoPts-2 do
  begin
    iX := iXp1;
    iXp1 := Items[0].XAxis.FofX(Items[0].XData^[j+1]);
    iXEnd := iXp1 - Round((ColumnGap) * (iXp1-iX) / 100);
    iYTop := Items[0].XAxis.MidY;
    iYNegBottom := iYTop;
    Sum := 0;
    NegSum := 0;
  {loop over every series:}
    for i := 0 to Self.Count-1 do
    begin
      if ((Items[i].NoPts > j) and
          (Items[i].Visible)) then
      begin
        ACanvas.Pen.Assign(Items[i].Pen);
        ACanvas.Brush.Assign(Items[i].Brush);
        if (Items[i].YData^[j] >= 0) then
        begin
          Sum := Sum + Items[i].YData^[j];
          iY := iYTop;
          iYTop := Items[0].YAxis.FofY(Sum);
          ACanvas.Rectangle(iX, iY, iXEnd, iYTop);
        end
        else
        begin
          NegSum := NegSum + Items[i].YData^[j];
          iYNeg := iYNegBottom;
          iYNegBottom := Items[0].YAxis.FofY(NegSum);
          ACanvas.Rectangle(iX, iYNeg, iXEnd, iYNegBottom);
        end;
      end; {for}
    end; {NoPts > 0}
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawNormStack
  Description: standard Drawing procedure for Normalized Stacked Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Normalized (percentage) Stacked Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawNormStack(ACanvas: TCanvas; ColumnGap: TPercent);
var
  i, j: Integer;
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXEnd,
  iY,
  iYNeg,
  iYTop,
  iYNegBottom: Integer;
  Sum, NegSum,
  Total, NegTotal: Single;
  //pSeries0: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit;

{all columns are plotted against the X and Y Axis and X Data of the first series,
 otherwise chaos can result.}
  if (Items[0].NoPts = 0) then exit;

  iXp1 := Items[0].XAxis.FofX(Items[0].XData^[0]);
{loop over every point:}
  for j := 0 to Items[0].NoPts-2 do
  begin
    iX := iXp1;
    iXp1 := Items[0].XAxis.FofX(Items[0].XData^[j+1]);
    iXEnd := iXp1 - Round((ColumnGap) * (iXp1-iX) / 100);
    iYTop := Items[0].XAxis.MidY;
    iYNegBottom := iYTop;
    Sum := 0;
    NegSum := 0;
    Total := 0;
    NegTotal := 0;
{count every series:}
    for i := 0 to Count-1 do
    begin
      if (Items[i].NoPts > j) then
      begin
        if (Items[i].YData^[j] >= 0) then
          Total := Total + Items[i].YData^[j]
         else
          NegTotal := NegTotal + Items[i].YData^[j];
      end; {NoPts > j}
    end; {count every series}
{prepare for conversion of data to percent:}
    Total := Total / 100;
    NegTotal := - NegTotal / 100;

{loop over every series:}
    for i := 0 to Count-1 do
    begin
      if (Items[i].NoPts > j) then
      begin
        ACanvas.Pen.Assign(Items[i].Pen);
        ACanvas.Brush.Assign(Items[i].Brush);
        if (Items[i].YData^[j] >= 0) then
        begin
          Sum := Sum + (Items[i].YData^[j] / Total);
          iY := iYTop;
          iYTop := Items[0].YAxis.FofY(Sum);
          if (Items[i].Visible) then
            ACanvas.Rectangle(iX, iY, iXEnd, iYTop);
        end
        else
        begin
          NegSum := NegSum + (Items[i].YData^[j] / NegTotal);
          iYNeg := iYNegBottom;
          iYNegBottom := Items[0].YAxis.FofY(NegSum);
          if (Items[i].Visible) then
            ACanvas.Rectangle(iX, iYNeg, iXEnd, iYNegBottom);
        end;
      end; {for}
    end; {NoPts > 0}
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawPie
  Description: standard Drawing procedure for Normalized Stacked Columns
       Author: Mat Ballard
 Date created: 12/20/2000
Date modified: 12/20/2000 by Mat Ballard
      Purpose: This draws all the series on the given canvas as Pie graphs.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawPie(ACanvas: TCanvas; Border: TBorder; NoRows: Integer);
var
  CellWidth,
  CellHeight,
  iSeries,
  iCol,
  jRow,
  NoPieCols,
  PieLeft,
  PieTop,
  PieWidth,
  PieHeight: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawPie: ACanvas is nil !');
{$ENDIF}

  if ((Count = 0) or (NoRows = 0)) then
    exit;

{remember the number of rows:}
  mNoPieRows := NoRows;
  NoPieCols := Trunc(0.99 + Self.Count / mNoPieRows);
{remember the border:}
  mPlotBorder.Left := Border.Left;
  mPlotBorder.Top := Border.Top;
  mPlotBorder.Right := Border.Right;
  mPlotBorder.Bottom := Border.Bottom;

{each Pie sits in a cell:}
  CellWidth := (mPlotBorder.Right - mPlotBorder.Left) div NoPieCols;
  CellHeight := (mPlotBorder.Bottom - mPlotBorder.Top) div mNoPieRows;
{... but does not occupy the entire cell:}
  PieWidth := Round(PIE_SIZE * CellWidth);
  PieHeight := Round(PIE_SIZE * CellHeight);
  if (PieHeight > PieWidth) then
    PieHeight := PieWidth;

  iSeries := 0;
  for iCol := 0 to NoPieCols-1 do
  begin
    for jRow := 0 to mNoPieRows-1 do
    begin
      if (iSeries >= Count) then break;
{indent the (left, top) a bit:}
      PieLeft := mPlotBorder.Left + iCol * CellWidth +
        (CellWidth-PieWidth) div 2;
      PieTop := mPlotBorder.Top + jRow * CellHeight +
        (CellHeight-PieHeight) div 2;
{draw it:}
      TSeries(Self.Items[iSeries]).DrawPie(
        ACanvas,
        PieLeft,
        PieTop,
        PieWidth,
        PieHeight);
      Inc(iSeries);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawPolar
  Description: standard Drawing procedure for Normalized Stacked Columns
       Author: Mat Ballard
 Date created: 12/20/2000
Date modified: 12/20/2000 by Mat Ballard
      Purpose: This draws all the series on the given canvas as a Polar graph.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawPolar(ACanvas: TCanvas; PolarRange: Single);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw: ACanvas is nil !');
{$ENDIF}

  if (PolarRange = 0) then
    PolarRange := TWO_PI;
  for i := 0 to Self.Count-1 do
  begin
    Items[i].DrawPolar(ACanvas, PolarRange);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw3DWire
  Description: standard Drawing procedure for 3D graphs
       Author: Mat Ballard
 Date created: 01/24/2001
Date modified: 01/24/2001 by Mat Ballard
      Purpose: This draws all the series as a 3D graph
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw3DWire(ACanvas: TCanvas; ZLink: Boolean);
var
  iSeries,
  j: Integer;
  Pt00, Pt01, Pt10, Pt11: TPoint;
{These correspond to Point[Series #, Point #]}
  ZShift0,
  ZShift1: TPoint;
  //pSeries0,  pSeries1: TSeries;
  ZAxis: TAxis;
begin
  ZAxis := TPlot(FPlot).ZAxis;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw3DWire: ACanvas is nil !');
  Assert(ZAxis <> nil, 'TSeriesList.Draw3DWire: ZAxis is nil !');
{$ENDIF}

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    ACanvas.Pen.Assign(Items[iSeries].Pen);
    //pSeries0 := TSeries(Items[iSeries]);
    Pt10.x := Low(Integer);
    if (Items[iSeries].NoPts > 0) then
    begin
      ZShift0 := ZAxis.dFofZ(Items[iSeries].ZData);
      Pt00.x := Items[iSeries].XAxis.FofX(Items[iSeries].XData^[0])+ ZShift0.x;
      Pt00.y := Items[iSeries].YAxis.FofY(Items[iSeries].YData^[0]) + ZShift0.y;
      if ((ZLink) and (iSeries < Count-1)) then
      begin
        //pSeries1 := TSeries(Items[iSeries+1]);
        if (Items[iSeries+1].NoPts > 0) then
        begin
          ZShift1 := ZAxis.dFofZ(Items[iSeries+1].ZData);
          Pt10.x := Items[iSeries+1].XAxis.FofX(Items[iSeries+1].XData^[0])+ ZShift1.x;
          Pt10.y := Items[iSeries+1].YAxis.FofY(Items[iSeries+1].YData^[0]) + ZShift1.y;
        end;
      end;

      for j := 1 to Items[iSeries].NoPts-1 do
      begin
        Pt01.x := Items[iSeries].XAxis.FofX(Items[iSeries].XData^[j]) + ZShift0.x;
        Pt01.y := Items[iSeries].YAxis.FofY(Items[iSeries].YData^[j]) + ZShift0.y;
        ACanvas.MoveTo(Pt00.x, Pt00.y);
        ACanvas.LineTo(Pt01.x, Pt01.y);
        if ((ZLink) and (iSeries < Count-1)) then
        begin
{Oh yes it was: Delphi isn't smart enough to see that if we got here,
 then it WAS initialized. Same applies to Pt10.x, Pt10.y.}
          if (Items[iSeries+1].NoPts > 0) then
          begin
            Pt11.x := Items[iSeries+1].XAxis.FofX(Items[iSeries+1].XData^[j]) + ZShift1.x;
            Pt11.y := Items[iSeries+1].YAxis.FofY(Items[iSeries+1].YData^[j]) + ZShift1.y;
            ACanvas.MoveTo(Pt10.x, Pt10.y);
            ACanvas.LineTo(Pt00.x, Pt00.y);
{no triangles please, we're British:
            ACanvas.LineTo(Pt11.x, Pt11.y);}
            Pt10.x := Pt11.x;
            Pt10.y := Pt11.y;
          end;
        end; {not the final series}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
      end; {loop over points}
      if (Pt10.x <> Low(Integer)) then
      begin
        ACanvas.MoveTo(Pt00.x, Pt00.y);
        ACanvas.LineTo(Pt10.x, Pt10.y);
      end;
    end; {NoPts}
  end; {over every series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw3DColumn
  Description: standard Drawing procedure for 3D graphs
       Author: Mat Ballard
 Date created: 06/01/2001
Date modified: 06/01/2001 by Mat Ballard
      Purpose: This draws all the series on the given canvas as a 3D Columns.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw3DColumn(ACanvas: TCanvas; ColumnGap: TPercent);
var
  iSeries,
  j: Integer;
  iX, iXp1, FofX, Height, dX: Integer;
  FofZ, FofZp1, dz: TPoint;
  ZAxis: TAxis;

{   NB: FofZ and dz are vectors in TPoint format.
                y
                ^
                |
                |
                |               FofX
                |                |
                |----------------/------------> x
               /                /
              /        4_______/5 ___
             /         /.    . /|  ^
            /         / .   . / |
           /         /_______/  |  Height
          /          7  . .  6  |
         /           |  ..   |  |
  FofZ _/            |  0....|..1 _v___
       /             | .     | /     dz
      z              3_______2/_____/
                     |       |
                     |<  dX >|            }
  procedure DrawCol(FofX, Height, dX: Integer; FofZ, dz: TPoint);
  var
    Vertex: array[0..7] of TPoint;
  begin
{calculate the positions of the six required vertices in the above diagram:}
    Vertex[0].x := FofX + FofZ.x;
    Vertex[4].x := Vertex[0].x;
    Vertex[3].x := Vertex[4].x + dz.x;
    Vertex[7].x := Vertex[3].x;
    Vertex[2].x := Vertex[3].x + dX;
    Vertex[6].x := Vertex[2].x;
    Vertex[1].x := Vertex[4].x + dX;
    Vertex[5].x := Vertex[1].x;

    Vertex[0].y := TPlot(FPlot).Axes[1].Bottom + FofZ.y; // TPlot(FPlot).Axes TPlot(FPlot).Axes
    Vertex[1].y := Vertex[0].y;
    Vertex[2].y := Vertex[1].y + dz.y;
    Vertex[3].y := Vertex[2].y;
    Vertex[4].y := Vertex[0].y + Height;
    Vertex[5].y := Vertex[4].y;
    Vertex[6].y := Vertex[2].y + Height;
    Vertex[7].y := Vertex[6].y;
{The rectangular front face:}
    ACanvas.Rectangle(Vertex[7].x, Vertex[7].y, Vertex[2].x, Vertex[2].y);
{Make it a little darker:}
    ACanvas.Brush.Color := GetDarkerColor(ACanvas.Brush.Color, 80);
{Draw the top parallelogram:}
    ACanvas.Polygon([Vertex[6], Vertex[7], Vertex[4], Vertex[5]]);
{Make it a little darker again:}
    ACanvas.Brush.Color := GetDarkerColor(ACanvas.Brush.Color, 70);
{Draw the right or left parallelogram:}
    if (dz.x > 0) then
      ACanvas.Polygon([Vertex[0], Vertex[3], Vertex[7], Vertex[4]]) {left}
     else
      ACanvas.Polygon([Vertex[1], Vertex[2], Vertex[6], Vertex[5]]); {right}
  end;

begin
  ZAxis := TPlot(FPlot).ZAxis;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw3DColumn: ACanvas is nil !');
  Assert(ZAxis <> nil, 'TSeriesList.Draw3DColumn: ZAxis is nil !');
{$ENDIF}

{Draw each individual series:}
{NB: the ZData values better be in ascending order !}
{set dz in case there is only one series !}
  dz := ZAxis.dFofZ(1);
  dX := 0;
  for iSeries := 0 to Self.Count-1 do
  begin
    if ((Items[iSeries].NoPts > 0) and
        (Items[iSeries].Visible)) then
    begin
      ACanvas.Pen.Assign(Items[iSeries].Pen);
      ACanvas.Brush.Assign(Items[iSeries].Brush);
{the Z co-ordinates are the same for every point in the series:}
      FofZ := ZAxis.dFofZ(Items[iSeries].ZData);
      if (iSeries < Self.Count-1) then
      begin
        FofZp1 := ZAxis.dFofZ(TSeries(Items[iSeries+1]).ZData);
        dz.x := (FofZp1.x - FofZ.x) * (100 - ColumnGap) div 100;
        dz.y := (FofZp1.y - FofZ.y) * (100 - ColumnGap) div 100;
      end;
{the starting X Point is:}
      iXp1 := Items[iSeries].XAxis.FofX(Items[iSeries].XData^[0]);
{loop over every point in each series:}
      for j := 0 to Items[iSeries].NoPts-1 do
      begin
        ACanvas.Brush.Color := Items[iSeries].Brush.Color;
        iX := iXp1;
        if (j < Items[iSeries].NoPts-1) then
        begin // we recalculate the gap based on the next point:
          iXp1 := Items[iSeries].XAxis.FofX(Items[iSeries].XData^[j+1]);
          dX := (iXp1-iX) * (100 - ColumnGap) div 100;
        end;
        FofX := Items[iSeries].XAxis.FofX(Items[iSeries].XData^[j]);
        Height := Items[iSeries].YAxis.FofY(Items[iSeries].YData^[j]) - Items[iSeries].YAxis.Bottom;// TPlot(FPlot).Axes[0].MidY;
        DrawCol(FofX, Height, dX, FofZ, dz);
      end; {for j}
    end; {NoPts > 0 & Visible}
  end; {for iSeries}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw3DContour
  Description: standard Drawing procedure for 3D contour graphs
       Author: Mat Ballard
 Date created: 03/06/2001
Date modified: 03/06/2001 by Mat Ballard
      Purpose: This draws all the series as a 3D contour graph
 Known Issues: Minor glitch that results in small "holes" - reason unknown
      Comment: Note that the paradigm is Y = F(X, Z).
               That is, X and Z are the independent variables, and Y is the function.
               Colour is based on Y Values, not Z.
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw3DContour(ACanvas: TCanvas; Contour: TContour);
var
  i, iSeries: Integer;
  ZFraction,
  TheMin,
  Span: Single;
{The original 3D data points:}
  Pt00, Pt01, Pt10, Pt11, PtCentre: T3DRealPoint;
{The transformed 2D screen points:}
  Q00, Q10, Q11, Q01, QCentre: TPoint;
  //pSeries0, pSeries1: TSeries;
  ZAxis: TAxis;

{Transform the 3D point to the 2D drawing surface:}
  function Trans3D2(APoint: T3DRealPoint): TPoint;
  var
    ZShift: TPoint;
  begin
    ZShift := ZAxis.dFofZ(APoint.Z);
    Result.x := TPlot(FPlot).Axes[0].FofX(APoint.X) + ZShift.x;
    Result.y := TPlot(FPlot).Axes[1].FofY(APoint.Y) + ZShift.y;
  end;

{This procedure performs a simple, single color rendition of a triangle:}
  procedure BltTriangle(Q00, Q10, Q11: TPoint; ZValue: Single);
  begin
    ACanvas.Pen.Color := Misc.Rainbow(ZValue, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
    ACanvas.Brush.Color := ACanvas.Pen.Color;
    ACanvas.Polygon([Q00, Q10, Q11]);
  end;

{This procedure performs a complex, three-color graded rendition of a triangle:}
  procedure RenderTriangle(Pt1, Pt2, Pt3: T3DRealPoint);
  var
    ii, nY: Integer;
    p1, p2, p3: p3DRealPoint; //Highest, medium, lowest points in Z direction
{Sides of the triangle:}
    P1P2, P1P3, P2P3,
{Pj is the left side of the triangle, Pk the right:}
    Pj, Pk, PjM1, PkM1: T3DRealPoint; //PjM1 = "P(j - 1)"
    dY, dxj, dzj, dxk, dzk: Single;
  begin
{sort the bastards first, on _Y_ values:}
    SortTriangleVerticesOnY(Pt1, Pt2, Pt3, p1, p2, p3);

{create the line vectors of the sides of the triangle:}
    P1P2.X := p2^.X - p1^.X;
    P1P2.Y := p2^.Y - p1^.Y;
    P1P2.Z := p2^.Z - p1^.Z;
    P1P3.X := p3^.X - p1^.X;
    P1P3.Y := p3^.Y - p1^.Y;
    P1P3.Z := p3^.Z - p1^.Z;
    P2P3.X := p3^.X - p2^.X;
    P2P3.Y := p3^.Y - p2^.Y;
    P2P3.Z := p3^.Z - p2^.Z;

{Get ready to loop: set the Pj and Pk to p1, the highest point:}
    Pj.X := p1^.X;
    Pj.Y := p1^.Y;
    Pj.Z := p1^.Z;
    Pk.X := p1^.X;
    Pk.Y := p1^.Y;
    Pk.Z := p1^.Z;
    Q10 := Trans3D2(Pj);
    Q11 := Q10;

{Estimate a colour granularity:}
    dY := Span / COLOUR_GRANULARITY;
    nY := Abs(Round(P1P3.Y / dY));
    if (nY < 2) then
      nY := 2;
    dY := P1P2.Y / nY;
    dxj := P1P2.X / nY;
    dzj := P1P2.Z / nY;
    dxk := P1P3.X * (dY / P1P3.Y);
    dzk := P1P3.Z * (dY / P1P3.Y);

    for ii := 1 to nY do
    begin
{Update last points:}
      PjM1.X := Pj.X;
      PjM1.Y := Pj.Y;
      PjM1.Z := Pj.Z;
      PkM1.X := Pk.X;
      PkM1.Y := Pk.Y;
      PkM1.Z := Pk.Z;
      Q00 := Q10;
      Q01 := Q11;
{move so that colour changes by dY:}
      Pj.Y := Pj.Y + dY;
      Pj.X := p1^.X + ii * dxj;
      Pj.Z := p1^.Z + ii * dzj;
      Pk.Y := Pj.Y;
      Pk.X := p1^.X + ii * dxk;
      Pk.Z := p1^.Z + ii * dzk;

{Draw:}
      ZFraction := (PjM1.Y - TheMin) / Span;
      ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
      ACanvas.Brush.Color := ACanvas.Pen.Color;
      Q10 := Trans3D2(Pj);
      Q11 := Trans3D2(Pk);
      if ((Q10.x <> Q00.x) or (Q10.y <> Q00.y) or
          (Q11.x <> Q01.x) or (Q11.y <> Q01.y)) then
        ACanvas.Polygon([Q00, Q01, Q11, Q10]);
    end; {for}

{Estimate a colour granularity:}
    dY := Span / COLOUR_GRANULARITY;
    nY := Abs(Round(P2P3.Y / dY));
    if (nY < 2) then
      nY := 2;
    dY := P2P3.Y / nY;
    dxj := P2P3.X / nY;
    dzj := P2P3.Z / nY;

    for ii := 1 to nY do
    begin
{Update last points:}
      PjM1.X := Pj.X;
      PjM1.Y := Pj.Y;
      PjM1.Z := Pj.Z;
      PkM1.X := Pk.X;
      PkM1.Y := Pk.Y;
      PkM1.Z := Pk.Z;
      Q00 := Q10;
      Q01 := Q11;
{move so that colour changes by dY:}
      Pj.Y := Pj.Y + dY;
      Pj.X := p2^.X + ii * dxj;
      Pj.Z := p2^.Z + ii * dzj;
{Calculate position of same Z on long right side:}
      Pk.Y := Pj.Y;
      ZFraction := (Pk.Y-p1^.Y)/(p3^.Y-p1^.Y);
      Pk.X := p1^.X + ZFraction * P1P3.X;
      Pk.Z := p1^.Z + ZFraction * P1P3.Z;
{Draw:}
      ZFraction := (PjM1.Y - TheMin) / Span;
      ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
      ACanvas.Brush.Color := ACanvas.Pen.Color;
      Q10 := Trans3D2(Pj);
      Q11 := Trans3D2(Pk);
      if ((Q10.x <> Q00.x) or (Q10.y <> Q00.y) or
          (Q11.x <> Q01.x) or (Q11.y <> Q01.y)) then
        ACanvas.Polygon([Q00, Q01, Q11, Q10]);
    end; {for}
  end;

begin
  ZAxis := TPlot(FPlot).ZAxis;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw3DContour: ACanvas is nil !');
  Assert(ZAxis <> nil, 'TSeriesList.Draw3DContour: ZAxis is nil !');
{$ENDIF}

  TheMin := Self.Ymin;
  Span := Self.YMax - TheMin;
  ACanvas.Brush.Style := bsSolid;
  //pSeries1 := nil;

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    //pSeries0 := TSeries(Items[iSeries]);
    if (Items[iSeries].NoPts > 0) then
    begin
{Assign the bottom-left point:}
      Pt00.X := Items[iSeries].XData^[0];
      Pt00.Y := Items[iSeries].YData^[0];
      Pt00.Z := Items[iSeries].ZData;
      if (iSeries < Count-1) then
      begin
        if (Items[iSeries+1].NoPts > 0) then
        begin
{Assign the top-left point:}
          Pt10.X := Items[iSeries+1].XData^[0];
          Pt10.Y := Items[iSeries+1].YData^[0];
          Pt10.Z := Items[iSeries+1].ZData;
        end;
      end;

      for i := 1 to Items[iSeries].NoPts-1 do
      begin
{Assign the bottom-right point:}
        Pt01.X := Items[iSeries].XData^[i];
        Pt01.Y := Items[iSeries].YData^[i];
        Pt01.Z := Items[iSeries].ZData;
{Oh yes it was: Delphi ain't that smart.}
        if ((iSeries < Count-1) and (Items[iSeries+1].NoPts > 0)) then
        begin
{Assign the top-right point:}
          Pt11.X := Items[iSeries+1].XData^[i];
          Pt11.Y := Items[iSeries+1].YData^[i];
          Pt11.Z := Items[iSeries+1].ZData;

{Calculate the centrum:}
          PtCentre.X := (Pt00.X + Pt01.X + Pt10.X + Pt11.X) / 4;
          PtCentre.Y := (Pt00.Y + Pt01.Y + Pt10.Y + Pt11.Y) / 4;
{Oh yes it was: see above}
          PtCentre.Z := (Pt00.Z + Pt01.Z + Pt10.Z + Pt11.Z) / 4;

{just how detailed will the plot be ?}
          case Contour.Detail of
            cdLow:
              begin
{No triangles; base colour on the Y Value:}
                ZFraction := (PtCentre.Y - TheMin) / Span;
                ACanvas.Brush.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
                if (Contour.WireFrame) then
                  ACanvas.Pen.Color := clBlack
                 else
                  ACanvas.Pen.Color := ACanvas.Brush.Color;
                ACanvas.Polygon([
                  Trans3D2(Pt00),
                  Trans3D2(Pt10),
                  Trans3D2(Pt11),
                  Trans3D2(Pt01)]);
              end;

            cdMedium:
              begin
                Q00 := Trans3D2(Pt00);
                Q10 := Trans3D2(Pt10);
                Q11 := Trans3D2(Pt11);
                Q01 := Trans3D2(Pt01);
                QCentre := Trans3D2(PtCentre);
{Left triangle, then Bottom, Top then Right:}
                ZFraction := ((Pt00.Y + Pt10.Y + PtCentre.Y)/3-TheMin) / Span;
                BltTriangle(Q00, Q10, QCentre, ZFraction);
                ZFraction := ((Pt00.Y + Pt01.Y + PtCentre.Y)/3-TheMin) / Span;
                BltTriangle(Q00, Q01, QCentre, ZFraction);
                ZFraction := ((Pt10.Y + Pt11.Y + PtCentre.Y)/3-TheMin) / Span;
                BltTriangle(Q10, Q11, QCentre, ZFraction);
                ZFraction := ((Pt01.Y + Pt11.Y + PtCentre.Y)/3-TheMin) / Span;
                BltTriangle(Q01, Q11, QCentre, ZFraction);
                if (Contour.WireFrame) then
                begin
                  ACanvas.Pen.Color := clBlack;
                  ACanvas.PolyLine([
                    Trans3D2(Pt00),
                    Trans3D2(Pt10),
                    Trans3D2(Pt11),
                    Trans3D2(Pt01),
                    Trans3D2(Pt00)]);
                end;
              end;

            cdHigh:
              begin
{Left triangle, then Bottom, Top then Right:}
                RenderTriangle(Pt00, Pt10, PtCentre);
                RenderTriangle(Pt00, Pt01, PtCentre);
                RenderTriangle(Pt10, Pt11, PtCentre);
                RenderTriangle(Pt01, Pt11, PtCentre);
                if (Contour.WireFrame) then
                begin
                  ACanvas.Pen.Color := clBlack;
                  ACanvas.PolyLine([
                    Trans3D2(Pt00),
                    Trans3D2(Pt10),
                    Trans3D2(Pt11),
                    Trans3D2(Pt01),
                    Trans3D2(Pt00)]);
                end;
              end;
          end; {case}

{update values:}
          Pt10.x := Pt11.x;
          Pt10.y := Pt11.y;
          Pt10.Z := Pt11.Z;
        end; {not the final series}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
        Pt00.Z := Pt01.Z;
      end; {loop over points}
    end; {NoPts}
  end; {over every series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawContour
  Description: standard Drawing procedure for 2.5D contour graphs
       Author: Mat Ballard
 Date created: 01/24/2001
Date modified: 07/15/2001 by Mat Ballard
      Purpose: This draws all the series as a 2.5D coloured contour graph
 Known Issues: Note that Y data is plotted as the color, and the Z data
               as screen Y
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawContour(ACanvas: TCanvas; Contour: TContour);
var
  i, iSeries: Integer;
  ZFraction,
  TheMin,
  Span: Single;
  Pt00, Pt01, Pt10, Pt11, PtCentre: T3DZPoint;
  pSeries0,
  pSeries1: TSeries;

{This procedure performs a simple, single color rendition of a triangle:}
  procedure BltTriangle(Pt1, Pt2, Pt3: T3DZPoint);
  begin
    ZFraction := ((Pt1.Z + Pt2.Z + Pt3.Z)/3-TheMin) / Span;
    ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
    ACanvas.Brush.Color := ACanvas.Pen.Color;
    ACanvas.Polygon([Point(Pt1.x,Pt1.y), Point(Pt2.x,Pt2.y), Point(Pt3.x,Pt3.y)]);
  end;

{This procedure performs a complex, three-color graded rendition of a triangle:}
  procedure RenderTriangle(Pt1, Pt2, Pt3: T3DZPoint);
  var
    ii, nZ: Integer;
    p1, p2, p3: p3DZPoint; //Highest, medium, lowest points in Z direction
{Sides of the triangle:}
    P1P2, P1P3, P2P3,
{Pj is the left side of the triangle, Pk the right:}
    Pj, Pk, PjM1, PkM1: T3DZPoint; //PjM1 = "P(j - 1)"
    dZ, dxj, dyj, dxk, dyk: Single;
  begin
{sort the bastards first, on Z values:}
    SortTriangleVertices(Pt1, Pt2, Pt3, p1, p2, p3);

{create the line vectors of the sides of the triangle:}
    P1P2.x := p2^.x - p1^.x;
    P1P2.y := p2^.y - p1^.y;
    P1P2.Z := p2^.Z - p1^.Z;
    P1P3.x := p3^.x - p1^.x;
    P1P3.y := p3^.y - p1^.y;
    P1P3.Z := p3^.Z - p1^.Z;
    P2P3.x := p3^.x - p2^.x;
    P2P3.y := p3^.y - p2^.y;
    P2P3.Z := p3^.Z - p2^.Z;

{Get ready to loop: set the Pj and Pk to p1, the highest point:}
    Pj.x := p1^.x;
    Pj.y := p1^.y;
    Pj.Z := p1^.Z;
    Pk.x := p1^.x;
    Pk.y := p1^.y;
    Pk.Z := p1^.Z;

{Draw the apex:}
    //ZFraction := (p1^.Z - TheMin) / Span;
{$IFDEF MSWINDOWS}
    //ACanvas.Pixels[p1^.x, p1^.y] := Misc.Rainbow(ZFraction);
{$ENDIF}
{$IFDEF LINUX}
    //ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
    //ACanvas.DrawPoint(p1^.x, p1^.y);
{$ENDIF}

{Estimate a colour granularity:}
    dZ := Span / COLOUR_GRANULARITY;
    nZ := Abs(Round(P1P3.Z / dZ));
    if (nZ < 2) then
      nZ := 2;
    dZ := P1P2.Z / nZ;
    dxj := P1P2.x / nZ;
    dyj := P1P2.y / nZ;
    dxk := P1P3.x * (dZ / P1P3.Z);
    dyk := P1P3.y * (dZ / P1P3.Z);

    for ii := 1 to nZ do
    begin
{Update last points:}
      PjM1.x := Pj.x;
      PjM1.y := Pj.y;
      PjM1.Z := Pj.Z;
      PkM1.x := Pk.x;
      PkM1.y := Pk.y;
      PkM1.Z := Pk.Z;
{move so that colour changes by dZ:}
      Pj.Z := Pj.Z + dZ;
      Pj.x := p1^.x + Round(ii * dxj);
      Pj.y := p1^.y + Round(ii * dyj);
      Pk.Z := Pj.Z; //Pk.Z + dZ;
      Pk.x := p1^.x + Round(ii * dxk);
      Pk.y := p1^.y + Round(ii * dyk);
{Draw:}
      ZFraction := (PjM1.Z - TheMin) / Span;
      ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
      ACanvas.Brush.Color := ACanvas.Pen.Color;
      if ((Pj.x <> PjM1.x) or (Pj.y <> PjM1.y) or
          (Pk.x <> PkM1.x) or (Pk.y <> PkM1.y)) then
        ACanvas.Polygon([
          Point(PjM1.x,PjM1.y),
          Point(PkM1.x,PkM1.y),
          Point(Pk.x,Pk.y),
          Point(Pj.x,Pj.y)]);
    end; {for}

{Estimate a colour granularity:}
    dZ := Span / COLOUR_GRANULARITY;
    nZ := Abs(Round(P2P3.Z / dZ));
    if (nZ < 2) then
      nZ := 2;
    dZ := P2P3.Z / nZ;
    dxj := P2P3.x / nZ;
    dyj := P2P3.y / nZ;

    for ii := 1 to nZ do
    begin
{Update last points:}
      PjM1.x := Pj.x;
      PjM1.y := Pj.y;
      PjM1.Z := Pj.Z;
      PkM1.x := Pk.x;
      PkM1.y := Pk.y;
      PkM1.Z := Pk.Z;
{move so that colour changes by dZ:}
      Pj.Z := Pj.Z + dZ;
      Pj.x := p2^.x + Round(ii * dxj);
      Pj.y := p2^.y + Round(ii * dyj);
      Pk.Z := Pj.Z;
{Calculate position of same Z on long right side:}
      ZFraction := (Pk.Z-p1^.Z)/(p3^.Z-p1^.Z);
      Pk.x := p1^.x + Round(ZFraction * P1P3.x);
      Pk.y := p1^.y + Round(ZFraction * P1P3.y);
{Draw:}
      ZFraction := (PjM1.Z - TheMin) / Span;
      ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
      ACanvas.Brush.Color := ACanvas.Pen.Color;
      if ((Pj.x <> PjM1.x) or (Pj.y <> PjM1.y) or
          (Pk.x <> PkM1.x) or (Pk.y <> PkM1.y)) then
        ACanvas.Polygon([
          Point(PjM1.x,PjM1.y),
          Point(PkM1.x,PkM1.y),
          Point(Pk.x,Pk.y),
          Point(Pj.x,Pj.y)]);
    end; {for}
  end;

begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawContour: ACanvas is nil !');
{$ENDIF}

  TheMin := Self.Ymin;
  Span := Self.YMax - TheMin;
  ACanvas.Brush.Style := bsSolid;
  pSeries1 := nil;

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    pSeries0 := TSeries(Items[iSeries]);
    if (pSeries0.NoPts > 0) then
    begin
{Assign the bottom-left point:}
      Pt00.x := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
{Note: for contours, we plot the Y data as a function of X and Z,
 so that Y determines the color, and Z determines the screen Y value.
 The reason for this is that most data will be in the form of scans (series)
 at repeated times.}
      Pt00.y := pSeries0.YAxis.FofY(pSeries0.ZData);
      Pt00.Z := pSeries0.YData^[0];
      if (iSeries < Count-1) then
      begin
        pSeries1 := TSeries(Items[iSeries+1]);
        if (pSeries1.NoPts > 0) then
        begin
{Assign the top-left point:}
          Pt10.x := pSeries1.XAxis.FofX(pSeries1.XData^[0]);
          Pt10.y := pSeries1.YAxis.FofY(pSeries1.ZData);
          Pt10.Z := pSeries1.YData^[0];
        end;
      end;

      for i := 1 to pSeries0.NoPts-1 do
      begin
{Assign the bottom-right point:}
        Pt01.x := pSeries0.XAxis.FofX(pSeries0.XData^[i]);
        Pt01.y := pSeries0.YAxis.FofY(pSeries0.ZData);
        Pt01.Z := pSeries0.YData^[i];
{Oh yes it was: Delphi ain't that smart.}
        if ((iSeries < Count-1) and (pSeries1.NoPts > 0)) then
        begin
{Assign the top-right point:}
          Pt11.x := pSeries1.XAxis.FofX(pSeries1.XData^[i]);
          Pt11.y := pSeries1.YAxis.FofY(pSeries1.ZData);
          Pt11.Z := pSeries1.YData^[i];

{Calculate the centrum:}
          PtCentre.x := (Pt00.x + Pt01.x + Pt10.x + Pt11.x) div 4;
          PtCentre.y := (Pt00.y + Pt01.y + Pt10.y + Pt11.y) div 4;
{Oh yes it was: see above}
          PtCentre.Z := (Pt00.Z + Pt01.Z + Pt10.Z + Pt11.Z) / 4;

{just how detailed will the plot be ?}
          case Contour.Detail of
            cdLow:
              begin
{No triangles:}
                ZFraction := (PtCentre.Z - TheMin) / Span;
                ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
                ACanvas.Brush.Color := ACanvas.Pen.Color;
                ACanvas.Polygon([
                  Point(Pt00.x,Pt00.y),
                  Point(Pt10.x,Pt10.y),
                  Point(Pt11.x,Pt11.y),
                  Point(Pt01.x,Pt01.y)]);
              end;

            cdMedium:
              begin
{Left triangle, then Bottom, Top then Right:}
                BltTriangle(Pt00, Pt10, PtCentre);
                BltTriangle(Pt00, Pt01, PtCentre);
                BltTriangle(Pt10, Pt11, PtCentre);
                BltTriangle(Pt01, Pt11, PtCentre);
              end;

            cdHigh:
              begin
{Left triangle, then Bottom, Top then Right:}
                RenderTriangle(Pt00, Pt10, PtCentre);
                RenderTriangle(Pt00, Pt01, PtCentre);
                RenderTriangle(Pt10, Pt11, PtCentre);
                RenderTriangle(Pt01, Pt11, PtCentre);
              end;
          end; {case}

{update values:}
          Pt10.x := Pt11.x;
          Pt10.y := Pt11.y;
          Pt10.Z := Pt11.Z;
        end; {not the final series}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
        Pt00.Z := Pt01.Z;
      end; {loop over points}
      {ACanvas.MoveTo(Pt00.x, Pt00.y);
      ACanvas.LineTo(Pt10.x, Pt10.y);}
    end; {NoPts}
  end; {over every series}

{draw the color scale:}
  DrawColorScale(ACanvas, TheMin, Span, Contour);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.SortTriangleVertices
  Description: sorts the vertices of a triangle from highest Z value to lowest
       Author: Mat Ballard
 Date created: 07/17/2001
Date modified: 07/17/2001 by Mat Ballard
      Purpose: 3D rendering
 Known Issues: Pti have to be var, so that the location is passed.
 ------------------------------------------------------------------------------}
procedure TSeriesList.SortTriangleVertices(var Pt1, Pt2, Pt3: T3DZPoint; var p1, p2, p3: p3DZPoint);
begin
  if (Pt1.Z >= Pt2.Z) then
  begin
    if (Pt2.Z >= Pt3.Z) then
    begin
      p1 := @Pt1;
      p2 := @Pt2;
      p3 := @Pt3;
    end
    else
    begin {2 < 3:}
      if (Pt1.Z >= Pt3.Z) then
      begin
        p1 := @Pt1;
        p2 := @Pt3;
        p3 := @Pt2;
      end
      else
      begin
        p1 := @Pt3;
        p2 := @Pt1;
        p3 := @Pt2;
      end;
    end;
  end
  else
  begin {1 < 2}
    if (Pt2.Z < Pt3.Z) then
    begin
      p1 := @Pt3;
      p2 := @Pt2;
      p3 := @Pt1;
    end
    else
    begin {1 < 2 >= 3:}
      if (Pt1.Z >= Pt3.Z) then
      begin
        p1 := @Pt2;
        p2 := @Pt1;
        p3 := @Pt3;
      end
      else
      begin
        p1 := @Pt2;
        p2 := @Pt3;
        p3 := @Pt1;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.SortTriangleVerticesOnY
  Description: sorts the vertices of a triangle from highest Y value to lowest
       Author: Mat Ballard
 Date created: 07/17/2001
Date modified: 07/17/2001 by Mat Ballard
      Purpose: 3D rendering
 Known Issues: Pti have to be var, so that the location is passed.
 ------------------------------------------------------------------------------}
procedure TSeriesList.SortTriangleVerticesOnY(var Pt1, Pt2, Pt3: T3DRealPoint; var p1, p2, p3: p3DRealPoint);
begin
  if (Pt1.Y >= Pt2.Y) then
  begin
    if (Pt2.Y >= Pt3.Y) then
    begin
      p1 := @Pt1;
      p2 := @Pt2;
      p3 := @Pt3;
    end
    else
    begin {2 < 3:}
      if (Pt1.Y >= Pt3.Y) then
      begin
        p1 := @Pt1;
        p2 := @Pt3;
        p3 := @Pt2;
      end
      else
      begin
        p1 := @Pt3;
        p2 := @Pt1;
        p3 := @Pt2;
      end;
    end;
  end
  else
  begin {1 < 2}
    if (Pt2.Y < Pt3.Y) then
    begin
      p1 := @Pt3;
      p2 := @Pt2;
      p3 := @Pt1;
    end
    else
    begin {1 < 2 >= 3:}
      if (Pt1.Y >= Pt3.Y) then
      begin
        p1 := @Pt2;
        p2 := @Pt1;
        p3 := @Pt3;
      end
      else
      begin
        p1 := @Pt2;
        p2 := @Pt3;
        p3 := @Pt1;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawLineContour
  Description: standard Drawing procedure for 2.5D contour graphs
       Author: Mat Ballard
 Date created: 01/24/2001
Date modified: 07/15/2001 by Mat Ballard
      Purpose: This draws all the series as a 2.5D coloured contour graph
 Known Issues: Note that Y data is plotted as the color, and the Z data
               as screen Y
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawLineContour(ACanvas: TCanvas; Contour: TContour);
var
  i, iSeries: Integer;
  ZFraction,
  TheMin,
  Span: Single;
  Pt00, Pt01, Pt10, Pt11, PtCentre: T3DZPoint;
  pSeries0,
  pSeries1: TSeries;

{This procedure performs a complex, three-color graded rendition of a triangle:}
  procedure RenderTriangle(Pt1, Pt2, Pt3: T3DZPoint);
  var
    ii, mZ, nZ: Integer;
    p1, p2, p3: p3DZPoint; //Highest, medium, lowest points in Z direction
{Sides of the triangle:}
    P1P2, P1P3, P2P3,
{Pj is the left side of the triangle, Pk the right:}
    Pj, Pk: T3DZPoint; //PjM1 = "P(j - 1)"
    //dZ, dxj, dyj, dxk, dyk: Single;
  begin
{sort the bastards first, on Z values:}
    SortTriangleVertices(Pt1, Pt2, Pt3, p1, p2, p3);

{create the line vectors of the sides of the triangle:}
    P1P2.x := p2^.x - p1^.x;
    P1P2.y := p2^.y - p1^.y;
    P1P2.Z := p2^.Z - p1^.Z;
    P1P3.x := p3^.x - p1^.x;
    P1P3.y := p3^.y - p1^.y;
    P1P3.Z := p3^.Z - p1^.Z;
    P2P3.x := p3^.x - p2^.x;
    P2P3.y := p3^.y - p2^.y;
    P2P3.Z := p3^.Z - p2^.Z;

{Get ready to loop: }
    nZ := Round((p1^.Z - Contour.Start) / Contour.Interval);
    mZ := Round((p3^.Z - Contour.Start) / Contour.Interval);

    for ii := mZ to nZ do
    begin
      Pk.Z := Contour.Start + ii * Contour.Interval;
      if ((p3^.Z <= Pk.Z) and (Pk.Z <= p1^.Z)) then
      begin
{Calculate position of same Z on long right side:}
        ZFraction := (Pk.Z-p1^.Z) / P1P3.Z;
        Pk.x := p1^.x + Round(ZFraction * P1P3.x);
        Pk.y := p1^.y + Round(ZFraction * P1P3.y);
{Calculate position of same Z on left or bottom side:}
        Pj.Z := Pk.Z;
        if (Pj.Z >= p2^.Z) then
        begin {Left, upper half-triangle}
          ZFraction := (Pj.Z-p1^.Z) / P1P2.Z;
          Pj.x := p1^.x + Round(ZFraction * P1P2.x);
          Pj.y := p1^.y + Round(ZFraction * P1P2.y);
        end
        else
        begin {Bottom, lower half-triangle}
          ZFraction := (Pj.Z-p2^.Z) / P2P3.Z;
          Pj.x := p2^.x + Round(ZFraction * P2P3.x);
          Pj.y := p2^.y + Round(ZFraction * P2P3.y);
        end;
{Draw:}
        if (Contour.Detail > cdMedium) then
        begin
          ZFraction := (Pj.Z - TheMin) / Span;
          ACanvas.Pen.Color := Misc.Rainbow(ZFraction, Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
        end;
        ACanvas.MoveTo(Pj.x,Pj.y);
        ACanvas.LineTo(Pk.x,Pk.y);
      end; {is in Z range}
    end; {for}
  end; {procedure RenderTriangle}

begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawLineContour: ACanvas is nil !');
{$ENDIF}

  pSeries1 := nil;
  TheMin := Self.YMin;
  Span := Self.YMax - Contour.Start;
  //ACanvas.Brush.Style := bsSolid;
  if (Contour.Detail = cdLow) then
    ACanvas.Pen.Color := clBlack;
  if (Contour.Detail = cdMedium) then
    ACanvas.Pen.Color := Items[0].Pen.Color;

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    pSeries0 := TSeries(Items[iSeries]);
    if (pSeries0.NoPts > 0) then
    begin
{Assign the bottom-left point:}
      Pt00.x := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
{Note: for contours, we plot the Y data as a function of X and Z,
 so that Y determines the color, and Z determines the screen Y value.
 The reason for this is that most data will be in the form of scans (series)
 at repeated times.}
      Pt00.y := pSeries0.YAxis.FofY(pSeries0.ZData);
      Pt00.Z := pSeries0.YData^[0];
      if (iSeries < Count-1) then
      begin
        pSeries1 := TSeries(Items[iSeries+1]);
        if (pSeries1.NoPts > 0) then
        begin
{Assign the top-left point:}
          Pt10.x := pSeries1.XAxis.FofX(pSeries1.XData^[0]);
          Pt10.y := pSeries1.YAxis.FofY(pSeries1.ZData);
          Pt10.Z := pSeries1.YData^[0];
        end;
      end;

      for i := 1 to pSeries0.NoPts-1 do
      begin
{Assign the bottom-right point:}
        Pt01.x := pSeries0.XAxis.FofX(pSeries0.XData^[i]);
        Pt01.y := pSeries0.YAxis.FofY(pSeries0.ZData);
        Pt01.Z := pSeries0.YData^[i];
{Oh yes it was: Delphi ain't that smart.}
        if ((iSeries < Count-1) and (pSeries1.NoPts > 0)) then
        begin
{Assign the top-right point:}
          Pt11.x := pSeries1.XAxis.FofX(pSeries1.XData^[i]);
          Pt11.y := pSeries1.YAxis.FofY(pSeries1.ZData);
          Pt11.Z := pSeries1.YData^[i];

{Calculate the centrum:}
          PtCentre.x := (Pt00.x + Pt01.x + Pt10.x + Pt11.x) div 4;
          PtCentre.y := (Pt00.y + Pt01.y + Pt10.y + Pt11.y) div 4;
{Oh yes it was: see above}
          PtCentre.Z := (Pt00.Z + Pt01.Z + Pt10.Z + Pt11.Z) / 4;

{just how detailed will the plot be ?}
          RenderTriangle(Pt00, Pt10, PtCentre);
          RenderTriangle(Pt00, Pt01, PtCentre);
          RenderTriangle(Pt10, Pt11, PtCentre);
          RenderTriangle(Pt01, Pt11, PtCentre);

{update values:}
          Pt10.x := Pt11.x;
          Pt10.y := Pt11.y;
          Pt10.Z := Pt11.Z;
        end; {not the final series}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
        Pt00.Z := Pt01.Z;
      end; {loop over points}
    end; {NoPts}
  end; {over every series}

{draw the color scale:}
  if (Contour.Detail > cdMedium) then
  begin
    ACanvas.Brush.Style := bsSolid;
    DrawColorScale(ACanvas, TheMin, Span, Contour);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawColorScale
  Description: draws the Color Scale for Contour plots
       Author: Mat Ballard
 Date created: 03/06/2001
Date modified: 03/06/2001 by Mat Ballard
      Purpose: contour plot details
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawColorScale(ACanvas: TCanvas; TheMin, Span: Single;
  Contour: TContour);
var
  i,
  iX, iXp1, iY,
  FontHeight: Integer;
  TheText: String;
begin
  iX := TPlot(FPlot).Axes[0].Right + TPlot(FPlot).Axes[0].Width div 50;
  iXp1 := iX + TPlot(FPlot).Axes[0].Width div 50;
  if (Contour.Detail = cdHigh) then
  begin
    for iY := TPlot(FPlot).Axes[1].Bottom downto TPlot(FPlot).Axes[1].Top do
    begin
      ACanvas.Pen.Color := Misc.Rainbow((TPlot(FPlot).Axes[1].Bottom - iY) / (TPlot(FPlot).Axes[1].Bottom - TPlot(FPlot).Axes[1].Top),
        Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
      ACanvas.Brush.Color := ACanvas.Pen.Color;
      ACanvas.FillRect(Rect(iX, iY, iXp1, iY+1));
    end;
  end
  else
  begin {low, medium, high:}
    for iY := TPlot(FPlot).Axes[1].Bottom downto TPlot(FPlot).Axes[1].Top do
    begin
      ACanvas.Pen.Color := Rainbow((TPlot(FPlot).Axes[1].Bottom - iY) / (TPlot(FPlot).Axes[1].Bottom - TPlot(FPlot).Axes[1].Top),
        Contour.Gamma, Contour.LambdaMin, Contour.LambdaSpan);
      ACanvas.MoveTo(iX, iY);
      ACanvas.LineTo(iXp1, iY);
    end;
  end;

{put some labels on it:}
  ACanvas.Font.Assign(TPlot(FPlot).Axes[1].Labels.Font);
  ACanvas.Brush.Style := bsClear;
  FontHeight := ACanvas.TextHeight('9') div 2;
  for i := 0 to 4 do
  begin
    iY := TPlot(FPlot).Axes[1].Bottom +
      i * (TPlot(FPlot).Axes[1].Top - TPlot(FPlot).Axes[1].Bottom) div 4 - FontHeight;
    TheText := TPlot(FPlot).Axes[1].LabelToStrF(TheMin);
    ACanvas.TextOut(iXp1+2, iY, TheText);
    TheMin := TheMin + Span / 4;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawHistory
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas IN History MODE
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawHistory(ACanvas: TCanvas; HistoryX: Single);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawHistory: ACanvas is nil !');
{$ENDIF}

  for i := 0 to Count-1 do
  begin
    Items[i].DrawHistory(ACanvas, HistoryX);
  end; {loop over series}
end;

{TSeriesList editing, both in designer and active modes -----------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.CloneSeries
  Description: Clones (Adds then Copies) a Series to a new one
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes, adds then copies a Series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.CloneSeries(TheSeries: Integer): TSeries;
begin
  if ((TheSeries < 0) or (TheSeries > Count-1)) then raise
    ERangeError.CreateFmt('I cannot clone Series #%d because there are only %d Series !', [TheSeries, Count]);

{so lets create the clone: the XDataSeries is either nil, or an existing series}
  if (Items[TheSeries].XDataSeries = nil) then
    Result := Add
   else
    Result := Self.AddDependent(Items[TheSeries].XDataSeries.Index);
  Result.Assign(Items[TheSeries]);

{move the cloned series up by 20 pixels:}
  Result.DeltaY := Items[TheSeries].DeltaY - 30;
  Result.Name := 'Clone Of ' + Items[TheSeries].Name;

  Result.ResetBounds;
  Result.GetBounds;
end;

{TSeriesList editing, both in designer and active modes -----------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.DataAsHTMLTable
  Description: creates a stringlist of the data as a HTML table
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: copying data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DataAsHTMLTable(var TheData: TStringList);
{This puts the data into a stringlist for saving or copying}
var
  i, j, k,
  NoPtsMax: Integer;
  SubHeader: String;
begin
  if (TheData = nil) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < Items[i].NoPts) then
      NoPtsMax := Items[i].NoPts;
  end;

{loop over all series:}
  SubHeader := #9 + '<tr>';
  for i := 0 to Count-1 do
  begin
{create the sub-headings:}
    if (Items[i].XDataSeries = nil) then
    begin
      SubHeader := SubHeader + '<td>' + Items[i].Name + ' - ' + Items[i].XAxis.Title.Caption + '</td>';
    end;
    SubHeader := SubHeader + '<td>' + Items[i].Name + ' - ' + Items[i].YAxis.Title.Caption + '</td>';

    case Items[i].SeriesType of
      stComplex: SubHeader := SubHeader + '<td>Imaginary</td>';
      stBubble: SubHeader := SubHeader + '<td>Radius</td>';
      stError: SubHeader := SubHeader + '<td>X Error</td><td>Y Error</td>';
      stMultiple:
        begin
          for j := 0 to Items[i].Multiplicity-1 do
            SubHeader := SubHeader + Format('<td>Extra-%d</td>', [j+1]);
        end;
    end;

{loop over points in each series:}
    for j := 0 to Items[i].NoPts-1 do
    begin
      if (Integer(TheData.Count) <= j) then
      begin
{start a new row:}
        TheData.Add(#9+#9+ '<tr>');
      end;
      if (Items[i].XDataSeries = nil) then
      begin
        TheData[j] := TheData[j] + '<td>' + FloatToStr(Items[i].XData^[j]) + '</td>';
      end;
      TheData[j] := TheData[j] + '<td>' + FloatToStr(Items[i].YData^[j]) + '</td>';
      for k := 0 to Items[i].Multiplicity-1 do
        TheData[j] := TheData[j] + '<td>' + FloatToStr(Items[i].ExtraData[k]^[j]) + '</td>';
    end; {loop over points}
    for j := Items[i].NoPts to NoPtsMax-1 do
    begin
      if (Integer(TheData.Count) <= j) then
      begin
{start a new row:}
        TheData.Add(#9+#9+ '<tr>');
      end;
      if (Items[i].XDataSeries = nil) then
      begin
        TheData[j] := TheData[j] + '<td></td>';
      end;
      TheData[j] := TheData[j] + '<td></td>';
      for k := 0 to Items[i].Multiplicity-1 do
        TheData[j] := TheData[j] + '<td></td>';
    end;
  end; {loop over series}
  SubHeader := SubHeader + '</tr>';
  TheData.Insert(0, SubHeader);
  TheData.Insert(0, '<table border=2 cellpadding=2 cellspacing=2>');
  TheData.Add('</table>');
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.AppendStream
  Description: puts the data collected since mLastSavedPoint into a stringlist for saving or copying.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saving data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.AppendStream(
  AsText: Boolean;
  Delimiter: Char;
  TheStream: TMemoryStreamEx);
var
  i, NoPtsMax: Integer;
begin
  if (TheStream = nil) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < Items[i].NoPts) then
      NoPtsMax := Items[i].NoPts;
  end;

  if (AsText) then
    SaveTextStream(TheStream, Delimiter, mLastSavedPoint, NoPtsMax)
   else
    SaveBinaryStream(TheStream, mLastSavedPoint, NoPtsMax);

  mLastSavedPoint := NoPtsMax-1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.SaveToStream
  Description: puts the data into a MemoryStream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 03/07/2001 by Mat Ballard
      Purpose: for saving or copying
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.SaveToStream(
  var TheStream: TMemoryStream;
  AsText: Boolean;
  Delimiter: Char);
var
  i, j: Integer;
  NoPtsMax: Integer;
  ALine{, BLine}: String;
  pLine: array [0..1023] of char;
begin
  Self.SaveSubHeaderStream(TheStream, Delimiter);

  if (AsText) then
    StrPCopy(pLine, 'Binary=0' + #13#10)
   else
    StrPCopy(pLine, 'Binary=1' + #13#10);
  TheStream.Write(pLine, StrLen(pLine));

  if (Count = 0) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < Items[i].NoPts) then
      NoPtsMax := Items[i].NoPts;
  end;

  if (AsText) then
  begin
    for i := 0 to Count-1 do
    begin
      if (i > 0) then
      begin
        ALine := ALine + Delimiter;
        //BLine := BLine + Delimiter;
      end;
      ALine := ALine + FloatToStr(Items[i].ZData);
      //BLine := BLine + FloatToStr(Items[i].StartTime);
      if (Items[i].XDataSeries = nil) then
      begin
        ALine := ALine + Delimiter;
        //BLine := BLine + Delimiter;
      end;
      if ((Assigned(Items[i].XStringData)) and
          (Items[i].XStringData.Count > 0)) then
      begin
        ALine := ALine + Delimiter;
        //BLine := BLine + Delimiter;
      end;
      for j := 0 to Items[i].Multiplicity-1 do
      begin
        ALine := ALine + Delimiter;
        //BLine := BLine + Delimiter;
      end;
    end;
    ALine := ALine + #13#10;
    //BLine := BLine + #13#10;
    TheStream.Write(Pointer(ALine)^, Length(ALine));
    //TheStream.Write(Pointer(BLine)^, Length(BLine));
{Get the actual data:}
    //if (NoPtsMax > 0) then
      SaveTextStream(TheStream, Delimiter, 0, NoPtsMax-1)
  end
  else
  begin
{ZData:}
    {ALine := 'ZData:';
    TheStream.Write(Pointer(ALine)^, Length(ALine));}
    for i := 0 to Count-1 do
      TheStream.Write(Items[i].ZData, SizeOf(Single));
{SrtartTimes:}
    {BLine := 'Start:';
    TheStream.Write(Pointer(BLine)^, Length(BLine));
    for i := 0 to Count-1 do
      TheStream.Write(Items[i].StartTime, SizeOf(TDateTime));}
{Get the actual data:}
    //if (NoPtsMax > 0) then
      SaveBinaryStream(TheStream, 0, NoPtsMax-1);
  end;

  FDataChanged := FALSE;
  if (NoPtsMax > 0) then
    mLastSavedPoint := NoPtsMax-1
end;


{------------------------------------------------------------------------------
    Procedure: TSeriesList.SaveSubHeaderStream
  Description: puts the data sub header onto a stream
       Author: Mat Ballard
 Date created: 08/06/2000
Date modified: 08/06/2000 by Mat Ballard
      Purpose: for saving or copying
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.SaveSubHeaderStream(
  TheStream: TMemoryStream;
  Delimiter: Char);
var
  i: Integer;
  //Str: String;
  SeriesNameLine,
  SeriesTypeLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine: String;

  procedure DoHeaderColumns(ii: Integer; aDelimiter: String);
  var
    jj: Integer;
  begin
{create the columns; first the X data / Name column:}
    SeriesNameLine := SeriesNameLine + aDelimiter + Items[ii].Name;
    SeriesTypeLine := SeriesTypeLine + aDelimiter +
      Format('%d%d', [Ord(Items[ii].SeriesType), Ord(Items[ii].Multiplicity)]);
    AxisNameLine := AxisNameLine + aDelimiter + Items[ii].YAxis.Title.Caption;
    if (Items[ii].XDataSeries = nil) then
    begin // normal internal X Data
{The X data belongs to this series:}
      DataTypeLine := DataTypeLine + aDelimiter + 'X';
      XDataSeriesLine := XDataSeriesLine + aDelimiter + '-';
    end
    else
    begin // X Data maintained in different series, so this is really a Y column:
      DataTypeLine := DataTypeLine + aDelimiter + 'Y';
      XDataSeriesLine := XDataSeriesLine + aDelimiter +
        IntToStr(Items[ii].XDataSeries.Index)
    end;
{Next, the X text data column:}
    if ((Assigned(Items[ii].XStringData)) and
        (Items[ii].XStringData.Count > 0)) then
    begin
      SeriesNameLine := SeriesNameLine + Delimiter;
      SeriesTypeLine := SeriesTypeLine + Delimiter;
      AxisNameLine := AxisNameLine + Delimiter;
      DataTypeLine := DataTypeLine + Delimiter + 'XTEXT';
      XDataSeriesLine := XDataSeriesLine + Delimiter + '-';
    end;
{Next, the normal Y data column:}
    if (Items[ii].XDataSeries = nil) then
    begin
      SeriesNameLine := SeriesNameLine + Delimiter;
      SeriesTypeLine := SeriesTypeLine + Delimiter;
      AxisNameLine := AxisNameLine + Delimiter;
      DataTypeLine := DataTypeLine + Delimiter + 'Y';
      XDataSeriesLine := XDataSeriesLine + Delimiter + '-';
    end;
{Finally, the "Extra" data columns:}
    for jj := 0 to Items[ii].Multiplicity-1 do
    begin
      SeriesNameLine := SeriesNameLine + Delimiter;
      SeriesTypeLine := SeriesTypeLine + Delimiter;
      AxisNameLine := AxisNameLine + Delimiter;
      DataTypeLine := DataTypeLine + Delimiter + 'EXTRA';
      XDataSeriesLine := XDataSeriesLine + Delimiter + '-';
    end;
  end;

begin
  if (TheStream = nil) then exit;

{loop over all the series:}
  if (Self.Count > 0) then
    DoHeaderColumns(0, '');
  for i := 1 to Self.Count-1 do
    DoHeaderColumns(i, Delimiter);

  SeriesNameLine := SeriesNameLine + #13#10;
  SeriesTypeLine := SeriesTypeLine + #13#10;
  AxisNameLine := AxisNameLine + #13#10;
  DataTypeLine := DataTypeLine + #13#10;
  XDataSeriesLine := XDataSeriesLine + #13#10;

  TheStream.Write(Pointer(SeriesNameLine)^, Length(SeriesNameLine));
  TheStream.Write(Pointer(SeriesTypeLine)^, Length(SeriesTypeLine));
  TheStream.Write(Pointer(AxisNameLine)^, Length(AxisNameLine));
  TheStream.Write(Pointer(DataTypeLine)^, Length(DataTypeLine));
  TheStream.Write(Pointer(XDataSeriesLine)^, Length(XDataSeriesLine));
end;

{------------------------------------------------------------------------------
    procedure: TSeriesList.SaveBinaryStream
  Description: gets the data as a binary stream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management: copying and saving
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.SaveBinaryStream(
  TheStream: TMemoryStream;
  Start, Finish: Integer);
var
  i: Integer;
  j, k: Integer;
  NullValue: Single;
  //pNullValue: array [0..16] of char;
  pLine: array [0..255] of char;

  procedure WriteStringData;
  begin
    if (Assigned(Items[j].XStringData))  then
      if (Items[j].XStringData.Count > 0)  then
      begin
        if (i < Items[j].XStringData.Count)  then
          StrPCopy(pLine, Items[j].XStringData.Strings[i] + #13#10)
        else
          StrPCopy(pLine, #13#10);
        TheStream.Write(pLine, StrLen(pLine));
      end;
  end;

begin
  MakeNAN(NullValue);
  {for i := 0 to SizeOf(Single)-1 do
    pNullValue[i] := 'x';
  pNullValue[SizeOf(Single)] := Chr(0);}

{all the data is written in BINARY format:}
  for i := Start to Finish do
  begin
{loop over all series:}
    for j := 0 to Count-1 do
    begin
      if (i < Items[j].NoPts) then
      begin
        if (Items[j].XDataSeries = nil) then
        begin
          TheStream.Write(Items[j].XData^[i], SizeOf(Single));
          WriteStringData;
        end;
        TheStream.Write(Items[j].YData^[i], SizeOf(Single));
        for k := 1 to Items[j].Multiplicity do
          TheStream.Write(Items[j].ExtraData[k-1]^[i], SizeOf(Single));
      end
      else
      begin
        if (Items[j].XDataSeries = nil) then
        begin
          TheStream.Write(NullValue, SizeOf(Single));
          WriteStringData;
        end;
        TheStream.Write(NullValue, SizeOf(Single));
        for k := 1 to Items[j].Multiplicity do
          TheStream.Write(NullValue, SizeOf(Single));
      end;
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    procedure: TSeriesList.SaveTextStream
  Description: gets the data as a text stream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management: copying and saving
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.SaveTextStream(
  TheStream: TMemoryStream;
  Delimiter: Char;
  Start, Finish: Integer);
var
  i, j, k: Integer;
  TheLine: String;

  procedure WriteStringData;
  begin
    if (Assigned(Items[j].XStringData))  then
      if (Items[j].XStringData.Count > 0)  then
      begin
        if (i < Integer(Items[j].XStringData.Count))  then
          TheLine := TheLine + Delimiter + Items[j].XStringData.Strings[i]
        else
          TheLine := TheLine + Delimiter;
      end;
  end;

begin
{all the data is written in text format:}
  for i := Start to Finish do
  begin
    TheLine := '';
{loop over all the remaining series:}
    for j := 0 to Count-1 do
    begin
{Add commas between (X,Y) points:}
      if (Length(TheLine) > 0) then
        TheLine := TheLine + Delimiter;
      if (i < Items[j].NoPts) then
      begin // 
        if (Items[j].XDataSeries = nil) then
        begin
          TheLine := TheLine + FloatToStr(Items[j].XData^[i]);
          WriteStringData;
        end;
        TheLine := TheLine + Delimiter + FloatToStr(Items[j].YData^[i]);
        for k := 1 to Items[j].Multiplicity do
          TheLine := TheLine + Delimiter + FloatToStr(Items[j].ExtraData[k-1]^[i]);
      end
      else
      begin
        if (Items[j].XDataSeries = nil) then
        begin
          TheLine := TheLine + Delimiter;
          WriteStringData;
        end;
        TheLine := TheLine + Delimiter;
        for k := 1 to Items[j].Multiplicity do
          TheLine := TheLine + Delimiter;
      end;
    end; {loop over points}
    TheLine := TheLine + #13#10;
    TheStream.Write(Pointer(TheLine)^, Length(TheLine));
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.LoadFromStream
  Description: Opens data on disk
       Author: Mat Ballard
 Date created: 04/25/2001
Date modified: 04/25/2001 by Mat Ballard
      Purpose: Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
               or decides to run it through ParseData instead
 Known Issues: Called by TPlot.LoadFromStream
 ------------------------------------------------------------------------------}
function TSeriesList.LoadFromStream(AStream: TMemoryStream;
  Delimiter: String;
  InitialSeriesCount: Integer;
  var AsText: Boolean): Boolean;
var
  MustAddSeries: Boolean;
  i, iSeries: Integer;
  TheLine,
  SeriesNameLine,
  SeriesTypeLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine,
  TheCell: String;
  TheStrings: TStringList;
  pNewSeries: TSeries;

  function GetColumn: String;
  begin
    GetWord(SeriesNameLine, Delimiter);
    GetWord(SeriesTypeLine, Delimiter);
    GetWord(AxisNameLine, Delimiter);
    Result := GetWord(DataTypeLine, Delimiter);
    GetWord(XDataSeriesLine, Delimiter);
  end;

begin
  TheStrings := nil;
{Possible Cases:
    1. Open, no Props:    InitialSeriesCount = 0,  Self.Count = 0  - Add
    2. Open, Props:       InitialSeriesCount = 0,  Self.Count = n  - Do not add
    3. Overlay:           InitialSeriesCount = n,  Self.Count = n  - Add
    4. Paste, clear:      InitialSeriesCount = 0,  Self.Count = 0  - Add
    5. Paste, overlay:    InitialSeriesCount = n,  Self.Count = n  - Add
    6. External call:     InitialSeriesCount = -1, Self.Count = n  - Add        }

  MustAddSeries := (InitialSeriesCount = Self.Count);

  try
{get the sub-header data:}
    SeriesNameLine := ReadLine(AStream);   // Complex			Bubble			Error				Multiple
    SeriesTypeLine := ReadLine(AStream);   // 11			21			32				44
    AxisNameLine := ReadLine(AStream);     // Y-Axis (mVolts)			Y-Axis (mVolts)			Y-Axis (mVolts)				Y-Axis (mVolts)
    DataTypeLine := ReadLine(AStream);     // X	Y	EXTRA	X	Y	EXTRA	X	Y	EXTRA	EXTRA	X	Y	EXTRA	EXTRA	EXTRA	EXTRA
    XDataSeriesLine := ReadLine(AStream);  // -	-	-	-	-	-	-	-	-	-	-	-	-	-	-	-
    iSeries := 0;
{Parse the header columns to set up the new Series:}
    while (Length(SeriesNameLine) > 0) do
    begin
{Is the X Data stored externally in another Series ?}
      TheCell := GetWord(XDataSeriesLine, Delimiter);
{Create a new Series:}
      if (Misc.IsInteger(TheCell)) then
      begin
        i := StrToInt(TheCell);
        if (MustAddSeries) then
          pNewSeries := AddDependent(InitialSeriesCount + i)
        else
        begin
          pNewSeries := Items[InitialSeriesCount + iSeries];
          pNewSeries.MakeXDataDependent(Items[InitialSeriesCount + i]);
        end;
      end
      else
      begin // this series depends on another (earlier) one:
        if (MustAddSeries) then
          pNewSeries := Add
         else
          pNewSeries := Items[InitialSeriesCount + iSeries];
      end;
{AxisList, OnStyleChange and OnDataChange are not set by streaming, so they have
 to be reset manually if the Series properties were streamed in:}
      //pNewSeries.AxisList := FAxisList;
      pNewSeries.OnStyleChange := Self.StyleChange;
      pNewSeries.OnDataChange := Self.DataChange;
      Inc(iSeries);
{Get its other properties:}
      TheCell := GetWord(SeriesNameLine, Delimiter);
      if (MustAddSeries) then
        pNewSeries.Name := TheCell;
      TheCell := GetWord(SeriesTypeLine, Delimiter);
{The first char is the SeriesType; the second the Multiplicity:}
      //if (MustAddSeries) then
      begin
        pNewSeries.SeriesType := TSeriesType(StrToInt(TheCell[1]));
        pNewSeries.Multiplicity := StrToInt(TheCell[2]);
      end;
      GetWord(AxisNameLine, Delimiter); // does not matter to data streaming, but does to copying into a spreadsheet
{The DataType can be either X, XTEXT, Y, or EXTRA:}
      TheCell := GetWord(DataTypeLine, Delimiter);
      if (TheCell = 'X') then // Skip this column: we now know what it is:
        TheCell := GetColumn;
      if (TheCell = 'XTEXT') then
      begin
        pNewSeries.MakeXDataString;
{Skip this column: we now know what it is:}
        TheCell := GetColumn;
      end;
{Skip the Y column, if it was not the first:}
      if (TheCell <> 'Y') then
        EComponentError.CreateFmt('TSeriesList.LoadFromStream: expected "Y", found "%s"', [TheCell]);
      for i := 0 to pNewSeries.Multiplicity-1 do
      begin
        if ('EXTRA' <> GetColumn) then
          EComponentError.CreateFmt('"EXTRA" not found in correct position in %s', [pNewSeries.Name]);
      end;
    end;

{Get the type of data:}
    TheLine := ReadLine(AStream);
    GetWord(TheLine, '=');
{'Binary=X': X=0 => Text, X=1 => Binary:}
    i := StrToInt(TheLine);

    if (i = 0) then
    begin
      TheStrings := TStringList.Create;
{although not documented, TStrings.LoadFromStream starts from
the CURRENT TStream.Position ! Ain't that nice !}
      TheStrings.LoadFromStream(AStream);
      Result := ConvertTextData(TheStrings, InitialSeriesCount, Delimiter);
      AsText := TRUE;
    end
    else
    begin
      Result := ConvertBinaryData(AStream, InitialSeriesCount);
      AsText := FALSE;
    end;

 finally
   if (Assigned(TheStrings)) then
    TheStrings.Free;
 end;

{new data has not changed:}
  FDataChanged := FALSE;
{Get the new bounds:}
  for i := InitialSeriesCount to Self.Count-1 do
    Items[i].GetBounds;
end;

{$IFDEF VER200_COMPATIBLE}
{------------------------------------------------------------------------------
     Function: TSeriesList.LoadFromStream_Ver200
  Description: Opens data on disk
       Author: Mat Ballard
 Date created: 04/25/2001
Date modified: 04/25/2001 by Mat Ballard
      Purpose: Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
               or decides to run it through ParseData instead
 Known Issues: Called by TPlot.LoadFromStream_Ver200
 ------------------------------------------------------------------------------}
function TSeriesList.LoadFromStream_Ver200(
  AStream: TMemoryStream; var AsText: Boolean): Boolean;
var
  TheResult: Boolean;
  ColCount,
  //FileVersion,
  i,
  iColumn,
  InitialSeriesCount,
  //LineLength,
  NoFileSeries: Integer;
  TheLine,
  SeriesNameLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine,
  TheCell: String;
  TheStrings: TStringList;
  SeriesInfo: pSeriesInfoArray;
  SeriesOfCol: pIntegerArray;
  pNewSeries: TSeries;
  //OldIgnoreChanges: Boolean;

  procedure CleanUp;
  begin
    if (Assigned(SeriesInfo)) then
      FreeMem(SeriesInfo, ColCount * SizeOf(TSeriesInfo));
    SeriesInfo := nil;
    if (Assigned(SeriesOfCol)) then
      FreeMem(SeriesOfCol, ColCount * SizeOf(Integer));
    SeriesOfCol := nil;
    if (Assigned(TheStrings)) then
      TheStrings.Free;
    TheStrings := nil;
  end;

begin
  //LoadFromStream_Ver200 := FALSE;
  ColCount := 1;
  SeriesInfo := nil;
  SeriesOfCol := nil;
  TheStrings := nil;

  InitialSeriesCount := Self.Count;
  try
{get the sub-header data:}
    SeriesNameLine := ReadLine(AStream);
    AxisNameLine := ReadLine(AStream);
    DataTypeLine := ReadLine(AStream);
    XDataSeriesLine := ReadLine(AStream);

{find out how many columns there are:}
    for i := 1 to Length(DataTypeLine) do
      if (DataTypeLine[i] = ',') then
        Inc(ColCount);

    if (ColCount < 2) then raise
      EFOpenError.CreateFmt('This file only has %d columns !', [ColCount]);

{allocate memory for the dynamic arrays, which are small:}
    GetMem(SeriesInfo, ColCount * SizeOf(TSeriesInfo));
    GetMem(SeriesOfCol, (ColCount+1) * SizeOf(Integer));
{this allocates more memory than SeriesInfo needs, but so what ?}

{Determine the number of series:}
    NoFileSeries := 0;
    for i := 0 to ColCount-1 do
    begin
      SeriesInfo^[i].XCol := 0;
      SeriesInfo^[i].XTextCol := -1;
    end;
{examine the columns one by one:}
    for iColumn := 1 to ColCount do
    begin
{No new column yet belongs to a Series:}
      SeriesOfCol^[iColumn] := -1;
      TheCell := GetWord(DataTypeLine, ',');
      if (TheCell = 'X') then
      begin
{we've found a X data column to add.}
        SeriesOfCol^[iColumn] := NoFileSeries;
        SeriesInfo^[NoFileSeries].XCol := iColumn;
        GetWord(XDataSeriesLine, ',');
        GetWord(SeriesNameLine, ',');
      end
      else if (TheCell = 'XTEXT') then
      begin
{we've found a X STRING data column to add.}
        SeriesOfCol^[iColumn] := NoFileSeries;
        SeriesInfo^[NoFileSeries].XTextCol := iColumn;
        GetWord(XDataSeriesLine, ',');
        GetWord(SeriesNameLine, ',');
      end
      else if (TheCell = 'Y') then
      begin
{we've found a Y data column to add.}
        SeriesInfo^[NoFileSeries].YCol := iColumn;
{find the X Column that this Y column uses:}
        TheCell := GetWord(XDataSeriesLine, ',');
        if (IsInteger(TheCell)) then
          SeriesInfo^[NoFileSeries].XSeriesIndex := StrToInt(TheCell) + InitialSeriesCount
        else
          SeriesInfo^[NoFileSeries].XSeriesIndex := -1;
{Add the new series:}
        pNewSeries := AddDependent(SeriesInfo^[NoFileSeries].XSeriesIndex);
        SeriesInfo^[NoFileSeries].Index := pNewSeries.Index;
        pNewSeries.Name := GetWord(SeriesNameLine, ',');
        Inc(NoFileSeries);
      end; {found a Y data column}
    end; {for}

{Get the type of data:}
    TheLine := ReadLine(AStream);
    GetWord(TheLine, '=');
{'Binary=X': X=0 => Text, X=1 => Binary:}
    i := StrToInt(TheLine);

{now we try to convert all the data:}
    if (i = 0) then
    begin {text}
      TheStrings := TStringList.Create;
{although not documented, TStrings.LoadFromStream_Ver200 starts from
the CURRENT TStream.Position ! Ain't that nice !}
      TheStrings.LoadFromStream(AStream);
      TheResult := Self.ConvertTextData_Ver200(ColCount, NoFileSeries, 0, ',', TheStrings, SeriesInfo);
      AsText := TRUE;
    end
    else if (i = 1) then
    begin {binary}
      TheResult := Self.ConvertBinaryData_Ver200(ColCount, NoFileSeries, AStream, SeriesInfo);
      AsText := FALSE;
    end
    else
    begin
      raise EFOpenError.Create('This file has no binary marker !');
    end;

{new data has not changed:}
    FDataChanged := FALSE;
  except
    CleanUp;
    raise;
  end; {try}

  for i := InitialSeriesCount to Self.Count-1 do
    Items[i].GetBounds;

  LoadFromStream_Ver200 := TheResult;
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: TSeriesList.GetNearestPoint
  Description: gets the nearest series and point to the input point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user selection of data on screen
 Return Value: the Index of the nearest point
 Known Issues: 1. this is a long and messy function: while some parts could be done
               lower done in TSeries, other bits can't. i am still not
               completely happy with it.
               2. We "GenerateXXXOutline" here for Columns and Pies, but not for XY types.
 ------------------------------------------------------------------------------}
function TSeriesList.GetNearestPoint(
  ThePlotType: TPlotType;
  ColumnGap,
  iX, iY: Integer;
  var TheSeries: Integer;
  var MinDistance: Single;
  var pSeries: TSeries): Integer;
var
  CellWidth, CellHeight,
  iCol, jRow, NoPieCols,
  i,
  PieLeft, PieTop, PieWidth, PieHeight: Integer;
  NearestPoint: Integer;
  Distance, dX, Span, X, Y,
  YSum, YNegSum, YSumOld, YNegSumOld, YTotal, YNegTotal: Single;

  function GetColumnIndex: Boolean;
  begin
    GetColumnIndex := FALSE;
{all column plots use Series[0].XData:}
    X := TPlot(FPlot).Axes[0].XofF(iX);
{bug out if outside span:}
    if (X < Items[0].XData^[0]) then exit;
    dX := ((100 - ColumnGap) / 100) * (
      Items[0].XData^[Items[0].NoPts-1] -
      Items[0].XData^[Items[0].NoPts-2]);
    if (X > Items[0].XData^[Items[0].NoPts-1] + dX) then exit;
{find the nearest point:}
    NearestPoint := Items[0].GetNearestPointToFX(iX);
    if (NearestPoint <> 0) then
      if (X < Items[0].XData^[NearestPoint]) then
        Dec(NearestPoint);
    if (NearestPoint = Items[0].NoPts-1) then
      Span := Items[0].XData^[NearestPoint] - Items[0].XData^[NearestPoint-1]
    else
      Span := Items[0].XData^[NearestPoint+1] - Items[0].XData^[NearestPoint];
    dX := ((100 - ColumnGap) / 100) * Span;
{was the click in the gap between bars ?}
    if (X > Items[0].XData^[NearestPoint] + dX) then
      exit;
    GetColumnIndex := TRUE;
  end;

begin
  Distance := 1.0e38;
  MinDistance := 1.0e38;
  TheSeries := -1;
  GetNearestPoint := -1;
  pSeries := nil;

  if (Self.Count = 0) then exit;

  case ThePlotType of
    ptXY: //, ptError, ptMultiple, ptBubble:
      begin
{loop over series: note that ptError and ptBubble skips every second series -
 the error/size ones}
        for i := 0 to Count-1 do
        begin
{Find the nearest point IN THIS SERIES:}
          NearestPoint := Items[i].GetNearestXYPoint(
            iX, iY,
            0, 0,
            Distance);
{Mirror, Mirror on the wall,
 who is the nearest one of all ?}
          if (Distance < MinDistance) then
          begin
            GetNearestPoint := NearestPoint;
            MinDistance := Distance;
            TheSeries := i;
            pSeries := Items[i];
          end;
{Note: we don't pSeries.GenerateXYOutline here, because that would be running
 that method every time the screen got clicked on, which is a bit of a drain.
 However, we do run pSeries.GenerateColumnOutline and pSeries.GeneratePieOutline
 because they are simple assignments.}
        end; {for}
      end;

    ptColumn:
      begin
        if (not GetColumnIndex) then
          exit;
{now home in: which series was it:}
        TheSeries := 0;
        if (Self.Count > 1) then
          TheSeries := Trunc(Self.Count * (X - Items[0].XData^[NearestPoint]) / dX);
{we now know which point in  which series:}
        pSeries := Items[TheSeries];
        Y := TPlot(FPlot).Axes[1].YofF(iY);
        if (Y > pSeries.YData^[NearestPoint]) then
          exit;
        if ((Y < 0) and (Y < pSeries.YData^[NearestPoint])) then
          exit;
        GetNearestPoint := NearestPoint;
        MinDistance := 0;
        pSeries.GenerateColumnOutline(
          TPlot(FPlot).Axes[0].FofX(pSeries.XData^[NearestPoint] + TheSeries * dX / Self.Count),
          TPlot(FPlot).Axes[1].FofY(0),
          TPlot(FPlot).Axes[0].FofX(pSeries.XData^[NearestPoint] + (TheSeries+1) * dX / Self.Count),
          TPlot(FPlot).Axes[1].FofY(pSeries.YData^[NearestPoint]));
      end;

    ptStack, ptNormStack:
      begin
        if (not GetColumnIndex) then
          exit;
        Y := TPlot(FPlot).Axes[1].YofF(iY);
{now home in: which series was it:}
        TheSeries := 0;
        YSum := 0;
        YNegSum := 0;
        YTotal := 0;
        YNegTotal := 0;
        if (ThePlotType = ptNormStack) then
{ptStack and ptNormStack are pretty similar expcept for ...}
        begin
          if ((Y < 0) or (Y > 100)) then
            exit;
{count every series:}
          for i := 0 to Count-1 do
          begin
            if (Items[i].YData^[NearestPoint] >= 0) then
              YTotal := YTotal + Items[i].YData^[NearestPoint]
             else
              YNegTotal := YNegTotal + Items[i].YData^[NearestPoint];
          end; {count every series}
{prepare for conversion of data to percent:}
          YTotal := YTotal / 100;
          YNegTotal := - YNegTotal / 100;
        end;
{loop over every series:}
        for i := 0 to Count-1 do
        begin
          pSeries := Items[i];
          if (pSeries.YData^[NearestPoint] >= 0) then
          begin
            YSumOld := YSum;
            if (ThePlotType = ptStack) then
              YSum := YSum + pSeries.YData^[NearestPoint]
             else {ptNormStack}
              YSum := YSum + pSeries.YData^[NearestPoint] / YTotal;
            if ((YSumOld < Y) and (Y < YSum)) then
            begin {Bingo !}
              GetNearestPoint := NearestPoint;
              MinDistance := 0;
              pSeries.GenerateColumnOutline(
                TPlot(FPlot).Axes[0].FofX(pSeries.XData^[NearestPoint]),
                TPlot(FPlot).Axes[1].FofY(YSumOld),
                TPlot(FPlot).Axes[0].FofX(pSeries.XData^[NearestPoint] + dX),
                TPlot(FPlot).Axes[1].FofY(YSum));
              break;
            end;
          end
          else
          begin
            YNegSumOld := YNegSum;
            if (ThePlotType = ptStack) then
              YNegSum := YNegSum + pSeries.YData^[NearestPoint]
             else {ptNormStack}
              YNegSum := YNegSum + pSeries.YData^[NearestPoint] / YNegTotal;
            if ((YNegSum < Y) and (Y < YNegSumOld)) then
            begin {Bingo !}
              GetNearestPoint := NearestPoint;
              MinDistance := 0;
              pSeries.GenerateColumnOutline(
                TPlot(FPlot).Axes[0].FofX(pSeries.XData^[NearestPoint]),
                TPlot(FPlot).Axes[1].FofY(YNegSumOld),
                TPlot(FPlot).Axes[0].FofX(pSeries.XData^[NearestPoint] + dX),
                TPlot(FPlot).Axes[1].FofY(YNegSum));
              break;
            end;
          end; {YData >= 0}
        end; {for i}
      end;

    ptPie:
      begin
{each Pie sits in a cell:}
        NoPieCols := Trunc(0.99 + Count / mNoPieRows);
        CellWidth := (mPlotBorder.Right - mPlotBorder.Left) div NoPieCols;
        CellHeight := (mPlotBorder.Bottom - mPlotBorder.Top) div mNoPieRows;
{... but does not occupy the entire cell:}
        PieWidth := Round(PIE_SIZE * CellWidth);
        PieHeight := Round(PIE_SIZE * CellHeight);
        if (PieHeight > PieWidth) then
          PieHeight := PieWidth;

        i := 0;
        for iCol := 0 to NoPieCols-1 do
        begin
          for jRow := 0 to mNoPieRows-1 do
          begin
            if (i >= Count) then break;
{indent the (left, top) a bit:}
            PieLeft := mPlotBorder.Left + iCol * CellWidth +
              (CellWidth-PieWidth) div 2;
            PieTop := mPlotBorder.Top + jRow * CellHeight +
              (CellHeight-PieHeight) div 2;
            pSeries := Items[i];
{Find the nearest point IN THIS SERIES:}
            NearestPoint := pSeries.GetNearestPieSlice(
              iX, iY,
              PieLeft, PieTop, PieWidth, PieHeight,
              Distance);
            if (Distance = 0) then
            begin
              GetNearestPoint := NearestPoint;
              MinDistance := Distance;
              TheSeries := i;
              pSeries.GeneratePieOutline(
                PieLeft,
                PieTop,
                PieWidth,
                PieHeight,
                NearestPoint);
              break;
            end;
            Inc(i);
          end; {jRow}
        end; {iCol}
      end; {ptPie}
    ptPolar:
      begin
      end;
  end; {case}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetSeriesOfZ
  Description: gets the series with ZData ZValue
       Author: Mat Ballard
 Date created: 04/25/2001
Date modified: 04/25/2001 by Mat Ballard
      Purpose: parsing data from strange files
 Return Value: the guilty series, or nil
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetSeriesOfZ(ZValue: Single): TSeries;
var
  i: Integer;
begin
  Result := nil;
  for i := 0 to Self.Count-1 do
  begin
    if (ZValue = Items[i].ZData) then
    begin
      Result := Items[i];
      break;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DoChange
  Description: event firing proedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnDataChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DoStyleChange;
begin
  if Assigned(FOnStyleChange) then OnStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DoDataChange
  Description: event firing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnDataChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DoDataChange;
begin
  FDataChanged := TRUE;
  if Assigned(FOnDataChange) then OnDataChange(Self);
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetTotalNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the TotalNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetTotalNoPts: Integer;
var
  i: Integer;
  Sum: Integer;
begin
{loop over all series:}
  Sum := 0;
  for i := 0 to Count-1 do
  begin
    Sum := Sum + Items[i].NoPts;
  end;
  Result := Sum;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetDisplaced
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MaxNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetDisplaced: Boolean;
var
  i: Integer;
begin
  Result := FALSE;
  for i := 0 to Count-1 do
  begin
    if ((Items[i].DeltaX <> 0) or (Items[i].DeltaY <> 0)) then
    begin
      Result := TRUE;
      Break;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetMaxNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MaxNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetMaxNoPts: Integer;
var
  i,
  TheMax: Integer;
begin
  TheMax := 0;
  if (Count > 0) then
  begin
    TheMax := Items[0].NoPts;
    for i := 1 to Count-1 do
    begin
      if (TheMax < Items[i].NoPts) then
        TheMax := Items[i].NoPts;
    end;
  end;
  Result := TheMax;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetMinNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MinNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetMinNoPts: Integer;
var
  i,
  TheMin: Integer;
begin
  TheMin := 0;
  if (Count > 0) then
  begin
    TheMin := Items[0].NoPts;
    for i := 1 to Count-1 do
    begin
      if (TheMin < Items[i].NoPts) then
        TheMin := Items[i].NoPts;
    end;
  end;
  Result := TheMin;
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
     Function: TSeriesList.ParseData
  Description: oversees the importation and pasting of data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: runs the ImportForm, and adds the new data as new Series
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ParseData(
  TheData: TStringList;
  TheHelpFile: String): Boolean;
var
  i,
  iColumn,
  jRow,
  NoXs, NoYs, NoZs, NoIgnores: Integer;
  iStart, iFinish: Integer;
  //Delimiter,
  ALine,
  ZDataLine,
  ZValue: String;
  AsText: Boolean;
  Ignore: pBooleanArray;
  ImportForm: TImportForm;
  AStream: TMemoryStream;

  function GetLine(RowIndex: Integer): String;
  begin
    Result := ImportForm.InfoGrid.Rows[RowIndex].CommaText;
    System.Delete(Result, 1, Pos(',', Result));
    if (ImportForm.Delimiter <> ',') then
      Result := StringReplace(Result, ',', ImportForm.Delimiter, [rfReplaceAll]);
  end;

begin
  Result := FALSE;

  ImportForm := TImportForm.Create(nil);
  for jRow := 0 to Min(100, TheData.Count-1) do
    ImportForm.DataListBox.Items.Add(TheData.Strings[jRow]);

  ImportForm.HelpFile := TheHelpFile;

  if (ImportForm.ShowModal = mrOK) then
  begin
    //Delimiter := ImportForm.Delimiter;
    if (not ImportForm.OverlayCheckBox.Checked) then
      Self.Clear;
    NoXs := 0;
    NoYs := 0;
    NoIgnores := 0;
    NoZs := 0;
    ReAllocMem(Ignore, ImportForm.InfoGrid.ColCount * SizeOf(Boolean));
    for iColumn := 1 to ImportForm.InfoGrid.ColCount-1 do
    begin
      Ignore^[iColumn-1] := FALSE;
      if (ImportForm.InfoGrid.Cells[iColumn, DATATYPE_LINE] = 'X') then
        Inc(NoXs)
      else if (ImportForm.InfoGrid.Cells[iColumn, DATATYPE_LINE] = 'Y') then
        Inc(NoYs)
      else if (ImportForm.InfoGrid.Cells[iColumn, DATATYPE_LINE] = 'Z') then
        Inc(NoZs)
      else if (Pos('Ignore', ImportForm.InfoGrid.Cells[iColumn, DATATYPE_LINE]) > 0) then
      begin // 'Ignore Column/Series'; do this here for efficiency of code:
        Inc(NoIgnores);
        Ignore^[iColumn-1] := TRUE;
      end;
    end; {for iColumn}

{The data might be in the form of:
    x1,y1,z1
    x2,y2,z2
    ...
 or in the form of columns of series (ie: like we save it):}
    if (NoZs > 0) then
    begin
      if ((NoXs = NoYs) and (NoYs = NoZs)) then
      begin
        Result := ConvertXYZData(ImportForm, TheData);
      end
      else
        ShowMessage('Sorry - your data is too complex for me to understand !');
    end
    else
    begin
{First remove the now-unwanted header material from TheData:}
      for jRow := 1 to ImportForm.TheFirstDataLine do
        TheData.Delete(0);
{remove the "Ignored" Columns from the remaining Data:}
      if (NoIgnores > 0) then
      begin
        for jRow := 0 to TheData.Count-1 do
        begin
          ALine := TheData.Strings[jRow];
          Clipboard.AsText := ALine;
          iStart := 1;
          iFinish := Misc.ForwardPos(ImportForm.Delimiter, ALine, iStart);
          iColumn := 0;
          while (iFinish > 0) do
          begin
            if (Ignore^[iColumn]) then
            begin
              System.Delete(ALine, iStart, iFinish-iStart+1);
              Clipboard.AsText := ALine;
            end
            else
            begin
              iStart := iFinish + 1;
            end;
{Note: if the last column is Ignored, iFinish will be -1: there is no trailing
 delimiter, so this last column will NOT be deleted;
 however, ConvertTextData just ignores this last column anyway.}
            iFinish := Misc.ForwardPos(ImportForm.Delimiter, ALine, iStart);
            Inc(iColumn);
          end;
          TheData.Strings[jRow] := ALine;
        end;
      end;

{Insert the neccessary header rows - as blanks:}
      for jRow := 1 to 12 do
        TheData.Insert(0, '');
{0:     Versions        TPlot=300
 1:     "               FileFormat=300
 2:     Title           TPlot - XY - Four SeriesTypes at Once !
 3:     Marker          Subheader
 4:     Series Names    Complex,,,Bubble,,,Error,,,,Multiple,,,,,
 5:     SeriesType/Mult 11,,,21,,,32,,,,44,,,,,
 6:     AxisNames       Y-Axis (mVolts),,,Y-Axis (mVolts),,,Y-Axis (mVolts),,,,Y-Axis (mVolts),,,,,
 7:     DataType        X,Y,EXTRA,X,Y,EXTRA,X,Y,EXTRA,EXTRA,X,Y,EXTRA,EXTRA,EXTRA,EXTRA
 8:     Depends X       -,-,-,-,-,-,-,-,-,-,-,-,-,-,-,-
 9:     Bin/Text        Binary=0
 10:    Z-Data          0,,,1,,,2,,,,3,,,,,
 11:    Time:           37585.4209073958,,,37585.4209073958,,,37585.4209073958,,,,37585.4209073958,,,,,}

{Parse the user-entered header data; first remove ignored columns:}
      ImportForm.CleanUpGrid(Self);
{Now put in the header information:}
      TheData.Strings[0] := 'TPlot=' + IntToStr(TPLOT_VERSION);
      TheData.Strings[1] := 'FileFormat=' + IntToStr(FILE_FORMAT_VERSION);
      TheData.Strings[2] := ImportForm.TitleEdit.Text;
      TheData.Strings[3] := SUBHEADER;
      TheData.Strings[4] := GetLine(SERIES_NAMES);
      TheData.Strings[5] := GetLine(SERIESTYPE_LINE);
      ALine := '';
      for i := 1 to ImportForm.InfoGrid.ColCount-1 do
        ALine := ALine + ImportForm.Delimiter;
      TheData.Strings[6] := ALine;
      TheData.Strings[7] := GetLine(DATATYPE_LINE);
      TheData.Strings[8] := GetLine(DEPENDS_ON_X);
      TheData.Strings[9] := 'Binary=0';
      TheData.Strings[10] := GetLine(Z_DATA_LINE);
{StartTime is a bastard: it could be a number, or s string date, or blank:}
      ALine := GetLine(TIME_DATA_LINE);
      ZDataLine := '';
      while (Length(ALine) > 0) do
      begin
        ZValue := GetWord(ALine, ImportForm.Delimiter);
        if (Length(ZValue) > 0) then
        begin
          if (not Misc.IsReal(ZValue)) then
          begin
            try
              ZValue := FloatToStr(StrToDateTime(ZValue));
            except
              ZValue := FloatToStr(Now);
            end;
          end;
        end;
        ZDataLine := ZDataLine + ZValue + ImportForm.Delimiter;
      end;
      SetLength(ZDataLine, Length(ZDataLine)-Length(ImportForm.Delimiter));
      TheData.Strings[11] := ZDataLine;

      if (ImportForm.Delimiter <> ',') then
      begin
        TheData.Strings[4] := StringReplace(TheData.Strings[4], ',', ImportForm.Delimiter, [rfReplaceAll]);
        TheData.Strings[7] := StringReplace(TheData.Strings[7], ',', ImportForm.Delimiter, [rfReplaceAll]);
        TheData.Strings[8] := StringReplace(TheData.Strings[8], ',', ImportForm.Delimiter, [rfReplaceAll]);
        TheData.Strings[10] := StringReplace(TheData.Strings[10], ',', ImportForm.Delimiter, [rfReplaceAll]);
      end;
      AStream := TMemoryStream.Create;
      TheData.SaveToStream(AStream);
{And _FINALLY_, load the data thru an "Stream" method:}
      Self.LoadFromStream(AStream, ImportForm.Delimiter, Self.Count, AsText);
      AStream.Free;
    end; {if (NoZs > 0) }
    ReAllocMem(Ignore, 0);
  end; {ShowModal}

  ImportForm.Free;
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertBinaryData
  Description: Adds binary data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: Given a AxisLocationArray, converts the text data to numeric data and adds it to the new Axis
 Return Value: TRUE is successful
 Known Issues: This procedure assumes that TheStream.Position points to the start
               of the binary data.
               Moved from TCustomPlot.
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertBinaryData(
  TheStream: TMemoryStream;
  InitialSeriesCount: Integer): Boolean;
var
  DataSize,
  i, j: Integer; {iColumn}
  NullValue,
  ZValue: Single;
  //AStartTime: TDateTime;
  X, Y: Single;
  Extra: array [0..3] of Single;
  XText: String;
begin
  DataSize := SizeOf(Single);
  MakeNAN(NullValue);
  
{get the ZData:}
  for i := InitialSeriesCount to Self.Count-1 do
  begin
    TheStream.Read(ZValue, DataSize);
    Items[i].ZData := ZValue;
  end;

{get the StartTime:
  for i := InitialSeriesCount to Self.Count-1 do
  begin
    TheStream.Read(AStartTime, SizeOf(TDateTime));
    Items[i].StartTime := AStartTime;
  end;}

  X := -1;
  XText := '';
  while (TheStream.Position < TheStream.Size) do
  begin            
    for i := InitialSeriesCount to Self.Count-1 do
    begin
      if (Items[i].XDataSeries = nil) then
        TheStream.Read(X, DataSize);
      if (Assigned(Items[i].XStringData)) then
        XText := ReadLine(TheStream);
      TheStream.Read(Y, DataSize);
      for j := 0 to Items[i].Multiplicity-1 do
        TheStream.Read(Extra[j], DataSize);
      if (Y <> NullValue) then
      begin
        if (Items[i].SeriesType = stNormal) then
        begin
          if (Assigned(Items[i].XStringData)) then
            Items[i].AddStringPoint(XText, X, Y, FALSE, FALSE)
           else
            Items[i].AddPoint(X, Y, FALSE, FALSE);
        end
        else
        begin
          if (Assigned(Items[i].XStringData)) then
            Items[i].AddStringPointEx(XText, X, Y, Extra, FALSE, FALSE)
           else
            Items[i].AddPointEx(X, Y, Extra, FALSE, FALSE);
        end;
      end; {if Valid}
    end; {for i}
  end; {for lines of data}

{for a subsequent SaveClick:}
  Result := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertTextData
  Description: Adds text data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: Given a pSeriesInfoArray, converts the text data to numeric data and adds it to the new Axis
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertTextData(
  TheData: TStringList;
  InitialSeriesCount: Integer;
  Delimiter: String): Boolean;
var
  i, iLine, j: Integer;
  X, Y: Single;
  Extra: array [0..3] of Single;
  TheCell,
  TheLine,
  XText: String;

  function GetValue: Single;
  begin
    TheCell := GetWord(TheLine, Delimiter);
    if (Length(TheCell) = 0) then
    begin
      Result := -999;
    end
     else
      Result := StrToFloat(TheCell)
  end;

begin
{The first line is Z Data:}
  TheLine := TheData.Strings[0];
  i := InitialSeriesCount;
  while (Length(TheLine) > 0) do
  begin
    TheCell := GetWord(TheLine, Delimiter);
    if (Length(TheCell) > 0) then
    begin
      Items[i].ZData := StrToFloat(TheCell);
      Inc(i);
    end;
  end;

{The second line is NO LONGER StartTime Data:
  TheLine := TheData.Strings[1];
  i := InitialSeriesCount;
  while (Length(TheLine) > 0) do
  begin
    TheCell := GetWord(TheLine, Delimiter);
    if (Length(TheCell) > 0) then
    begin
      Items[i].StartTime := StrToFloat(TheCell);
      Inc(i);
    end;
  end;}

  XText := '';
  for iLine := 2 to TheData.Count-1 do
  begin
    TheLine := TheData.Strings[iLine];
    for i := InitialSeriesCount to Self.Count-1 do
    begin
      //if (Items[i].XDataSeries = nil) then
        X := GetValue;
      if (Assigned(Items[i].XStringData)) then
        XText := GetWord(TheLine, Delimiter);
{get the Y Date:}
      Y := GetValue;
      for j := 0 to Items[i].Multiplicity-1 do
        Extra[j] := GetValue;
      if (Items[i].SeriesType = stNormal) then
      begin
        if (Assigned(Items[i].XStringData)) then
          Items[i].AddStringPoint(XText, X, Y, FALSE, FALSE)
         else
          Items[i].AddPoint(X, Y, FALSE, FALSE);
      end
      else
      begin
        if (Assigned(Items[i].XStringData)) then
          Items[i].AddStringPointEx(XText, X, Y, Extra, FALSE, FALSE)
         else
          Items[i].AddPointEx(X, Y, Extra, FALSE, FALSE);
      end;
    end; {for i}
  end; {for iLine}

{for a subsequent SaveClick:}
  Result := TRUE;
end;
{Alternative version of reading in the actual data:}
      (*for jRow := ImportForm.TheFirstDataLine to TheData.Count-1 do
      begin
        TheLine := TheData[jRow];
        X := NullValue;
        XText := '';
        Y := NullValue;
        for i := 0 to 3 do
          Extra[i] := NullValue;
        for iColumn := 1 to ImportForm.InfoGrid.Rows[DATATYPE_LINE].Count-1 do
        begin
          TheCell := GetWord(TheLine, Delimiter);
          if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iCol] = 'X') then
          begin
            X := StrToFloat(TheCell);
          end
          else if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iCol] = 'XTEXT') then
          begin
            XText := TheCell;
          end
          else if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iCol] = 'Y') then
          begin
            Y := StrToFloat(TheCell);
            iExtra := 0;
          else if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iCol] = 'EXTRA') then
          begin
            Extra[iExtra] := StrToFloat(TheCell);
            Inc(iExtra);
          end;
        end; {for iColumn}
      end; {for jRow} *)


{$IFDEF VER200_COMPATIBLE}
{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertBinaryData_Ver200
  Description: Adds binary data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: Given a AxisLocationArray, converts the text data to numeric data and adds it to the new Axis
 Return Value: TRUE is successful
 Known Issues: This procedure assumes that TheStream.Position points to the start
               of the binary data.
               Moved from TCustomPlot.
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertBinaryData_Ver200(
  ColCount,
  SeriesCount: Integer;
  TheStream: TMemoryStream;
  SeriesInfo: pSeriesInfoArray): Boolean;
var
  DataSize,
  i: Integer; {iColumn}
  MemPosition: Longint;
  NullValue,
  ZValue: Single;
  AStartTime: TDateTime;
  pValues: pSingleArray;
  pLine: array [0..32] of char;
  XTextValue: String;
begin
  //ConvertBinaryData_Ver200 := FALSE;
  GetMem(pValues, ColCount * SizeOf(Single));
  DataSize := SizeOf(Single);

{calculate our "ignore" value:}
  MakeNAN(NullValue);

{check for ZData:}
  MemPosition := TheStream.Position;
  TheStream.Read(pLine, 6);
  pLine[6] := Chr(0);
  if (StrComp(pLine, 'ZData:') = 0) then
  begin
    for i := 0 to SeriesCount-1 do
    begin
      TheStream.Read(ZValue, DataSize);
      TSeries(Self.Items[SeriesInfo^[i].Index]).ZData := ZValue;
    end;
  end
  else
    TheStream.Position := MemPosition;

{check for StartTime:}
  MemPosition := TheStream.Position;
  TheStream.Read(pLine, 6);
  pLine[6] := Chr(0);
  if (StrComp(pLine, 'Start:') = 0) then
  begin
    for i := 0 to SeriesCount-1 do
    begin
      TheStream.Read(AStartTime, SizeOf(TDateTime));
      //TSeries(Self.Items[SeriesInfo^[i].Index]).StartTime := AStartTime;
    end;
  end
   else
    TheStream.Position := MemPosition;

  while (TheStream.Position < TheStream.Size) do
  begin
    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        TheStream.Read(SeriesInfo^[i].XValue, DataSize);
      if (SeriesInfo^[i].XTextCol > 0) then
        XTextValue := ReadLine(TheStream);
      TheStream.Read(SeriesInfo^[i].YValue, DataSize);

      if (SeriesInfo^[i].XTextCol > 0) then
        TSeries(Self.Items[SeriesInfo^[i].Index]).AddStringPoint(
          XTextValue,
          SeriesInfo^[i].XValue,
          SeriesInfo^[i].YValue,
          FALSE, FALSE)
      else
        TSeries(Self.Items[SeriesInfo^[i].Index]).AddPoint(
          SeriesInfo^[i].XValue,
          SeriesInfo^[i].YValue,
          FALSE, FALSE);
    end; {for iColumn}
  end; {for lines of data}

  FreeMem(pValues, ColCount * SizeOf(Single));
{for a subsequent SaveClick:}
  ConvertBinaryData_Ver200 := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertTextData_Ver200
  Description: Adds text data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: Given a pSeriesInfoArray, converts the text data to numeric data and adds it to the new Axis
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertTextData_Ver200(
  ColCount,
  SeriesCount,
  FirstLine: Integer;
  Delimiter: String;
  TheData: TStringList;
  SeriesInfo: pSeriesInfoArray): Boolean;
var
  i,
  jRow: Integer;
  TheCell,
  TheLine: String;
  TheSortedLine: TStringList;
begin
{Does this contain Z Data ?}
  if (Pos('ZData', TheData.Strings[FirstLine]) > 0) then
  begin
    TheLine := TheData.Strings[0];
    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        GetWord(TheLine, Delimiter);
      if (SeriesInfo^[i].XTextCol > 0) then
        GetWord(TheLine, Delimiter);
      TheCell := GetWord(TheLine, Delimiter);
      TSeries(Self.Items[SeriesInfo^[i].Index]).ZData :=
        StrToFloat(TheCell);
    end;
    Inc(FirstLine);
  end;

{Does this contain StartTimes ?}
  if (Pos('Start', TheData.Strings[FirstLine]) > 0) then
  begin
    TheLine := TheData.Strings[0];
    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        GetWord(TheLine, Delimiter);
      if (SeriesInfo^[i].XTextCol > 0) then
        GetWord(TheLine, Delimiter);
      TheCell := GetWord(TheLine, Delimiter);
      TSeries(Self.Items[SeriesInfo^[i].Index]).StartTime :=
        StrToFloat(TheCell);
    end;
    Inc(FirstLine);
  end;

  TheSortedLine := TStringList.Create;
  for i := 0 to ColCount-1 do
    TheSortedLine.Add('');

  for jRow := FirstLine to TheData.Count-1 do
  begin
    TheLine := TheData.Strings[jRow];

    for i := 0 to ColCount-1 do
    begin
      TheSortedLine.Strings[i] := GetWord(TheLine, Delimiter);
    end;

    for i := 0 to SeriesCount-1 do
    begin
      try
        if (SeriesInfo^[i].XCol = -2) then
          SeriesInfo^[i].XValue := (jRow - FirstLine)
        else if (SeriesInfo^[i].XCol > 0) then
          SeriesInfo^[i].XValue := StrToFloat(TheSortedLine.Strings[SeriesInfo^[i].XCol-1]);
        if (Length(TheSortedLine.Strings[SeriesInfo^[i].YCol-1]) > 0) then
        begin
          SeriesInfo^[i].YValue := StrToFloat(TheSortedLine.Strings[SeriesInfo^[i].YCol-1]);
          if (SeriesInfo^[i].XTextCol > 0) then
            TSeries(Self.Items[SeriesInfo^[i].Index]).AddStringPoint(
              TheSortedLine.Strings[SeriesInfo^[i].XTextCol-1],
              SeriesInfo^[i].XValue,
              SeriesInfo^[i].YValue,
              FALSE, FALSE)
           else
            TSeries(Self.Items[SeriesInfo^[i].Index]).AddPoint(
              SeriesInfo^[i].XValue,
              SeriesInfo^[i].YValue,
              FALSE, FALSE);
        end;
      except
        SeriesInfo^[i].XValue := -999999;
      end;
    end; {for i}
  end; {for lines of data}
{cleanup:}
  TheSortedLine.Free;

{Make the new Series visible:}
  for i := 0 to SeriesCount-1 do
  begin
    if (SeriesInfo^[i].Index >= 0) then
      TSeries(Self.Items[SeriesInfo^[i].Index]).Visible := TRUE;
  end;

{for a subsequent SaveClick:}
  ConvertTextData_Ver200 := TRUE;
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertXYZData
  Description: Processes XYZ text data
       Author: Mat Ballard
 Date created: 04/19/2001
Date modified: 04/27/2001 by Mat Ballard
      Purpose: data importation
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertXYZData(ImportForm: TImportForm; TheData: TStringList): Boolean;
var
  iColumn,
  jRow,
  NoXYZs: Integer;
  TheCell, TheLine: String;
  X, Y, Z: Single;
  pSeries: TSeries;
begin
  try
    for jRow := ImportForm.TheFirstDataLine to TheData.Count-1 do
    begin
      TheLine := TheData[jRow];
      NoXYZs := 0;
      X := -999999;
      Y := -999999;
      Z := -999999;
      for iColumn := 1 to ImportForm.InfoGrid.Rows[DATATYPE_LINE].Count-1 do
      begin
        TheCell := GetWord(TheLine, ImportForm.Delimiter);
        if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iColumn] = 'X') then
        begin
          X := StrToFloat(TheCell);
          Inc(NoXYZs);
        end
        else if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iColumn] = 'Y') then
        begin
          Y := StrToFloat(TheCell);
          Inc(NoXYZs);
        end
        else if (ImportForm.InfoGrid.Rows[DATATYPE_LINE].Strings[iColumn] = 'Z') then
        begin
          Z := StrToFloat(TheCell);
          Inc(NoXYZs);
        end;
  {test for a triple:}
        if (NoXYZs = 3) then
        begin
          NoXYZs := 0;
          pSeries := Self.GetSeriesOfZ(Z);
          if (pSeries = nil) then
          begin
            pSeries := Add;
            pSeries.ZData := Z;
          end;
          pSeries.AddPoint(X, Y, FALSE, TRUE);
        end;
      end;
    end;
    Result := TRUE;
  except
    Result := FALSE;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetItem
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/25/2002 by Mat Ballard
      Purpose: interface to Series property, which is the default property
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetItem(Index: Integer): TSeries;
begin
  if (Index < Count) then
    Result := TSeries(inherited GetItem(Index))
   else
    Result := nil;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.SetItem
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/25/2002 by Mat Ballard
      Purpose: interface to Series property, which is the default property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.SetItem(Index: Integer; Value: TSeries);
begin
  inherited SetItem(Index, Value);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetAxisList
  Description: very protected property Get function
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: restrict setting of the axis list
 Known Issues:
 ------------------------------------------------------------------------------
procedure TSeriesList.SetAxisList(Value: TCollection);
begin
  if (FAxisList = nil) then
    FAxisList := Value
   else
    EComponentError.Create('TSeriesList: AxisList has already been set !');
end; }


{------------------------------------------------------------------------------
    Procedure: TSeriesList.Update
  Description: Collection override
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: responds to changes in style of the member series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.Update(Item: TCollectionItem);
begin
  DoStyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.StyleChange
  Description: target of all of the Items[i] OnStyleChange events
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: responds to changes in style of the member series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.StyleChange(
  Sender: TObject);
begin
  if (FIgnoreChanges) then exit;

  DoStyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DataChange
  Description: target of all of the Items[i] OnStyleChange events
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: responds to changes in data of the member series
 Known Issues: get up to 3 screen re-draws
 ------------------------------------------------------------------------------}
procedure TSeriesList.DataChange(
  Sender: TObject);
begin
  if (IgnoreChanges) then exit;

  DoDataChange;
end;


end.
