unit Import;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Parser.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 04/18/2001
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To allow users to paste or import complex data into TPlot.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Buttons, ComCtrls, Controls, ExtCtrls, Forms, Graphics, Grids, StdCtrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Buttons, ComCtrls, Controls, ExtCtrls, Forms, Graphics, Grids, StdCtrls,
{$ENDIF}
{$IFDEF LINUX}
  QTypes, Types,
  QButtons, QComCtrls, QControls, QExtCtrls, QForms, QGraphics, QGrids, QStdCtrls,
{$ENDIF}
  Misc, Plotdefs;

type
  TDelimiter = (dlNone, dlTab, dlComma, dlSpace, dlColon, dlSemiColon,
                dlLineFeed, dlCarriageReturn);
  TDelimiters = array[TDelimiter] of string;
  PDelimiters = ^TDelimiters;

const
  COLUMN_NOS = 0;
  SERIES_NAMES = 1;
  DATATYPE_LINE = 2;
  DEPENDS_ON_X = 3;
  Z_DATA_LINE = 4;
  TIME_DATA_LINE = 5;
  FIRST_LINE_OF_DATA = 6;
  SERIESTYPE_LINE = 6; // ain't that sneaky ?

  {Delimiters: array[TDelimiter] of string =
    ('', #9, ',', ' ', ';', ':', #10, 'Type your own');}
  DelimiterNames: TDelimiters =
    ('None', 'Tab  ->', 'Comma  ,', 'Space   ', 'Colon  ;', 'Semicolon  :', 'Line Feed  ', 'Carriage Return  <-');
  Delimiters: TDelimiters =
    ('', #9, ',', ' ', ';', ':', #10, #13);
  DataTypes: array [0..6] of string = ('Ignore Series', 'Ignore', 'X', 'XTEXT', 'Y', 'EXTRA', 'Z');

type
  TColumnType = (ctIgnore, ctX, ctXText, ctY, ctExtra);

  TImportForm = class(TForm)
    DataListBox: TListBox;
    DelimiterLabel: TLabel;
    DelimiterComboBox: TComboBox;
    SetGroupBox: TGroupBox;
    SeriesNamesButton: TButton;
    FirstLineOfDataButton: TButton;
    HelpBitBtn: TBitBtn;
    BitBtn2: TBitBtn;
    OKBitBtn: TBitBtn;
    InfoGrid: TStringGrid;
    PickXDataComboBox: TComboBox;
    ZDataLineButton: TButton;
    NoSeriesLabel: TLabel;
    ExpandBitBtn: TBitBtn;
    StartTimeLineButton: TButton;
    DataTypeComboBox: TComboBox;
    OverlayCheckBox: TCheckBox;
    TitleEdit: TEdit;
    Title: TLabel;
    TestButton: TButton;
    procedure FormCreate(Sender: TObject);
    procedure DataListBoxClick(Sender: TObject);
    procedure FirstLineOfDataButtonClick(Sender: TObject);
    procedure SeriesNamesButtonClick(Sender: TObject);
    procedure InfoGridClick(Sender: TObject);
    procedure PickXDataComboBoxClick(Sender: TObject);
    procedure HelpBitBtnClick(Sender: TObject);
    procedure ZDataLineButtonClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure ExpandBitBtnClick(Sender: TObject);
    procedure FormClick(Sender: TObject);
    procedure StartTimeLineButtonClick(Sender: TObject);
    procedure DataTypeComboBoxClick(Sender: TObject);
    procedure DelimiterComboBoxClick(Sender: TObject);
    procedure CleanUpGrid(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure InfoGridMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
  private
    OriginalHeight: Integer;
    {DelimiterNames: array[TDelimiter] of String;}
    NumberOfSeries: Integer;
    TheCol, TheRow: Integer;
    //InstructionNo: Integer;
    XDataPresent: Boolean;
    AppHint: TNotifyEvent;
    Status: TStatusBar;

  public
{NB: Delimiters have to be public, so it cannot be a const array.}
    Delimiter: String;
    TheFirstDataLine: Integer;
    TheSeriesNamesLine: Integer;
    TheSeriesUnitsLine: Integer;
    TheZDataLine: Integer;
    TheTimeDataLine: Integer;
    CurrentLine: Integer;

    procedure DoHintsFromResource;
    procedure DisplayHint(Sender: TObject);
    procedure CalcColumns(ALine: String; iRow: Integer);
    procedure CheckDelimiter;
    procedure CalculateNoSeries;
    procedure CheckLocalXData;
  end;

var
  ImportForm: TImportForm;

implementation

{$R *.dfm}
{$R Arrows.res}

const
{  Instructions: array[1..7] of string =
   ('1. Select the first line of data, and click on the "1st line of data" button. This enables the "ZData Line" button',
    '2. Select the line containing the names of the Series, by clicking on the "Series Names" button, or just type into the Grid.',
    '3. Expand the Grid, if needed by clicking on the "Expand" button.',
    '4. Set the column type: X, XTEXT, Y, Z or Ignore, by clicking in the cell in the "X or Y Data ?" line, if needed',
    '5. For Series that depend on the X Data in others, set the column in which the X Data is located, if needed',
    '6. If there is a line (row) of Z Data, select it by clicking on the "Z Data Line" button; ditto with Start Times',
    '7. If you want to add the Z or Time Data manually, type it into the last row of the grid under the Y values');
}
  GridHints: array[COLUMN_NOS..FIRST_LINE_OF_DATA] of string =
   ('Each column of data needs to be specified',
    'Enter the Names of the Series in this row, above the first column of (X) data',
    'What type of data is this ? Or just Ignore this column ?',
    'Does this Series depend on the X Data in another (Series) Column ?',
    'Enter any Z data on this row - or just leave blank',
    'Enter any Time data on this row - or just leave blank',
    'This is the first line of data');

{------------------------------------------------------------------------------
    Procedure: TImportForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/09/2001 by Mat Ballard
      Purpose: sets the position, the Delimiters and their names, and putsw headings on the stringgrid
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.FormCreate(Sender: TObject);
var
  i: Integer;
begin
{$IFDEF DELPHI1}
  Status := TPanel.Create(Self);
  Status.Align := alBottom;
{$ELSE}
  Status := TStatusBar.Create(Self);
  Status.SimplePanel := TRUE;
{$ENDIF}
  Status.Height := 20;
  Status.Parent := Self;

  DoHintsFromResource;
{$IFDEF MSWINDOWS}
  Self.BorderStyle := bsSizeable;
{$ENDIF}
{$IFDEF LINUX}
  Self.BorderStyle := fbsSizeable;
{$ENDIF}
  Self.Scaled := FALSE;
  Self.HorzScrollBar.Visible := FALSE;
  Self.VertScrollBar.Visible := FALSE;

  Self.Left := 5;
  Self.Top := 10;
  OriginalHeight := OKBitBtn.Top + 2 * OKBitBtn.Height;
  Self.ClientWidth := ExpandBitBtn.Left + ExpandBitBtn.Width + DataListBox.Left;
  Self.ClientHeight := OriginalHeight;

  CurrentLine := 0;
  AppHint := Application.OnHint;
  Application.OnHint := DisplayHint;

  TheTimeDataLine := -1;
  TheZDataLine := -1;
  TheFirstDataLine := -1;
  TheSeriesNamesLine := -1;
  TheSeriesUnitsLine := -1;
  XDataPresent := TRUE;

  //InstructionNo := 1;
  //InstructionsLabel.Caption := Instructions[InstructionNo];

  for i := Ord(dlNone) to Ord(dlCarriageReturn) do
    DelimiterComboBox.Items.Add(DelimiterNames[TDelimiter(i)]);

  for i := 0 to 6 do
    DataTypeComboBox.Items.Add(DataTypes[i]);

  with InfoGrid do
  begin
    Cells[0, COLUMN_NOS] := 'Column Number';
    Cells[0, SERIES_NAMES] := 'Series Names';
    Cells[0, FIRST_LINE_OF_DATA] := 'First Line Of Data';
    Cells[0, DATATYPE_LINE] := 'Type of Data'; // also XTEXT, EXTRA, Ignore
    Cells[0, DEPENDS_ON_X] := 'X Data In Series:';
    Cells[0, Z_DATA_LINE] := 'Z Data';
    Cells[0, TIME_DATA_LINE] := 'Start Times';
    DefaultColWidth := 80;
    ColWidths[0] := 150;
  end;

  ExpandBitBtn.Glyph.LoadFromResourceName(HInstance, 'ARROWR');
end;

procedure TImportForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Application.OnHint := AppHint;
end;

procedure TImportForm.DisplayHint(Sender: TObject);
begin
{$IFDEF DELPHI1}
  Status.Caption := Application.Hint;
{$ELSE}
  Status.SimpleText := Application.Hint;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.DoHintsFromResource
  Description: standard loading of hints from resources
       Author: Mat Ballard
 Date created: 06/25/2001
Date modified: 06/25/2001 by Mat Ballard
      Purpose: display in different languages
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.DoHintsFromResource;
begin
  DelimiterComboBox.Hint := 'What character (s) seperate values ?';
  SeriesNamesButton.Hint := 'This is the line containing the names of the Series' + #13#10 +
    'Selecting a line will also fill in the "X, Y or Z Data" line in the grid';
  FirstLineOfDataButton.Hint := 'Make the current line the first line of XY data';
  ZDataLineButton.Hint := 'This is the line containing the Z Data + #10 +If not set, or typed in, you will be prompted for values for 3D plots.';
  StartTimeLineButton.Hint := 'This is the line containing the Start Times.';
  InfoGrid.Hint := 'Click on these cells and you might get a surprise !';
  ExpandBitBtn.Hint := 'Expand (or contract) this window horizontally to full screen width';
  DataTypeComboBox.Hint := 'Pick a Data Type for this column';
  PickXDataComboBox.Hint := 'If this Series depends X Data in another Series, pick it';
  //InstructionSpinButton.Hint := 'Scroll through the instructions using these arrow buttons';
  OverlayCheckBox.Hint := 'If this is checked, then the data will be overlaid';
  TitleEdit.Hint := 'Enter the title of this pasted data here';
end;


{------------------------------------------------------------------------------
    Procedure: TImportForm.DataListBoxClick
  Description: responds to user selection of a line
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: places the selected line into the Edit box, and checks it for Delimiters
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.DataListBoxClick(Sender: TObject);
begin
  //LineEdit.Text := DataListBox.Items[DataListBox.ItemIndex];
  CheckDelimiter;
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.CheckDelimiter
  Description: determines the Delimiter in a line
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets Delimiter
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.CheckDelimiter;
var
  iDelimiter, TheDelimiter: TDelimiter;
  TheLine: String;
begin
  TheDelimiter := dlNone;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  for iDelimiter := dlTab to dlLineFeed do
  begin
    if (Pos(Delimiters[iDelimiter], TheLine) > 0) then
    begin
      TheDelimiter := iDelimiter;
      break;
    end;
  end;
  DelimiterComboBox.ItemIndex := Ord(TheDelimiter);
  Delimiter := Delimiters[TheDelimiter];
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.CalculateNoSeries
  Description: Calculates the number of Series (Y columns)
       Author: Mat Ballard
 Date created: 04/09/2001
Date modified: 04/09/2001 by Mat Ballard
      Purpose: see Description
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.CalculateNoSeries;
var
  iColumn: Integer;
begin
  NumberOfSeries := 0;
  for iColumn := 2 to InfoGrid.ColCount-1 do
  begin
    if (InfoGrid.Cells[iColumn, DATATYPE_LINE] = 'Y') then
      Inc(NumberOfSeries);
  end;
  NoSeriesLabel.Caption := Format('There seems to be %d Series', [NumberOfSeries]);
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.FirstLineOfDataButtonClick
  Description: responds to the selection of the first line of data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fills the InfoGrid cells with the parsed line
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.FirstLineOfDataButtonClick(Sender: TObject);
var
  TheLine: String;
  iColumn: Integer;
begin
  SeriesNamesButton.Enabled := TRUE;
  ZDataLineButton.Enabled := TRUE;
  StartTimeLineButton.Enabled := TRUE;
  OKBitBtn.Enabled := TRUE;

  TheFirstDataLine := DataListBox.ItemIndex;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  CalcColumns(TheLine, FIRST_LINE_OF_DATA);

  InfoGrid.Cells[1, DATATYPE_LINE] := 'X';
  for iColumn := 2 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, DATATYPE_LINE] := 'Y';
  end;

  {InfoGrid.Cells[1, DEPENDS_ON_X] := '';
  for iColumn := 2 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, DEPENDS_ON_X] := '1';
  end;}

  CalculateNoSeries;  
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.SeriesNamesButtonClick
  Description: responds to the selection of the Series names
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fills the InfoGrid Series Names row with the parsed line
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.SeriesNamesButtonClick(Sender: TObject);
var
  iColumn, jColumn: Integer;
  TheLine: String;
begin
  TheSeriesNamesLine := DataListBox.ItemIndex;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  CalcColumns(TheLine, SERIES_NAMES);
{Guess the X Columns:}
  for iColumn := 1 to InfoGrid.ColCount-1 do
  begin
    if (Length(InfoGrid.Cells[iColumn, SERIES_NAMES]) > 0) then
    begin
      InfoGrid.Cells[iColumn, DATATYPE_LINE] := 'X';
    end;
  end;
{Guess the EXTRA Columns, if any:}
  for iColumn := 1 to InfoGrid.ColCount-1 do
  begin
    if (InfoGrid.Cells[iColumn, DATATYPE_LINE] = 'X') then
    begin
      jColumn := iColumn + 2;
      while (jColumn < InfoGrid.ColCount) do
      begin
        if (InfoGrid.Cells[jColumn, DATATYPE_LINE] = 'Y') then
        begin
          InfoGrid.Cells[jColumn, DATATYPE_LINE] := 'EXTRA';
          Inc(jColumn);
        end
         else
          break;
      end;
    end;
  end;
end;

procedure TImportForm.CalcColumns(ALine: String; iRow: Integer);
var
  iColumn: Integer;
begin
  iColumn := 1;
  while (Length(ALine) > 0) do
  begin
    if (InfoGrid.ColCount <= iColumn) then
      InfoGrid.ColCount := iColumn+1;
    InfoGrid.Cells[iColumn, iRow] :=
      GetWord(ALine, Delimiter);
    Inc(iColumn);
  end;
  //InfoGrid.ColCount := iColumn;

  for iColumn := 1 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, COLUMN_NOS] := IntToStr(iColumn);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.InfoGridClick
  Description: responds to the selection of a cell in the StringGrid
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the InfoGrid options, contents of some cells, and PickXDataComboBox position and visibility
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.InfoGridClick(Sender: TObject);
var
  Rect: TRect;
  iColumn: Integer;
begin
  with InfoGrid do
  begin
    if (((Row = SERIES_NAMES) or
         (Row = Z_DATA_LINE)) and
         (Col > 0)) then
      Options := Options + [goEditing]
     else
      Options := Options - [goEditing];

    if ((Row = DATATYPE_LINE) and (Col >= 1)) then
    begin
      TheCol := Col;
      TheRow := Row;
      {Rect := TRect(CellRect(Col, Row));}
      Rect := CellRect(TheCol, TheRow);
      DataTypeComboBox.Left :=
        InfoGrid.Left + Rect.Left;
      DataTypeComboBox.Top :=
        InfoGrid.Top + Rect.Top;
      DataTypeComboBox.Width := Rect.Right - Rect.Left;
      DataTypeComboBox.Font.Height := DataTypeComboBox.Font.Height +
        (Rect.Bottom - Rect.Top) - DataTypeComboBox.Height;
      DataTypeComboBox.ItemIndex := DataTypeComboBox.Items.IndexOf(Cells[Col, Row]);
      DataTypeComboBox.Visible := TRUE;

      CalculateNoSeries;
    end;

{OK, this is messy: if we are looking at the row that contains the dependent series indexes,
 then a series can be dependent if it has a Y column, but no X[TEXT] column
 immediately to the left:}
    if ((Row = DEPENDS_ON_X) and
        (Cells[Col, Row-1] = 'Y') and
        (Cells[Col-1, Row-1][1] <> 'X')) then
    begin
      TheCol := Col;
      TheRow := Row;
      Rect := CellRect(TheCol, TheRow);
      PickXDataComboBox.Left :=
        InfoGrid.Left + Rect.Left;
      PickXDataComboBox.Top :=
        InfoGrid.Top + Rect.Top;
      PickXDataComboBox.Width := Rect.Right - Rect.Left;
      PickXDataComboBox.Height := Rect.Bottom - Rect.Top;
      PickXDataComboBox.Clear;
      PickXDataComboBox.Items.Add('-');
      for iColumn := 1 to ColCount-1 do
      begin
        if (Length(Cells[iColumn, SERIES_NAMES]) > 0) then
          PickXDataComboBox.Items.Add(Cells[iColumn, SERIES_NAMES]);
      end;
      PickXDataComboBox.Visible := TRUE;
    end;

{Is there X Data ?}
    CheckLocalXData;
  end; {with InfoGrid}
end;

{------------------------------------------------------------------------------
    Procedure: TImportForm.PickXDataComboBoxClick
  Description: responds to the selection of a particular X Data column
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the XData column
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TImportForm.PickXDataComboBoxClick(Sender: TObject);
begin
  PickXDataComboBox.Visible := FALSE;
  if (PickXDataComboBox.ItemIndex >= 0) then
    InfoGrid.Cells[TheCol, TheRow] := PickXDataComboBox.Items[PickXDataComboBox.ItemIndex];
end;

procedure TImportForm.HelpBitBtnClick(Sender: TObject);
{$IFDEF LINUX}
var
  TheHelpFile: String;
{$ENDIF}
begin
{$IFDEF LINUX}
  TheHelpFile := 'hs' + IntToStr(HelpBitBtn.HelpContext) + '.htm';
{$ENDIF}
end;

procedure TImportForm.ZDataLineButtonClick(Sender: TObject);
var
  TheLine: String;
  iColumn: Integer;
begin
  TheZDataLine := DataListBox.ItemIndex;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  iColumn := 1;
  while (Length(TheLine) > 0) do
  begin
    if (InfoGrid.ColCount <= iColumn) then
      InfoGrid.ColCount := iColumn+1;
    InfoGrid.Cells[iColumn, Z_DATA_LINE] :=
      GetWord(TheLine, Delimiter);
    Inc(iColumn);
  end;
end;

procedure TImportForm.StartTimeLineButtonClick(Sender: TObject);
var
  TheLine: String;
  iColumn: Integer;
begin
  TheTimeDataLine := DataListBox.ItemIndex;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  iColumn := 1;
  while (Length(TheLine) > 0) do
  begin
    if (InfoGrid.ColCount <= iColumn) then
      InfoGrid.ColCount := iColumn+1;
    InfoGrid.Cells[iColumn, TIME_DATA_LINE] :=
      GetWord(TheLine, Delimiter);
    Inc(iColumn);
  end;
end;

procedure TImportForm.FormResize(Sender: TObject);
begin
  Self.ClientHeight := OriginalHeight;
  DataListBox.Width := Self.ClientWidth - 2*DataListBox.Left;
  InfoGrid.Width := Self.ClientWidth - 2*DataListBox.Left;
end;

procedure TImportForm.ExpandBitBtnClick(Sender: TObject);
begin
  if (ExpandBitBtn.Tag = 0) then
  begin
    Self.Width := Screen.Width - Self.Left;
    ExpandBitBtn.Caption := 'Contract';
    ExpandBitBtn.Tag := 1;
    ExpandBitBtn.Glyph.LoadFromResourceName(HInstance, 'ARROWL');
  end
  else
  begin
    Self.Width := SetGroupBox.Left + SetGroupBox.Width + HelpBitBtn.Left;
    ExpandBitBtn.Caption := 'Expand';
    ExpandBitBtn.Tag := 0;
    ExpandBitBtn.Glyph.LoadFromResourceName(HInstance, 'ARROWR');
  end;
end;

procedure TImportForm.FormClick(Sender: TObject);
begin
  DataTypeComboBox.Visible := FALSE;
end;

procedure TImportForm.DataTypeComboBoxClick(Sender: TObject);

  procedure FillRightWith(Str: String);
  var
    iColumn, YFound: Integer;
  begin
    YFound := 0;
    if (InfoGrid.Cells[TheCol, TheRow] = 'Y') then
      Inc(YFound);
    for iColumn := TheCol+1 to InfoGrid.ColCount-1 do
    begin
      if (InfoGrid.Cells[iColumn, TheRow] = 'X') then
        break;
      if (Pos('Ignore', InfoGrid.Cells[iColumn, TheRow]) > 0) then
        break;
      if (InfoGrid.Cells[iColumn, TheRow] = 'Y') then
        Inc(YFound);
      if (YFound >= 1) then
      begin
        if (Length(InfoGrid.Cells[iColumn, SERIES_NAMES]) > 0) then
          break;
      end;
      InfoGrid.Cells[iColumn, TheRow] := Str;
    end;
  end;

begin
  DataTypeComboBox.Visible := FALSE;
  InfoGrid.Cells[TheCol, TheRow] := DataTypeComboBox.Text;
  if (DataTypeComboBox.ItemIndex = 0) then
  begin // Ignore Series
    FillRightWith(DataTypes[1]);
  end
  else if (DataTypeComboBox.ItemIndex = 1) then
  begin // Ignore
    if ((TheCol < InfoGrid.ColCount-1) and
        (InfoGrid.Cells[TheCol+1, TheRow] = 'Y') and
        (Length(InfoGrid.Cells[TheCol+1, TheRow+1]) = 0)) then
    begin
      InfoGrid.Cells[TheCol+1, TheRow+1] := InfoGrid.Cells[1, SERIES_NAMES];  // depends on Series in Col 1
    end;
  end
  else if (DataTypeComboBox.ItemIndex = 5) then
  begin // EXTRA
    FillRightWith(DataTypes[5]);
  end;
  CalculateNoSeries;
end;

procedure TImportForm.CheckLocalXData;
var
  iColumn, iXColumn: Integer;
  LocalXDataPresent: Boolean;
  TheText: String;
begin
{Is there X Data ?}
  iXColumn := 0;
  with InfoGrid do
  begin
    LocalXDataPresent := FALSE;
    for iColumn := 1 to ColCount-1 do
    begin
      if (Cells[iColumn, DATATYPE_LINE] = 'X') then
      begin
        LocalXDataPresent := TRUE;
        iXColumn := iColumn;
        break;
      end;
    end;
    if (LocalXDataPresent) then
    begin
      if (not XDataPresent) then
{Oh merde ! we have to put ALL the X's back in !}
      begin
        TheText := IntToStr(iXColumn);
        for iColumn := 1 to ColCount-1 do
          if (Cells[iColumn, DATATYPE_LINE] = 'Y') then
            Cells[iColumn, DEPENDS_ON_X] := TheText;
      end;
    end
    else
    begin
      for iColumn := 1 to ColCount-1 do
        Cells[iColumn, DEPENDS_ON_X] := '';
    end;
    XDataPresent := LocalXDataPresent;
  end; {with InfoGrid}
end;

procedure TImportForm.DelimiterComboBoxClick(Sender: TObject);
begin
  Delimiter := Delimiters[TDelimiter(DelimiterComboBox.ItemIndex)];
end;

procedure TImportForm.CleanUpGrid(Sender: TObject);
var
  iColumn, iSeries, NoExtra: Integer;
  Names: TStringList;

  procedure BlowAway(iCol: Integer);
  var
    j, k: Integer;
  begin
    for j := 0 to InfoGrid.RowCount-1 do
      for k := iCol to InfoGrid.ColCount-2 do
        InfoGrid.Cells[k, j] :=InfoGrid.Cells[k+1, j];
    InfoGrid.ColCount := InfoGrid.ColCount - 1;
  end;

  procedure WriteSeriesType;
  begin
    case NoExtra of
      0: InfoGrid.Cells[iSeries, SERIESTYPE_LINE] := '00';
      1: InfoGrid.Cells[iSeries, SERIESTYPE_LINE] := '11';
      2: InfoGrid.Cells[iSeries, SERIESTYPE_LINE] := '32';
      4: InfoGrid.Cells[iSeries, SERIESTYPE_LINE] := '44';
    else
      EComponentError.CreateFmt('There seems to be %d EXTRA columns - this does not work', [NoExtra]);
    end;
  end;


begin
  iColumn := 1;
  Names := TStringList.Create;

  while (iColumn < InfoGrid.ColCount) do
  begin
    if (Length(InfoGrid.Cells[iColumn, SERIES_NAMES]) > 0) then
      Names.Add(InfoGrid.Cells[iColumn, SERIES_NAMES]);
    if (InfoGrid.Cells[iColumn, DATATYPE_LINE] = DataTypes[0]) then
    begin // Ignore Series
      BlowAway(iColumn);
    end
    else if (InfoGrid.Cells[iColumn, DATATYPE_LINE] = DataTypes[1]) then
    begin // Ignore
      if ((iColumn = InfoGrid.ColCount - 1) or  // Last column, or no series names
          (Length(InfoGrid.Cells[iColumn, SERIES_NAMES]) = 0)) then
        BlowAway(iColumn)
      else
      begin
        InfoGrid.Cells[iColumn, DATATYPE_LINE] := InfoGrid.Cells[iColumn+1, DATATYPE_LINE];
        InfoGrid.Cells[iColumn, DEPENDS_ON_X] := InfoGrid.Cells[iColumn+1, DEPENDS_ON_X];
        InfoGrid.Cells[iColumn, FIRST_LINE_OF_DATA] := InfoGrid.Cells[iColumn+1, FIRST_LINE_OF_DATA];
        BlowAway(iColumn+1);
      end;
    end
     else
      Inc(iColumn);
  end; {while}

{Convert dependent columns to Series (Index):}
  iColumn := 1;
  while (iColumn < InfoGrid.ColCount) do
  begin
    if (Length(InfoGrid.Cells[iColumn, DEPENDS_ON_X]) > 0) then
    begin // dependent series
      iSeries := Names.IndexOf(InfoGrid.Cells[iColumn, DEPENDS_ON_X]);
      InfoGrid.Cells[iColumn, DEPENDS_ON_X] := IntToStr(iSeries);
    end;
    Inc(iColumn);
  end; {while}

{What series type and multiplicity is this ?}
  iColumn := 1;
  iSeries := -1;
  NoExtra := 0;
{Add a row for the SeriesType}
  //InfoGrid.RowCount := InfoGrid.RowCount + 1;
  while (iColumn < InfoGrid.ColCount) do
  begin
    InfoGrid.Cells[iColumn, SERIESTYPE_LINE] := '';
    if (Length(InfoGrid.Cells[iColumn, SERIES_NAMES]) > 0) then
    begin
      if (iSeries > 0) then
        WriteSeriesType;
      iSeries := iColumn;
      NoExtra := 0;
    end;
    if (InfoGrid.Cells[iColumn, DATATYPE_LINE] = 'EXTRA') then
      Inc(NoExtra);

    Inc(iColumn);
  end; {while}
  if (NoExtra > 0) then
    WriteSeriesType;

  Names.Free;
end;

procedure TImportForm.InfoGridMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
var
  ACol, ARow: Longint;
begin
  InfoGrid.MouseToCell(X, Y, ACol, ARow);
  if ((CurrentLine <> ARow) and (ARow <= FIRST_LINE_OF_DATA)) then
  begin
    CurrentLine := ARow;
    InfoGrid.Hint := GridHints[CurrentLine];
  end;
end;

end.                                                 
