unit Plot;

{.$ DEFINE DEBUG}

{$I Plot.inc}

{$IFDEF DELPHI4}
  {$DEFINE DELPHI4_DOWN}
{$ENDIF}
{$IFDEF BCB3}
  {$DEFINE DELPHI4_DOWN}
{$ENDIF}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Plot.PAS, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999-2002 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Acknowledgements to:
    Anders Melanders
      - TGifImage (http://www.melander.dk)

    Edmund H. Hand
      - pnglib
    Jack Goman <jack@SharePower.VirtualAve.net>
      - PngUnit
    Eric Engler (http://www.geocities.com/SiliconValley/Network/2114/)
      - TPngImage in Pnglib24a

    Atanas Stoyanov (http://www.poboxes.com/astoyanov/) and his marvelous
      - MemProof

    Renate Schaaf <schaaf@math.usu.edu>, 1993
    Alin Flaider <aflaidar@datalog.ro>, 1996
    Hallvard Vassbotn <hallvard.vassbotn@c2i.net>,
      - TParser10
    Stefan Hoffmeister <Stefan.Hoffmeister@Uni-Passau.de>, 1997
      - TParser10,
      - Linux LoadStr and LoadBitmapFromResource bugfix,
      - most useful comments on the kylix newsgroups
    John Shemitz
      - second most useful comments on the kylix newsgroups,
        and the rotated Kylix text code.
    David J Taylor, Jeff Weiss and Juergen Loewner <davidtaylor@writeme.com>
      - CurveFit

    Thanks (or blame) to:
        http://www.freetranslation.com/
        for the French, German, Italian, Norwegian, Portuguese and Spanish translations.
    Thanks to Amy Khai Luong <akluong@sandia.gov> for help in pointers and solutions to D6 compatibility.
    Thanks to Rakesh Nagarajan <rakesh@pathbox.wustl.edu> for help in pointers and solutions to BCB5 compatibility.

Last Modified: 30 Sept 2003
Current Version: 3.06

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/ or
        http://tplot.sourceforge.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Kylix issues:
    - Yes, there is no printing (yet)
    - Yes, there is no GIF support (yet - but with PNG who needs it ?)
    - Note the peculiar work-arounds that apply to resources in Kylix 1:
      you have to patch QGraphics see ResourceFix.pas, or FreeClx on SourceForge.
    - No, everything else works.
    - Help does not work yet - but check out:
      http://Chemware.hypermart.net/tplot/help/ to see where we are heading

Kylix tricks and traps:
    - symlinks: put Symlinks from your home directory to your
      development and kylix directories:
      >cd ~
      >ln -s /opt/kylix
      >ln -s /opt/kylix/help kylixhelp
      >ln -s /mnt/dos/delphi/components/tplot
    - case sensitivity - filenames, resources, unit names, etc, etc, etc.
    - permissions: suggest you make $(KYLIX)/bin user-writable, and put final binaries there
    - put XKill on the taskbar - you'll need it !
    - from the command line: "ps -A", then "kill nnn"
    - CLX objects and classes don't correspond 100% to VCL ones:
      there are some old properties and methods missing, and some new ones
    - eg: QForms.TForm is different from Forms.TForm.
      This means you can open a form saved from Delphi into Linux, and vice-versa,
      _BUT_ you will get errors about non-existent properties. eg:
          Delphi bitches about:
            BorderStyle,
            HelpType,
            Color (different clXXXs),
            Font.Weight
          Kylix bitches about:
            BorderStyle,
      If you don't save a form from the current IDE, then it will hopefully compile
      (and run) OK, BUT you may get exceptions when you open a dialog.

Known Issues:
    - D7 has a weird behaviour: it occassionally insists that:
        ""
      To fix this, just click on "OK" _TWICE_, then ignore it. It does not affect the final binary.  
    - D1, D2 and D3 are no longer supported. The chosen localization technique (resourcestring)
      is not supported under D1 and D2, and we now overload certain methods, which requires D4 or above.
    - Will move to minimal D5 support soon: this will enable the use of official dynamic arrays,
      and so enable .NET compatibility
    - Printing does not (yet) work under Kylix.
    - $IFDEFs: TPlot is a cross-platform, multi-compiler component. $IFDEFs are therefore unavoidable.
    - Explicit dereferencing (eg: XData^[i]) _WAS_ required for compatibility
      with Delphi 1. Besides, I prefer it to distinguish dynamic from static arrays.
    - Since TPlotMenu, etc are useless without TPlot, there is only the one
      registration unit: TPlot_Reg.pas.
    - if you work across different versions of Delphi, ALWAYS save any form from
      the lowest version of Delphi (eg: Delphi 1), because otherwise the DFM
      files will contain extra properties that will cause stream read errors in
      lower versions of Delphi.


History:
********
The history of TPlot is documented in the file:
    History.txt
- just right-click and "Open File at Cursor"


     Comparison of TPlot with TChart
     ********************************
TChart version 4 (in Delphi 5), "Fastline" series, versus TPlot 3.00:
Windows 2000, sp3, no IDE, 10 Oct 2002
Delphi 5, PII-266/128 Dell laptop:
                             TPlot                                   TChart
No Pts        init       fps      bytes     leak      init       fps      bytes     leak
     101      0.02     29.04     30608        12      0.02     13.78     10924       300
    1001      0.02     15.97     36772         0      0.02     11.12     40664         0
   10001      0.05     15.68    110432         0      0.05      3.56    328760         0
  100001      0.26      8.83    829280         0      0.26      0.45   3486212         0
 1000001      7.24      1.71   7968608         0      7.24      0.05  33399024         0

Linux, Mandrake 9.0 (kernel 2.4.19-16.mdk), Kylix 3, no IDE:
No Pts        init       fps      bytes     leak      init       fps      bytes     leak
     101      0.28     12.88     546000 ?
 1000001      2.45      2.00   52924000

Notes:
        1. memory usage under Windows was determined using GetHeapStatus.
        2. memory usage under Linux was determined using top, as SIZE + SHARE.

Comments:
 1. TPlot is 2 - 30 x faster than TChart, and version 3.0 is a little bit slower than 2.xx.
 2. TPlot initially uses more memory than TChart. However, as the number of data
    points rises, it uses less.
 3. TPlot shines with large data sets, where it is both much faster and uses less memory.
    This makes it ideal for scientific and engineering applications.
 4. Graphics are generally faster under W2K than Linux.
 5. Memory operations are much faster under Linux, so that the Linux GUI is
    _FASTER_ than W2K for large data sets.

Feature Comparison:
                      TPlot            TChart
    Data file IO       yes               Pro version
    Extensible file IO yes               no
    Click & drag       yes               no
      objects
    Popup menus        yes               no
    Main menu          yes               no
    Toolbar            yes               no
    Context sensitive  yes               no
    Web images         yes               only GIF in Pro version
    Web pages          yes               no
    Sticky notes       yes               no
    Dynamic,           yes               no
      movable axes
    Math functions:
      Average          yes              yes
      Compression      yes               no
      Contraction      yes               no
      Differentiation  yes               no
      Integration      yes               no
      Linear Fits      yes               no
      Polynomial Fits  yes               no
      Moving Average   yes               no
      Nearest Point    yes               no
      Position         yes               no
      Smoothing        yes               no
      Splines          yes               no
      User defined     yes               no
    True 3D plots      yes               Pro version
    Database support   in development    yes

-----------------------------------------------------------------------------}
{TO BE DONE:}
{TODO 1 -o DB Expert -c Database : debug and thoroughly test TDBPlot}
{TODO 2 -o all -c Linux/Help : get the HTML Help system working under Linux}
{TODO 2 -o all -c Linux/Printing : Printing under Kylix}
{TODO 2 -o all -c Linux : fix bug that prevents Collections from functioning in the Kylix IDE}
{TODO 2 -o all -c Linux : fix bug that prevents TPlotToolbar from functioning (in Demo1)}
{TODO 2 -o Mat B -c Linux : fix bugs in Label and Title drawing}
{TODO 2 -o Mat B -c Windows/Apache : tidy up non-GUI stuff}
{TODO 2 -o Mat B -c Linux/Apache : produce a completely "X-free" version for web development using GD library}
{TODO 5 -o all -c Linux/Images : implementation of GIFs under Linux: low priority when PNG so good}
{TODO 3 -o Mat B -c Financial : addition of a "Financial" menu, similar to "File", "Edit", etc, and populate it with goodies like RSI.}
{TODO 3 -o Mat B -c Math/Science : Marry Parser10 to a good minimization algorithm (eg: Simplex) so that users can fit _ANY_ function.}
{TODO 4 -o any -c Math/Science : Polars suck: need to make nicer}
{TODO 4 -o any -c Math/Science : ptGantt type ?}
{TODO 4 -o any -c Math/Science : ptBar type ?}
{TODO 4 -o any -c Math/Science : point dragging ?}
{TODO 2 -o any -c GUI : OpenGL 3D version - T3DPlot?}
{TODO 2 -o any -c GUI : move sundry options into TOptions}
{TODO 1 -o BC++ Expert -c BC++ Builder : better BC++ support}


interface

uses
  Classes, SysUtils, ActnList, TypInfo, 
  {$IFDEF NO_MATH}Nomath,{$ELSE}Math,{$ENDIF}
{$IFDEF GUI}
  {$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Buttons, Clipbrd, Controls, Dialogs,
  Extctrls, Forms, Graphics, Menus, Printers, Stdctrls,
  {$ENDIF}
  {$IFDEF WIN32}
  Windows,
  Buttons, Clipbrd, ComCtrls, Controls, Dialogs,
  Extctrls, Forms, Graphics, Menus, Printers, Stdctrls,
  Messages,
  {$ENDIF}
  {$IFDEF LINUX}
  Types, Untranslated,
  QTypes, QButtons, QClipbrd, QComCtrls, QControls, QDialogs,
  QExtctrls, QForms, QGraphics, QImgList, QMenus, QPrinters, QStdctrls,
  {$ENDIF}

  {$IFDEF GIF}
  GIFImage,
  {$ENDIF}
  {$IFDEF PNG}
    {$IFDEF MSWINDOWS}
  Pngimage,
    {$ENDIF}
  {$ENDIF}
  Parser10,
  Aboutdlg, Axisedit, Plotzoom, Propedit, Seredit,
{$ELSE} {console mode plot:}
  Printers,
  gd, gd_io, gdWindows, gdGraphics, {gdWinapi,}
{$ENDIF}
  Axis, Data, Misc, Options3, Plotdefs, Datalist, Titles;

const
  //FileExtensions: array[1..5] of string = ('.csv', '.txt', '.htm', '.html', '.*');

  _INSTRUCTION = 'Click and Drag to move, Right Click and Drag to Act, Double-Click to edit,' + #13#10 +
    'Right-Click for Popup, and Shift-Click and Drag to Zoom In !';

  SUBHEADER = 'Subheader';

  DEF_EXTENSION = '.plot';
  PROP_EXTENSION = '.props';

  crScope = 1;
  crX = 2;

  ImageExtensions: array[0..4] of string =
    ('wmf', 'emf', 'bmp', 'gif', 'png');

{$IFDEF MSWINDOWS}
  PICTURE_TYPES =
    'Metafile (picture)|*.wmf'
    + '|Enhanced Metafile (picture)|*.emf'
    + '|Bitmap|*.bmp'
  {$IFDEF GIF}
    + '|Compuserve GIF|*.gif'
  {$ENDIF}
  {$IFDEF PNG}
    + '|Web Graphic|*.png'
  {$ENDIF}
    ;
{$ENDIF}
{$IFDEF LINUX}
  PICTURE_TYPES =
    'Metafile picture (*.wmf)'
    + '|Enhanced Metafile picture (*.emf)'
    + '|Bitmap (*.bmp)'
  {$IFDEF GIF}
    + '|Compuserve GIF (*.gif)'
  {$ENDIF}
  {$IFDEF PNG}
    + '|Web Graphic (*.png)'
  {$ENDIF}
    ;
{$ENDIF}


type
{NB: many other types are in PlotDefs.pas}
  TOnPaintEvent = procedure(Sender: TObject; ACanvas: TCanvas) of object;

  TOnFileEvent = TOnMessageEvent;
{When a file is opened or closed, the app can be notified of the new file name
 using this event.}

  TOnHeaderEvent = procedure(Sender: TObject; TheStream: TMemoryStream) of object;
{When data is opened, the "user-added" Header (eg: the run date, flow rate,
 comments, etc) is passed back to the user for processing.}

  TOnRequestHeaderEvent = TOnHeaderEvent;
{When data is saved or copied, then the "user" can add additional data via the
 Header: eg: the run date, flow rate, comments, etc.}

  TOnRequestHTMLHeaderEvent = procedure(Sender: TObject; Header: TStringList) of object;
{When data is saved or copied, then the "user" can add additional data via the
 Header: eg: the run date, flow rate, comments, etc.}

  TOnSelectionEvent = procedure(Sender: TObject; Sel: TRect) of object;
{When the user clicks and drags over a region, this event is fired.}

  TOnDualSelectionEvent = procedure(Sender: TObject; Sel1, Sel2: TRect) of object;
{When the user clicks and drags over TWO regions, this event is fired.}


{******************************************************************************}
  TPlot = class;

{******************************************************************************}
  TPlotCollection = class(TOwnedCollection)
  private
    FPlot: TPlot;
  protected
    procedure Update(Item: TCollectionItem); override;
{The standard Update method for a visual component.}
  public
    constructor Create(AOwner: TPersistent; ItemClass: TCollectionItemClass);
    property Plot: TPlot read FPlot write FPlot;
{This is the Plot to which this Collection belongs.}
  end;

{******************************************************************************}
  TAxisList = class(TPlotCollection)
  private
    function GetItem(Index: Integer): TAxis;
    function GetZAxis: TAxis;
    procedure SetItem(Index: Integer; Value: TAxis);
    procedure SetZAxis(Value: TAxis);
  protected

  public
    function Add: TAxis;
{This creates a new Axis and adds it to the Collection.}
{}
{Note that it is not "override", because the ancestor is not virtual !}
{$IFDEF DELPHI4_DOWN}
    procedure Delete(Index: Integer);
{$ENDIF}

    property Items[Index: Integer]: TAxis read GetItem write SetItem; default;
{These are the Axes in the Collection.}
    property ZAxis: TAxis read GetZAxis write SetZAxis;
{This is the Z Axis, which is (currently) unique.}
  end;

{******************************************************************************}
  TNoteList = class(TPlotCollection)
  private
    function GetItem(Index: Integer): TNote;
    procedure SetItem(Index: Integer; Value: TNote);
  protected
  public
    function Add: TNote;
    function AddEx(ACaption: String): TNote;
{This creates a new Note and adds it to the Collection.}
{}
{Note that it is not "override", because the ancestor is not virtual !}
{$IFDEF DELPHI4_DOWN}
    procedure Delete(Index: Integer);
{$ENDIF}
    property Items[Index: Integer]: TNote read GetItem write SetItem; default;
{These are the Notes in the Collection.}
  end;

{******************************************************************************}
  TTextList = class(TPlotCollection)
  private
    function GetItem(Index: Integer): TText;
    procedure SetItem(Index: Integer; Value: TText);
  protected
  public
    function Add: TText;
    function AddEx(ACaption: String): TText;
{This creates a new Text box and adds it to the Collection.}
{}
{Note that it is not "override", because the ancestor is not virtual !}
{$IFDEF DELPHI4_DOWN}
    procedure Delete(Index: Integer);
{$ENDIF}
    property Items[Index: Integer]: TText read GetItem write SetItem; default;
{These are the Text boxes in the Collection.}
  end;


{******************************************************************************}
{$IFDEF GUI}
{this is the normal component}
  TCustomPlot = class(TCustomPanel)
{$ELSE}
{this is the highly specialized component for Apache developers who want a
 graphing component that works under bash alone and requires no GUI:}
  TCustomPlot = class(TComponent)
{$ENDIF}
  private
{property editor variables:}
    FAbout: String;
    FAxesProperties: String;
    FDataProperties: String;
    FProperties: String;
    FSeriesProperties: String;

{LISTS:
 The Axes is created and managed in the Plot unit and TCustomPlot component.
 The specific axes are:
   0 .. X Axis
   1 .. Primary Y Axis
   2 .. Secondary Y Axis
   3 .. Tertiary Y Axis
   4 .. etc.}
    FAxes: TAxisList;
    FNotes: TNoteList;
    FTexts: TTextList;

    //FActionList: TActionList;

{Here are all the private member properties that GUI ancestors usually supply,
 but must be created in the non-GUI version.}
{$IFNDEF GUI}
    FCanvas: TCanvas;
    FColor: TColor;
    FFont: TFont;
{This is the font of the hint-like message at the bottom of the graph.}
    FHeight: Integer;
    FWidth: Integer;
{$ENDIF}

    FColor2: TColor;
    FColorPlot: TColor;
    FColorType: TGridType;
    FBorder: TBorder;
    // FBubbleSize: TPercent; now in TSeries
    FColumnGap: TPercent;
    FContour: TContour;
    FCreatedBy: String;
    FDescription: String;
    FDefaultExtension: String;
    FDisplayMode: TDisplayMode;
    FDisplayHistory: Single;
    FEditable: Boolean;

    FFileName,      {D:\Data\Delphi\Plot\Test3.csv}
{where are the file types ?}
      mOpenDriveDir: String;      {NB: SavePath == FileDriveDir}
{$IFDEF GUI}
      mOverlayDriveDir: String;   {T:\Projects\}
      mImageDriveDir: String;     {D:\Data\Images}
{$ENDIF}
    FFileSavePrompt: Boolean;
{Overlay Management:}
    FFirstOverlay: Integer;

    FGrids: TGrids;
    FHelpFile: String;
    FHighFont: TFont;
    FInstructions: String;
{The Legend of Series}
    FLegend: TLegend;
    FMovable: Boolean;
    FPieRowCount: Byte;
{$IFDEF GUI}
    //FPlotMenu: TMainMenu;
    //FPlotToolBar: TToolBar;
    FPlotActionList: TActionList;
{$ENDIF}
    FPlotType: TPlotType;
    FPolarRange: Single;
    FPrintOrientation: TPrinterOrientation;
    FOutlineWidth: Integer;
{The results of things like least-squares fits:}
    FSaveOptions: TSaveOptions;
    FScreenJob: TScreenJob;
    FStatus: set of TStatus;
    FStartTime: TDateTime;
    //FUpdating: Boolean;
    FTitle: TTitle;
    //FXAxis: TAxis;
    FXYFastAt: Longint;
    //FYAxis: TAxis;
    //ZAxis: TAxis;
    //FZRatio: TPercent; // the length of the Z Axis relative to the X and Y Axes;
    FZLink: Boolean;
    FZSeriesNames: Boolean;

{events:}
    FOnAfterPaint: TOnPaintEvent;
    FOnAfterDraw: TOnPaintEvent;
    FOnBeforePaint: TOnPaintEvent;
    FOnBeforeDraw: TOnPaintEvent;
    FOnStyleChange: TNotifyEvent;
    FOnDataChange: TNotifyEvent;
    FOnFileNew: TNotifyEvent;
    FOnFileOpen: TOnFileEvent;
    FOnFileClose: TOnFileEvent;
    FOnHeader: TOnHeaderEvent;
    FOnHeaderRequest: TOnRequestHeaderEvent;
    FOnHTMLHeaderRequest: TOnRequestHTMLHeaderEvent;
    FOnSelection: TOnSelectionEvent;
    FOnDualSelection: TOnDualSelectionEvent;
    FOnWarning: TOnWarningEvent;

    //mBorderList: TBorderList;
{the four borders, clickable by the user:}
(*
    mLeftBorder: TAngleRect;
    mTopBorder: TAngleRect;
    mRightBorder: TAngleRect;
    mBottomBorder: TAngleRect;
*)

    mClientRect: TRect;

{extremely private members:}
{the current, which depends on the file type and is ORed with FSaveOptions}
    mAsText: Boolean;
{What file types ?}
    mOpenFilterIndex,
    mSaveFilterIndex,
    mImageFilterIndex: Integer;

{$IFDEF GUI}
    mPageButtons: array[0..3] of TBitmap;
{$ENDIF}
{Ignore changes in sub-components during painting:}
    mPaintLock: Integer;
{Has Loaded been called ?}
    //mNOTLoaded: Boolean;
{Do we need to clean up the screen (click) management ?}
    mZeroStuff: Boolean;
{Exactly how many series was there when we started the Open/Overlay/Paste ?}
    mInitialSeriesCount: Integer;
{Shall we forbid any Name changes ?}
    mNameFrozen: Boolean;

{The list of (TRectangle descended) objects on screen:}
    ScreenObjectList: TList;
    //NoBasicScreenObjects: Integer;

{$IFDEF GUI}
{The timer to start click-and-drag operations:}
    MouseTimer: TTimer;
{the position of the mousedown:}
    MouseStart: TPoint;
{the starting position of the object:}
    ClickedObjectOffset: TPoint;
{$ENDIF}

{which object(s) were clicked ?}
    FClickedObjectType: TObjectType;
    pClickedObject: TCollectionItem;
    SecondClickedObjectType: TObjectType;
    pSecondClickedObject: TCollectionItem;

{The rectangular outline of a ScreenObject that is dragged around the screen:}
    Selection: TAngleRect;
{$IFDEF GUI}
{The first selection in a Dual Selection operation:}
    Sel1: TRect;
    Sel2: TRect;
{Popup menus:}
    //FPlotPopUpMenu: TPopupMenu;
    FDragPopUpMenu: TPopupMenu;
    WhichPopUpMenu: TPopupMenu;
    WhichPopUpItems: array[0..1] of TMenuItem;
{$ENDIF}

{the currently selected (eg: by a mouse-click) series:}
    TheSeries: Integer;
    FClickedSeries: TSeries;
{$IFDEF GUI}
    ThePointNumber: Integer;

{The clipboard format number for HTML:}
    ClipBoardFormatForHTML: Integer; {aka CF_HTML}
{$ENDIF}

    FileExtensions: array[0..5] of String;
   {= (
    FDefaultExtension,
    'csv',
    'txt',
    'htm',
    'html',
    '*');}

    FileTypes: String;
  {=
    'Plot Files|*.' + FDefaultExtension
    + '|Comma Sep Var Files|*.csv'
    + '|Text Files|*.txt'
    + '|HTML Files|*.htm;*.html'
    + '|All Files|*.*';}

{Get functions}
{$IFDEF GUI}
    function GetClickAndDragDelay: Integer;
  {$IFDEF COMPILER4_UP}
{while TImageList exists from Delphi 2 onwards, TMenu and TPopupMenu
 do not have an Images property until Delphi 4}
    //function GetImages: TImageList; {TCustomImageList}
    //procedure SetImages(Value: TImageList);
  {$ENDIF}
    function GetAxisFromUser(StartAxis: Word): TAxis;
    function GetSeriesFromUser: TSeries;
    function GetNoteFromUser(DoDelete: Boolean): TText;
{$ENDIF}
    function GetNoSeries: Word;
    function GetNoYAxes: Integer;
    function GetSeries(Index: Integer): TSeries;
    function GetFilterIndex(Ext: String): Integer;

{SetProcedures:}
{The main geometry manager.}
    procedure SetAxisDimensions(ACanvas: TCanvas);
{Sets the behaviour upon adding new data points.}
    procedure SetDisplayModeHistory(HistoryValue: Single; ScalingValue: TDisplayMode);

{$IFDEF GUI}
  {$IFDEF MSWINDOWS}
    {Puts CreatedBy and Description into the metafile.}
    procedure SetMetafileDescription;
  {$ENDIF}
{$ENDIF}

{Responding to mouse events, click & drag:}
{$IFDEF GUI}
    procedure GetTheClickedObject(X, Y: Integer);
    procedure MouseTimeOut(Sender: TObject);
    procedure MoveTheClickedObjectTo(X, Y: Integer);
    procedure OutlineTheClickedObject;
    procedure StretchTheClickedObjectTo(X, Y: Integer);
    procedure SwapEnds;
{These two respond to a user choice between overlaying objects:}
    procedure MoveTheClickedObjectClick(Sender: TObject);
    procedure MoveSecondClickedObjectClick(Sender: TObject);
{Sets the width of the outline for screen objects like lines: axes and borders.}
{$ENDIF}


{$IFDEF GUI}
  {$IFDEF WIN32}
{This does not fire. Anyone know why ? TPanel}
    procedure WMKeyDown(var Message: TWMKey); message WM_KEYDOWN;
  {$ENDIF}
{$ENDIF}

{------------------------------------------------------------------------------}
  protected
    FSeriesList: TSeriesList;
{Huh ?! Why protected for both the internal variable and the property ?}
{Because the property is later Published in TPlot, and TDBPlot needs access to
 the internal variable, but still has to hide the property.}

    function GetXAxis: TAxis;
    function GetYAxis: TAxis;
    function GetZAxis: TAxis;

    function GetZAngle: Word;
    {function GetZLength: Word;}
{Property Set Procedures:}
{$IFNDEF GUI}
    procedure SetColor(Value: TColor);
    procedure SetFont(Value: TFont);
    procedure SetHeight(Value: Integer);
    procedure SetWidth(Value: Integer);
{$ENDIF}
    procedure SetColor2(Value: TColor);
    procedure SetColorPlot(Value: TColor);
    procedure SetColorType(Value: TGridType);
    procedure SetName(const NewName: TComponentName); override;

    procedure SetXAxis(Value: TAxis);
    procedure SetYAxis(Value: TAxis);
    procedure SetZAxis(Value: TAxis);

    procedure SetZLink(Value: Boolean);
    procedure SetZSeriesNames(Value: Boolean);
    procedure SetZAngle(Value: Word);
    (*procedure SetZLength(Value: Word);
    procedure SetZRatio(Value: TPercent);*)
    procedure SetXYFastAt(Value: Longint);
    //procedure SetBubbleSize(Value: TPercent);
    procedure SetColumnGap(Value: TPercent);
    procedure SetDefaultExtension(Value: String);
    procedure SetDisplayMode(Value: TDisplayMode);
    procedure SetHistory(Value: Single);
    procedure SetInstructions(Value: String);
{Handles file names for saving and opening:}
    procedure SetFileName(Value: String);
    procedure SetNoYAxes(Value: Integer);
    procedure SetPlotType(Value: TPlotType);
      procedure RemoveZAxis;
    procedure SetPolarRange(Value: Single);
    procedure SetOutlineWidth(Value: Integer);
    procedure SetPieRowCount(Value: Byte);
    procedure SetNoSeries(Value: Word); virtual;
    procedure SetAxes(Value: TAxisList);
    procedure SetNotes(Value: TNoteList);
    procedure SetSeriesList(Value: TSeriesList);
    procedure SetStartTime(Value: TDateTime);

    procedure SetTexts(Value: TTextList);
    //procedure SetActionList(Value: TActionList);
{$IFDEF GUI}
    procedure SetClickAndDragDelay(Value: Integer);
{$ENDIF}
    procedure SetOnSelection(Value: TOnSelectionEvent);
    procedure SetOnDualSelection(Value: TOnDualSelectionEvent);

{Copying data:}
{$IFDEF GUI}
    procedure CopyText; virtual;
{Copies the data as tab-delimited text to the Clipboard, with any
 TCustomPlot.Owner added header.}
    procedure CopyHTML(Format: Word); virtual;
{Copies the data as HTML to the Clipboard in CF_HTML format, with any
 TCustomPlot.Owner added header.}
{Copying pictures:}
    procedure CopyBitmap; virtual;
{Does what it says.}
    procedure CopyDrawing(Enhanced: Boolean); virtual;
{Does what it says.}
{$ENDIF}


{$IFDEF GUI}
    function GetBitmap: TBitmap; virtual;
{Does what it says.}

  {$IFDEF MSWINDOWS}
    function GetDrawing: TMetafile; virtual;
  {$ENDIF}
  {$IFDEF LINUX}
    function GetDrawing: TDrawing; virtual;
  {$ENDIF}
  {$IFDEF GIF}
    function GetGIF: TGIFImage; virtual;
    function GetGIFSize(W, H: Integer): TGIFImage; virtual;
  {$ENDIF}
{$ENDIF}

{$IFDEF GUI}
  {$IFDEF PNG}
    {$IFDEF MSWINDOWS}
    function GetPng: TPngImage; virtual;
    function GetPngSize(W, H: Integer): TPngImage; virtual;
      {$ENDIF}
      {$IFDEF LINUX}
    function GetPng: TBitmap; virtual;
    function GetPngSize(W, H: Integer): TBitmap; virtual;
    {$ENDIF}
  {$ENDIF}
{$ELSE}
    function GetPng: Pointer; virtual;
    function GetPngSize(W, H: Integer): Pointer; virtual;
{$ENDIF}

{$IFDEF GUI}
    procedure CreateMenus;
{Creates the three popup menus.}
    procedure CreatePageButtons;
    procedure DestroyPageButtons;
    procedure PageButtonClick(Index: Integer);
{$ENDIF}

    procedure DoStyleChange(Sender: TObject); dynamic;
    procedure DoOutOfBounds(Sender: TObject; iX, iY: Integer); dynamic;
    procedure DoDataChange(Sender: TObject); dynamic;
    procedure DoFileClose(AFileName: String); dynamic;
    procedure DoFileOpen(AFileName: String); dynamic;
    procedure DoHeader(TheStream: TMemoryStream); dynamic;
    procedure DoHeaderRequest(TheStream: TMemoryStream); dynamic;
    procedure DoHTMLHeaderRequest(TheHeader: TStringList); dynamic;
    procedure DoSelection(Sel1: TRect); dynamic;
    procedure DoDualSelection(Sel1, Sel2: TRect); dynamic;

{$IFDEF GUI}
    procedure DblClick; override;
{Some items - ie: the Title Captions, can be edited directly.}
    {procedure KeyDown(var Key: Word; Shift: TShiftState); override;}
{This processes certain key strokes.}
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
{Implements dragging with the mouse.}
    procedure MouseMove(Shift: TShiftState; X,Y: Integer); override;
{Further implements dragging with the mouse.}
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X,Y: Integer); override;
{Implements dragging with the mouse, and right clicking for a popup menu.}
    procedure ProcessClickedObject(ObjectType: TObjectType; pObject: TCollectionItem); virtual;
{Applies effects of clicking and dragging to the selected object.}
    procedure Paint; override;
{The normal paint procedure. Most of the work is done in Draw.}
    procedure PaintInstructions;
{Just paints the instructions.}
{$ENDIF} {GUI}

    procedure PaintLock;
{This increments the paint lock number to prevent painting.}
    procedure PaintUnLock;
{This decrements the paint lock number to prevent painting.}

{Paint and Draw:}
    procedure DrawGrid(ACanvas: TCanvas);
{This draws grid for xy-type plots - it is called by Draw.}
    procedure DrawWalls(ACanvas: TCanvas);
{This draws the XY, (XZ and YZ) walls for 2D (and 3D) plots - it is called by Draw.}
    procedure FillWall(ACanvas: TCanvas; Col1, Col2: TColor; FillType: TGridType; TheWall: TQuad);
{This draws the XY, XZ and YZ grids for 3D plots - it is called by Draw.}
    procedure ZeroScreenStuff; dynamic;
{This re-initializes all the mouse-selection related variables}

{$IFDEF GUI}
    //procedure DetermineMenuVisibility;
{Sets the Visibility of the items of the internal Popup Menu.}
    //procedure SetSeriesVisibility(Value: Boolean);
{Makes series-related menu items visible or otherwise}
    //procedure SetSeriesEnabledness(TheMenu: TMenu);
{Makes the Series-related items of an external menu Enabled or otherwise}
{$ENDIF}

{Properties that will be published in the non-custom descendant:}
    Property About: string read FAbout write FAbout stored False;
{Displays the "About" dialog box for this component.}
    Property AxesProperties: string read FAxesProperties write FAxesProperties stored False;
{Displays the "Axis Editor" dialog box for the FAxes subcomponent.}
    Property ClickedObjectType: TObjectType read FClickedObjectType;
{The type of object that has just been clicked.}
    Property ClickedSeries: TSeries read FClickedSeries;
{The series that has just been selected by clicking or picking.}
    Property DataProperties: string read FDataProperties write FDataProperties stored False;
{Displays the "Data Editor" dialog box.}
    Property FirstOverlay: Integer read FFirstOverlay;
{The index of the first overlay.}
    Property Properties: string read FProperties write FProperties stored False;
{Displays the "Property Editor" dialog box.}
    Property SeriesProperties: string read FSeriesProperties write FSeriesProperties stored False;
{Displays the "Series Editor" dialog box.}

    Property Axes: TAxisList read FAxes write SetAxes; // stored False;
{This is a list of all the axes. The 0th is X, the 1st Y.}
    Property Notes: TNoteList read FNotes write SetNotes;
{This is a list of all the Notes: TNote.}
    Property Texts: TTextList read FTexts write SetTexts; {.$ IFDEF DELPHI2_UP stored FALSE $.ENDIF}
{This is a list of all the Text: TText boxes.}

    //Property ActionList: TActionList read FActionList write SetActionList; // stored False;
{This is a list of all the Plot actions.}

{$IFDEF GUI}
    Property Bitmap: TBitmap read GetBitmap;
{$ENDIF}

{$IFNDEF GUI}
    property Canvas: TCanvas read FCanvas write FCanvas;
    property Color: TColor read FColor write SetColor;
    property Font: TFont read FFont write SetFont;
{This is the font of the hint-like message at the bottom of the graph.}
    property Height: Integer read FHeight write SetHeight;
    property Width: Integer read FWidth write SetWidth;
{$ENDIF}

    property Color2: TColor read FColor2 write SetColor2 default clWhite;
{This is the second wall color. If it differs from Color, then we get a gradient.
 NB: TControl.Color already exists !}
    property ColorPlot: TColor read FColorPlot write SetColorPlot default clWhite;
{This is the Plot color. If it differs from Color, then we see it.}
    property ColorType: TGridType read FColorType write SetColorType default gtHorizontal;
{If the Color and Color2 are differernt, how are they drawn ?}
    property Contour: TContour read FContour write FContour;
{Contour is a persistent object that contains numerous properties relevant to
 contour plots: color schemes, detail, wireframes, etc.}
    property Grids: TGrids read FGrids write FGrids;
{Grids is a persistent object that contains numerous properties relevant to
 Grids and Walls: color schemes, pens, etc.}

{This returns a Bitmap of the Plot.}
{$IFDEF GUI}
  {$IFDEF MSWINDOWS}
    Property Drawing: TMetafile read GetDrawing;
  {This returns a Metafile/Drawing of the Plot.}
  {$ENDIF}
  {$IFDEF LINUX}
    Property Drawing: TDrawing read GetDrawing;
  {This returns a Metafile/Drawing of the Plot.}
  {$ENDIF}
  {$IFDEF GIF}
    Property GIF: TGIFImage read GetGIF;
{This returns a Bitmap of the Plot IN GIF FORMAT.}
  {$ENDIF}
{$ENDIF}

{$IFDEF PNG}
  {$IFDEF MSWINDOWS}
    Property Png: {$IFDEF GUI}TPngImage{$ELSE}Pointer{$ENDIF} read GetPng;
  {$ENDIF}
  {$IFDEF LINUX}
    Property Png: {$IFDEF GUI}TBitmap{$ELSE}Pointer{$ENDIF} read GetPng;
{This returns a Bitmap of the Plot IN PNG FORMAT.}
  {$ENDIF}
{$ENDIF}

{$IFDEF GUI}
    //Property PlotPopUpMenu: TPopupMenu read FPlotPopUpMenu stored FALSE;
{The public exposure of the popupmenu for PlotMenu's use.}
{$ENDIF}
    Property Series: TSeriesList read FSeriesList write SetSeriesList; {? default}
{This is the data in a list of data series.}

    Property ScreenJob: TScreenJob read FScreenJob write FScreenJob stored False;
{This is the job that is in progress in response to user input.}

{Displays the "Series Editor" dialog box.}
    Property Border: TBorder read FBorder write FBorder;
{Manages the geometry of TCustomPlot: where the axes are, where they can go, etc.}
    Property ColumnGap: TPercent read FColumnGap write SetColumnGap;
{This is the percentage gap between groups of Columns in ptColumn PlotType.}
    Property HelpFile: String read FHelpFile write FHelpFile;
{When this is set to "", TPlot uses Plot.hlp as its context-sensitive help file.}
{}
{When set to some other file, it uses that instead.}
    Property Instructions: String read FInstructions write SetInstructions; // stored FALSE;
{This is a message to the user, at the bottom-left of the graph, in the current
 Font. It disappears on a MouseDown event.}

{$IFDEF GUI}
    Property ClickAndDragDelay: Integer read GetClickAndDragDelay write SetClickAndDragDelay;
{The delay (in milliseconds) before a clicked object becomes draggable.}
{$ENDIF}

    Property CreatedBy: String read FCreatedBy write FCreatedBy;
{A string that is stored in the metafile description.}
{}
{This is also used as the "Author" value in a PNG file, if PNG is defined.}
    Property Description: String read FDescription write FDescription;
{A string that is stored in the metafile description.}
{}
{This is also used as the "Description" value in a PNG file, if PNG is defined.}

    Property DisplayHistory: Single read FDisplayHistory write SetHistory;
{The width of the X Axis when in History mode.}
    Property DisplayMode: TDisplayMode read FDisplayMode write SetDisplayMode;
{See TDisplayMode.}
    Property Editable: Boolean read FEditable write FEditable;
{Are screen objects like axes Editable ?}

    Property DefaultExtension: String read FDefaultExtension write SetDefaultExtension;
{What is the default extension of Plot files ?}

    Property FileName: String read FFileName write SetFileName stored False;
{This is the FileName to which the data is saved, or opened from.}
{}
{If FileName is blank, and an OpenClick or SaveClick occurs, then the standard
 file dialog box appears to let the user pick a name.}
    Property FileSavePrompt: Boolean read FFileSavePrompt write FFileSavePrompt default FALSE;
{When TPlot is destroyed, do we prompt to save changed files ?}

    property HighFont: TFont read FHighFont write FHighFont;
{The font for annotation of the Highs and Lows.}
{$IFDEF GUI}
  {$IFDEF COMPILER4_UP}
    //property Images: TImageList read GetImages write SetImages; {TCustomImageList}
{The images for the popupmenu.}
  {$ENDIF}
{$ENDIF}
    property Legend: TLegend read FLegend write FLegend;
{The list of series with their line styles. This is a moveable, on-screen object.}
    //Property Link: TLink read FLink write FLink;
{Another composite property that contains properties relevant to drawing links
 between series on screen.}
    Property PlotType: TPlotType read FPlotType write SetPlotType;
{What type of plot is this ?}
    Property PolarRange: Single read FPolarRange write SetPolarRange;
{What does 360 correspond to in a polar graph ? Examples are:
    2 Pi (6.28...),
    360 (degrees),
    60 (minutes),
    24 (hours)
    100 (%)
Get the idea ?    }
    Property Movable: Boolean read FMovable write FMovable;
{Are screen objects like axes movable ?}
    Property PieRowCount: Byte read FPieRowCount write SetPieRowCount default 1;
{The number of rows of Pie Graphs.}
    Property NoSeries: Word read GetNoSeries write SetNoSeries stored FALSE;
{The number of Series. Setting this can both create new series or free existing series.}
    Property NoYAxes: Integer read GetNoYAxes write SetNoYAxes stored FALSE;
{The total number of Y Axes (primary, secondary, tertiary, etc).}
    Property PrintOrientation: TPrinterOrientation
      read FPrintOrientation write FPrintOrientation;
{Shall we print the graph in Landscape or Portrait mode ?}
    Property OutlineWidth: Integer read FOutlineWidth write SetOutlineWidth;
{This is the width of the outline for screen objects like lines: axes and borders.}
    Property SaveOptions: TSaveOptions read FSaveOptions write FSaveOptions;
{Shall we save the data as Text or binary ?}
{}
{Shall we also save the Plot properties when we save the data ?
 If we do, then we:
    1. Save the properties in a seperate file;
    2. Look for a properties file to open.}
    //Property Updating: Boolean read FUpdating write FUpdating;
{Is the component updating itself ? If it is, then we not not redraw.}
    property StartTime: TDateTime read FStartTime write SetStartTime;
{The time at which data addition started. Only used when XAxis is drawn with
 units of Absolute Time. }
    Property Title: TTitle read FTitle write FTitle;
{The title of the graph, including its geometry, font and visibility.}
    Property XAxis: TAxis read GetXAxis write SetXAxis stored FALSE;
{This is the X Axis. Every nice graph should have an X Axis.}
    Property YAxis: TAxis read GetYAxis write SetYAxis stored FALSE;
{This is the Y Axis. Every nice graph should have at least one Y Axis.}
{}
{Each Series must know what Y Axes it is being plotted against:
 Primary (this one) or Secondary.}
    Property ZAxis: TAxis read GetZAxis write SetZAxis;
{This is the Z Axis. Every nice 3D graph should have an Z Axis.}
{}
{This is fairly experimental at the moment, so use with caution.
 It is also nil with non-3D PlotTypes.}
(*    Property ZLength: Word read GetZLength write SetZLength stored FALSE;
{The Length made by the Z Axis, if any, with the vertical, in a clockwise direction.}
    Property ZRatio: TPercent read FZRatio write SetZRatio default 40;
{The length of the Z Axis relative to the X and Y Axes.}  *)
    Property XYFastAt: Longint read FXYFastAt write SetXYFastAt default 10000;
{At how many data points should we switch to a fast drawing alforithm.}
    Property ZAngle: Word read GetZAngle write SetZAngle default 225;
{The angle made by the Z Axis, if any, with the vertical, in a clockwise direction.}
    Property ZLink: Boolean read FZLink write SetZLink default TRUE;
{Should we link 3D series together in the Z direction ?}
    property ZSeriesNames: Boolean read FZSeriesNames write SetZSeriesNames default FALSE;
{Should the Z Axis display the names of the Series ?}

{Events:}
    Property OnAfterPaint: TOnPaintEvent read FOnAfterPaint write FOnAfterPaint;
    Property OnAfterDraw: TOnPaintEvent read FOnAfterDraw write FOnAfterDraw;
    Property OnBeforePaint: TOnPaintEvent read FOnBeforePaint write FOnBeforePaint;
    Property OnBeforeDraw: TOnPaintEvent read FOnBeforeDraw write FOnBeforeDraw;

    Property OnStyleChange: TNotifyEvent read FOnStyleChange write FOnStyleChange;
    Property OnDataChange: TNotifyEvent read FOnDataChange write FOnDataChange;

    Property OnFileNew: TNotifyEvent read FOnFileNew write FOnFileNew;
{When a New file is created, the app can be notified of this using this event.}

    Property OnFileOpen: TOnFileEvent read FOnFileOpen write FOnFileOpen;
{When a file is opened, the app can be notified of the new file name using this event.}

    Property OnFileClose: TOnFileEvent read FOnFileClose write FOnFileClose;
{When a file is closed, the app can be notified of the new file name using this event.}

    Property OnHeader: TOnHeaderEvent read FOnHeader write FOnHeader;
{When data is opened, this event passes the header information back to the "user".}

    Property OnHeaderRequest: TOnRequestHeaderEvent read FOnHeaderRequest write FOnHeaderRequest;
{When data is saved or copied, this event allows the user to add a header
 to the data.}

    Property OnHTMLHeaderRequest: TOnRequestHTMLHeaderEvent read FOnHTMLHeaderRequest write FOnHTMLHeaderRequest;
{When data is copied as HTML, this event allows the user to add a header
 to the data.}

    Property OnSelection: TOnSelectionEvent read FOnSelection write SetOnSelection;
{When the user selects a region, then this event is fired.}
{}
{Note that after firing, this event is set to nil, so you have to reset it every usage.}
    Property OnDualSelection: TOnDualSelectionEvent read FOnDualSelection write SetOnDualSelection;
{When the user selects TWO regions, then this event is fired.}
{}
{Note that after firing, this event is set to nil, so you have to reset it every usage.}

    Property OnWarning: TOnWarningEvent read FOnWarning write FOnWarning;
{This is used for general error messages.}

{------------------------------------------------------------------------------}
  public
    Constructor Create(AOwner: TComponent); override;
{The usual Constructor, where sub-components are created, properties set, etc.}
    Destructor Destroy; override;
{The usual Destructor, where sub-components are destroyed.}
    procedure Loaded; override;

    procedure Clear(Cancellable: Boolean); virtual;
{Saves any changed files (at user request) and then clears the SeriesList.}

    procedure Draw(ACanvas: TCanvas); virtual;
{This draws the graph on a Canvas. The canvas can be:}
{}
{    1. Self.Canvas - ie: on screen;}
{    2. a Bitmap, for copying and saving.}
{    3. a MetafileCanvas, for copying and saving.}
{    4. the printer.}
{}
{It is called by Paint, and also many of the copying and saving routines.}

    procedure Resize; {$IFDEF GUI}override;{$ENDIF}
{The normal Resize procedure - it manages screen geometry.}
{$IFNDEF GUI}
    procedure Refresh;
{$ENDIF}

    function GetFileDriveDir: String;  {D:\Data\Delphi\Plot}

    function GetPlotTypeAsString: String;

{$IFDEF GUI}
    procedure ShowAbout(Sender: TObject);
{This displays the components "About" dialog box.}
    procedure ShowAxes;
{This displays the properties of the Series in the SeriesList.}
    procedure ShowData;
{This displays the properties of the Data in the DataList.}
    procedure ShowProperties;
{This displays the properties of the Plot.}
    procedure ShowSeries;
{This displays the properties of the Series in the SeriesList.}
{$ENDIF}

{Wrappers for the SeriesList functions; the first three also set the
 OnStyleChange event and the Visibility property:}
    function AddSeries(XSeriesIndex: Integer): TSeries; virtual;
{This adds a new, empty series to the list.}
    function AddExternalData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): TSeries; virtual;
{This adds a new, empty series to the list, and sets its data to point to the
 external XPointer, YPointer data.}
    function AddInternalData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): TSeries; virtual;
{This adds a new, empty series to the list, and copies the data from the
 XPointer, YPointer data.}
    function CloneSeries(TheSeries: Integer): TSeries; virtual;
{This adds a new, empty series to the list, copies the data and properties from
 TheSeries into the new clone, and changes the color and Y Displacement.}
    procedure DeleteSeries(Index: Integer); virtual;
{This deletes TheSeries from the list.}

    procedure AddData(NoPoints: Integer; XYArray: pXYArray); virtual;
{Add a slice of readings to the internal series.
 These will become the Nth points in every series.}
{}
{This method is extremely useful for data acquisition.}

    procedure Trace;
{This draws all the Series in an erasable mode without re-drawing the Axes,
 Titles or Legend.}
{}
{More specifically, the first call to Trace draws the Series, the second erases them.}
{}
{This is useful for rapidly changing data such as an oscilloscope. To use it:}
{}
{    1. Set up the Axes, etc.}
{    2. Create each Series you need.}
{    3a. Use AllocateNoPts for each one, or:}
{    3b. Allocate and manage the memory yourself, then use PointToData.}
{    4. Dump you data into each Series via the XData and YData properties.}
{    5. Trace the data.}
{    6. Get more data.}
{    7. Trace the data again to erase the old image.}
{    8. Repeat steps (4)-(7) indefinitely.}
{}
{Note that in step (4), you can either use the XData and YData properties as:}
{}
{    a. FClickedSeries.XData^[i];}
{    b. pX, pY: pSingle; pX := FClickedSeries.XData; pY := FClickedSeries.YData;}

{All of the xxxClick methods}
{$IFDEF GUI}
    procedure ProcessKeyDown(var Key: Word;
  Shift: TShiftState);
{This processes a keystroke sent to TPlot by the main application.}
    procedure CopyClick(Sender: TObject);
{This responds to a user selection of "Copy" in the popupmenu,
 and copies the data as text, as a bitmap, and as an enhanced metafile.}
    procedure PrintClick(Sender: TObject);
{This prints the graph.}
    procedure ShowAllClick(Sender: TObject);
{This shows (.Visible := TRUE) ALL objects.}
    procedure HideClick(Sender: TObject);
{This hides (.Visible := FALSE) the selected object.}
    procedure ShowHideAllSeriesClick(Sender: TObject; IsVisible: Boolean);
{This hides (.Visible := FALSE) all the series.}
    procedure PositionClick(Sender: TObject);
{This reports the Position of the mouse (right) click.}
    procedure NearestPointClick(Sender: TObject);
{This reports the Position of the nearest data series point to the mouse
 (right) click.}
    procedure DeleteSeriesClick(Sender: TObject);
{This deletes the currently selected series.}
    procedure CopySeriesClick(Sender: TObject);
{This copies the selected series to the clipboard in tab-delimited form.}
    procedure NewSeriesClick(Sender: TObject);
{This creates a new, empty series.}
    procedure CloneSeriesClick(Sender: TObject);
{This creates a new series, and copies the data of the selected series  into it.}
    procedure ModeClick(Sender: TObject);
    {procedure ModeNoneClick(Sender: TObject);
    procedure ModeRunClick(Sender: TObject);
    procedure ModeHistoryClick(Sender: TObject);}
{This sets the Display Mode of the graph.}
    function CanPaste: Boolean;
{Can we paste data from the Clipboard into TPlot ?}
    procedure PasteClick(Sender: TObject);
{This pastes (Tab Delimited) data from the clipboard.}
    procedure LineBestFitClick(Sender: TObject);
{This initiates a line of best fit determination.}
    procedure TwoRegionLineBestFitClick(Sender: TObject);
{This initiates a line of best fit determination over two different regions.}

    procedure ForwardFourierClick(Sender: TObject);
{This Forward fast fourier [inverse] transforms all series.
 Note that one of two different methods are employed, depending on the number of points.}
    procedure InverseFourierClick(Sender: TObject);
{This Backward / Inverse fast fourier [inverse] transforms all series.}

    procedure FunctionClick(Sender: TObject);
{This creates a new series which is a function of the existing series.}

    procedure SmoothSeriesClick(Sender: TObject);
{This smoothes the currently seleccted data series.}
    procedure CompressSeriesClick(Sender: TObject);
{This Compresss the currently selected data series by a factor of 2, 3, 4, etc.}
    procedure CompressAllSeriesClick(Sender: TObject);
{This Compresss ALL data series by a factor of 2, 3, 4, etc.}
    procedure ContractSeriesClick(Sender: TObject);
{This contracts the currently selected data series by a factor of 2, 3, 4, etc.}
    procedure ContractAllSeriesClick(Sender: TObject);
{This contracts ALL data series by a factor of 2, 3, 4, etc.}

    procedure LegendClick(Sender: TObject);
{This responds to a user selection of "Legend" in the popupmenu.}
    procedure NoteBordersClick(Sender: TObject);
{This responds to a user selection of "Show/Hide Note Borders" in the popupmenu.}
    procedure EditAxisClick(Sender: TObject);
{This responds to a user selection of "Axis ..." in the popupmenu,
 runs the Edit Axis Dialog, and assigns any changes to the appropriate axis.}
    procedure ApplyAxisChange(Sender: TObject);
{Apply changes from the AxisEditor to the Axes.}
    procedure NewNoteClick(Sender: TObject);
{This responds to a user selection of "New Note ..." in the popupmenu,
 creates the note, and gets the user to set it and place it.}
    procedure NewTextClick(Sender: TObject);
{This responds to a user selection of "New Text ..." in the popupmenu,
 creates the Text, and gets the user to set it and place it.}
    procedure DeleteNoteClick(Sender: TObject);
{This responds to a user selection of "Delete Note ..." in the popupmenu,
 checks with the user, then deletes it.}
    procedure EditFontClick(Sender: TObject);
{This responds to a user selection of "Font ..." in the popupmenu,
 runs the FontDialog, and assigns any changes to the appropriate object.}
    procedure EditPointClick(Sender: TObject);
{This responds to a user selection of "Point ..." in the popupmenu,
 runs the Series.PointEdit method, which displays and runs the PointEditor.}
    procedure EditDataClick(Sender: TObject);
{This responds to a user selection of "Edit ... Data ..." in the popupmenu,
 and displays and runs the DataEditor.}
    procedure EditSeriesClick(Sender: TObject);
{This responds to a user selection of "Edit ... Series ..." in the popupmenu,
 and displays and runs the SeriesEditor.}
    procedure ApplySeriesChange(Sender: TObject);
{Apply changes from the SeriesEditor to the Series.}

    procedure ResetDisplacementClick(Sender: TObject);
{This sets the Displacement properties DeltaX and DeltaY to ZeroScreenStuff.}
    procedure EditPropertiesClick(Sender: TObject);
{This responds to a user selection of "Properties ..." in the popupmenu,
 runs the PropertiesDialog, and assigns any changes to the appropriate objects.}
    procedure ApplyPropertiesChange(Sender: TObject);
{This applies changes from the PropertiesDialog.}

    procedure NewClick(Sender: TObject);
{This responds to a user selection of "New" in the File popupmenu,
 and clears all the data and resets the graph.}

  {$IFDEF FINANCE}
{Now all the financial functions:}
    procedure ADLClick(Sender: TObject);
{This runs the ADL financial calculation.}
  {$ENDIF}

    procedure OpenClick(Sender: TObject);
{This responds to a user selection of "Open ..." in the popupmenu,
 runs the Open Dialog, and calls the LoadFromFile method.}
    procedure ClearOverlaysClick(Sender: TObject);
{This responds to a user selection of "Clear Overlays" in the popupmenu,
 and removes any Overlays.}
    procedure OverlayClick(Sender: TObject);
{This responds to a user selection of "Overlay Data" in the popupmenu,
 runs the Overlay Dialog, and calls the LoadFromFile method.}
    procedure SaveImageClick(Sender: TObject);
{This responds to a user selection of "Save Image" in the popupmenu,
 runs the Save Dialog, and calls the SaveAsBitmap, SaveAsGIF or the SaveAsDrawing method.}
    procedure SaveClick(Sender: TObject);
{This responds to a user selection of "Save Data" in the popupmenu,
 runs the Save Dialog if FileName is blank, and calls the SaveToFile method.}
    procedure SaveAsClick(Sender: TObject);
{This responds to a user selection of "Save Data As" in the popupmenu,
 runs the Save Dialog, and calls the SaveToFile method.}
    procedure SetAsNormalClick(Sender: TObject);
{Defines the current view == Mins and Maxes of axes, as the Normal view.}
    procedure NormalViewClick(Sender: TObject);
{Zooms the screen the screen to the Normal view.}
    procedure ManualZoomClick(Sender: TObject);
{Zooms the screen using a manual dialog box.}
    procedure ApplyZoom(Sender: TObject);
{Applies the zoom.}
    procedure CopyHTMLClick(Sender: TObject);
{Copies the data as HTML to the Clipboard in CF_TEXT format, with any
 TCustomPlot.Owner added header.}
    procedure DisplaceClick(Sender: TObject);
{This responds to a user selection of "Displace" in the popupmenu, and runs
 the Displacement Form, which moves the selected Series from its origin.}
    procedure DifferentiateClick(Sender: TObject);
{This responds to a user selection of "Differentiate" in the popupmenu,
 and replaces the selected series with its differential.
{}
{(Hint: Clone the series first !)}
    procedure IntegrateClick(Sender: TObject);
{This responds to a user selection of "Integrate" in the popupmenu, and replaces
 the selected series with its integral.}
{}
{(Hint: Clone the series first !)}
    procedure IntegralClick(Sender: TObject);
{This responds to a user selection of "Integral" in the popupmenu,
 and calculates the integral of the selected series over the user-selected
 (by click-and-drag) range.}
    procedure SortClick(Sender: TObject);
{This runs the Sort method.}
    procedure SplineClick(Sender: TObject);
{This responds to a user menu click and performs a cubic spline interpolation
 of  the currently selected data series by calling the Spline method.}
    function Spline(ASeries: Integer): TSeries;
{This performs a cubic spline interpolation of ASeries by calling the
 TSeries.Spline method to place the cubic spline into a new data series.}
    procedure ZoomInClick(Sender: TObject);
{}
    procedure HighsClick(Sender: TObject);
{Finds and displays the Highs (peaks) and/or Lows (troughs) of a series.}
    procedure MovingAverageClick(Sender: TObject);
{Calculates and displays the Moving Average of a series.}
    procedure AverageClick(Sender: TObject);
{Calculates the Average of a series over a range.}
    procedure LinearizeClick(Sender: TObject);
{Linearizes a series over a range.}
    procedure ZeroClick(Sender: TObject);
{Zeros a series over a range.}
    procedure DeleteAxisClick(Sender: TObject);
{Deletes the axis selected by the user.}

{$ENDIF} {GUI}
    procedure NewYAxisClick(Sender: TObject);
{Adds a new axis, based on either the selected or last axis.}
    function CreateNewNote(ALeft, ATop: Integer; ACaption: String): TNote;
{Create a new note programmatically.}
    function CreateNewText(ALeft, ATop: Integer; ACaption: String): TText;
{Create a new Text programmatically.}

{While this is a "Click" method, it is used in some screen non-GUI screen redraws:}
    procedure ZoomOutClick(Sender: TObject);
{Zooms the screen out after a zoom-in operation.}

    procedure OpenFile(TheFile: String);

    procedure LoadFromFile(AFileName: String); virtual;
{Loads the data from a file, using LoadFromStream.}
    procedure LoadFromStream(AStream: TMemoryStream; Delimiter: String); virtual;
{Loads the data from a stream. Returns TRUE if file is a TPlot file.}
{Property File manipulation:}
    procedure LoadPropertiesFromFile(AFileName: String);
{Loads the Plot Properties from a file.}
    procedure SavePropertiesToFile(AFileName: String);
{Saves the Plot Properties to a file.}
{Property stream manipulation:}
    procedure LoadPropertiesFromStream(AStream: TMemoryStream);
{Loads the Plot Properties from a file.}
    procedure ForceKeyProperties(AStream: TMemoryStream);
{This tries to force the reading of properties from a non-3.00 properties file.}
    function IsPlotFile(AStream: TMemoryStream; var FileVersion: Integer): Boolean;
{Is the file in this stream a TPlot file ?}
    procedure AppendToFile;
{Appends the data as text to a text file.}
    procedure SaveToFile(AFileName: String); virtual;
{Saves the data as text to a text file, with any TCustomPlot.Owner added header.}
    procedure SaveToHTMLFile(AFileName: String); virtual;
{Saves the data as HTML text file, with any TCustomPlot.Owner added header.}
    procedure SaveToStream(var TheStream: TMemoryStream; InText: Boolean; Delimiter: Char); virtual;
{Saves the data to a stream, with any TCustomPlot.Owner added header.}
    procedure SaveToHTMLStream(ImageFileName: String; var TheStream: TMemoryStream); virtual;
{Saves the data to a stream in HTML format, with any TCustomPlot.Owner added header.}
    procedure SaveToHTMLStringList(ImageFileName: String; var TheData: TStringList); virtual;
{Saves the data to a stream in HTML format, with any TCustomPlot.Owner added header.}
    //procedure SaveToHTMLTemplateStream(TheFileName: String; var TheStream: TMemoryStream); virtual;
{Saves the data to a stream in HTML format, with any TCustomPlot.Owner added header.}

{$IFDEF GUI}
{Saving as a picture:}
    procedure SaveAsBitmap(AFileName: String); virtual;
{Does what it says.}
    procedure SaveAsDrawing(AFileName: String); virtual;
  {$IFDEF GIF}
    procedure SaveAsGIF(AFileName: String; W: Integer = -1; H: Integer = -1); virtual;
  {$ENDIF}
{Does what it says.}
{}
{If GIF is defined (in Plot.inc) then saving as a GIF is enabled.}
{}
{Commercial developers note: the use of GIFs requires a license from Unisys:
 http://www.unisys.com/unisys/lzw/}
{$ENDIF}

{$IFDEF PNG}
    procedure SaveAsPNG(AFileName: String; W: Integer = -1; H: Integer = -1); virtual;
{$ENDIF}
{Does what it says.}
{}
{If PNG is defined (in Plot.inc) then saving as a PNG is enabled.
 Note that PNG is always enabled under Kylix.}

    procedure MakeDummyData(NoPoints: Integer);
{This procedure is used to generate some dummy data for testing purposes.}

    procedure DeleteAxis(AnAxis: TAxis; Confirm: Boolean);
{Deletes the selected axis.}
{$IFDEF GUI}
    function GetIndicesFromTag(TheTag: Integer; var i, j: Integer): Boolean;
{Get the i and j indices for the menu from the Tag.}
    //procedure SetPlotMenu(Value: TMainMenu);
{Sets the PlotMenu property.}
    //procedure SetPlotToolBar(Value: TToolBar);
{Sets the PlotToolbar property.}
    procedure SetPlotActionList(Value: TActionList);
    procedure SetRightDragItems;
{Sets the PlotToolbar property.}
{$ENDIF}

{------------------------------------------------------------------------------}
  published

  end;
{End of declaration of TCustomPlot.}


{******************************************************************************}
  TPlot = class(TCustomPlot)
{------------------------------------------------------------------------------}
  protected

{------------------------------------------------------------------------------}
  public
{$IFDEF GUI}
    Property Bitmap;
{This returns a Bitmap of the Plot.}
    Property Drawing;
{This returns a Drawing/metafile of the Plot. Very useful for printing.}
  {$IFDEF GIF}
    Property GIF;
{This returns an image of the Plot IN GIF FORMAT.}
  {$ENDIF}
{$ENDIF}

    Property ClickedObjectType;
{The type of object that has just been clicked.}

    Property ClickedSeries;
{The series that has just been selected by clicking or picking.}

{$IFDEF PNG}
    Property Png;
{This returns a Bitmap of the Plot IN PNG FORMAT.}
{$ENDIF}

    Property FirstOverlay;
{The index of the first overlay.}    
{$IFDEF GUI}
    //Property PlotPopUpMenu;
{The public exposure of the popupmenu for PlotMenu's use.}
{$ENDIF}

    //Property Series;
{This provides direct access to the series maintained by SeriesList.}

    Property ScreenJob;
{This is the job that is in progress in response to user input.}

    property StartTime;
{The time at which data addition started. Only used when XAxis is drawn with
 units of Absolute Time. }

    Property ZAxis;
{This is the Z Axis. Every nice 3D graph should have an Z Axis.}
{}
{This is fairly experimental at the moment, so use with caution.
 It is also nil with non-3D PlotTypes.}

{------------------------------------------------------------------------------}
  published
{All the protected properties of TCustomPlot:}
    Property About;
{Displays the "About" dialog box for this component.}
    Property AxesProperties;
{Displays the "Axis Editor" dialog box for the FAxes subcomponent.}
    Property DataProperties;
{Displays the "Data Editor" dialog box.}
    Property Properties;
{Displays the "Property Editor" dialog box.}
    Property SeriesProperties;
{Displays the "Series Editor" dialog box.}
    Property Axes;
{This is a list of all the axes. The 0th is X, the 1st Y.}
    Property Notes;
{This is a list of all the Notes: TNote.}
    Property Series;
{This is the data in a list of data series.}
    Property Texts;
{This is a list of all the Text: TText boxes.}
    //Property ActionList;
{This is a list of all the Actions.}

    Property Border;
{Manages the geometry of TCustomPlot: where the axes are, where they can go, etc.}
{$IFDEF GUI}
    Property ClickAndDragDelay;
{The delay (in milliseconds) before a clicked object becomes draggable.}
{$ENDIF}

{$IFNDEF GUI}
    property Height;
    property Width;
{$ENDIF}
    property Contour;
{Contour is a persistent object that contains numerous properties relevant to
 contour plots: color schemes, detail, wireframes, etc.}
    property Color2;
{This is the second wall color. If it differs from Color, then we get a gradient.
 NB: TControl.Color already exists !}
    property ColorPlot;
{This is the Plot color. If it differs from Color, then we see it.}
    property ColorType;
{If the Color and Color2 are differernt, how are they drawn ?}
    property Grids;
{Grids is a persistent object that contains numerous properties relevant to
 Grids and Walls: color schemes, pens, etc.}

    Property ColumnGap;
{This is the percentage gap between groups of Columns in ptColumn PlotType.}

    Property CreatedBy;
{For a metafile, this is a string that is stored in the metafile.}
    Property Description;
{For a metafile, this is a string that is stored in the metafile.}

    Property DisplayHistory;
{The width of the X Axis when in History mode.}
    Property DisplayMode;
{See TDisplayMode.}
    Property DefaultExtension;
{What is the default extension of Plot files ?}
    Property Editable;
{Are screen objects like axes Editable ?}
    Property FileName;
{This is the filename to which the data is saved, or opened from.}
{}
{If FileName is blank, and an OpenClick or SaveClick occurs, then the standard
 file dialog box appears to let the user pick a name.}
    Property FileSavePrompt;
{When TPlot is destroyed, do we prompt to save changed files ?}
    property HighFont;
{The font for annotation of the Highs and Lows.}

{$IFDEF GUI}
  {$IFDEF COMPILER4_UP}
    //property Images;
{A publication of the Popup Menu's Images property}
  {$ENDIF}
{$ENDIF}

    Property HelpFile;
{When this is set to "", TPlot uses Plot.hlp as its context-sensitive help file.}
{}
{When set to some other file, it uses that instead.}

    Property Instructions;
{This is a hint-like message at the bottom of the graph.
 It used to be Caption - a string - but now has to be a
 stringlist to be fully Delphi-compatible (to avoid API calls).
 Remember Kylix !
 It disappears upon a MouseDown. See Font.}

    property Legend;
{The list of series with their line styles. This is a moveable, on-screen object.}
    Property Movable;
{Are screen objects like axes movable ?}
    //Property Link;
{Another composite property that contains properties relevant to drawing links
 between series on screen.}
    //Property Multiplicity;
{When the PlotType is ptMultiple, the series are grouped into multiples of Multiplicity.
 Otherwise ignored.}
    //property MultiplePen;
{The pen to use for the verticle lines of ptMultiple PlotTypes (eg: High-Low).}
    //property MultiJoin;
{Which two series, by number, to join with one rectangular symbol, in a ptMultiple plot.
 Eg: if Multiplicity is 4, and MultiJoin = '2,3', then you have a "Candle" or
 High-Low-Open-Close plot.}    
    property PieRowCount;
{The number of rows of Pie Graphs.}
    property NoSeries;
{The number of Series.}
    property NoYAxes;
{The total number of Y Axes (primary, secondary, tertiary, etc).}
    Property OutlineWidth;
{This is the width of the outline for screen objects like lines: axes and borders.}
    Property PlotType;
{What type of plot is this ?}
    Property PolarRange;
{What does 360 correspond to in a polar graph ? Examples are:
    2 Pi (6.28...),
    360 (degrees),
    60 (minutes),
    24 (hours)
    100 (%)
Get the idea ?    }
    Property PrintOrientation;
{Shall we print the graph in Landscape or Portrait mode ?}
    Property SaveOptions;
{Shall we save the data as Text or binary ?}
{}
{Shall we also save the Plot properties when we save the data ?
 If we do, then we:
    1. Save the properties in a seperate file;
    2. Look for a properties file to open.}
    Property Title;
{The title of the graph, including its geometry, font and visibility.}
    //Property WallColor;
{What colour is our Wall ? Default is clYellow.}
    Property XAxis;
{This is the X Axis. Every nice graph should have an X Axis.}
    Property YAxis;
{This is the Y Axis. Every nice graph should have at least one Y Axis.}
{}
{Each Series must know what Y Axes it is being plotted against:
 Primary (this one) or Secondary.}
(*    Property ZLength;
{The Length made by the Z Axis, if any, with the vertical, in a clockwise direction.}
    Property ZLink;
{Should we link 3D series together in the Z direction ?}
    Property ZRatio;
{The length of the Z Axis relative to the X and Y Axes.} *)
    Property XYFastAt;
{At how many data points should we switch to a fast drawing alforithm.}
    Property ZAngle;
{The angle made by the Z Axis, if any, with the vertical, in a clockwise direction.}
    Property ZLink;
{Should we link 3D series together in the Z direction ?}
    property ZSeriesNames;
{Should the Z Axis display the names of the Series ?}

{Events:}
    Property OnAfterPaint;
    Property OnAfterDraw;
    Property OnBeforePaint;
    Property OnBeforeDraw;
    Property OnStyleChange;
    Property OnDataChange;

    Property OnFileNew;
{When a New file is created, the app can be notified of this using this event.}

    Property OnFileOpen;
{When a file is opened, the app can be notified of the new file name using this event.}

    Property OnFileClose;
{When a file is closed, the app can be notified of the new file name using this event.}

    Property OnHeader;
{When data is opened, this event passes the header information back to the "user".}

    Property OnHeaderRequest;
{When data is saved or copied, this event allows the user to add a header
 to the data.}

    Property OnHTMLHeaderRequest;
{When data is copied as HTML, this event allows the user to add a header
 to the data.}

    Property OnSelection;
    Property OnDualSelection;
    Property OnWarning;

{all the "extra" TPanel properties in D4 (alias "VCL Bloat"):}

{Now all the TPanel properties-------------------------------------------------}
{all the TPanel properties in D1:}
    property Color;
    property Font;
{This is the font of the hint-like message at the bottom of the graph.}
{$IFDEF GUI}
    property Align;
    {property Alignment; TPanel.Alignment not required}
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    {property FullRepaint; TPanel.FullRepaint not required}
    {property Locked; TPanel.Locked not required}
    {property Caption; - replaced by Instructions}
  {$IFDEF MSWINDOWS}
    property Ctl3D;
    property DragCursor;
  {$ENDIF}
    property DragMode;
    property Enabled;
    property ParentColor;
  {$IFDEF MSWINDOWS}
    property Locked;
    property ParentCtl3D;
  {$ENDIF}
    property ParentFont;
    property ParentShowHint;
{ Note: D1 to D4 were quite happy for:
        PopupMenu := FPlotPopUpMenu;
  FPlotPopUpMenu was then run by inherited MouseUp.
  However, D5 introduced TControl.WMContextMenu, which ran BEFORE MouseDown.
  This went spastic when it tried to Popup the PopupMenu.

  We have therefore returned to hiding FPlotPopUpMenu, and running it manually.}
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;

  {$IFDEF COMPILER3_UP}
    {$IFDEF MSWINDOWS}
    property FullRepaint;
    property OnStartDrag;
    {$ENDIF}
  {$ENDIF}

  {$IFDEF COMPILER4_UP}
    property Anchors;
    // property AutoSize; - leads to bizzare behaviour
    {$IFDEF MSWINDOWS}
    property BiDiMode;
    {$ENDIF}
    property Constraints;
    {$IFDEF MSWINDOWS}
    property UseDockManager default True;
    property DockSite;
    property DragKind;
    property ParentBiDiMode;
    property OnCanResize;
    {$ENDIF}
    property OnConstrainedResize;
    //property OnContextPopup; - not required because of internal popup menu
    {$IFDEF MSWINDOWS}
    property OnDockDrop;
    property OnDockOver;
    property OnEndDock;
    property OnGetSiteInfo;
    property OnStartDock;
    property OnUnDock;
    {$ENDIF}
  {$ENDIF} {COMPILER4_UP}
{$ENDIF} {GUI}

{$IFDEF COMPILER5_UP}
{$ENDIF}
{$IFDEF COMPILER6_UP}
{$ENDIF}
{$IFDEF COMPILER7_UP}
{$ENDIF}
  end;


implementation
{$R Cursor32.res}

{$IFDEF GUI}
uses
  {Plotmenu, Plottoolbar,} Plotactionlist;
{$ENDIF}

resourcestring
  sAcknowledgements =
    'TPlot Version %s' + #10 +
    'Released under the Mozilla Public License' + #10 + #10 +
    'Acknowledgements to' + #10 +
    '  Anders Melanders - TGifImage' + #10 +
    '  Edmund H. Hand, Dominique Louis, Uberto Barbini and ' + #10 +
    '    Eric Engler - TPngimage' + #10 +
    '  Atanas Stoyanov - MemProof' + #10 +
    '  Renate Schaaf, Alin Flaider, Hallvard Vassbotn and ' + #10 +
    '    Stefan Hoffmeister - TParser10' + #10 +
    '  John Shemitz - rotated Kylix Text' + #10 +
    '  David J Taylor, Jeff Weiss and Juergen Loewner - CurveFit';
  sActionInstructions =
    'For best use of TPlot, also do the following:' + #13#10 + #13#10 +
    '1. Create an OnKeyDown event handler in this form, and enter this code:' + #13#10 +
    '  %s' + #13#10 + #13#10 +
    'This code snippet is on the clipboard.';
  sCode1 = '  Plot1.ProcessKeyDown(Key, Shift);';

{Let the coding begin ! *******************************************************}

{TPlotCollection methods ------------------------------------------------------}
constructor TPlotCollection.Create(AOwner: TPersistent; ItemClass: TCollectionItemClass);
begin
  inherited Create(AOwner, ItemClass);
  FPlot := TPlot(AOwner);
end;

procedure TPlotCollection.Update(Item: TCollectionItem);
begin
{$IFDEF GUI}
  FPlot.Invalidate;
{$ENDIF}
end;

{TAxisList methods ------------------------------------------------------------}
function TAxisList.Add: TAxis;
begin
  Result := TAxis(inherited Add);
  Result.OnChange := FPlot.DoStyleChange;
  Result.OnOutOfBounds := FPlot.DoOutOfBounds;
  case Self.Count of
    1: Result.AxisType := atPrimaryX;
    2: Result.AxisType := atPrimaryY;
    3:
      begin
        Result.AxisType := atSecondaryY;
        Result.Intercept := Self.Items[0].Max;
      end;
  else
    Result.AxisType := atTertiaryY;
    Result.Intercept := 1.10 * Self.Items[0].Max;
  end;
  Result.Tag := Ord(soYAxis);
  Result.Labels.Tag := Ord(soYAxisLabel);
  Result.Title.Tag := Ord(soYAxisTitle);
end;

{$IFDEF DELPHI4_DOWN}
procedure TAxisList.Delete(Index: Integer);
begin
  TCollectionItem(Self.Items[Index]).Free;
end;
{$ENDIF}

function TAxisList.GetItem(Index: Integer): TAxis;
begin
  if (Index < Count) then
    Result := TAxis(inherited GetItem(Index))
   else
    Result := nil;
end;

function TAxisList.GetZAxis: TAxis;
var
  i: Integer;
  AnAxis: TAxis;
begin
  Result := nil;
  for i := 2 to Self.Count-1 do // 0 = X, 1 = Y _ALWAYS_
  begin
    AnAxis := TAxis(inherited GetItem(i));
    if (AnAxis.AxisType = atZ) then
    begin
      Result := AnAxis;
      break;
    end;
  end;
end;

procedure TAxisList.SetItem(Index: Integer; Value: TAxis);
begin
  inherited SetItem(Index, Value);
end;

procedure TAxisList.SetZAxis(Value: TAxis);
begin
  if (Value.AxisType = atZ) then
    Self.ZAxis.Assign(Value);
end;

{TNoteList methods ------------------------------------------------------------}
function TNoteList.Add: TNote;
begin
  Result := AddEx('');
end;

function TNoteList.AddEx(ACaption: String): TNote;
begin
  Result := TNote.Create(Self);
  Result.Tag := Ord(soNote);
  Result.OnCaptionChange := FPlot.DoStyleChange;
  Result.Name := IntToStr(Self.Count);
  if (Length(ACaption) > 0) then
    Result.Caption := ACaption
   else
    Result.Caption := 'New Note #' + Result.Name;
  Result.MoveTo(Random(FPlot.Width-50), Random(FPlot.Height-20));
  Result.ArrowLeft := Result.Left - 20;
  Result.ArrowTop := Result.Top -20;
end;

{$IFDEF DELPHI4_DOWN}
procedure TNoteList.Delete(Index: Integer);
begin
  TCollectionItem(Self.Items[Index]).Free;
end;
{$ENDIF}

function TNoteList.GetItem(Index: Integer): TNote;
begin
  if (Index < Count) then
    Result := TNote(inherited GetItem(Index))
   else
    Result := nil;
end;

procedure TNoteList.SetItem(Index: Integer; Value: TNote);
begin
  inherited SetItem(Index, Value);
end;

{TTextList methods ------------------------------------------------------------}
function TTextList.Add: TText;
begin
  Result := AddEx('');
end;

function TTextList.AddEx(ACaption: String): TText;
begin
  Result := TText(inherited Add);
  Result.Tag := Ord(soText);
  Result.OnCaptionChange := FPlot.DoStyleChange;
  Result.Name := IntToStr(Self.Count);
  if (Length(ACaption) > 0) then
    Result.Caption := ACaption
   else
    Result.Caption := 'New Text Box #' + Result.Name;
  Result.MoveTo(Random(FPlot.Width-50), Random(FPlot.Height-20));
end;

{$IFDEF DELPHI4_DOWN}
procedure TTextList.Delete(Index: Integer);
begin
  TCollectionItem(Self.Items[Index]).Free;
end;
{$ENDIF}

function TTextList.GetItem(Index: Integer): TText;
begin
  if (Index < Count) then
    Result := TText(inherited GetItem(Index))
   else
    Result := nil;
end;

procedure TTextList.SetItem(Index: Integer; Value: TText);
begin
  inherited SetItem(Index, Value);
end;



{------------------------------------------------------------------------------
{TCustomPlot methods --------------------------------------------------------------}
{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ShowAbout
  Description: Property editor methods, et cetera
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 03/29/2001 by Mat Ballard
      Purpose: design-time user interface
 Known Issues: TAboutDlg adds about 8 K to the app
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ShowAbout(Sender: TObject);
var
  AboutDlg: TAboutDlg;
begin
  AboutDlg := TAboutDlg.Create(nil);
  AboutDlg.Comments := Format(sAcknowledgements,
    [FloatToStrF(TPLOT_VERSION / 100, ffFixed, 5, 2)]);
  AboutDlg.Execute;
  AboutDlg.Free;
end;

procedure TCustomPlot.ShowData;
begin
  EditDataClick(Self);
end;

procedure TCustomPlot.ShowSeries;
begin
  EditSeriesClick(Self);
end;

procedure TCustomPlot.ShowAxes;
begin
  EditAxisClick(Self);
end;

procedure TCustomPlot.ShowProperties;
begin
  EditPropertiesClick(Self);
end;
{$ENDIF}

{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Create
  Description: class constructor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initialize variables and create sub-components
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TCustomPlot.Create(AOwner: TComponent);
begin
  FStatus := [sNew, sUpdating];
{First call the ancestor:}
  inherited Create(AOwner);

{$IFNDEF GUI}
  FCanvas := nil;
  FFont := TFont.Create;
{$ENDIF}

  //FInstructions := TStringList.Create;

{create the border of the graph:}
  FBorder := TBorder.Create(Self);

{create the user-clickable edges of the graph (for clicking and dragging):}
(*
  mLeftBorder := TAngleRect.Create(nil);
  mTopBorder := TAngleRect.Create(nil);
  mRightBorder := TAngleRect.Create(nil);
  mBottomBorder := TAngleRect.Create(nil);
*)

{create the list of sundry temporary objects on the screen:}
  ScreenObjectList := TList.Create;
{create the lists of notes and texts:}
  FNotes := TNoteList.Create(Self, TNote);
  FTexts := TTextList.Create(Self, TText);

{create the list of all axes:}
  FAxes := TAxisList.Create(Self, TAxis);

{create the list of data series:}
  FSeriesList := TSeriesList.Create(Self);
  //FSeriesList.AxisList := FAxes;
  mInitialSeriesCount := -1;

{create the graph title:}
  FTitle := TTitle.Create(nil);

{create the plot legend:}
  FLegend := TLegend.CreateList(nil, FSeriesList);

{create the contour and grid information:}
  FContour := TContour.Create;
  FGrids := TGrids.Create;
  (*FZLink := TRUE;
  FZRatio := 40;*)

{the font for annotation of the highs and lows:}
  FHighFont := TFont.Create;

{create the object that is clicked and dragged:}
  Selection := TAngleRect.Create(nil);
  Selection.FireEvents := FALSE;

{nil out the page buttons:}
{$IFDEF GUI}
  mPageButtons[0] := nil;
  mPageButtons[1] := nil;
  mPageButtons[2] := nil;
  mPageButtons[3] := nil;
{$ENDIF}

{Users can edit some screen text:}
  FEditable := TRUE;
{Users can move screen objects:}
  FMovable := TRUE;

{20% gap between columns:}
  FColumnGap := 20;

{our default help file:}
  FHelpFile := 'Plot.hlp';

{only one row of pie graphs:}
  FPieRowCount := 1;

{One whole rotation corrsponds to ...}
  FPolarRange := TWO_PI;

  FXYFastAt := 10000;

{$IFDEF GUI}
{No mouse timer as yet:}
  MouseTimer := TTimer.Create(Self);
  MouseTimer.Enabled := FALSE;
  MouseTimer.OnTimer := MouseTimeOut;
{$ENDIF}

{no series has been selected yet:}
  FClickedSeries := nil;
  TheSeries := -1;

  FDisplayHistory := 30;
  FFirstOverlay := -1;
{$IFDEF GUI}
  //FPlotMenu := nil;
  FPlotActionList := nil;
{$ENDIF}
  FPrintOrientation := poLandscape;
{ZeroScreenStuff everything to do with mice:}
  FScreenJob := sjNone;
  FOnSelection := nil;
  FOnDualSelection := nil;
  FSaveOptions := [soProperties];

  DefaultExtension := DEF_EXTENSION;
  FileExtensions[1] := '.csv';
  FileExtensions[2] := '.txt';
  FileExtensions[3] := '.htm';
  FileExtensions[4] := '.html';
  FileExtensions[5] := '.*';
  FileName := 'plot1' + DefaultExtension;

  mOpenFilterIndex := 1;
  mSaveFilterIndex := 1;
  mImageFilterIndex := 1;

  //BevelGap := 1;

{Take the hint:}
  FInstructions := _INSTRUCTION;

{This is a typical size:}
  Height := 300;
  Width := 400;

(*
{Name the user-clickable edges of the graph (for clicking and dragging):}
  mLeftBorder.Name := 'Left Border';
  mLeftBorder.Tag := Ord(soLeftBorder);
  mTopBorder.Name := 'Top Border';
  mTopBorder.Tag := Ord(soTopBorder);
  mRightBorder.Name := 'Right Border';
  mRightBorder.Tag := Ord(soRightBorder);
  mBottomBorder.Name := 'Bottom Border';
  mBottomBorder.Tag := Ord(soBottomBorder);
*)

{... and initialize the Plot Title:}
  FTitle.Caption := ClassName;
  FTitle.Name := 'Plot Title';
  FTitle.Font.Size := LARGE_FONT_SIZE;
  FTitle.Tag := Ord(soTitle);
  FTitle.FireEvents := TRUE;

{... and initialize the plot legend:}
  FLegend.Name := 'Legend ...';
  FLegend.Font.Size := SMALL_FONT_SIZE;
  FLegend.Tag := Ord(soLegend);

{Initialize the Instruction font:}
  Font.Name := sArial;
  Font.Size := SMALL_FONT_SIZE;
  Font.Color := clRed;
{the font for annotation of the highs and lows:}
  FHighFont.Name := sArial;
  FHighFont.Size := SMALL_FONT_SIZE;

{load the list of screen objects:}
  ScreenObjectList.Add(Selection);      {soNone}
  ScreenObjectList.Add(FTitle);         {soTitle}
  ScreenObjectList.Add(FLegend);        {soLegend}
(*
  ScreenObjectList.Add(mLeftBorder);     {soLeftBorder}
  ScreenObjectList.Add(mTopBorder);      {soTopBorder}
  ScreenObjectList.Add(mRightBorder);    {soRightBorder}
  ScreenObjectList.Add(mBottomBorder);   {soBottomBorder}
*)

{$IFDEF GUI}
  CreateMenus;
{$ENDIF}

{Set all the OnChanges:}
  FBorder.OnChange := DoStyleChange;
  FTitle.OnChange := DoStyleChange;
  FSeriesList.OnStyleChange := DoStyleChange;
  FSeriesList.OnDataChange := DoDataChange;
  FLegend.OnChange := DoStyleChange;
  FContour.OnChange := DoStyleChange;
  FGrids.OnChange := DoStyleChange;

{Set some colors:}
  Color := clWhite;
  FColor2 := clWhite;
  FColorPlot := clWhite;
  FColorType := gtHorizontal;

{load some cursors:}
{$IFDEF GUI}
  {$IFDEF MSWINDOWS}
  Screen.Cursors[crScope] := {WinProcs.}LoadCursor(HInstance, 'crScope');
  Screen.Cursors[crX] := {WinProcs.}LoadCursor(HInstance, 'crX');
  {$ENDIF}
{$ENDIF}

{Loaded has not run yet:}
  //mNOTLoaded := TRUE;
  mPaintLock := 0;
  if ((csDesigning in ComponentState) and not (csLoading in AOwner.ComponentState)) then
  begin
    ShowMessageFmt(sActionInstructions, [sCode1]);
    ClipBoard.AsText := sCode1;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Loaded
  Description: "Loaded allows a component to initialize itself after all its parts have been loaded from a stream."
       Author: Mat Ballard
 Date created: 09/05/2002
Date modified: 09/05/2002 by Mat Ballard
      Purpose: Property Management
 Known Issues: "After reading all the property values for all the components,
               Delphi calls the Loaded methods of each component in the order
               the components were created. This gives the components a chance
               to initialize any data that depends on the values of other
               components or other parts of itself."
               NB: Loaded is NOT called directly if:
               1. TPlot is created dynamically in an App;
               2. When TPlot is placed on a form in the IDE.
               So we have to kludge calling Loaded in Resize and MakeDummyData
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Loaded;
begin
  inherited Loaded;
  if (FAxes.Count = 0) then
  begin
{Create the X Axis and initialize it:}
    FAxes.Add; // which actually returns a TCollectionItem
    FAxes[0].AxisType := atPrimaryX;
    FAxes[0].Title.Units := 'seconds';
{Now the Y Axis:}
    FAxes.Add;
    FAxes[1].AxisType := atPrimaryY;
    FAxes[1].Title.Units := 'mVolts';
  end
  else if (FAxes.Count = 1) then
  begin
    EComponentError.Create('Error: Loaded: Only ONE Axis !!!');
  end;

  //if (FSeriesList.AxisList = nil) then FSeriesList.AxisList := FAxes;
  FSeriesList.Loaded;
  
  FStatus := FStatus - [sUpdating];
  ZeroScreenStuff;
  FStatus := FStatus + [sLoaded];

{set the user-clickable border thicknesses:}
  SetOutlineWidth(10);
{this fires SetAxisDimensions.}

  if (csDesigning in ComponentState) then
    MakeDummyData(20);
end;

{functions called by constructor ----------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MakeDummyData
  Description: procedure to add two series, with sine wave data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: 1. test TCustomPlot
               2. display capabilities in design mode.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MakeDummyData(NoPoints: Integer);
var
  i, j: Integer;
  Amplitude,
  MidY,
  Phase,
  StepSize,
  X, Y: Single;
  Extra: array [0..3] of Single;
  pASeries: TSeries;

const
  EPSILON = 0.2;

begin
  if (NoPoints <= 1) then exit;
  PaintLock;
  if not (sLoaded in FStatus) then Loaded;

{clean up first:}
  for i := 0 to FSeriesList.Count-1 do
    TSeries(FSeriesList[i]).DelData;

  if (FSeriesList.Count = 0) then
  begin
    FSeriesList.Add;
    if (FPlotType in [ptLineContour, ptContour, pt3DContour, pt3DWire, pt3DColumn]) then
    begin
      for i := 1 to 7 do
        FSeriesList.AddDependent(0);
    end
    else
    begin
{add two series, the second depending on the first:}
{this second series uses the X data of the first series:}
      FSeriesList.AddDependent(0);
      TSeries(FSeriesList[1]).OnStyleChange := DoStyleChange;
    end;
  end;

{seed it:}
  Randomize;
{set the phase for trig functions:}
  Phase := Random;
{initialize: calculate the step size:}
  StepSize := (FAxes[0].Max - FAxes[0].Min) / (NoPoints-1);
{... and the size:}
  Amplitude := (FAxes[1].Max - FAxes[1].Min) / 2;
  MidY := (FAxes[1].Max + FAxes[1].Min) / 2;
  for i := 0 to NoPoints-1 do
  begin
    for j := 0 to FSeriesList.Count-1 do
    begin
      pASeries := TSeries(FSeriesList[j]);
      case FPlotType of
        ptPolar:
          begin
            X := (i / (NoPoints-1)) * Self.FPolarRange;
            Y := FAxes[0].Max / 2 + (Amplitude/3) * Sin(X + (j+1) * Phase);
{Don't fire any events, and don't adjust axes:}
            pASeries.AddPoint(X, Y, FALSE, FALSE);
          end;
        ptLineContour, ptContour, pt3DContour, pt3DWire, pt3DColumn{, pt3DSurface}:
          begin
            X := FAxes[0].Min + i * StepSize;
            Y := Exp(-j / 2.0) * (MidY + Amplitude * Sin(X));
{Don't fire any events, and don't adjust axes:}
            pASeries.AddPoint(X, Y, FALSE, FALSE);
          end;
        else
          begin
            X := FAxes[0].Min + i * StepSize;
            Y := MidY + Amplitude * Sin(X + (j+1) * Phase) + Random;
            case pASeries.SeriesType of
              stNormal: pASeries.AddPoint(X, Y, FALSE, FALSE);
              stComplex:
                begin
                  Extra[0] := MidY + Amplitude * Cos(X + (j+1) * Phase) + Random;
                  pASeries.AddPointEx(X, Y, Extra, FALSE, FALSE);
                end;
              stBubble:
                begin
                  Extra[0] := 0.5 + Random;
                  pASeries.AddPointEx(X, Y, Extra, FALSE, FALSE);
                end;
              stError:
                begin
                  Extra[0] := (1 + Random) / 5;
                  Extra[1] := (1 + Random) / 3;
                  pASeries.AddPointEx(X, Y, Extra, FALSE, FALSE);
                end;
              stMultiple:
                begin
                  case pASeries.Multiplicity of
                    1: Extra[0] := Y + EPSILON + Random;
                    2:
                      begin
                        Extra[0] := Y + EPSILON + Random;
                        Extra[1] := Y - EPSILON - Random;
                      end;
                    3:
                      begin
                        Extra[0] := Y + EPSILON + Random;
                        Extra[1] := Y - EPSILON - Random;
                        Extra[2] := Extra[1] - EPSILON - Random;
                      end;
                    4:
                      begin
                        Extra[1] := Y + EPSILON + Random;
                        Extra[0] := Extra[1] + EPSILON + Random;
                        Extra[2] := Y - EPSILON - Random;
                        Extra[3] := Extra[2] - EPSILON - Random;
                      end;
                  else
                  end;
                  pASeries.AddPointEx(X, Y, Extra, FALSE, FALSE);
                end;
            end;
          end;
      end; {case}
      TSeries(FSeriesList[j]).ZData := j;
    end; {for j}
  end; {for i}
  PaintUnLock;
  ZoomOutClick(Self);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CreateMenus
  Description: creates popup menus that are accessible by right-click
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/20/2000 by Mat Ballard
      Purpose: modularize user-interface code
 Known Issues: this was a bitch to get right !
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CreateMenus;
var
  i: Integer;
{This following is just a dummy matrix of menu items used to create sub-menus,
 then removed and the menuitems freed.}
  TempMenuItem: TMenuItem;

begin
{We create the "which" popup menu:}
  WhichPopUpMenu := TPopUpMenu.Create(Self);
  for i := 0 to 1 do begin
    WhichPopUpItems[i] := TMenuItem.Create(Self);
    WhichPopUpItems[i].Tag := i;
    WhichPopUpMenu.Items.Add(WhichPopUpItems[i]);
  end;
{Note: we set the Instructions just before we display this popup:}
  WhichPopUpItems[0].OnClick := MoveTheClickedObjectClick;
  WhichPopUpItems[1].OnClick := MoveSecondClickedObjectClick;

  FDragPopUpMenu := TPopupMenu.Create(Self);
  for i := 0 to 8 do
  begin
    TempMenuItem := TMenuItem.Create(Self);
    //TempMenuItem.Action := FPlotActionList.Actions[RightDragItems[i]];
    FDragPopUpMenu.Items.Add(TempMenuItem);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CreatePageButtons
  Description: creates the Page buttons and assigns their properties
       Author: Mat Ballard
 Date created: 04/28/2001
Date modified: 04/28/2001 by Mat Ballard
      Purpose: user interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CreatePageButtons;
var
  i: Integer;
begin
  if (mPageButtons[0] = nil) then
  begin
    for i := 0 to 3 do
    begin
      mPageButtons[i] := {$IFDEF GUI}TBitmap{$ELSE}gdGraphics.TBitmap{$ENDIF}.Create;
    end;
    mPageButtons[0].LoadFromResourceName(HInstance, 'BMLEFT');
    mPageButtons[1].LoadFromResourceName(HInstance, 'BMRIGHT');
    mPageButtons[2].LoadFromResourceName(HInstance, 'BMUP');
    mPageButtons[3].LoadFromResourceName(HInstance, 'BMDOWN');
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DestroyPageButtons
  Description: frees the Page buttons
       Author: Mat Ballard
 Date created: 04/28/2001
Date modified: 04/28/2001 by Mat Ballard
      Purpose: user interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DestroyPageButtons;
var
  i: Integer;
begin
  if (Assigned(mPageButtons[0])) then
  begin
    for i := 0 to 3 do
    begin
      mPageButtons[i].Free;
      mPageButtons[i] := nil;
    end;
  end;
end;
{$ENDIF} {GUI CreateMenus}

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PageButtonClick
  Description: changes the ranges of the axes in Zoomed mode
       Author: Mat Ballard
 Date created: 04/28/2001
Date modified: 04/28/2001 by Mat Ballard
      Purpose: User Interface control of axis ranges and zooming
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PageButtonClick(Index: Integer);
var
  Gap,
  OldVar,
  NewVar,
  MinMax: Single;
begin
  case Index of
    0: {left}
      begin
        Gap := FAxes[0].Max - FAxes[0].Min;
        OldVar := FAxes[0].Min;
        NewVar := OldVar - Gap;
        MinMax := FSeriesList.Xmin;
        if (NewVar < MinMax) then
          NewVar := MinMax;
        FAxes[0].Min := NewVar;
        FAxes[0].Max := NewVar + Gap;
      end;
    1: {right}
      begin
        Gap := FAxes[0].Max - FAxes[0].Min;
        OldVar := FAxes[0].Max;
        NewVar := OldVar + Gap;
        MinMax := FSeriesList.Xmax;
        if (NewVar > MinMax) then
          NewVar := MinMax;
        FAxes[0].Max := NewVar;
        FAxes[0].Min := NewVar - Gap;
      end;
    2: {up}
      begin
        Gap := FAxes[1].Max - FAxes[1].Min;
        OldVar := FAxes[1].Max;
        NewVar := OldVar + Gap;
        MinMax := FSeriesList.Ymax;
        if (NewVar > MinMax) then
          NewVar := MinMax;
        FAxes[1].Max := NewVar;
        FAxes[1].Min := NewVar - Gap;
      end;
    3: {down}
      begin
        Gap := FAxes[1].Max - FAxes[1].Min;
        OldVar := FAxes[1].Min;
        NewVar := OldVar - Gap;
        MinMax := FSeriesList.Ymin;
        if (NewVar < MinMax) then
          NewVar := MinMax;
        FAxes[1].Min := NewVar;
        FAxes[1].Max := NewVar + Gap;
      end;
  end;
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: free sub-components
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TCustomPlot.Destroy;
begin
{nil out the events:}
  FOnStyleChange := nil;
  FOnFileNew := nil;
  FOnFileOpen := nil;
  FOnFileClose := nil;
  FOnHeader := nil;
  FOnHeaderRequest := nil;
  FOnHTMLHeaderRequest := nil;
  FOnSelection := nil;
  FOnDualSelection := nil;

{Clear the SeriesList, thereby provoking a file save if required:}
  Clear(FALSE);

{$IFDEF GUI}
  DestroyPageButtons;
{$ENDIF}

{Free the Collections of objects; note that they free their collected objects:}
  FAxes.Free;
  FNotes.Free;
  FTexts.Free;

{Free the experimental Action List}
  //FActionList.Free;

{Free the lists of objects:}
  ScreenObjectList.Free;
  FSeriesList.Free;

{Free the visual sub-components}
  //FInstructions.Free;
  FLegend.Free;
  FTitle.Free;
  FBorder.Free;

{free non-visible screen elements:}
(*
  mLeftBorder.Free;
  mTopBorder.Free;
  mRightBorder.Free;
  mBottomBorder.Free;
*)
  Selection.Free;

{Free fonts, pens, etc:}
  FHighFont.Free;
  FContour.Free;
  FGrids.Free;

{$IFDEF GUI}
  MouseTimer.Free;
{Free menus:}
  //FPlotPopUpMenu.Free;
  FDragPopUpMenu.Free;
  WhichPopUpMenu.Free;
{$ELSE}
  FFont.Free;
{$ENDIF}

{then call ancestor:}
  inherited Destroy;
end;

{End Constructor and Destructor: ----------------------------------------------}

{Get functions ----------------------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TCustomPlot.GetSeries
  Description: private property Get function
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: interface to Series property, which is the default property
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetSeries(
  Index: Integer): TSeries;
begin
  if ((Index < 0) or (Index >= FSeriesList.Count)) then raise
    ERangeError.CreateFmt('There is no series %d: valid indices are from 0 to %d', [Index, FSeriesList.Count-1]);
  GetSeries := TSeries(FSeriesList[Index]);
end;

{Set procedures ---------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetName
  Description: Sets the plot name
       Author: Mat Ballard
 Date created: 06/05/2002
Date modified: 06/05/2002 by Mat Ballard
      Purpose: prohibits run-time name changes
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetName(const NewName: TComponentName);
begin
  if (not mNameFrozen) then
    inherited SetName(NewName);
end;

(* ----------------------------------------------------------------------------
   |                                                                           |
   |                                                                           |
   |       ************************************************************        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |       *    |                                                     *        |
   |    ___*    |_____________________________________________________*        |
   |     | *   /                                                      *        |
   |     | *  /                                                       *        |
   |    dy * /                                                        *        |
   |     | */                                                         *        |
   |    _|_************************************************************        |
   |       |-dx-|                                                              |
   |                                                                           |
   ----------------------------------------------------------------------------   *)

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetAxisDimensions
  Description: geometry manager
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets up the border, axes and Title position
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetAxisDimensions(ACanvas: TCanvas);
var
  i: Integer;
  TheOrigin, TheVector, Delta: TPoint;
  AnAxis: TAxis;
  ASeries: TSeries;

  procedure LimitXAxis;
  begin
    if (FAxes[0].Intercept < FAxes[1].Min) then
    begin
      FAxes[0].Intercept := FAxes[1].Min;
      FAxes[0].DoTicks;
    end
    else if (FAxes[0].Intercept > FAxes[1].Max) then
    begin
      FAxes[0].Intercept := FAxes[1].Max;
      FAxes[0].DoTicks;
    end;
  end;

  procedure LimitYAxis;
  begin
    if (FAxes[1].Intercept < FAxes[0].Min) then
    begin
      FAxes[1].Intercept := FAxes[0].Min;
      FAxes[1].DoTicks;
    end
    else if (FAxes[1].Intercept > FAxes[0].Max) then
    begin
      FAxes[1].Intercept := FAxes[0].Max;
      FAxes[1].DoTicks;
    end;
  end;

  procedure LimitZAxis;
  begin
{Intercept on the X Axis:}
    if (ZAxis.Intercept < FAxes[0].Min) then
      ZAxis.Intercept := FAxes[0].Min;
    if (ZAxis.Intercept > FAxes[0].Max) then
      ZAxis.Intercept := FAxes[0].Max;
{Intercept on the Y Axis:}
    if (ZAxis.ZInterceptY < FAxes[1].Min) then
      ZAxis.ZInterceptY := FAxes[1].Min;
    if (ZAxis.ZInterceptY > FAxes[1].Max) then
      ZAxis.ZInterceptY := FAxes[1].Max;
  end;

begin
  if (csLoading in ComponentState) then exit;

  PaintLock;

  if ((FPlotType = ptContour) or
      (FPlotType = ptLineContour)) then
  begin
    if (FAxes[1].AutoScale) then
    begin
      FAxes[1].Min := FSeriesList.ZMin;
      FAxes[1].Max := FSeriesList.ZMax;
    end;
  end;

  if (Assigned(ZAxis)) then
  begin
{NB: if this is the first draw of the Z Axis after 2D plot types, then the
 Z Axis geometry has not yet been calculated. We therefore have to calculate
 the Z Axis geometry so as to have a valid Vector.}
    ZAxis.DoGeometry;
    TheVector := ZAxis.Vector;
    if (TheVector.X < 0) then
    begin  // DeltaX < 0, Z Axis points left
      Delta.X := TheVector.X * FBorder.Width div (FBorder.Width - TheVector.X);
      FAxes[0].Left := FBorder.Left - Delta.X;
      FAxes[0].Width := FBorder.Width + Delta.X;
    end else if (TheVector.X > 0) then begin // DeltaX > 0, Z Axis points right
      Delta.X := TheVector.X * FBorder.Width div (FBorder.Width + TheVector.X);
      FAxes[0].Left := FBorder.Left;
      FAxes[0].Width := FBorder.Width - Delta.X;
    end;
    if (TheVector.Y < 0) then
    begin // DeltaY < 0, Z Axis points UP
      Delta.Y := TheVector.Y * FBorder.Height div (FBorder.Height - TheVector.Y);
      FAxes[1].Top := FBorder.Top - Delta.Y;
      FAxes[1].Height := FBorder.Height + Delta.Y;
    end else if (TheVector.Y > 0) then begin // DeltaY > 0, Z Axis points DOWN
      Delta.Y := TheVector.Y * FBorder.Height div (FBorder.Height + TheVector.Y);
      FAxes[1].Top := FBorder.Top;
      FAxes[1].Height := FBorder.Height - Delta.Y;
    end;
  end else begin
{do the simple bits, where the border sets the axis lengths:}
    FAxes[0].Left := FBorder.Left;
    FAxes[0].Width := FBorder.Width;
    FAxes[1].Top := FBorder.Top;
    FAxes[1].Height := FBorder.Height;
  end;

  if (FAxes[0].AutoScale) then
  begin
    FAxes[0].Min := FSeriesList.XMin;
    FAxes[0].Max := FSeriesList.XMax;
  end;

{Set height of all other Y Axes:}
  for i := 2 to FAxes.Count-1 do
  begin // all Y Axes have the same Top and Height as the Primary:
    if (FAxes[i].AxisType >= atPrimaryY) then
    begin
      FAxes[i].Top := FAxes[1].Top;
      FAxes[i].Height := FAxes[1].Height;
    end;
  end;

{Set Scaling of all Y Axes}
(*  for i := 1 to FAxes.Count-1 do
  begin
    if (FAxes[i].AxisType >= atPrimaryY) then
    begin
      if (FAxes[i].AutoScale) then
      begin
        if ((FPlotType = ptContour) or
            (FPlotType = ptLineContour)) then
        begin
          FAxes[1].Min := FSeriesList.ZMin;
          FAxes[1].Max := FSeriesList.ZMax;
        end
        else
        begin
          FAxes[1].Min := FSeriesList.YMin;
          FAxes[1].Max := FSeriesList.YMax;
        end; {Contour}
      end; {AutoScale}
    end; {is Y Axis}
  end; {loop over Axes} *)

  if (FAxes.Count = 2) then
  begin {only 1 y axis:}
    if (FAxes[1].AutoScale) then
    begin
      if ((FPlotType = ptContour) or
          (FPlotType = ptLineContour)) then
        FAxes[1].SetMinMax(FSeriesList.ZMin, FSeriesList.ZMax)
       else
        FAxes[1].SetMinMax(FSeriesList.YMin, FSeriesList.YMax);
    end; {AutoScale}
  end
  else
  begin
    for i := 1 to FAxes.Count-1 do
      FAxes[i].ClearNewMinMax;
    for i := 0 to FSeriesList.Count-1 do
    begin
      ASeries := FSeriesList[i];
      AnAxis := ASeries.YAxis;
      if (AnAxis.AxisType >= atPrimaryY) then
      begin
        if (AnAxis.AutoScale) then
        begin
          if ((FPlotType = ptContour) or
              (FPlotType = ptLineContour)) then
            AnAxis.StoreNewMinMax(FSeriesList.ZMin, FSeriesList.ZMax)
           else
            AnAxis.StoreNewMinMax(ASeries.YMin, ASeries.YMax);
        end; {AutoScale}
      end; {Y Axis}
    end; {for Series}
    for i := 1 to FAxes.Count-1 do
      if ((FAxes[i].AutoScale) and (FAxes[i].AxisType >= atPrimaryY)) then
        FAxes[i].UseNewMinMax;
  end;

{Limit the X Axis intercept:}
  LimitXAxis;
  if (FAxes[0].AutoZero) then
  begin
    if ((FAxes[1].Min <= 0) and (0 <= FAxes[1].Max)) then
      FAxes[0].Intercept := 0
     else
      FAxes[0].Intercept := FAxes[1].Min;
  end;

{Limit the Y Axis intercept:}
  LimitYAxis;
  if (FAxes[1].AutoZero) then
  begin
    if ((FAxes[0].Min <= 0) and (0 <= FAxes[0].Max)) then
      FAxes[1].Intercept := 0
     else
      FAxes[1].Intercept := FAxes[0].Min;
  end;

{Limit the secondary Y Axes intercepts:}
  for i := 2 to FAxes.Count-1 do
  begin
    case FAxes[i].AxisType of
      atPrimaryY, atSecondaryY:
        begin
          if (FAxes[i].Intercept < FAxes[0].Min) then
            FAxes[i].Intercept := FAxes[0].Min;
          if (FAxes[i].Intercept > FAxes[0].Max) then
            FAxes[i].Intercept := FAxes[0].Max;
        end;
      atTertiaryY:
        begin
          if (FAxes[i].Intercept < FAxes[0].XofF(1)) then
            FAxes[i].Intercept := FAxes[0].XofF(1);
          if (FAxes[i].Intercept > FAxes[0].XofF(Width-2)) then
            FAxes[i].Intercept := FAxes[0].XofF(Width-2);
        end
    else
      FAxes[i].DoGeometry;
    end;
  end;

{Set the screen positions based on the Intercepts:}
  FAxes[0].MidY := FAxes[1].FofY(FAxes[0].Intercept);
  FAxes[1].MidX := FAxes[0].FofX(FAxes[1].Intercept);
  for i := 2 to FAxes.Count-1 do
  begin
    if (FAxes[i].AxisType >= atPrimaryY) then
      FAxes[i].MidX := FAxes[0].FofX(FAxes[i].Intercept);
  end;

  if (Assigned(ZAxis)) then
  begin
{Limit the Z Axis interceptS:}
    LimitZAxis;
    if (ZAxis.AutoZero) then
      if ((ZAxis.Min <= 0) and (0 <= ZAxis.Max)) then
        ZAxis.Intercept := 0;
{Set the screen positions based on the Intercepts:}
    TheOrigin.x := FAxes[0].FofX(ZAxis.Intercept);
    TheOrigin.y := FAxes[1].FofY(ZAxis.ZInterceptY);
    ZAxis.FireEvents := FALSE;
    ZAxis.Origin := TheOrigin;
    ZAxis.FireEvents := TRUE;
    {if ((ZAxis.EndX <= 0) or (Width <= ZAxis.EndX) or (Height <= ZAxis.EndY)) then
    begin
      FBorder.ScaleBy(ZAxis.EndX, ZAxis.EndY);
      SetAxisDimensions;
    end;}
  end;

{Put the main title into place:}
  FTitle.DoCanvasGeometry(ACanvas);
  FTitle.Left := FBorder.Left + (FBorder.Width) * FTitle.Position div 100;
  if (FTitle.DefaultSide) then
    FTitle.Top := (FBorder.Top - FTitle.Height) div 2
   else
    FTitle.Top := Height - 3 * FTitle.Height div 2;

{do the borders for click-and-drag purposes:}
(*
  mLeftBorder.Top := FBorder.Top;
  mLeftBorder.Bottom := FBorder.Bottom;
  mLeftBorder.MidX := FBorder.Left;

  mRightBorder.Top := FBorder.Top;
  mRightBorder.Bottom := FBorder.Bottom;
  mRightBorder.MidX := FBorder.Right;

  mTopBorder.Left := FBorder.Left;
  mTopBorder.Right := FBorder.Right;
  mTopBorder.MidY := FBorder.Top;

  mBottomBorder.Left := FBorder.Left;
  mBottomBorder.Right := FBorder.Right;
  mBottomBorder.MidY := FBorder.Bottom;
*)
  PaintUnLock;
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetClickAndDragDelay
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the ClickAndDragDelay Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetClickAndDragDelay(Value: Integer);
begin
  MouseTimer.Interval := Value;
end;
{$ENDIF} {GUI SetClickAndDragDelay}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetColor2
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Color2 for gradient backgrounds
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetColor2(Value: TColor);
begin
  if (FColor2 <> Value) then
  begin
    FColor2 := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetColorPlot
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Color of the Plot area
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetColorPlot(Value: TColor);
begin
  if (FColorPlot <> Value) then
  begin
    FColorPlot := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetColorType
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Color scheme for environments
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetColorType(Value: TGridType);
begin
  if (FColorType <> Value) then
  begin
    FColorType := Value;
    DoStyleChange(Self);
  end;
end;

{$IFNDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetColor
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Color for NON_GUI environments
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetColor(Value: TColor);
begin
  if (FColor <> Value) then
  begin
    FColor := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetFont
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Font for NON_GUI environments
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetHeight
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Height for NON_GUI environments
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetHeight(Value: Integer);
begin
  if (FHeight <> Value) then
  begin
    FHeight := Value;
    Resize;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetWidth
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/23/2001
Date modified: 08/23/2001 by Mat Ballard
      Purpose: sets the Width for NON_GUI environments
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetWidth(Value: Integer);
begin
  if (FWidth <> Value) then
  begin
    FWidth := Value;
    Resize;
  end;
end;
{$ENDIF} {GUI SetColor}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetColumnGap
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Gap between Columns in ptColumn mode
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetColumnGap(Value: TPercent);
begin
  if (FColumnGap <> Value) then
  begin
    FColumnGap := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetDefaultExtension
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DefaultExtension for TPlot files
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetDefaultExtension(
  Value: String);
begin
  if (Length(Value) = 0) then
    exit;

  if (Value[1] <> '.') then
    Value := '.' + Value;
  FDefaultExtension := Value;
  FileExtensions[0] := Value;
  FileTypes :=
{$IFDEF MSWINDOWS}
    'Plot Files|*' + Value +
    '|Comma Sep Var Files|*.csv' +
    '|Text Files|*.txt' +
    '|HTML Files|*.htm;*.html' +
    '|All Files|*.*';
{$ENDIF}
{$IFDEF LINUX}
    'Plot Files(*' + Value + ')' +
    '|Comma Sep Var Files (*.csv)' +
    '|Text Files (*.txt)' +
    '|HTML Files(*.htm)' +
    '|All Files (*.*)';
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetDisplayMode
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DisplayMode property, which is how
               graphs are updated when more data is Added
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetDisplayMode(
  Value: TDisplayMode);
begin
  if (FDisplayMode <> Value) then
    SetDisplayModeHistory(FDisplayHistory, Value);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetDisplayModeHistory
  Description: adjusts axes and sets DisplayMode and History properties
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: DisplayMode and History must be set and the graph updated simultaneously.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetDisplayModeHistory(
  HistoryValue: Single;
  ScalingValue: TDisplayMode);
begin
  if (ScalingValue = dmHistory) then
  begin
{we are changing to History from normal behaviour,
 or we are in History mode:}
    FAxes[0].Min := -HistoryValue;
    FAxes[0].Max := 0;
    FAxes[1].Intercept := -HistoryValue;
  end
  else if (FDisplayMode = dmHistory) then
  begin
{We are changing from History to normal behaviour.
 We therefore need to reset the X Axis dimensions:}
    if (ScalingValue = dmRun) then
      FAxes[0].Max := 1.5 * FSeriesList.Xmax
    else
      FAxes[0].Max := FSeriesList.Xmax;
    FAxes[0].Min := FSeriesList.Xmin;
    FAxes[1].Intercept := FAxes[0].Min;
  end;
  FDisplayHistory := HistoryValue;
  FDisplayMode := ScalingValue;
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetFileName
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/25/2001 by Mat Ballard
      Purpose: sets and parses the FileName property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetFileName(Value: String);
begin
  if (FFileName <> Value) then
  begin
    FFileName := Value;
    FTitle.Caption := ExtractFileRoot(Value);
  end;
end;


function TCustomPlot.GetFileDriveDir: String;  {D:\Data\Delphi\Plot}
var
  FileDriveDir: String;
begin
  FileDriveDir := ExtractFilePath(FFileName);
  if (Length(FileDriveDir) = 0) then
    FileDriveDir := GetCurrentDir;
  if (FileDriveDir[Length(FileDriveDir)] <> DIR_SEPERATOR) then
    FileDriveDir := FileDriveDir + DIR_SEPERATOR;
  GetFileDriveDir := FileDriveDir;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetFilterIndex
  Description: gets the file filter index from the extension
       Author: Mat Ballard
 Date created: 08/10/2000
Date modified: 08/10/2000 by Mat Ballard
      Purpose: file management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetFilterIndex(
  Ext: String): Integer;
var
  i: Integer;
begin
{the default filterindex is actually '*'}
  GetFilterIndex := 5;  //".*"
  for i := 0 to 4 do
  begin
    if (LowerCase(Ext) = FileExtensions[i]) then
    begin
      GetFilterIndex := i+1;
      break;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetHistory
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the History property: which is how far back
               a History graph goes
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetHistory(
  Value: Single);
begin
  if (FDisplayHistory <> Value) then
    SetDisplayModeHistory(Value, FDisplayMode);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetInstructions
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 10/09/1999
Date modified: 10/09/2000 by Mat Ballard
      Purpose: sets the Instructions property
 Known Issues: Fixed IDE crash when Instructions set
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetInstructions(Value: String);
begin
  FInstructions := Value;
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetInstructionsText
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 07/23/2001
Date modified: 07/23/2001 by Mat Ballard
      Purpose: sets the Instructions property by its text
 Known Issues:
 ------------------------------------------------------------------------------
procedure TCustomPlot.SetInstructionText(Value: String);
begin
  FInstructions.Text := Value;
  DoStyleChange(Self);
end;}

{$IFDEF GUI}
  {$IFDEF MSWINDOWS}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetMetafileDescription
  Description: sets the CreatedBy and Description properties
               if they are not yet set
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: fully utilise enhanced metafile capabilities to
               put keywords into WMF
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetMetafileDescription;
var
  i: Integer;
begin
  if (Length(FCreatedBy) = 0) then
    FCreatedBy := 'Chemware';
  {$IFDEF GUI}
  FCreatedBy := InputBox('Author', 'Please enter your name', FCreatedBy);
  {$ENDIF}

  if (Length(FDescription) = 0) then
  begin
    FDescription := FTitle.Caption + ': ';
    for i := 0 to FSeriesList.Count-1 do
    begin
      FDescription := FDescription + TSeries(FSeriesList[i]).Name + ', ';
    end;
{remove trailing ', ':}
    SetLength(FDescription, Length(FDescription)-2);
  end;
  {$IFDEF GUI}
  FDescription := InputBox('Author', 'Please enter a description', FDescription);
  {$ENDIF}
end;
  {$ENDIF} {MSWINDOWS SetMetafileDescription}
{$ENDIF} {GUI SetMetafileDescription}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetPieRowCount
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the PieRowCount property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetPieRowCount(Value: Byte);
begin
  if ((Value > 0) and
      (Value <= FSeriesList.Count) and
      (Value <> FPieRowCount)) then
  begin
    FPieRowCount := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetNoSeries
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 22/12/2000
Date modified: 22/12/2000 by Mat Ballard
      Purpose: sets the NoSeries property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetNoSeries(Value: Word);
begin
  if (Value > FSeriesList.Count) then
  begin
{note that the series are added with independent X data:}
    while (FSeriesList.Count < Value) do
      FSeriesList.Add;
    if (csDesigning in ComponentState) then
      MakeDummyData(20);
  end
  else if (Value < FSeriesList.Count) then
  begin
    while (FSeriesList.Count >  Value) do
      FSeriesList.Delete(FSeriesList.Count-1);
  end;
end;

{procedure TCustomPlot.SetActionList(Value: TActionList);
begin
  FActionList.Assign(Value);
end;}

procedure TCustomPlot.SetAxes(Value: TAxisList);
begin
  FAxes.Assign(Value);
  Self.DoStyleChange(Self);
end;

procedure TCustomPlot.SetNotes(Value: TNoteList);
begin
  FNotes.Assign(Value);
  Self.DoStyleChange(Self);
end;

procedure TCustomPlot.SetSeriesList(Value: TSeriesList);
begin
  FSeriesList.Assign(Value);
  Self.DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetStartTime
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 02/28/2003
Date modified: 02/28/2003 by Mat Ballard
      Purpose: sets the SetStartTime Property
 Known Issues: formerly in TSeries 
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetStartTime(Value: TDateTime);
begin
  FStartTime := Value;
  DoStyleChange(Self);
end;

procedure TCustomPlot.SetTexts(Value: TTextList);
begin
  FTexts.Assign(Value);
  Self.DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetOutlineWidth
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the OutlineWidth property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetOutlineWidth(Value: Integer);
var
  i: Integer;
begin
  if (FOutlineWidth <> Value) then
  begin
  {Set border widths:}
    FOutlineWidth := Value;
(*
    mLeftBorder.Height := FOutlineWidth;
    mLeftBorder.Width := FOutlineWidth;
    mTopBorder.Height := FOutlineWidth;
    mTopBorder.Width := FOutlineWidth;
    mRightBorder.Height := FOutlineWidth;
    mRightBorder.Width := FOutlineWidth;
    mBottomBorder.Height := FOutlineWidth;
    mBottomBorder.Width := FOutlineWidth;
*)
  {Set axis widths:}
    //if not (csLoading in ComponentState) then
    for i := 0 to FAxes.Count-1 do
      FAxes[i].Breadth := FOutlineWidth;

    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetOnSelection
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the OnSelection event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetOnSelection(Value: TOnSelectionEvent);
begin
  FOnSelection := Value;
  ScreenJob := sjSelection;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetOnDualSelection
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the OnDualSelection event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetOnDualSelection(Value: TOnDualSelectionEvent);
begin
  FOnDualSelection := Value;
  ScreenJob := sjDualSelection1;
end;


{The painting/drawing methods -----------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PaintLock
  Description: prevents painting of the graph
       Author: Mat Ballard
 Date created: 08/02/2002
Date modified: 08/02/2002 by Mat Ballard
      Purpose: increments the paint lock number to prevent painting
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PaintLock;
begin
  Inc(mPaintLock);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PaintUnLock
  Description: allows painting of the graph
       Author: Mat Ballard
 Date created: 08/02/2002
Date modified: 08/02/2002 by Mat Ballard
      Purpose: decrements the paint lock number to allow painting
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PaintUnLock;
begin
  Dec(mPaintLock);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Paint
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: paints the background, border, then draws the graph: NOT called by graphics and printer.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Paint;
var
  iX, iY: Integer;
  TopColor,
  BottomColor: TColor;

  procedure AdjustColors(Bevel: TPanelBevel);
  begin
    TopColor := clBtnHighlight;
    if Bevel = bvLowered then TopColor := clBtnShadow;
    BottomColor := clBtnShadow;
    if Bevel = bvLowered then BottomColor := clBtnHighlight;
  end;

begin
  try
    if (mPaintLock > 0) then exit;

  {$IFDEF DELPHI3_UP}
    Assert(Canvas <> nil, 'TCustomPlot.Paint: Canvas is nil !');
  {$ENDIF}

  {$IFDEF THREAD_SAFE}
{This is the heavyweight locking mechanism for multi-threaded apps:}
    Canvas.Lock;
  {$ENDIF}

    if Assigned(FOnBeforePaint) then
      OnBeforePaint(Self, Canvas);

    mClientRect := GetClientRect;

    Canvas.Pen.Mode := pmCopy;
    Canvas.Pen.Style := psSolid;

{Do borders and Bevels:}
    //BevelGap := BorderWidth;
    if BevelOuter <> bvNone then
    begin
      //Inc(BevelGap, BevelWidth);
      AdjustColors(BevelOuter);
      Frame3D(Canvas, mClientRect, TopColor, BottomColor, BevelWidth);
    end;
    Frame3D(Canvas, mClientRect, Color, Color, BorderWidth);
    if BevelInner <> bvNone then
    begin
      //Inc(BevelGap, BevelWidth);
      AdjustColors(BevelInner);
      Frame3D(Canvas, mClientRect, TopColor, BottomColor, BevelWidth);
    end;

{paint the background:}
    if ((Color = FColor2) or (FColorType = gtNone)) then
    begin
      Canvas.Brush.Color := Color;
      Canvas.FillRect(mClientRect)
    end;

    Canvas.Brush.Style := bsClear;

    Draw(Canvas);

    Canvas.Brush.Color := Color;
    Canvas.Brush.Style := bsClear;
{The Instructions are usually an instruction to the user.
   As such, it does not need to be copied or printed,
   so it is placed here, rather than in the "Draw" method:}
    PaintInstructions;

{ditto with the PageButtons}
    if (Assigned(mPageButtons[0])) then
    begin
{Right:}
      iX := Self.Width - mPageButtons[1].Width - 1;
      iY := Self.Height - mPageButtons[1].Height - 1;
      Canvas.Draw(iX, iY, mPageButtons[1]);
{Down}
      iX := iX - mPageButtons[3].Width;
      Canvas.Draw(iX, iY, mPageButtons[3]);
{Up}
      Canvas.Draw(iX, iY - mPageButtons[2].Height, mPageButtons[2]);
{Left}
      iX := iX - mPageButtons[0].Width;
      Canvas.Draw(iX, iY, mPageButtons[0]);
    end;

{Do we need to check that a 3D plot has not exceeded the control boundaries ?}
  {if (FPlotType >= pt3DContour) then
    begin
      if ((ZAxis.EndX < 0) or (Width < ZAxis.EndX) or (Height < ZAxis.EndY)) then
        FBorder.ScaleBy(ZAxis.EndX, ZAxis.EndY);
      Invalidate;
    end;}

    if Assigned(FOnAfterPaint) then
      OnAfterPaint(Self, Canvas);

  {$IFDEF THREAD_SAFE}
    Canvas.UnLock;
  {$ENDIF}
  finally
    mPaintLock := 0;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PaintInstructions
  Description: draws the instructions
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: tell the user what to do
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PaintInstructions;
var
  FontHeight,
  iX, iY, iPos, Len: Integer;
  Str: String;
begin
{The Instructions are usually an instruction to the user.
 As such, it does not need to be copied or printed,
 so it is placed here, rather than in the "Draw" method:}
  Len := Length(FInstructions);
  if (Len > 0) then
  begin
    Canvas.Font.Assign(Font);
    FontHeight := Canvas.TextHeight('Wp');
{Adjust the Position appropriately:}
    iX := BevelWidth + 5;
    iY := Height - BevelWidth - 5 - FontHeight;
{how many lines ?}
    repeat
      iPos := Misc.BackPos(#10, FInstructions, Len);
      Str := Copy(FInstructions, iPos+1, Len - iPos);
      Canvas.TextOut(iX, iY, Str);
      Dec(iY, FontHeight);
      Len := iPos - 1;
      if (FInstructions[Len] = #13) then
        Dec(Len);
    until iPos = 0;
    {for i := FInstructions.Count-1 downto 0 do
    begin
      Canvas.TextOut(iX, iY, FInstructions[i]);
      Dec(iY, FontHeight);
    end;}
  end;
end;
{$ENDIF} {GUI Paint}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Draw
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: draws the graph on a canvas: graphics and printers call this procedure directly
 Known Issues:
        Notes: order of drawing is:
               1. Bevels in Paint
               2. Instructions in Paint
               3. Outer background
               4. Inner (Plot) background
               5. Border, if any
               6. Title
               7. Walls, if any
               8. Grid, if any
               9. Axes
              10. Legend
              11. Series
              12. Notes, if any
              13. text, if any
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Draw(
  ACanvas: TCanvas);
var
  i: Integer;
  Pts: TQuad;
begin
  if (mPaintLock > 0) then exit;

{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TCustomPlot.Draw: ACanvas is nil !');
{$ENDIF}

{$IFDEF THREAD_SAFE}
{This is the heavyweight locking mechanism fro multi-threaded apps:}
  ACanvas.Lock;
{$ENDIF}

{This is our own lightweight locking mechanism that prevents multiple screen redraws:}
  PaintLock;

{We have to set the Axis dimensions first because much of the subsequent Wall
 and Grid operations depend on the axes in the new TBorder mode:}
  SetAxisDimensions(ACanvas);

{Fill in the background:}
  if ((Color <> FColor2) and (FColorType > gtNone)) then
  begin // with a gradient:
    Pts[0] := Point(mClientRect.Left, mClientRect.Top);
    Pts[1] := Point(mClientRect.Right, mClientRect.Top);
    Pts[2] := Point(mClientRect.Right, mClientRect.Bottom);
    Pts[3] := Point(mClientRect.Left, mClientRect.Bottom);
    FillWall(Canvas, Color, FColor2, FColorType, Pts);
  end;
{Fill in the Plot:}
  if ((FColorPlot <> Color) or (FColorPlot <> FColor2)) then
  begin
    Canvas.Brush.Color := ColorPlot;
    Canvas.FillRect(Rect(FAxes[0].Left, FAxes[1].Top, FAxes[0].Right, FAxes[1].Bottom));
  end;

  if Assigned(FOnBeforeDraw) then
    OnBeforeDraw(Self, ACanvas);

  if (FBorder.Visible) then
  begin
    Canvas.Brush.Color := FAxes[0].Pen.Color;
{$IFDEF WIN32}
    Canvas.FrameRect(Rect(FAxes[0].Left, FAxes[1].Top, FAxes[0].Right+1, FAxes[1].Bottom+1));
{$ENDIF}
{$IFDEF LINUX}
    Canvas.Rectangle(FAxes[0].Left, FAxes[1].Top, FAxes[0].Right+1, FAxes[1].Bottom+1);
{$ENDIF}
  end;

  Canvas.Brush.Style := bsClear;

  FTitle.Draw(ACanvas);

  if (FGrids.WallType > gtNone) then
    DrawWalls(ACanvas);
  if (FGrids.GridType > gtNone) then
    DrawGrid(ACanvas);

  FAxes[0].Draw(ACanvas, FAxes[1].Vector);
  FAxes[1].Draw(ACanvas, FAxes[0].Vector);
  for i := 2 to FAxes.Count-1 do
    FAxes[i].Draw(ACanvas, FAxes[1].Vector);

  FLegend.Draw(ACanvas);

  ACanvas.Font.Assign(FHighFont);
  case FPlotType of
    ptXY:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.Draw(ACanvas, FXYFastAt)
         else
          FSeriesList.DrawHistory(ACanvas, FDisplayHistory);
      end;
    ptColumn:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawColumns(ACanvas, FColumnGap);
      end;
    ptStack:
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawStack(ACanvas, FColumnGap);
    ptNormStack:
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawNormStack(ACanvas, FColumnGap);
    ptPie:
        if (FDisplayMode < dmHistory) then
        begin
          FSeriesList.DrawPie(ACanvas, FBorder, FPieRowCount);
        end;
    ptPolar:
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawPolar(ACanvas, FPolarRange);
    ptLineContour:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawLineContour(ACanvas, FContour);
      end;
    ptContour:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawContour(ACanvas, FContour);
      end;
    pt3DContour:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.Draw3DContour(ACanvas, FContour);
      end;
    pt3DWire:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.Draw3DWire(ACanvas, FZLink);
      end;
    pt3DColumn:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.Draw3DColumn(ACanvas, FColumnGap)
      end;
  end;

  for i := 0 to FNotes.Count-1 do
    TNote(FNotes.Items[i]).Draw(ACanvas);
  for i := 0 to FTexts.Count-1 do
    TText(FTexts.Items[i]).Draw(ACanvas);

  if Assigned(FOnAfterDraw) then
    OnAfterDraw(Self, ACanvas);

  PaintUnLock;
{$IFDEF THREAD_SAFE}
  ACanvas.UnLock;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DrawGrid
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/06/2002 by Mat Ballard
      Purpose: draws the grid for xy-type plots
     Comments:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DrawGrid(
  ACanvas: TCanvas);
var
  iX, iY: Integer;
  X, Y, Z: Single;
  dZ,
  TheTickStart: TPoint;
begin
{reset the pen:}
  ACanvas.Brush.Style := bsClear;
  ACanvas.Pen.Color := FGrids.Color;
  ACanvas.Pen.Width := 1;
  ACanvas.Pen.Style := FGrids.Style;

  if (FPlotType >= pt3DContour) then
    dZ := ZAxis.Vector;

{do the XY and XZ grids:}
  if (FGrids.GridType > gtHorizontal) then
  begin
    X := FAxes[0].StepStart - FAxes[0].StepSize;
    if (X < FAxes[0].Min) then
      X := FAxes[0].StepStart;
    while (X <= FAxes[0].Max) do
    begin
      iX := FAxes[0].FofX(X);
{XY: vertical |||}
      ACanvas.MoveTo(iX, FAxes[1].Top);
      ACanvas.LineTo(iX, FAxes[1].Bottom);
{XZ: vertical ///}
      if (FPlotType >= pt3DContour) then
        ACanvas.LineTo(iX + dZ.x, FAxes[1].Bottom + dZ.y);
      X := FAxes[0].GetNextXValue(X);
    end;
  end;

{do the XY and YZ grids:}
  if ((FGrids.GridType = gtHorizontal) or (FGrids.GridType = gtBoth)) then
  begin
    Y := FAxes[1].StepStart - FAxes[1].StepSize;
    if (Y < FAxes[1].Min) then
      Y := FAxes[1].StepStart;
    while (Y <= FAxes[1].Max) do
    begin
      iY := FAxes[1].FofY(Y);
      if (iY <> FAxes[0].MidY) then
      begin
{XY: horizontal ---}
        ACanvas.MoveTo(FAxes[0].Left, iY);
        ACanvas.LineTo(FAxes[0].Right, iY);
        if (FPlotType >= pt3DContour) then
        begin
          if (ZAxis.Angle >= 180) then
          begin
{YZ: horizontal ///}
            ACanvas.MoveTo(FAxes[0].Left, iY);
            ACanvas.LineTo(FAxes[0].Left + dZ.x, iY + dZ.y);
          end
          else
          begin
{YZ: horizontal ///}
            ACanvas.MoveTo(FAxes[0].Right, iY);
            ACanvas.LineTo(FAxes[0].Right + dZ.x, iY + dZ.y);
          end;
        end;
      end;
      Y := FAxes[1].GetNextXValue(Y);
    end;
  end;

{do the horizontal Xs and vertical Zs:}
  if (FPlotType >= pt3DContour) then
  begin
    Z := ZAxis.StepStart;
    while (Z <= ZAxis.Max) do
    begin
      TheTickStart := ZAxis.dFofZ(Z);
      Inc(TheTickStart.x, FAxes[0].Left);
      Inc(TheTickStart.y, FAxes[1].Bottom);
{XZ: horizontal: ---}
      if ((FGrids.GridType = gtHorizontal) or (FGrids.GridType = gtBoth)) then
      begin
        ACanvas.MoveTo(TheTickStart.x + FAxes[0].Width, TheTickStart.y);
        ACanvas.LineTo(TheTickStart.x, TheTickStart.y);
      end;
{YZ: vertical}
      if (FGrids.GridType > gtHorizontal) then
      begin
        if (ZAxis.Angle >= 180) then
        begin
          ACanvas.MoveTo(TheTickStart.x, TheTickStart.y);
          ACanvas.LineTo(TheTickStart.x, TheTickStart.y - FAxes[1].Height);
        end
        else
        begin
          ACanvas.MoveTo(TheTickStart.x + FAxes[0].Width, TheTickStart.y);
          ACanvas.LineTo(TheTickStart.x + FAxes[0].Width, TheTickStart.y - FAxes[1].Height);
        end;
      end;
      Z := ZAxis.GetNextXValue(Z);
    end; {while}
  end; {if FPlotType >= pt3DContour}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.FillWall
  Description: painting the graph
       Author: Mat Ballard
 Date created: 08/06/2002
Date modified: 09/01/2002 by Mat Ballard
      Purpose: draws the grids for 3D plots
     Comments:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.FillWall(ACanvas: TCanvas; Col1, Col2: TColor; FillType: TGridType; TheWall: TQuad);
var
  iix, iiy, iix1, iiy1, iStep, Span, aHeight, aWidth: Integer;
  dX, dY: Single;
begin
  if (Col1 = Col2) then
  begin
    ACanvas.Pen.Style := psClear;
    ACanvas.Brush.Color := FGrids.WallColor;
    ACanvas.Polygon(TheWall);
  end
  else if (FillType = gtHorizontal) then    //
  begin                                            //   0--dX--0---aWidth---1
    ACanvas.Pen.Style := psSolid;                  //   |     /            /
    Span := TheWall[3].y - TheWall[0].y;           //   S    S            /
    dX := (TheWall[3].x - TheWall[0].x) / Span;    //   p   p            /
    aWidth := TheWall[1].x - TheWall[0].x;         //   a  a            /
    iStep := 0;                                    //   n n            /
    iiy1 := TheWall[1].y;                          //   |/            /
    for iiy := TheWall[0].y to TheWall[3].y do     //   3------------2
    begin                                          //
      ACanvas.Pen.Color :=
        Misc.GetColorMix(Col1, Col2, (iiy-TheWall[0].y)/Span);
      iix := TheWall[0].x + Round(dX * iStep);
      ACanvas.MoveTo(iix, iiy);
      ACanvas.LineTo(iix+aWidth, iiy1);
      Inc(iStep);
      Inc(iiy1);
    end;
  end
  else //if (FillType = gtHorizontal) then  //
  begin                                            //   0------0----Span----1
    ACanvas.Pen.Style := psSolid;                  //   |     /            /
    Span := TheWall[1].x - TheWall[0].x;           //   |    /            /
    dY := (TheWall[1].y - TheWall[0].y) / Span;    //   d   /            /
    aHeight := TheWall[3].y - TheWall[0].y;        //   Y  /            /
    iStep := 0;                                    //   | /            /
    iix1 := TheWall[3].x;                          //   |/            /
    for iix := TheWall[0].x to TheWall[1].x do     //   3------------2
    begin                                          //
      ACanvas.Pen.Color :=
        Misc.GetColorMix(Col1, Col2, (iix-TheWall[0].x)/Span);
      iiy := TheWall[0].y + Round(dY * iStep);
      ACanvas.MoveTo(iix, iiy);
      ACanvas.LineTo(iix1, iiy+aHeight);
      Inc(iStep);
      Inc(iix1);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DrawWalls
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/06/2002 by Mat Ballard
      Purpose: draws the grids for 3D plots
     Comments:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DrawWalls(
  ACanvas: TCanvas);
var
  dZ: TPoint;
  TheWall: TQuad;
begin
  ACanvas.Brush.Style := bsSolid;

{Calculate the XY wall:
  TheWall[0].x := FBorder.Left;       //        1  0--------1  0
  TheWall[0].y := FBorder.Top;        //       /|  |        |  |\
  TheWall[1].x := FBorder.Right;      //      / |  |        |  | \
  TheWall[1].y := FBorder.Top;        //     0  |  |        |  |  1
  TheWall[2].x := FBorder.Right;      //     |  |  |        |  |  |
  TheWall[2].y := FBorder.Bottom;     //     |  |  |        |  |  |
  TheWall[3].x := FBorder.Left;       //     |  2  3--------2  3  |
  TheWall[3].y := FBorder.Bottom;     //     | /                \ |
                                      //     |/    0--------1    \|
                                      //         /        /
                                      //        3--------2         }
{Calculate the XY wall:}
  TheWall[0].x := FAxes[0].Left;       //        1  0--------1  0
  TheWall[0].y := FAxes[1].Top;        //       /|  |        |  |\
  TheWall[1].x := FAxes[0].Right;      //      / |  |        |  | \
  TheWall[1].y := FAxes[1].Top;        //     0  |  |        |  |  1
  TheWall[2].x := FAxes[0].Right;      //     |  |  |        |  |  |
  TheWall[2].y := FAxes[1].Bottom;     //     |  |  |        |  |  |
  TheWall[3].x := FAxes[0].Left;       //     |  2  3--------2  3  |
  TheWall[3].y := FAxes[1].Bottom;     //     | /                \ |
                                      //     |/    0--------1    \|
                                      //         /        /
                                      //        3--------2
{Draw the XY wall:}
  if ((FPlotType < ptLineContour) or (FPlotType > ptLineContour)) then
  begin
    if (FGrids.WallType = gtHorizontal) then
      FillWall(ACanvas, FGrids.WallColor, FGrids.WallColor2, FGrids.WallType, TheWall)
     else
      FillWall(ACanvas, FGrids.WallColor2, FGrids.WallColor, FGrids.WallType, TheWall)
  end;

{3D plots:}
  if (FPlotType >= pt3DContour) then
  begin
{Calculate the YZ wall:}
    dZ := ZAxis.Vector;

    if (ZAxis.Angle >= 180) then
    begin
      TheWall[0].x := FAxes[0].Left + dZ.x;
      TheWall[0].y := FAxes[1].Top + dZ.y;
      TheWall[1].x := FAxes[0].Left;
      TheWall[1].y := FAxes[1].Top;
      TheWall[2].x := FAxes[0].Left;
      TheWall[2].y := FAxes[1].Bottom;
      TheWall[3].x := TheWall[0].x;
      TheWall[3].y := FAxes[1].Bottom + dZ.y;
    end
    else
    begin
      TheWall[0].x := FAxes[0].Right;
      TheWall[0].y := FAxes[1].Top;
      TheWall[1].x := FAxes[0].Right + dZ.x;
      TheWall[1].y := FAxes[1].Top + dZ.y;
      TheWall[2].x := TheWall[1].x;
      TheWall[2].y := FAxes[1].Bottom + dZ.y;
      TheWall[3].x := FAxes[0].Right;
      TheWall[3].y := FAxes[1].Bottom;
    end;
{Draw the YZ left wall:}
    FillWall(ACanvas, FGrids.WallColor, FGrids.WallColor2, FGrids.WallType, TheWall);

{Calculate the XZ floor:}
    TheWall[0].x := FAxes[0].Left;
    TheWall[0].y := FAxes[1].Bottom;
{These four are repeated because if angle < 180 ...}
    TheWall[1].x := FAxes[0].Right;
    TheWall[1].y := FAxes[1].Bottom;
    TheWall[2].x := FAxes[0].Right + dZ.x;
    TheWall[2].y := FAxes[1].Bottom + dZ.y;
    TheWall[3].x := FAxes[0].Left + dZ.x;
    TheWall[3].y := TheWall[2].y;

{Draw the XZ floor:}
    FillWall(ACanvas, FGrids.WallColor2, FGrids.WallColor, FGrids.WallType, TheWall)
  end;
  ACanvas.Brush.Style := bsClear;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Trace
  Description: This traces all series: useful for Oscilloscopes
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Draws all Series in erasable mode
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Trace;
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(Canvas <> nil, 'TCustomPlot.Trace: Canvas is nil !');
{$ENDIF}

  for i := 0 to FSeriesList.Count-1 do
  begin
    TSeries(FSeriesList[i]).Trace(Canvas);
  end;
end;

{$IFNDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Refresh
  Description: overrides ancestor's ReSize
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: responds to a resize of the Plot
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Refresh;
begin
  Resize;
end;
{$ENDIF} {GUI Refresh}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Resize
  Description: overrides ancestor's ReSize
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: responds to a resize of the Plot
 Known Issues:
 ------------------------------------------------------------------------------}

procedure TCustomPlot.Resize;
begin
  PaintLock;
  if not (sLoaded in FStatus) then Loaded;
  if not (sUpdating in FStatus) then
    SetAxisDimensions(Self.Canvas);
{$IFDEF GUI}
  inherited Resize;
{$ENDIF}
  PaintUnLock;
end;

{Mousey stuff -----------------------------------------------------------------}

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DblClick
  Description: overrides ancestor's DblClick
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: activates in-place editing of titles
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DblClick;
begin
{get rid of the mouse moving timer:}
  MouseTimer.Enabled := FALSE;

  if (FEditable) then
  begin
    if ((FClickedObjectType = soTitle) or
        (FClickedObjectType = soXAxisTitle) or
        (FClickedObjectType = soYAxisTitle) or
        (FClickedObjectType = soText) or
        (FClickedObjectType = soNote)) then
    begin
      TText(pClickedObject).StartEdit(Self);
    end {Title or axis caption}
    else if (FClickedObjectType = soLegend) then
    begin
      TLegend(pClickedObject).StartEdit(Self, Point(Selection.Left, Selection.Top));
    end; {Legend}
  end; {editable}

  inherited DblClick;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.KeyDown
  Description: KeyDown event handler
       Author: Mat Ballard
 Date created: 04/22/2001
Date modified: 04/22/2001 by Mat Ballard
      Purpose: This processes certain key strokes.
 Known Issues: Does not work: a CustomPanel does not seem to be able to gain focus.
 ------------------------------------------------------------------------------}
{procedure TCustomPlot.KeyDown(var Key: Word; Shift: TShiftState);
var
  OldVar, Gap: Single;
begin
  case Key of
    VK_LEFT:
      begin
        if (ssCtrl in Shift) then
        begin
          if (FAxes[0].Min > FSeriesList.Xmin) then
          begin
            OldVar := FAxes[0].Min;
            Gap := FAxes[0].Max - FAxes[0].Min;
            FAxes[0].Min := FAxes[0].Min - Gap;
            FAxes[0].Max := FAxes[0].Max - Gap;
          end;
        end;
      end;
    VK_RIGHT:
      begin
        if (ssCtrl in Shift) then
        begin
          if (FAxes[0].Max < FSeriesList.Xmax) then
          begin
            OldVar := FAxes[0].Max;
            Gap := FAxes[0].Max - FAxes[0].Min;
            FAxes[0].Max := FAxes[0].Max + Gap;
            FAxes[0].Min := FAxes[0].Min + Gap;
          end;
        end;
      end;
  end;

  inherited KeyDown(Key, Shift);
end;}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseDown
  Description: MouseDown event handler
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/31/2000 by Mat Ballard
      Purpose: The start of all mouse routines
 Known Issues: MouseDown gets called AFTER DblClick !
      Changes: GetTheClickedObject now moved to within the 'if (FScreenJob = sjNone) then'
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseDown(
  Button: TMouseButton;
  Shift: TShiftState;
  X,
  Y: Integer);
begin
{Nuke any Caption:}
  if (Length(FInstructions) > 0) then
  begin
    FInstructions := '';
    Invalidate;
  end;

  if (mZeroStuff) then
    ZeroScreenStuff;

{record the beginning:}
  MouseStart.x := X;
  MouseStart.y := Y;
{set the moving object co-ordinates:}
  Selection.MoveTo(X, Y);
  Selection.Height := 1;
  Selection.Width := 1;

{if no ScreenJob has been set yet, it could be several things:}
  if (FScreenJob = sjNone) then
  begin
{no job yet}
{what got clicked ?}
    GetTheClickedObject(X, Y);
    if (Button = mbLeft) then
    begin
{left click:}
      if (ssShift in Shift) then
      begin
{We want to zoom in:}
        FScreenJob := sjZoomIn;
      end
      else if ((FClickedObjectType > soNone) and (FMovable)) then
      begin
{left clicks can lead to click and drag:}
        MouseTimer.Enabled := TRUE;
      end;
    end {left button}
    else if (Button = mbRight) then
    begin {Right Button}
      if (FClickedObjectType > soNone) then
      begin
        if (FClickedObjectType <> soSeries) then
        begin
          TAngleRect(pClickedObject).CopyTo(Selection);
          Selection.FireEvents := FALSE;
        end;
        OutlineTheClickedObject;
      end;
    end;
{NOTE: if it is the right button, then the popup menu will be displayed at the end of the MouseUp}
  end; {if sjNone}

  case FScreenJob of
    {sjNone: already done}
    {sjDrag: set by MouseTimeOut
    sjRightDrag: set in MouseMove}
    sjHide: HideClick(Self);
    sjZoomIn:
      begin
        Screen.Cursor := crSize;
        OutlineTheClickedObject;
      end;
    {sjEditAxis:}
    {sjTouchNote:}
    sjMoveNotePointer: ZeroScreenStuff;
    {sjEditFont:
    sjEditPoint:
    sjEditSeries:
    sjCopySeries: ;
    sjDisplace:   ;
    sjCloneSeries: ;
    sjDeleteSeries: all done by popupmenu or option}
    sjPosition: PositionClick(Self);
    sjNearestPoint: NearestPointClick(Self);
    sjLinearize,
    sjZero,
    sjAverage,
    sjContractSeries,
    sjContractAllSeries,
    {sjSplineSeries: ;
    sjHighs,
    sjLows,
    sjMovingAverage,
    sjSmoothSeries:   ;
    sjSortSeries: ;
    sjDifferentiate:   ;
    sjIntegrate: all done by popupmenu or option}
    sjIntegral,
    sjLineOfBestFit,
    sjDualLineBestFit1,
    sjDualLineBestFit2,
    sjSelection,
    sjDualSelection1,
    sjDualSelection2:
      begin
        Screen.Cursor := crSize;
        Selection.Outline(Canvas);
      end;
  end;

  inherited MouseDown(Button, Shift, X, Y);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseMove
  Description: MouseMove event handler
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Moves the dashed outline around the screen; how it moves depends on the object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseMove(
  Shift: TShiftState;
  X,
  Y: Integer);
var
  APosition,
  NewLeft,
  NewTop,
  dx, dy: Integer;
  pAxis: TAxis;
  ACenter: TPoint;
begin
  MouseTimer.Enabled := FALSE;

  if (FScreenJob = sjMoveNotePointer) then
    TNote(pClickedObject).TracePointerTo(Canvas, X, Y);

  if (ssLeft in Shift) then
  begin
    case FScreenJob of
      {sjNone:}
      sjDrag:
        begin
          case FClickedObjectType of
            soTitle:
              begin
                NewLeft := X - ClickedObjectOffset.x;
                if (NewLeft < FBorder.Left) then
                  NewLeft := FBorder.Left
                else if (NewLeft > FBorder.Right - Selection.Width) then
                  NewLeft := FBorder.Right - Selection.Width;
                if (Y < Height div 2) then
                  NewTop := (FBorder.Top - Selection.Height) div 2
                 else
                  NewTop := Height - 3*Selection.Height div 2;
                MoveTheClickedObjectTo(NewLeft, NewTop);
              end;
            soXAxis, soTopBorder, soBottomBorder:
              MoveTheClickedObjectTo(TRectangle(pClickedObject).Left, Y - ClickedObjectOffset.y);
            soYAxis, soLeftBorder, soRightBorder:
              MoveTheClickedObjectTo(X - ClickedObjectOffset.x, TRectangle(pClickedObject).Top);
            soZAxis:
              begin
                dx := X - MouseStart.x;
                dy := Y - MouseStart.y;
                if (Abs(dx) >= Abs(dy)) then
                  MoveTheClickedObjectTo(X - ClickedObjectOffset.x, TRectangle(pClickedObject).Top)
                 else
                  MoveTheClickedObjectTo(TRectangle(pClickedObject).Left, Y - ClickedObjectOffset.y);
              end;
            soZAxisEnd:
              begin
                Selection.Outline(Canvas);
                Selection.Vector := Point(X-Selection.Origin.x, Y - Selection.Origin.y);
                Selection.Outline(Canvas);
              end;
            soXAxisTitle, soYAxisTitle, soZAxisTitle:
              begin //pAxis.Vector
                Selection.Outline(Canvas);
{Which Y Axis owns this Title ?}
                pAxis := TAxis(TTitle(pClickedObject).Axis);
                case pAxis.Angle of
                  0, 180: APosition := 100 - 100 * (Y + ClickedObjectOffset.y - pAxis.EndY) div pAxis.Height;
                else // is a Z Axis:
                  APosition := 100 * (X - ClickedObjectOffset.x - pAxis.Origin.x) div (pAxis.EndX - pAxis.Origin.x);
                end;
                if (APosition < 0) then APosition := 0;
                if (APosition > 100) then APosition := 100;
                Selection.Centre := pAxis.GetTitleCentre(APosition, pAxis.IsDefaultSide(X, Y));
                Selection.Tag := APosition;
                Selection.Outline(Canvas);
              end;
            soXAxisLabel, soYAxisLabel, soZAxisLabel:
              begin
                Selection.Outline(Canvas);
{Which Y Axis owns this Label ?}
                pAxis := TAxisLabel(pClickedObject).Axis;
                ACenter := pAxis.Labels.Centre;
                if (pAxis.SameSideAs(X, Y, ACenter)) then
                  Selection.Centre := ACenter
                 else
                  Selection.Centre := pAxis.Reflect(ACenter);
                Selection.Outline(Canvas);
              end;
            soLegend, soText, soNote:
              begin {all of these can move freely:}
                MoveTheClickedObjectTo(
                  X - ClickedObjectOffset.x,
                  Y - ClickedObjectOffset.y);
              end;
            soSeries:
              begin
                FClickedSeries.MoveBy(Canvas, FPlotType, X-Selection.Left, Y-Selection.Top, FOutlineWidth);
                Selection.MoveTo(X, Y);
              end;
          end; {end case FClickedObjectType}
        end; {case sjDrag}
      {sjHide:}
      sjZoomIn:
        StretchTheClickedObjectTo(X, Y);
      {sjEditAxis: ;}
      {sjEditFont: ;
      sjEditPoint: ;
      sjEditSeries: ;
      sjCopySeries: ;
      sjDisplace:   ;
      sjCloneSeries: ;
      sjDeleteSeries: all done by popupmenu or option}
      {sjPosition: already done, or by popupmenu}
      {sjNearestPoint: already done, or by popupmenu}
      sjLinearize,
      sjZero,
      sjAverage,
      sjContractSeries,
      sjContractAllSeries,
      {sjSplineAxis: ;
      sjHighs,
      sjLows,
      sjMovingAverage,
      sjSmoothSeries:   ;
      sjSortSeries: ;
      sjDifferentiate:   ;
      sjIntegrate: all done by popupmenu or option}
      sjIntegral,
      sjLineOfBestFit,
      sjDualLineBestFit1,
      sjDualLineBestFit2,
      sjSelection,
      sjDualSelection1,
      sjDualSelection2:
        StretchTheClickedObjectTo(X, Y);
    end;
  end {if (ssLeft in Shift)}
  else if (ssRight in Shift) then
  begin {aha ! a right click and drag operation !}
    FScreenJob := sjRightDrag;
    StretchTheClickedObjectTo(X, Y);
  end;

  inherited MouseMove(Shift, X, Y);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseUp
  Description: MouseUp event handler
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 05/30/2001 by Mat Ballard
      Purpose: Reacts to the user finishing an action with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseUp(
  Button: TMouseButton;
  Shift: TShiftState;
  X,
  Y: Integer);
var
  Point: TPoint;
begin
  MouseTimer.Enabled := FALSE;

  if (Button = mbLeft) then
  begin
    case FScreenJob of
      {sjNone:}
      sjDrag:
        begin
          OutlineTheClickedObject;
          if (SecondClickedObjectType = soNone) then
          begin
            if (FClickedObjectType = soZAxisEnd) then
            begin
              ZAxis.Vector := Selection.Vector;
              ZeroScreenStuff;
            end
             else
              ProcessClickedObject(FClickedObjectType, pClickedObject);
          end
          else
          begin
            WhichPopUpItems[0].Caption := 'Move the ' + TRectangle(pClickedObject).Name;
            WhichPopUpItems[1].Caption := 'Move the ' + TRectangle(pSecondClickedObject).Name;
            Point.x := X;
            Point.y := Y;
            Point := ClientToScreen(Point);
            mZeroStuff := TRUE;
            WhichPopUpMenu.Popup(Point.x, Point.y);
          end;
        end;
      {sjHide:}
      sjZoomIn:
        begin
          OutlineTheClickedObject;
          SwapEnds;
          ZoomInClick(Self);
        end;
      {sjEditAxis: ;}
      sjMoveNotePointer: ZeroScreenStuff;
      {sjEditFont: ;
      sjEditPoint: ;
      sjEditSeries: ;
      sjCopySeries: ;
      sjDisplace:   ;
      sjCloneSeries: ;
      sjDeleteSeries: ;}
      {sjPosition: already done, or by popupmenu}
      {sjNearestPoint: already done, or by popupmenu}
      sjAverage:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          AverageClick(Self);
        end;
      sjLinearize:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          LinearizeClick(Self);
        end;
      sjZero:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          ZeroClick(Self);
        end;
      sjContractSeries:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          ContractSeriesClick(Self);
        end;
      sjContractAllSeries:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          ContractAllSeriesClick(Self);
        end;
      {sjSplineSeries: ;
      sjHighs,
      sjLows,
      sjMovingAverage,
      sjSmoothSeries:   ;
      sjSortSeries: ;
      sjDifferentiate:   ;
      sjIntegrate:        ;}
      sjIntegral:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          IntegralClick(Self);
        end;
      sjLineOfBestFit:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          LineBestFitClick(Self);
        end;
      sjDualLineBestFit1:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          TwoRegionLineBestFitClick(Self);
        end;
      sjDualLineBestFit2:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          LineBestFitClick(Self);
        end;
      sjSelection:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          Selection.AssignToRect(Sel1);
          DoSelection(Sel1);
        end;
      sjDualSelection1:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          Selection.AssignToRect(Sel1);
          ScreenJob := sjDualLineBestFit2;
          FInstructions := 'Click and Drag over the SECOND region';
          DoStyleChange(Self);
        end;
      sjDualSelection2:
        begin
          Selection.Outline(Canvas);
          SwapEnds;
          Selection.AssignToRect(Sel2);
          DoDualSelection(Sel1, Sel2);
        end;
    end; {end case}
    //DoStyleChange(Self);
  end {end Left Button}
  else if (Button = mbRight) then
  begin {Right Button}
    //OutlineTheClickedObject;
{what does that do to menu visibility ?}
    //DetermineMenuVisibility;
{we no longer let the ancestor run the popup:
 see note above on 'property PopupMenu':}
    {Point.x := X;
    Point.y := Y;
    Point := ClientToScreen(Point);
    mZeroStuff := TRUE;
    if (FScreenJob = sjRightDrag) then
    begin
      FDragPopUpMenu.Popup(Point.x, Point.y);
    end
     else
      FPlotPopUpMenu.Popup(Point.x, Point.y);}
  end; {end Right Button}

{inherited runs the popup if neccessary:}
  inherited MouseUp(Button, Shift, X, Y);
end;
{$ENDIF} {GUI DblClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoSelection
  Description: Fires the Selection event
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoSelection(Sel1: TRect);
begin
  if Assigned(FOnSelection) then
    OnSelection(Self, Sel1);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoDualSelection
  Description: Fires the DualSelection event
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoDualSelection(Sel1, Sel2: TRect);
begin
  if Assigned(FOnDualSelection) then
    OnDualSelection(Self, Sel1, Sel2);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: SwapEnds
  Description: Swaps the selection's Left-Right and Top-Bottom if needed
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: manageing region selections
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SwapEnds;
var
  iX: Integer;
begin
{swap Left and Right:}
  if (Selection.Left > Selection.Right) then
  begin
    iX := Selection.Left;
    Selection.Left := Selection.Right;
    Selection.Right := iX;
  end;

{swap Top and Bottom:}
  if (Selection.Top > Selection.Bottom) then
  begin
    iX := Selection.Top;
    Selection.Top := Selection.Bottom;
    Selection.Bottom := iX;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseTimeOut
  Description: responds to the mouse button being held down
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Frees the timer, identifies the clicked object, outlines it, and prepares to move it
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseTimeOut(
  Sender: TObject);
{If this is fired, then the user has held the mouse still for one second
 on a screen object: this is a cue for a move.}
begin
  MouseTimer.Enabled := FALSE;

  if (FClickedObjectType = soNone) then exit;

  if (FScreenJob = sjTouchNotePointer) then
  begin
  {$IFDEF COMPILER3_UP}
    Screen.Cursor := crScope;
  {$ENDIF}
    FScreenJob := sjMoveNotePointer;
  end
  else
  begin
    FScreenJob := sjDrag;
  {$IFDEF COMPILER3_UP}
    Screen.Cursor := crHandPoint;
  {$ENDIF}
    if (FClickedObjectType > soNone) then
    begin
      if (FClickedObjectType = soSeries) then
        FClickedSeries.Outline(Self.Canvas, FPlotType, FOutlineWidth)
      else
      begin
        TAngleRect(pClickedObject).CopyTo(Selection);
        Selection.FireEvents := FALSE;
        Selection.Outline(Canvas);
      end;
    end;
  end; {sjTouchNote}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetTheClickedObject
  Description: identifies the object(s) that was clicked on
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/15/2001 by Mat Ballard
      Purpose: sets FClickedObjectType and SecondClickedObjectType, and TheSeries if it was a Axis.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.GetTheClickedObject(
  X,
  Y: Integer);
var
  i: Integer;
  MinDistance: Single;
  TheRect: TRect;

  function ObjectClicked(ARect: TAngleRect): Boolean;
  begin
    if (ARect.ClickedOn(X, Y)) then
    begin
      Result := TRUE;
      if (pClickedObject = nil) then
      begin
        pClickedObject := ARect;
        FClickedObjectType := TObjectType(TAngleRect(pClickedObject).Tag);
        ClickedObjectOffset.x := X - TAngleRect(pClickedObject).Left;
        ClickedObjectOffset.y := Y - TAngleRect(pClickedObject).Top;
        if (FClickedObjectType = soZAxis) then
          if (TAxis(pClickedObject).EndClickedOn(MouseStart.x, MouseStart.y, FOutlineWidth)) then
          begin
            FClickedObjectType := soZAxisEnd;
            exit;
          end;
      end
      else if (FClickedObjectType <> soZAxisEnd) then
      begin // once we have found a Z Axis End, that's it !
  {there are two objects under the mouse: usually an axis and a border:}
        pSecondClickedObject := ARect;
        SecondClickedObjectType := TObjectType(TAngleRect(pSecondClickedObject).Tag);
        if (SecondClickedObjectType = soZAxis) then
          if (TAxis(pSecondClickedObject).EndClickedOn(MouseStart.x, MouseStart.y, FOutlineWidth)) then
          begin
{If the second clicked object is an Z Axis end, that takes priority:}
            Inc(SecondClickedObjectType);
            pClickedObject := pSecondClickedObject;
            FClickedObjectType := SecondClickedObjectType;
            pSecondClickedObject := nil;
            SecondClickedObjectType := soNone;
          end;
        exit;
      end;
    end
     else
      Result := FALSE;
  end;


begin
  pClickedObject := nil;
  FClickedObjectType := soNone;
  pSecondClickedObject := nil;
  SecondClickedObjectType := soNone;

{page control while zoomed in ?}
  if (Assigned(mPageButtons[0])) then
  begin
    if ((X > Self.Width - 3*mPageButtons[0].Width) and
        (Y > Self.Height - 2*mPageButtons[0].Height)) then
    begin
{ x2x031  is the layout:}
      if (Y > Self.Height - mPageButtons[0].Height) then
      begin
        if (X < Self.Width - 2*mPageButtons[0].Width) then
        begin
          PageButtonClick(0);
          exit;
        end
        else if (X < Self.Width - mPageButtons[0].Width) then
        begin
          PageButtonClick(3);
          exit;
        end
        else
        begin
          PageButtonClick(1);
          exit;
        end
      end
      else
      begin
        if ((X >= Self.Width - 2*mPageButtons[0].Width) and
            (X < Self.Width - mPageButtons[0].Width)) then
        begin
          PageButtonClick(2);
          exit;
        end
      end;
    end;
  end;

  for i := 0 to FAxes.Count-1 do
  begin
    ObjectClicked(TAngleRect(FAxes[i]));
    ObjectClicked(TAngleRect(FAxes[i].Title));
    ObjectClicked(TAngleRect(FAxes[i].Labels));
  end;

  for i := 0 to FNotes.Count-1 do
  begin {identify the clicked-on object:}
    if (not ObjectClicked(TAngleRect(FNotes[i]))) then
    begin
      if ((Abs(FNotes[i].ArrowLeft - X) < FOutlineWidth) or
               (Abs(FNotes[i].ArrowTop - Y) < FOutlineWidth)) then
      begin {it is a note end:}
        FClickedObjectType := soNote;
        pClickedObject := FNotes[i];
        FScreenJob := sjTouchNotePointer;
      end;
    end;
  end; {for FNotes}

  for i := 0 to FTexts.Count-1 do
    ObjectClicked(TAngleRect(FTexts[i]));

  for i := 1 to ScreenObjectList.Count-1 do
    ObjectClicked(TAngleRect(ScreenObjectList.Items[i]));

  if ((FClickedObjectType = soLegend) or (SecondClickedObjectType = soLegend)) then
  begin
    //if (FClickedSeries = nil) then
    //begin
    i := FLegend.GetHit(X, Y, TheRect);
    FClickedSeries := FSeriesList[i];
    //end;
  end;

{If nothing yet clicked on ...}
  if (FClickedObjectType = soNone) then
  begin
    if ((FPlotType = ptXY) or
        (FPlotType = ptColumn) or
        (FPlotType = ptStack) or
        (FPlotType = ptNormStack) or
        (FPlotType = ptPie)) then
    begin
{was it a Series ?}
      ThePointNumber := FSeriesList.GetNearestPoint(
        FPlotType,
        FColumnGap,
        X, Y, //Selection.Left, Selection.Top,
        TheSeries,
        MinDistance,
        FClickedSeries);
{give it a wide hit range:}
      if (MinDistance < (FOutlineWidth)) then
      begin
        FClickedObjectType := soSeries;
        if (FPlotType = ptXY) then
          FClickedSeries.GenerateXYOutline;
      end;
    end; {FPlotType}
  end; {FClickedObjectType = soNone}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ProcessClickedObject
  Description: Adjusts the geometry of a moved (clicked and dragged) object
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets appropriate property(ies) of the object that has been manipulated on screen
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ProcessClickedObject(ObjectType: TObjectType; pObject: TCollectionItem);
var
  NewMousePoint: TPoint;
  pAxis: TAxis;
begin
  case ObjectType of
    soTitle:
      begin
        FTitle.Position := 100 * (Selection.Left - FBorder.Left) div
          FBorder.Width;
        FTitle.DefaultSide := (Selection.MidY < Height div 2);
      end;

    soLegend:
      begin
        FLegend.MoveTo(Selection.Left, Selection.Top);
      end;

    soXAxis:
      begin
        FAxes[0].AutoZero := FALSE;
        FAxes[0].Intercept := FAxes[1].YofF(Selection.MidY);
      end;

    soYAxis:
      begin
        pAxis := TAxis(pObject);
        pAxis.AutoZero := FALSE;
        pAxis.Intercept := FAxes[0].XofF(Selection.MidX);
      end;

    soZAxis:
      begin
        ZAxis.AutoZero := FALSE;
        ZAxis.Intercept := FAxes[0].XofF(Selection.Origin.x);
        ZAxis.ZInterceptY := FAxes[1].YofF(Selection.Origin.y);
      end;

    soXAxisTitle, soYAxisTitle, soZAxisTitle:
      begin
        pAxis := TAxis(TTitle(pClickedObject).Axis);
        pAxis.Title.Position := Selection.Tag;
        pAxis.Title.DefaultSide := pAxis.IsDefaultSide(Selection.Centre.x, Selection.Centre.y);
      end;

    soXAxisLabel, soYAxisLabel, soZAxisLabel:
      begin
        pAxis := TAxis(TAxisLabel(pClickedObject).Axis);
        pAxis.Labels.DefaultSide := pAxis.IsDefaultSide(Selection.Centre.x, Selection.Centre.y);
      end;

(*
    soLeftBorder: FBorder.Left := Selection.MidX;
    soRightBorder: FBorder.Right := Selection.MidX;
    soTopBorder: FBorder.Top := Selection.MidY;
    soBottomBorder: FBorder.Bottom := Selection.MidY;
*)
    soText: TText(pObject).MoveTo(Selection.Left, Selection.Top);

    soNote:
      begin
        TNote(pObject).MoveTo(Selection.Left, Selection.Top);
{now lets move the note pointer:}
  {$IFDEF COMPILER3_UP}
        Screen.Cursor := crScope;
  {$ENDIF}
        FScreenJob := sjMoveNotePointer;
        NewMousePoint.x := TNote(pObject).ArrowLeft;
        NewMousePoint.Y := TNote(pObject).ArrowTop;
        ClientToScreen(NewMousePoint);
  {$IFDEF BCB}
        SetCursorPos(NewMousePoint.x, NewMousePoint.y);
  {$ELSE} {BCB doesn't like:}
        Mouse.CursorPos := ClientToScreen(NewMousePoint);
  {$ENDIF}
        exit;
      end;

    soSeries: { already moved by direct change of DeltaX and DeltaY properties}
      begin

      end;
  end; {case TheObjectType}

  ZeroScreenStuff;
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OutlineTheClickedObject
  Description: Outlines The Clicked Object
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are manipulating with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OutlineTheClickedObject;
begin
  if (FClickedObjectType = soSeries) then
    FClickedSeries.Outline(Self.Canvas, FPlotType, FOutlineWidth)
  else if (FClickedObjectType > soNone) then
  begin
    //Selection.Left := TRectangle(pClickedObject).Left;
    Selection.Outline(Self.Canvas);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MoveTheClickedObjectTo
  Description: This moves the clicked object outline TO (X, Y)
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are moving with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MoveTheClickedObjectTo(X, Y: Integer);
begin
  if ((Selection.Left = X) and (Selection.Top = Y)) then exit;

{erase the old outline:}
  OutlineTheClickedObject;

{re-initialize the Selection:}
  Selection.MoveTo(X, Y);

{create the new outline:}
  OutlineTheClickedObject;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.StretchTheClickedObjectTo
  Description: Stretch The Clicked Object To
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are selecting with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.StretchTheClickedObjectTo(X, Y: Integer);
{This moves the far (right, bottom) point of the Stretched object outline TO (X, Y).}
begin
  if ((Selection.Right = X) and (Selection.Bottom = Y)) then
    exit;

{erase the old outline:}
  //OutlineTheSelection;
  Selection.Outline(Canvas);

{re-initialize the Selection:}
  Selection.Right := X;
  Selection.Bottom := Y;

{create the new outline:}
  //OutlineTheSelection;
  Selection.Outline(Canvas);
end;

{------------------------------------------------------------------------------
    Procedure: MoveTheClickedObjectClick
  Description: deals with the end of a screen operation at MouseUp
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Processes the first Clicked Object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MoveTheClickedObjectClick(
  Sender: TObject);
begin
  ProcessClickedObject(FClickedObjectType, pClickedObject);
end;

{------------------------------------------------------------------------------
    Procedure: MoveSecondClickedObjectClick
  Description: deals with the end of a screen operation at MouseUp
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Processes the SECOND Clicked Object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MoveSecondClickedObjectClick(
  Sender: TObject);
begin
  ProcessClickedObject(SecondClickedObjectType, pSecondClickedObject);
end;
{$ENDIF} {GUI SwapEnds}

{General public methods -----------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AddData
  Description: Add data to the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: adds data to all Series simultaneously, avoiding re-drawing the data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AddData(
  NoPoints: Integer;
  XYArray: pXYArray);
var
  i: Integer;
begin
  {if (Canvas.LockCount > 0) then
    ShowMessageFmt('LockCount = %d !', [Canvas.LockCount]);}
  if ((NoPoints > 0) and (NoPoints <= FSeriesList.Count)) then
  begin
    if (FDisplayMode <> dmHistory) then
    begin
      for i := 0 to NoPoints-1 do
      begin
        TSeries(FSeriesList[i]).AddDrawPoint(XYArray^[i].X, XYArray^[i].Y, Canvas);
      end;
    end
    else
    begin
      case FPlotType of
        ptXY:
          begin
{erase the old curve:}
            FSeriesList.DrawHistory(Canvas, FDisplayHistory);
            for i := 0 to NoPoints-1 do
            begin
              TSeries(FSeriesList[i]).AddPoint(XYArray^[i].X, XYArray^[i].Y, FALSE, FALSE);
            end;
{draw the new one:}
            FSeriesList.DrawHistory(Canvas, FDisplayHistory);
          end;
        ptColumn:
          begin
          end;
      end;
    end;
  end
  else
    EMathError.CreateFmt('AddData: you must add between 1 and %d points !', [FSeriesList.Count]);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.AddSeries
  Description: wrapper for TSeriesList.Add
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new, empty data Series to the graph
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.AddSeries(
  XSeriesIndex: Integer): TSeries;
begin
  AddSeries := FSeriesList.AddDependent(XSeriesIndex);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.AddExternal
  Description: wrapper for TSeriesList.AddExternal
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new data Series that is maintained elsewhere to the graph
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.AddExternalData(
  XPointer,
  YPointer: pSingleArray;
  NumberOfPoints: Integer): TSeries;
begin
  AddExternalData := FSeriesList.AddExternal(XPointer, YPointer, NumberOfPoints);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.AddInternal
  Description: wrapper for TSeriesList.AddInternal
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new data Series from elsewhere to the graph, and saves it internally
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.AddInternalData(
  XPointer,
  YPointer: pSingleArray;
  NumberOfPoints: Integer): TSeries;
begin
  AddInternalData := FSeriesList.AddInternal(XPointer, YPointer, NumberOfPoints);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CloneSeries
  Description: wrapper for TSeriesList.CloneSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Clones the specified Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.CloneSeries(
  TheSeries: Integer): TSeries;
begin
  Result := FSeriesList.CloneSeries(TheSeries);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteSeries
  Description: wrapper for TSeriesList.DeleteSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Deletes the specified Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteSeries(
  Index: Integer);
begin
  FSeriesList.DeleteSeries(Index, not(csDesigning in ComponentState));
end;

{Responding to user (right) click menus -------------------------------------}
{$IFDEF GUI}

(*------------------------------------------------------------------------------
    Procedure: TCustomPlot.ProcessKeyDown
  Description: This processes a keystroke sent to TPlot by the main application
       Author: Mat Ballard
 Date created: 12/05/2002
Date modified: 12/05/2002 by Mat Ballard
      Purpose: GUI management
 Known Issues:
     Comments: Requires developer to place code in their form to work
               The required code is to add an "OnKeyDown" event handler:
               double-click in the "OnKeyDown" Event in the Form's Object Inspector,
               and call this method using the Key and Shift values; eg:

procedure TForm1.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  MyPlot.ProcessKeyDown(Key, Shift);
end;
 -----------------------------------------------------------------------------*)
procedure TCustomPlot.ProcessKeyDown(var Key: Word;
  Shift: TShiftState);
var
  Handled: Boolean;
{$IFDEF COMPILER4_UP}
  Mulitplier: Integer;
  Pos, PlotPos: TPoint;
{$ENDIF}
begin
  Handled := FALSE;
{$IFDEF COMPILER4_UP}
  if (ssShift in Shift) then
    Mulitplier := 10
  else if (ssCtrl in Shift) then
    Mulitplier := 100
   else
    Mulitplier := 1;
  Pos := Mouse.CursorPos;
  PlotPos := Self.ScreenToClient(Pos);
  if ((0 < PlotPos.x) and (PlotPos.x < Self.Width) and
      (0 < PlotPos.y) and (PlotPos.y < Self.Height)) then
  begin
{$ENDIF}
    case Key of
{$IFDEF COMPILER4_UP}
      VK_RIGHT:
        begin
          Pos.x := Pos.x + Mulitplier;
          Handled := TRUE;
        end;
      VK_LEFT:
        begin
          Pos.x := Pos.x - Mulitplier;
          Handled := TRUE;
        end;
      VK_UP:
        begin
          Pos.y := Pos.y - Mulitplier;
          Handled := TRUE;
        end;
      VK_DOWN:
        begin
          Pos.y := Pos.y + Mulitplier;
          Handled := TRUE;
        end;
{$ENDIF}
      Ord('c'):
        begin
          if (ssCtrl in Shift) then Self.CopyClick(Self);
          Handled := TRUE;
        end;
      Ord('v'):
        begin
          if (ssCtrl in Shift) then Self.PasteClick(Self);
          Handled := TRUE;
        end;
    end;
    if (Handled) then
      Key := 0;
{$IFDEF COMPILER4_UP}
    Mouse.CursorPos := Pos;
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyClick
  Description: The public copying method
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: copies the graph to the clipboard, in all formats simultaneously
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyClick(
  Sender: TObject);
begin
  {$IFDEF MSWINDOWS}
  ClipBoardFormatForHTML := RegisterClipboardFormat('cfHTML');
  ClipBoard.Open;
  try
{copy all three formats to the clipboard at once:}
    CopyText;
    CopyHTML(ClipBoardFormatForHTML);
    CopyBitmap;
    CopyDrawing(TRUE);
  finally
    ClipBoard.Close;
  end;
  {$ENDIF}
  {$IFDEF LINUX}
  CopyText;
  CopyHTML(ClipBoardFormatForHTML);
  CopyBitmap;
  {$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PrintClick
  Description: The public printing method
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Print the graph
 Known Issues: We kludge this one a bit by creating a metafile, then playing it
               on the printer canvas. It would be nicer to draw directly on the
               printer canvas.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PrintClick(
  Sender: TObject);
var
  i,
  Copies: Integer;
  //Copies: Integer; currently ignored by the Qt dialog
  PrintBorder: TRect;
  {$IFDEF MSWINDOWS}
  PrintDialog: TPrintDialog;
  HorzSizeMM, VertSizeMM: Integer;
  {$ENDIF}
  {$IFDEF LINUX}
  ThePrinter: TPrinter;
  {$ENDIF}
begin
  {$IFDEF LINUX}
  ThePrinter := Printer;
  ThePrinter.Orientation := FPrintOrientation;
  if (ThePrinter.ExecuteSetup) then
  begin
    ThePrinter.Title := ExtractFileName(Application.ExeName) + ' - ' + FTitle.Caption;
    {with ThePrinter do
    begin
      ShowMessageFmt(ThePrinter.Printers.Text + #10 +
        'Copies = %d (%d)' + #10 +
        'Orientation = %d' + #10 +
        'PageWidth = %d' + #10 +
        'PageHeight = %d' + #10 +
        'Margins = (%d, %d)' + #10 +
        'XDPI = %d' + #10 +
        'YDPI = %d' + #10 +
        'OutputDevice: ' + OutputDevice + #10 +
        'Title: ' + Title,
        [Copies, ThePrinter.PrintAdapter.Copies, Ord(Orientation),
         PageWidth, PageHeight, Margins.cx, Margins.cy, XDPI, YDPI]);
    end;}
    if (ThePrinter.Copies < 1) then
      ThePrinter.Copies := 1;

{i like 1" borders:}
    PrintBorder.Left := 1 * ThePrinter.XDPI;
    PrintBorder.Top := 1 * ThePrinter.YDPI;
    PrintBorder.Right := ThePrinter.PageWidth - 1 * ThePrinter.XDPI;
    PrintBorder.Bottom := ThePrinter.PageHeight - 1 * ThePrinter.YDPI;

    ThePrinter.BeginDoc;
    for i := 1 to ThePrinter.Copies do
    begin
      if (i > 1) then
        ThePrinter.NewPage;
      ThePrinter.Canvas.Start(TRUE);
      ThePrinter.Canvas.StretchDraw(PrintBorder, Self.Drawing);
      //ThePrinter.PrintAdapter.Canvas.StretchDraw(PrintBorder, Self.Drawing);

      ThePrinter.Canvas.Pen.Color := clBlack;
      ThePrinter.Canvas.Pen.Width := 9;
      ThePrinter.Canvas.Pen.Style := psSolid;
      ThePrinter.Canvas.MoveTo(PrintBorder.Left, PrintBorder.Top);
      ThePrinter.Canvas.LineTo(PrintBorder.Right, PrintBorder.Bottom);
      ThePrinter.Canvas.MoveTo(PrintBorder.Right, PrintBorder.Top);
      ThePrinter.Canvas.LineTo(PrintBorder.Left, PrintBorder.Bottom);

      ThePrinter.Canvas.Stop;
    end;
    ThePrinter.EndDoc;
  end;
  {$ENDIF}

  {$IFDEF MSWINDOWS}
  Printer.Orientation := FPrintOrientation;

  PrintDialog := TPrintDialog.Create(Self);
  PrintDialog.Options := [poPrintToFile, poWarning];
  if (PrintDialog.Execute) then
  begin
    if (PrintDialog.Copies > 1) then
      Copies := PrintDialog.Copies
     else
      Copies := 1;
    Printer.Title := Application.ExeName + ' - ' + FTitle.Caption;

    HorzSizeMM := GetDeviceCaps(Printer.Handle, HORZSIZE);
    VertSizeMM := GetDeviceCaps(Printer.Handle, VERTSIZE);

{Set the margins to 25 mm:}
    PrintBorder.Left := 25 * (Printer.PageWidth div HorzSizeMM);
    PrintBorder.Top := 25 * (Printer.PageHeight div VertSizeMM);
    PrintBorder.Right := Printer.PageWidth - PrintBorder.Left;
    PrintBorder.Bottom := Printer.PageHeight - PrintBorder.Top;

{note: the D4 TPrinter has a Copies property, but not all printers support it,
 so do printing the slow way:}
    Printer.BeginDoc;
    for i := 1 to Copies do
    begin
      if (i > 1) then
        Printer.NewPage;
      Printer.Canvas.StretchDraw(PrintBorder, Self.Drawing);
    end;

    Printer.EndDoc;
    //DoStyleChange(Self);
  end; {PrintDialog}
  PrintDialog.Free;
  {$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ShowAllClick
  Description: Shows/reveals all screen objects
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: makes everything visible.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ShowAllClick(Sender: TObject);
var
  i: Integer;
begin
  if (not FLegend.Visible) then
    LegendClick(Self);
  for i := 0 to FSeriesList.Count-1 do
    FSeriesList[i].Visible := TRUE;

  for i := 1 to ScreenObjectList.Count-1 do
    TRectangle(ScreenObjectList.Items[i]).Visible := TRUE;

  DoStyleChange(Self);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.HideClick
  Description: Hides part of the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: makes the selected object invisible
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.HideClick(Sender: TObject);
begin
  case FClickedObjectType of
    soTitle: FTitle.Visible := FALSE;
    soLegend: FLegend.Visible := FALSE;
    soNote: TNote(pClickedObject).Visible := FALSE;
    soXAxis: FAxes[0].Visible := FALSE;
    soYAxis: TAxis(pClickedObject).Visible := FALSE;
    soXAxisTitle: FAxes[0].Title.Visible := FALSE;
    soYAxisTitle: TTitle(pClickedObject).Visible := FALSE;
    soXAxisLabel: FAxes[0].Labels.Visible := FALSE;
    soYAxisLabel: TAxisLabel(pClickedObject).Visible := FALSE;
    soSeries: FClickedSeries.Visible := FALSE;
  else
    FInstructions := 'Click on the object you want to hide';
    ScreenJob := sjHide;
  end;
  DoStyleChange(Self);
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ShowHideAllSeriesClick
  Description: Shows or hides all of the graph series
       Author: Mat Ballard
 Date created: 10/17/2003
Date modified: 10/17/2003 by Mat Ballard
      Purpose: GUI management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ShowHideAllSeriesClick(Sender: TObject; IsVisible: Boolean);
var
  i: Integer;
begin
  for i := 0 to Self.FSeriesList.Count-1 do
    Self.FSeriesList[i].Visible := IsVisible;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PositionClick
  Description: Where the hell are we ?
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Displays (and copies) the current mouse click position, in USER units
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PositionClick(Sender: TObject);
var
  Msg: String;
begin
  if (MouseStart.x >= 0) then
  begin
    Msg := Format('The screen co-ordinates are (%d, %d),',
      [MouseStart.x, MouseStart.y]) + CRLF +
        'and the Position is (' +
          FAxes[0].LabelToStrF(FAxes[0].XofF(MouseStart.x));
    if (Length(FAxes[0].Title.Units) > 0) then
      Msg := Msg + ' ' + FAxes[0].Title.Units;
    Msg := Msg + ', ' +
      FAxes[1].LabelToStrF(FAxes[1].YofF(MouseStart.y));
    if (Length(FAxes[1].Title.Units) > 0) then
      Msg := Msg + ' ' + FAxes[1].Title.Units;
    Msg := Msg + ').';
    ShowMessage(Msg);
    ClipBoard.AsText := Msg;
    ZeroScreenStuff;
  end
  else
  begin
    ScreenJob := sjPosition;
    Screen.Cursor := crScope;
    FInstructions := 'Click on the position you want the details of';
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NearestPointClick
  Description: Where the hell is it ?
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: finds the nearest point of the nearest Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NearestPointClick(
  Sender: TObject);
var
  NearestiX,
  NearestiY: Integer;
  MinDistance: Single;
  Msg: String;
begin
  if (MouseStart.x > 0) then
  begin
    ThePointNumber := FSeriesList.GetNearestPoint(
      FPlotType,
      FColumnGap,
      Selection.Left, Selection.Top,
      TheSeries,
      MinDistance,
      FClickedSeries);
    Msg := 'The nearest point is #' + IntToStr(ThePointNumber) + ' in ' +
      FClickedSeries.Name + CRLF + ' At (' +
        FAxes[0].LabelToStrF(FClickedSeries.XData^[ThePointNumber]);
    if (Length(FAxes[0].Title.Units) > 0) then
      Msg := Msg + ' ' + FAxes[0].Title.Units;
    Msg := Msg + ', ' +
      FAxes[1].LabelToStrF(FClickedSeries.YData^[ThePointNumber]);
    if (Length(FAxes[1].Title.Units) > 0) then
      Msg := Msg + ' ' + FAxes[1].Title.Units;
    Msg := Msg + ').';

    Canvas.Pen.Color := clRed;
    Canvas.Pen.Mode := pmNotXOR;
    Canvas.Pen.Style := psSolid;
    Canvas.Pen.Width := 2;
    NearestiX := FClickedSeries.XAxis.FofX(FClickedSeries.XData^[ThePointNumber]);
    NearestiY := FClickedSeries.YAxis.FofY(FClickedSeries.YData^[ThePointNumber]);
    Canvas.Ellipse(NearestiX-10, NearestiY-10, NearestiX+10, NearestiY+10);

    ShowMessage(Msg);
    ClipBoard.AsText := Msg;

      {Pen.Color := clBlack;
      Pen.Mode := pmNotXOR;
      Pen.Style := psSolid;}
    Canvas.Ellipse(NearestiX-10, NearestiY-10, NearestiX+10, NearestiY+10);

    ZeroScreenStuff;
  end
  else
  begin
    ScreenJob := sjNearestPoint;
    Screen.Cursor := crX;
    FInstructions := 'Click near the point you want';
    DoStyleChange(Self);
  end;
end;
{$ENDIF} {HideClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteSeriesClick
  Description: wrapper for TSeriesList.DeleteSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Deletes the selected Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteSeriesClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FSeriesList.DeleteSeries(TheSeries, not(csDesigning in ComponentState));
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopySeriesClick
  Description: wrapper for TSeriesList.CopySeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Copies the selected Series (as text)
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopySeriesClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
    FClickedSeries.CopyToClipBoard;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NewSeriesClick
  Description: wrapper for TSeriesList.AddSeries
       Author: Mat Ballard
 Date created: 04/16/2001
Date modified: 04/16/2001 by Mat Ballard
      Purpose: Adds a new Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NewSeriesClick(
  Sender: TObject);
var
  Index: Integer;
  pXSeries, pNewSeries: TSeries;
  OptionsForm: TOptions3Form;
begin
  OptionsForm := TOptions3Form.Create(nil);
  OptionsForm.Caption := 'New Series';
  OptionsForm.QuestionLabel.Caption := 'Does this new series use the X Data of an existing series ?';
  OptionsForm.SetListCaption('Pick one');
  OptionsForm.Add('No - it is independent');
  for Index := 0 to FSeriesList.Count-1 do
    OptionsForm.Add(TSeries(FSeriesList[Index]).Name);

  if (OptionsForm.ShowModal = mrOK) then
  begin
    Index := OptionsForm.ItemIndex;
    if (Index = 0) then
    begin
      pNewSeries := FSeriesList.Add;
    end
    else
    begin
      pXSeries := FSeriesList[Index];
      if (Assigned(pXSeries.XDataSeries)) then
        pXSeries := pXSeries.XDataSeries;
      pNewSeries := FSeriesList.AddDependent(pXSeries.Index);
    end;
    pNewSeries.EditData(FHelpFile);
  end;
  OptionsForm.Free;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CloneSeriesClick
  Description: wrapper for TSeriesList.CloneSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Clones the selected Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CloneSeriesClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FSeriesList.CloneSeries(TheSeries);
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ModeClick
  Description: Changes how the graph appears and reacts to new data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DisplayMode property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ModeClick(
  Sender: TObject);
var
  Index: Integer;
  DisplayHistoryStr: String;
  OptionsForm: TOptions3Form;
begin
  OptionsForm := TOptions3Form.Create(nil);
  OptionsForm.Caption := 'Display Mode ?';
  OptionsForm.QuestionLabel.Caption := 'How do you want to display the data when a new point is added?';
  OptionsForm.SetListCaption('Pick one');
  OptionsForm.Add('Normal - expand the Axes if necessary');
  OptionsForm.Add('None - do nothing');
  OptionsForm.Add('Run - if neccessary, expand the Y-Axis but double the X-Axis');
  OptionsForm.Add('History - only show the most recent data');

  if (OptionsForm.ShowModal = mrOK) then
  begin
    Index := OptionsForm.ItemIndex;
    if (Index = Ord(dmHistory)) then
    begin
      DisplayHistoryStr := FloatToStr(FDisplayHistory);
      if (InputQuery('History Range', '', DisplayHistoryStr)) then
      FDisplayHistory := StrToFloat(DisplayHistoryStr);
    end;
    SetDisplayMode(TDisplayMode(Index));
  end;
  OptionsForm.Free;
end;
{$ENDIF} {GUI CopyClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CanPaste
  Description: Can we paste data from the Clipboard into TPlot ?
       Author: Mat Ballard
 Date created: 11/28/1999
Date modified: 11/28/2000 by Mat Ballard
      Purpose: check in with the clipboard, looking for text data
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.CanPaste: Boolean;
{$IFDEF GUI}
  {$IFDEF LINUX}
var
  FormatList: TStringList;
  i: Integer;
  {$ENDIF}
begin
  {$IFDEF MSWINDOWS}
  CanPaste := (ClipBoard.HasFormat(CF_TEXT));
  {$ENDIF}
  {$IFDEF LINUX}
  CanPaste := FALSE;
  FormatList := TStringList.Create;
  Clipboard.SupportedFormats(FormatList);
  for i := 0 to FormatList.Count-1 do
  begin
    if (Pos('text/plain', FormatList.Strings[i]) > 0) then
    begin
      CanPaste := TRUE;
      exit;
    end;
  end;
  FormatList.Free;
  {$ENDIF}
{$ELSE}
begin
  Result := FALSE;
{$ENDIF}
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PasteClick
  Description: Pastes data from the Clipboard
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Collects the data from the Clipboard and runs the ParseData method on it.
 Known Issues: limited to 32 K under D1. Can be fixed with messy memory management.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PasteClick(
  Sender: TObject);
var
  TheStrings: TStringList;
  TheData: TMemoryStream;
begin
  TheStrings := TStringList.Create;
  TheStrings.Text := Clipboard.AsText;
  TheData := TMemoryStream.Create;
  TheStrings.SaveToStream(TheData);
  mInitialSeriesCount := FSeriesList.Count;
  Self.LoadFromStream(TheData, #9);
  TheData.Free;
  TheStrings.Free;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LineBestFitClick
  Description: Initiates a Line of Best Fit.
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 05/30/2001 by Mat Ballard
      Purpose: Sets the Instructions and the ScreenJob
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LineBestFitClick(
  Sender: TObject);
var
  pTextBox: TText;
  Str: String;

  procedure AddATextBox;
  begin
    if (Length(Str) > 0) then
    begin
      pTextBox := FTexts.AddEx(Str);
      pTextBox.Font.Color := FClickedSeries.Pen.Color;
      pTextBox.MoveTo(Selection.Left, Selection.Top);
    end;
  end;

begin
  if (GetSeriesFromUser <> nil) then
  begin
    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjLineOfBestFit)) then
    begin
{$IFDEF POLYNOMIAL_FIT}
      Str := FClickedSeries.PolyBestFit(Selection.Left, Selection.Right, 0, 0);
{$ELSE}
      Str := FClickedSeries.LineBestFit(Selection.Left, Selection.Right, 0, 0);
{$ENDIF}
      AddATextBox;
      ZeroScreenStuff;
    end
    else if (FScreenJob = sjDualLineBestFit2) then
    begin
{$IFDEF POLYNOMIAL_FIT}
      Str := FClickedSeries.PolyBestFit(Sel1.Left, Sel1.Right, Selection.Left, Selection.Right);
{$ELSE}
      Str := FClickedSeries.LineBestFit(Sel1.Left, Sel1.Right, Selection.Left, Selection.Right);
{$ENDIF}
      AddATextBox;
      ZeroScreenStuff;
    end
    else
    begin
      FInstructions := 'Click and Drag over the region to fit.';
      ScreenJob := sjLineOfBestFit;
      DoStyleChange(Self);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.TwoRegionLineBestFitClick
  Description: Initiates a Two Region Line of Best Fit.
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 05/30/2001 by Mat Ballard
      Purpose: Sets the FInstructions and the ScreenJob
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.TwoRegionLineBestFitClick(Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjDualLineBestFit1)) then
    begin
      Selection.AssignToRect(Sel1);
      ScreenJob := sjDualLineBestFit2;
      FInstructions := 'Click and Drag over the SECOND region  to fit';
      DoStyleChange(Self);
    end
    else
    begin
      FInstructions := 'Click and Drag over the FIRST region to fit.';
      ScreenJob := sjDualLineBestFit1;
      DoStyleChange(Self);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ForwardFourierClick
  Description: Fast Fourier [inverse] Transforms all series.
       Author: Mat Ballard
 Date created: 08/27/2001
Date modified: 08/27/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ForwardFourierClick(Sender: TObject);
begin
  FSeriesList.FourierAllSeries(TRUE);
  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.InverseFourierClick
  Description: Fast Fourier [inverse] Transforms all series.
       Author: Mat Ballard
 Date created: 08/27/2001
Date modified: 08/27/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.InverseFourierClick(Sender: TObject);
begin
  FSeriesList.FourierAllSeries(FALSE);
  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.FunctionClick
  Description: Creates a new series which is a function of existing series
       Author: Mat Ballard
 Date created: 04/03/2001
Date modified: 04/03/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.FunctionClick(Sender: TObject);
begin
  FSeriesList.CreateFunctionSeries;
  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SmoothSeriesClick
  Description: Smoothes the selected data Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Obtains the Smoothing Order then runs the selected Series' Smooth method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SmoothSeriesClick(
  Sender: TObject);
var
  SmoothOrder: Integer;
  SmoothStr: String;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    SmoothStr := '10';
    if (InputQuery('Smoothing ' + FClickedSeries.Name,
                   'Enter the Smoothing Order (2..20)',
                   SmoothStr)) then
    begin
      try
        SmoothOrder := StrToInt(SmoothStr);
        FClickedSeries.Smooth(SmoothOrder);
        DoStyleChange(Self);
      except
        ShowMessage('Smoothing Failed !');
      end;
    end;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SortClick
  Description: sorts the selected data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management and manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SortClick(Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FClickedSeries.Sort;
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

procedure TCustomPlot.SplineClick(Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    Self.Spline(TheSeries);
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.Spline
  Description: wrapper for TSeriesList.Spline
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new, empty data Series to the graph
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.Spline(ASeries: Integer): TSeries;
var
  Density: Word;
  TheString: String;
begin
  Result := nil;
{if it isn't already ...}
  FClickedSeries := TSeries(FSeriesList[ASeries]);

  TheString := '1';
  if (InputQuery('Cubic Spline', 'Please enter the number of divisions', TheString)) then
  begin
    Density := StrToInt(TheString);

    Result := FSeriesList.Add;
    Result.AllocateNoPts(FClickedSeries.NoPts * (Density + 1));

    FClickedSeries.DoSpline(Density, Result);

    Result.Name := 'Cubic Spline of ' + FClickedSeries.Name;
    Result.Pen.Style := psDot;
    Result.Visible := TRUE;
{should we call FClickedSeries.ClearSpline ?}
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CompressSeriesClick
  Description: Reduces the number of data points of the selected Series by averaging the data
       Author: Mat Ballard
 Date created: 10/15/2000
Date modified: 10/15/2000 by Mat Ballard
      Purpose: data management and manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CompressSeriesClick(
  Sender: TObject);
var
  CompressRatio: Integer;
  CompressStr: String;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if (FClickedSeries.XDataRefCount > 0) then raise
      EComponentError.CreateFmt(
        'Cannot Compress %s !' + CRLF + '%d other series depend on it !',
        [FClickedSeries.Name, FClickedSeries.XDataRefCount]);

    if (Assigned(FClickedSeries.XDataSeries)) then raise
      EComponentError.CreateFmt(
        'Cannot Compress %s !' + CRLF + 'It depends on the X Data in %s !',
        [FClickedSeries.Name, FClickedSeries.XDataSeries.Name]);

    CompressStr := '10';
    if (InputQuery('Compressing ' + FClickedSeries.Name,
                   'Enter the Compression Ratio (2..20)',
                   CompressStr)) then
    begin
      try
        CompressRatio := StrToInt(CompressStr);
        FClickedSeries.Compress(CompressRatio);
      except
        ShowMessage('Compression Failed !');
      end;
    end;
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CompressAllSeriesClick
  Description: Reduces the number of data points in ALL Series by averaging the data
       Author: Mat Ballard
 Date created: 10/15/2000
Date modified: 10/15/2000 by Mat Ballard
      Purpose: data management and manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CompressAllSeriesClick(
  Sender: TObject);
var
  CompressRatio: Integer;
  CompressStr: String;
  i: Integer;
begin
  CompressStr := '10';
  if (InputQuery('Compressing ALL Series',
                 'Enter the Compression Ratio (2..20)',
                 CompressStr)) then
  begin
    try
      CompressRatio := StrToInt(CompressStr);
      for i := 0 to FSeriesList.Count-1 do
        TSeries(FSeriesList[i]).Compress(CompressRatio);
    except
      ShowMessage('Compression Failed !');
    end;
  end;
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ContractSeriesClick
  Description: Reduces the number of data points of the selected Series by throwing away the ends
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 05/30/2001 by Mat Ballard
      Purpose: data management and manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ContractSeriesClick(
  Sender: TObject);
var
  TheLeft,
  TheRight: Integer;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if (FClickedSeries.XDataRefCount > 0) then raise
      EComponentError.CreateFmt(
        'Cannot Contract %s !' + CRLF + '%d other series depend on it !',
        [FClickedSeries.Name, FClickedSeries.XDataRefCount]);

    if (Assigned(FClickedSeries.XDataSeries)) then raise
      EComponentError.CreateFmt(
        'Cannot Contract %s !' + CRLF + 'It depends on the X Data in %s !',
        [FClickedSeries.Name, FClickedSeries.XDataSeries.Name]);

    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjContractSeries)) then
    begin
      TheLeft := FClickedSeries.GetNearestPointToFX(Selection.Left);
      TheRight := FClickedSeries.GetNearestPointToFX(Selection.Right);
      FClickedSeries.Contract(TheLeft, TheRight);
      ZeroScreenStuff;
    end
    else
    begin
      FScreenJob := sjContractSeries;
      FInstructions := 'Click and Drag over the (X-) data to keep - ' + #13#10 +
        '  all data outside this region is deleted.';
      DoStyleChange(Self);
    end;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ContractAllSeriesClick
  Description: Reduces the number of data points of the selected Series by throwing away the ends
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 05/30/2001 by Mat Ballard
      Purpose: data management and manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ContractAllSeriesClick(
  Sender: TObject);
var
  TheLeft,
  TheRight: Integer;
  i: Integer;
begin
  if ((FScreenJob = sjRightDrag) or
     (FScreenJob = sjContractAllSeries)) then
  begin
    TheLeft := TSeries(FSeriesList[0]).GetNearestPointToFX(Selection.Left);
    TheRight := TSeries(FSeriesList[0]).GetNearestPointToFX(Selection.Right);
    for i := 0 to FSeriesList.Count-1 do
      TSeries(FSeriesList[i]).Contract(TheLeft, TheRight);
    ZeroScreenStuff;
  end
  else
  begin
    FScreenJob := sjContractAllSeries;
    FInstructions := 'Click and Drag over the (X-) data to keep - ' + #13#10 +
      '  all data outside this region is deleted.';
    DoStyleChange(Self);
  end;
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LegendClick
  Description: Sets the Legend Direction
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LegendClick(Sender: TObject);
var
  Index: Integer;
begin
{Vertical -> Horizontal -> Invisible:}
  if (FLegend.Visible) then
  begin
    if (FLegend.Direction = drHorizontal) then
    begin
      FLegend.Direction := drVertical;
      FLegend.Visible := FALSE;
    end
     else
      FLegend.Direction := drHorizontal;
  end
  else
  begin
    FLegend.Visible := TRUE;
  end;
{$IFDEF GUI}
  if (Assigned(Self.FPlotActionList)) then // mnuView, mnuLegend
  begin
    Index := TPlotActionList(FPlotActionList).GetIndexFromMenuPosition(mnuView, Ord(mnuLegend));
    if (FLegend.Visible) then
    begin
    if (FLegend.Direction = drHorizontal) then
      TAction(FPlotActionList.Actions[Index]).Caption := 'Legend ... Hide'
     else
      TAction(FPlotActionList.Actions[Index]).Caption := 'Legend ... Horizontal';
    end
     else
      TAction(FPlotActionList.Actions[Index]).Caption := 'Legend ... Visible';
  end;
{$ENDIF}
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NoteBordersClick
  Description: Sets the Visibility of Note Borders
       Author: Mat Ballard
 Date created: 08/15/2001
Date modified: 08/15/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NoteBordersClick(
  Sender: TObject);
var
{$IFDEF GUI}
  Show: Boolean;
{$ENDIF}
  i: Integer;
begin
{$IFDEF GUI}
  Show := FALSE;
{$ENDIF}
  for i := 0 to FNotes.Count-1 do
  begin
    if (FNotes.Items[i].Border) then
    begin
      FNotes.Items[i].Border := FALSE;
{$IFDEF GUI}
      Show := TRUE;
{$ENDIF}
    end
    else
    begin
      FNotes.Items[i].Border := TRUE;
    end;
  end;

{$IFDEF GUI}
  if Assigned(FPlotActionList) then
  begin
    i := TPlotActionList(FPlotActionList).GetIndexFromMenuPosition(mnuView, Ord(mnuNoteBorders));
    if (Show) then // mnuView, mnuNoteBorders
      TAction(FPlotActionList.Actions[i]).Caption := 'Show Note Borders'
    else
      TAction(FPlotActionList.Actions[i]).Caption := 'Hide Note Borders';
  end;
{$ENDIF}

  ZeroScreenStuff;
{$IFDEF GUI}
  Invalidate;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditAxisClick
  Description: Runs the Axis Editor of the selected Axis
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 06/28/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditAxisClick(
  Sender: TObject);
var
  i: Integer;
  AxisEditor: TAxisEditorForm;
  AXP: TAxisProperty;
  pAxis: TAxis;
begin
  if (GetAxisFromUser(0) <> nil) then
  begin
    AxisEditor := TAxisEditorForm.Create(nil);
    AxisEditor.ThePlot := TObject(Self);

    if (FDisplayMode = dmHistory) then
      AxisEditor.HistoryMode := TRUE;

{Iterate over all axes:}
    for i := 0 to FAxes.Count-1 do
    begin
      pAxis := FAxes[i];
      AXP.TitlePosition := pAxis.Title.Position;
      AXP.LabelFormat := pAxis.Labels.NumberFormat;
      AXP.LabelDigits := pAxis.Labels.Digits;
      AXP.LabelPrecision := pAxis.Labels.Precision;
      AXP.DefaultSide := pAxis.Labels.DefaultSide;
      AXP.TitleDefaultSide := pAxis.Title.DefaultSide;
      AXP.PenColor := pAxis.Pen.Color;
      AXP.PenWidthIndex := pAxis.Pen.Width;
      AXP.PenStyleIndex := Ord(pAxis.Pen.Style);
      AXP.TickAuto := pAxis.AutoTick;
      AXP.TickSize := pAxis.TickSize;
      //AXP.TickDirection := pAxis.TickDirection;
      AXP.TickStepSize := pAxis.StepSize;
      AXP.TickStepStart := pAxis.StepStart;
      AXP.TickMinors := pAxis.TickMinor;
      AXP.ScaleMin := pAxis.Min;
      AXP.ScaleMax := pAxis.Max;
      AXP.ScaleIntercept := pAxis.Intercept;
      AXP.ScaleAuto := pAxis.AutoScale;
      AXP.ScaleLog := pAxis.LogScale;
      AXP.ArrowSize := pAxis.ArrowSize;
      //AXP.ArrowDirection := pAxis.Alignment;
      AXP.Visible := pAxis.Visible;
      AXP.LimitLower := pAxis.LimitLower;
      AXP.LimitUpper := pAxis.LimitUpper;
      AXP.LimitsVisible := pAxis.LimitsVisible;
      if (FAxes[i].AxisType = atZ) then
      begin
        AXP.ZAngle := pAxis.Angle;
        AXP.ZLength := pAxis.Width;
        AXP.ZInterceptY := pAxis.ZInterceptY;
      end
      else
      begin
        AXP.ZAngle := 0;
        AXP.ZLength := 0;
      end;
      AxisEditor.AddAxis(pAxis.Title.Caption, AXP);
    end;

    AxisEditor.NoComboBox.ItemIndex := TAxis(pClickedObject).Index;
    AxisEditor.SelectAxis(TAxis(pClickedObject).Index);
    AxisEditor.HelpFile := FHelpFile;

    if (AxisEditor.ShowModal = mrOK) then
      ApplyAxisChange(AxisEditor)
     {$IFDEF GUI}else Invalidate{$ENDIF};

    AxisEditor.Free;

    ZeroScreenStuff;
  end; {GetAxisFrom User}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ApplyAxisChange
  Description: This applies changes from the PropertiesDialog.
       Author: Mat Ballard
 Date created: 03/28/2001
Date modified: 03/28/2001 by Mat Ballard
      Purpose: User interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ApplyAxisChange(Sender: TObject);
var
  i: Integer;
  pAXP: ^TAxisProperty;
  pAxis: TAxis;
begin
  for i := 0 to FAxes.Count-1 do
  begin
    pAXP := TAxisEditorForm(Sender).AxisPropertyList.Items[i];
    pAxis := FAxes[i];
    if (FEditable) then
      pAxis.Title.Caption := TAxisEditorForm(Sender).AxisNames.Strings[i];
    pAxis.Title.Position := pAXP^.TitlePosition;
    pAxis.Labels.NumberFormat := pAXP^.LabelFormat;
    pAxis.Labels.Digits := pAXP^.LabelDigits;
    pAxis.Labels.Precision := pAXP^.LabelPrecision;
    pAxis.Labels.DefaultSide := pAXP^.DefaultSide;
    pAxis.Title.DefaultSide := pAXP^.TitleDefaultSide;
    pAxis.Pen.Color := pAXP^.PenColor;
    pAxis.Pen.Width := pAXP^.PenWidthIndex;
    pAxis.Pen.Style := TPenStyle(pAXP^.PenStyleIndex);
    pAxis.AutoTick := pAXP^.TickAuto;
    pAxis.TickSize := pAXP^.TickSize;
    //pAxis.TickDirection := pAXP^.TickDirection;
    pAxis.TickMinor := pAXP^.TickMinors;
    pAxis.StepSize := pAXP^.TickStepSize;
    pAxis.StepStart := pAXP^.TickStepStart;
    pAxis.AutoScale := pAXP^.ScaleAuto;
    pAxis.SetMinMax(pAXP^.ScaleMin, pAXP^.ScaleMax);
    pAxis.Intercept := pAXP^.ScaleIntercept;
    pAxis.ArrowSize := pAXP^.ArrowSize;
    //pAxis.Alignment := pAXP^.ArrowDirection;
    pAxis.LimitLower := pAXP^.LimitLower;
    pAxis.LimitUpper := pAXP^.LimitUpper;
    pAxis.LimitsVisible := pAXP^.LimitsVisible;
    if (FAxes[i].AxisType = atZ) then
    begin
      pAxis.Angle := pAXP^.ZAngle;
      pAxis.Width := pAXP^.ZLength;
      pAxis.ZInterceptY := pAXP^.ZInterceptY;
    end;
{moved here to prevent log exceptions:}
    pAxis.LogScale := pAXP^.ScaleLog;
    pAxis.Visible := pAXP^.Visible;
    //pAxis.DoTicks;
    pAxis.ReScale;
  end; {for}
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NewNoteClick
  Description: Creates a new note
       Author: Mat Ballard
 Date created: 11/1/2000
Date modified: 08/17/2001 by Mat Ballard
      Purpose: appearance management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NewNoteClick(Sender: TObject);
var
  ACaption: String;
  ANote: TNote;
begin
  ACaption := InputBox('New Note', 'Please enter the new ', 'New Note');
  ANote := CreateNewNote(Selection.Left, Selection.Top, ACaption);
{Now let the user move the pointer:}
  pClickedObject := ANote;
  FClickedObjectType := soNote;
  Screen.Cursor := crScope;
  ScreenJob := sjMoveNotePointer;
  FInstructions := 'Move the cursor to the new note pointer then click';
end;
{$ENDIF} {GUI PasteSeriesClick}

{------------------------------------------------------------------------------
     Function: TCustomPlot.CreateNewNote
  Description: Creates a new note programmatically
       Author: Mat Ballard
 Date created: 08/16/2001
Date modified: 08/16/2001 by Mat Ballard
      Purpose: appearance management
 Known Issues:
 Return Value: the created note
 ------------------------------------------------------------------------------}
function TCustomPlot.CreateNewNote(ALeft, ATop: Integer; ACaption: String): TNote;
begin
  Result := TNote(FNotes.Add);
  Result.Caption := ACaption;
  Result.Name := IntToStr(FNotes.Count);
  Result.MoveTo(ALeft, ATop);
  Result.ArrowLeft := ALeft + 1;
  Result.ArrowTop := ATop + 1;
  DoStyleChange(Self);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NewTextClick
  Description: Creates a new Text
       Author: Mat Ballard
 Date created: 11/1/2000
Date modified: 08/17/2001 by Mat Ballard
      Purpose: appearance management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NewTextClick(Sender: TObject);
var
  ACaption: String;
begin
  ACaption := InputBox('New Text', 'Please enter the new ', 'New Text');
  CreateNewText(Selection.Left, Selection.Top, ACaption);
end;
{$ENDIF} {GUI NewTextClick}

{------------------------------------------------------------------------------
     Function: TCustomPlot.CreateNewText
  Description: Creates a new Text programmatically
       Author: Mat Ballard
 Date created: 08/16/2001
Date modified: 08/16/2001 by Mat Ballard
      Purpose: appearance management
 Known Issues:
 Return Value: the created Text
 ------------------------------------------------------------------------------}
function TCustomPlot.CreateNewText(ALeft, ATop: Integer; ACaption: String): TText;
begin
  Result := TText(FTexts.Add);
  Result.Caption := ACaption;
  Result.Name := IntToStr(FTexts.Count);
  Result.MoveTo(ALeft, ATop);
  //Result.Tag := Ord(soText);
  //Result.OnCaptionChange := Self.DoStyleChange;

  //FTexts.Add(Result);
  DoStyleChange(Self);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteNoteClick
  Description: Deletes a note or text
       Author: Mat Ballard
 Date created: 11/14/2000
Date modified: 05/29/2002 by Mat Ballard
      Purpose: appearance management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteNoteClick(Sender: TObject);
begin
  GetNoteFromUser(TRUE);
  {if (GetNoteFromUser <> nil) then
  begin
    if (TObject(pClickedObject) is TNote) then
    begin
      FNotes.Delete(TCollectionItem(pClickedObject).Index);
    end;
    if (TObject(pClickedObject) is TText) then
    begin
      FTexts.Delete(TCollectionItem(pClickedObject).Index);
    end;
  end;}
  DoStyleChange(Self);
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditFontClick
  Description: Edits the font of the selected screen object
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the Font common Dialog box, and applies the results to the selected object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditFontClick(Sender: TObject);
var
  i: Integer;
  TheFont: TFont;
  FontDialog: TFontDialog;
  OptionsForm: TOptions3Form;
  Str: String;
  //FClickedObjectType: TObjectType;
begin
{has the user already selected an object ?}
  TheFont := nil;
  OptionsForm := nil;
  FontDialog := nil;
  if (pClickedObject <> nil) then
  begin
    case FClickedObjectType of
      soTitle: TheFont := FTitle.Font;
      soLegend: TheFont := FLegend.Font;
      soXAxis, soYAxis, soZAxis,
      soXAxisLabel, soYAxisLabel, soZAxisLabel: TheFont := TAxisLabel(pClickedObject).Font;
      soXAxisTitle, soYAxisTitle, soZAxisTitle: TheFont := TTitle(pClickedObject).Font;
      soText: TheFont := TText(pClickedObject).Font;
      soNote: TheFont := TNote(pClickedObject).Font;
    end;
  end;

  if (TheFont = nil) then
  begin
{get the user to select an object:}
    OptionsForm := TOptions3Form.Create(nil);
{change to check boxes:}
    OptionsForm.SetType(otCheck);
    OptionsForm.Caption := 'Edit which Fonts ?';
    OptionsForm.QuestionLabel.Caption := 'Which Fonts you want to edit ?';
    OptionsForm.SetListCaption('Pick one');
    OptionsForm.AddObject('Plot Title', FTitle.Font);
    OptionsForm.AddObject('Legend', FLegend.Font);
    for i := 0 to FAxes.Count-1 do
    begin
      OptionsForm.AddObject(FAxes[i].Name, FAxes[i].Labels.Font);
      OptionsForm.AddObject(FAxes[i].Title.Caption, FAxes[i].Title.Font);
    end;
    for i := 0 to FNotes.Count-1 do
    begin
      Str := Copy(FNotes.Items[i].Caption, 1, 20);
      OptionsForm.AddObject(Str, FNotes.Items[i].Font);
    end;
    for i := 0 to FTexts.Count-1 do
    begin
      Str := Copy(FTexts.Items[i].Caption, 1, 20);
      OptionsForm.AddObject(Str, FTexts.Items[i].Font);
    end;

    if (OptionsForm.ShowModal = mrOK) then
    begin
      TheFont := TFont(OptionsForm.SelectedObject);
    end;
  end; {if object selected}

  if (TheFont <> nil) then
  begin
    FontDialog := TFontDialog.Create(Self);
    FontDialog.Font.Assign(TheFont);
    if (FontDialog.Execute) then
    begin
      FontDialog.Tag := -999; // flag that Font Dialog has been OK'ed
      TheFont.Assign(FontDialog.Font);
      DoStyleChange(Self);
    end;
  end;

  if (Assigned(OptionsForm)) then
  begin
    if (Assigned(FontDialog) and (FontDialog.Tag = -999)) then
    begin {New font was selected - was it for multiple items ?}
      for i := 0 to OptionsForm.ItemCount-1 do
      begin
        if (OptionsForm.Checked(i)) then
        begin
          TheFont := TFont(OptionsForm.Objects(i));
          TheFont.Assign(FontDialog.Font);
        end;
      end;
    end;
    OptionsForm.Free;
  end;
  if (Assigned(FontDialog)) then
    FontDialog.Free;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditPropertiesClick
  Description: Edits the other properties of the Plot
       Author: Mat Ballard
 Date created: 10/10/2000
Date modified: 10/10/2000 by Mat Ballard
      Purpose: Runs the Properties Dialog box, and applies the results to the selected objects
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditPropertiesClick(
  Sender: TObject);
var
  PlotPropertyEditor: TPlotPropertyEditorForm;
begin
  PlotPropertyEditor := TPlotPropertyEditorForm.Create(nil);
  PlotPropertyEditor.ThePlot := TObject(Self);
  PlotPropertyEditor.HelpFile := FHelpFile;

  PlotPropertyEditor.PlotTypeComboBox.ItemIndex := Ord(FPlotType);
  PlotPropertyEditor.PlotColorEdit.Color := Self.ColorPlot;
  PlotPropertyEditor.PlotColorEdit.Text := ColorToString(Self.ColorPlot);
  PlotPropertyEditor.BackColorEdit.Color := Self.Color;
  PlotPropertyEditor.BackColorEdit.Text := ColorToString(Self.Color);
  PlotPropertyEditor.BackColorComboBox.ItemIndex := Ord(FColorType);
  PlotPropertyEditor.BackColor2Edit.Color := Self.Color2;
  PlotPropertyEditor.BackColor2Edit.Text := ColorToString(Self.Color2);
  PlotPropertyEditor.BordersCheckBox.Checked := FBorder.Visible;
  PlotPropertyEditor.BorderGapNEdit.AsInteger := FBorder.Gap;
  PlotPropertyEditor.BorderGapMinNEdit.AsInteger := FBorder.GapMin;
  PlotPropertyEditor.BorderGapMaxNEdit.AsInteger := FBorder.GapMax;
  PlotPropertyEditor.ClickAndDragDelayNEdit.AsInteger := MouseTimer.Interval;
  PlotPropertyEditor.ColumnGapNEdit.AsInteger := FColumnGap;
  PlotPropertyEditor.ContourDetailComboBox.ItemIndex := Ord(FContour.Detail);
  PlotPropertyEditor.ContourIntervalNEdit.AsReal := FContour.Interval;
  PlotPropertyEditor.ContourStartNEdit.AsReal := FContour.Start;
  PlotPropertyEditor.ContourWireFrameCheckBox.Checked := FContour.WireFrame;
  PlotPropertyEditor.GridComboBox.ItemIndex := Ord(FGrids.GridType);
  PlotPropertyEditor.GridStyleComboBox.ItemIndex := Ord(FGrids.Style);
  PlotPropertyEditor.GridColorEdit.Color := FGrids.Color;
  PlotPropertyEditor.GridColorEdit.Text := ColorToString(FGrids.Color);
  PlotPropertyEditor.PolarRangeNEdit.AsReal := FPolarRange;
  PlotPropertyEditor.PieRowCountComboBox.ItemIndex := FPieRowCount - 1;
  PlotPropertyEditor.PrintOrientationComboBox.ItemIndex := Ord(FPrintOrientation);
  PlotPropertyEditor.WallTypeComboBox.ItemIndex := Ord(FGrids.WallType);
  PlotPropertyEditor.WallColorEdit.Color := FGrids.WallColor;
  PlotPropertyEditor.WallColorEdit.Text := ColorToString(FGrids.WallColor);
  PlotPropertyEditor.WallColor2Edit.Color := FGrids.WallColor2;
  PlotPropertyEditor.WallColor2Edit.Text := ColorToString(FGrids.WallColor2);
  PlotPropertyEditor.XYFastAtNEdit.AsInteger := FXYFastAt;
  if (Assigned(ZAxis)) then
  begin
    PlotPropertyEditor.ZAxisAngleNEdit.AsInteger := ZAxis.Angle;
    PlotPropertyEditor.ZLengthNEdit.AsInteger := ZAxis.Length;
  end;
  PlotPropertyEditor.ZSeriesLinkCheckBox.Checked := FZLink;
  PlotPropertyEditor.ZSeriesNamesCheckBox.Checked := FZSeriesNames;
  PlotPropertyEditor.ColorMinNEdit.AsInteger := Round(FContour.LambdaMin);
  PlotPropertyEditor.ColorMaxNEdit.AsInteger := Round(FContour.LambdaMax);
  PlotPropertyEditor.GammaNEdit.AsReal := FContour.Gamma;
  PlotPropertyEditor.CreatedByEdit.Text := FCreatedBy;
  PlotPropertyEditor.DescriptionEdit.Text := FDescription;
  if (FLegend.Visible) then
  begin
    if (FLegend.Direction = drHorizontal) then
      PlotPropertyEditor.LegendRadioGroup.ItemIndex := 0
     else
      PlotPropertyEditor.LegendRadioGroup.ItemIndex := 1;
  end
   else
    PlotPropertyEditor.LegendRadioGroup.ItemIndex := 2;

  if (PlotPropertyEditor.ShowModal = mrOK) then
    ApplyPropertiesChange(PlotPropertyEditor)
   else
    Invalidate;

  PlotPropertyEditor.Free;

  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ApplyPropertiesChange
  Description: This applies changes from the PropertiesDialog.
       Author: Mat Ballard
 Date created: 03/28/2001
Date modified: 03/28/2001 by Mat Ballard
      Purpose: User interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ApplyPropertiesChange(Sender: TObject);
begin
  with TPlotPropertyEditorForm(Sender) do
  begin
    PaintLock;
    MouseTimer.Interval := ClickAndDragDelayNEdit.AsInteger;
    Self.ColorPlot := PlotColorEdit.Color;
    Self.Color := BackColorEdit.Color;
    FColorType := TGridType(BackColorComboBox.ItemIndex);
    Self.Color2 := BackColor2Edit.Color;
    FBorder.Visible := BordersCheckBox.Checked;
    FBorder.Gap := BorderGapNEdit.AsInteger;
    FBorder.GapMin := BorderGapMinNEdit.AsInteger;
    FBorder.GapMax := BorderGapMaxNEdit.AsInteger;
    FColumnGap := ColumnGapNEdit.AsInteger;
    FContour.Detail := TContourDetail(ContourDetailComboBox.ItemIndex);
    FContour.Interval := ContourIntervalNEdit.AsReal;
    FContour.Start := ContourStartNEdit.AsReal;
    FContour.WireFrame := ContourWireFrameCheckBox.Checked;
    FGrids.GridType := TGridType(GridComboBox.ItemIndex);
    FGrids.Style := TPenStyle(GridStyleComboBox.ItemIndex);
    FGrids.Color := GridColorEdit.Color;
    FGrids.WallType := TGridType(WallTypeComboBox.ItemIndex);
    FGrids.WallColor := WallColorEdit.Color;
    FGrids.WallColor2 := WallColor2Edit.Color;
    FPieRowCount := PieRowCountComboBox.ItemIndex + 1;
    FPolarRange := PolarRangeNEdit.AsReal;
    FPrintOrientation := TPrinterOrientation(PrintOrientationComboBox.ItemIndex);
    FXYFastAt := XYFastAtNEdit.AsInteger;
    FContour.LambdaMin := ColorMinNEdit.AsInteger;
    FContour.LambdaMax := ColorMaxNEdit.AsInteger;
    FContour.Gamma := GammaNEdit.AsReal;
    FZLink := ZSeriesLinkCheckBox.Checked;
    FZSeriesNames := ZSeriesNamesCheckBox.Checked;
    if (FEditable) then
    begin
      FCreatedBy := CreatedByEdit.Text;
      FDescription := DescriptionEdit.Text;
    end;
    FLegend.Visible := TRUE;
    case LegendRadioGroup.ItemIndex of
      0: FLegend.Direction := drHorizontal;
      1: FLegend.Direction := drVertical;
      2:
        begin
          FLegend.Direction := drVertical;
          FLegend.Visible := FALSE;
        end;
    end;
    PlotType := TPlotType(PlotTypeComboBox.ItemIndex);
    if (Assigned(ZAxis)) then
    begin
      ZAxis.Angle := ZAxisAngleNEdit.AsInteger;
      ZAxis.Length := ZLengthNEdit.AsInteger;
    end;
    PaintUnLock;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditPointClick
  Description: Runs the Point Editor of the selected data point
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditPointClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FClickedSeries.EditPoint(ThePointNumber, FHelpFile);
    //DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditDataClick
  Description: Runs the Data Editor for the selected Series
       Author: Mat Ballard
 Date created: 03/13/2001
Date modified: 03/13/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditDataClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FClickedSeries.EditData(FHelpFile);
  end; {GetSeries}
  ZeroScreenStuff;
  Invalidate;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditSeriesClick
  Description: Runs the Series Editor of the selected data Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditSeriesClick(
  Sender: TObject);
var
  i: Integer;
  SeriesEditor: TSeriesEditorForm;
  ASP: TSeriesProperty;
  aSeries: TSeries;
begin
  FClickedSeries := GetSeriesFromUser;
  if (Assigned(FClickedSeries)) then
  begin
    SeriesEditor := TSeriesEditorForm.Create(nil);
    SeriesEditor.ThePlot := TObject(Self);

{Load the Y Axis Combo Box:}
    for i := 1 to FAxes.Count-1 do
    begin
      if (FAxes[i].AxisType >= atPrimaryY) then
        SeriesEditor.YAxisComboBox.Items.AddObject(FAxes[i].Title.Caption, FAxes[i]);
    end;

{Iterate over all series:}
    for i := 0 to FSeriesList.Count-1 do
    begin
      aSeries := TSeries(FSeriesList[i]);
      ASP.BrushColor := aSeries.Brush.Color;
      ASP.BrushStyleIndex := Ord(aSeries.Brush.Style);
      ASP.BubbleSize := aSeries.BubbleSize;
      ASP.DeltaX := aSeries.DeltaX;
      ASP.DeltaY := aSeries.DeltaY;
      ASP.ExternalXSeries := aSeries.XDataSeries <> nil;
      ASP.IsLine := aSeries.IsLine;
{returns 0..MY_COLORS_MAX}
      ASP.PenColor := aSeries.Pen.Color;
      ASP.PenWidthIndex := aSeries.Pen.Width;
      ASP.PenStyleIndex := Ord(aSeries.Pen.Style);
      ASP.SeriesTypeIndex := Ord(aSeries.SeriesType);
      ASP.ShadeLimits := aSeries.ShadeLimits;
      ASP.ShowFit := aSeries.ShowFit;
      ASP.SymbolIndex := Ord(aSeries.Symbol);
      ASP.SymbolSize := aSeries.SymbolSize;
      ASP.SymbolCheck := aSeries.SymbolCheck;
      ASP.Visible := aSeries.Visible;
      ASP.XDataIndependent := not (Assigned(aSeries.XDataSeries));
      ASP.YAxisIndex := aSeries.YAxisIndex;
      //ASP.YAxis := aSeries.YAxis;
      SeriesEditor.AddSeries(aSeries.Name, ASP);
    end;
    SeriesEditor.NoComboBox.ItemIndex := FClickedSeries.Index;
    SeriesEditor.SelectSeries(FClickedSeries.Index);

    SeriesEditor.HelpFile := FHelpFile;

    if (SeriesEditor.ShowModal = mrOK) then
      ApplySeriesChange(SeriesEditor)
     else
      Invalidate;

    SeriesEditor.Free;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ApplySeriesChange
  Description: This applies changes from the SeriesEditor Dialog.
       Author: Mat Ballard
 Date created: 03/28/2001
Date modified: 03/28/2001 by Mat Ballard
      Purpose: User interface management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ApplySeriesChange(Sender: TObject);
var
  i: Integer;
  aSeries: TSeries;
  pASP: ^TSeriesProperty;
begin
  for i := 0 to FSeriesList.Count-1 do
  begin
    pASP := TSeriesEditorForm(Sender).SeriesPropertyList.Items[i];
    aSeries := TSeries(FSeriesList[i]);
    aSeries.Brush.Color := pASP^.BrushColor;
    aSeries.Brush.Style := TBrushStyle(pASP.BrushStyleIndex);
    aSeries.BubbleSize := pASP^.BubbleSize;
    aSeries.DeltaX := pASP^.DeltaX;
    aSeries.DeltaY := pASP^.DeltaY;
    aSeries.IsLine := pASP^.IsLine;
    if (FEditable) then
      aSeries.Name := TSeriesEditorForm(Sender).SeriesNames.Strings[i];
    aSeries.Pen.Color := pASP^.PenColor;
    aSeries.Pen.Width := pASP^.PenWidthIndex;
    aSeries.Pen.Style := TPenStyle(pASP^.PenStyleIndex);
    aSeries.SeriesType := TSeriesType(pASP^.SeriesTypeIndex);
    aSeries.ShadeLimits := pASP^.ShadeLimits;
    aSeries.ShowFit := pASP^.ShowFit;
    aSeries.Symbol := TSymbol(pASP^.SymbolIndex);
    aSeries.SymbolSize := pASP^.SymbolSize;
    aSeries.SymbolCheck := pASP^.SymbolCheck;
    aSeries.Visible := pASP^.Visible;
    aSeries.YAxisIndex := pASP^.YAxisIndex;
    if ((pASP^.XDataIndependent) and (Assigned(aSeries.XDataSeries))) then
    begin
{this series did depend on X Data in another series,
but user wants to make it independent:}
      aSeries.MakeXDataIndependent;
    end;
  end; {for}
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ResetDisplacementClick
  Description: Puts the selected Axis back where it came from
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DeltaX and DeltaY properties of the selected Axis to ZeroScreenStuff
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ResetDisplacementClick(Sender: TObject);
var
  i: Integer;
begin
  if (FClickedSeries = nil) then
  begin
    for i := 0 to FSeriesList.Count-1 do
    begin
      Series[i].DeltaX := 0;
      Series[i].DeltaY := 0;
    end;
  end
  else
  begin
    FClickedSeries.DeltaX := 0;
    FClickedSeries.DeltaY := 0;
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetNoteFromUser
  Description: Gets the user to select (if not already done so) a Note
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 05/29/2002 by Mat Ballard
      Purpose: user interface management
 Known Issues: had to use multiple exits - unclean, but better than 5-level ifs
 ------------------------------------------------------------------------------}
function TCustomPlot.GetNoteFromUser(DoDelete: Boolean): TText;
var
  OptionsForm: TOptions3Form;
  //ACollectionItem: TCollectionItem;
  i, Index: Integer;
  Str: String;
begin
  Result := nil;
  if (FNotes.Count + FTexts.Count = 0) then
    exit;

{has the user already selected an object ?}
  if ((FClickedObjectType = soNote) or
      (FClickedObjectType = soText)) then
  begin
    Result := TText(pClickedObject);
    Result.Free;
    exit;
  end;

  if (FNotes.Count + FTexts.Count = 1) then
  begin
{there is only one Note or Text:}
    if (FNotes.Count = 1) then
    begin
      pClickedObject := FNotes[0];
      FClickedObjectType := soNote;
    end
    else if (FTexts.Count = 1) then
    begin
      pClickedObject := FTexts[0];
      FClickedObjectType := soText;
    end;
    Result := TText(pClickedObject);
    Result.Free;
    exit;
  end;

  {get the user to select an object:}
  OptionsForm := TOptions3Form.Create(nil);
{Change to check boxes:}
  OptionsForm.SetType(otCheck);
  OptionsForm.Caption := 'Which Notes ?';
  if (DoDelete) then
    OptionsForm.QuestionLabel.Caption := 'Which Notes/Texts do you want to Delete ?'
   else
    OptionsForm.QuestionLabel.Caption := 'Which Notes/Texts do you want ?';
  for i := 0 to FNotes.Count-1 do
  begin
    Str := Copy(FNotes.Items[i].Caption, 1, 20);
    OptionsForm.AddObject(Str, FNotes.Items[i]);
  end;
  for i := 0 to FTexts.Count-1 do
  begin
    Str := Copy(FTexts.Items[i].Caption, 1, 20);
    OptionsForm.AddObject(Str, TText(FTexts.Items[i]));
  end;

  if (OptionsForm.ShowModal = mrOK) then
  begin
    Index := OptionsForm.ItemIndex;
    if (Index >= 0) then
    begin
      pClickedObject := TCollectionItem(OptionsForm.SelectedObject);
      FClickedObjectType := TObjectType(TText(pClickedObject).Tag);
      Result := TText(pClickedObject);
      if (DoDelete) then
      begin
        for i := 0 to OptionsForm.Checks.Count-1 do
          if (OptionsForm.Checks.CheckBoxes[i].Checked) then
          begin
            TCollectionItem(OptionsForm.Checks.Items.Objects[i]).Free;
            {ACollectionItem := OptionsForm.Checks.Items.Objects[i];
            ACollectionItem.Collection.Delete(ACollectionItem.Index);}
          end;
      end;
    end;
  end;
  OptionsForm.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetSeriesFromUser
  Description: Gets the user to select (if not already done so) a Axis
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: user interface management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetSeriesFromUser: TSeries;
var
  OptionsForm: TOptions3Form;
  i: Integer;
begin
  Result := nil;
{has the user already selected an object ?}
  //if (FClickedObjectType = soSeries) then
  if (Assigned(FClickedSeries)) then
  begin
    Result := FClickedSeries;
  end
  else
  begin
    ThePointNumber := 0;
    if (FSeriesList.Count = 1) then
    begin
{there is only one Series:}
      FClickedObjectType := soSeries;
      TheSeries := 0;
      FClickedSeries := FSeriesList[0];
      Result := FClickedSeries;
    end;

    if (FClickedObjectType <> soSeries) then
  {still no Series selected:}
    begin
  {get the user to select an object:}
      OptionsForm := TOptions3Form.Create(nil);
      OptionsForm.Caption := 'Which Series ?';
      OptionsForm.QuestionLabel.Caption := 'Which Series you want to work on ?';
      for i := 0 to FSeriesList.Count-1 do
      begin
        OptionsForm.Add(TSeries(FSeriesList[i]).Name);
      end;
      if (OptionsForm.ShowModal = mrOK) then
      begin
        FClickedObjectType := soSeries;
        FClickedSeries := FSeriesList[OptionsForm.ItemIndex];
        Result := FClickedSeries;
      end;
      OptionsForm.Free;
    end; {if object selected}
  end; {if clicked object is Series}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetAxisFromUser
  Description: Gets the user to select (if not already done so) a Axis
       Author: Mat Ballard
 Date created: 06/25/1999
Date modified: 06/25/2000 by Mat Ballard
      Purpose: user interface management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetAxisFromUser(StartAxis: Word): TAxis;
var
  OptionsForm: TOptions3Form;
  i: Integer;
begin
{has the user already selected an object ?}
{see GetTheClickedObject}
  if (Assigned(pClickedObject)) then
    if (TObjectType(TRectangle(pClickedObject).Tag) in [soXAxis, soYAxis, soZAxis]) then
    begin
      Result := TAxis(pClickedObject);
      exit;
    end;

{still no Axis selected:}
{get the user to select an object:}
  OptionsForm := TOptions3Form.Create(nil);
  OptionsForm.Caption := 'Which Axis ?';
  OptionsForm.QuestionLabel.Caption := 'Which Axis you want to work on ?';
  for i := StartAxis to FAxes.Count-1 do
  begin
    OptionsForm.AddObject(FAxes[i].Name, FAxes[i]);
  end;

  if (OptionsForm.ShowModal = mrOK) then
  begin
    pClickedObject := TAxis(OptionsForm.SelectedObject);
    FClickedObjectType := TObjectType(TRectangle(pClickedObject).Tag);
    Result := TAxis(pClickedObject);
  end
  else
  begin
    pClickedObject := nil;
    FClickedObjectType := soNone;
    Result := nil;
  end;
  OptionsForm.Free;
end;
{$ENDIF} {GUI DeleteNoteClick}

{File manipulation ------------------------------------------------------------}
{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NewClick
  Description: Creates a new, blank Graph
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NewClick(
  Sender: TObject);
begin
  Self.Clear(TRUE);
  FFilename := '';
  if Assigned(FOnFileNew) then OnFileNew(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OpenClick
  Description: Opens a file on disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the File Open common dialog then the LoadFromFile method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OpenClick(
  Sender: TObject);
var
  Ext: String;
  OpenDialog: TOpenDialog;
begin
{We have to display a File Open Dialog:}
  OpenDialog := TOpenDialog.Create(Self);
  OpenDialog.Title := 'Open ...';
  OpenDialog.Filter := FileTypes;
  OpenDialog.Options := [ofOverwritePrompt];

  Ext := ExtractFileExt(FFileName);
  mOpenFilterIndex := GetFilterIndex(Ext);
  OpenDialog.FilterIndex := mOpenFilterIndex;

  OpenDialog.FileName := '*' + Ext;

  if (Length(mOpenDriveDir) > 0) then
    OpenDialog.InitialDir := mOpenDriveDir
   else
    OpenDialog.InitialDir := GetFileDriveDir;

  if (OpenDialog.Execute) then
  begin
    OpenFile(OpenDialog.FileName);
    mOpenFilterIndex := OpenDialog.FilterIndex;
  end;
  OpenDialog.Free;
end;
{$ENDIF} {GUI NewClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OpenFile
  Description: Opens a file on disk
       Author: Mat Ballard
 Date created: 08/12/2000
Date modified: 08/12/2000 by Mat Ballard
      Purpose: Called from TPlotMenu.HandleFileClick
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OpenFile(
  TheFile: String);
begin
  if (FileExists(TheFile)) then
  begin
{Delete any existing Series:}
    Clear(FALSE);
    mInitialSeriesCount := FSeriesList.Count;
    FileName := TheFile;
    mOpenDriveDir := ExtractFilePath(FFileName);
{Finally, Open it:}
    LoadFromFile(FFileName);

    DoFileOpen(FFileName);
  end;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoFileOpen
  Description: Fires the OnFileOpen event
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoFileOpen(AFileName: String);
begin
  if Assigned(FOnFileOpen) then
    OnFileOpen(Self, AFileName);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ClearOverlaysClick
  Description: Gets rid of the overlaid data
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the FSeriesList.Delete method for each overlaid Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ClearOverlaysClick(
  Sender: TObject);
var
  i: Integer;
begin
  if (FFirstOverlay < 0) then raise
    EComponentError.Create('There are no Overlays to Clear !');

  for i := FSeriesList.Count-1 downto FFirstOverlay do
  begin
    //TSeries(FSeriesList[i]).Free; - no longer needed
    FSeriesList.Delete(i);
  end;
  FFirstOverlay := -1;
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OverlayDataClick
  Description: Overlays data
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the Open common dialog, then LoadFromFile the selected files
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OverlayClick(
  Sender: TObject);
var
  i: Integer;
  OverlayDialog: TOpenDialog;
begin
{We have to display a File Overlay Dialog:}
  OverlayDialog := TOpenDialog.Create(Self);
  OverlayDialog.Title := 'Overlay Data As';
  OverlayDialog.Filter := FileTypes;
  OverlayDialog.Options :=
    [ofFileMustExist, ofPathMustExist, ofAllowMultiSelect];

  OverlayDialog.FileName := '*' + FDefaultExtension;

  OverlayDialog.FilterIndex := mOpenFilterIndex;

  if (Length(mOverlayDriveDir) > 0) then
    OverlayDialog.InitialDir := mOpenDriveDir
   else
    OverlayDialog.InitialDir := GetFileDriveDir;

  if (OverlayDialog.Execute) then
  begin
{This may be a second or third overlay:}
    if (FFirstOverlay < 0) then
      FFirstOverlay := FSeriesList.Count;
    mInitialSeriesCount := FSeriesList.Count;
    for i := 0 to OverlayDialog.Files.Count - 1 do
    begin
      mOverlayDriveDir := ExtractFilePath(OverlayDialog.Files.Strings[i]);
{Finally, Overlay it:}
      Self.LoadFromFile(OverlayDialog.Files.Strings[i]);
    end;
  end;
  OverlayDialog.Free;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveImageClick
  Description: saves the current plot as an image
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Save Image" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveImageClick(Sender: TObject);
var
  SaveImageDialog: TSaveDialog;
  Extension, ImageName: String;
begin
{We have to display a File Save Dialog:}
  SaveImageDialog := TSaveDialog.Create(Self);
  SaveImageDialog.Title := 'Save Image As';
  SaveImageDialog.Filter := PICTURE_TYPES;
  SaveImageDialog.Options := [ofOverwritePrompt];

  if (Length(mImageDriveDir) = 0) then
    if (Length(FFileName) > 0) then
      mImageDriveDir := GetFileDriveDir;

  SaveImageDialog.FilterIndex := mImageFilterIndex;
{which starts off at zero, then  may change}

  if (Length(FFileName) > 0) then
  begin
    SaveImageDialog.FileName :=
      ChangeFileExt(FFileName, ImageExtensions[mImageFilterIndex-1]);
  end
  else
  begin
    SaveImageDialog.FileName :=
      'plot1' + ImageExtensions[mImageFilterIndex-1];
  end;

  SaveImageDialog.InitialDir := mImageDriveDir;

  if (SaveImageDialog.Execute) then
  begin
    ImageName := SaveImageDialog.FileName;
    mImageDriveDir := ExtractFilePath(ImageName);

    Extension := LowerCase(ExtractFileExt(ImageName));
    if (Length(Extension) = 0) then
    begin
      Extension := '.' +
        ImageExtensions[SaveImageDialog.FilterIndex-1];
      ImageName := ImageName + Extension;
    end;

{Finally, save it:}
{We base this on the extension, rather than FilterIndex:}
    if (Extension = '.bmp') then
    begin
      mImageFilterIndex := 3;
      SaveAsBitMap(ImageName);
    end;
    if (Extension = '.wmf') then
    begin
      mImageFilterIndex := 1;
      SaveAsDrawing(ImageName);
    end
    else if (Extension = '.emf') then
    begin
      mImageFilterIndex := 2;
      SaveAsDrawing(ImageName);
    end;
{$IFDEF GIF}
    if (Extension = '.gif') then
    begin
      mImageFilterIndex := 4;
      SaveAsGIF(ImageName);
    end;
{$ENDIF}
{$IFDEF PNG}
    if (Extension = '.png') then
    begin
      mImageFilterIndex := 5;
      SaveAsPNG(ImageName);
    end;
{$ENDIF}
  end;
  SaveImageDialog.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveClick
  Description: Saves the graph
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs SaveToFile or SaveAsClick
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveClick(Sender: TObject);
begin
  if (sSaved in FStatus) then
    SaveToFile(FFileName)
   else
    SaveAsClick(Sender);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsClick
  Description: Saves the data to disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the Save common dialog box then SaveToFile
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsClick(Sender: TObject);
var
  AFileName,
  Ext: String;
  SaveDialog: TSaveDialog;
begin
{We have to display a File Save Dialog:}
  SaveDialog := TSaveDialog.Create(Self);
  SaveDialog.Title := 'Save Data As';
  SaveDialog.Filter := FileTypes;
  SaveDialog.Options := [ofOverwritePrompt];

  Ext := ExtractFileExt(FFileName);
  mSaveFilterIndex := GetFilterIndex(Ext);
  SaveDialog.FilterIndex := mSaveFilterIndex;

  SaveDialog.InitialDir := GetFileDriveDir;
  SaveDialog.FileName := ExtractFileName(FFileName);

  if (SaveDialog.Execute) then
  begin
    AFileName := SaveDialog.FileName;
    Ext := ExtractFileExt(AFileName);
    if (Length(Ext) = 0) then
    begin
      Ext := FileExtensions[SaveDialog.FilterIndex-1];
      AFileName := AFileName + Ext;
    end;

{Double-whammy problem: save with 'plot'extension, but different filter,
 should save in that (text) format.
 Save with other extension, but FilterIndex=0 (plot type), then extension
 should override, so also save in text.}

    if (Ext = FileExtensions[0]) then
    begin
      mSaveFilterIndex := 1;
      if (SaveDialog.FilterIndex = 1) then
        mAsText := FALSE or (soAsText in FSaveOptions) //binary plot format
      else
        mAsText := TRUE; // text file format
    end
    else if (Ext = FileExtensions[1]) then
    begin {.csv}
      mSaveFilterIndex := 2;
      mAsText := TRUE;
    end {.csv}
    else if (Ext = FileExtensions[2]) then
    begin
      mSaveFilterIndex := 3;
      mAsText := TRUE;
    end {.txt}
    else if ((Ext = FileExtensions[3]) or
             (Ext = FileExtensions[4])) then
    begin
      mSaveFilterIndex := 4;
      //mAsText := TRUE; maybe, maybe not !
    end {.html}
    else
    begin
      mSaveFilterIndex := 5;
      mAsText := TRUE;
    end; {.*}

{Finally, save it:}
    if (mSaveFilterIndex = 4) then
      Self.SaveToHTMLFile(AFileName)
    else
      Self.SaveToFile(AFileName);
    mSaveFilterIndex := SaveDialog.FilterIndex;
  end;
  SaveDialog.Free;
end;
{$ENDIF} {GUI ClearOverlaysClick}

{Saving data to disk --------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LoadFromFile
  Description: Opens data on disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Opens the data and feeds it into LoadFromStream
 Known Issues: Can be called by either OpenClick or OverlayClick
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LoadFromFile(AFileName: String);
var
  PropsFile: String;
  TheStream: TMemoryStream;
begin
  TheStream := TMemoryStream.Create;
  try
{First, load properties, but only if we are opening, not overlaying,
 and also if the properties file exists:}
    if (FSeriesList.Count = 0) then
    begin
      PropsFile := ChangeFileExt(FFileName, PROP_EXTENSION);
      if (FileExists(PropsFile)) then
      begin
        if (Assigned(ZAxis)) then
          RemoveZAxis;
{IMPORTANT NOTE: this caused a lot of pain: when Collections are streamed in,
 they are first emptied. This means that:
        a. all series are destroyed;
        b. all axes are destroyed;
        c. all notes, texts are destroyed;
        d. all their events and aliases are invalid.
 The solution to this problem is to override the TCollection.Clear method for
 Axes, Series, etc. Unfortunately, this is impossible because Borland decided
 to make Clear _Static_ !}
        LoadPropertiesFromFile(PropsFile);
      end;
{NB: this clears all collections (Axes, Notes, etc), then re-creates them.}
    end;
{Next, load the data file:}
    TheStream.LoadFromFile(AFileName);
    FileName := AFileName;
{And "parse" it from the memory stream:}
    Self.LoadFromStream(TheStream, ',');
  finally
    TheStream.Free;
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.IsPlotFile
  Description: Examines the memory stream, and decides if it is a plot file
       Author: Mat Ballard
 Date created: 11/25/2002
Date modified: 11/25/2002 by Mat Ballard
      Purpose: File opening / overlaying / data pasting
 Known Issues: Can be called by either OpenClick or OverlayClick
 ------------------------------------------------------------------------------}
function TCustomPlot.IsPlotFile(AStream: TMemoryStream; var FileVersion: Integer): Boolean;
var
  TheLine: String;
begin
  Result := FALSE;
{Line the first:}
  TheLine := ReadLine(AStream);
  if (Pos('TPlot', TheLine) = 0) then exit;

{Line the second:}
  TheLine := ReadLine(AStream);
  if (Pos('FileFormat', TheLine) = 0) then exit;
  GetWord(TheLine, '=');
  FileVersion := StrToInt(TheLine);
  if (FileVersion > MAX_FILE_VERSION) then exit;

  FTitle.Caption := ReadLine(AStream);

  TheLine := ReadLine(AStream);
  if ('StartTime' = GetWord(TheLine, '=')) then
    FStartTime := StrToFloat(TheLine);

{Now comes the developer-defined header:}
  DoHeader(AStream);

  TheLine := ReadLine(AStream);
  if (TheLine <> SUBHEADER) then
  begin
{either a stuffed file, or the developer has done a naughty
and overrun his own header; we therefore try to find it from the beginning,
then reset the stream position:}
    AStream.Position := 0;
    if (not (FindStringInStream(SUBHEADER, AStream))) then exit;
    TheLine := ReadLine(AStream);
  end;
  Result := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LoadFromStream
  Description: Opens data on disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
               or decides to run it through ParseData instead
 Known Issues: Can be called by either OpenClick or OverlayClick
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LoadFromStream(AStream: TMemoryStream; Delimiter: String);
var
  FileVersion: Integer;
  CanOpen: Boolean;
{$IFDEF GUI}
  TheStrings: TStringList;
{$ENDIF}
begin
  PaintLock;

  AStream.Seek(0, soFromBeginning);
  CanOpen := IsPlotFile(AStream, FileVersion);
  if (CanOpen) then
  begin
    if (FileVersion = 300) then
      FSeriesList.LoadFromStream(AStream, Delimiter, mInitialSeriesCount, mAsText)
{$IFDEF VER200_COMPATIBLE}
    else if (FileVersion = 200) then
      FSeriesList.LoadFromStream_Ver200(AStream, mAsText)
{$ENDIF}
     else
      CanOpen := FALSE;
  end;

  if (not CanOpen) then
  begin
{$IFDEF GUI}
{maybe it's just a text file:}
    AStream.Seek(0, soFromBeginning);
    TheStrings := TStringList.Create;
    try
      TheStrings.LoadFromStream(AStream);
      FSeriesList.ParseData(TheStrings, FHelpFile);
      {mAsText := TRUE;}
    finally
      if (Assigned(TheStrings)) then
        TheStrings.Free;
    end;
{$ENDIF}
  end; {IsPlotFile}

  mInitialSeriesCount := -1;
  PaintUnLock;
  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoHeader
  Description: Informs the user of a user-defined file header
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: Fires the OnHeader event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoHeader(TheStream: TMemoryStream);
begin
  if assigned(FOnHeader) then
    OnHeader(Self, TheStream);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveToFile
  Description: Saves the data as text
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 03/07/2001 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event, then the GetData of SeriesList
 Known Issues:
      Comment: Note that we now use a TStream.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveToFile(
  AFileName: String);
var
  TheStream: TMemoryStream;
begin
  TheStream := TMemoryStream.Create;
{create the data in binary or text format:}
  Self.SaveToStream(TheStream, mAsText, ',');

{determine the file name:}
  if (Length(AFileName) > 0) then
    SetFileName(AFileName);

{save it:}
  TheStream.SaveToFile(FFileName);

  FStatus := FStatus + [sSaved];

  TheStream.Free;

  if (ExtractFileExt(FFileName) = FileExtensions[0]) then
    if (soProperties in FSaveOptions) then
      SavePropertiesToFile(ChangeFileExt(AFileName, PROP_EXTENSION));
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveToHTMLFile
  Description: Saves the data as text
       Author: Mat Ballard
 Date created: 08/25/2001
Date modified: 08/25/2001 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event, then the GetData of SeriesList
 Known Issues:
      Comment: Note that we now use a TStream.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveToHTMLFile(
  AFileName: String);
var
  AnImageFile: String;
  TheData: TStringList;
begin
  AnImageFile := ChangeFileExt(ExtractFileName(AFileName), '.png');
  TheData := TStringList.Create;

  SaveToHTMLStringList(AnImageFile, TheData);

  TheData.SaveToFile(AFileName);
  TheData.Free;
{$IFDEF PNG}
  AnImageFile := ChangeFileExt(AFileName, '.png');
  SaveAsPng(AnImageFile);
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveToHTMLStream
  Description: Saves the data as text
       Author: Mat Ballard
 Date created: 08/25/2001
Date modified: 08/25/2001 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event, then the GetData of SeriesList
 Known Issues:
      Comment: Note that we now use a TStream.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveToHTMLStream(ImageFileName: String; var TheStream: TMemoryStream);
var
  TheData: TStringList;
begin
  if (TheStream = nil) then
    TheStream := TMemoryStream.Create;
  TheData := TStringList.Create;

  SaveToHTMLStringList(ImageFileName, TheData);

  TheData.SaveToStream(TheStream);
  TheData.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveToHTMLStringList
  Description: Saves the data as HTML to a stringlist for other functions
       Author: Mat Ballard
 Date created: 08/25/2001
Date modified: 08/25/2001 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event, then the GetData of SeriesList
 Known Issues:
      Comment: Note that we now use a TStream.
 ------------------------------------------------------------------------------}
{TODO 4 -o all -c HTML : remove peculiar characters at the end of the the copy}
procedure TCustomPlot.SaveToHTMLStringList(ImageFileName: String; var TheData: TStringList);
var
  i: Integer;
  TheHeader: TStringList;
begin
  if (TheData = nil) then
    TheData := TStringList.Create;

  TheHeader := TStringList.Create;
  DoHTMLHeaderRequest(TheHeader);

{$IFDEF PNG}
  if (Length(ImageFileName) > 0) then
  begin
    TheHeader.Insert(0, '<p>');
    TheHeader.Insert(0, '<img src="' + ImageFileName + '">');
  end;
{$ENDIF}
  TheHeader.Insert(0, '<p>');
  TheHeader.Insert(0, '<h1>' + FTitle.Caption + '</h1>');
  TheHeader.Insert(0, '<body bgcolor="white">');
  TheHeader.Insert(0, '</head>');
  TheHeader.Insert(0, '<title>' + FTitle.Caption + '</title>');
  TheHeader.Insert(0, '<head>');
  TheHeader.Insert(0, '<html>');

{create the data in text format:}
  FSeriesList.DataAsHTMLTable(TheData);
{insert the header:}
  for i := TheHeader.Count-1 downto 0 do
    TheData.Insert(0, TheHeader[i]);
  TheHeader.Free;
{finish document:}
  TheData.Add('</body>');
  TheData.Add('</html>');
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveToStream
  Description: Saves the data as text
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 03/07/2001 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event, then the GetData of SeriesList
 Known Issues:
      Comment: Note that we now use a TStream.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveToStream(var TheStream: TMemoryStream;
  InText: Boolean;
  Delimiter: Char);
var
  pLine: array [0..1023] of char;
begin
  if (TheStream = nil) then
    TheStream := TMemoryStream.Create;

{D1 does not like Pointer(TheLine)^:
  TheLine := 'TPlot=' + IntToStr(TPLOT_VERSION) + CRLF;
  TheStream.Write(Pointer(TheLine)^, Length(TheLine));
so:}
  StrPCopy(pLine, 'TPlot=' + IntToStr(TPLOT_VERSION) + CRLF);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, 'FileFormat=' + IntToStr(FILE_FORMAT_VERSION) + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  StrPCopy(pLine, FTitle.Caption + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  StrPCopy(pLine, 'StartTime=' + FloatToStr(FStartTime) + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  DoHeaderRequest(TheStream);

  StrPCopy(pLine, SUBHEADER + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

{create the data in binary or text format:}
  FSeriesList.SaveToStream(TheStream, InText, Delimiter);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoHeaderRequest
  Description: Asks the user for a header
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoHeaderRequest(TheStream: TMemoryStream);
begin
  if assigned(FOnHeaderRequest) then
    OnHeaderRequest(Self, TheStream);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LoadPropertiesFromFile
  Description: Opens the properties of this instance of TPlot
       Author: Mat Ballard
 Date created: 08/03/2000
Date modified: 11/24/2002 by Mat Ballard
      Purpose: Reads the appearance of the Plot
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LoadPropertiesFromFile(AFileName: String);
var
  AStream: TMemoryStream;
begin
  if (FileExists(AFileName)) then
  begin
    AStream := TMemoryStream.Create;
    AStream.LoadFromFile(AFileName);
    Self.LoadPropertiesFromStream(AStream);
    AStream.Free;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ForceKeyProperties
  Description: This tries to force the reading of properties from a non-3.00 properties file
       Author: Mat Ballard
 Date created: 11/24/2002
Date modified: 11/24/2002 by Mat Ballard
      Purpose: Property streaming management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ForceKeyProperties(AStream: TMemoryStream);
var
  IsBinary: Boolean;
  i: Integer;
  AColor: Longint;
  BStream: TMemoryStream;
  Properties: TStringList;
begin
  AStream.Seek(0, soFromBeginning);
  Properties := TStringList.Create;
  Properties.LoadFromStream(AStream);
  IsBinary := FALSE;

{inspect first line for binary values:}
  for i := 1 to Length(Properties.Strings[0]) do
  begin
    if (Properties.Strings[0][i] < #30) then
    begin
      IsBinary := TRUE;
      break;
    end;
  end;
  Properties.Free;

{shall we convert to binary ?}
  if (not IsBinary) then
  begin
    BStream := TMemoryStream.Create;
    AStream.Seek(0, soFromBeginning);
    ObjectTextToResource(AStream, BStream{, sofText});
    //BStream.Seek(0, soFromBeginning); - not needed: done by CopyFrom
    AStream.CopyFrom(BStream, 0);
    BStream.Free;
  end;

  AStream.Seek(0, soFromBeginning);
  {Self.Width := GetIntegerProperty('Width', AStream);
  Self.Height := GetIntegerProperty('Height', AStream);}
(*
  Border.Left := Round(GetNumberProperty('Border.Left', AStream));
  Border.Top := Round(GetNumberProperty('Border.Top', AStream));
  Border.RightGap := Round(GetNumberProperty('Border.RightGap', AStream));
  Border.BottomGap := Round(GetNumberProperty('Border.BottomGap', AStream));
*)  
  FDisplayHistory := GetNumberProperty('DisplayHistory', AStream);
  DisplayMode := TDisplayMode(GetEnumValue(TypeInfo(TDisplayMode), GetStringProperty('DisplayMode', AStream)));
  FGrids.GridType := TGridType(GetEnumValue(TypeInfo(TGridType), GetStringProperty('Grid', AStream)));
  FGrids.Style := TPenStyle(GetEnumValue(TypeInfo(TPenStyle), GetStringProperty('GridStyle', AStream)));
  if IdentToColor(GetStringProperty('WallColor', AStream), AColor) then
    FGrids.WallColor := TColor(AColor);
  FPlotType := TPlotType(GetEnumValue(TypeInfo(TPlotType), GetStringProperty('PlotType', AStream)));
  FAxes[0].Min := GetNumberProperty('XAxis.Min', AStream);
  FAxes[0].Max := GetNumberProperty('XAxis.Max', AStream);
  FAxes[0].Title.Caption := GetStringProperty('XAxis.Title.Caption', AStream);
  FAxes[1].Min := GetNumberProperty('YAxis.Min', AStream);
  FAxes[1].Max := GetNumberProperty('YAxis.Max', AStream);
  AStream.Seek(0, soFromBeginning);
  if (FPlotType in [ptContour, pt3DContour]) then
  begin
    FContour.Detail := TContourDetail(GetEnumValue(TypeInfo(TContourDetail), GetStringProperty('ContourDetail', AStream)));
    FContour.Interval := GetNumberProperty('ContourInterval', AStream);
    FContour.Start := GetNumberProperty('ContourStart', AStream);
    FContour.WireFrame := GetBooleanProperty('ContourWireFrame', AStream);
  end;
  {if (FPlotType = ptMultiple) then
  begin
    FLink.No := Round(GetNumberProperty('Multiplicity', AStream));
    FLink.Join := GetStringProperty('MultiJoin', AStream);
  end;}
  if (FPlotType = ptPie) then
  begin
    FPieRowCount := Round(GetNumberProperty('PieRowCount', AStream));
    FAxes[0].Visible := FALSE;
    FAxes[1].Visible := FALSE;
  end;
  if (FPlotType in [ptLineContour, ptContour, pt3DContour]) then
  begin
    FContour.Detail := TContourDetail(GetEnumValue(TypeInfo(TContourDetail), GetStringProperty('ContourDetail', AStream)));
    FContour.Interval := GetNumberProperty('ContourInterval', AStream);
    FContour.WireFrame := GetBooleanProperty('ContourWireFrame', AStream);
  end;
  if (FPlotType in [pt3DContour, pt3DWire, pt3DColumn]) then
  begin
    FZLink := GetBooleanProperty('ZLink', AStream);
    Self.ZAngle := Round(GetNumberProperty('ZAngle', AStream));
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LoadPropertiesFromStream
  Description: Opens the properties of this instance of TPlot
       Author: Mat Ballard
 Date created: 11/24/2002
Date modified: 11/24/2002 by Mat Ballard
      Purpose: Saves the appearance of the Plot
 Known Issues:
      Comment: Note that if AFileName is blank, we use a name generated from
               the FileName property (in SetFileName).
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LoadPropertiesFromStream(AStream: TMemoryStream);
begin
  PaintLock;
  try
    mNameFrozen := TRUE;
{This is what causes multiple screen redraws. But why ?}
    AStream.ReadComponent(Self);
  except
    on EReadError do
    begin
      FInstructions := 'Problem reading properties';
      ForceKeyProperties(AStream);
      ZoomOutClick(Self);
    end;
  end;
  mNameFrozen := FALSE;
  PaintUnLock;
end;

  {function LookForInText(AStr, Props: String): String;
  var
    i, j, OldPos: Integer;
  begin
    OldPos := j;
    j := Misc.ForwardPos(AStr, Props, i);
    if (j > 0) then
    begin
      Result := Copy(Props, i, j-i+1);
    end
    else
    begin
      j := OldPos;
      Result := '0';
    end;
  end;}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SavePropertiesToFile
  Description: Saves the properties of this instance of TPlot
       Author: Mat Ballard
 Date created: 08/03/2000
Date modified: 08/03/2000 by Mat Ballard
      Purpose: Saves the appearance of the Plot
 Known Issues:
      Comment: Note that if AFileName is blank, we use a name generated from
               the FileName property (in SetFileName).
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SavePropertiesToFile(AFileName: String);
var
  FileStream: TFileStream;
begin
  FileStream := TFileStream.Create(AFileName, fmCreate or fmShareExclusive);
  FileStream.WriteComponent(Self);
  FileStream.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AppendToFile
  Description: Appends the data to FileName
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the GetData method of SeriesList, then the Append method of the new TFileList
 Known Issues: Needs work on GetData and testing
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AppendToFile;
var
  TheStream: TMemoryStreamEx;
begin
  if (FileExists(FFileName)) then
  begin
{create the FileList, an extension of TStringList:}
    TheStream := TMemoryStreamEx.Create;

{create the data in text format:}
    FSeriesList.AppendStream((soAsText in FSaveOptions), ',', TheStream);
{save it:}
    TheStream.AppendToFile(FFileName);

    TheStream.Free;
  end
  else
  begin
    Self.SaveClick(Self);
  end;
end;

{Copying data to the clipboard ----------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoHTMLHeaderRequest
  Description: Asks the user for their HTML header data
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: fires the OnHTMLHeaderRequest event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoHTMLHeaderRequest(TheHeader: TStringList);
begin
  if assigned(FOnHTMLHeaderRequest) then
    OnHTMLHeaderRequest(Self, TheHeader);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyHTML
  Description: Copies data as HTML to Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Creates a Header, fires the OnHTMLHeaderRequest event, then runs
               the DataAsHTMLTable method of SeriesList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyHTML(Format: Word);
var
  TheData: TStringList;
{$IFDEF MSWINDOWS}
  i: Integer;
  Size,
  LineLength: LongInt;
  pText,
  TextPtr: PChar;
  TextHandle: THandle;
{$ENDIF}
begin
  TheData := TStringList.Create;

  SaveToHTMLStringList('', TheData);

{$IFDEF MSWINDOWS}
{Calculate the size:}
  Size := 8;
  for i := 0 to TheData.Count-1 do
    Inc(Size, Length(TheData[i])+2);
{save it:}
  TextHandle := GlobalAlloc(GMEM_MOVEABLE, Size);
  TextPtr := GlobalLock(TextHandle);

  pText := TextPtr;
  for i := 0 to TheData.Count - 1 do
  begin
    LineLength := Length(TheData[i]);
    if LineLength <> 0 then
    begin
{for some unknown reason, this Move works !}
      System.Move(Pointer(TheData[i])^, pText^, LineLength);
      Inc(pText, LineLength);
    end;
    pText^ := #13;
    Inc(pText);
    pText^ := #10;
    Inc(pText);
  end;
{we need to use SetAsHandle because Format may be CF_HTML:}
  ClipBoard.SetAsHandle(Format, TextHandle);
  GlobalUnlock(TextHandle);
{$ENDIF} {MSWINDOWS}
{$IFDEF LINUX}
  Clipboard.AsText := TheData.Text;
{$ENDIF}
  TheData.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyText
  Description: Copies data as tab-delimited text to Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Creates a Header, fires the OnHeaderRequest event, then runs the GetData method of SeriesList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyText;
var
  TheStream: TMemoryStream;
{$IFDEF MSWINDOWS}
  TextPtr: Pointer;
  TextHandle: THandle;
{$ENDIF}
{$IFDEF LINUX}
  TheText: TStringList;
{$ENDIF}
begin
  TheStream := TMemoryStream.Create;
{create the data in TEXT format:}
  Self.SaveToStream(TheStream, TRUE, #9);

{save it:}
{$IFDEF MSWINDOWS}
  TextHandle := GlobalAlloc(GMEM_MOVEABLE, TheStream.Size+1);
  TextPtr := GlobalLock(TextHandle);
{for some unknown reason, this Move works !}
  System.Move(TheStream.Memory^, TextPtr^, TheStream.Size);
  ClipBoard.SetAsHandle(CF_TEXT, TextHandle);
  GlobalUnlock(TextHandle);
{$ENDIF}
{$IFDEF LINUX}
  TheText := TStringList.Create;
  TheText.LoadFromStream(TheStream);
  ClipBoard.AsText := TheText.Text;
  TheText.Free;
{$ENDIF}
  TheStream.Free;
end;

{Copying picture to the clipboard -------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyBitmap
  Description: Copies a picture of the graph as a bitmap to the Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over Bitmap then copies Bitmap
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyBitmap;
begin
  ClipBoard.Assign(Self.BitMap);
end;
{$ENDIF}

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetBitmap
  Description: Returns a picture of the graph as a bitmap
       Author: Mat Ballard
 Date created: 04/25/2001
Date modified: 04/25/2001 by Mat Ballard
      Purpose: Runs Draw method over Bitmap
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetBitmap: TBitmap;
begin
  Result := TBitMap.Create;
  Result.Height := Height;
  Result.Width := Width;

  Draw(Result.Canvas);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsBitmap
  Description: Saves a picture of the graph as a bitmap
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over Bitmap then saves Bitmap
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsBitmap(
  AFileName: String);
begin
  Self.Bitmap.SaveToFile(AFileName);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetDrawing
  Description: Returns a picture of the graph as a Windows Metafile to the Clipboard
       Author: Mat Ballard
 Date created: 05/16/2001
Date modified: 05/16/2001 by Mat Ballard
      Purpose: developer rendering image of TPlot
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF MSWINDOWS}
function TCustomPlot.GetDrawing: TMetafile;
var
  AMetafileCanvas: TMetafileCanvas;
begin
  Result := TMetafile.Create;
  {$IFDEF COMPILER3_UP}
  Result.Enhanced := TRUE;
  {$ENDIF}
  Result.Height := Height;
  Result.Width := Width;

  SetMetafileDescription;
{create the metafile canvas to draw on:}
  AMetafileCanvas :=
    TMetafileCanvas.CreateWithComment(Result, 0, FCreatedBy, FDescription);

{draw the graph on the metafile:}
  Draw(AMetafileCanvas);
  AMetafileCanvas.Free;
end;
{$ENDIF} {MSWINDOWS}

{$IFDEF LINUX}
function TCustomPlot.GetDrawing: TDrawing;
begin
  Result := TDrawing.Create;
  Result.Height := Height;
  Result.Width := Width;

  Result.Canvas.Start;
  try
    Draw(Result.Canvas);
  finally
    Result.Canvas.Stop;
  end;
end;
{$ENDIF} {LINUX}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyDrawing
  Description: Copies a picture of the graph as a Windows Metafile to the Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over AMetafileCanvas then copies Metafile
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyDrawing(Enhanced: Boolean);
{$IFDEF MSWINDOWS}
var
  AFormat: Word;
  AData: THandle;
  APalette: HPALETTE;
begin
  Self.Drawing.SaveToClipboardFormat(AFormat, AData, APalette);
  ClipBoard.SetAsHandle(AFormat, AData);
{$ENDIF}
{$IFDEF LINUX}
begin
  Clipboard.Assign(Self.Drawing);
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsDrawing
  Description: Saves a picture of the graph as a Windows Metafile
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over Metafile Canvas then saves Metafile
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsDrawing(
  AFileName: String);
begin
  Self.Drawing.SaveToFile(AFileName);
end;

{$IFDEF GIF}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetGIF
  Description: Saves a picture of the graph as a GIF file
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over ABitmap of AGifImage then saves AGifImage
 Known Issues: 1. Requires Anders Melander's TGifImage
               2. Package dependency problems means that it is easier to let end users add this functionality
 ------------------------------------------------------------------------------}
function TCustomPlot.GetGIF: TGIFImage;
var
  ABitMap: TBitMap;
begin
  ABitMap := TBitMap.Create;
  ABitMap.PixelFormat := pf24bit;
  ABitMap.Height := Height;
  ABitMap.Width := Width;

  Self.Draw(ABitMap.Canvas);

  Result := TGIFImage.Create;
  Result.Assign(ABitMap);
  ABitMap.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetGIFSize
  Description: Saves a picture of the graph at a specified size as a GIF file
       Author: Mat Ballard
 Date created: 09/25/2003
Date modified: 09/25/2003 by Mat Ballard
      Purpose: Runs Draw method over ABitmap of AGifImage then saves AGifImage
 Known Issues: 1. Requires Anders Melander's TGifImage
               2. Package dependency problems means that it is easier to let end users add this functionality
 ------------------------------------------------------------------------------}
function TCustomPlot.GetGIFSize(W, H: Integer): TGIFImage;
var
  ABitMap: TBitMap;
begin
  ABitMap := TBitMap.Create;
  ABitMap.PixelFormat := pf24bit;
  if (W > 0) and (H > 0) then
  begin
    ABitMap.Height := H;
    ABitMap.Width := W;
  end else begin
    ABitMap.Height := Height;
    ABitMap.Width := Width;
  end;

  Self.Draw(ABitMap.Canvas);

  Result := TGIFImage.Create;
  Result.Assign(ABitMap);
  ABitMap.Free;
end;

procedure TCustomPlot.SaveAsGIF(AFileName: String; W: Integer = -1; H: Integer = -1);
begin
  if (W > 0) and (H > 0) then
    with GetGIFSize(W, H) do SaveToFile(AFileName)
   else
    GIF.SaveToFile(AFileName);
end;
{$ENDIF}
{$ENDIF} {GUI CopyDrawing}

{$IFDEF GUI}
  {$IFDEF PNG}
    {$IFDEF MSWINDOWS}
function TCustomPlot.GetPNG: TPngimage;
var
  ABitmap: TBitmap;
begin
  ABitmap := TBitmap.Create;
{We need 24 bit otherwise the 8 bit PNG renders white as black !}
  ABitmap.PixelFormat := pf24bit;
  ABitMap.Height := Height;
  ABitMap.Width := Width;

  Draw(ABitmap.Canvas);

  Result := TPngimage.Create;
  Result.Title := FTitle.Caption;
  Result.Author := FCreatedBy;
  Result.Description := FDescription;
  Result.Software := ExtractFileName(Application.ExeName);
  Result.CopyFromBmp(ABitmap);
  ABitmap.Free;
end;

function TCustomPlot.GetPNGSize(W, H: Integer): TPngimage;
var
  ABitmap: TBitmap;
begin
  ABitmap := TBitmap.Create;
{We need 24 bit otherwise the 8 bit PNG renders white as black !}
  ABitmap.PixelFormat := pf24bit;
  if (W > 0) and (H > 0) then
  begin
    ABitMap.Height := H;
    ABitMap.Width := W;
  end else begin
    ABitMap.Height := Height;
    ABitMap.Width := Width;
  end;

  Draw(ABitmap.Canvas);

  Result := TPngimage.Create;
  Result.Title := FTitle.Caption;
  Result.Author := FCreatedBy;
  Result.Description := FDescription;
  Result.Software := ExtractFileName(Application.ExeName);
  Result.CopyFromBmp(ABitmap);
  ABitmap.Free;
end;
    {$ENDIF} {MSWINDOWS}
    {$IFDEF LINUX}
function TCustomPlot.GetPNG: TBitmap;
begin
  Result := TBitMap.Create;
  Result.Format := 'PNG';
  Result.Height := Height;
  Result.Width := Width;

  Draw(Result.Canvas);
end;

function TCustomPlot.GetPNGSize(W, H: Integer): TBitmap;
begin
  Result := TBitMap.Create;
  Result.Format := 'PNG';
  if (W > 0 and H > 0) then
  begin
    Result.Height := H;
    Result.Width := W;
  end else begin
    Result.Height := Height;
    Result.Width := Width;
  end;

  Draw(Result.Canvas);
end;
    {$ENDIF} {LINUX}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsPNG
  Description: Saves a picture of the graph as a PNG file
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over ABitmap of APngImage then saves APngImage
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsPng(AFileName: String; W: Integer = -1; H: Integer = -1);
begin
  with GetPngSize(W, H) do SaveToFile(AFileName);
end;
  {$ENDIF} {PNG}
{$ELSE}
function TCustomPlot.GetPNG: Pointer;
var
  im: PgdImage;
begin
  im := gdImageCreate(Width, Height);

  gdImageDestroy(im);
  Result := nil;
end;

function TCustomPlot.GetPNGSize(W, H: Integer): Pointer;
var
  im: PgdImage;
begin
  if (W > 0 and H > 0) then
  begin
    im := gdImageCreate(W, H);
  end else begin
    im := gdImageCreate(Width, Height);
  end;

  gdImageDestroy(im);
  Result := nil;
end;
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsPNG
  Description: Saves a picture of the graph as a PNG file
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over gd library then saves APngImage
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsPng(AFileName: String; W: Integer = -1; H: Integer = -1);
var
  im: PgdImage;
  AFile: FILE;
begin
  im := gdImageCreate(Width, Height);
  AssignFile(AFile, AFileName);
  gdImagePng(im, @AFile);
  CloseFile(AFile);
  gdImageDestroy(im);
end;
{$ENDIF} {GUI GetPNG}

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetAsNormalClick
  Description: Defines the current view == Mins and Maxes of axes, as the Normal view.
       Author: Mat Ballard
 Date created: 08/12/2000
Date modified: 08/12/2000 by Mat Ballard
      Purpose: Zoom == Axis Min/Max management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetAsNormalClick(Sender: TObject);
var
  i: Integer;
  pTheAxis: TAxis;
begin
  for i := 0 to FAxes.Count-1 do
  begin
    pTheAxis := FAxes[i];
    pTheAxis.ZoomIntercept := pTheAxis.Intercept;
    pTheAxis.ZoomMin := pTheAxis.Min;
    pTheAxis.ZoomMax := pTheAxis.Max;
  end;
{$IFDEF GUI}  // mnuView, mnuNormal
  if Assigned(FPlotActionList) then
  begin
    i := TPlotActionList(FPlotActionList).GetIndexFromMenuPosition(mnuView, Ord(mnuNormal));
    TAction(FPlotActionList.Actions[i]).Enabled := TRUE;
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NormalViewClick
  Description: Zooms to the Normal view
       Author: Mat Ballard
 Date created: 08/12/2000
Date modified: 08/12/2000 by Mat Ballard
      Purpose: Zoom == Axis Min/Max management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NormalViewClick(Sender: TObject);
var
  i: Integer;
  pTheAxis: TAxis;
begin
  for i := 0 to FAxes.Count-1 do
  begin
    pTheAxis := FAxes[i];
    if (pTheAxis.ZoomMin < pTheAxis.ZoomMax) then
    begin
      pTheAxis.SetMinMax(pTheAxis.ZoomMin, pTheAxis.ZoomMax);
      pTheAxis.Intercept := pTheAxis.ZoomIntercept;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ManualZoomClick
  Description: Manually Zooms In
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the ZoomForm and adjusts Axes accordingly
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ManualZoomClick(Sender: TObject);
var
  ZoomForm: TZoomForm;
begin
  ZoomForm := TZoomForm.Create(Self);
  ZoomForm.ThePlot := Self;
  ZoomForm.XMinNEdit.AsReal := FAxes[0].Min;
  ZoomForm.XMaxNEdit.AsReal := FAxes[0].Max;
  ZoomForm.YMinNEdit.AsReal := FAxes[1].Min;
  ZoomForm.YMaxNEdit.AsReal := FAxes[1].Max;

  ZoomForm.HelpFile := FHelpFile;

  if (ZoomForm.ShowModal = mrOK) then
    ApplyZoom(ZoomForm);

  ZoomForm.Free;
  Invalidate;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ApplyZoom
  Description: Manually Zooms In
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: adjusts Axes accordingly
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ApplyZoom(Sender: TObject);
var
  i,
  PixelYMin,
  PixelYMax: Integer;
  pTheYAxis: TAxis;
begin
  FAxes[0].Min := TZoomForm(Sender).XMinNEdit.AsReal;
  FAxes[0].Max := TZoomForm(Sender).XMaxNEdit.AsReal;
  FAxes[1].Min := TZoomForm(Sender).YMinNEdit.AsReal;
  FAxes[1].Max := TZoomForm(Sender).YMaxNEdit.AsReal;
  PixelYMin := FAxes[1].FofY(FAxes[1].Min);
  PixelYMax := FAxes[1].FofY(FAxes[1].Max);
  for i := 2 to FAxes.Count-1 do
  begin
    pTheYAxis := TAxis(FAxes[i]);
    pTheYAxis.SetMinMax(pTheYAxis.YofF(PixelYMin), pTheYAxis.YofF(PixelYMax));
  end;
  Invalidate;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyHTMLClick
  Description: copys the data as HTML
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: ... in CF_TEXT format
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyHTMLClick(Sender: TObject);
begin
  CopyHTML(CF_TEXT);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetIndicesFromTag
  Description: Gets the i and j indices from the Tag
       Author: Mat Ballard
 Date created: 05/25/2000
Date modified: 05/25/2000 by Mat Ballard
      Purpose: interfacing with external components
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetIndicesFromTag(TheTag: Integer; var i, j: Integer): Boolean;
var
  MenuIndex: Integer;
begin
{TPlotToolBar throws some strange tags at this function:
  if (TheTag <= TAG_BASE) then raise
    ERangeError.CreateFmt('GetIndicesFromTag: ' + sInvalidTag, [Tag]);}

  Result := FALSE;
  MenuIndex := TheTag - TAG_BASE;
  if ((0 <= MenuIndex) and (MenuIndex < 200)) then
  begin
    if (MenuIndex <= Ord(High(TFileMenus))) then
    begin
      i := 0;
      j := MenuIndex;
      Result := TRUE;
    end
    else
    begin
      MenuIndex := MenuIndex - Ord(High(TFileMenus)) - 1;
      if (MenuIndex <= Ord(High(TEditMenus))) then
      begin
        i := 1;
        j := MenuIndex;
        Result := TRUE;
      end
      else
      begin
        MenuIndex := MenuIndex - Ord(High(TEditMenus)) - 1;
        if (MenuIndex <= Ord(High(TViewMenus))) then
        begin
          i := 2;
          j := MenuIndex;
          Result := TRUE;
        end
        else
        begin
          MenuIndex := MenuIndex - Ord(High(TViewMenus)) - 1;
          if (MenuIndex <= Ord(High(TCalcMenus))) then
          begin
            i := 3;
            j := MenuIndex;
            Result := TRUE;
          end
{$IFDEF FINANCE}
          else
          begin
            MenuIndex := MenuIndex - Ord(High(TCalcMenus)) - 1;
            if (MenuIndex <= Ord(High(TFinanceMenus))) then
            begin
              i := 4;
              j := MenuIndex;
              Result := TRUE;
            end
{$ENDIF}
          else
          begin
            Result := FALSE;
            EComponentError.CreateFmt('GetIndicesFromTag: cannot resolve tag %d', [TheTag]);
          end;
        end;
      end;
    end;
  end;
  if (Result) then
{$IFDEF DELPHI3_UP}
//    Assert(FPlotPopUpMenu.Items[i].Items[j].Tag = TheTag, 'TCustomPlot.GetIndicesFromTag: Tags do not match !');
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DisplaceClick
  Description: runs the "Displacement" dialog box
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Displace" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DisplaceClick(Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FClickedSeries.Displace(FHelpFile);
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZoomInClick
  Description: Zooms in using the mouse
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 05/30/2001 by Mat Ballard
      Purpose: responds to "Zoom In" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZoomInClick(Sender: TObject);
var
  i: Integer;
  pYAxis: TAxis;
begin
  if ((FScreenJob = sjRightDrag) or
      (FScreenJob = sjZoomIn)) then
  begin
    FAxes[0].Min := FAxes[0].XofF(Selection.Left);
    FAxes[0].Max := FAxes[0].XofF(Selection.Right);
    for i := 1 to FAxes.Count-1 do
    begin
      pYAxis := TAxis(FAxes[i]);
      if (pYAxis.AxisType >= atPrimaryY) then
      begin
        pYAxis.SetMinMax(pYAxis.YofF(Selection.Bottom), pYAxis.YofF(Selection.Top));
      end;
    end;
    CreatePageButtons;
    ZeroScreenStuff;
  end
  else
  begin
    FInstructions := 'Click and drag over the region you want to zoom in on';
    FScreenJob := sjZoomIn;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DifferentiateClick
  Description: Replaces the selected Axis with its differential
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Differential" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DifferentiateClick(Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FClickedSeries.Differentiate;
    ZoomOutClick(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.IntegrateClick
  Description: Replaces the selected Axis with its differential
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Integrate" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.IntegrateClick(Sender: TObject);
begin
  if (GetSeriesFromUser <> nil) then
  begin
    FClickedSeries.Integrate;
    ZoomOutClick(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.IntegralClick
  Description: calculates the integral of the selected Series over a selected range
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 05/30/2001 by Mat Ballard
      Purpose: responds to "Integral" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.IntegralClick(Sender: TObject);
var
  Msg: String;
  Sum,
  TheLeft,
  TheRight: Single;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjIntegral)) then
    begin
      TheLeft := FAxes[0].XofF(Selection.Left);
      TheRight := FAxes[0].XofF(Selection.Right);
      Sum := FClickedSeries.Integral(TheLeft, TheRight);
      Msg := 'The integral of ' + FClickedSeries.Name + ' from ' + CRLF;
      Msg := Msg + FAxes[0].LabelToStrF(TheLeft) + ' to ' +
        FAxes[0].LabelToStrF(TheRight);
      if (Length(FAxes[0].Title.Units) > 0) then
        Msg := Msg + ' ' + FAxes[0].Title.Units;
      Msg := Msg + ' is ' + CRLF +
        Format('%g', [Sum]);
      if ((Length(FAxes[0].Title.Units) > 0) and
          (Length(FClickedSeries.YAxis.Title.Units) > 0)) then
      begin
        Msg := Msg + ' (' + FClickedSeries.YAxis.Title.Units + '.' +
          FAxes[0].Title.Units + ')';
      end
      else if (Length(FAxes[0].Title.Units) > 0) then
      begin
        Msg := Msg + ' (' + FAxes[0].Title.Units + ')';
      end
      else if (Length(FClickedSeries.YAxis.Title.Units) > 0) then
      begin
        Msg := Msg + ' (' + FClickedSeries.YAxis.Title.Units + ')';
      end;
      ShowMessage(Msg);
      ClipBoard.AsText := Msg;
      ZeroScreenStuff;
    end
    else
    begin
      FScreenJob := sjIntegral;
      FInstructions := 'Click and Drag over the (X-) region to calculate the Integral';
      DoStyleChange(Self);
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetClickAndDragDelay
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ClickAndDragDelay Property
 Return value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetClickAndDragDelay: Integer;
begin
  GetClickAndDragDelay := MouseTimer.Interval;
end;
{$ENDIF} {GUI SetAsNormalClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZoomOutClick
  Description: Zooms out after Zooming In
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Resets the axes Min and Max values to those of the Axis
 Known Issues: required even if GUI is not defined
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZoomOutClick(Sender: TObject);
var
  i: Integer;
  aSeries: TSeries;
begin
  FAxes[0].AutoScale := TRUE;
  FAxes[0].AutoTick := TRUE;
  FAxes[0].AutoZero := TRUE;

  for i := 1 to FSeriesList.Count-1 do
  begin
    aSeries := FSeriesList[i];
    if ((aSeries.Visible) and (aSeries.YAxis.AxisType >= atPrimaryY)) then
    begin
      aSeries.YAxis.AutoScale := TRUE;
      aSeries.YAxis.AutoTick := TRUE;
      aSeries.YAxis.AutoZero := TRUE;
    end;
  end;

  case FPlotType of
    ptColumn:
      begin
        for i := 1 to FAxes.Count-1 do
          TAxis(FAxes[i]).Min := 0;
        for i := 0 to FSeriesList.Count-1 do
        begin
          aSeries := FSeriesList[i];
          if (aSeries.Visible) then
            if (aSeries.YAxis.Max < Round(aSeries.YMax + 0.999)) then
              aSeries.YAxis.Max := Round(aSeries.YMax + 0.999);
        end;
      end;

    ptStack:
      begin
        YAxis.Min := 0;
        if (FSeriesList.TotalNoPts > 0) then
          YAxis.Max := Round(FSeriesList.Count * FSeriesList.Ymax + 0.999);
      end;

    ptNormStack:
      begin
        for i := 1 to FAxes.Count-1 do
        begin
          TAxis(FAxes[i]).Min := 0;
          TAxis(FAxes[i]).Max := 100;
        end;
      end;

    ptPolar:
      begin
        FAxes[0].Min := Round(FSeriesList.Xmin - 0.999);
        FAxes[0].Max := Round(FSeriesList.Xmax + 0.999);
        FAxes[1].Min := Round(FSeriesList.Ymin - 0.999);
        FAxes[1].Max := Round(FSeriesList.Ymax + 0.999);
        if (FAxes[0].Min > -FAxes[0].Max) then
          FAxes[0].Min := -FAxes[0].Max;
        if (FAxes[0].Max < -FAxes[0].Min) then
          FAxes[0].Max := -FAxes[0].Min;
        if (FAxes[1].Min > -FAxes[1].Max) then
          FAxes[1].Min := -FAxes[1].Max;
        if (FAxes[1].Max < -FAxes[1].Min) then
          FAxes[1].Max := -FAxes[1].Min;
        FAxes[0].Intercept := 0;
        FAxes[1].Intercept := 0;
      end;

    ptLineContour:
      begin
        FContour.Start := FSeriesList.Ymin;
        FContour.Interval := (FSeriesList.Ymax - FContour.Start) / 10;
      end;

    pt3DContour, pt3DWire, pt3DColumn:
      begin
        ZAxis.SetMinMax(FSeriesList.Zmin, FSeriesList.Zmax);
      end;
  end; {case}
  //FAxes[0].Intercept := FAxes[1].Min;

{$IFDEF GUI}
  DestroyPageButtons;
{$ENDIF}
  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZeroScreenStuff
  Description: cleans up after a menu event handler - XXXClick
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: mouse management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZeroScreenStuff;
begin
  mZeroStuff := FALSE;
  FScreenJob := sjNone;
  FClickedObjectType := soNone;
  pClickedObject := nil;
  SecondClickedObjectType := soNone;
  pSecondClickedObject := nil;
  TheSeries := -1;
  FClickedSeries := nil;
  FOnSelection := nil;
  FOnDualSelection := nil;
{$IFDEF GUI}
  Screen.Cursor := crDefault;
  MouseStart.x := -100;
  MouseStart.y := -100;
{$ENDIF}
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.HighsClick
  Description: Calculates the Highs (peaks) and/or Lows (troughs) of a series
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: data processing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.HighsClick(Sender: TObject);
var
  OptionsForm: TOptions3Form;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    OptionsForm := TOptions3Form.Create(nil);
    OptionsForm.Caption := 'Highs and Lows';
    OptionsForm.QuestionLabel.Caption := 'What do you want to do ?';
    OptionsForm.Add('Hide Highs and Lows');
    OptionsForm.Add('Show Highs');
    OptionsForm.Add('Show Lows');
    OptionsForm.Add('Show Highs and Lows');

    if (OptionsForm.ShowModal = mrOK) then
    begin
      case OptionsForm.ItemIndex of
        0: FClickedSeries.ClearHighsLows;
        1:
          begin
            FClickedSeries.FindHighsLows(0, FClickedSeries.NoPts, 5);
            FClickedSeries.HighLow := FClickedSeries.HighLow + [hlHigh];
          end;
        2:
          begin
            FClickedSeries.FindHighsLows(0, FClickedSeries.NoPts, 5);
            FClickedSeries.HighLow := FClickedSeries.HighLow + [hlLow];
          end;
        3:
          begin
            FClickedSeries.FindHighsLows(0, FClickedSeries.NoPts, 5);
            FClickedSeries.HighLow := FClickedSeries.HighLow + [hlLow, hlHigh];
          end;
      end;
    end;

    OptionsForm.Free;

    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MovingAverageClick
  Description: Calculates and displays the Moving Average of a series
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: data smoothing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MovingAverageClick(Sender: TObject);
var
  Span: Integer;
  SpanStr: String;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    SpanStr := '10';
    if (InputQuery('Calculation of the Moving Average', 'Enter the number of points to average over', SpanStr)) then
    begin
      try
        Span := StrToInt(SpanStr);
        FClickedSeries.MovingAverage(Span);
      except
        ShowMessage('Moving Average Failed !');
      end;
    end;
    DoStyleChange(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AverageClick
  Description: Calculates the Average of a series over a range
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 05/30/2001 by Mat Ballard
      Purpose: data processing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AverageClick(Sender: TObject);
var
  Msg: String;
  Sum,
  TheLeft,
  TheRight: Single;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjAverage)) then
    begin
      TheLeft := FAxes[0].XofF(Selection.Left);
      TheRight := FAxes[0].XofF(Selection.Right);
      Sum := FClickedSeries.Average(TheLeft, TheRight);
      Msg := 'The Average of ' + FClickedSeries.Name + ' from ' + CRLF;
      Msg := Msg + FAxes[0].LabelToStrF(TheLeft) + ' to ' +
        FAxes[0].LabelToStrF(TheRight);
      if (Length(FAxes[0].Title.Units) > 0) then
        Msg := Msg + ' ' + FAxes[0].Title.Units;
      Msg := Msg + ' is ' + CRLF +
        Format('%g', [Sum]);
      if (Length(FClickedSeries.YAxis.Title.Units) > 0) then
      begin
        Msg := Msg + ' (' + FClickedSeries.YAxis.Title.Units + ')';
      end;
      ShowMessage(Msg);
      ClipBoard.AsText := Msg;
      ZeroScreenStuff;
    end
    else
    begin
      FScreenJob := sjAverage;
      FInstructions := 'Click and Drag over the (X-) region to calculate the Average';
      DoStyleChange(Self);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LinearizeClick
  Description: Linearizes Y data of a series over a range
       Author: Mat Ballard
 Date created: 05/30/2001
Date modified: 05/30/2001 by Mat Ballard
      Purpose: data processing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LinearizeClick(Sender: TObject);
var
  TheLeft,
  TheRight: Single;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjLinearize)) then
    begin
      TheLeft := FAxes[0].XofF(Selection.Left);
      TheRight := FAxes[0].XofF(Selection.Right);
      FClickedSeries.Linearize(TheLeft, TheRight);
      ZeroScreenStuff;
      Invalidate;
    end
    else
    begin
      FScreenJob := sjLinearize;
      FInstructions := 'Click and Drag over the (X-) region that you want Linearized';
      DoStyleChange(Self);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZeroClick
  Description: Calculates the Zero of a series over a range
       Author: Mat Ballard
 Date created: 05/30/2001
Date modified: 05/30/2001 by Mat Ballard
      Purpose: data processing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZeroClick(Sender: TObject);
var
  TheLeft,
  TheRight: Single;
begin
  if (GetSeriesFromUser <> nil) then
  begin
    if ((FScreenJob = sjRightDrag) or
        (FScreenJob = sjZero)) then
    begin
      TheLeft := FAxes[0].XofF(Selection.Left);
      TheRight := FAxes[0].XofF(Selection.Right);
      FClickedSeries.Zero(TheLeft, TheRight);
      ZeroScreenStuff;
      Invalidate;
    end
    else
    begin
      FScreenJob := sjZero;
      FInstructions := 'Click and Drag over the (X-) region that you want to set to Zero';
      DoStyleChange(Self);
    end;
  end;
end;
{$ENDIF} {GUI HighsClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NewYAxisClick
  Description: Adds a new axis, based on either the selected or last axis.
       Author: Mat Ballard
 Date created: 06/12/2000
Date modified: 06/12/2000 by Mat Ballard
      Purpose: axis management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NewYAxisClick(Sender: TObject);
//var
  //TheTemplate: Integer;
  //NewAxis: TAxis;
begin
{has the user already selected an object ?}
{see GetTheClickedObject and ZeroScreenStuff
  TheTemplate := FAxes.Count - 1;
  if (Assigned(pClickedObject)) then
    if (TObjectType(TRectangle(pClickedObject).Tag) in [soXAxis, soYAxis, soZAxis]) then
      TheTemplate := pClickedObject.Index;  }

  //NewAxis :=
  FAxes.Add;
  //NewAxis.Assign(TAxis(FAxes[TheTemplate]));

(*{Move the new axis to the right:}
  if (FAxes.Count = 3) then
  begin
{this new one is a secondary axis:}
    NewAxis.AxisType := atSecondaryY;
    NewAxis.Intercept := FAxes[0].Max;
  end
  else
  begin
    NewAxis.AxisType := atTertiaryY;
{place the new axis half-way between the last and the right hand side:}
    NewAxis.Intercept := FAxes[0].XofF((NewAxis.MidX + Width) div 2);
  end; *)

{if this over-runs the panel width, then wrap around:
  if (NewAxis.Left > Width) then
    NewAxis.Left := NewAxis.Left - Width;}

{... and rename it:}
  //NewAxis.Name := 'Copy  Of  ' + NewAxis.Name;
  //NewAxis.Title.Caption := 'Copy  Of  ' + NewAxis.Title.Caption;

{re-initialize:}
  ZeroScreenStuff;
{we need to call re-size to force a SetAxisDimensions:}
  DoStyleChange(Self);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteAxisClick
  Description: Deletes an axis
       Author: Mat Ballard
 Date created: 06/12/2000
Date modified: 06/12/2000 by Mat Ballard
      Purpose: axis management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteAxisClick(Sender: TObject);
var
  pAxis: TAxis;
begin
  pAxis := GetAxisFromUser(2);
  if (Assigned(pAxis)) then
    DeleteAxis(pAxis, TRUE);
end;
{$ENDIF} {GUI DeleteAxisClick}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteAxis
  Description: Deletes the selected axis
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: axis management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteAxis(AnAxis: TAxis; Confirm: Boolean);
begin
{$IFDEF GUI}
  if (Confirm) then
  begin
    Confirm := (mrNo =
      MessageDlg(
  {$IFDEF LINUX}
      'Delete Secondary Axis',
  {$ENDIF}
        'Are you sure you want to delete ' + AnAxis.Title.Caption,
        mtWarning,
        [mbYes, mbNo],
        0));
  end;

  if (not Confirm) then
  begin
    FAxes.Delete(AnAxis.Index);
    DoStyleChange(Self);
  end;
{$ELSE}
  FAxes.Delete(AnAxis.Index);
  DoStyleChange(Self);
{$ENDIF}
end;

{$IFDEF GUI}
  {$IFDEF FINANCE}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ADLClick
  Description: This runs the ADL financial calculation
       Author: Mat Ballard
 Date created: 07/25/2001
Date modified: 07/25/2001 by Mat Ballard
      Purpose: standard Financial menu item event handler
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ADLClick(Sender: TObject);
begin
  ShowMessage('Sorry - Financial calculations are not yet enabled !');
end;
  {$ENDIF} {FINANCE}
{$ENDIF} {GUI}

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetNoSeries
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 25/25/2000
Date modified: 25/25/2000 by Mat Ballard
      Purpose: gets the number of Series
 Return Value: integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetNoSeries: Word;
begin
  GetNoSeries := FSeriesList.Count;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetNoYAxes
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 06/25/2000
Date modified: 08/25/2002 by Mat Ballard
      Purpose: gets the number of Y Axes
 Return Value: integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetNoYAxes: Integer;
var
  i: Integer;
begin
  Result := 0;
  for i := 1 to FAxes.Count-1 do
    if (FAxes[i].AxisType >= atPrimaryY) then
      Inc(Result);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetNoYAxes
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the number of Y Axes
 Known Issues: it is a fairly brutal way of setting the number of Y Axes
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetNoYAxes(Value: Integer);
var
  i: Integer;
  Change: Integer;
begin
  if (Value < 1) then exit;

  if (Value > 15) then raise
    EComponentError.CreateFmt('You must be joking ! I only support 15 Y Axes - not %d !', [Value]);

  Change := Value - GetNoYAxes;
  if (Change < 0) then
  begin
    i := FAxes.Count-1;
    while (Change < 0) do
    begin
      if (FAxes[i].AxisType >= atPrimaryY) then
      begin // blow this one away
        FAxes.Delete(FAxes[i].Index);
        Dec(i);
        Inc(Change);
      end
      else // skip this one
        Dec(i);
    end;
  end
  else if (Change > 0) then
  begin {add some axes:}
    for i := FAxes.Count-1 to Value-1 do
      NewYAxisClick(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.RemoveZAxis
  Description: Does what it says
       Author: Mat Ballard
 Date created: 12/02/2002
Date modified: 12/02/2002 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.RemoveZAxis;
begin
  ZAxis.OnChange := nil;
  FAxes.Delete(ZAxis.Index);
  ZAxis := nil;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetPlotType
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/20/2000
Date modified: 09/16/2002 by Mat Ballard
      Purpose: sets the PlotType property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetPlotType(Value: TPlotType);
var
  i: Integer;
  Temp: Single;

  procedure AddZAxis;
  var
    DefAngle: TDegrees;
    AnAxis: TAxis;
  begin
    DefAngle := Self.ZAngle;
    AnAxis := FAxes.Add;
    AnAxis.AxisType := atZ;
    //FAxes.ZAxis := ZAxis;
    AnAxis.Left := FAxes[0].Left;
    AnAxis.Top := FAxes[1].Bottom;
    AnAxis.Length := FAxes[0].Width div 3;
    AnAxis.Title.Units := 'cm';
    AnAxis.Angle := DefAngle;
  end;

begin
  if (FPlotType = Value) then exit;

  PaintLock;

  if not (sLoaded in FStatus) then Loaded;

  if (Assigned(ZAxis)) then
    if (Value < pt3DContour) then
      RemoveZAxis;

  FPlotType := Value;

{clean up axes:}
  for i := 0 to FAxes.Count-1 do
  begin
    FAxes[i].FireEvents := FALSE;
    FAxes[i].Visible := TRUE;
  end;

{clear previous Series pen settings:}
  for i := 0 to FSeriesList.Count-1 do
  begin
    Series[i].Pen.Width := 1;
    Series[i].Symbol := syNone;
  end;

  FLegend.Visible := TRUE;

  if (csDesigning in ComponentState) then
    FSeriesList.ClearSeries;

  case FPlotType of
    ptXY:
      if (csDesigning in ComponentState) then
        NoSeries := 2;

    ptColumn:
      begin
        if (csDesigning in ComponentState) then
          NoSeries := 2;
        YAxis.Min := 0;
      end;

    ptStack:
      begin
        if (csDesigning in ComponentState) then
          NoSeries := 2;
        YAxis.Min := 0;
        if (FSeriesList.TotalNoPts > 0) then
          YAxis.Max := Round(FSeriesList.Count * FSeriesList.Ymax + 0.999);
      end;

    ptNormStack:
      begin
        if (csDesigning in ComponentState) then
          NoSeries := 2;
        YAxis.SetMinMax(0, 100);
      end;

    ptPie:
      begin
        if (csDesigning in ComponentState) then
          NoSeries := 2;
        for i := 0 to FAxes.Count-1 do
          FAxes[i].Visible := FALSE;
      end;

    ptPolar:
      begin
        if (csDesigning in ComponentState) then
          NoSeries := 1;
        if (FAxes[0].Min > -FAxes[0].Max) then
          FAxes[0].Min := -FAxes[0].Max;
        if (FAxes[0].Max < -FAxes[0].Min) then
          FAxes[0].Max := -FAxes[0].Min;
        if (FAxes[1].Min > -FAxes[1].Max) then
          FAxes[1].Min := -FAxes[1].Max;
        if (FAxes[1].Max < -FAxes[1].Min) then
          FAxes[1].Max := -FAxes[1].Min;
        FAxes[0].Intercept := 0;
        FAxes[1].Intercept := 0;
        FAxes[1].Title.Visible := FALSE;
      end;

    ptLineContour, ptContour:
      begin
        if (csDesigning in ComponentState) then
          NoSeries := 10;
        FLegend.Visible := FALSE;
      end;

    pt3DContour, pt3DWire, pt3DColumn:
      begin
        if (ZAxis = nil) then
          AddZAxis;
        if (csDesigning in ComponentState) then
          NoSeries := 10;
        FLegend.Visible := FALSE;
      end;
  end;

  if (csDesigning in ComponentState) then
    MakeDummyData(20);

{re-scale the primary X Axis:}
  if (FSeriesList.TotalNoPts > 0) then
  begin
    if (FSeriesList.Xmin >= 0) then
      Temp := 0
    else
      Temp := Round(FSeriesList.Xmin - 0.9);
    XAxis.SetMinMax(Temp, Round(FSeriesList.Xmax + 0.9));
  end;

{re-scale the primary Y Axis:}
  if (FSeriesList.TotalNoPts > 0) then
  begin
    if (FSeriesList.Ymin >= 0) then
      Temp := 0
    else
      Temp := Round(FSeriesList.Ymin - 0.9);
    YAxis.SetMinMax(Temp, Round(FSeriesList.Ymax + 0.9));
  end;

{deal with axes:}
  //SetAxisDimensions;
  for i := 0 to FAxes.Count-1 do
  begin
    FAxes[i].Intercept := 0;
    FAxes[i].FireEvents := TRUE;
  end;

  PaintUnLock;

  DoStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetPolarRange
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/20/2000
Date modified: 01/16/2001 by Mat Ballard
      Purpose: sets the PolarRange property
 Known Issues: under development
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetPolarRange(Value: Single);
begin
  if (FPolarRange <> Value) then
  begin
    FPolarRange := Value;
    DoStyleChange(Self);
  end;
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetPlotActionList
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 05/05/2003
Date modified: 05/05/2003 by Mat Ballard
      Purpose: sets the PlotActionList property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetPlotActionList(Value: TActionList);
begin
  FPlotActionList := Value;
  if (not (csLoading in ComponentState)) then
    SetRightDragItems;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetRightDragItems
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 05/05/2003
Date modified: 05/05/2003 by Mat Ballard
      Purpose: sets the Right Drag Popup Menu Items 
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetRightDragItems;
var
  i: Integer;
const
  RightDragItems: array[0..8] of Integer = (
    51, {mnuView, mnuZoomIn}
    27, {mnuEdit, mnuLinearize}
    28, {mnuEdit, mnuZero}
    57, {mnuCalc, mnuCalcAverage}
    60, {mnuCalc, mnuContractSeries}
    61, {mnuCalc, mnuContractAllSeries}
    70, {mnuCalc, mnuIntegral}
    72, {mnuCalc, mnuLineOfBestFit}
    73); {mnuCalc, mnuTwoRegionLineOfBestFit}
begin
  if (Assigned(FPlotActionList) and (FPlotActionList.ActionCount > 73)) then
  begin
    for i := 0 to 8 do
      FDragPopUpMenu.Items[i].Action := FPlotActionList.Actions[RightDragItems[i]];
  end;
end;
{$ENDIF} {SetPopupOptions}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Clear
  Description: Saves any changed files (at user request) and then clears the SeriesList.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management: wraps TSeriesList.ClearSeries
 Known Issues: it is a fairly brutal way of setting the number of Y Axes
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Clear(Cancellable: Boolean);
{$IFDEF GUI}
var
  TheResult: Integer;
  TheMessage: String;
  {$IFDEF MSWINDOWS}
  pMessage: array[0..255] of Char;
  {$ENDIF}
{$ENDIF}
begin
  if (csDesigning in ComponentState) then exit;

  if (FSeriesList.Count > 0) then
  begin
{$IFDEF GUI}
    if ((FFileSavePrompt) and (FSeriesList.DataChanged)) then
    begin
      if (Length(FFileName) > 0) then
        TheMessage := ExtractFileName(FFileName)
       else
        TheMessage := 'this file';
       TheMessage := 'Save  ' + TheMessage + '  before closing it ?';
{NB: MessageDlg provokes an access violation because the Window may have gone:}
  {$IFDEF MSWINDOWS}
      StrPCopy(pMessage, TheMessage);
      if (Cancellable) then
        TheResult := {Windows.}MessageBox(0, pMessage, pChar('File has Changed'), MB_YESNOCANCEL + MB_ICONQUESTION)
       else
        TheResult := {Windows.}MessageBox(0, pMessage, pChar('File has Changed'), MB_YESNO + MB_ICONQUESTION);
  {$ENDIF}
  {$IFDEF LINUX}
      if (Cancellable) then
        TheResult := MessageDlg('Save File',
          TheMessage,
          mtWarning,
          [mbYes, mbNo, mbCancel],
          0)
       else
        TheResult := MessageDlg('Save File',
          TheMessage,
          mtWarning,
          [mbYes, mbNo],
          0);
  {$ENDIF} {LINUX}
      case TheResult of
        mrYes: SaveClick(Self);
        mrNo: ;
        mrCancel: Exit;
      end;
    end;
{$ENDIF} {GUI}
    FSeriesList.ClearSeries; // which also blows away overlays
    FFirstOverlay := -1;
    DoFileClose(FFileName);
  end;
  FFileName := '';
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoStyleChange
  Description: informs the user of a change in the plot that requires a repaint
       Author: Mat Ballard
 Date created: 02/20/2001
Date modified: 02/20/2001 by Mat Ballard
      Purpose: Fires the OnStyleChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoStyleChange(Sender: TObject);
begin
  if ((not (sUpdating in FStatus)) and (mPaintLock = 0)) then
  begin
    //SetAxisDimensions;

    if assigned(FOnStyleChange) then OnStyleChange(Sender);
    {$IFDEF GUI}Invalidate;{$ENDIF}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoOutOfBounds
  Description: handles a change in the plot that requires a complete rescaling
       Author: Mat Ballard
 Date created: 09/08/2002
Date modified: 09/08/2002 by Mat Ballard
      Purpose: makes 3D graphs visible
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoOutOfBounds(Sender: TObject; iX, iY: Integer);
begin
  if (iX < 0) then
  begin
    DoStyleChange(Self);
  end;
  if (iY > Height) then
  begin
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoDataChange
  Description: informs the user of a change in the data of the plot that requires a repaint
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: Fires the OnDataChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoDataChange(Sender: TObject);
begin
  if assigned(FOnDataChange) then OnDataChange(Sender);
  {$IFDEF GUI}Invalidate;{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoFileClose
  Description: informs the user of a file close
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: Fires the OnFileClose event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoFileClose(AFileName: String);
begin
  if assigned(FOnFileClose) then
    OnFileClose(Self, AFileName);
end;

{$IFDEF GUI}
  {$IFDEF WIN32}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.WMKeyDown
  Description: "listens" for keydown events
       Author: Mat Ballard
 Date created: 09/30/2002
Date modified: 09/30/2002 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.WMKeyDown(var Message: TWMKey);
var
  Key: Word;
  Shift: TShiftState;
begin
  Key := Message.CharCode;
  Shift := KeyDataToShiftState(Message.KeyData);
  ProcessKeyDown(Key, Shift);
  if not (Message.CharCode = 0) then inherited;
  //ShowMessage('Got a KeyDown !');
end;
  {$ENDIF}
{$ENDIF}

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetXAxis
  Description: standard property getting method
       Author: Mat Ballard
 Date created: 23/11/2002
Date modified: 23/11/2002 by Mat Ballard
      Purpose: gets the X Axis
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetXAxis: TAxis;
begin
  Result := FAxes[0];
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetYAxis
  Description: standard property getting method
       Author: Mat Ballard
 Date created: 23/11/2002
Date modified: 23/11/2002 by Mat Ballard
      Purpose: gets the Y Axis
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetYAxis: TAxis;
begin
  Result := FAxes[1];
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetZAxis
  Description: standard property getting method
       Author: Mat Ballard
 Date created: 23/11/2002
Date modified: 23/11/2002 by Mat Ballard
      Purpose: gets the Z Axis
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetZAxis: TAxis;
begin
  Result := FAxes.ZAxis;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetPlotTypeAsString
  Description: standard property getting method
       Author: Mat Ballard
 Date created: 23/07/2001
Date modified: 023/07/2001 by Mat Ballard
      Purpose: gets the PlotType as a string (for BCB users mainly)
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetPlotTypeAsString: String;
begin
  Result := Copy(TypInfo.GetEnumName(TypeInfo(TPlotType), Ord(FPlotType)), 3, 99);
end;


{------------------------------------------------------------------------------
     Function: TCustomPlot.GetZAngle
  Description: standard property getting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: gets the angle of the Z Axis, if any
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetZAngle: Word;
begin
  if (Assigned(ZAxis)) then
    GetZAngle := ZAxis.Angle
   else
    GetZAngle := 225;
end;

(*
{------------------------------------------------------------------------------
     Function: TCustomPlot.GetZLength
  Description: standard property getting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: gets the Length of the Z Axis, if any
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetZLength: Word;
begin
  if (Assigned(ZAxis)) then
    GetZLength := ZAxis.Length
   else
    GetZLength := 100;
end; *)

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetXAxis
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: sets the X Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetXAxis(Value: TAxis);
begin
  FAxes[0].Assign(Value);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetYAxis
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: sets the angle of the Y Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetYAxis(Value: TAxis);
begin
  FAxes[1].Assign(Value);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetZAxis
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: sets the angle of the Z Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetZAxis(Value: TAxis);
begin
  if (Assigned(ZAxis)) then
    FAxes.ZAxis.Assign(Value);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetZAngle
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: sets the angle of the Z Axis, if any
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetZAngle(Value: Word);
begin
  if (Assigned(ZAxis)) then
    if (Value <> ZAxis.Angle) then
      ZAxis.Angle := Value;
end;

(*
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetZLength
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: sets the Length of the Z Axis, if any
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetZLength(Value: Word);
begin
  if (Assigned(ZAxis)) then
    ZAxis.Length := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetZRatio
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 09/07/2003
Date modified: 09/07/2003 by Mat Ballard
      Purpose: sets the ZRatio property - that is, the ratio of the length of
               the Z Axis to the X and Y Axes
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetZRatio(Value: TPercent);
begin
  if (Value <> FZRatio) then
  begin
    FZRatio := Value;
    DoStyleChange(Self);
  end;
end;
*)

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetZLink
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2001 by Mat Ballard
      Purpose: sets the Link property - that is, should we link 3D series ?
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetZLink(Value: Boolean);
begin
  if (Value <> FZLink) then
  begin
    FZLink := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetZSeriesNames
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 01/07/2003
Date modified: 01/07/2003 by Mat Ballard
      Purpose: sets the Link property - that is, should we link 3D series ?
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetZSeriesNames(Value: Boolean);
begin
  if (Value <> FZSeriesNames) then
  begin
    FZSeriesNames := Value;
    DoStyleChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetXYFastAt
  Description: standard property setting method
       Author: Mat Ballard
 Date created: 05/07/2001
Date modified: 05/07/2001 by Mat Ballard
      Purpose: sets the Bridge point of the data - when it switches to the
               hyperfast draw algorithm
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetXYFastAt(Value: Longint);
begin
  if ((Value <> FXYFastAt) and (Value > 100)) then
  begin
    FXYFastAt := Value;
    DoStyleChange(Self);
  end;
end;


end.

