unit Plotactionlist;

{$I Plot.inc}

{.$ DEFINE DEBUG}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: PlotActionList.pas, released 12 September 2003.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 03/28/2003
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Known Issues:
        - highly experimental

-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, ActnList, Typinfo, {$IFDEF DEBUG}Clipbrd,{$ENDIF}
{$IFDEF WINDOWS}
  Actions do not work under Win16
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  ComCtrls, Controls, Dialogs,
{$ENDIF}
{$IFDEF LINUX}
  QComCtrls, QControls, QDialogs,
{$ENDIF}

  Filerhack, Plotdefs, Plot, PlotMenu, PlotPopupMenu, Menus, PlotToolBar, Misc;

type
  TPlotActionList = class(TActionList)
{TPlotActionList contains all the 80-odd actions of TPlot.}
  private
    FPlot: TPlot;
    FPlotMenu: TPlotMenu;
    FPlotToolBar: TPlotToolBar;
    FPlotPopupMenu: TPlotPopupMenu;
    F_Menus: TMainOptions;
    F_File: TFileOptions;
    F_Edit: TEditOptions;
    F_View: TViewOptions;
    F_Calc: TCalcOptions;
{$IFDEF FINANCE}
    F_Finance: TFinanceOptions;
{$ENDIF}

  protected
    function GetPlot(Target: TObject): TPlot; virtual;
{GetPlot is part of the Notification mechanism for linking components.}
    procedure ClearActions;
    procedure SetPlot(Value: TPlot);
{SetPlot is part of the Notification mechanism for linking components.
 It sets the Plot, and so creates the Actions, and sets the "OnExecute" methods
 for every Action.}
    procedure Loaded; override;
{Once streaming is completed, we need to set the Action events in the Loaded method.}
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
{Notification is part of the Notification mechanism for linking components.}
    procedure CreateActions;
{Create the actions from the resource Caption.res.}
    procedure SetOnExecutes;
{Set the OnExecute event handler for all Actions.}
    procedure DetermineEnabledNess;
    procedure DetermineVisibility;
{Which Actions are enabled ?}

(*    procedure DefineProperties(Filer: TFiler); override;
    procedure LoadActions(Reader: TReader);
    procedure StoreActions(Writer: TWriter); *)

  public
    Constructor Create(AOwner: TComponent); override;
{This is the standard constructor.}
    procedure SetPlotMenu(Value: TPlotMenu);
{This sets the value of the linked PlotMenu, as part of the Notification mechanism.}
    procedure SetPlotPopupMenu(Value: TPlotPopupMenu);
{This sets the value of the linked PlotPopupMenu, as part of the Notification mechanism.}
    procedure SetPlotToolbar(Value: TPlotToolbar);
{This sets the value of the linked PlotToolbar, as part of the Notification mechanism.}
    procedure SetupClients;
{This sets up the clients and points them to this components Actions.}
    function UpdateAction(Action: TBasicAction): Boolean; override;
{This updates the "enabledness" of all actions, and depends on the state of the associated Plot.}
    function GetIndexFromMenuPosition(Menu: TMainMenus; jValue: Integer): Integer;
{This returns the Index of the action for a particular menu item.}
    function GetMenuPositionFromIndex(Index: Integer): TPoint;
{This returns the position of a menu item associated with a particular action.}
  published
    property _Menus: TMainOptions read F_Menus write F_Menus;
    property _File: TFileOptions read F_File write F_File;
    property _Edit: TEditOptions read F_Edit write F_Edit;
    property _View: TViewOptions read F_View write F_View;
    property _Calc: TCalcOptions read F_Calc write F_Calc;
{$IFDEF FINANCE}
    property _Finance: TFinanceOptions read F_Finance write F_Finance;
{$ENDIF}
    property Plot: TPlot read FPlot write SetPlot;
{This is the Plot to which this ToolBar is connected.}
  end;

const
  MENUITEM_DIVIDER = -1;
  MENU_START = -2;

implementation

{$R Captions.res}

{------------------------------------------------------------------------------
  Constructor: TPlotActionList.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2003
Date modified: 04/25/2003 by Mat Ballard
      Purpose: initialization
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TPlotActionList.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPlot := nil;
  FPlotMenu := nil;
  FPlotToolBar := nil;
  FPlotPopupMenu := nil;
{$IFDEF FINANCE}
  F_Menus := [mnuFile, mnuEdit, mnuView, mnuCalc, mnuFinance];
{$ELSE}
  F_Menus := [mnuFile, mnuEdit, mnuView, mnuCalc];
{$ENDIF}
  F_File := [mnuNew, mnuOpen, mnuOverlayDiv, mnuOverlay, mnuClearOverlays, mnuSaveDiv, mnuSave, mnuSaveAs, mnuSaveImage, mnuPrintDiv, mnuPrint];
  F_Edit := [mnuCopy, mnuCopyHTML, mnuCopySeries, mnuPaste, mnuDisplaceDiv, mnuDisplace, mnuResetDisplacement, mnuEditSeriesDiv, mnuNewSeries, mnuCloneSeries, mnuEditPoint, mnuEditData, mnuEditSeries, mnuDeleteSeries, mnuLinearize, mnuZero, mnuAxisDiv, mnuNewYAxis, mnuEditAxis, mnuDeleteY2Axis, mnuEditFontDiv, mnuNewNote, mnuNewText, mnuDeleteNote, mnuEditFont, mnuPropertiesDiv, mnuProperties];
  F_View := [mnuHide, mnuShowAll, mnuNoteBorders, mnuDisplayModeDiv, mnuDisplayMode, mnuLegend, mnuZoomDiv, mnuSetAsNormal, mnuNormal, mnuManualZoom, mnuZoomIn, mnuZoomOut];
  F_Calc := [mnuPosition, mnuNearestPoint, mnuCalcAverageDiv, mnuCalcAverage, mnuCompressSeries, mnuCompressAllSeries, mnuContractSeries, mnuContractAllSeries, mnuCubicSpline, mnuHighs, mnuMovingAverage, mnuSmoothSeries, mnuSortSeries, mnuCalculusDiv, mnuDifferentiate, mnuIntegrate, mnuIntegral, mnuLineOfBestFitDiv, mnuLineOfBestFit, mnuTwoRegionLineOfBestFit, mnuFunctionDiv, mnuFourier, mnuInverseFourier, mnuFunction];
end;


{------------------------------------------------------------------------------
    Procedure: TPlotActionList.Loaded
  Description: sets tool button click events
       Author: Mat Ballard
 Date created: 08/08/2003
Date modified: 08/08/2003 by Mat Ballard
      Purpose: Actions are either created by CreateActions or the streaming system
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.Loaded;
begin
  inherited Loaded;
  CreateActions;
  SetOnExecutes;
  SetupClients;
end;

(*
{------------------------------------------------------------------------------
     Function: TCustomPlot.DefineProperties
  Description: Defines Property streams
       Author: Mat Ballard
 Date created: 08/15/2001
Date modified: 08/15/2001 by Mat Ballard
      Purpose: Allows the streaming of the properties of dynamic sub-components
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.DefineProperties(Filer: TFiler);
begin
  inherited; { allow base classes to define properties }
  Filer.DefineProperty('Actions',
    LoadActions,
    StoreActions,
    TRUE);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.LoadActions
  Description: Loads the Property stream of the Axes
       Author: Mat Ballard
 Date created: 08/15/2001
Date modified: 08/15/2001 by Mat Ballard
      Purpose: Allows the streaming of the properties of dynamic sub-components
 Known Issues:
       Note 1: The number of Axes has already been defined by the TPlot.NoSeries property
       Note 2: Axes 0 and 1 (X and Y) have already been stored.
 ------------------------------------------------------------------------------}
procedure TPlotActionList.LoadActions(Reader: TReader);
var
  i, Count: Integer;
begin
  if Reader.ReadBoolean then
  begin
    Count := Reader.ReadInteger;
    for i := 0 to Count-1 do
    begin
      Reader.ReadString;
      TReaderHack(Reader).ReadProperties(TAction(Self.Actions[i]));
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.StoreActions
  Description: Writes the Property streams of Axes
       Author: Mat Ballard
 Date created: 08/15/2001
Date modified: 08/15/2001 by Mat Ballard
      Purpose: Allows the streaming of the properties of dynamic sub-components
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.StoreActions(Writer: TWriter);
var
  i: Integer;
begin
  Writer.WriteBoolean(Self.ActionCount > 0);
  Writer.WriteInteger(Self.ActionCount);
  for i := 0 to Self.ActionCount-1 do
  begin
    Writer.WriteString(Format('Action%d', [i]));
    TWriterHack(Writer).WriteProperties(TAction(Self.Actions[i]));
  end;
end;
 *)

{------------------------------------------------------------------------------
     Function: TPlotActionList.GetPlot;
  Description: Check target object is a Plot
       Author: Mat Ballard
 Date created: 04/20/2003
Date modified: 05/02/2003 by Mat Ballard
      Purpose: design-time user interface
 Known Issues:
 ------------------------------------------------------------------------------}
function TPlotActionList.GetPlot(Target: TObject): TPlot;
begin
  Result := Target as TPlot;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.ClearActions;
  Description: gets rid of all the TPlot actions
       Author: Mat Ballard
 Date created: 09/20/2003
Date modified: 09/02/2003 by Mat Ballard
      Purpose: design-time user interface
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.ClearActions;
var
  i: Integer;
  AnAction: TAction;
begin
  for i := Self.ActionCount-1 downto 0 do
  begin
    if (Self.Actions[i].Tag < 0) or (Self.Actions[i].Tag > CAPTION_BASE) then
    begin
      AnAction := TAction(Self.Actions[i]);
      AnAction.ActionList := nil;
      AnAction.Free;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.SetPlot;
  Description: sets the Plot
       Author: Mat Ballard
 Date created: 04/20/2003
Date modified: 05/02/2003 by Mat Ballard
      Purpose: design-time user interface
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.SetPlot(Value: TPlot);
begin
  if (Value <> FPlot) then
  begin
    {ClearActions;
    //FPlot.PopupMenu := nil;
    if (Assigned(FPlotPopupMenu)) then FPlotPopupMenu.Clear;
    if (Assigned(FPlotToolBar)) then FPlotToolBar.Clear;}
    if (Value = nil) then
    begin
      FPlot := Value;
{$IFDEF GUI}
      FPlot.SetPlotActionList(TActionList(nil));
{$ENDIF}
    end
    else
    begin
      FPlot := Value;
      Value.FreeNotification(Self);
{$IFDEF GUI}
      FPlot.SetPlotActionList(TActionList(Self));
{$ENDIF}
      if not (csLoading in ComponentState) then
      begin
        CreateActions;
        SetOnExecutes;
        SetupClients;
        FPlot.PopupMenu := FPlotPopupMenu;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.Notification;
  Description: Manages FPlot
       Author: Mat Ballard
 Date created: 04/20/2003
Date modified: 05/02/2003 by Mat Ballard
      Purpose: part of the Notification mechanism
 Known Issues: Note deletion of attached Plot component
 ------------------------------------------------------------------------------}
procedure TPlotActionList.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = Plot) then
    FPlot := nil;
end;

{------------------------------------------------------------------------------
     Function: TPlotActionList.GetIndexFromMenuPosition;
  Description: Returns an Index from the menu position
       Author: Mat Ballard
 Date created: 04/20/2003
Date modified: 05/02/2003 by Mat Ballard
      Purpose: menu management
 Known Issues:
 ------------------------------------------------------------------------------}
function TPlotActionList.GetIndexFromMenuPosition(Menu: TMainMenus; jValue: Integer): Integer;
var
  i: Integer;
begin
  Result := 0;
  for i := 0 to Ord(Menu)-1 do
  begin
    Result := Result + NoMenuItems[i] + 2;
  end;
  if (jValue >= 0) then
    Result := Result + jValue + 1;
{$IFDEF DEBUG}
  case Menu of
    mnuFile: ClipBoard.AsText := Format('(mnuFile, %s) = #%d',
      [GetEnumName(TypeInfo(TFileMenus), jValue), Result]);
    mnuEdit: ClipBoard.AsText := Format('(mnuEdit, %s) = #%d',
      [GetEnumName(TypeInfo(TEditMenus), jValue), Result]);
    mnuView: ClipBoard.AsText := Format('(mnuView, %s) = #%d',
      [GetEnumName(TypeInfo(TViewMenus), jValue), Result]);
    mnuCalc: ClipBoard.AsText := Format('(mnuCalc, %s) = #%d',
      [GetEnumName(TypeInfo(TCalcMenus), jValue), Result]);
    mnuHelp: ClipBoard.AsText := Format('(mnuHelp, %s) = #%d',
      [GetEnumName(TypeInfo(THelpMenus), jValue), Result]);
  end;
{$ENDIF}
end;

{------------------------------------------------------------------------------
     Function: TPlotActionList.GetMenuPositionFromIndex;
  Description: Returns the menu position from an Index
       Author: Mat Ballard
 Date created: 04/20/2003
Date modified: 05/02/2003 by Mat Ballard
      Purpose: menu management
 Known Issues:
 ------------------------------------------------------------------------------}
function TPlotActionList.GetMenuPositionFromIndex(Index: Integer): TPoint;
var
  i, j, Tagindex: Integer;
begin
  Result.x := -1;
  Result.y := -1;
  TagIndex := 0;
  for i := 0 to Ord(High(TMainMenus))-1 do
  begin
    for j := 0 to NoMenuItems[i] do
    begin
      if (Index = TagIndex) then
      begin
        Result.x := i;
        Result.y := j;
        exit;
      end;
      Inc(TagIndex);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.CreateActions
  Description: Creates all the actions
       Author: Mat Ballard
 Date created: 08/08/2003
Date modified: 08/08/2003 by Mat Ballard
      Purpose: Actions are either created by CreateActions or the streaming system
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.CreateActions;
var
  i, j, ImageIndex, TagIndex: Integer;
  Category, Name: String;
  TempAction: TAction;

  function CleanName: String;
  var
    ii: Integer;
  begin
    Result := '';
    for ii := 1 to Length(Name) do
    begin
      if (Name[ii] in ['a'..'z', 'A'..'Z']) then Result := Result + Name[ii];
      if (Name[ii] = ' ') then Result := Result + '_';
      if (Name[ii] in ['0'..'9']) then
        if (ii > 1) then
          Result := Result + Name[ii];
    end;
  end;

begin
  if (Self.ActionCount = 0) then
  begin // no actions created yet
    TagIndex := 0;
    ImageIndex := 0;
{create the menus in each sub-menu:}
    for i := 0 to Ord(High(TMainMenus))-1 do
    begin
      Category := Copy(GetEnumName(TypeInfo(TMainMenus), i), 4, 99);
{Create the "Start of menu" action placeholder. This action does not execute anything.}
      TempAction := TAction.Create(Self);
      TempAction.Name := Category;
      TempAction.Caption := Category;
      TempAction.Tag := MENU_START; // use Tags to identify this as a Sub Menu End
      TempAction.Category := Category;
{add it to the list of Actions:}
      Self.AddAction(TempAction);
      //TempAction.ActionList := Self;
      for j := 0 to NoMenuItems[i] do
      begin
        TempAction := TAction.Create(Self);
        Name  := LoadStr(TagIndex + CAPTION_BASE);
        if (Name = '-') then
        begin
{Create the "Divider" action placeholder. This action does not execute anything.}
          TempAction.Name := Format('Div_%d', [TagIndex]);
          TempAction.Caption := '-';
          TempAction.Tag := MENUITEM_DIVIDER; // use Tags to identify this as a Menu Divider
          TempAction.Category := Category;
        end
        else
        begin
          TempAction.Name := CleanName;
          TempAction.Caption := Name;
          TempAction.Tag := TagIndex + CAPTION_BASE;
          TempAction.ImageIndex := ImageIndex;
          TempAction.Category := Category;
          TempAction.Hint := LoadStr(TagIndex + HINT_BASE);
          Inc(ImageIndex);
        end; {not a divider}
        Self.AddAction(TempAction);
        //TempAction.ActionList := Self;
        Inc(TagIndex);
      end; {for j}
    end; {for i}
{Add a help menu:}
    TempAction := TAction.Create(Self);
    TempAction.Name := 'Help';
    TempAction.Caption := 'Help';
    TempAction.Tag := MENU_START; // use Tags to identify this as a Menu
    TempAction.Category := 'Help';
    TempAction.ActionList := Self;
{And the "About" menu item:}
    TempAction := TAction.Create(Self);
    TempAction.Name := 'About';
    TempAction.Caption := 'About ...';
    TempAction.Tag := TagIndex; // use Tags to identify this as a Menu
    TempAction.Category := 'Help';
    TempAction.ActionList := Self;
  end; {need to create Actions}
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.SetOnExecutes
  Description: sets tool button click events
       Author: Mat Ballard
 Date created: 08/08/2003
Date modified: 08/08/2003 by Mat Ballard
      Purpose: Actions are either created by CreateActions or the streaming system
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.SetOnExecutes;
begin
  if (Assigned(FPlot)) then
  begin
{File:}
    Self.Actions[0].OnExecute := nil;
    Self.Actions[1].OnExecute := FPlot.NewClick;
    Self.Actions[2].OnExecute := FPlot.OpenClick;
    //Self.Actions[3].OnExecute := FPlot.nil;
    Self.Actions[4].OnExecute := FPlot.OverlayClick;
    Self.Actions[5].OnExecute := FPlot.ClearOverlaysClick;
    //Self.Actions[6].OnExecute := FPlot.nil;
    Self.Actions[7].OnExecute := FPlot.SaveClick;
    Self.Actions[8].OnExecute := FPlot.SaveAsClick;
    Self.Actions[9].OnExecute := FPlot.SaveImageClick;
    //Self.Actions[10].OnExecute := FPlot.nil;
    Self.Actions[11].OnExecute := FPlot.PrintClick;
{Edit:}
    Self.Actions[12].OnExecute := nil;
    Self.Actions[13].OnExecute := FPlot.CopyClick;
    Self.Actions[14].OnExecute := FPlot.CopyHTMLClick;
    Self.Actions[15].OnExecute := FPlot.CopySeriesClick;
    Self.Actions[16].OnExecute := FPlot.PasteClick;
    //Self.Actions[17].OnExecute := FPlot.nil;
    Self.Actions[18].OnExecute := FPlot.DisplaceClick;
    Self.Actions[19].OnExecute := FPlot.ResetDisplacementClick;
    //Self.Actions[20].OnExecute := FPlot.nil;
    Self.Actions[21].OnExecute := FPlot.NewSeriesClick;
    Self.Actions[22].OnExecute := FPlot.CloneSeriesClick;
    Self.Actions[23].OnExecute := FPlot.EditPointClick;
    Self.Actions[24].OnExecute := FPlot.EditDataClick;
    Self.Actions[25].OnExecute := FPlot.EditSeriesClick;
    Self.Actions[26].OnExecute := FPlot.DeleteSeriesClick;
    Self.Actions[27].OnExecute := FPlot.LinearizeClick;
    Self.Actions[28].OnExecute := FPlot.ZeroClick;
    //Self.Actions[29].OnExecute := FPlot.nil;
    Self.Actions[30].OnExecute := FPlot.NewYAxisClick;
    Self.Actions[31].OnExecute := FPlot.EditAxisClick;
    Self.Actions[32].OnExecute := FPlot.DeleteAxisClick;
    //Self.Actions[33].OnExecute := FPlot.nil;
    Self.Actions[34].OnExecute := FPlot.NewNoteClick;
    Self.Actions[35].OnExecute := FPlot.NewTextClick;
    Self.Actions[36].OnExecute := FPlot.DeleteNoteClick;
    Self.Actions[37].OnExecute := FPlot.EditFontClick;
    //Self.Actions[38].OnExecute := FPlot.nil;
    Self.Actions[39].OnExecute := FPlot.EditPropertiesClick;
{View:}
    Self.Actions[40].OnExecute := nil;
    Self.Actions[41].OnExecute := FPlot.HideClick;
    Self.Actions[42].OnExecute := FPlot.ShowAllClick;
    Self.Actions[43].OnExecute := FPlot.NoteBordersClick;
    //Self.Actions[44].OnExecute := FPlot.nil;
    Self.Actions[45].OnExecute := FPlot.ModeClick;
    Self.Actions[46].OnExecute := FPlot.LegendClick;
    //Self.Actions[47].OnExecute := FPlot.nil;
    Self.Actions[48].OnExecute := FPlot.SetAsNormalClick;
    Self.Actions[49].OnExecute := FPlot.NormalViewClick;
    Self.Actions[50].OnExecute := FPlot.ManualZoomClick;
    Self.Actions[51].OnExecute := FPlot.ZoomInClick;
    Self.Actions[52].OnExecute := FPlot.ZoomOutClick;
{Calculate:}
    Self.Actions[53].OnExecute := nil;
    Self.Actions[54].OnExecute := FPlot.PositionClick;
    Self.Actions[55].OnExecute := FPlot.NearestPointClick;
    //Self.Actions[56].OnExecute := FPlot.nil;
    Self.Actions[57].OnExecute := FPlot.AverageClick;
    Self.Actions[58].OnExecute := FPlot.CompressSeriesClick;
    Self.Actions[59].OnExecute := FPlot.CompressAllSeriesClick;
    Self.Actions[60].OnExecute := FPlot.ContractSeriesClick;
    Self.Actions[61].OnExecute := FPlot.ContractAllSeriesClick;

    Self.Actions[62].OnExecute := FPlot.SplineClick;
    Self.Actions[63].OnExecute := FPlot.HighsClick;
    Self.Actions[64].OnExecute := FPlot.MovingAverageClick;
    Self.Actions[65].OnExecute := FPlot.SmoothSeriesClick;
    Self.Actions[66].OnExecute := FPlot.SortClick;
    //Self.Actions[67].OnExecute := FPlot.nil;
    Self.Actions[68].OnExecute := FPlot.DifferentiateClick;
    Self.Actions[69].OnExecute := FPlot.IntegrateClick;
    Self.Actions[70].OnExecute := FPlot.IntegralClick;
    //Self.Actions[71].OnExecute := FPlot.nil;
    Self.Actions[72].OnExecute := FPlot.LineBestFitClick;
    Self.Actions[73].OnExecute := FPlot.TwoRegionLineBestFitClick;
    //Self.Actions[74].OnExecute := FPlot.nil;
    Self.Actions[75].OnExecute := FPlot.ForwardFourierClick;
    Self.Actions[76].OnExecute := FPlot.InverseFourierClick;
    Self.Actions[77].OnExecute := FPlot.FunctionClick;

{$IFDEF FINANCE}
    Self.Actions[78].OnExecute := nil;
    Self.Actions[79].OnExecute := FPlot.ADLClick;
    Self.Actions[80].OnExecute := FPlot.ADLClick;
    Self.Actions[81].OnExecute := FPlot.ADLClick;
    Self.Actions[82].OnExecute := FPlot.ADLClick;
    Self.Actions[83].OnExecute := FPlot.ADLClick;
    Self.Actions[84].OnExecute := FPlot.ADLClick;
    Self.Actions[85].OnExecute := FPlot.ADLClick;
    Self.Actions[86].OnExecute := FPlot.ADLClick;
    Self.Actions[87].OnExecute := FPlot.ADLClick;
    Self.Actions[88].OnExecute := FPlot.ADLClick;
    Self.Actions[89].OnExecute := FPlot.ADLClick;
    Self.Actions[90].OnExecute := FPlot.ADLClick;
    Self.Actions[91].OnExecute := FPlot.ADLClick;
    Self.Actions[92].OnExecute := FPlot.ADLClick;
    Self.Actions[93].OnExecute := FPlot.ADLClick;
    Self.Actions[94].OnExecute := FPlot.ADLClick;
    Self.Actions[95].OnExecute := FPlot.ADLClick;
    Self.Actions[96].OnExecute := FPlot.ADLClick;
{Help:}
    Self.Actions[97].OnExecute := nil;
    Self.Actions[98].OnExecute := FPlot.ShowAbout;
{$ELSE}
{Help:}
    Self.Actions[78].OnExecute := nil;
    Self.Actions[79].OnExecute := FPlot.ShowAbout;
{$ENDIF}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.SetPlotMenu
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 05/05/2003 by Mat Ballard
      Purpose: sets the PlotMenu property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.SetPlotMenu(Value: TPlotMenu);
begin
  FPlotMenu := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.SetPlotPopupMenu
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 05/05/2003 by Mat Ballard
      Purpose: sets the PlotPopupMenu property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.SetPlotPopupMenu(Value: TPlotPopupMenu);
begin
  FPlotPopupMenu := Value;
  if (Assigned(FPlot)) then
    FPlot.PopupMenu := FPlotPopupMenu;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.SetPlotToolBar
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 05/05/2003 by Mat Ballard
      Purpose: sets the PlotToolBar property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.SetPlotToolBar(Value: TPlotToolBar);
begin
  FPlotToolBar := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.SetupClients
  Description: override that sets enabledness and visibility of Actions
       Author: Mat Ballard
 Date created: 05/05/2003
Date modified: 05/05/2003 by Mat Ballard
      Purpose: Action management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.SetupClients;
begin
  DetermineVisibility;
  if Assigned(FPlotMenu) then
  begin
    FPlotMenu.CreateMenus;
    FPlotMenu.SetActions;
  end;
  if Assigned(FPlotPopupMenu) then
  begin
    FPlotPopupMenu.CreateMenus;
    FPlotPopupMenu.SetActions;
  end;
  if Assigned(FPlotToolBar) then
  begin
    FPlotToolBar.CreateToolButtons;
    FPlotToolBar.SetActions;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.UpdateAction
  Description: override that sets enabledness and visibility of Actions
       Author: Mat Ballard
 Date created: 05/05/2003
Date modified: 05/05/2003 by Mat Ballard
      Purpose: Action management
 Known Issues:
 ------------------------------------------------------------------------------}
function TPlotActionList.UpdateAction(Action: TBasicAction): Boolean;
begin
  Result := False;
  DetermineEnabledNess;
  if Assigned(OnUpdate) then OnUpdate(Action, Result);
end;

{------------------------------------------------------------------------------
    Procedure: TPlotActionList.DetermineEnabledNess
  Description: Sets the Enabledness of Axis-related menus
       Author: Mat Ballard
 Date created: 04/17/2000
Date modified: 04/17/2003 by Mat Ballard
      Purpose: menu management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotActionList.DetermineEnabledNess;
var
  AnySeries, AnyPoints, EnoughPoints: Boolean;
  TotalNoPts: Integer;
begin
{$IFNDEF SHOWALLMENUS}
  if (Assigned(FPlot)) then
  begin
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuFile, Ord(mnuClearOverlays))]).Enabled :=
      (FPlot.FirstOverlay >= 0);

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuEditAxis))]).Enabled := TRUE;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuEditFont))]).Enabled := TRUE;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuDeleteNote))]).Enabled :=
      (FPlot.Texts.Count > 0);
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuDeleteY2Axis))]).Enabled :=
      (FPlot.Axes.Count > 2);

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuView, Ord(mnuHide))]).Enabled := TRUE;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuView, Ord(mnuNoteBorders))]).Enabled :=
      (FPlot.Notes.Count + FPlot.Notes.Count > 0);
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuView, Ord(mnuLegend))]).Enabled := (FPlot.Series.Count > 1);

    //SetSeriesEnabledNess;

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuDisplaceDiv))]).Enabled :=
      TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuDisplace))]).Enabled;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCalcAverageDiv))]).Enabled :=
      TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCalcAverage))]).Enabled;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuLineOfBestFitDiv))]).Enabled :=
      TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuLineOfBestFit))]).Enabled;

{Now do the Series-related stuff:}
    TotalNoPts := FPlot.Series.TotalNoPts;
    AnySeries := (FPlot.Series.Count > 0);
    AnyPoints := (TotalNoPts > 0);
    EnoughPoints := (TotalNoPts >= 20);

{The following are independent of whether or not a Series has been selected:}
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuFile, Ord(mnuSave))]).Enabled :=
      (AnyPoints and FPlot.Series.DataChanged);
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuFile, Ord(mnuSaveAs))]).Enabled := AnyPoints;

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuCopyHTML))]).Enabled := AnySeries;

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuNearestPoint))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCompressAllSeries))]).Enabled := EnoughPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuContractAllSeries))]).Enabled := EnoughPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuFunctionDiv))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuFunction))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuFourier))]).Enabled := EnoughPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuInverseFourier))]).Enabled := EnoughPoints;

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuCopySeries))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuDisplace))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuResetDisplacement))]).Enabled := FPlot.Series.Displaced;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuCloneSeries))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuEditPoint))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuEditData))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuEditSeries))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuDeleteSeries))]).Enabled := AnySeries;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuLinearize))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuEdit, Ord(mnuZero))]).Enabled := AnyPoints;

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCalcAverage))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCompressSeries))]).Enabled :=
      EnoughPoints and
      (Assigned(FPlot.ClickedSeries)) and
      (not ((Assigned(FPlot.ClickedSeries.XDataSeries)) or (FPlot.ClickedSeries.XDataRefCount > 0)));
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuContractSeries))]).Enabled :=
      TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCompressSeries))]).Enabled;

    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCubicSpline))]).Enabled := (TotalNoPts > 3);
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuHighs))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuMovingAverage))]).Enabled := (TotalNoPts > 3);
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuSmoothSeries))]).Enabled := EnoughPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuSortSeries))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuCalculusDiv))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuDifferentiate))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuIntegrate))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuIntegral))]).Enabled := AnyPoints;
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuLineOfBestFit))]).Enabled := (TotalNoPts > 2);
    TAction(Self.Actions[GetIndexFromMenuPosition(mnuCalc, Ord(mnuTwoRegionLineOfBestFit))]).Enabled := (TotalNoPts > 4);
  end;
{$ENDIF}
end;

procedure TPlotActionList.DetermineVisibility;
var
  Index, j: Integer;
  iMenu: TMainMenus;
begin
{$IFNDEF SHOWALLMENUS}
  if (Assigned(FPlot)) then
  begin
{create the menus in each sub-menu:}
    for iMenu := mnuFile to High(TMainMenus) do
    begin
      Index := GetIndexFromMenuPosition(iMenu, -1);
      TAction(Self.Actions[Index]).Visible := iMenu in _Menus;
    end;
    for j := 0 to NoMenuItems[0] do
    begin
      Index := GetIndexFromMenuPosition(mnuFile, j);
      TAction(Self.Actions[Index]).Visible := TFileMenus(j) in _File;
    end; {for j}
    for j := 0 to NoMenuItems[1] do
    begin
      Index := GetIndexFromMenuPosition(mnuEdit, j);
      TAction(Self.Actions[Index]).Visible := TEditMenus(j) in _Edit;
    end; {for j}
    for j := 0 to NoMenuItems[2] do
    begin
      Index := GetIndexFromMenuPosition(mnuView, j);
      TAction(Self.Actions[Index]).Visible := TViewMenus(j) in _View;
    end; {for j}
    for j := 0 to NoMenuItems[3] do
    begin
      Index := GetIndexFromMenuPosition(mnuCalc, j);
      TAction(Self.Actions[Index]).Visible := TCalcMenus(j) in _Calc;
    end; {for j}
  {$IFDEF FINANCE}
    for j := 0 to NoMenuItems[4] do
    begin
      Index := GetIndexFromMenuPosition(mnuFinance, j);
      TAction(Self.Actions[Index]).Visible := TFinanceMenus(j) in _Finance;
    end; {for j}
  {$ENDIF}
  end;
{$ENDIF}
end;


end.
