unit Plotdefs;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Menus.pas, released 16 october 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 09/25/2002
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Known Issues:
-----------------------------------------------------------------------------}

{
HOWTO add menu items:

Warning: this is a complex multistep process!

1. Edit TPlot.wb1 and add the menu item(s) you want.
   It is best to add them in their correct position near rows 209-269.
   Add its hint similarly, in the rows below.
   Make sure you update the indices correctly.

2. Open Images32.res in the Image Editor.

3. for i := MaxImageNo downto NewIndex do
      ImageIndex := ImageIndex + NoNewMenuItems;

4. create your new 16x16x256 bitmap(s), and rename them to NewIndex, ....

5. Save Images32.Res.

6. Open Plotxxx_xx.bpg, open Plotxxx_Rxx.dpk, and ...

7. In Plotdefs.pas, insert your new mnuMyNewMenuItem(s).

8. In PlotImageList.pas, redefine NO_IMAGES to the new value.

9. In Plot.pas, create your new MyNewMenuClick method(s).

10. In Plot.pas, go to the CreateOnClicks procedure of TContour.CreateMenus, and add:
  FPlotPopUpMenu.Items[Ord(mnuXXX)].Items[Ord(mnuMyNewMenuItem)].OnClick := MyNewMenuClick;

11. Similarly add the Caption and Hint.

20. Change
          TPlot.DetermineMenuVisibility,
          TPlot.SetSeriesVisibility and
          TPlot.SetSeriesEnabledness
          TPopupOptions.Create
  as required.

21. If you are support languages than English, try to translate your new menu caption and hint.

22. Export the neccessary resourcestrings from the spreadsheet to the TPlot/lang/eng.txt (and other) files.
}

{Setup Information:
 ******************
Delphi 5:
  KEY                                                             VALUE                                                              DATA
  HKEY_LOCAL_MACHINE\SOFTWARE\Borland\Delphi\5.0                  App                                                                C:\Program Files\Borland\Delphi5\Bin\Delphi32.exe
  HKEY_CURRENT_USER\Software\Borland\Delphi\5.0\Known Packages\   $(DELPHI)\Projects\Bpl\Misc300_Design.bpl                          Chemware Miscellaneous Components (Design time)
Delphi 6:
  HKEY_LOCAL_MACHINE\SOFTWARE\Borland\Delphi\6.0                  App                                                                C:\Program Files\Borland\Delphi6\Bin\Delphi32.exe
  HKEY_CURRENT_USER\Software\Borland\Delphi\6.0\Known Packages\   c:\program files\borland\delphi6\Projects\Bpl\Misc300_Design.bpl   Chemware Miscellaneous Components (Design time)
Delphi 7: ?

Borland Shared:
 HKEY_LOCAL_MACHINE\SOFTWARE\Borland\Borland Shared\SharedFilesDir = C:\Program Files\Common Files\Borland Shared

}

interface

uses
  Classes, SysUtils,
{$IFDEF GUI}
  {$IFDEF WINDOWS}
  Menus, Graphics, Wintypes, Dialogs,
  {$ENDIF}
  {$IFDEF WIN32}
  Menus, Graphics, Windows, Dialogs,
  {$ENDIF}
  {$IFDEF LINUX}
  Types,
  QGraphics, QMenus, QDialogs,
  {$ENDIF}
{$ELSE}
  gd, gd_io, gdWindows, gdGraphics, {gdWinapi,}
{$ENDIF}
  Misc;

{Misc and TPlot now only work for Compiler 3 up !}
{$IFNDEF COMPILER3_UP}
  Misc and TPlot now only work for Compiler 3 up !
{$ENDIF}

const
  TPLOT_VERSION = 311;

  FILE_FORMAT_VERSION = 300;
  MAX_FILE_VERSION = 300;

  NAN = 0.0 / 0.0;
  INFINITY = 1.0 / 0.0;
  NEG_INFINITY = -1.0 / 0.0;

{$IFDEF POLYNOMIAL_FIT}
  POLYORDER = 9;
{$ELSE}
  POLYORDER = 1;
{$ENDIF}

{Where are our resources hiding ?}
  TAG_BASE = 1000;
  CAPTION_BASE = 3000;
  HINT_BASE = 4000;

{a little bit of trigonometry for TAngleRect:}
  TWO_PI = 2*PI;
  PI_ON_TWO = PI/2;
  THREE_PI_ON_TWO = 3*PI/2;

{$IFDEF MSWINDOWS}
  SMALL_FONT_SIZE = 8;
  MEDIUM_FONT_SIZE = 9;
  LARGE_FONT_SIZE = 14;
{$ENDIF}
{$IFDEF LINUX}
  SMALL_FONT_SIZE = 10;
  MEDIUM_FONT_SIZE = 12;
  LARGE_FONT_SIZE = 18;
{$ENDIF}

{This roughly controls the number of unique colors in contour plots:}
  COLOUR_GRANULARITY = 256;
{Default: 256;
 - larger values may give a slightly improved image;
 - smaller values lead to false banding.}

resourcestring
  sArial = 'Arial';

type
{$IFDEF BCB3}
  Longword = Cardinal;
{$ENDIF}  

  TDegrees = 0..359;

  //TSide = (siNormal, siOpposite);
  TDirection = (drHorizontal, drVertical);
{Screen objects are sometimes Horizontal or Vertical.}

  TStatus = (sLoaded, sNew, sSaved, sDataChanged, sStyleChanged, sUpdating);
{What is the status of the plot ?}

{Different possible plot types: these determine how the series are interpreted,
 and so how data are displayed on screen}
  TPlotType = (
    ptXY,
    {ptError,    // These are all now SeriesTypes
    ptMultiple,
    ptBubble,}
    ptColumn,
    ptStack,
    ptNormStack,
    ptPie,
    ptPolar,
    ptLineContour,
    ptContour,
    pt3DContour,
    pt3DWire,
    pt3DColumn);
{ptXY          - a normal XY plot that everyone does in sigh school

 ptError       - every second (odd indexed) series is interpreted as the "error bar" of the
                 previous (even indexed) series.
                  
 ptMultiple    - each "point" is composed of a multiple of points from
                 the different series. This is a generalised extension of the previous
                 ptError type. It is used for High-Low-Close, High-Low-Open-Close, etc
                 plots. Eg:
  high, low        high, average, low        high, low, open, close;
     X                     X                           H
     |                     |                           |
     |                     |                           O
     |                     O                           |
     |                     |                           |
     |                     |                           |
     |                     |                           |
     |                     |                           C
     |                     |                           |
     X                     X                           L
                 See the Multiplicity property !

 ptColumn      - columns of data: the last X value is taken as Xn + (Xn - Xn-1),
                 so the appearance is:
                              (X1, Y1)----(X2, Y1)
                              |                  |
                              |                  (Xn, Yn)----(Xn+dX, Yn)
           (X0, Y0)----(X1, Y0)                  |                     |
           |                  |                  |                     |
           |                  |                  |                     |
           |                  |                  |                     |
           |                  |                  |                     |
           (X0, 0)---------(X1, 0)------------(X2, 0)---------(Xn+dX, 0)
                - when there are N (> 1) series, each gap is divided by (N+1):
                              -------
                              |     |
                              |     |
           -------            |     |
           |     |------      |     |
           |     |     |      |     |------
           |     |     |      |     |     |
           |     |     |      |     |     |
           (X0, 0)---------(X1, 0)------------(X2, 0)

 ptStack       - a stacked column, with each series on to of each other

 ptNormStack   - like a stack, but normalized to 100%

 ptPie         - very boring pie graphs

 ptPolar       - (x, y) = (r*cos(theta), r*sin(theta))

 ptContour     - a two-dimensional plot where the color represents the Y value.

 pt3DContour   - a three-dimensional plot where the color represents the Y value.

 pt3DWire      - a three-dimensional plot in which each series in order as succesive slices through the surface
}

  TSeriesType = (stNormal, stComplex, stBubble, stError, stMultiple);
{These are the Series Types. The require additional storage:
     stNormal:   X, Y arrays
     stComplex:  X, Y PLUS one more array
     stBubble:   X, Y PLUS one more array
     stError:    X, Y PLUS TWO more arrays
     stMultiple: X, Y PLUS N more arrays; 0 < N < 4; N == Multiplicity
 and so "interact" with the DataStatus.
 They determine what additional data is stored, and how it is drawn.}

  TScreenJob = (sjNone,
                sjDrag,
                sjRightDrag,
                sjHide,
                sjZoomIn,
                sjEditAxis,
                sjTouchNotePointer,
                sjMoveNotePointer,
                sjEditFont,
                sjEditPoint,
                sjEditSeries,
                sjFlashEdit,
                sjCopySeries,
                sjDisplace,
                sjCloneSeries,
                sjDeleteSeries,
                sjLinearize,
                sjZero,
                sjPosition,
                sjNearestPoint,
                sjAverage,
                sjContractSeries,
                sjContractAllSeries,
                sjSplineSeries,
                sjHighs,
                sjLows,
                sjMovingAverage,
                sjSmoothSeries,
                sjSortSeries,
                sjDifferentiate,
                sjIntegrate,
                sjIntegral,
                sjLineOfBestFit,
                sjDualLineBestFit1, sjDualLineBestFit2,
                sjSelection,
                sjDualSelection1, sjDualSelection2);
{What particular operation is to be carried out ?}
{}
{Note that a 2-region line of best fit is a 2 stage operation.}

  TObjectType = (soNone,
                 soTitle,
                 soLegend,
                 soXAxis, soXAxisTitle, soXAxisLabel,
                 soYAxis, soYAxisTitle, soYAxisLabel,
                 soZAxis, soZAxisTitle, soZAxisLabel, soZAxisEnd,
                 soLeftBorder, soTopBorder, soRightBorder, soBottomBorder,
                 soSeries,
                 soText,
                 soNote);
{What object on the screen got clicked on ?}
{}
{Note that the soYAxis, soYAxisTitle, and soYAxisLabel are generic: several
 sub-components can share this type.}

  TDisplayMode = (dmNormal, dmNone, dmRun, dmHistory);
{What do we do when new data is added ?}
{}
{    dmNormal: Check (and adjust DisplayMode if neccessary) with the addition of every point.}
{    dmNone: Nothing;}
{    dmRun: Increase the Span of the X Axis by 100% to accomodate the new data,
 and alter (increase) the scale of the Y Axis if needed.}
{    dmHistory: Similar to dmRun, except that the X-Axis runs from (Min-Max)
 through (x-Max) to 0.}
{}
{Note that with dmRun we can expect more data with increasing X values.
 Rather than force a complete screen re-draw every time a data point is
 added, we extend the X Axis by 100%.}
{}
{History mode, dmHistory, deserves a few more words. In this mode TContour
 displays the data over the last History 's worth of points - older data is not
 displayed. The result is a graph that shows the last few seconds or minutes of
 data in great detail.}

  TContourDetail = (cdLow, cdMedium, cdHigh);

  TGridType = (gtNone, gtHorizontal, gtVertical, gtBoth);

  TQuad = array[0..3] of TPoint;

  TSymbol = (syNone,
    syCircle, sySquare,
    syUpTriangle, syDownTriangle,
    syPlus, syCross, syStar,
    syDash, syVertDash, syLeftDash, syRightDash);
{These are the different symbols that are used to represent the data points.}

{the actual menus:}
  TMainMenus = (
    mnuFile,
    mnuEdit,
    mnuView,
    mnuCalc,
{$IFDEF FINANCE}
    mnuFinance,
{$ENDIF}
    mnuHelp);
  TMainOptions = set of TMainMenus;

  TFileMenus = (
    mnuNew,
    mnuOpen,
    mnuOverlayDiv, {convention: Name the Divs after the following menuitem}
    mnuOverlay,
    mnuClearOverlays,
    mnuSaveDiv,
    mnuSave,
    mnuSaveAs,
    mnuSaveImage,
    mnuPrintDiv,
    mnuPrint);
  TFileOptions = set of TFileMenus;

  TEditMenus = (
    mnuCopy,
    mnuCopyHTML,
    mnuCopySeries,
    mnuPaste,
    mnuDisplaceDiv,
    mnuDisplace,
    mnuResetDisplacement,
    mnuEditSeriesDiv,
    mnuNewSeries,
    mnuCloneSeries,
    mnuEditPoint,
    mnuEditData,
    mnuEditSeries,
    mnuDeleteSeries,
    mnuLinearize,
    mnuZero,
    mnuAxisDiv,
    mnuNewYAxis,
    mnuEditAxis,
    mnuDeleteY2Axis,
    mnuEditFontDiv,
    mnuNewNote,
    mnuNewText,
    mnuDeleteNote,
    mnuEditFont,
    mnuPropertiesDiv,
    mnuProperties); {19}
  TEditOptions = set of TEditMenus;

  TViewMenus = (
    mnuHide,
    mnuShowAll,
    mnuNoteBorders,
    mnuDisplayModeDiv,
    mnuDisplayMode,
    mnuLegend,
    mnuZoomDiv,
    mnuSetAsNormal,
    mnuNormal,
    mnuManualZoom,
    mnuZoomIn,
    mnuZoomOut);
  TViewOptions = set of TViewMenus;

  TCalcMenus = (
    mnuPosition,
    mnuNearestPoint,
    mnuCalcAverageDiv,
    mnuCalcAverage,
    mnuCompressSeries,
    mnuCompressAllSeries,
    mnuContractSeries,
    mnuContractAllSeries,
    mnuCubicSpline,
    mnuHighs,
    mnuMovingAverage,
    mnuSmoothSeries,
    mnuSortSeries,
    mnuCalculusDiv,
    mnuDifferentiate,
    mnuIntegrate,
    mnuIntegral,
    mnuLineOfBestFitDiv,
    mnuLineOfBestFit,
    mnuTwoRegionLineOfBestFit,
    mnuFunctionDiv,
    mnuFourier,
    mnuInverseFourier,
    mnuFunction);
  TCalcOptions = set of TCalcMenus;

{$IFDEF FINANCE}
{These have been taken from:
 http://www.stockcharts.com/education/What/IndicatorAnalysis/
}
  TFinanceMenus = (
    mnuADL,
    mnuAroon,
    mnuADX,
    mnuATR,
    mnuBB,
    mnuCCI,
    mnuCMF,
    mnuMACD,
    mnuMA,
    mnuPVO,
    mnuPO,
    mnuPR,
    mnuPVB,
    mnuRSI,
    mnuStochOsc,
    mnuStochRSI,
    mnuTRIN,
    mnuWilliams);
  TFinanceOptions = set of TFinanceMenus;
{$ENDIF}

  THelpMenus = (
    mnuAbout
    );

  TSaveOption = (soAsText, soProperties);
  TSaveOptions = set of TSaveOption;

  {T3DPoint = Record
    x: Integer;
    y: Integer;
    z: Integer;
  end;
  p3DPoint = ^T3DPoint;}

  T3DZPoint = Record
    x: Integer;
    y: Integer;
    Z: Single;
  end;
  p3DZPoint = ^T3DZPoint;
{Used in TSeriesList for 3D stuff.}

  T3DRealPoint = Record
    X: Single;
    Y: Single;
    Z: Single;
  end;
  p3DRealPoint = ^T3DRealPoint;
{Used in TSeriesList for 3D stuff.}

  TPoints = array [0..600] of TPoint;
  pTPoints = ^TPoints;
{Used in TSeries.DrawPie.}

  TSeriesInfo = Record       {The series are 0, 1, 2}
    Index: Integer;          {What is the Index of this series of data in the SeriesList ?}
    XCol: Integer;           {in which column is this series' X data ?}
    XTextCol: Integer;       {in which column is this series' X STRING data, if any ?}
    YCol: Integer;           {in which column is this series' Y data ?}
    XSeriesIndex: Integer;   {what is the index of the series that contains the X Data in the SeriesList ?}
    XValue: Single;          {the X Value, after a string conversion}
    YValue: Single;          {the Y Value, after a string conversion}
  end;

{A TSeriesInfoArray is never declared directly - instead, it is used as a
 template for a dynamic array}
  TSeriesInfoArray = array[0..1023] of TSeriesInfo;
  pSeriesInfoArray = ^TSeriesInfoArray;

  TContour = class(TPersistent)
  private
    FDetail: TContourDetail;
    FInterval: Single;
    FStart: Single;
    FWireFrame: Boolean;
    FGamma: Single;
    FLambdaMin: Integer;
    FLambdaMax: Integer;
    FLambdaSpan: Integer;
    FOnChange: TNotifyEvent;
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure SetContourDetail(Value: TContourDetail);
    procedure SetInterval(Value: Single);
    procedure SetStart(Value: Single);
    procedure SetWireFrame(Value: Boolean);
    procedure SetGamma(Value: Single);
    procedure SetLambdaMin(Value: Integer);
    procedure SetLambdaMax(Value: Integer);
  public
    Constructor Create; {$IFDEF DELPHI4_UP}reintroduce;{$ENDIF}
    Destructor Destroy; override;
  published
    property Detail: TContourDetail read FDetail write SetContourDetail default cdLow;
{This is the detail of the colour interpolation in a contour graph.
     cdLow: each triangular tile has uniform colour; this is pretty grainy
     cdMedium: each pixel in each triangular tile has its own colour: this makes a good bitmap, but a lousy metafile
     cdHigh: each rectangular pixel in each triangular tile has its own colour: this makes a good metafile
 See SeriesList.DrawContour.}
    property Interval: Single read FInterval write SetInterval;
{This is the granularity of the intervals in a contour graph. See SeriesList.DrawContour.}
    property Start: Single read FStart write SetStart;
{This is the granularity of the Starts in a contour graph. See SeriesList.DrawContour.}
    property WireFrame: Boolean read FWireFrame write SetWireFrame default FALSE;
{This is the presence or absence of a wireframe on the 3D surface.}
    property Gamma: Single read FGamma write SetGamma;
{The color intensity correction for Contour plots:}
    property LambdaMin: Integer read FLambdaMin write SetLambdaMin default 380{LAMBDA_MIN};
{The minimum wavelength (color) for Contour plots; default is almost Violet:}
    property LambdaMax: Integer read FLambdaMax write SetLambdaMax default 780{LAMBDA_MAX};
{The maximum wavelength (color) for Contour plots; default is almost Red:}
    property LambdaSpan: Integer read FLambdaSpan;
{The gap between the minimum and maximum wavelengths.}
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TGrids = class(TPersistent)
  private
    FColor: TColor;
    FStyle: TPenStyle;
    FGridType: TGridType;
    FWallColor: TColor;
    FWallColor2: TColor;
    FWallType: TGridType;
    FOnChange: TNotifyEvent;
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure SetColor(Value: TColor);
    procedure SetWallColor2(Value: TColor);
    procedure SetStyle(Value: TPenStyle);
    procedure SetGridType(Value: TGridType);
    procedure SetWallColor(Value: TColor);
    procedure SetWallType(Value: TGridType);

  public
    constructor Create; {$IFDEF DELPHI4_UP}reintroduce;{$ENDIF}
    destructor Destroy; override;
  published
    property Color: TColor read FColor write SetColor default clGray;
{What colour is our grid ? Default is clGray.}
    property Style: TPenStyle read FStyle write SetStyle;
{What pen style (dash, dot, etc) do we want for Grids ?}
    property GridType: TGridType read FGridType write SetGridType;
{Do we want a grid ?}
    property WallColor: TColor read FWallColor write SetWallColor default clYellow;
{What colour is our Wall ? Default is clYellow.}
    property WallColor2: TColor read FWallColor2 write SetWallColor2 default clYellow;
{What is the SECOND colour of our grid ? Default is clGray.}
{}
{NB: if WallColor = WallColor2, then the walls are a solid color; if they are different,
 then there is a color gradient.}
    property WallType: TGridType read FWallType write SetWallType;
{Do we want a Wall ?}
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;


  TLimits = class(TPersistent)
  private
    FAxisIndex: Byte;
    FPen: TPen;
    FLower: Single;
    FUpper: Single;
    FVisible: Boolean;
    FOnChange: TNotifyEvent;
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure SetAxisIndex(Value: Byte);
    procedure SetPen(Value: TPen);
    procedure SetLower(Value: Single);
    procedure SetUpper(Value: Single);
    procedure SetVisible(Value: Boolean);
  public
    Constructor Create; {$IFDEF DELPHI4_UP}reintroduce;{$ENDIF}
    Destructor Destroy; override;
  published
    property AxisIndex: Byte read FAxisIndex write SetAxisIndex;
    property Pen: TPen read FPen write SetPen;
    property Lower: Single read FLower write SetLower;
    property Upper: Single read FUpper write SetUpper;
    property Visible: Boolean read FVisible write SetVisible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

const
  NoMenuItems: array [0..Ord(High(TMainMenus))] of Integer =
    (Ord(High(TFileMenus)),
     Ord(High(TEditMenus)),
     Ord(High(TViewMenus)),
     Ord(High(TCalcMenus)),
{$IFDEF FINANCE}Ord(High(TFinanceMenus)),{$ENDIF}
     Ord(High(THelpMenus)));


implementation


{******************************************************************************}
{------------------------------------------------------------------------------
  Constructor: TContour.Create;
  Description: standard constructor
       Author: Mat Ballard
 Date created: 07/20/2002
Date modified: 07/20/2002 by Mat Ballard
      Purpose: sets default values
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TContour.Create;
begin
{distance between line contours is 1:}
  FInterval := 1;
{Set the color-control variables for contours:}
  FGamma := 0.80;
  FLambdaMin := Round(LAMBDA_MIN);
  FLambdaMax := Round(LAMBDA_MAX);
  FLambdaSpan := FLambdaMax - FLambdaMin;
  FOnChange := nil;
end;

Destructor TContour.Destroy;
begin
  FOnChange := nil;
  inherited Destroy;
end;

procedure TContour.AssignTo(Dest: TPersistent);
begin
  TContour(Dest).Detail := FDetail;
  TContour(Dest).Interval := FInterval;
  TContour(Dest).Start := FStart;
  TContour(Dest).WireFrame := FWireFrame;
  TContour(Dest).Gamma := FGamma;
  TContour(Dest).LambdaMin := FLambdaMin;
  TContour(Dest).LambdaMax := FLambdaMax;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetContourDetail
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/08/2001
Date modified: 02/08/2002 by Mat Ballard
      Purpose: sets the granularity of the colour interpolation in a contour graph.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetContourDetail(Value: TContourDetail);
begin
  if (FDetail <> Value) then
  begin
    FDetail := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetInterval
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 07/15/2001
Date modified: 07/15/2002 by Mat Ballard
      Purpose: sets the granularity of the intervals in a contour graph.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetInterval(Value: Single);
begin
  if (FInterval <> Value) then
  begin
    FInterval := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetStart
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 07/15/2001
Date modified: 07/15/2002 by Mat Ballard
      Purpose: sets the granularity of the Starts in a contour graph.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetStart(Value: Single);
begin
  if (FStart <> Value) then
  begin
    FStart := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetWireFrame
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 07/15/2001
Date modified: 07/15/2002 by Mat Ballard
      Purpose: sets the presence or absence of a wireframe on the 3D surface
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetWireFrame(Value: Boolean);
begin
  if (FWireFrame <> Value) then
  begin
    FWireFrame := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetGamma
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2002 by Mat Ballard
      Purpose: sets the Gamma color adjustment parameter
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetGamma(Value: Single);
begin
  if ((Value <> FGamma) and (0 < Value) and (Value <= 1)) then
  begin
    FGamma := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetLambdaMin
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/20012 by Mat Ballard
      Purpose: sets the LambdaMin color adjustment parameter
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetLambdaMin(Value: Integer);
begin
  if ((Value <> FLambdaMin) and (Value >= LAMBDA_MIN) and (Value < FLambdaMax)) then
  begin
    FLambdaMin := Value;
    FLambdaSpan := FLambdaMax - FLambdaMin;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TContour.SetLambdaMax
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 01/07/2001
Date modified: 01/07/2002 by Mat Ballard
      Purpose: sets the LambdaMax color adjustment parameter
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TContour.SetLambdaMax(Value: Integer);
begin
  if ((Value <> FLambdaMax) and (Value <= LAMBDA_MAX) and (Value > FLambdaMin)) then
  begin
    FLambdaMax := Value;
    FLambdaSpan := FLambdaMax - FLambdaMin;
    if Assigned(FOnChange) then OnChange(Self);
  end;  
end;


{******************************************************************************}
{------------------------------------------------------------------------------
  Constructor: TGrids.Create;
  Description: standard constructor
       Author: Mat Ballard
 Date created: 07/20/2002
Date modified: 07/20/2002 by Mat Ballard
      Purpose: sets default values
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TGrids.Create;
begin
  FColor := clGray ;
  FStyle := psDot;
  FWallColor := clYellow;
  FWallColor2 := clYellow;
  FOnChange := nil;
end;

Destructor TGrids.Destroy;
begin
  FOnChange := nil;
  inherited Destroy;
end;

procedure TGrids.AssignTo(Dest: TPersistent);
begin
  TGrids(Dest).Color := FColor;
  TGrids(Dest).Style := FStyle;
  TGrids(Dest).GridType := FGridType;
  TGrids(Dest).WallType := FWallType;
  TGrids(Dest).WallColor := FWallColor;
  TGrids(Dest).WallColor2 := FWallColor2;
  //TGrids(Dest).WallStyle := FWallStyle;
end;

{------------------------------------------------------------------------------
    Procedure: TGrids.SetGridsDetail
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/08/2001
Date modified: 02/08/2002 by Mat Ballard
      Purpose: sets the granularity of the colour interpolation in a Grids graph.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TGrids.SetColor(Value: TColor);
begin
  FColor := Value;
  if Assigned(FOnChange) then OnChange(Self);
end;

procedure TGrids.SetStyle(Value: TPenStyle);
begin
  if (FStyle <> Value) then
  begin
    FStyle := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TGrids.SetGridType(Value: TGridType);
begin
  if (FGridType <> Value) then
  begin
    FGridType := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TGrids.SetWallType(Value: TGridType);
begin
  if (FWallType <> Value) then
  begin
    FWallType := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TGrids.SetWallColor(Value: TColor);
begin
  if (FWallColor <> Value) then
  begin
    FWallColor := Value;
    FWallColor2 := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TGrids.SetWallColor2(Value: TColor);
begin
  FWallColor2 := Value;
  if Assigned(FOnChange) then OnChange(Self);
end;

{procedure TGrids.SetWallStyle(Value: TBrushStyle);
begin
  if (FWallStyle <> Value) then
  begin
    FWallStyle := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;}

{******************************************************************************}
{------------------------------------------------------------------------------
  Constructor: TLimits.Create;
  Description: standard constructor
       Author: Mat Ballard
 Date created: 07/20/2001
Date modified: 07/20/2001 by Mat Ballard
      Purpose: sets default values
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TLimits.Create;
begin
  FAxisIndex := 1;
  FUpper := 7;
  FLower := 3;
  FPen := TPen.Create;
  FPen.Color := clRed;
  FPen.Style := psDot;
  FOnChange := nil;
end;

Destructor TLimits.Destroy;
begin
  FOnChange := nil;
  FPen.Free;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TLimits.AssignTo;
  Description: standard copier
       Author: Mat Ballard
 Date created: 04/20/2001
Date modified: 04/20/2001 by Mat Ballard
      Purpose: copies values from a source
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLimits.AssignTo(Dest: TPersistent);
begin
  TLimits(Dest).Upper := FUpper;
  TLimits(Dest).Lower := FLower;
  TLimits(Dest).Visible := FVisible;
end;

procedure TLimits.SetUpper(Value: Single);
begin
  if (FUpper <> Value) then
  begin
    FUpper := Value;
    if (FUpper < FLower) then
      FLower := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TLimits.SetAxisIndex(Value: Byte);
begin
  if (FAxisIndex <> Value) then
  begin
    FAxisIndex := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TLimits.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
  if Assigned(FOnChange) then OnChange(Self);
end;

procedure TLimits.SetLower(Value: Single);
begin
  if (FLower <> Value) then
  begin
    FLower := Value;
    if (FLower > FUpper) then
      FLower := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;

procedure TLimits.SetVisible(Value: Boolean);
begin
  if (FVisible <> Value) then
  begin
    FVisible := Value;
    if Assigned(FOnChange) then OnChange(Self);
  end;
end;


end.



