unit Plotmenu;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Axis.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 02/25/2000
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
The purpose of this module is to provide the developer using TPlot a quick
method of rapidly implementing menu access to the TPlot functions.

It will be augmented by a TPlotAction module at a later stage.

Known Issues:
    - Since TPlotMenu is useless without TPlot, there is only the one
      registration unit: TPlot_Reg.pas.
    - Is INVISIBLE in D1 Runtime - works perfectly in the designer, but
      no menu is present in running applications. If anyone wants to help fix
      this, please do so and let me know the answer.
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, ActnList, Typinfo,
{$IFDEF WINDOWS}
  Menus,
{$ENDIF}
{$IFDEF WIN32}
  Forms, Menus,
{$ENDIF}
{$IFDEF LINUX}
  QForms, QMenus,
{$ENDIF}

  Misc, Plot, Plotdefs;

const
  TPLOTMENU_VERSION = 300;

  FILE_TAG = 1000;
  HELP_TAG = 1095;
  ABOUT_TAG = 1096;
  REOPEN_TAG = 1097;
  EXIT_DIV_TAG = 1098;
  EXIT_TAG = 1099;

  mnuReopen = 2;

type
  TPlotMenu = class(TMainMenu)
  private
{The plot to which this menu is linked:}
    //FPlot: TPlot;
{The PlotActionList to which this menu is linked:}
    FPlotActionList: TActionList;
{and the limit of the number of those files:}
    FHistoryLimit: Byte;
{Creates the menus after the Plot property is set:}
  protected
    procedure CreateReopenSubMenu;
{Creates the Reopen menu item and Submenu and adds it to the File menu:}
    function GetHistory: String;
{Gets the History List as a string:}

    procedure HandleFileClick(Sender: TObject);
{Called when the a Reopen File menuitem is clicked on.}
    //procedure Loaded; override;
{The standard overridden method in which the Menus are created if needed.}
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
{Implementation of the standard Notification method for linked components.}
    procedure SetPlotActionList(Value: TActionList);
{Sets the associated PlotActionList, creates the menus and sets their Actions.}
  public
    property History: String read GetHistory stored FALSE;
{A History List, CRLF delimited, of previously closed files.
 Just use it as:
   MyStringList.Create;
   MyStringList.Text := PlotMenu1.History;}

    constructor Create(AOwner: TComponent); override;
{The standard constructor.}
    destructor Destroy; override;
{The standard destructor.}

    procedure AddHistory(Value: String);
{Adds one more file to the History List.}
    procedure CreateMenus;
{Creates the menus from the Actions}
    procedure SetActions;
{Points each menu item at FPlotActionsList's Actions}
    function MenuExists(ATag: Integer; AName: String): Boolean;
{Does AMenu already exist in Self ?}
{}
{The basis for comparison is both the Tag and Caption}
  published
    property HistoryLimit: Byte read FHistoryLimit write FHistoryLimit;
{The limit of the number of files in the History List.}
    property PlotActionList: TActionList read FPlotActionList write SetPlotActionList;
{The PlotActionList to which this TPlotMenu is linked.}
  end;


implementation

uses
  PlotActionList;

{Constructor and Destructor ---------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TPlotMenu.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: initialization
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TPlotMenu.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPlotActionList := nil;
  FHistoryLimit := 10;
end;

{------------------------------------------------------------------------------
   Destructor: TPlotMenu.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the Plot Property
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TPlotMenu.Destroy;
begin
  if FPlotActionList <> nil then
    FPlotActionList.FreeNotification(Self);
  FPlotActionList := nil;
  inherited Destroy;
end;

(*{------------------------------------------------------------------------------
    Procedure: TPlotMenu.Loaded
  Description: standard Loaded override
       Author: Mat Ballard
 Date created: 04/25/2003
Date modified: 04/25/2003 by Mat Ballard
      Purpose: prevent duplication of menu items
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotMenu.Loaded;
begin
  inherited Loaded;
  if (Self.Items.Count < 4) then
    CreateMenus;
  SetActions;
end;*)

{Get functions and Set procedures ---------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TPlotMenu.AddHistory
  Description: Adds one more file to the History List
       Author: Mat Ballard
 Date created: 09/01/2000
Date modified: 09/01/2000 by Mat Ballard
      Purpose: History List management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotMenu.AddHistory(Value: String);
var
  i: Integer;
  pReopen: TMenuItem;
  TempMenuItem: TMenuItem;
begin
  if (csDestroying in ComponentState) then exit;
  if (Length(Value) = 0) then exit;

  pReopen := Self.Items[0].Items[mnuReopen];
  if (pReopen = nil) then
    exit;

{$IFNDEF DELPHI1}
  for i := 0 to pReopen.Count-1 do
  begin
    if (pReopen.Items[i].Hint = Value) then
    begin
{Change the order of existing entries:}
      pReopen.Items[i].MenuIndex := 0;
      exit;
    end;
  end;
{$ENDIF}

  TempMenuItem := TMenuItem.Create(Self);
{Note: we store the full pathname in the Hint because D5 mangles the Caption:
 the AutoHotKeys property does not work ! Also looks neater !}
  TempMenuItem.Caption := ExtractFileName(Value);
  TempMenuItem.Hint := Value;
  TempMenuItem.OnClick := HandleFileClick;
  pReopen.Insert(0, TempMenuItem);

  while (pReopen.Count > FHistoryLimit) do
    pReopen.Delete(pReopen.Count-1);

  pReopen.Enabled := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotMenu.GetHistory
  Description: Gets the History List as a string.
       Author: Mat Ballard
 Date created: 08/13/2000
Date modified: 08/13/2000 by Mat Ballard
      Purpose: History List management
 Known Issues:
 ------------------------------------------------------------------------------}
function TPlotMenu.GetHistory: String;
var
  i: Integer;
  TheString: String;
  pReopen: TMenuItem;
begin
  TheString := '';
  pReopen := Self.Items[0].Items[mnuReopen];

  for i := 0 to pReopen.Count-1 do
    TheString := TheString + pReopen.Items[i].Caption;

  GetHistory := TheString;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotMenu.SetPlotActionList
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the PlotActionList Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotMenu.SetPlotActionList(Value: TActionList);
begin
  if (Value <> FPlotActionList) then
  begin
    if (Value = nil) then
    begin
      FPlotActionList := Value;
{$IFDEF GUI}
      TPlotActionList(FPlotActionList).SetPlotMenu(TPlotMenu(nil));
{$ENDIF}
    end
    else
    begin
      FPlotActionList := Value;
      Value.FreeNotification(Self);
      if Assigned(FPlotActionList.Images) then
        Self.Images := FPlotActionList.Images;
{$IFDEF GUI}
      if (not (csLoading in ComponentState)) then
      begin
        if (Self.Items.Count < 4) then
          CreateMenus;
        SetActions;
      end;
      TPlotActionList(FPlotActionList).SetPlotMenu(TPlotMenu(Self));
{$ENDIF}
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotMenu.SetActions
  Description: sets the Actions for the menu items
       Author: Mat Ballard
 Date created: 04/25/2003
Date modified: 04/25/2003 by Mat Ballard
      Purpose: Action management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotMenu.SetActions;
var
  i, j: Integer;
begin
  if (Assigned(FPlotActionList)) then
  begin
    for i := 0 to Self.Items.Count-1 do
    begin
      for j := 0 to Self.Items[i].Count-1 do
        if ((0 <= Self.Items[i].Items[j].Tag) and
            (Self.Items[i].Items[j].Tag < FPlotActionList.ActionCount)) then
          Self.Items[i].Items[j].Action := FPlotActionList.Actions[Self.Items[i].Items[j].Tag];
    end;
  end;
end;

{General methods --------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CreateMenus
  Description: creates popup menus that are accessible by right-click
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/20/2000 by Mat Ballard
      Purpose: modularize user-interface code
 Known Issues: this was a bitch to get right !
 ------------------------------------------------------------------------------}
procedure TPlotMenu.CreateMenus;
{$IFDEF GUI}
var
  i: Word;
  MenuIndex: Integer;
{This following is just a dummy matrix of menu items used to create sub-menus,
 then removed and the menuitems freed.}
  TempMenu: array [0..31] of array [0..0] of TMenuItem;
  TempMenuItem: TMenuItem;
begin
  if ((Self.Items.Count < 4) and (Assigned(FPlotActionList))) then
  begin
    MenuIndex := -1;
    for i := 0 to FPlotActionList.ActionCount-1 do
    begin
      if (FPlotActionList.Actions[i].Tag = MENU_START) then
        Inc(MenuIndex);
{don't re-create a menu if it already exists:}
      if (not MenuExists(FPlotActionList.Actions[i].Tag,
        FPlotActionList.Actions[i].Name)) then
      begin
        if (FPlotActionList.Actions[i].Tag = MENU_START) then
        begin
{we create a temporary menu array to add the submenu, and then later remove it:}
          TempMenu[MenuIndex][0] := TMenuItem.Create(Self);
          TempMenu[MenuIndex][0].Name := Format('Menu%d', [i]);
          TempMenu[MenuIndex][0].Tag := -999;
          TempMenuItem := NewSubMenu(
            TAction(FPlotActionList.Actions[i]).Caption,
            0,
            'mnu' + FPlotActionList.Actions[i].Name,
            TempMenu[MenuIndex]);
{Make the tag of the Menu Item the same as index of its action:}
          TempMenuItem.Tag := i;
{Developers can insert menus using TMainMenu.merge:}
          TempMenuItem.GroupIndex := 10*MenuIndex;
          Self.Items.Add(TempMenuItem);
        end {Menu Start}
        else if ((FPlotActionList.Actions[i].Tag = MENUITEM_DIVIDER) or (FPlotActionList.Actions[i].Tag >= 0)) then
        begin {Menu item:}
          TempMenuItem := TMenuItem.Create(Self);
          TempMenuItem.Tag := i;
          TempMenuItem.Name := 'mnu' + FPlotActionList.Actions[i].Name;
{add the TempMenuItem to the popup:}
          Self.Items[MenuIndex].Add(TempMenuItem);
        end;
{This is used later in SetActions.}
      end; {MenuExists}
    end; {i over Actions}

{remove the temporary menu array used to create the submenu:}
    for i := 0 to Ord(High(TMainMenus)) do
    begin
      if (Self.Items[i].Items[0].Tag = -999) then
      begin
        Self.Items[i].Remove(TempMenu[i][0]);
        TempMenu[i][0].Free;
      end;
    end; {i over submenus}

{do what it says:}
    CreateReopenSubMenu;

{Add a divider to the File menu:}
    if (not MenuExists(EXIT_DIV_TAG, 'ExitDiv')) then
    begin
      TempMenuItem := TMenuItem.Create(Self);
      TempMenuItem.Caption := '-';
      TempMenuItem.Name := 'ExitDiv';
      TempMenuItem.Tag := EXIT_DIV_TAG;
      Self.Items[0].Add(TempMenuItem);
    end;

{Add an "Exit" to the File menu:}
    if (not MenuExists(EXIT_TAG, 'ExitMenuItem')) then
    begin
      TempMenuItem := TMenuItem.Create(Self);
      TempMenuItem.Caption := 'E&xit';
      TempMenuItem.Tag := EXIT_TAG;
      TempMenuItem.Name := 'ExitMenuItem';
      Self.Items[0].Add(TempMenuItem);
    end;
  end; {FPlotActionList <> nil}
{$ELSE}
begin  
{$ENDIF} {GUI}  
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CreateReopenSubMenu
  Description: creates popup menus that are accessible by right-click
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/20/2000 by Mat Ballard
      Purpose: modularize user-interface code
 Known Issues: this was a bitch to get right !
 ------------------------------------------------------------------------------}
procedure TPlotMenu.CreateReopenSubMenu;
var
{This following is just a dummy matrix of menu items used to create sub-menus,
 then removed and the menuitems freed.}
  TempMenu: array [0..0] of TMenuItem;
  TempMenuItem: TMenuItem;
begin
{don't create menus when the Plot property is streamed in:}
  if (csLoading in ComponentState) then exit;

{don't re-create a menu if it already exists:}
  if ((Self.Items[0].Items[mnuReopen].Tag = REOPEN_TAG) and
      (Self.Items[0].Items[mnuReopen].Caption = '&Reopen')) then
    exit;

{we create a temporary menu array to add the submenu, and then later remove it:}
  TempMenu[0] := TMenuItem.Create(Self);
  TempMenu[0].Visible := FALSE;
  TempMenuItem := NewSubMenu(
    '&Reopen',
    0,
    'ReopenSubMenu',
    TempMenu);

  TempMenuItem.Tag := REOPEN_TAG;
{disable the "Reopen" menu because it has no entries:}
{$IFNDEF SHOWALLMENUS}
  TempMenuItem.Enabled := FALSE;
{$ENDIF}

  Self.Items[0].Insert(mnuReopen, TempMenuItem);
{remove the temporary menu array used to create the submenu:}
  Self.Items[0].Items[mnuReopen].Remove(TempMenu[0]);
{then free it:}
  TempMenu[0].Free;
end;

{------------------------------------------------------------------------------
    Procedure: TPlotMenu.HandleFileClick
  Description: Click Event handler for all Reopen file menus
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to user clicks of the FilaName menus by firing the
               TPlot.Open method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotMenu.HandleFileClick(Sender: TObject);
begin
  //if (FPlot = nil) then exit;

  //FPlot.OpenFile(TMenuItem(Sender).Hint);
end;

{------------------------------------------------------------------------------
    Procedure: TPlotMenu.Notification
  Description: handles changes in Plot property
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Notifies TPlot that it is no longer loved
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPlotMenu.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = PlotActionList) then
  begin
    FPlotActionList := nil;
  end;
end;

{------------------------------------------------------------------------------
     Function: TPlotMenu.MenuExists
  Description: Does this menu exist ? Based on Tag and Name
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: do we need to add a menu item ?
 Return Value: Boolean;
 Known Issues:
 ------------------------------------------------------------------------------}
function TPlotMenu.MenuExists(ATag: Integer; AName: String): Boolean;
var
  i,
  j: Integer;
begin
  Result := FALSE;
  for i := 0 to Self.Items.Count-1 do
  begin
{the menus:}
    if ((Self.Items[i].Tag = ATag) and
        (Pos(AName, Self.Items[i].Name) = 4)) then
    begin
      Result := TRUE;
      exit;
    end;
    for j := 0 to Self.Items[i].Count-1 do
    begin
{the submenus:}
      if ((Self.Items[i].Items[j].Tag = ATag) and
          (Pos(AName, Self.Items[i].Items[j].Name) = 4)) then
      begin
        Result := TRUE;
        exit;
      end;
    end;
  end;
end;


end.
