unit Seredit;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: SerEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 02/25/2000
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To facilitate user manipluation of Series properties

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, TypInfo,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Buttons, Controls, {Dialogs,} Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Buttons, Controls, {Dialogs,} Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF LINUX}
  QButtons, QControls, QDialogs, QExtCtrls, QForms, QGraphics, QStdCtrls,
{$ENDIF}

  Axis, Misc, NEdit, Plotdefs, Coloredt;

type
  TSeriesProperty = record
    BrushColor: TColor;
    BrushStyleIndex: Integer;
    BubbleSize: TPercent;
    DeltaX: Integer;
    DeltaY: Integer;
    ExternalXSeries: Boolean;
    IsLine: Boolean;
    PenColor: TColor;
    PenWidthIndex: Integer;
    PenStyleIndex: Integer;
    SeriesTypeIndex: Integer;
    ShadeLimits: Boolean;
    ShowFit: Boolean;
    SymbolIndex: Integer;
    SymbolCheck: Boolean;
    SymbolSize: Integer;
    YAxisIndex: Byte;
    //YAxis: TAxis;
    XDataIndependent: Boolean;
    Visible: Boolean;
  end;

  TSeriesEditorForm = class(TForm)
    HelpBitBtn: TBitBtn;
    CancelBitBtn: TBitBtn;
    OKBitBtn: TBitBtn;
    VisibleCheckBox: TCheckBox;
    PenGroupBox: TGroupBox;
    PenColorLabel: TLabel;
    PenWidthLabel: TLabel;
    PenWidthComboBox: TComboBox;
    PenStyleComboBox: TComboBox;
    PenStyleLabel: TLabel;
    NoComboBox: TComboBox;
    DisplacementGroupBox: TGroupBox;
    DeltaXLabel: TLabel;
    DeltaYLabel: TLabel;
    SeriesNumberLabel: TLabel;
    NameEdit: TEdit;
    YAxisLabel: TLabel;
    YAxisComboBox: TComboBox;
    XDataIndependentCheckBox: TCheckBox;
    BrushGroupBox: TGroupBox;
    BrushColorLabel: TLabel;
    BrushStyleLabel: TLabel;
    BrushStyleComboBox: TComboBox;
    DeltaXNEdit: TNEdit;
    DeltaYNEdit: TNEdit;
    PenColorEdit: TColorEdit;
    BrushColorEdit: TColorEdit;
    ApplyBitBtn: TBitBtn;
    SymbolGroupBox: TGroupBox;
    SymbolSizeLabel: TLabel;
    SymbolComboBox: TComboBox;
    SymbolSizeNEdit: TNEdit;
    SymbolTypeLabel: TLabel;
    ShadedCheckBox: TCheckBox;
    IsLineCheckBox: TCheckBox;
    ShowFitCheckBox: TCheckBox;
    BubbleSizeNEdit: TNEdit;
    SeriesTypeComboBox: TComboBox;
    Label1: TLabel;
    MultiplicityLabel: TLabel;
    MultiplicityComboBox: TComboBox;
    BubbleSizeLabel: TLabel;
    SymbolCheckCheckBox: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NoComboBoxClick(Sender: TObject);
    procedure PenWidthComboBoxClick(Sender: TObject);
    procedure PenStyleComboBoxClick(Sender: TObject);
    procedure SymbolComboBoxClick(Sender: TObject);
    procedure VisibleCheckBoxClick(Sender: TObject);
    procedure YAxisComboBoxClick(Sender: TObject);
    procedure DeltaXNEditChange(Sender: TObject);
    procedure DeltaYNEditChange(Sender: TObject);
    procedure SymbolSizeNEditChange(Sender: TObject);
    procedure NameEditChange(Sender: TObject);
    procedure XDataIndependentCheckBoxClick(Sender: TObject);
    procedure PenColorEditChange(Sender: TObject);
    procedure BrushColorEditChange(Sender: TObject);
    procedure ApplyBitBtnClick(Sender: TObject);
    procedure ShadedCheckBoxClick(Sender: TObject);
    procedure IsLineCheckBoxClick(Sender: TObject);
    procedure ShowFitCheckBoxClick(Sender: TObject);
    procedure BubbleSizeNEditChange(Sender: TObject);
    procedure SeriesTypeComboBoxClick(Sender: TObject);
    procedure SymbolCheckCheckBoxClick(Sender: TObject);
  private
    pASP: ^TSeriesProperty;
    CurrentIndex: Integer;

  public
    SeriesPropertyList: TList;
    SeriesNames: TStringList;
    ThePlot: TObject;
    
    function AddSeries(AName: String; ASeriesProperty: TSeriesProperty): Integer;
    procedure SelectSeries(Index: Integer);
    procedure DoHintsFromResource;
  end;

var
  SeriesEditorForm: TSeriesEditorForm;

implementation

{$R *.dfm}

uses
  Plot, Data;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position, populates the PenColor combo, and initializes SeriesNames and SeriesPropertyList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.FormCreate(Sender: TObject);
var
  i: Integer;
begin
  DoHintsFromResource;

{set combo and edit box widths:}
  SetDialogGeometry(Self, ApplyBitBtn, HelpBitBtn.Left);
{the exceptions:}
  NoComboBox.Width := 50;
  NameEdit.Width := BrushGroupBox.Width;
  YAxisComboBox.Width := 100;

{Populate the combo boxes:}
  for i := 0 to Ord(High(TBrushStyle)) do
    BrushStyleComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TBrushStyle), i), 3, 99));
  for i := 0 to 20 do
    PenWidthComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TPenStyle)) do
    PenStyleComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TPenStyle), i), 3, 99));
  for i := 0 to Ord(High(TSymbol)) do
    SymbolComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TSymbol), i), 3, 99));
  for i := 0 to Ord(High(TSeriesType)) do
    SeriesTypeComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TSeriesType), i), 3, 99));
  for i := 0 to MAX_MULTIPLICITY-1 do
    MultiplicityComboBox.Items.Add(IntToStr(i));

  SymbolComboBox.ItemIndex := 0;
  SeriesPropertyList := TList.Create;
  SeriesNames := TStringList.Create;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DoHintsFromResource
  Description: standard loading of hints from resources
       Author: Mat Ballard
 Date created: 06/25/2001
Date modified: 06/25/2001 by Mat Ballard
      Purpose: display Hints in different languages
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DoHintsFromResource;
begin
  ApplyBitBtn.Hint := 'Apply these properties now !';
  BubbleSizeNEdit.Hint := 'The percentage size of the biggest bubble relative to the Y Axis';
  BrushGroupBox.Hint := 'For symbols and columns';
  DisplacementGroupBox.Hint := 'The distance (in pixels) that the Series is moved from its true position';
  IsLineCheckBox.Hint := 'Display this series as a line in Column mode ?';
  NoComboBox.Hint := 'Which series to work on';
  NameEdit.Hint := 'The name of the Series';
  PenGroupBox.Hint := 'The lines that connect each point in the Series';
  ShadedCheckBox.Hint := 'Shade the regions of this Series above and below the axis limits';
  ShowFitCheckBox.Hint := 'Shall we display the fitted line or curve ?';
  SymbolGroupBox.Hint := 'Symbol shape and size';
  VisibleCheckBox.Hint := 'Make this series appear and disappear';
  XDataIndependentCheckBox.Hint := 'Make this Series independent of X Data in another Series';
  YAxisComboBox.Hint := 'The Y Axis that this series is plotted against';
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.FormDestroy
  Description: standard FormDestroy procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the SeriesNames and SeriesPropertyList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to SeriesPropertyList.Count-1 do
    FreeMem(SeriesPropertyList.Items[i], SizeOf(TSeriesProperty));

  SeriesPropertyList.Free;
  SeriesNames.Free;
end;

{------------------------------------------------------------------------------
     Function: TSeriesEditorForm.AddSeries
  Description: adds a new series to the list
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: stores the Series properties
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesEditorForm.AddSeries(AName: String; ASeriesProperty: TSeriesProperty): Integer;
var
  i: Integer;
  pDestination: Pointer;
  pSource,
  pDest: PChar;
begin
  NoComboBox.Items.Add(IntToStr(SeriesNames.Count));

  SeriesNames.Add(AName);

  GetMem(pDestination, SizeOf(ASeriesProperty));
  pSource := @ASeriesProperty;
{NB: this causes terminal access violations:
  System.Move(pSource, pDestination, SizeOf(TSeriesProperty));}
  pDest := pDestination;
  for i := 1 to SizeOf(ASeriesProperty) do
  begin
    pDest^ := pSource^;
    Inc(pSource);
    Inc(pDest);
  end;

  SeriesPropertyList.Add(pDestination);
  AddSeries := SeriesPropertyList.Count;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SelectSeries
  Description: responds to selection of a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SelectSeries(Index: Integer);
begin
  pASP := SeriesPropertyList.Items[Index];
  CurrentIndex := Index;

  NameEdit.Text := SeriesNames.Strings[Index];

  BubbleSizeNEdit.Visible := (pASP^.SeriesTypeIndex = Ord(stBubble));
  BubbleSizeLabel.Enabled := BubbleSizeNEdit.Visible;
  MultiplicityComboBox.Visible := (pASP^.SeriesTypeIndex = Ord(stMultiple));
  MultiplicityLabel.Enabled := MultiplicityComboBox.Visible;
  SymbolCheckCheckBox.Enabled := (pASP^.SeriesTypeIndex = Ord(stError)) or
    (pASP^.SeriesTypeIndex = Ord(stMultiple));

  BrushColorEdit.Color := pASP^.BrushColor;
  BrushColorEdit.Text := ColorToString(pASP^.BrushColor)+ ' ...';
  BrushStyleComboBox.ItemIndex := pASP^.BrushStyleIndex;
  DeltaXNEdit.AsInteger := pASP^.DeltaX;
  DeltaYNEdit.AsInteger := pASP^.DeltaY;
  IsLineCheckBox.Checked := pASP^.IsLine;
  PenColorEdit.Color := pASP^.PenColor;
  PenColorEdit.Text := ColorToString(pASP^.PenColor)+ ' ...';
  PenWidthComboBox.ItemIndex := pASP^.PenWidthIndex;
  PenStyleComboBox.ItemIndex := pASP^.PenStyleIndex;
  SeriesTypeComboBox.ItemIndex := pASP^.SeriesTypeIndex;
  ShadedCheckBox.Checked := pASP^.ShadeLimits;
  ShowFitCheckBox.Checked := pASP^.ShowFit;
  SymbolComboBox.ItemIndex := pASP^.SymbolIndex;
  SymbolCheckCheckBox.Checked := pASP^.SymbolCheck;
  SymbolSizeNEdit.AsInteger := pASP^.SymbolSize;
  VisibleCheckBox.Checked := pASP^.Visible;
{We define YAxisIndex to run from 1 to FAxisList.Count-1}
  YAxisComboBox.ItemIndex := pASP^.YAxisIndex-1;
  //YAxisComboBox.ItemIndex := YAxisComboBox.Items.IndexOfObject(pASP^.YAxis);
  XDataIndependentCheckBox.Checked := pASP^.XDataIndependent;
  XDataIndependentCheckBox.Enabled := pASP^.ExternalXSeries;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.NoComboBoxClick
  Description: responds to user selection of a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues: see SelectSeries above
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.NoComboBoxClick(Sender: TObject);
begin
  SelectSeries(NoComboBox.ItemIndex);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.PenWidthComboBoxClick
  Description: responds to selection of a new Pen Width for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.PenWidthComboBoxClick(Sender: TObject);
begin
  pASP^.PenWidthIndex := PenWidthComboBox.ItemIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.PenStyleComboBoxClick
  Description: responds to selection of a new Pen Style for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.PenStyleComboBoxClick(Sender: TObject);
begin
  pASP^.PenStyleIndex := PenStyleComboBox.ItemIndex;
  if (PenStyleComboBox.ItemIndex > Ord(psSolid)) then
    PenWidthComboBox.ItemIndex := 1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SymbolComboBoxClick
  Description: responds to selection of a new Symbol for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SymbolComboBoxClick(Sender: TObject);
begin
  pASP^.SymbolIndex := SymbolComboBox.ItemIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.VisibleCheckBoxClick
  Description: responds to changes to the Visibility of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.VisibleCheckBoxClick(Sender: TObject);
begin
  pASP^.Visible := VisibleCheckBox.Checked;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.YAxisComboBoxClick
  Description: responds to changes in the selected Y Axis (primary or 2nd)
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.YAxisComboBoxClick(Sender: TObject);
begin
  pASP^.YAxisIndex := YAxisComboBox.ItemIndex + 1;
  //pASP^.YAxis := TAxis(YAxisComboBox.Items.Objects[YAxisComboBox.ItemIndex]);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DeltaXNEditChange
  Description: responds to changes in DeltaX
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DeltaXNEditChange(Sender: TObject);
begin
  pASP^.DeltaX := DeltaXNEdit.AsInteger;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DeltaYNEditChange
  Description: repsonds to changes in DeltaY
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DeltaYNEditChange(Sender: TObject);
begin
  pASP^.DeltaY := DeltaYNEdit.AsInteger;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SymbolSizeNEditChange
  Description: repsonds to changes in SymbolSize
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SymbolSizeNEditChange(Sender: TObject);
begin
  pASP^.SymbolSize := SymbolSizeNEdit.AsInteger;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.NameEditChange
  Description: repsonds to changes in the Series Name
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.NameEditChange(Sender: TObject);
begin
  SeriesNames.Strings[CurrentIndex] := NameEdit.Text;
end;

procedure TSeriesEditorForm.XDataIndependentCheckBoxClick(Sender: TObject);
begin
  pASP^.XDataIndependent := XDataIndependentCheckBox.Checked;
end;

procedure TSeriesEditorForm.PenColorEditChange(Sender: TObject);
begin
  pASP^.PenColor := PenColorEdit.Color;
  pASP^.BrushColor := Misc.GetPalerColor(pASP.PenColor, 70);
end;

procedure TSeriesEditorForm.BrushColorEditChange(Sender: TObject);
begin
  pASP^.BrushColor := BrushColorEdit.Color;
end;

procedure TSeriesEditorForm.ApplyBitBtnClick(Sender: TObject);
begin
{$IFDEF GUI}
  TPlot(ThePlot).ApplySeriesChange(Self);
{$ENDIF}
end;


procedure TSeriesEditorForm.ShadedCheckBoxClick(Sender: TObject);
begin
  pASP^.ShadeLimits := ShadedCheckBox.Checked;
end;

procedure TSeriesEditorForm.IsLineCheckBoxClick(Sender: TObject);
begin
  pASP^.IsLine := IsLineCheckBox.Checked;
end;

procedure TSeriesEditorForm.ShowFitCheckBoxClick(Sender: TObject);
begin
  pASP^.ShowFit := ShowFitCheckBox.Checked;
end;

procedure TSeriesEditorForm.BubbleSizeNEditChange(Sender: TObject);
begin
  pASP^.BubbleSize := BubbleSizeNEdit.AsInteger;
end;

procedure TSeriesEditorForm.SeriesTypeComboBoxClick(Sender: TObject);
begin
  pASP^.SeriesTypeIndex := SeriesTypeComboBox.ItemIndex;

  BubbleSizeNEdit.Visible := (pASP^.SeriesTypeIndex = Ord(stBubble));
  BubbleSizeLabel.Enabled := BubbleSizeNEdit.Visible;
  MultiplicityComboBox.Visible := (pASP^.SeriesTypeIndex = Ord(stMultiple));
  MultiplicityLabel.Enabled := MultiplicityComboBox.Visible;
  SymbolCheckCheckBox.Enabled := (pASP^.SeriesTypeIndex = Ord(stError)) or
    (pASP^.SeriesTypeIndex = Ord(stMultiple));
end;

procedure TSeriesEditorForm.SymbolCheckCheckBoxClick(Sender: TObject);
begin
  pASP^.SymbolCheck := SymbolCheckCheckBox.Checked;
end;

end.
