unit Titles;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are used with permission, subject to the Mozilla
Public License Version 1.1 (the "License"); you may not use this file except
in compliance with the License. You may obtain a copy of the License at

    http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Titles.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 11/09/2002
Current Version: 3.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Purpose:
This unit contains many sub-components:
     TRectangle
     TAngleRect
     TBorder
     TText
     TLegend
     TTitle
     TNote
that manage various screen areas and objects for TPlot.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils,
{$IFDEF GUI}
  {$IFDEF WINDOWS}
  Wintypes,
  TrimStr,
  Controls, Dialogs, Graphics, stdctrls,
  {$ENDIF}
  {$IFDEF WIN32}
  Windows,
  Controls, Dialogs, Graphics, stdctrls,
  {$ENDIF}
  {$IFDEF LINUX}
  Types, Untranslated,
  QControls, QDialogs, QGraphics, Qstdctrls,
  {$ENDIF}
{$ELSE}
  gd, gd_io, gdWindows, gdGraphics, {gdWinapi,}
{$ENDIF}
  {$IFDEF NO_MATH}NoMath,{$ELSE}Math,{$ENDIF}
  Misc, Plotdefs, Polytest;

const
  SIN_30 = 0.5;
  SIN_60 = 0.86602540378443864676372317075294;
  COS_30 = 0.86602540378443864676372317075294;
  COS_60 = 0.5;

type
{TRectangle *******************************************************************}
  TRectangle = class(TCollectionItem) //TPersistent) //
{TRectangle is extremely important to the functioning of TPlot.}
{}
{Not only is it used for a variety of objects, it is also the base class
 of many on-screen objects: Axes, Titles, Captions, Legends, etc.}
{}
{As well as having the properties that one would expect for a rectangle,
 it also has some properties that are useful to its various descendants:
 Name, Tag and Visibility to be precise.}
  private
    //FAlignment: TAlignment;
    FFireEvents: Boolean;
    FLeft: Integer;
    FTop: Integer;
    FHeight: Integer;
    FWidth: Integer;
    FName: String;
{NOTE WELL: FPlot replaces FOwner when we went for Collections:}
    //FOwner: TPersistent;
    //FPlot: TComponent;
    FTag: Longint;
    FVisible: Boolean;

  protected
    FOnChange: TNotifyEvent;
{Events are normally private/published, but D1 does not allow descendants to
 check the assignment of events: FOnChange is inaccessible, and OnChange
 cannot be examined because it is a function.}

{Get functions for virtual properties:}
    function GetCentre: TPoint; virtual;
    function GetBottom: Integer;
    function GetRight: Integer;
    function GetMidX: Integer; virtual;
    function GetMidY: Integer; virtual;

    procedure SetName(Value: String); virtual;
{This sets the name of the TRectangle. This is overridden later to also set the Caption.}

{Set Geometry procedures that we may wish to override later:}
    procedure SetLeft(Value: Integer); virtual;
{This sets the Left screen position property. It also moves the Right by the
 same amount, thereby preserving the Width. Ie: it moves the whole TRectangle.}
    procedure SetMidX(Value: Integer); virtual;
{This sets the MidX screen virtual position property. It thereby moves the Left and Right.}
    procedure SetMidY(Value: Integer); virtual;
{This sets the MidY screen virtual position property. It thereby moves the Top and Bottom.}
    procedure SetTop(Value: Integer); virtual;
{This sets the Top screen position property. It also moves the Bottom by the
 same amount, thereby preserving the Height. Ie: it moves the whole TRectangle.}
    //procedure SetAlignment(Value: TAlignment);
    procedure SetRight(Value: Integer); virtual;
    procedure SetBottom(Value: Integer); virtual;
    procedure SetVisible(Value: Boolean); virtual;

{Set procedures for virtual properties:}
    procedure SetCentre(NewCentre: TPoint); virtual;
    procedure SetHeight(Value: Integer); virtual;
    procedure SetWidth(Value: Integer); virtual;

    procedure DoHandleChange; virtual;
    procedure DoGeometry; virtual;

{All Set methods call this to handle the OnChange logic.}
  public
    property Centre: TPoint read GetCentre write SetCentre;
{The centre of the rectangle.}
    property Tag: Longint read FTag write FTag;
{The usual Tag property, as in TComponent.Tag}
{}
{However, DO NOT USE THIS PROPERTY !
 It is used by TPlot to store the object type, and hence control the visible
 behaviour of the TRectangle descendant.}

{These are Public here, because TBorder is based on the (Top,Left) and {RightGap, BottomGap.
 They are published in some descendants.}
    property Height: Integer read FHeight write SetHeight;
{The standard Height property.}
    property Width: Integer read FWidth write SetWidth;
{The standard Width property.}

    Constructor Create(AOwner: TCollection); override;
{The standard constructor, where standard properties are set.}
    Destructor Destroy; override;
{The standard destructor, where the OnChange event is "freed".}

    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;
{TRectangle's implementation of the standard Assign(To) method.}
    procedure AssignToRect(Dest: TRect);
{TRectangle's implementation of a non-standard AssignTo method.}
    function ClickedOn(iX, iY: Integer): Boolean; virtual;
{Was this TRectangle clicked on ?}
    procedure MoveTo(iX, iY: Integer); virtual;
{Move the rectangle to a new (Top, Left) location.}
    procedure MoveBy(dX, dY: Integer); virtual;
{Move the rectangle by (iX, iY) from (Top, Left) to (Top + iX, Left + iY) .}
{$IFDEF GUI}
    procedure Outline(ACanvas: TCanvas); virtual;
{Draws an Outline around this rectangle.}
{$ENDIF}

  published
    property FireEvents: Boolean read FFireEvents write FFireEvents default TRUE;
{Do we fire events in response to a geometry change ?
 For the TText descendants, the answer is no, because they dance around
 the screen with every repaint.}
    property Name: String read FName write SetName;
{This is what is displayed when the user is offered a choice.
 Eg: "Move the X-Axis Caption or the Bottom Border ?".}
    property Left: Integer read FLeft write SetLeft;
{This is the usual position property.}
    property Top: Integer read FTop write SetTop;
{This is the usual position property.}
    Property Visible: Boolean read FVisible write SetVisible default TRUE;
{Is the Rectangle (or its descendants) visible ?}

{Now the virtual properties:}
    property Right: Integer read GetRight write SetRight stored FALSE;
{This is the usual position property.}
    property Bottom: Integer read GetBottom write SetBottom stored FALSE;
{This is the usual position property.}
    property MidX: Integer read GetMidX write SetMidX stored FALSE;
{The X midpoint of the TRectangle.}
    property MidY: Integer read GetMidY write SetMidY stored FALSE;
{The Y midpoint of the TRectangle.}

    Property OnChange: TNotifyEvent read FOnChange write FOnChange;
{The standard OnChange event (for geometry).}
  end;

{TBorder **********************************************************************}
  TGapPercent = 0..40;
  TBorder = class (TComponent) //(TCollectionItem)
  private
    FFireEvents: Boolean;
    FGap: TGapPercent;
    FGapMin: Word;
    FGapMax: Word;
    FVisible: Boolean;
  protected
    FOnChange: TNotifyEvent;
{Events are normally private/published, but D1 does not allow descendants to
 check the assignment of events: FOnChange is inaccessible, and OnChange
 cannot be examined because it is a function.}
    function GetLeft: Integer;
    function GetTop: Integer;
    function GetRight: Integer;
    function GetBottom: Integer;
    function GetWidth: Integer;
    function GetHeight: Integer;

    procedure DoHandleChange;

    procedure SetGap(Value: TGapPercent);
    procedure SetGapMin(Value: Word);
    procedure SetGapMax(Value: Word);
    procedure SetVisible(Value: Boolean);
  public
    property Left: Integer read GetLeft;
    property Top: Integer read GetTop;
    property Right: Integer read GetRight;
    property Bottom: Integer read GetBottom;
    property Width: Integer read GetWidth;
    property Height: Integer read GetHeight;

    Constructor Create(AOwner: TComponent); override;
{The standard constructor, where standard properties are set.}
  published
    property FireEvents: Boolean read FFireEvents write FFireEvents default TRUE;
{Do we fire events in response to a geometry change ?}
    property Gap: TGapPercent read FGap write SetGap default 10;
    property GapMin: Word read FGapMin write SetGapMin default 60;
    property GapMax: Word read FGapMax write SetGapMax default 200;
    property Visible: Boolean read FVisible write SetVisible default FALSE;
{events:}
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

{TAngleRect *******************************************************************}
{TAngleRect is a TRectangle that has been rotated:
        - by Angle
        - about the Origin "O", where the Origin is defined to be:
      ^                                  Origin.x := FLeft;
      |                                  Origin.y := FTop;
      |
      |                                   - Angle between vertical and Vector (Axis)
      |  __-                              = 0  for Y Axis
     _O-----\------------------>          = 90  for X Axis
  _--  \     \                            = 225  for Z Axis
 \      \     \                           ~ 160 here in this diagram
  \      \     \
   \      \     \                        Real properties are:
    \      \     \                       Angle
     \      \     \
      \      C     \                     Virtual properties are:
       \      \     \                    TrueLeft, TrueTop, TrueRight, TrueBottom,
        \      \     \                   TrueHeight, TrueWidth, TrueCentre,
         \      \     \                  Length, Centre, Vector;
          \      \     \
           \      \     \                Rotation by :
            \      \  __--                [NewX]  =  [cos  sin][X]
             \    __V-                    [NewY]     [-sin cos][Y]
              \_--
}

  TOriginType = (orTopLeft, orMidLeft, orCentre);
{Text boxes get rotated about their top-left corner,
 Axes get rotated about their mid-left side/end,
 Titles on Axes get rotated about centres ?}

  TAngleRect = class(TRectangle)
  private
    FAngle: TDegrees;
{Note: Axes need to be defined in terms of an Origin + Angle;
       Origin.x == FLeft; Origin.y := MidY;
       Labels need to be defined in terms of a Centre and an Angle.}
    FEndX, FEndY: Integer;
    FOrigin: TPoint;
    FSin,
    FCos: Extended;

    mLeftIndex: Byte;
{The index of the left-most point:
          Left, Right, Top, Bottom points are in order:
    mLeftIndex,    +1,  +2,     +3 }

  protected
    mAngleRadians: Single;
    FP: TQuad;
{These are the points that define a TAngleRect. They are calculated in DoGeometry,
 and used internally for Outlining and ClickedOn.}

    //property Origin: TPoint read FOrigin;
{Text boxes get rotated about their top-left corner,
 Axes get rotated about their mid-left side/end,
 Titles on Axes get rotated about centres ?}

    property mSin: Extended read FSin;
{This is the Sine of the Angle, and is calculated in SetAngle.}
    property mCos: Extended read FCos;
{This is the Cosine of the Angle, and is calculated in SetAngle.}

    procedure DoGeometry; override;
{This calculates the FP points, which store the geometry of the TAngleRect.}
    procedure DoHandleChange; override;
{All Set methods call this to handle the OnChange logic.}

    function RotatePoint(APoint: TPoint): TPoint;
    function RotatePointAbout(APoint, AnOrigin: TPoint): TPoint;

{Get functions for virtual properties:}
    function GetCentre: TPoint; override;
    function GetVector: TPoint; virtual;
    function GetLength: Integer; virtual;
    function GetBreadth: Integer; virtual;
    function GetMidX: Integer; override;
{This gets the MidX screen virtual position property}
    function GetMidY: Integer; override;
{This gets the MidY screen virtual position property}

{Set procedures for REAL properties:}
    procedure SetAngle(Value: TDegrees); virtual;
{Sets the angle of the rectangle to the vertical.}
    procedure SetCentre(NewCentre: TPoint); override;
{This sets the centre of the rotated rectangle.}
    procedure SetOrigin(Value: TPoint);
{This sets the Origin REAL screen position property.}
    procedure SetLength(Value: Integer); virtual;
    procedure SetBreadth(Value: Integer); virtual;
    procedure SetVector(Value: TPoint); virtual;
{This sets the Vector virtual screen position property,
 thereby changing the Length and Angle.}
    procedure SetMidX(Value: Integer); override;
{This sets the MidX screen virtual position property.
 It thereby moves the Left and Right.}
    procedure SetMidY(Value: Integer); override;
{This sets the MidY screen virtual position property.
 It thereby moves the Top and Bottom.}

  public
    property EndX: Integer read FEndX;
{The "End" of the rectangle/Axis, as opposed to the "Origin": eg: Right for a X Axis.}
    property EndY: Integer read FEndY;
{The "End" of the Rectangle: eg: Top for a Y Axis.}
    property Origin: TPoint read FOrigin write SetOrigin;
{The origin of the Rectangle: eg: (MidY, Left) for an X Axis.}
{Now the virtual properties:}
    property Vector: TPoint read GetVector write SetVector;
{The Vector of the Rectangle: eg: (Width, 0) for an X Axis.}

    Constructor Create(AOwner: TCollection); override;
{The standard constructor, where standard properties are set.}
    Destructor Destroy; override;
{The standard destructor, where the OnChange event is "freed".}

    function ClickedOn(iX, iY: Integer): Boolean; override;
{Was this TRectangle clicked on ?}
    procedure SetAngles(aAngle: TDegrees; aSin, aCos: Extended);
{This is how we set the angle geometry without doing the hard work of calculating trigs.}
    procedure AssignTo(Dest: TPersistent); override;
{TAngleRect's implementation of the standard Assign(To) method.}
    procedure CopyTo(Dest: TPersistent); 
{This copies TAngleRect's Geometry to Dest, assuming that Dest also has these properties.}
    procedure MoveTo(NewX, NewY: Integer); override;
{Move the rectangle to a new (Top, Left) location.}
    procedure MoveBy(dX, dY: Integer); override;
{Move the rectangle by (iX, iY) from (Top, Left) to (Top + iX, Left + iY) .}
    //procedure SetNewGeometry(NewOrigin: TPoint; NewAngle: TDegrees; NewLength, NewBreadth: Integer);
{Move the rectangle to a new position and angle.}
{$IFDEF GUI}
    procedure Outline(ACanvas: TCanvas); override;
{Draws an Outline around this rectangle.}
{$ENDIF}
    procedure Draw(ACanvas: TCanvas); virtual;
{Draws this rotated rectangle.}

  published
{Real properties:}
    property Angle: TDegrees read FAngle write SetAngle default 90;
{The angle between the rotated rectangle and the vertical.}
    property Length: Integer read GetLength write SetLength stored FALSE;
{Length is less confusing than Width for a Y or Z Axis - but (in our definitions) it is the same thing.}
    property Breadth: Integer read GetBreadth write SetBreadth stored FALSE;
{Breadth is less confusing than Width for a Y or Z Axis - but (in our definitions) it is the same thing.}

end;

{TText *********************************************************************}
  TText = class(TAngleRect)
{A TText inherits the positional behaviour of TRectangle,
 and adds a Caption and a Font.}
  private
    FBorder: Boolean;
    FCaption: String;
    FFont: TFont;

    FOnCaptionChange: TNotifyEvent;

{the in-place editor:}
{$IFDEF GUI}
    FTextEdit: TEdit;
    FTextEditFinshed: Boolean;
{$ENDIF}
    mMaxTextLen, mTextLines: Integer;

  protected
{This sets the name after the Caption is set.}
    function GetCaption: String; virtual;

    procedure SetBorder(Value: Boolean);
    procedure SetCaption(Value: String); virtual;
{This sets the Caption and calls CreateName.}
    procedure SetFont(Value: TFont);
{This sets the font.}
{$IFDEF GUI}
    procedure TextEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;
    procedure EndEdit;
{$ENDIF}

  public
    constructor Create(AOwner: TCollection); override;
    destructor Destroy; override;

{The standard constructor, where standard properties are set.}
    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;
{TText's implementation of the standard Assign(To) method.}
    procedure Draw(ACanvas: TCanvas); override;
{Draw this axis.}
    procedure DoCanvasGeometry(ACanvas: TCanvas);
{$IFDEF GUI}
    procedure StartEdit(AParent: TWinControl);
{$ENDIF}

  published
    property Border: Boolean read FBorder write SetBorder;
{Do we draw a Border around the text ?}
    property Caption: String read GetCaption write SetCaption;
{This is the text that is displayed on the screen.
 Eg: "X-Axis". Setting the Caption also sets the Name:}
{}
{ FName := FCaption + ' Caption';}
    property Font: TFont read FFont write SetFont;
{The font used to display the caption.}
    property OnCaptionChange: TNotifyEvent read FOnCaptionChange write FOnCaptionChange;
{Has the Caption changed ?}
  end;

{Begin TTitle Declarations ****************************************************}
  TTitle = class(TText)
{This is an extended TText that dances around the screen depending on
 Alignment, Orientation and Direction, and Draws itself.}
  private
    FAxis: TCollectionItem;
    FDefaultSide: Boolean;
    FUnits: String;
    FPosition: TPercent;
  protected
    function GetUnits: String;

    procedure SetName(Value: String); override;
    procedure SetPosition(Value: TPercent);
    procedure SetDefaultSide(Value: Boolean);
    procedure SetUnits(Value: String);
{This handles the tricky question of Caption and Units. If Value contains a
 pair of brackets '()', then the contents of these brackets is taken to be the
 Units. If it doesn't, then the FullCaption is constructed from Value and Units.}

  public
    Property Axis: TCollectionItem read FAxis;
{What Axis is this label linked to ? If any (could be global title).}

    constructor Create(Axis: TCollectionItem); reintroduce;
{The standard constructor, where standard properties are set.}

    procedure AssignTo(Dest: TPersistent); override;
{TTitle's implementation of the standard Assign(To) method.}
    //procedure Draw(ACanvas: TCanvas); override;
{This Draws the TTitle on the given Canvas. It calls DoGeometry, and also
 various API functions to create a vertical font if neccessary.}
    {procedure Assign(Source: TPersistent); override;}

  published
    property DefaultSide: Boolean read FDefaultSide write SetDefaultSide default TRUE;
{Is the caption on the Default Side ?}
{}
{Note that the "Default Side" of a Plot Title it the Top, and that of an Axis
 depends on the Angle and the screen location of the Axis.}
    Property Position: TPercent read FPosition write SetPosition default 50;
{This is how far along the Axis (or Plot) that the centre of the Title is located.}
    property Units: String read GetUnits write SetUnits;
{These are the physical units, eg: mm, mV, seconds, etc, of the Axis.}
  end;

{TLegend **********************************************************************}
  TLegend = class(TAngleRect)
{This is an extended TText that dances around the screen depending on
 Alignment, Orientation and Direction, and Draws itself.}
  private
    FCheckboxes: Boolean;
    FDirection : TDirection;
    FFont: TFont;
    FFontHeight: Integer;
    FStringWidth: Integer;
    FCheckWidth: Integer;
    FLineWidth: Integer;
    FSeriesList: TCollection;

{the in-place editor:}
{$IFDEF GUI}
    FLegendEdit: TEdit;
    FLegendEditFinshed: Boolean;
    TheEditSeries: Integer;
{$ENDIF}

  protected
    function GetItemWidth: Integer;
    //function GetDirection: TDirection;
    procedure SetCheckboxes(Value: Boolean);
    procedure SetDirection(Value: TDirection);
    procedure SetFont(Value: TFont);

{$IFDEF GUI}
    procedure LegendEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState); virtual;
    procedure EndEdit;
{$ENDIF}

  public
    property FontHeight: Integer read FFontHeight;
    property ItemWidth: Integer read GetItemWidth;

    constructor CreateList(AOwner: TCollection; SeriesList: TCollection); virtual;
    destructor Destroy; override;

    procedure AssignTo(Dest: TPersistent); override;
{TLegend's implementation of the standard Assign(To) method.}
    function GetHit(iX, iY: Integer; var TheRect: TRect): Integer;
{The rectangle of the series name under the point iX, iY.}
    procedure Draw(ACanvas: TCanvas); override;
{$IFDEF GUI}
    procedure StartEdit(AParent: TWinControl; ThePoint: TPoint);
{$ENDIF}

  published
    property CheckBoxes: Boolean read FCheckBoxes write SetCheckBoxes default TRUE;
{Display Checkboxes ?}
    property Direction : TDirection read FDirection write SetDirection default drVertical;
{Is the Legend drawn Horizontal or Vertical ?}
    property Font: TFont read FFont write SetFont;
{The font used to display the caption.}
  end;

{TNote ************************************************************************}
  TNote = class(TText)
  private
    FArrowLeft: Integer;
    FArrowTop: Integer;
    FArrowLeftReal: Single;
    FArrowTopReal: Single;
    FLeftReal: Single;
    FTopReal: Single;

    ArrowStartLeft: Integer;
    ArrowStartTop: Integer;

  protected
    procedure SetLeft(Value: Integer); override;
    procedure SetTop(Value: Integer); override;
    procedure SetArrowLeft(Value: Integer); virtual;
    procedure SetArrowTop(Value: Integer); virtual;
    procedure SetArrowLeftReal(Value: Single); virtual;
    procedure SetArrowTopReal(Value: Single); virtual;
    procedure SetLeftReal(Value: Single); virtual;
    procedure SetTopReal(Value: Single); virtual;

  public
    constructor Create(AOwner: TCollection); override;
    {destructor Destroy; override;}
    procedure AssignTo(Dest: TPersistent); override;

    {procedure Assign(Source: TPersistent); override;}
    procedure Draw(ACanvas: TCanvas); override;
{This Draws the TNote on the given Canvas. It calls DoGeometry, and also
 various API functions to create a vertical font if neccessary.}
    procedure TracePointerTo(ACanvas: TCanvas; iX, iY: Integer);
  published
    property ArrowLeft: Integer read FArrowLeft write SetArrowLeft;
    property ArrowTop: Integer read FArrowTop write SetArrowTop;
    property ArrowLeftReal: Single read FArrowLeftReal write SetArrowLeftReal;
    property ArrowTopReal: Single read FArrowTopReal write SetArrowTopReal;
    property LeftReal: Single read FLeftReal write SetLeftReal;
    property TopReal: Single read FTopReal write SetTopReal;
  end;

implementation

uses
  Axis, Data, Plot;


{Constructor and Destructor:---------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TRectangle.Create
  Description: Standard Constructor for TRectangle
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues: TCollection.Add does not call this: it has the wrong signature !
 ------------------------------------------------------------------------------}
Constructor TRectangle.Create(AOwner: TCollection);
begin
{First call the ancestor:}
//  if (AOwner is TCollection) then
  inherited Create(AOwner); 

  //FOwner := AOwner;

{we insert the default values that cannot be "defaulted":}
  //FAlignment := taCenter;
  FLeft := 10;
  FTop := 10;
  FWidth := 100;
  FHeight := 100;
  FVisible := TRUE;
{global change event handler:}
  FOnChange := nil;
{we do fire events with a geometry change:}
  FireEvents := TRUE;
end;

{------------------------------------------------------------------------------
   Destructor: TRectangle.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: frees the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TRectangle.Destroy;
begin
  FOnChange := nil;
{then call ancestor:}
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TRectangle.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FLeft := TRectangle(Source).Left;
  FTop := TRectangle(Source).Top;
  FRight := TRectangle(Source).Right;
  FBottom := TRectangle(Source).Bottom;
end;}

{------------------------------------------------------------------------------
    Procedure: TRectangle.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.AssignTo(Dest: TPersistent);
begin
{we DON'T call the ancestor, because TPersistent.AssignTo simply throws an
 exception:
  inherited AssignTo(Dest);}
  TRectangle(Dest).Left := FLeft;
  TRectangle(Dest).Top := FTop;
  TRectangle(Dest).Height := FHeight;
  TRectangle(Dest).Width := FWidth;
  TRectangle(Dest).FireEvents := FFireEvents;
  TRectangle(Dest).Visible := FVisible;
  TRectangle(Dest).OnChange := FOnChange;
  //TRectangle(Dest).Name := FName;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.AssignToRect
  Description: non-standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2001
Date modified: 07/06/2001 by Mat Ballard
      Purpose: implements a non-standard AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.AssignToRect(Dest: TRect);
begin
  Dest.Left := FLeft;
  Dest.Top := FTop;
  Dest.Bottom := GetBottom;
  Dest.Right := GetRight;
end;

{Begin Set and Get Functions and Procedures----------------------------------}
{Get Functions for virtual properties ---------------------------------------}
{------------------------------------------------------------------------------
     Function: TRectangle.GetHeight
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Height, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetCentre: TPoint;
begin
  Result.x := GetMidX;
  Result.y := GetMidY;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetHeight
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Height, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetBottom: Integer;
begin
  Result := FTop + FHeight;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetWidth
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Width, which is a virtual property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetRight: Integer;
begin
  Result := FLeft + FWidth;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetMidX
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the MidX, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetMidX: Integer;
begin
  Result := FLeft + FWidth div 2;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetMidY
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the MidY, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetMidY: Integer;
begin
  Result := FTop + FHeight div 2;
end;

{Set Procedures -------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TRectangle.SetName
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/25/2002
Date modified: 08/25/2002 by Mat Ballard
      Purpose: sets the Name; not needed in TRectangle, but in descendants
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetName(Value: String);
begin
  FName := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetAlignment
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Alignment
 Known Issues:
 ------------------------------------------------------------------------------}
(*procedure TRectangle.SetAlignment(Value: TAlignment);
begin
  if (Value <> FAlignment) then
  begin
    FAlignment := Value;
    DoHandleChange;
  end;
end;*)

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetLeft
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Left
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetLeft(Value: Integer);
begin
  if (Value <> FLeft) then
  begin
    FLeft := Value;
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetTop
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Top
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetTop(Value: Integer);
begin
  if (Value <> FTop) then
  begin
    FTop := Value;
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetRight
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Right, a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetRight(Value: Integer);
var
  OldValue: Integer;
begin
  OldValue := FLeft + FWidth;
  if (Value <> OldValue) then
  begin
    Inc(FWidth, Value - OldValue);
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetBottom
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Bottom, a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetBottom(Value: Integer);
var
  OldValue: Integer;
begin
  OldValue := FTop + FHeight;
  if (Value <> OldValue) then
  begin
    Inc(FHeight, Value - OldValue);
    //DoGeometry;
    DoHandleChange;
  end;
end;

{Set procedures for virtual properties ---------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TRectangle.SetCentre
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Centre Property
 Known Issues: we do this by changing the origin
 ------------------------------------------------------------------------------}
procedure TRectangle.SetCentre(NewCentre: TPoint);
var
  OldValue: TPoint;
begin
  OldValue.x := FLeft + FWidth div 2;
  OldValue.y := FTop + FHeight div 2;
  if ((OldValue.x <> NewCentre.x) or (OldValue.y <> NewCentre.y)) then
    MoveBy((NewCentre.x - OldValue.x), (NewCentre.y - OldValue.y));
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetHeight
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Height
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetHeight(Value: Integer);
begin
  if ((Value > 0) and (FHeight <> Value)) then
  begin
    FHeight := Value;
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetWidth
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Width
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetWidth(Value: Integer);
begin
  if ((Value > 0) and (Value <> FWidth)) then
  begin
    FWidth := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetMidX
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the MidX, a virtual property, by moving the Left
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetMidX(Value: Integer);
var
  OldMidX: Integer;
begin
  OldMidX := FLeft + FWidth div 2;
  if (Value <> OldMidX) then
  begin
    Inc(FLeft, Value - OldMidX);
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetMidY
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the MidY, a virtual property, by moving the Top and Bottom
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetMidY(Value: Integer);
var
  OldMidY: Integer;
begin
  OldMidY := FTop + FHeight div 2;
  if (Value <> OldMidY) then
  begin
    Inc(FTop, Value - OldMidY);
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetVisible
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Visibility
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetVisible(Value: Boolean);
begin
  if (FVisible <> Value) then
  begin
    FVisible := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.ClickedOn
  Description: Was this TRectangle clicked on ?
       Author: Mat Ballard
 Date created: 01/22/2001
Date modified: 01/22/2001 by Mat Ballard
      Purpose: screen click management
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.ClickedOn(iX, iY: Integer): Boolean;
begin
  Result := ((FVisible) and
    (FLeft <= iX) and
    (iX <= (FLeft + FWidth)) and
    (FTop <= iY) and
    (iY <= (FTop + FHeight)));
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TRectangle.Outline
  Description: Draws an Outline around this rectangle
       Author: Mat Ballard
 Date created: 01/22/2001
Date modified: 01/22/2001 by Mat Ballard
      Purpose: gives the user a guide to what they are moving with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.Outline(ACanvas: TCanvas);
begin
  ACanvas.Pen.Color := clBlack;
  ACanvas.Pen.Mode := pmNotXOR;
  ACanvas.Pen.Style := psDash;
  ACanvas.MoveTo(FLeft, FTop);
  ACanvas.LineTo(FLeft+FWidth, FTop);
  ACanvas.LineTo(FLeft+FWidth, FTop+FHeight);
  ACanvas.LineTo(FLeft, FTop+FHeight);
  ACanvas.LineTo(FLeft, FTop);
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TRectangle.MoveTo
  Description: Move the rectangle to a new (Top, Left) location.
       Author: Mat Ballard
 Date created: 06/12/2001
Date modified: 06/12/2001 by Mat Ballard
      Purpose: location / geometry management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.MoveTo(iX, iY: Integer); 
begin
  if not ((FLeft = iX) and (FTop = iY)) then
  begin
    FLeft := iX;
    FTop := iY;
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.MoveBy
  Description: Move the rectangle by (iX, iY) from (Top, Left) to (Top + iX, Left + iY).
       Author: Mat Ballard
 Date created: 06/12/2001
Date modified: 06/12/2001 by Mat Ballard
      Purpose: location / geometry management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.MoveBy(dX, dY: Integer);
begin
  if ((dX = 0) and (dY = 0)) then
  begin
    Inc(FLeft, dX);
    Inc(FTop, dY);
    //DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.DoHandleChange
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/27/2000
Date modified: 02/27/2000 by Mat Ballard
      Purpose: all Change Event firing passes through here
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.DoHandleChange;
begin
{Imagine that a Border changes: since it is not visible, we cannot leave DoGeometry up to Draw.
 Similarly, if a Canvas changes (eg: printing), we have to re-do geometry upon draw,
   even if TRectangle soe not change !}
  DoGeometry;
{NOTE: once all this lot is working, let's try to remove DoGeometry from here.}  
  if (FireEvents and assigned(FOnChange) and FVisible) then
  begin
    OnChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.DoGeometry
  Description: geometry manager
       Author: Mat Ballard
 Date created: 02/27/2000
Date modified: 02/27/2000 by Mat Ballard
      Purpose: overridden by some descendants
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.DoGeometry;
begin

end;

{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TBorder.Create
  Description: Standard Constructor for TBorder
       Author: Mat Ballard
 Date created: 09/25/2003
Date modified: 09/25/2003 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TBorder.Create(AOwner: TComponent);
begin
{Note: the owner is TPlot !}
  inherited Create(AOwner);
  FFireEvents := TRUE;
  FGap := 10;
  FGapMin := 50;
  FGapMax := 200;
  FVisible := FALSE;
end;

procedure TBorder.DoHandleChange;
begin
  if Assigned(FOnChange) then OnChange(Self);
end;

function TBorder.GetLeft: Integer;
var
  TheGap: Integer;
begin
  TheGap := FGap * TPlot(Owner).Width div 100;
  if (TheGap < FGapMin) then
    TheGap := FGapMin;
  if (TheGap > FGapMax) then
    TheGap := FGapMax;
  Result := TheGap;
end;

function TBorder.GetTop: Integer;
var
  TheGap: Integer;
begin
  TheGap := FGap * TPlot(Owner).Height div 100;
  if (TheGap < FGapMin) then
    TheGap := FGapMin;
  if (TheGap > FGapMax) then
    TheGap := FGapMax;
  Result := TheGap;
end;

function TBorder.GetRight: Integer;
var
  TheGap: Integer;
begin
  TheGap := FGap * TPlot(Owner).Width div 100;
  if (TheGap < FGapMin) then
    TheGap := FGapMin;
  if (TheGap > FGapMax) then
    TheGap := FGapMax;
  Result := TPlot(Owner).Width - TheGap;
end;

function TBorder.GetBottom: Integer;
var
  TheGap: Integer;
begin
  TheGap := FGap * TPlot(Owner).Width div 100;
  if (TheGap < FGapMin) then
    TheGap := FGapMin;
  if (TheGap > FGapMax) then
    TheGap := FGapMax;
  Result := TPlot(Owner).Height - TheGap;
end;

function TBorder.GetWidth: Integer;
var
  TheGap: Integer;
begin
  TheGap := FGap * TPlot(Owner).Width div 100;
  if (TheGap < FGapMin) then
    TheGap := FGapMin;
  if (TheGap > FGapMax) then
    TheGap := FGapMax;
  Result := TPlot(Owner).Width - 2*TheGap;
end;

function TBorder.GetHeight: Integer;
var
  TheGap: Integer;
begin
  TheGap := FGap * TPlot(Owner).Height div 100;
  if (TheGap < FGapMin) then
    TheGap := FGapMin;
  if (TheGap > FGapMax) then
    TheGap := FGapMax;
  Result := TPlot(Owner).Height - 2*TheGap;
end;

procedure TBorder.SetGap(Value: TGapPercent);
begin
  FGap := Value;
  DoHandleChange;
end;

procedure TBorder.SetGapMin(Value: Word);
begin
  FGapMin := Value;
  DoHandleChange;
end;

procedure TBorder.SetGapMax(Value: Word);
begin
  FGapMax := Value;
  DoHandleChange;
end;

procedure TBorder.SetVisible(Value: Boolean);
begin
  FVisible := Value;
  DoHandleChange;
end;

{TBorder ends -----------------------------------------------------------------}

{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TAngleRect.Create
  Description: Standard Constructor for TRectangle
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TAngleRect.Create(AOwner: TCollection);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{we insert the default values that cannot be "defaulted":}
  Angle := 90;
end;

{------------------------------------------------------------------------------
   Destructor: TAngleRect.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: frees the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TAngleRect.Destroy;
begin
  FOnChange := nil;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TAngleRect(Dest).Angle := FAngle;
  //TAngleRect(Dest).Origin := FOrigin;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.CopyTo
  Description: like an AssignTo method, but only copies Geometry
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 09/06/2002 by Mat Ballard
      Purpose: assignment of Clicked Object geometry to Selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.CopyTo(Dest: TPersistent);
begin
  TAngleRect(Dest).Left := FLeft;
  TAngleRect(Dest).Top := FTop;
  TAngleRect(Dest).Height := FHeight;
  TAngleRect(Dest).Width := FWidth;
  TAngleRect(Dest).Angle := FAngle;
  TAngleRect(Dest).DoGeometry;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.DoHandleChange
  Description: Sets several properties at once
       Author: Mat Ballard
 Date created: 02/27/2000
Date modified: 02/27/2000 by Mat Ballard
      Purpose: This is how we set the angle geometry without doing the hard work of calculating trigs.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetAngles(aAngle: TDegrees; aSin, aCos: Extended);
begin
  FAngle := aAngle;
  mAngleRadians := Pi * FAngle / 180;
  FSin := aSin;
  FCos := aCos;
  DoGeometry;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRectangle.DoHandleChange
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/27/2000
Date modified: 02/27/2000 by Mat Ballard
      Purpose: all Change Event firing passes through here
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.DoHandleChange;
begin
  //DoGeometry;
  if (FireEvents and assigned(FOnChange) and FVisible) then OnChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.DoGeometry
  Description: works out where the corners of the rectangle are.
       Author: Mat Ballard
 Date created: 06/08/2001
Date modified: 06/08/2000 by Mat Ballard
      Purpose: geometry management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.DoGeometry;
var
  i: Integer;
begin
  if (FAngle = 90) then
  begin // X Axis, or normal object
    FP[0].x := FLeft;
    FP[0].y := FTop;
    FP[1].x := FP[0].x + FWidth;
    FP[1].y := FP[0].y;
    FP[2].x := FP[1].x;
    FP[2].y := FP[0].y + FHeight;
    FP[3].x := FP[0].x;
    FP[3].y := FP[2].y;
    FEndX := FP[1].x;
    FEndY := (FP[1].y + FP[2].y) div 2;
    FOrigin.x := FP[0].x;
    FOrigin.y := (FP[0].y + FP[3].y) div 2;
  end
  else
  begin // Y or Z Axis, or vertical Y Axis Title
    FP[1].x := FLeft;
    FP[1].y := FTop;
    FP[0].x := FP[1].x;
    FP[0].y := FP[1].y + Length;
    FP[2].x := FP[1].x + Breadth;
    FP[2].y := FP[1].y;
    FP[3].x := FP[2].x;
    FP[3].y := FP[0].y;
    FOrigin.x := (FP[0].x + FP[3].x) div 2;
    FOrigin.y := FP[0].y;
    if (FAngle <> 0) then
    begin  // Z Axis or Title
      for i := 0 to 3 do
      begin
        FP[i] := RotatePointAbout(FP[i], FOrigin);
      end;
    end;
    FEndX := (FP[1].x + FP[2].x) div 2; // but End does move
    FEndY := (FP[1].y + FP[2].y) div 2;
  end;
end;

{------------------------------------------------------------------------------
     Function: TAngleRect.RotatePoint
  Description: rotates a point about the Centre
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: geometry management
 Return Value: TPoint;
 Known Issues:
    Equations: Rotation by :
                [NewX]  =  [cos  sin][X]
                [NewY]     [-sin cos][Y]
 ------------------------------------------------------------------------------}
function TAngleRect.RotatePoint(APoint: TPoint): TPoint;
begin
  Result.x := Round(FCos * APoint.x - FSin * APoint.y);
  Result.y := Round(FSin * APoint.x + FCos * APoint.y);
end;

{------------------------------------------------------------------------------
     Function: TAngleRect.RotatePointAbout
  Description: rotates a point about an arbitrary point
       Author: Mat Ballard
 Date created: 10/28/2001
Date modified: 10/28/2001 by Mat Ballard
      Purpose: geometry management
 Return Value: TPoint;
 Known Issues:
 ------------------------------------------------------------------------------}
function TAngleRect.RotatePointAbout(APoint, AnOrigin: TPoint): TPoint;
begin
  APoint.x := APoint.x - AnOrigin.x;
  APoint.y := APoint.y - AnOrigin.y;
  Result.x := Round(FCos * APoint.x - FSin * APoint.y);
  Result.y := Round(FSin * APoint.x + FCos * APoint.y);
  Result.x := Result.x + AnOrigin.x;
  Result.y := Result.y + AnOrigin.y;
end;

{Begin Set and Get Functions and Procedures----------------------------------}
{------------------------------------------------------------------------------
     Function: TAngleRect.GetVector
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 06/06/2001 by Mat Ballard
      Purpose: Gets the Vector, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TAngleRect.GetVector: TPoint;
begin
  Result.x := FP[1].x - FP[0].x;
  Result.y := FP[1].y - FP[0].y;
end;

{Get Functions for VIRTUAL properties ---------------------------------------}
function TAngleRect.GetLength: Integer;
begin
  if (FAngle = 0) then
    Result := FHeight
   else
    Result := FWidth;
end;

function TAngleRect.GetBreadth: Integer;
begin
  if (FAngle = 0) then
    Result := FWidth
   else
    Result := FHeight;
end;

{This gets the MidX screen virtual position property}
function TAngleRect.GetMidX: Integer;
begin
  Result := (FP[0].x + FP[1].x + FP[2].x + FP[3].x) shr 2;
end;

{This gets the MidY screen virtual position property}
function TAngleRect.GetMidY: Integer;
begin
  Result := (FP[0].y + FP[1].y + FP[2].y + FP[3].y) shr 2;
end;

{This gets the new True centre of the Rectangle.}
function TAngleRect.GetCentre: TPoint;
begin
  Result.x := GetMidX;
  Result.y := GetMidY;
end;

{Set Procedure for REAL properties ---------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAngleRect.SetAngle
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Angle Property: clockwise angle between vertical and long axis
 Known Issues: Windows defines angles for fonts ANTICLOCKWISE in 0.1 increments from X Axis;
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetAngle(Value: TDegrees);
begin
  if (FAngle <> Value) then
  begin
    FAngle := Value;
    mAngleRadians := Pi * FAngle / 180;
    if (FAngle = 0) then
    begin
      FSin := 0;
      FCos := 1;
    end
    else if (FAngle = 90) then
    begin
      FSin := 1;
      FCos := 0;
    end
    else if (FAngle = 180) then
    begin
      FSin := 0;
      FCos := -1;
    end
    else if (FAngle = 270) then
    begin
      FSin := -1;
      FCos := 0;
    end
    else
    begin
  {this is twice as fast as calling them individually:}
      SinCos(mAngleRadians, FSin, FCos);
    end;

{What is the index of the "Left" point ? (order: Left, Top, Right, Bottom)}
    case FAngle of
      1..90: mLeftIndex := 0;
      91..180: mLeftIndex := 3;
      181..270: mLeftIndex := 2;
      0, 271..359: mLeftIndex := 1;
    end;

    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.SetCentre
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Centre Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetCentre(NewCentre: TPoint);
var
  OldValue: TPoint;
begin
  OldValue.x := GetMidX;
  OldValue.y := GetMidY;
  if ((OldValue.x <> NewCentre.x) or (OldValue.y <> NewCentre.y)) then
    MoveBy((NewCentre.x - OldValue.x), (NewCentre.y - OldValue.y));
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.SetOrigin
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Origin, a virtual property, by moving the Right
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetOrigin(Value: TPoint);
begin
  if ((Origin.x <> Value.x) or (Origin.y <> Value.y)) then
  begin
    if (FAngle = 90) then
    begin // X Axis, or normal object
      FLeft := Value.x;
      FTop := Value.y - Breadth div 2;
    end
    else
    begin // Y or Z Axis, or vertical Y Axis Title
      FTop := Value.y - Length;
      FLeft := Value.x - Breadth div 2;
    end;
    DoHandleChange;
  end;
end;

{Set Procedures for VIRTUAL properties ----------------------------------------}
procedure TAngleRect.SetLength(Value: Integer);
begin
  if (FAngle = 0) then
  begin
    if (FHeight <> Value) then
    begin
      FHeight := Value;
      DoGeometry;
      DoHandleChange;
    end;
  end
  else
  begin
    if (FWidth <> Value) then
    begin
      FWidth := Value;
      DoGeometry;
      DoHandleChange;
    end;
  end;
end;

procedure TAngleRect.SetBreadth(Value: Integer);
begin
  if (FAngle = 0) then
  begin
    if (FWidth <> Value) then
    begin
      FWidth := Value;
      DoHandleChange;
    end;
  end
  else
  begin
    if (FHeight <> Value) then
    begin
      FHeight := Value;
      DoHandleChange;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.SetMidX
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the MidX
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetMidX(Value: Integer);
var
  OldValue: Integer;
begin
  OldValue := GetMidX;
  if (OldValue <> Value) then
  begin
    FLeft := Value - FWidth div 2;
    DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.SetMidY
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the MidY property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetMidY(Value: Integer);
var
  OldValue: Integer;
begin
  OldValue := GetMidY; // WRONG ! FP has not been calculated yet !
  if (OldValue <> Value) then
  begin
    FTop := Value - FHeight div 2;
    DoGeometry;
    DoHandleChange;
  end;
end;

{Set Procedures for VIRTUAL properties ---------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAngleRect.SetVector
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 06/06/2001
Date modified: 06/06/2001 by Mat Ballard
      Purpose: sets the Vector, a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.SetVector(Value: TPoint);
begin
  Angle := Round(GetAngleDeg(Value.x, Value.y));
  Length := Round(Sqrt(Value.x*Value.x + Value.y*Value.y));
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.ClickedOn
  Description: Was this TRectangle clicked on ?
       Author: Mat Ballard
 Date created: 01/22/2001
Date modified: 01/22/2001 by Mat Ballard
      Purpose: screen click management
 Known Issues: does not work yet !
 ------------------------------------------------------------------------------}
function TAngleRect.ClickedOn(iX, iY: Integer): Boolean;
begin
{$IFDEF GUI}
  if ((FAngle = 0) or (FAngle = 90)) then
    Result := inherited ClickedOn(iX, iY)
  else
  begin
    Result := PolyTest.PointInPolygonTest(iX, iY, 4, FP);
  end;
{$ELSE}
  Result := FALSE;
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.MoveTo
  Description: Move the rectangle to a new (Top, Left) location.
       Author: Mat Ballard
 Date created: 06/12/2001
Date modified: 06/12/2001 by Mat Ballard
      Purpose: location / geometry management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.MoveTo(NewX, NewY: Integer);
begin
  if ((FLeft <> NewX) or (FTop <> NewY)) then
  begin
    FLeft := NewX;
    FTop := NewY;
    DoGeometry;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TAngleRect.MoveBy
  Description: Move the rectangle by (iX, iY) from (Top, Left) to (Top + iX, Left + iY).
       Author: Mat Ballard
 Date created: 06/12/2001
Date modified: 06/12/2001 by Mat Ballard
      Purpose: location / geometry management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.MoveBy(dX, dY: Integer);
begin
  if ((dX <> 0) or (dY <> 0)) then
  begin
    Inc(FLeft, dX);
    Inc(FTop, dY);
    DoGeometry;
  end;
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TAngleRect.Outline
  Description: Draws an Outline around this rectangle
       Author: Mat Ballard
 Date created: 01/22/2001
Date modified: 01/22/2001 by Mat Ballard
      Purpose: gives the user a guide to what they are moving with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.Outline(ACanvas: TCanvas);
begin
  ACanvas.Pen.Color := clBlack;
  ACanvas.Pen.Mode := pmNotXOR;
  ACanvas.Pen.Style := psDot;
  ACanvas.PolyLine([FP[0], FP[1], FP[2], FP[3], FP[0]]);
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TAngleRect.Draw
  Description: Draws this rectangle
       Author: Mat Ballard
 Date created: 01/22/2001
Date modified: 01/22/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAngleRect.Draw(ACanvas: TCanvas);
begin
  ACanvas.PolyLine(Self.FP);
end;

{TText -------------------------------------------------------------------}
{Constructor and Destructor -------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TText.Create
  Description: Standard Constructor for TText
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TText.Create(AOwner: TCollection);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{Create font:}
  FFont := TFont.Create;
  FFont.Name := sArial;
  FFont.Size := SMALL_FONT_SIZE;

{$IFDEF GUI}  
  FTextEdit := nil;
  FTextEditFinshed := FALSE;
{$ENDIF}
end;

destructor TText.Destroy;
begin
  FFont.Free;
  inherited Destroy;
end;

{Begin Get Functions -------------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TText.GetCaption
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Caption of TText
 Known Issues:
 ------------------------------------------------------------------------------}
function TText.GetCaption: String;
begin
  Result := FCaption;
end;
{End Get Functions -------------------------------------------------------}

{Begin Set Procedures -------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TText.SeTBorder
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/15/2001
Date modified: 09/15/2002 by Mat Ballard
      Purpose: Does the text have a border ?
 Known Issues: This used to be in TNote
 ------------------------------------------------------------------------------}
procedure TText.SetBorder(Value: Boolean);
begin
  if (Value <> FBorder) then
  begin
    FBorder := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TText.SetCaption
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Caption of TText
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.SetCaption(Value: String);
var
  iLen, iPos: Integer;
begin
  if (Value <> FCaption) then
  begin
    FCaption := Value;
    //CreateName;
{establish the dimensions of the text:}
    iPos := 1;
    iLen := 0;
    mMaxTextLen := 0;
    mTextLines := 1;
    while (iPos <= System.Length(Value)) do
    begin
      if (Value[iPos] = #10) then
      begin
        Inc(mTextLines);
        if (mMaxTextLen < iLen) then mMaxTextLen := iLen;
        iLen := 0;
      end
      else
      begin
        Inc(iLen);
      end;
      Inc(iPos);
    end;
{a single line has no #10:}
  if (mMaxTextLen < iLen) then mMaxTextLen := iLen;
    DoHandleChange;
    if assigned(FOnCaptionChange) then OnCaptionChange(Self);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TText.SetFont
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Font
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
  DoHandleChange;
end;

{General purpose methods ------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TText.CreateName
  Description: protected procedure to set a useful name
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Name, generally in response to a Caption change
 Known Issues:
 ------------------------------------------------------------------------------}
(*procedure TText.CreateName;
begin
{eg: Caption: X Axis
     Name: X Axis Caption.}
  Name := FCaption + ' Caption';
end;*)

{------------------------------------------------------------------------------
    Procedure: TText.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TText.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FCaption := TText(Source).Caption;
  FFont.Assign(TText(Source).Font);
end;}

{------------------------------------------------------------------------------
    Procedure: TText.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TText(Dest).Caption := FCaption;
  TText(Dest).Font.Assign(FFont);
  TText(Dest).OnCaptionChange := FOnCaptionChange;
end;

{------------------------------------------------------------------------------
    Procedure: TText.Draw
  Description: public Drawing method
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Draws the Caption, either horizontally or vertically, at the desired position
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.DoCanvasGeometry(ACanvas: TCanvas);
var
  TheText: String;
  TextSize: TSize;
begin
  TheText := GetCaption;
  if (System.Length(TheText) = 0) then exit;

{Set up font and brush:}
  ACanvas.Font.Assign(FFont);
  ACanvas.Brush.Style := bsClear;

{calculate textbox size:}
  TextSize := ACanvas.TextExtent(TheText);
  if (mTextLines > 1) then
  begin
    Self.Length := mMaxTextLen * TextSize.cx div System.Length(TheText);
    Self.Breadth := mTextLines * TextSize.cy;
  end
  else
  begin
    Self.Length := TextSize.cx;
    Self.Breadth := TextSize.cy;
  end;

{This may not be needed:}
  DoGeometry;
end;

{------------------------------------------------------------------------------
    Procedure: TText.Draw
  Description: public Drawing method
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Draws the Caption, either horizontally or vertically, at the desired position
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.Draw(ACanvas: TCanvas);
var
  iY, TextHeight: Integer;
  TheText: String;
  NewAngle: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TText.Draw: ACanvas is nil !');
{$ENDIF}
{$IFDEF GUI}
  if (FTextEditFinshed) then
    EndEdit;
{$ENDIF}
  if (not FVisible) then exit;

  TheText := GetCaption;
  TextHeight := ACanvas.Font.Height;
  if (FAngle = 90) then
  begin
{output text to screen:}
    iY := Top;
{$IFDEF LINUX}
    //Dec(iY, Abs(TextHeight));
{$ENDIF}
    while (Pos(#10, TheText) > 0) do
    begin
      ACanvas.TextOut(Left, iY, GetWord(TheText, #10));
      Inc(iY, Abs(TextHeight));
    end;
    ACanvas.TextOut(Left, iY, TheText);
  end
  else if (FAngle = 0) then
  begin
{$IFDEF GUI}
    Misc.TextOutAngle(ACanvas, 90, FP[0].x, FP[0].y, TheText);
{$ELSE}
    ACanvas.TextOutAngle(90, FP[0].x, FP[0].y, TheText);
{$ENDIF}
  end
  else
  begin // Z Axis: oddball angle:
    NewAngle := (360 + 90 - FAngle) mod 360;
{$IFDEF GUI}
    Misc.TextOutAngle(ACanvas, NewAngle, FP[0].x, FP[0].y, TheText);
{$ELSE}
    ACanvas.TextOutAngle(NewAngle, FP[0].x, FP[0].y, TheText);
{$ENDIF}
  end;
{Set up pen ...}
  ACanvas.Pen.Color := ACanvas.Font.Color;
  //ACanvas.Pen.Mode := pmCopy;
  //ACanvas.Pen.Style := psSolid;
  if (FBorder) then
    ACanvas.PolyLine([FP[0], FP[1], FP[2], FP[3], FP[0]]);
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TText.StartEdit
  Description: starts user editing of Text
       Author: Mat Ballard
 Date created: 08/09/2001
Date modified: 08/09/2001 by Mat Ballard
      Purpose: Creates a TEdit and sets its properties
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.StartEdit(AParent: TWinControl);
begin
  Assert(FTextEdit = nil, 'TText.StartEdit: FTextEdit is not nil !');

  FTextEdit := TEdit.Create(nil);
  FTextEdit.Font.Assign(FFont);
  FTextEdit.AutoSize := TRUE;
  FTextEdit.Text := GetCaption;
  FTextEdit.Parent := AParent;
  FTextEdit.OnKeyDown := TextEditKeyDown;
  FTextEdit.Hint := 'Type in the new caption, then press <-Enter, or Escape to cancel.';
  FTextEdit.ShowHint := TRUE;
  FTextEdit.Left := FLeft;
  FTextEdit.Top := FTop;
  FTextEdit.SetFocus;
end;

{------------------------------------------------------------------------------
    Procedure: TText.EndEdit
  Description: ends user editing of Text
       Author: Mat Ballard
 Date created: 08/09/2001
Date modified: 08/09/2001 by Mat Ballard
      Purpose: cleanup of TEdit
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.EndEdit;
begin
  Assert(FTextEdit <> nil, 'TText.Draw: FTextEdit = nil !');
  FTextEdit.OnKeyDown := nil;
  FTextEdit.Free;
  FTextEdit := nil;
  FTextEditFinshed := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TText.TextEditKeyDown
  Description: KeyDown event handler of FTextEdit in-place editor
       Author: Mat Ballard
 Date created: 08/08/2001
Date modified: 08/08/2001 by Mat Ballard
      Purpose: Cancel the FTextEditor if Esc pressed, or save the changed Title.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TText.TextEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
var
  i: Integer;
begin
  if (Key = VK_ESCAPE) then  //4096
  begin
    FTextEdit.Visible := FALSE;
    //Key := 0;
    FTextEditFinshed := TRUE;
  end;

  if (Key = VK_RETURN) then //4100
  begin
    if ((ssShift in Shift) or (ssCtrl in Shift)) then
    begin
      Key := 10;
      i := FTextEdit.SelStart;
      FTextEdit.Text :=
        Copy(FTextEdit.Text, 1, i) + #10 +
        Copy(FTextEdit.Text, i+1, 999);
      FTextEdit.SelStart := i + 1;
    end
    else
    begin
      SetCaption(FTextEdit.Text);
      FTextEdit.Visible := FALSE;
      Key := 0;
      FTextEditFinshed := TRUE;
      DoHandleChange;
    end;
  end;
end;
{$ENDIF} {GUI StartEdit}

{TTitle -------------------------------------------------------------------}
{Constructor and Destructor -------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TTitle.Create
  Description: Standard Constructor for TTitle
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TTitle.Create(Axis: TCollectionItem);
begin
{First call the ancestor:}
  inherited Create(nil);

  FAxis := Axis;
  FDefaultSide := TRUE;
  FPosition := 50;
end;

{Begin Get functions -----------------------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TTitle.GetUnits
  Description: protected property Get function
       Author: Mat Ballard
 Date created: 09/25/2002
Date modified: 09/25/2002 by Mat Ballard
      Purpose: gets the Units in the Caption
 Known Issues:
 ------------------------------------------------------------------------------}
function TTitle.GetUnits: String;
var
  Str: String;
begin
  Str := FCaption;
  GetWord(Str, '(');
  Result := GetWord(Str, ')');
end;

{End Get functions --------------------------------------------------------------}

{Begin Set procedures --------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TTitle.SetName
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/25/2002
Date modified: 08/25/2002 by Mat Ballard
      Purpose: sets the Name; not needed in TRectangle, but in descendants
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetName(Value: String);
begin
  FName := Value;
  SetCaption(Getword(Value, ' Title'));
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetPosition
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Position, which is where the title is located along the Axis or Plot
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetPosition(Value: TPercent);
begin
  if (FPosition <> Value) then
  begin
    FPosition := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetDefaultSide
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 08/25/2002
Date modified: 08/25/2002 by Mat Ballard
      Purpose: sets the Side (of the axis) on which text is drawn
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetDefaultSide(Value: Boolean);
begin
  if (Value <> FDefaultSide) then
  begin
    FDefaultSide := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetUnits
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Units (eg: Furlongs / Fortnight ^3), and also the FullCaption
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetUnits(Value: String);
var
  Str: String;
begin
  if (Value <> GetUnits) then
  begin
    Str := GetWord(FCaption, '(') + ' (' + Value;
    GetWord(FCaption, ')');
    Str := Str + ')' + FCaption;
    SetCaption(Str);
  end;
end;

{Drawing ----------------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TTitle.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TTitle(Dest).DefaultSide := FDefaultSide;
  TTitle(Dest).Position := FPosition;
  TTitle(Dest).Units := FUnits;
end;

{TLegend ----------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TLegend.Create
  Description: Standard Constructor for TLegend
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TLegend.Createlist(AOwner: TCollection; SeriesList: TCollection);
begin
{First call the ancestor:}
  inherited Create(AOwner);
  FSeriesList := SeriesList;

{Create font:}
  FFont := TFont.Create;
  FFont.Name := sArial;
  FFont.Size := SMALL_FONT_SIZE;
  FCheckBoxes := TRUE;
  FDirection := drVertical;
  FTag := Ord(soLegend);

{$IFDEF GUI}
  FLegendEdit := nil;
  FLegendEditFinshed := FALSE;
{$ENDIF}
end;

destructor TLegend.Destroy;
begin
  FFont.Free;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues: does not assign the Serieslist !
 ------------------------------------------------------------------------------}
procedure TLegend.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TLegend(Dest).CheckBoxes := FCheckBoxes;
  TLegend(Dest).Direction := FDirection;
  TLegend(Dest).Font.Assign(FFont);
end;

{Get functions ----------------------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TLegend.GetHit
  Description: Interprets mouse click position
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the region of the line of the Legend under the input position
 Known Issues:
 ------------------------------------------------------------------------------}
function TLegend.GetHit(iX, iY: Integer; var TheRect: TRect): Integer;
var
  SeriesWidth,
  TheHit,
  TheOffset: Integer;
begin
  if (FDirection = drHorizontal) then
  begin
    SeriesWidth :=  FCheckWidth + FLineWidth + FStringWidth + 5;
    TheHit := (iX - Left) div SeriesWidth;
    TheRect.Left := Left + TheHit * SeriesWidth;
    TheRect.Right := TheRect.Left + SeriesWidth;
    TheRect.Top := Top;
    TheRect.Bottom := Bottom;
    TheOffset := (iX - Left) mod SeriesWidth;
  end
  else
  begin {drVertical}
    TheHit := (iY - Top) div FFontHeight;
    if (TheHit = FSeriesList.Count) then
      TheHit := FSeriesList.Count - 1;
    TheRect.Left := Left;
    TheRect.Right := Right;
    TheRect.Top := Top + TheHit * FFontHeight;
    TheRect.Bottom := TheRect.Top + FFontHeight;
    TheOffset := iX - Left;
  end;
  if (TheOffset <= FCheckWidth) then
    TSeries(FSeriesList.Items[TheHit]).Visible :=
      not TSeries(FSeriesList.Items[TheHit]).Visible;
  GetHit := TheHit;
end;

function TLegend.GetItemWidth: Integer;
begin
  GetItemWidth :=  FCheckWidth + FLineWidth + FStringWidth;
end;

{Set procedures ---------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TLegend.SetCheckBoxes
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 04/12/2001
Date modified: 04/12/2001 by Mat Ballard
      Purpose: sets whether or not the CheckBoxes are visible
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetCheckBoxes(Value: Boolean);
begin
  if (Value <> FCheckboxes) then
  begin
    FCheckboxes := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.SetDirection
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Direction
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetDirection(Value: TDirection);
begin
  if (Value <> FDirection) then
  begin
    FDirection := Value;
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.SetFont
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Font
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.Draw
  Description: draws the legend
       Author: Mat Ballard
 Date created: 04/19/2001
Date modified: 09/11/2002 by Mat Ballard
      Purpose: screen drawing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.Draw(ACanvas: TCanvas);
var
  Chars,
  i,
  iX,
  iMaxChars,
  MaxChars,
  LineY,
  TextY: Integer;

const
  SERIES_BUTTON_COLOR = clLime;
{$IFDEF LINUX}
  //ARect: TRect;
{$ENDIF}

  procedure DoGeometry;
  begin
{Allow for symbols and lines:}
    FStringWidth := ACanvas.TextWidth(
      TSeries(FSeriesList.Items[iMaxChars]).Name);
    FFontHeight := ACanvas.TextHeight('Ap');
    if (FCheckBoxes) then
    begin
      FCheckWidth := 4 * ACanvas.TextWidth('W') div 3;
      //if (Odd(FCheckWidth)) then Inc(FCheckWidth);
    end
     else
      FCheckWidth := 0;
    FLineWidth := 2 * FCheckWidth + TSeries(FSeriesList.Items[0]).SymbolSize;

    if (FDirection = drHorizontal) then
    begin
      Height := FFontHeight;
  {       <-----LineWidth---->
   <Check><Line><Symbol><Line><Text---StringWidth--->}
      FWidth := FSeriesList.Count *
        (FCheckWidth + FLineWidth + FStringWidth + 5);
{NB: This used to be Width; changed to FWidth to prevent redraw.}
    end
    else
    begin
{NB: These used to be Width; changed to FWidth to prevent redraw.}
      FHeight := FSeriesList.Count * FFontHeight;
      FWidth := FCheckWidth + FLineWidth + FStringWidth + 5;
    end;
  end;

  procedure DrawCheck(IsVisible: Boolean; X, Y, Size: Integer);
  begin
    ACanvas.Pen.Color := clBtnFace;
    ACanvas.Pen.Width := 3;
    ACanvas.Pen.Style := psSolid;
    ACanvas.Brush.Color := clWhite;
    ACanvas.Brush.Style := bsSolid;
    ACanvas.Rectangle(X, Y, X + Size, Y + Size);

    ACanvas.Pen.Width := 1;
    ACanvas.Pen.Color := clBtnShadow;
    ACanvas.MoveTo(X, Y+Size-2);
    ACanvas.LineTo(X, Y);
    ACanvas.LineTo(X+Size-1, Y);

    ACanvas.Pen.Color := clWhite;
    ACanvas.MoveTo(X+Size-1, Y+1);
    ACanvas.LineTo(X+Size-1, Y+Size-1);
    ACanvas.LineTo(X, Y+Size-1);

    ACanvas.Pen.Color := clBlack;
    ACanvas.MoveTo(X+1, Y+Size-2);
    ACanvas.LineTo(X+1, Y+1);
    ACanvas.LineTo(X+Size-2, Y+1);

    if (IsVisible) then
    begin
      ACanvas.Brush.Color := SERIES_BUTTON_COLOR;
      ACanvas.Ellipse(X+3, Y+3, X+Size-3, Y+Size-3);
    end;
  end;

  procedure DrawItem;
  begin
    DrawCheck(TSeries(FSeriesList.Items[i]).Visible,
      iX, LineY-FCheckWidth div 2, FCheckWidth);
    ACanvas.Pen.Assign(TSeries(FSeriesList.Items[i]).Pen);
    ACanvas.MoveTo(iX + FCheckWidth+2, LineY);
    ACanvas.LineTo(iX + FCheckWidth + FLineWidth, LineY);
    ACanvas.Brush.Assign(TSeries(FSeriesList.Items[i]).Brush);
    TSeries(FSeriesList.Items[i]).DrawSymbol(ACanvas,
      iX + FCheckWidth + FLineWidth div 2, LineY);
    ACanvas.Brush.Style := bsClear;
{output text to screen:}
    ACanvas.Font.Color := ACanvas.Pen.Color;
    ACanvas.TextOut(iX + FCheckWidth + FLineWidth+1, TextY,
      TSeries(FSeriesList.Items[i]).Name);
  end;

begin
{$IFDEF GUI}
  if (FLegendEditFinshed) then
    EndEdit;
{$ENDIF}

{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TLegend.Draw: ACanvas is nil !');
{$ENDIF}

  if (not FVisible) then exit;
  if (FSeriesList.Count = 0) then exit;

  ACanvas.Font.Assign(Self.Font);

  MaxChars := 0;
  iMaxChars := -1;
  for i := 0 to FSeriesList.Count-1 do
  begin
    Chars := System.Length(TSeries(FSeriesList.Items[i]).Name);
    if (MaxChars < Chars) then
    begin
      MaxChars := Chars;
      iMaxChars := i;
    end;
  end;
  if (iMaxChars < 0) then exit;

  DoGeometry;

  LineY := Self.Top + Self.FontHeight div 2;
  TextY := Self.Top;

  iX := FLeft;
  if (Self.Direction = drVertical) then
  begin
    for i := 0 to FSeriesList.Count-1 do
    //for i := 0 to SeriesList.Count-1 do
    begin
      DrawItem;
      Inc(LineY, FFontHeight);
      Inc(TextY, FFontHeight);
    end;
  end
  else {Horizontal}
  begin
{Note: in Horizontal mode, the size of each series name is:}
{<---LineWidth---><---StringWidth---><-LineWidth->}
{<-Symbol + Line-><-Name of Series--><--Space---->}
    LineY := Self.Top + Self.FontHeight div 2;
    TextY := Self.Top;
    for i := 0 to FSeriesList.Count-1 do
    begin
      DrawItem;
      iX := iX + FCheckWidth + FLineWidth + FStringWidth + 5;
    end; {for}
{NB: This used to be Width; changed to FWidth to prevent redraw.}
    FWidth := iX - FLeft;
    FHeight := Self.FontHeight;
  end; {if}
end;

{$IFDEF GUI}
{------------------------------------------------------------------------------
    Procedure: TLegend.StartEdit
  Description: starts user editing of Legend
       Author: Mat Ballard
 Date created: 08/09/2001
Date modified: 08/09/2001 by Mat Ballard
      Purpose: Creates a TEdit and sets its properties
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.StartEdit(AParent: TWinControl; ThePoint: TPoint);
var
  TheRect: TRect;
begin
  Assert(FLegendEdit = nil, 'TLegend.StartEdit: FLegendEdit is not nil !');

  TheEditSeries := GetHit(ThePoint.x, ThePoint.y, TheRect);

  FLegendEdit := TEdit.Create(nil);
  FLegendEdit.Font.Assign(FFont);
  FLegendEdit.Text := TSeries(FSeriesList.Items[TheEditSeries]).Name;
  FLegendEdit.Parent := AParent;
  FLegendEdit.OnKeyDown := LegendEditKeyDown;
  FLegendEdit.Hint := 'Type in the new caption, then press <-Enter, or Escape to cancel.';
  FLegendEdit.ShowHint := TRUE;
  FLegendEdit.Left := TheRect.Left;
  FLegendEdit.Top := TheRect.Top;
  FLegendEdit.Width := TheRect.Right - TheRect.Left;
  FLegendEdit.SetFocus;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.EndEdit
  Description: ends user editing of Legend
       Author: Mat Ballard
 Date created: 08/09/2001
Date modified: 08/09/2001 by Mat Ballard
      Purpose: cleanup of TEdit
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.EndEdit;
begin
  Assert(FLegendEdit <> nil, 'TLegend.Draw: FLegendEdit = nil !');
  FLegendEdit.OnKeyDown := nil;
  FLegendEdit.Free;
  FLegendEdit := nil;
  FLegendEditFinshed := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.LegendEditKeyDown
  Description: KeyDown event handler of FLegendEdit in-place editor
       Author: Mat Ballard
 Date created: 08/08/2001
Date modified: 08/08/2001 by Mat Ballard
      Purpose: Cancel the FLegendEditor if Esc pressed, or save the changed Title.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.LegendEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_ESCAPE) then  //4096
  begin
    FLegendEdit.Visible := FALSE;
    //Key := 0;
    FLegendEditFinshed := TRUE;
  end;

  if (Key = VK_RETURN) then //4100
  begin
{this will throw an exception if Tag is not a valid TObjectType:}
    TSeries(FSeriesList.Items[TheEditSeries]).Name := FLegendEdit.Text;
    FLegendEdit.Visible := FALSE;
    Key := 0;
    FLegendEditFinshed := TRUE;
    DoHandleChange;
  end;
end;
{$ENDIF} {GUI StartEdit}

{TNote ------------------------------------------------------------------------}
procedure TNote.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TNote(Dest).ArrowLeft := FArrowLeft;
  TNote(Dest).ArrowTop := FArrowTop;
  TNote(Dest).ArrowLeftReal := FArrowLeftReal;
  TNote(Dest).ArrowTopReal := FArrowTopReal;
  TNote(Dest).LeftReal := FLeftReal;
  TNote(Dest).TopReal := FTopReal;
end;

constructor TNote.Create(AOwner: TCollection);
begin
  inherited Create(AOwner);
{a note can only ever be a note:}
  Tag := Ord(soNote);
  FBorder := TRUE;
end;

{destructor TNote.Destroy;
begin

end;}

{------------------------------------------------------------------------------
    Procedure: TNote.Draw
  Description: public Drawing method
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 09/25/2002 by Mat Ballard
      Purpose: Draws the Caption, either horizontally or vertically, then draws the arrow
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.Draw(ACanvas: TCanvas);
begin
{$IFDEF GUI}
  if (FTextEditFinshed) then
    EndEdit;
{$ENDIF}
  if (not FVisible) then exit;
  if (System.Length(FCaption) = 0) then exit;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TNote.Draw: ACanvas is nil !');
{$ENDIF}

  ACanvas.Font.Assign(FFont);
  ACanvas.Pen.Color := FFont.Color;
  ACanvas.Brush.Style := bsClear;

{we do the geometry;
 note that if a Zoom-in has occurred, then the note position changes:}

  FLeft := TNoteList(GetOwner).Plot.XAxis.FofX(FLeftReal);
  FTop := TNoteList(GetOwner).Plot.YAxis.FofY(FTopReal);
  FArrowLeft := TNoteList(GetOwner).Plot.XAxis.FofX(FArrowLeftReal);
  FArrowTop := TNoteList(GetOwner).Plot.YAxis.FofY(FArrowTopReal);

{why re-invent the wheel ?}
  inherited Draw(ACanvas);

{now deal with that pesky arrow:}  
  if (FArrowLeft < Left) then
    ArrowStartLeft := Left
  else if (FArrowLeft > Left+Width) then
    ArrowStartLeft := Left + Width
  else
    ArrowStartLeft := Left + Width div 2;

  if (FArrowTop < Top) then
    ArrowStartTop := Top
  else if (FArrowTop > Top+Height) then
    ArrowStartTop := Top + Height
  else
    ArrowStartTop := Top + Height div 2;

  if ((FArrowLeft < Left) or
      (FArrowLeft > Left+Width) or
      (FArrowTop < Top) or
      (FArrowTop > Top+Height)) then
  begin
    ACanvas.MoveTo(ArrowStartLeft, ArrowStartTop);
    ACanvas.LineTo(FArrowLeft, FArrowTop);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.TracePointerTo
  Description: Moves the end of the note pointer to X, Y
       Author: Mat Ballard
 Date created: 11/23/2000
Date modified: 11/23/2000 by Mat Ballard
      Purpose: manages movement of the note pointer
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.TracePointerTo(ACanvas: TCanvas; iX, iY: Integer);
begin
{rub out the old line:}
  ACanvas.Pen.Mode := pmNotXOR;
  if ((FArrowLeft < Left) or
      (FArrowLeft > Left+Width) or
      (FArrowTop < Top) or
      (FArrowTop > Top+Height)) then
  begin
    ACanvas.MoveTo(ArrowStartLeft, ArrowStartTop);
    ACanvas.LineTo(FArrowLeft, FArrowTop);
  end;
{go to new coordinates:}
  ArrowLeft := iX;
  ArrowTop := iY;
{draw the new one:}
  ACanvas.MoveTo(ArrowStartLeft, ArrowStartTop);
  ACanvas.LineTo(FArrowLeft, FArrowTop);
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetArrowLeft
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 11/15/2000
Date modified: 11/15/2000 by Mat Ballard
      Purpose: sets the ArrowLeft, which also re-calculates the RealArrowLeft
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetArrowLeft(Value: Integer);
begin
  if (FArrowLeft <> Value) then
  begin
    FArrowLeft := Value;
    //FArrowLeftReal := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).XAxis.XofF(FArrowLeft);
    FArrowLeftReal := TNoteList(GetOwner).Plot.XAxis.XofF(FArrowLeft);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetArrowTop
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 11/15/2000
Date modified: 11/15/2000 by Mat Ballard
      Purpose: sets the ArrowTop, which also re-calculates the RealArrowTop
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetArrowTop(Value: Integer);
begin
  if (FArrowTop <> Value) then
  begin
    FArrowTop := Value;
    //FArrowTopReal := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).YAxis.YofF(FArrowTop);
    FArrowTopReal := TNoteList(GetOwner).Plot.YAxis.YofF(FArrowTop);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetLeft
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the (Screen)Left, which also re-calculates the LeftReal
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetLeft(Value: Integer);
begin
  if (Value <> FLeft) then
  begin
    FLeft := Value;
    //FLeftReal := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).XAxis.XofF(FLeft);
    FLeftReal := TNoteList(GetOwner).Plot.XAxis.XofF(FLeft);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetTop
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the (Screen)Top, which also re-calculates the TopReal
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetTop(Value: Integer);
begin
  if (Value <> FTop) then
  begin
    FTop := Value;
    //FTopReal := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).YAxis.YofF(FTop);
    FTopReal := TNoteList(GetOwner).Plot.YAxis.YofF(FTop);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetLeftReal
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the (Data)LeftReal, which also re-calculates the Left
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetLeftReal(Value: Single);
begin
  if (FLeftReal <> Value) then
  begin
    FLeftReal := Value;
    //FLeft := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).XAxis.FofX(FLeftReal);
    FLeft := TNoteList(GetOwner).Plot.XAxis.FofX(FLeftReal);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetTopReal
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the (Data)TopReal, which also re-calculates the Top
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetTopReal(Value: Single);
begin
  if (FTopReal <> Value) then
  begin
    FTopReal := Value;
    //FTop := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).YAxis.FofY(FTopReal);
    FTop := TNoteList(GetOwner).Plot.YAxis.FofY(FTopReal);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetArrowLeftReal
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the (Data)ArrowLeftReal, which also re-calculates the Left
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetArrowLeftReal(Value: Single);
begin
  if (FArrowLeftReal <> Value) then
  begin
    FArrowLeftReal := Value;
    //FLeft := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).XAxis.FofX(FArrowLeftReal);
    FArrowLeft := TNoteList(GetOwner).Plot.XAxis.FofX(FArrowLeftReal);
    DoHandleChange;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TNote.SetArrowTopReal
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the (Data)ArrowTopReal, which also re-calculates the Top
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TNote.SetArrowTopReal(Value: Single);
begin
  if (FArrowTopReal <> Value) then
  begin
    FArrowTopReal := Value;
    //FTop := {$IFDEF GUI}TPlot{$ELSE}TConsolePlot{$ENDIF}(FOwner).YAxis.FofY(FArrowTopReal);
    FArrowTop := TNoteList(GetOwner).Plot.YAxis.FofY(FArrowTopReal);
    DoHandleChange;
  end;
end;


end.
