unit gdGraphics;

{$P+,S-,W-,R-,T-,X+,H+}
{$C PRELOAD}

interface

uses
  SysUtils, Classes, Consts, Typinfo,
  {$IFDEF NO_MATH}NoMath,{$ELSE}Math,{$ENDIF}
  gd, gdWindows, {gdWinapi,}

  Misc;

{ Graphics Objects }

type
  PColor = ^TColor;
  TColor = -$7FFFFFFF-1..$7FFFFFFF;


const
{$IFDEF WIN32}
  PATH_DELIMITER = '\';
{$ENDIF}
{$IFDEF LINUX}
  PATH_DELIMITER = '/'
{$ENDIF}
  { Color Types }
  clScrollBar = TColor(COLOR_SCROLLBAR or $80000000);
  clBackground = TColor(COLOR_BACKGROUND or $80000000);
  clActiveCaption = TColor(COLOR_ACTIVECAPTION or $80000000);
  clInactiveCaption = TColor(COLOR_INACTIVECAPTION or $80000000);
  clMenu = TColor(COLOR_MENU or $80000000);
  clWindow = TColor(COLOR_WINDOW or $80000000);
  clWindowFrame = TColor(COLOR_WINDOWFRAME or $80000000);
  clMenuText = TColor(COLOR_MENUTEXT or $80000000);
  clWindowText = TColor(COLOR_WINDOWTEXT or $80000000);
  clCaptionText = TColor(COLOR_CAPTIONTEXT or $80000000);
  clActiveBorder = TColor(COLOR_ACTIVEBORDER or $80000000);
  clInactiveBorder = TColor(COLOR_INACTIVEBORDER or $80000000);
  clAppWorkSpace = TColor(COLOR_APPWORKSPACE or $80000000);
  clHighlight = TColor(COLOR_HIGHLIGHT or $80000000);
  clHighlightText = TColor(COLOR_HIGHLIGHTTEXT or $80000000);
  clBtnFace = TColor(COLOR_BTNFACE or $80000000);
  clBtnShadow = TColor(COLOR_BTNSHADOW or $80000000);
  clGrayText = TColor(COLOR_GRAYTEXT or $80000000);
  clBtnText = TColor(COLOR_BTNTEXT or $80000000);
  clInactiveCaptionText = TColor(COLOR_INACTIVECAPTIONTEXT or $80000000);
  clBtnHighlight = TColor(COLOR_BTNHIGHLIGHT or $80000000);
  cl3DDkShadow = TColor(COLOR_3DDKSHADOW or $80000000);
  cl3DLight = TColor(COLOR_3DLIGHT or $80000000);
  clInfoText = TColor(COLOR_INFOTEXT or $80000000);
  clInfoBk = TColor(COLOR_INFOBK or $80000000);

  clBlack = TColor($000000);
  clMaroon = TColor($000080);
  clGreen = TColor($008000);
  clOlive = TColor($008080);
  clNavy = TColor($800000);
  clPurple = TColor($800080);
  clTeal = TColor($808000);
  clGray = TColor($808080);
  clSilver = TColor($C0C0C0);
  clRed = TColor($0000FF);
  clLime = TColor($00FF00);
  clYellow = TColor($00FFFF);
  clBlue = TColor($FF0000);
  clFuchsia = TColor($FF00FF);
  clAqua = TColor($FFFF00);
  clLtGray = TColor($C0C0C0);
  clDkGray = TColor($808080);
  clWhite = TColor($FFFFFF);
  clNone = TColor($1FFFFFFF);
  clDefault = TColor($20000000);

  ScreenLogPixels = 96; // that's what it is on my 1024x768 laptop

const
  cmBlackness = BLACKNESS;
  cmDstInvert = DSTINVERT;
  cmMergeCopy = MERGECOPY;
  cmMergePaint = MERGEPAINT;
  cmNotSrcCopy = NOTSRCCOPY;
  cmNotSrcErase = NOTSRCERASE;
  cmPatCopy = PATCOPY;
  cmPatInvert = PATINVERT;
  cmPatPaint = PATPAINT;
  cmSrcAnd = SRCAND;
  cmSrcCopy = SRCCOPY;
  cmSrcErase = SRCERASE;
  cmSrcInvert = SRCINVERT;
  cmSrcPaint = SRCPAINT;
  cmWhiteness = WHITENESS;

  Colors: array[0..41] of TIdentMapEntry = (
    (Value: clBlack; Name: 'clBlack'),
    (Value: clMaroon; Name: 'clMaroon'),
    (Value: clGreen; Name: 'clGreen'),
    (Value: clOlive; Name: 'clOlive'),
    (Value: clNavy; Name: 'clNavy'),
    (Value: clPurple; Name: 'clPurple'),
    (Value: clTeal; Name: 'clTeal'),
    (Value: clGray; Name: 'clGray'),
    (Value: clSilver; Name: 'clSilver'),
    (Value: clRed; Name: 'clRed'),
    (Value: clLime; Name: 'clLime'),
    (Value: clYellow; Name: 'clYellow'),
    (Value: clBlue; Name: 'clBlue'),
    (Value: clFuchsia; Name: 'clFuchsia'),
    (Value: clAqua; Name: 'clAqua'),
    (Value: clWhite; Name: 'clWhite'),
    (Value: clScrollBar; Name: 'clScrollBar'),
    (Value: clBackground; Name: 'clBackground'),
    (Value: clActiveCaption; Name: 'clActiveCaption'),
    (Value: clInactiveCaption; Name: 'clInactiveCaption'),
    (Value: clMenu; Name: 'clMenu'),
    (Value: clWindow; Name: 'clWindow'),
    (Value: clWindowFrame; Name: 'clWindowFrame'),
    (Value: clMenuText; Name: 'clMenuText'),
    (Value: clWindowText; Name: 'clWindowText'),
    (Value: clCaptionText; Name: 'clCaptionText'),
    (Value: clActiveBorder; Name: 'clActiveBorder'),
    (Value: clInactiveBorder; Name: 'clInactiveBorder'),
    (Value: clAppWorkSpace; Name: 'clAppWorkSpace'),
    (Value: clHighlight; Name: 'clHighlight'),
    (Value: clHighlightText; Name: 'clHighlightText'),
    (Value: clBtnFace; Name: 'clBtnFace'),
    (Value: clBtnShadow; Name: 'clBtnShadow'),
    (Value: clGrayText; Name: 'clGrayText'),
    (Value: clBtnText; Name: 'clBtnText'),
    (Value: clInactiveCaptionText; Name: 'clInactiveCaptionText'),
    (Value: clBtnHighlight; Name: 'clBtnHighlight'),
    (Value: cl3DDkShadow; Name: 'cl3DDkShadow'),
    (Value: cl3DLight; Name: 'cl3DLight'),
    (Value: clInfoText; Name: 'clInfoText'),
    (Value: clInfoBk; Name: 'clInfoBk'),
    (Value: clNone; Name: 'clNone'));

type
{now the Delphi stuff:}
  EInvalidGraphic = class(Exception);
  EInvalidGraphicOperation = class(Exception);

  TResData = record
    Handle: THandle;
  end;

  TFontPitch = (fpDefault, fpVariable, fpFixed);
  TFontName = type string;
  TFontCharset = 0..255;

  { Changes to the following types should be reflected in the $HPPEMIT directives. }

  TFontDataName = string[LF_FACESIZE - 1];
  TFontStyle = (fsBold, fsItalic, fsUnderline, fsStrikeOut);
  TFontStyles = set of TFontStyle;
  TFontStylesBase = set of TFontStyle;

  {TFontData = record
    Handle: HFont;
    Height: Integer;
    Pitch: TFontPitch;
    Style: TFontStylesBase;
    Charset: TFontCharset;
    Name: TFontDataName;
  end;}

  TPenStyle = (psSolid, psDash, psDot, psDashDot, psDashDotDot, psClear,
    psInsideFrame);
  TPenMode = (pmBlack, pmWhite, pmNop, pmNot, pmCopy, pmNotCopy,
    pmMergePenNot, pmMaskPenNot, pmMergeNotPen, pmMaskNotPen, pmMerge,
    pmNotMerge, pmMask, pmNotMask, pmXor, pmNotXor);

  TPenData = record
    Handle: HPen;
    Color: TColor;
    Width: Integer;
    Style: TPenStyle;
  end;

  TBrushStyle = (bsSolid, bsClear, bsHorizontal, bsVertical,
    bsFDiagonal, bsBDiagonal, bsCross, bsDiagCross);

  TBrushData = record
    Handle: HBrush;
    Color: TColor;
    ///Bitmap: TBitmap;
    Style: TBrushStyle;
  end;

  PResource = ^TResource;
  TResource = record
    Next: PResource;
    RefCount: Integer;
    Handle: THandle;
    HashCode: Word;
    case Integer of
      0: (Data: TResData);
      //1: (Font: TFontData);
      2: (Pen: TPenData);
      3: (Brush: TBrushData);
  end;

  { TProgressEvent is a generic progress notification event which may be
        used by TGraphic classes with computationally intensive (slow)
        operations, such as loading, storing, or transforming image data.
    Event params:
      Stage - Indicates whether this call to the OnProgress event is to
        prepare for, process, or clean up after a graphic operation.  If
        OnProgress is called at all, the first call for a graphic operation
        will be with Stage = psStarting, to allow the OnProgress event handler
        to allocate whatever resources it needs to process subsequent progress
        notifications.  After Stage = psStarting, you are guaranteed that
        OnProgress will be called again with Stage = psEnding to allow you
        to free those resources, even if the graphic operation is aborted by
        an exception.  Zero or more calls to OnProgress with Stage = psRunning
        may occur between the psStarting and psEnding calls.
      PercentDone - The ratio of work done to work remaining, on a scale of
        0 to 100.  Values may repeat or even regress (get smaller) in
        successive calls.  PercentDone is usually only a guess, and the
        guess may be dramatically altered as new information is discovered
        in decoding the image.
      RedrawNow - Indicates whether the graphic can be/should be redrawn
        immediately.  Useful for showing successive approximations of
        an image as data is available instead of waiting for all the data
        to arrive before drawing anything.  Since there is no message loop
        activity during graphic operations, you should call Update to force
        a control to be redrawn immediately in the OnProgress event handler.
        Redrawing a graphic when RedrawNow = False could corrupt the image
        and/or cause exceptions.
      Rect - Area of image that has changed and needs to be redrawn.
      Msg - Optional text describing in one or two words what the graphic
        class is currently working on.  Ex:  "Loading" "Storing"
        "Reducing colors".  The Msg string can also be empty.
        Msg strings should be resourced for translation,  should not
        contain trailing periods, and should be used only for
        display purposes.  (do not: if Msg = 'Loading' then...)
  }

  TProgressStage = (psStarting, psRunning, psEnding);
  TProgressEvent = procedure (Sender: TObject; Stage: TProgressStage;
    PercentDone: Byte; RedrawNow: Boolean; const R: TRect; const Msg: string) of object;

  IChangeNotifier = interface
    ['{1FB62321-44A7-11D0-9E93-0020AF3D82DA}']
    procedure Changed;
  end;

  TGraphicsObject = class(TPersistent)
  private
    FOnChange: TNotifyEvent;
    FOwnerLock: PRTLCriticalSection;
  protected
    procedure Changed; dynamic;
    procedure Lock;
    procedure Unlock;
  public
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OwnerCriticalSection: PRTLCriticalSection read FOwnerLock write FOwnerLock;
  end;

  TFont = class(TGraphicsObject)
  private
    FCharset: TFontCharset;
    FColor: TColor;
    FPixelsPerInch: Integer;
    FHeight: Integer;
    FName: String;
    FNotify: IChangeNotifier;
    FPitch: TFontPitch;
    FStyle: TFontStyles;
  protected
    procedure Changed; override;
    function GetHeight: Integer;
    function GetName: TFontName;
    function GetPitch: TFontPitch;
    function GetSize: Integer;
    function GetStyle: TFontStyles;
    function GetCharset: TFontCharset;
    procedure SetColor(Value: TColor);
    procedure SetHeight(Value: Integer);
    procedure SetName(const Value: TFontName);
    procedure SetPitch(Value: TFontPitch);
    procedure SetSize(Value: Integer);
    procedure SetStyle(Value: TFontStyles);
    procedure SetCharset(Value: TFontCharset);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    property FontAdapter: IChangeNotifier read FNotify write FNotify;
    property PixelsPerInch: Integer read FPixelsPerInch write FPixelsPerInch;
  published
    property Charset: TFontCharset read GetCharset write SetCharset;
    property Color: TColor read FColor write SetColor;
    property Height: Integer read GetHeight write SetHeight;
    property Name: TFontName read GetName write SetName;
    property Pitch: TFontPitch read GetPitch write SetPitch default fpDefault;
    property Size: Integer read GetSize write SetSize stored False;
    property Style: TFontStyles read GetStyle write SetStyle;
  end;

  TPen = class(TGraphicsObject)
  private
    FColor: TColor;
    FMode: TPenMode;
    FStyle: TPenStyle;
    FWidth: Integer;
  protected
    function GetColor: TColor;
    procedure SetColor(Value: TColor);
    procedure SetMode(Value: TPenMode);
    function GetStyle: TPenStyle;
    procedure SetStyle(Value: TPenStyle);
    function GetWidth: Integer;
    procedure SetWidth(Value: Integer);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property Color: TColor read GetColor write SetColor default clBlack;
    property Mode: TPenMode read FMode write SetMode default pmCopy;
    property Style: TPenStyle read GetStyle write SetStyle default psSolid;
    property Width: Integer read GetWidth write SetWidth default 1;
  end;

  TBrush = class(TGraphicsObject)
  private
    FColor: TColor;
    FStyle: TBrushStyle;
  protected
    ///function GetBitmap: TBitmap;
    ///procedure SetBitmap(Value: TBitmap);
    function GetColor: TColor;
    procedure SetColor(Value: TColor);
    function GetStyle: TBrushStyle;
    procedure SetStyle(Value: TBrushStyle);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    ///property Bitmap: TBitmap read GetBitmap write SetBitmap;
  published
    property Color: TColor read GetColor write SetColor default clWhite;
    property Style: TBrushStyle read GetStyle write SetStyle default bsSolid;
  end;

  TFillStyle = (fsSurface, fsBorder);
  TFillMode = (fmAlternate, fmWinding);

  TCopyMode = Longint;

  TCanvasStates = (csHandleValid, csFontValid, csPenValid, csBrushValid);
  TCanvasState = set of TCanvasStates;
  TCanvasOrientation = (coLeftToRight, coRightToLeft);
  TPixelFormat = (pfDevice, pf1bit, pf4bit, pf8bit, pf15bit, pf16bit, pf24bit, pf32bit, pfCustom);

  TCanvas = class(TPersistent)
  private
{Pointer to the gd image:}
    im: PgdImage;

    FFont: TFont;
    FPen: TPen;
    FBrush: TBrush;
    FPenPos: TPoint;
    FCopyMode: TCopyMode;
    FOnChange: TNotifyEvent;
    FOnChanging: TNotifyEvent;
    FLock: TRTLCriticalSection;
    FLockCount: Integer;
    FPixelFormat: TPixelFormat;
    FTextFlags: Longint;
    FTransparentColor: TColor;

    mLineStyle: array [0..11] of Integer;
    mColorStyle: Integer;
    FWidth: Integer;
    FHeight: Integer;

    procedure SetStyle(Value: TPenStyle; ColorIndex: Integer);
    procedure HandleNeeded;

    function GetCanvasOrientation: TCanvasOrientation;
    function GetClipRect: TRect;
    function GetHandle: PgdImage;
    function GetPenPos: TPoint;
    function GetPixel(X, Y: Integer): TColor;
    procedure SetBrush(Value: TBrush);
    procedure SetFont(Value: TFont);
    procedure SetHandle(Value: PgdImage);
    procedure SetPen(Value: TPen);
    procedure SetPenPos(Value: TPoint);
    procedure SetPixel(X, Y: Integer; Value: TColor);
  protected
    procedure Changed; virtual;
    procedure Changing; virtual;
    function GetColorIndex(Color: TColor): Integer;
  public
    constructor Create;
    constructor CreateFull(AWidth, AHeight: Integer; AColor: TColor; APixelFormat: TPixelFormat);
    destructor Destroy; override;
    procedure Arc(X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer); overload;
    procedure Arc(X1, Y1, X2, Y2, StartAngle, EndAngle, iType: Integer); overload;
    ///procedure BrushCopy(const Dest: TRect; Bitmap: TBitmap; const Source: TRect; Color: TColor);
    procedure Chord(X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
    procedure CopyRect(const Dest: TRect; Canvas: TCanvas;
      const Source: TRect);
    ///procedure Draw(X, Y: Integer; Graphic: TGraphic);
    procedure DrawFocusRect(const Rect: TRect);
    procedure Ellipse(X1, Y1, X2, Y2: Integer); overload;
    procedure Ellipse(const Rect: TRect); overload;
    procedure FillRect(const Rect: TRect);
    procedure FloodFill(X, Y: Integer; Color: TColor; FillStyle: TFillStyle);
    procedure FrameRect(const Rect: TRect);
    procedure LineTo(X, Y: Integer);
    procedure Lock;
    procedure MoveTo(X, Y: Integer);
    procedure Pie(X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
    procedure Polygon(const Points: array of TPoint);
    procedure Polyline(const Points: array of TPoint);
    procedure PolyBezier(const Points: array of TPoint);
    procedure PolyBezierTo(const Points: array of TPoint);
    procedure Rectangle(X1, Y1, X2, Y2: Integer); overload;
    procedure Rectangle(const Rect: TRect); overload;
    procedure Refresh; virtual;
    procedure RoundRect(X1, Y1, X2, Y2, X3, Y3: Integer);
    ///procedure StretchDraw(const Rect: TRect; Graphic: TGraphic);
    function TextExtent(const Text: string): TSize;
    function TextHeight(const Text: string): Integer;
    procedure TextOut(X, Y: Integer; const Text: string);
    procedure TextOutAngle(Angle, X, Y: Integer; Text: String);

    procedure TextRect(Rect: TRect; X, Y: Integer; Text: string);
    function TextWidth(const Text: string): Integer;
    function TryLock: Boolean;
    procedure Unlock;

    property ClipRect: TRect read GetClipRect;
    property Handle: PgdImage read GetHandle write SetHandle;
    property LockCount: Integer read FLockCount;
    property CanvasOrientation: TCanvasOrientation read GetCanvasOrientation;
    property PenPos: TPoint read GetPenPos write SetPenPos;
    property Pixels[X, Y: Integer]: TColor read GetPixel write SetPixel;
    property TextFlags: Longint read FTextFlags write FTextFlags;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnChanging: TNotifyEvent read FOnChanging write FOnChanging;
  published
    property Brush: TBrush read FBrush write SetBrush;
    property CopyMode: TCopyMode read FCopyMode write FCopyMode default cmSrcCopy;
    property Font: TFont read FFont write SetFont;
    property Pen: TPen read FPen write SetPen;
  end;

  TGraphic = class(TPersistent)
  private
    FOnChange: TNotifyEvent;
    FOnProgress: TProgressEvent;
    FModified: Boolean;
    FTransparent: Boolean;
    FPaletteModified: Boolean;
    procedure SetModified(Value: Boolean);
  protected
    constructor Create; virtual;
    procedure Changed(Sender: TObject); virtual;
    procedure DefineProperties(Filer: TFiler); override;
    procedure Draw(ACanvas: TCanvas; const Rect: TRect); virtual; abstract;
    function Equals(Graphic: TGraphic): Boolean; virtual;
    function GetEmpty: Boolean; virtual; abstract;
    function GetHeight: Integer; virtual; abstract;
    function GetPalette: HPALETTE; virtual;
    function GetTransparent: Boolean; virtual;
    function GetWidth: Integer; virtual; abstract;
    procedure Progress(Sender: TObject; Stage: TProgressStage;
      PercentDone: Byte;  RedrawNow: Boolean; const R: TRect; const Msg: string); dynamic;
    procedure ReadData(Stream: TStream); virtual;
    procedure SetHeight(Value: Integer); virtual; abstract;
    procedure SetPalette(Value: HPALETTE); virtual;
    procedure SetTransparent(Value: Boolean); virtual;
    procedure SetWidth(Value: Integer); virtual; abstract;
    procedure WriteData(Stream: TStream); virtual;
  public
    procedure LoadFromFile(const Filename: string); virtual;
    procedure SaveToFile(const Filename: string); virtual;
    procedure LoadFromStream(Stream: TStream); virtual; abstract;
    procedure SaveToStream(Stream: TStream); virtual; abstract;
    procedure LoadFromClipboardFormat(AFormat: Word; AData: THandle;
      APalette: HPALETTE); virtual; abstract;
    procedure SaveToClipboardFormat(var AFormat: Word; var AData: THandle;
      var APalette: HPALETTE); virtual; abstract;
    property Empty: Boolean read GetEmpty;
    property Height: Integer read GetHeight write SetHeight;
    property Modified: Boolean read FModified write SetModified;
    property Palette: HPALETTE read GetPalette write SetPalette;
    property PaletteModified: Boolean read FPaletteModified write FPaletteModified;
    property Transparent: Boolean read GetTransparent write SetTransparent;
    property Width: Integer read GetWidth write SetWidth;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnProgress: TProgressEvent read FOnProgress write FOnProgress;
  end;

  TGraphicClass = class of TGraphic;

  TBitmapHandleType = (bmDIB, bmDDB);
  TTransparentMode = (tmAuto, tmFixed);
  TJPEGQualityRange = -1..100;

  TBitmap = class(TGraphic)
  private
    FCanvas: TCanvas;
    FIgnorePalette: Boolean;
    FMaskBitsValid: Boolean;
    ///FMaskValid: Boolean;
    FTransparentColor: TColor;
    FTransparentMode: TTransparentMode;

    FWidth: Integer;
    FHeight: Integer;
    FBackgroundColor: TColor;
    FPixelFormat: TPixelFormat;
    FCompressionQuality: TJPEGQualityRange;

    procedure Changing(Sender: TObject);
    procedure CopyImage(AHandle: HBITMAP; APalette: HPALETTE; DIB: TDIBSection);
    ///procedure DIBNeeded;
    ///procedure FreeContext;
    function GetCanvas: TCanvas;
    ///function GetHandle: HBITMAP; virtual;
    function GetHandleType: TBitmapHandleType;
    function GetMaskHandle: HBITMAP; virtual;
    ///function GetMonochrome: Boolean;
    function GetPixelFormat: TPixelFormat;
    ///function GetScanline(Row: Integer): Pointer;
    function GetTransparentColor: TColor;
    ///procedure NewImage(NewHandle: HBITMAP; NewPalette: HPALETTE;
      ///const NewDIB: TDIBSection; OS2Format: Boolean; RLEStream: TStream = nil);
    procedure ReadStream(Stream: TStream; Size: Longint);
    ///procedure ReadDIB(Stream: TStream; ImageSize: LongWord);
    ///procedure SetHandle(Value: HBITMAP);
    procedure SetHandleType(Value: TBitmapHandleType); virtual;
    procedure SetMaskHandle(Value: HBITMAP);
    ///procedure SetMonochrome(Value: Boolean);
    procedure SetPixelFormat(Value: TPixelFormat);
    procedure SetTransparentColor(Value: TColor);
    procedure SetTransparentMode(Value: TTransparentMode);
    function TransparentColorStored: Boolean;
    procedure WriteStream(Stream: TStream; WriteSize: Boolean);
  protected
    procedure Changed(Sender: TObject); override;
    procedure Draw(ACanvas: TCanvas; const Rect: TRect); override;
    function GetEmpty: Boolean; override;
    function GetHeight: Integer; override;
    ///function GetPalette: HPALETTE; override;
    function GetWidth: Integer; override;
    ///procedure HandleNeeded;
    procedure MaskHandleNeeded;
    ///procedure PaletteNeeded;
    ///procedure ReadData(Stream: TStream); override;
    procedure SetHeight(Value: Integer); override;
    ///procedure SetPalette(Value: HPALETTE); override;
    procedure SetWidth(Value: Integer); override;
    ///procedure WriteData(Stream: TStream); override;
  public
    constructor Create; override;
    constructor CreateFull(Width, Height: Integer; BackgroundColor: TColor; APixelFormat: TPixelFormat);
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    procedure Dormant;
    procedure FreeImage;
    ///procedure LoadFromClipboardFormat(AFormat: Word; AData: THandle;
      ///APalette: HPALETTE); override;
    procedure LoadFromFile(const Filename: string); override;
    procedure LoadFromStream(Stream: TStream); override;
    procedure LoadFromResourceName(Instance: THandle; const ResName: String);
    procedure LoadFromResourceID(Instance: THandle; ResID: Integer);
    procedure Mask(TransparentColor: TColor);
    ///function ReleaseHandle: HBITMAP;
    function ReleaseMaskHandle: HBITMAP;
    ///function ReleasePalette: HPALETTE;
    ///procedure SaveToClipboardFormat(var Format: Word; var Data: THandle;
      ///var APalette: HPALETTE); override;
    procedure SaveToFile(const Filename: string); override;
    procedure SaveToStream(Stream: TStream); override;
    property Canvas: TCanvas read GetCanvas;
    ///property Handle: HBITMAP read GetHandle write SetHandle;
    property HandleType: TBitmapHandleType read GetHandleType write SetHandleType;
    property IgnorePalette: Boolean read FIgnorePalette write FIgnorePalette;
    property CompressionQuality: TJPEGQualityRange read FCompressionQuality write FCompressionQuality;
    property MaskHandle: HBITMAP read GetMaskHandle write SetMaskHandle;
    ///property Monochrome: Boolean read GetMonochrome write SetMonochrome;
    property PixelFormat: TPixelFormat read GetPixelFormat write SetPixelFormat;
    ///property ScanLine[Row: Integer]: Pointer read GetScanLine;
    property TransparentColor: TColor read GetTransparentColor
      write SetTransparentColor stored TransparentColorStored;
    property TransparentMode: TTransparentMode read FTransparentMode
      write SetTransparentMode default tmAuto;
  end;


var
  BitmapImageLock: TRTLCriticalSection;
  CounterLock: TRTLCriticalSection;

  function MulDiv(nNumber, nNumerator, nDenominator: Integer): Integer;
  procedure InitializeCriticalSection(var lpCriticalSection: TRTLCriticalSection);
  procedure DeleteCriticalSection(var lpCriticalSection: TRTLCriticalSection);
  procedure EnterCriticalSection(var lpCriticalSection: TRTLCriticalSection);
  procedure LeaveCriticalSection(var lpCriticalSection: TRTLCriticalSection);

  function GetSysColor(nIndex: Integer): DWORD;
  function ColorToRGB(Color: TColor): Longint;
  function ColorToString(Color: TColor): string;
  function StringToColor(const S: string): TColor;
  procedure GetColorValues(Proc: TGetStrProc);
  function ColorToIdent(Color: Longint; var Ident: string): Boolean;
  function IdentToColor(const Ident: string; var Color: Longint): Boolean;
  procedure GetCharsetValues(Proc: TGetStrProc);
  function CharsetToIdent(Charset: Longint; var Ident: string): Boolean;
  function IdentToCharset(const Ident: string; var Charset: Longint): Boolean;
  function GetDInColors(BitCount: Word): Integer;

{ Exception routines }
  procedure InvalidOperation(Str: PResStringRec);
  procedure InvalidGraphic(Str: PResStringRec);
  procedure InvalidBitmap;
  ///procedure InvalidIcon;
  ///procedure InvalidMetafile;
  procedure OutOfResources;


implementation

function MulDiv(nNumber, nNumerator, nDenominator: Integer): Integer;
var
  a, b, c: Int64;
begin
  a := nNumber;
  b := nNumerator;
  c := nDenominator;
  a := a * b div c;
  Result := Integer(a);
end;

procedure InitializeCriticalSection(var lpCriticalSection: TRTLCriticalSection);
begin
end;

procedure DeleteCriticalSection(var lpCriticalSection: TRTLCriticalSection);
begin
end;

procedure EnterCriticalSection(var lpCriticalSection: TRTLCriticalSection);
begin
end;

procedure LeaveCriticalSection(var lpCriticalSection: TRTLCriticalSection);
begin
end;

function GetSysColor(nIndex: Integer): DWORD;
begin
  case nIndex of
    0: Result := 13160660;
    1: Result := 0;
    2: Result := 6956042;
    3: Result := 8421504;
    4: Result := 13160660;
    5: Result := 16777215;
    6: Result := 0;
    7: Result := 0;
    8: Result := 0;
    9: Result := 16777215;
    10: Result := 13160660;
    11: Result := 13160660;
    12: Result := 8421504;
    13: Result := 6956042;
    14: Result := 16777215;
    15: Result := 13160660;
    16: Result := 8421504;
    17: Result := 8421504;
    18: Result := 0;
    19: Result := 13160660;
    20: Result := 16777215;
    21: Result := 4210752;
    22: Result := 13160660;
    23: Result := 0;
    24: Result := 14811135;
    25: Result := 11908533;
    26: Result := 16711680;
    27: Result := 15780518;
    28: Result := 12632256;
  else
    Result := 0;  
  end;
end;

function ColorToRGB(Color: TColor): Longint;
begin
  if Color < 0 then
    Result := GetSysColor(Color and $000000FF) else
    Result := Color;
end;

function ColorToString(Color: TColor): string;
begin
  if not ColorToIdent(Color, Result) then
    FmtStr(Result, '%s%.8x', [HexDisplayPrefix, Color]);
end;

function StringToColor(const S: string): TColor;
begin
  if not IdentToColor(S, Longint(Result)) then
    Result := TColor(StrToInt(S));
end;

procedure GetColorValues(Proc: TGetStrProc);
var
  I: Integer;
begin
  for I := Low(Colors) to High(Colors) do Proc(Colors[I].Name);
end;

function ColorToIdent(Color: Longint; var Ident: string): Boolean;
begin
  Result := IntToIdent(Color, Ident, Colors);
end;

function IdentToColor(const Ident: string; var Color: Longint): Boolean;
begin
  Result := IdentToInt(Ident, Color, Colors);
end;

{ TGraphicsObject }

procedure TGraphicsObject.Changed;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TGraphicsObject.Lock;
begin
  if Assigned(FOwnerLock) then EnterCriticalSection(FOwnerLock^);
end;

procedure TGraphicsObject.Unlock;
begin
  if Assigned(FOwnerLock) then LeaveCriticalSection(FOwnerLock^);
end;

{ TFont }

const
  FontCharsets: array[0..17] of TIdentMapEntry = (
    (Value: 0; Name: 'ANSI_CHARSET'),
    (Value: 1; Name: 'DEFAULT_CHARSET'),
    (Value: 2; Name: 'SYMBOL_CHARSET'),
    (Value: 77; Name: 'MAC_CHARSET'),
    (Value: 128; Name: 'SHIFTJIS_CHARSET'),
    (Value: 129; Name: 'HANGEUL_CHARSET'),
    (Value: 130; Name: 'JOHAB_CHARSET'),
    (Value: 134; Name: 'GB2312_CHARSET'),
    (Value: 136; Name: 'CHINESEBIG5_CHARSET'),
    (Value: 161; Name: 'GREEK_CHARSET'),
    (Value: 162; Name: 'TURKISH_CHARSET'),
    (Value: 177; Name: 'HEBREW_CHARSET'),
    (Value: 178; Name: 'ARABIC_CHARSET'),
    (Value: 186; Name: 'BALTIC_CHARSET'),
    (Value: 204; Name: 'RUSSIAN_CHARSET'),
    (Value: 222; Name: 'THAI_CHARSET'),
    (Value: 238; Name: 'EASTEUROPE_CHARSET'),
    (Value: 255; Name: 'OEM_CHARSET'));

procedure GetCharsetValues(Proc: TGetStrProc);
var
  I: Integer;
begin
  for I := Low(FontCharsets) to High(FontCharsets) do Proc(FontCharsets[I].Name);
end;

function CharsetToIdent(Charset: Longint; var Ident: string): Boolean;
begin
  Result := IntToIdent(Charset, Ident, FontCharsets);
end;

function IdentToCharset(const Ident: string; var Charset: Longint): Boolean;
begin
  Result := IdentToInt(Ident, CharSet, FontCharsets);
end;

function GetDInColors(BitCount: Word): Integer;
begin
  case BitCount of
    1, 4, 8: Result := 1 shl BitCount;
  else
    Result := 0;
  end;
end;

{ Exception routines }

procedure InvalidOperation(Str: PResStringRec); 
begin
  raise EInvalidGraphicOperation.CreateRes(Str);
end;

procedure InvalidGraphic(Str: PResStringRec);
begin
  raise EInvalidGraphic.CreateRes(Str);
end;

procedure InvalidBitmap; 
begin
  InvalidGraphic(@SInvalidBitmap);
end;

{procedure InvalidIcon;
begin
  InvalidGraphic(@SInvalidIcon);
end;

procedure InvalidMetafile;
begin
  InvalidGraphic(@SInvalidMetafile);
end;}

procedure OutOfResources;
begin
  raise EOutOfResources.Create(SOutOfResources);
end;


{ TFont ***********************************************************************}
constructor TFont.Create;
begin
  FCharset := DEFAULT_CHARSET;
  FColor := clWindowText;
  FHeight := -13;
  FName := 'Arial';
  FPitch := fpDefault;
  FStyle := [];
  FPixelsPerInch := ScreenLogPixels; 
end;

destructor TFont.Destroy;
begin
end;

procedure TFont.Changed;
begin
  inherited Changed;
  if FNotify <> nil then FNotify.Changed;
end;

procedure TFont.Assign(Source: TPersistent);
begin
  if Source is TFont then
  begin
    Lock;
    try
      TFont(Source).Lock;
      try
        Charset := TFont(Source).Charset;
        Color := TFont(Source).Color;
        Height := TFont(Source).Height;
        Name := TFont(Source).Name;
        Pitch := TFont(Source).Pitch;
        Style := TFont(Source).Style;
        if PixelsPerInch <> TFont(Source).PixelsPerInch then
          Size := TFont(Source).Size;
      finally
        TFont(Source).Unlock;
      end;
    finally
      Unlock;
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

procedure TFont.SetColor(Value: TColor);
begin
  if FColor <> Value then
  begin
    FColor := Value;
    Changed;
  end;
end;

function TFont.GetHeight: Integer;
begin
  Result := FHeight;
end;

procedure TFont.SetHeight(Value: Integer);
begin
  FHeight := Value;
end;

function TFont.GetName: TFontName;
begin
  Result := FName;
end;

procedure TFont.SetName(const Value: TFontName);
begin
  if (FileExists(DEFAULT_FONTPATH + PATH_DELIMITER + Value)) then
    FName := Value;
end;

function TFont.GetSize: Integer;
begin
  Result := -MulDiv(FHeight, 72, FPixelsPerInch);
end;

procedure TFont.SetSize(Value: Integer);
begin
  FHeight := -MulDiv(Value, FPixelsPerInch, 72);
end;

function TFont.GetStyle: TFontStyles;
begin
  Result := FStyle;
end;

procedure TFont.SetStyle(Value: TFontStyles);
var
  NewFontName: String;
begin
  NewFontName := FName;
  NewFontName := GetWord(NewFontName, ' Bold');
  NewFontName := GetWord(NewFontName, ' Italic');
  if (fsBold in Value) then
    NewFontName := NewFontName + ' Bold';
  if (fsItalic in Value) then
    NewFontName := NewFontName + ' Italic';
  if (FileExists(DEFAULT_FONTPATH + '\' + NewFontName)) then
  begin
    FName := NewFontName;
  end
  else
  begin
    Exclude(Value, fsBold);
    Exclude(Value, fsItalic);
  end;
  FStyle := Value;
end;

function TFont.GetPitch: TFontPitch;
begin
  Result := FPitch;
end;

procedure TFont.SetPitch(Value: TFontPitch);
begin
  if (FPitch <> Value) then
  begin
    FPitch := Value;
    case FPitch of
      fpDefault: Name := 'Arial';
      fpFixed: Name := 'Courier New';
      fpVariable: Name := 'Times New Roman';
    end;
  end;
end;

function TFont.GetCharset: TFontCharset;
begin
  Result := FCharset;
end;

procedure TFont.SetCharset(Value: TFontCharset);
begin
  if (FCharset <> Value) then
  begin
    FCharset := Value;
    if (FCharset = ANSI_CHARSET) then
      Name := 'Courier New'
    else if (FCharset = SYMBOL_CHARSET) then
      Name := 'Symbol';
  end;
end;

{ TPen ************************************************************************}

constructor TPen.Create;
begin
  FMode := pmCopy;
  FColor := clBlack;
  FStyle := psSolid;
  FWidth := 1;
end;

destructor TPen.Destroy;
begin
end;

procedure TPen.Assign(Source: TPersistent);
begin
  if Source is TPen then
  begin
    Lock;
    try
      TPen(Source).Lock;
      try
        Color := TPen(Source).Color;
        Style := TPen(Source).Style;
        Width := TPen(Source).Width;
        SetMode(TPen(Source).FMode);
      finally
        TPen(Source).Unlock;
      end;
    finally
      Unlock;
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

function TPen.GetColor: TColor;
begin
  Result := FColor;
end;

procedure TPen.SetColor(Value: TColor);
begin
  FColor := Value;
end;

procedure TPen.SetMode(Value: TPenMode);
begin
  if FMode <> Value then
  begin
    FMode := Value;
    Changed;
  end;
end;

function TPen.GetStyle: TPenStyle;
begin
  Result := FStyle;
end;

procedure TPen.SetStyle(Value: TPenStyle);
begin
  FStyle := Value;
end;

function TPen.GetWidth: Integer;
begin
  Result := FWidth;
end;

procedure TPen.SetWidth(Value: Integer);
begin
  if Value >= 0 then
  begin
    FWidth := Value;
  end;
end;

{ TBrush **********************************************************************}

constructor TBrush.Create;
begin
  ///FResource := BrushManager.AllocResource(DefBrushData);
end;

destructor TBrush.Destroy;
begin
  ///BrushManager.FreeResource(FResource);
end;

procedure TBrush.Assign(Source: TPersistent);
begin
  if Source is TBrush then
  begin
    Lock;
    try
      TBrush(Source).Lock;
      try
        Color := TBrush(Source).Color;
        Style := TBrush(Source).Style;
      finally
        TBrush(Source).Unlock;
      end;
    finally
      Unlock;
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

{///function TBrush.GetBitmap: TBitmap;
begin
  Result := FResource^.Brush.Bitmap;
end;

procedure TBrush.SetBitmap(Value: TBitmap);
var
  BrushData: TBrushData;
begin
  BrushData := DefBrushData;
  BrushData.Bitmap := Value;
  SetData(BrushData);
end;}

function TBrush.GetColor: TColor;
begin
  Result := FColor;
end;

procedure TBrush.SetColor(Value: TColor);
begin
  FColor := Value;
  if FStyle = bsClear then FStyle := bsSolid;
end;

function TBrush.GetStyle: TBrushStyle;
begin
  Result := FStyle;
end;

procedure TBrush.SetStyle(Value: TBrushStyle);
begin
  FStyle := Value;
  if FStyle = bsClear then FColor := clWhite;
end;

{ TCanvas *********************************************************************}

constructor TCanvas.Create;
begin
  inherited Create;
  InitializeCriticalSection(FLock);
  FFont := TFont.Create;
  FFont.OwnerCriticalSection := @FLock;
  FPen := TPen.Create;
  FPen.OwnerCriticalSection := @FLock;
  FBrush := TBrush.Create;
  FBrush.OwnerCriticalSection := @FLock;
  FCopyMode := cmSrcCopy;
end;

constructor TCanvas.CreateFull(AWidth, AHeight: Integer; AColor: TColor; APixelFormat: TPixelFormat);
begin
  Create;
  FWidth := AWidth;
  FHeight := AHeight;
  HandleNeeded;
end;

procedure TCanvas.HandleNeeded;
begin
  if (FPixelFormat = pf8Bit) then
  begin
    Handle := gdImageCreate(FWidth, FHeight);
{The first color allocated sets the background:}
    GetColorIndex(FColor);
  end
  else
  begin
    Handle := gdImageCreateTrueColor(FWidth, FHeight);
{The first color allocated DOES NOT set the background, which remains black:}
    gdImageFilledRectangle(Handle, 0, 0, FWidth-1, FHeight-1, GetColorIndex(FColor));
  end;
end;

destructor TCanvas.Destroy;
begin
  gdImageDestroy(Handle);

  FFont.Free;
  FPen.Free;
  FBrush.Free;
  DeleteCriticalSection(FLock);
  inherited Destroy;
end;

procedure TCanvas.Arc(X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
var
  s, e: Integer;
begin
  s := Round(Misc.GetAngleDeg(X3, Y3));
  e := Round(Misc.GetAngleDeg(X4, Y4));
  Arc(X1, Y1, X2, Y2, s, e, gdArc);
end;

{NB: iType = (gdArc, gdChord, gdPie):}
procedure TCanvas.Arc(X1, Y1, X2, Y2, StartAngle, EndAngle, iType: Integer);
begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  if (FBrush.Style = bsSolid) then
    gdImageFilledArc(Handle, (X1+X2) div 2, (Y1+Y2) div 2, X2-X1, Y2-Y1, StartAngle, EndAngle, GetColorIndex(FBrush.Color), iType);
  if (iType = gdPie) then
    iType := iType or gdEdged;
  if ((FBrush.Style <> bsSolid) or
      (FPen.Color <> FBrush.Color) or
      (FPen.Style <> psSolid)) then
  begin
    mColorStyle := GetColorIndex(FPen.Color);
    if (FPen.Style <> psSolid) then
      SetStyle(FPen.Style, mColorStyle);
    gdImageFilledArc(Handle, (X1+X2) div 2, (Y1+Y2) div 2, X2-X1, Y2-Y1, StartAngle, EndAngle, mColorStyle, iType+gdNoFill);
  end;
  FPenPos.x := X1;
  FPenPos.y := Y1;
  Changed;
end;

(*
procedure TCanvas.BrushCopy(const Dest: TRect; Bitmap: TBitmap;
  const Source: TRect; Color: TColor);
const
  ROP_DSPDxax = $00E20746;
var
  SrcW, SrcH, DstW, DstH: Integer;
  crBack, crText: TColorRef;
  MaskDC: HDC;
  Mask: TBitmap;
  MaskHandle: HBITMAP;
begin
  if Bitmap = nil then Exit;
  Lock;
  try
    Changing;
    RequiredState([csHandleValid, csBrushValid]);
    Bitmap.Canvas.Lock;
    try
      DstW := Dest.Right - Dest.Left;
      DstH := Dest.Bottom - Dest.Top;
      SrcW := Source.Right - Source.Left;
      SrcH := Source.Bottom - Source.Top;

      if Bitmap.TransparentColor = Color then
      begin
        Mask := nil;
        MaskHandle := Bitmap.MaskHandle;
        MaskDC := CreateCompatibleDC(0);
        MaskHandle := SelectObject(MaskDC, MaskHandle);
      end
      else
      begin
        Mask := TBitmap.Create;
        Mask.Assign(Bitmap);
        { Replace Color with black and all other colors with white }
        Mask.Mask(Color);
        Mask.Canvas.RequiredState([csHandleValid]);
        MaskDC := Mask.Canvas.FHandle;
        MaskHandle := 0;
      end;

      try
        Bitmap.Canvas.RequiredState([csHandleValid]);
        { Draw transparently or use brush color to fill background }
        if Brush.Style = bsClear then
        begin
          TransparentStretchBlt(FHandle, Dest.Left, Dest.Top, DstW, DstH,
            Bitmap.Canvas.FHandle, Source.Left, Source.Top, SrcW, SrcH,
            MaskDC, Source.Left, Source.Top);
        end
        else
        begin
          StretchBlt(FHandle, Dest.Left, Dest.Top, DstW, DstH,
            Bitmap.Canvas.FHandle, Source.Left, Source.Top, SrcW, SrcH, SrcCopy);
          crText := SetTextColor(Self.FHandle, 0);
          crBack := SetBkColor(Self.FHandle, $FFFFFF);
          StretchBlt(Self.FHandle, Dest.Left, Dest.Top, DstW, DstH,
            MaskDC, Source.Left, Source.Top, SrcW, SrcH, ROP_DSPDxax);
          SetTextColor(Self.FHandle, crText);
          SetBkColor(Self.FHandle, crBack);
        end;
      finally
        if Assigned(Mask) then Mask.Free
        else
        begin
          if MaskHandle <> 0 then SelectObject(MaskDC, MaskHandle);
          DeleteDC(MaskDC);
        end;
      end;
    finally
      Bitmap.Canvas.Unlock;
    end;
    Changed;
  finally
    Unlock;
  end;
end;
*)

procedure TCanvas.Chord(X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
var
  s, e: Integer;
begin
  s := Round(Misc.GetAngleDeg(X3, Y3));
  e := Round(Misc.GetAngleDeg(X4, Y4));
  Arc(X1, Y1, X2, Y2, s, e, gdChord);
end;

procedure TCanvas.CopyRect(const Dest: TRect; Canvas: TCanvas;
  const Source: TRect);
begin
  Changing;
  if (((Source.Right-Source.Left) = (Source.Right-Source.Left)) and
      ((Source.Bottom-Source.Top) = (Source.Bottom-Source.Top))) then
  begin
    gdImageCopy(Handle, Canvas.Handle,
      Dest.Left, Dest.Top,
      Source.Left, Source.Top,
      (Source.Right-Source.Left), (Source.Bottom-Source.Top));
  end
  else
  begin
    gdImageCopyResized(Handle, Canvas.Handle,
      Dest.Left, Dest.Top,
      Source.Left, Source.Top,
      (Dest.Right-Dest.Left), (Dest.Bottom-Dest.Top),
      (Source.Right-Source.Left), (Source.Bottom-Source.Top));
  end;           
  Changed;
end;
(*
procedure TCanvas.Draw(X, Y: Integer; Graphic: TGraphic);
begin
  if (Graphic <> nil) and not Graphic.Empty then
  begin
    Changing;
    RequiredState([csHandleValid]);
    SetBkColor(FHandle, ColorToRGB(FBrush.Color));
    SetTextColor(FHandle, ColorToRGB(FFont.Color));
    ///Graphic.Draw(Self, Rect(X, Y, X + Graphic.Width, Y + Graphic.Height));
    Changed;
  end;
end;
*)
procedure TCanvas.DrawFocusRect(const Rect: TRect);
begin
  Changing;
  Rectangle(Rect);
  Changed;
end;

procedure TCanvas.Ellipse(X1, Y1, X2, Y2: Integer);
begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  if (FBrush.Style = bsSolid) then
    gdImageFilledEllipse(Handle, X1, Y1, X2-X1, Y2-Y1, GetColorIndex(FBrush.Color));
  if ((FBrush.Style <> bsSolid) or
      (FPen.Color <> FBrush.Color) or
      (FPen.Style <> psSolid)) then
  begin
    mColorStyle := GetColorIndex(FPen.Color);
    if (FPen.Style <> psSolid) then
      SetStyle(FPen.Style, mColorStyle);
    gdImageEllipse(Handle, X1, Y1, X2-X1, Y2-Y1, mColorStyle);
  end;
  FPenPos.x := (X1+X2) div 2;
  FPenPos.y := (Y1+Y2) div 2;
  Changed;
end;

procedure TCanvas.Ellipse(const Rect: TRect);
begin
  Ellipse(Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
end;

procedure TCanvas.FillRect(const Rect: TRect);
begin
  Changing;
  gdImageFilledRectangle(Handle, Rect.Left, Rect.Top, Rect.Right-1, Rect.Bottom-1, GetColorIndex(FBrush.Color));
  FPenPos.x := Rect.Left;
  FPenPos.y := Rect.Top;
  Changed;
end;

procedure TCanvas.FloodFill(X, Y: Integer; Color: TColor;
  FillStyle: TFillStyle);
const
  FillStyles: array[TFillStyle] of Word =
    (FLOODFILLSURFACE, FLOODFILLBORDER);
begin
  Changing;
  if (fsSurface = FillStyle) then
    gdImageFill(Handle, X, Y, GetColorIndex(FBrush.Color));
  if (fsBorder = FillStyle) then
    gdImageFillToBorder(Handle, X, Y, GetColorIndex(Color), GetColorIndex(FBrush.Color));
  FPenPos.x := X;
  FPenPos.y := Y;
  Changed;
end;

procedure TCanvas.FrameRect(const Rect: TRect);
begin
  Changing;
  gdImageSetThickness(Handle, 1);
  mColorStyle := GetColorIndex(FPen.Color);
  if (FPen.Style <> psSolid) then
    SetStyle(FPen.Style, mColorStyle);
  gdImageRectangle(Handle, Rect.Left, Rect.Top, Rect.Right, Rect.Bottom, mColorStyle);
  FPenPos.x := Rect.Left;
  FPenPos.y := Rect.Top;
  Changed;
end;

procedure TCanvas.SetStyle(Value: TPenStyle; ColorIndex: Integer);
var
  i: Integer;
begin
  case Value of
    psSolid, psInsideFrame: for i := 0 to 11 do mLineStyle[i] := ColorIndex;
    psDash:
      begin
        for i := 0 to 5 do mLineStyle[i] := ColorIndex;
        for i := 6 to 11 do mLineStyle[i] := gdTransparent;
      end;
    psDot:
      begin
        for i := 0 to 5 do
        begin
          mLineStyle[2*i] := ColorIndex;
          mLineStyle[2*i+1] := gdTransparent;
        end;
      end;
    psDashDot:
      begin
        for i := 0 to 6 do mLineStyle[i] := ColorIndex;
        mLineStyle[7] := gdTransparent;
        mLineStyle[8] := gdTransparent;
        mLineStyle[9] := ColorIndex;
        mLineStyle[10] := gdTransparent;
        mLineStyle[11] := gdTransparent;
      end;
    psDashDotDot:
      begin
        for i := 0 to 6 do mLineStyle[i] := ColorIndex;
        mLineStyle[7] := gdTransparent;
        mLineStyle[8] := ColorIndex;
        mLineStyle[9] := gdTransparent;
        mLineStyle[10] := ColorIndex;
        mLineStyle[11] := gdTransparent;
      end;
    psClear: for i := 0 to 11 do mLineStyle[i] := gdTransparent;
  end;
  gdImageSetStyle(Handle, @mLineStyle[0], 12);
  mColorStyle := gdStyled;
end;

procedure TCanvas.LineTo(X, Y: Integer);
begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  mColorStyle := GetColorIndex(FPen.Color);
  if (FPen.Style <> psSolid) then
    SetStyle(FPen.Style, mColorStyle); // this sets mColorStyle
  gdImageLine(Handle, FPenPos.x, FPenPos.y, X, Y, mColorStyle);
  FPenPos.x := X;
  FPenPos.y := Y;
  Changed;
end;

procedure TCanvas.Lock;
begin
  EnterCriticalSection(CounterLock);
  Inc(FLockCount);
  LeaveCriticalSection(CounterLock);
  EnterCriticalSection(FLock);
end;

procedure TCanvas.MoveTo(X, Y: Integer);
begin
  FPenPos.x := X;
  FPenPos.y := Y;
end;

procedure TCanvas.Pie(X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
var
  s, e: Integer;
begin
  s := Round(Misc.GetAngleDeg(X3, Y3));
  e := Round(Misc.GetAngleDeg(X4, Y4));
  Arc(X1, Y1, X2, Y2, s, e, gdPie);
end;

type
  PPoints = ^TPoints;
  TPoints = array[0..0] of TPoint;

procedure TCanvas.Polygon(const Points: array of TPoint);
begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  //gdImagePolygon(Handle, Points, int pointsTotal, int color);
  Changed;
end;

procedure TCanvas.Polyline(const Points: array of TPoint);
var
  i: Integer;
begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  mColorStyle := GetColorIndex(FPen.Color);
  if (FPen.Style <> psSolid) then
    SetStyle(FPen.Style, mColorStyle);
  for i := 1 to Length(Points)-1 do
  begin
    gdImageLine(Handle, Points[i-1].x, Points[i-1].y, Points[i].x, Points[i].y, mColorStyle);
  end;
  FPenPos.x := Points[Length(Points)-1].x;
  FPenPos.x := Points[Length(Points)-1].y;
  Changed;
end;

procedure TCanvas.PolyBezier(const Points: array of TPoint);
begin
  Changing;
  Changed;
end;

procedure TCanvas.PolyBezierTo(const Points: array of TPoint);
begin
  Changing;
  Changed;
end;

procedure TCanvas.Rectangle(X1, Y1, X2, Y2: Integer);
begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  if (FBrush.Style = bsSolid) then
    gdImageFilledRectangle(Handle, X1, Y1, X2-X1, Y2-Y1, GetColorIndex(FBrush.Color));
  if ((FBrush.Style <> bsSolid) or
      (FPen.Color <> FBrush.Color) or
      (FPen.Style <> psSolid)) then
  begin
    mColorStyle := GetColorIndex(FPen.Color);
    if (FPen.Style <> psSolid) then
      SetStyle(FPen.Style, mColorStyle);
    gdImageRectangle(Handle, X1, Y1, X2-X1, Y2-Y1, mColorStyle);
  end;

  gdImageRectangle(Handle, X1, Y1, X2, Y2, GetColorIndex(FPen.Color));
  Changed;
end;

procedure TCanvas.Rectangle(const Rect: TRect);
begin
  Rectangle(Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
end;

procedure TCanvas.Refresh;
begin
end;

procedure TCanvas.RoundRect(X1, Y1, X2, Y2, X3, Y3: Integer);
{NB: X3, Y3 are width and height of ellipse at corners}

  procedure Outline(TheColorStyle: Integer);
  begin
{Draw top-left arc:}
    gdImageArc(Handle, X1 + (X3 div 2), Y1 + (Y3 div 2), X3, Y3, 270, 0, TheColorStyle);
{Draw top-right arc:}
    gdImageArc(Handle, X2 - (X3 div 2), Y1 + (Y3 div 2), X3, Y3, 0, 90, TheColorStyle);
{Draw bottom-left arc:}
    gdImageArc(Handle, X1 + (X3 div 2), Y1 - (Y3 div 2), X3, Y3, 180, 270, TheColorStyle);
{Draw bottom-right arc:}
    gdImageArc(Handle, X2 - (X3 div 2), Y1 - (Y3 div 2), X3, Y3, 90, 180, TheColorStyle);
{Draw the four lines:}
    gdImageLine(Handle, X1 + (X3 div 2), Y1, X2 - (X3 div 2), Y1, TheColorStyle);
    gdImageLine(Handle, X1 + (X3 div 2), Y2, X2 - (X3 div 2), Y2, TheColorStyle);
    gdImageLine(Handle, X1, Y1 + (Y3 div 2), X1, Y2 - (Y3 div 2), TheColorStyle);
    gdImageLine(Handle, X2, Y1 + (Y3 div 2), X2, Y2 - (Y3 div 2), TheColorStyle);
  end;

begin
  Changing;
  gdImageSetThickness(Handle, FPen.Width);
  Outline(GetColorIndex(FPen.Color));
  gdImageFill(Handle, (X1+X2) div 2, (Y1+Y2) div 2, GetColorIndex(FBrush.Color));
  if (FPen.Style <> psSolid) then
  begin
    SetStyle(FPen.Style, GetColorIndex(FPen.Color));
    Outline(gdStyled);
  end;
  Changed;
end;

function TCanvas.GetCanvasOrientation: TCanvasOrientation;
var
  Point: TPoint;
begin
  Result := coLeftToRight;
  if (FTextFlags and ETO_RTLREADING) <> 0 then
  begin
{just for gd:}
    Point.x := 0;
    Point.y := 0;
    if Point.X <> 0 then Result := coRightToLeft
  end;
end;

procedure TCanvas.TextOut(X, Y: Integer; const Text: String);
begin
  TextOutAngle(0, X, Y, Text);
end;

{------------------------------------------------------------------------------
    Procedure: TextOutAngle
  Description: draws angled text on the input canvas
      Authors: Mat Ballard
 Date created: 10/15/2002
Date modified: 10/15/2002 by Mat Ballard
      Purpose: Vertical fonts
 Known Issues: derived from the very early GPC work;
               ACanvas.Font does not remain rotated
               Note: Angle of rotation is Anti-Clockwise in Win32,
               Clockwise in Qt/Linux
 ------------------------------------------------------------------------------}
procedure TCanvas.TextOutAngle(Angle, X, Y: Integer; Text: String);
var
  ColorIndex: Integer;
  dAngle: double;
  brect: array [0..7] of Integer;
{0      lower left corner, X position
 1      lower left corner, Y position
 2      lower right corner, X position
 3      lower right corner, Y position
 4      upper right corner, X position
 5      upper right corner, Y position
 6      upper left corner, X position
 7      upper left corner, Y position}
begin
  Changing;
  dAngle := Angle;
  ColorIndex := GetColorIndex(FFont.Color);
  gdImageStringFT(Handle, @brect[0], ColorIndex, PChar(FFont.Name), FFont.Size, dAngle, X, Y, PChar(Text));
  if (fsUnderline in FFont.Style) then
  begin
    gdImageSetThickness(Handle, 2);
    gdImageLine(Handle, brect[0], brect[1], brect[2], brect[3], ColorIndex);
  end;
  if (fsStrikeout in FFont.Style) then
  begin
    gdImageSetThickness(Handle, 2);
    gdImageLine(Handle,
      (brect[0] + brect[6]) div 2,
      (brect[1] + brect[7]) div 2,
      (brect[2] + brect[4]) div 2,
      (brect[3] + brect[5]) div 2,
      ColorIndex);
  end;
  MoveTo(brect[4], brect[5]);
  Changed;
end;

procedure TCanvas.TextRect(Rect: TRect; X, Y: Integer; Text: string);
var
  ColorIndex, NoChars: Integer;
  brect: array [0..7] of Integer;
begin
  Changing;
  ColorIndex := GetColorIndex(FFont.Color);
{obtain brect so that we can size the image}
  gdImageStringFT(nil, @brect[0], ColorIndex, PChar(FFont.Name), FFont.Size, 0, 0, 0, PChar(Text));
  if ((Rect.Right - Rect.Left) < (brect[2] - brect[0])) then
  begin
    NoChars := Length(Text) * (Rect.Right - Rect.Left) div (brect[2] - brect[0]);
    SetLength(Text, NoChars);
  end;
  TextOut(X, Y, Text);
  Changed;
end;

function TCanvas.TextExtent(const Text: string): TSize;
var
  brect: array [0..7] of Integer;
{0      lower left corner, X position
 1      lower left corner, Y position
 2      lower right corner, X position
 3      lower right corner, Y position
 4      upper right corner, X position
 5      upper right corner, Y position
 6      upper left corner, X position
 7      upper left corner, Y position}
begin
  gdImageStringFT(nil, @brect[0], GetColorIndex(FFont.Color), PChar(FFont.Name), FFont.Size, 0.0, 0, 0, PChar(Text));
  Result.cx := brect[2] - brect[0];
  Result.cy := brect[3] - brect[5];
end;

function TCanvas.TextWidth(const Text: string): Integer;
begin
  Result := TextExtent(Text).cX;
end;

function TCanvas.TextHeight(const Text: string): Integer;
begin
  Result := TextExtent(Text).cY;
end;

function TCanvas.TryLock: Boolean;
begin
  EnterCriticalSection(CounterLock);
  try
    Result := FLockCount = 0;
    if Result then Lock;
  finally
    LeaveCriticalSection(CounterLock);
  end;
end;

procedure TCanvas.Unlock;
begin
  LeaveCriticalSection(FLock);
  EnterCriticalSection(CounterLock);
  Dec(FLockCount);
  LeaveCriticalSection(CounterLock);
end;

procedure TCanvas.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
end;

procedure TCanvas.SetHandle(Value: PgdImage);
begin
  if (im = nil) then
  begin
    im := Value;
  end
  else
  begin
    gdImageDestroy(im);
    im := Value;
  end;
end;

procedure TCanvas.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
end;

procedure TCanvas.SetBrush(Value: TBrush);
begin
  FBrush.Assign(Value);
end;

function TCanvas.GetPenPos: TPoint;
begin
  Result := Point(FPenPos.x, FPenPos.y);
end;

procedure TCanvas.SetPenPos(Value: TPoint);
begin
  MoveTo(Value.X, Value.Y);
end;

function TCanvas.GetPixel(X, Y: Integer): TColor;
var
  c: Integer;
begin
  c := gdImageGetPixel(Handle, X, Y);
  Result := gdImageRed(Handle, c) + gdImageBlue(Handle, c) shr 8 + gdImageGreen(Handle, c) shr 16;
end;

procedure TCanvas.SetPixel(X, Y: Integer; Value: TColor);
begin
  Changing;
  gdImageSetPixel(Handle, X, Y, GetColorIndex(Value));
  Changed;
end;

function TCanvas.GetClipRect: TRect;
begin
  Result.Left := 0;
  Result.Top := 0;
  Result.Right := Handle.sx-1;
  Result.Bottom := Handle.sy-1;
end;

function TCanvas.GetColorIndex(Color: TColor): Integer;
var
  R, G, B: Integer;
begin
  R := Color and 255;
  G := (Color shr 8) and 255;
  B := (Color shr 16) and 255;
{Find the existing color:}
  Result := gdImageColorResolve(Handle, R, G, B);
end;

procedure TCanvas.Changing;
begin
  if Assigned(FOnChanging) then FOnChanging(Self);
end;

procedure TCanvas.Changed;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TCanvas.GetHandle: PgdImage;
begin
  if (im = nil) then
  begin
    HandleNeeded;
  end;
  Result := im;
end;

{ TGraphic ********************************************************************}

constructor TGraphic.Create;
begin
  inherited Create;
end;

procedure TGraphic.Changed(Sender: TObject);
begin
  FModified := True;
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TGraphic.DefineProperties(Filer: TFiler);

  function DoWrite: Boolean;
  begin
    if Filer.Ancestor <> nil then
      Result := not (Filer.Ancestor is TGraphic) or
        not Equals(TGraphic(Filer.Ancestor))
    else
      Result := not Empty;
  end;

begin
  Filer.DefineBinaryProperty('Data', ReadData, WriteData, DoWrite);
end;

function TGraphic.Equals(Graphic: TGraphic): Boolean;
var
  MyImage, GraphicsImage: TMemoryStream;
begin
  Result := (Graphic <> nil) and (ClassType = Graphic.ClassType);
  if Empty or Graphic.Empty then
  begin
    Result := Empty and Graphic.Empty;
    Exit;
  end;
  if Result then
  begin
    MyImage := TMemoryStream.Create;
    try
      WriteData(MyImage);
      GraphicsImage := TMemoryStream.Create;
      try
        Graphic.WriteData(GraphicsImage);
        Result := (MyImage.Size = GraphicsImage.Size) and
          CompareMem(MyImage.Memory, GraphicsImage.Memory, MyImage.Size);
      finally
        GraphicsImage.Free;
      end;
    finally
      MyImage.Free;
    end;
  end;
end;

function TGraphic.GetPalette: HPALETTE;
begin
  Result := 0;
end;

function TGraphic.GetTransparent: Boolean;
begin
  Result := FTransparent;
end;

procedure TGraphic.LoadFromFile(const Filename: string);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(Filename, fmOpenRead or fmShareDenyWrite);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TGraphic.Progress(Sender: TObject; Stage: TProgressStage;
  PercentDone: Byte; RedrawNow: Boolean; const R: TRect; const Msg: string);
begin
  if Assigned(FOnProgress) then
    FOnProgress(Sender, Stage, PercentDone, RedrawNow, R, Msg);
end;

procedure TGraphic.ReadData(Stream: TStream);
begin
  LoadFromStream(Stream);
end;

procedure TGraphic.SaveToFile(const Filename: string);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(Filename, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TGraphic.SetPalette(Value: HPalette);
begin
end;

procedure TGraphic.SetModified(Value: Boolean);
begin
  if Value then
    Changed(Self) else
    FModified := False;
end;

procedure TGraphic.SetTransparent(Value: Boolean);
begin
  if Value <> FTransparent then
  begin
    FTransparent := Value;
    Changed(Self);
  end;
end;

procedure TGraphic.WriteData(Stream: TStream);
begin
  SaveToStream(Stream);
end;

{ TBitmap *********************************************************************}
constructor TBitmap.Create;
begin
  inherited Create;
  FTransparentColor := clDefault;
  HandleType := bmDIB;
  FCanvas := nil;
end;

constructor TBitmap.CreateFull(Width, Height: Integer; BackgroundColor: TColor; APixelFormat: TPixelFormat);
begin
  Create;
  FTransparentColor := BackgroundColor;
  SetPixelFormat(APixelFormat);
  FCanvas := TCanvas.Create(Width, Height, TransparentColor, (FPixelFormat = pf24bit) or (FPixelFormat = pf32bit));
end;

destructor TBitmap.Destroy;
begin
  if (FCanvas <> nil) then
    FCanvas.Free;
  inherited Destroy;
end;

procedure TBitmap.Assign(Source: TPersistent);
{var
  DIB: TDIBSection;}
begin
  if (Source = nil) or (Source is TBitmap) then
  begin
    {///EnterCriticalSection(BitmapImageLock);
    try
      if Source <> nil then
      begin
        TBitmap(Source).FImage.Reference;
        FImage.Release;
        FImage := TBitmap(Source).FImage;
        FTransparent := TBitmap(Source).FTransparent;
        FTransparentColor := TBitmap(Source).FTransparentColor;
        FTransparentMode := TBitmap(Source).FTransparentMode;
      end
      else
      begin
        FillChar(DIB, Sizeof(DIB), 0);
        NewImage(0, 0, DIB, False);
      end;
    finally
      LeaveCriticalSection(BitmapImageLock);
    end;
    PaletteModified := Palette <> 0;
    Changed(Self);}
  end
  else inherited Assign(Source);
end;

procedure TBitmap.CopyImage(AHandle: HBITMAP; APalette: HPALETTE; DIB: TDIBSection);
begin
end;

procedure TBitmap.Changing(Sender: TObject);
begin
end;

procedure TBitmap.Changed(Sender: TObject);
begin
  FMaskBitsValid := False;
  inherited Changed(Sender);
end;

procedure TBitmap.Dormant;
begin
end;

procedure TBitmap.Draw(ACanvas: TCanvas; const Rect: TRect);
{///var
  OldPalette: HPalette;
  RestorePalette: Boolean;
  DoHalftone: Boolean;
  Pt: TPoint;
  BPP: Integer;
  MaskDC: HDC;
  Save: THandle;}
begin
(*  with Rect, FImage do
  begin
    ACanvas.RequiredState(csAllValid);
    PaletteNeeded;
    OldPalette := 0;
    RestorePalette := False;

    if FPalette <> 0 then
    begin
      OldPalette := SelectPalette(ACanvas.FHandle, FPalette, True);
      RealizePalette(ACanvas.FHandle);
      RestorePalette := True;
    end;
    BPP := GetDeviceCaps(ACanvas.FHandle, BITSPIXEL) *
      GetDeviceCaps(ACanvas.FHandle, PLANES);
    DoHalftone := (BPP <= 8) and (BPP < (FDIB.dsbm.bmBitsPixel * FDIB.dsbm.bmPlanes));
    if DoHalftone then
    begin
      GetBrushOrgEx(ACanvas.FHandle, pt);
      SetStretchBltMode(ACanvas.FHandle, HALFTONE);
      SetBrushOrgEx(ACanvas.FHandle, pt.x, pt.y, @pt);
    end else if not Monochrome then
      SetStretchBltMode(ACanvas.Handle, STRETCH_DELETESCANS);
    try
      { Call MaskHandleNeeded prior to creating the canvas handle since
        it causes FreeContext to be called. }
      if Transparent then MaskHandleNeeded;
      Canvas.RequiredState(csAllValid);
      if Transparent then
      begin
        Save := 0;
        MaskDC := 0;
        try
          MaskDC := GDICheck(CreateCompatibleDC(0));
          Save := SelectObject(MaskDC, FMaskHandle);
          TransparentStretchBlt(ACanvas.FHandle, Left, Top, Right - Left,
            Bottom - Top, Canvas.FHandle, 0, 0, FDIB.dsbm.bmWidth,
            FDIB.dsbm.bmHeight, MaskDC, 0, 0);
        finally
          if Save <> 0 then SelectObject(MaskDC, Save);
          if MaskDC <> 0 then DeleteDC(MaskDC);
        end;
      end
      else
        StretchBlt(ACanvas.FHandle, Left, Top, Right - Left, Bottom - Top,
          Canvas.FHandle, 0, 0, FDIB.dsbm.bmWidth,
          FDIB.dsbm.bmHeight, ACanvas.CopyMode);
    finally
      if RestorePalette then
        SelectPalette(ACanvas.FHandle, OldPalette, True);
    end;
  end;   *)
end;

{ FreeImage:
  If there are multiple references to the image, create a unique copy of the image.
  If FHandle = FDIBHandle, the DIB memory will be updated when the drawing
  handle is drawn upon, so no changes are needed to maintain image integrity.
  If FHandle <> FDIBHandle, the DIB will not track with changes made to
  the DDB, so destroy the DIB handle (but keep the DIB pixel format info).  }
procedure TBitmap.FreeImage;
begin
end;

function TBitmap.GetEmpty;
begin
  Result := (FCanvas.Handle = nil);
end;

function TBitmap.GetCanvas: TCanvas;
begin
  if FCanvas = nil then
  begin
{you get either 8 or 24 bit images:}
    FCanvas := TCanvas.Create(FWidth, FHeight, FBackgroundColor, (PixelFormat = pf24bit));
    FCanvas.OnChange := Changed;
    FCanvas.OnChanging := Changing;
  end;
  Result := FCanvas;
end;

(*
{ Since the user might modify the contents of the HBITMAP it must not be
  shared by another TBitmap when given to the user nor should it be selected
  into a DC. }
function TBitmap.GetHandle: HBITMAP;
begin
  FreeContext;
  HandleNeeded;
  Changing(Self);
  Result := FImage.FHandle;
end;
*)

function TBitmap.GetHandleType: TBitmapHandleType;
begin
  Result := bmDIB
end;

function TBitmap.GetHeight: Integer;
begin
  Result := FCanvas.Handle^.sy;
end;

function TBitmap.GetMaskHandle: HBITMAP;
begin
  ///MaskHandleNeeded;
  ///Result := FImage.FMaskHandle;
end;
(*
function TBitmap.GetMonochrome: Boolean;
begin
  ///with FImage.FDIB.dsbm do Result := (bmPlanes = 1) and (bmBitsPixel = 1);
end;

function TBitmap.GetPalette: HPALETTE;
begin
  PaletteNeeded;
  Result := FImage.FPalette;
end;
*)
function TBitmap.GetPixelFormat: TPixelFormat;
begin
  Result := FPixelFormat;
end;

(*
function TBitmap.GetScanLine(Row: Integer): Pointer;
begin
  Changing(Self);
  with FImage.FDIB, dsbm, dsbmih do
  begin
    if (Row < 0) or (Row > bmHeight) then
      InvalidOperation(@SScanLine);
    DIBNeeded;
    GDIFlush;
    if biHeight > 0 then  // bottom-up DIB
      Row := biHeight - Row - 1;
    Integer(Result) := Integer(bmBits) +
      Row * BytesPerScanline(biWidth, biBitCount, 32);
  end;
end;
*)

function TBitmap.GetTransparentColor: TColor;
begin
  if FTransparentColor = clDefault then
  begin
    {if Monochrome then
      Result := clWhite
    else}
      Result := Canvas.Pixels[0, Height - 1];
  end
  else Result := ColorToRGB(FTransparentColor);
  Result := Result or $02000000;
end;

function TBitmap.GetWidth: Integer;
begin
  Result := FCanvas.Handle^.sx;
end;
(*
procedure TBitmap.DIBNeeded;
begin
  with FImage do
  begin
    if (FHandle = 0) or (FDIBHandle <> 0) then Exit;
    PaletteNeeded;
    if FDIB.dsbmih.biSize = 0 then
    begin
      GetObject(FHandle, sizeof(FDIB), @FDIB);
      with FDIB, dsbm, dsbmih do
      begin
        biSize := sizeof(dsbmih);
        biWidth := bmWidth;
        biHeight := bmHeight;
        biPlanes := 1;
        biBitCount := bmPlanes * bmBitsPixel;
      end;
    end;
    FDIBHandle := CopyBitmap(FHandle, FPalette, FPalette, FDIB, nil);
  end;
end;

procedure TBitmap.FreeContext;
begin
  if (FCanvas <> nil) then TBitmapCanvas(FCanvas).FreeContext;
end;


procedure TBitmap.HandleNeeded;
begin
  with FImage do
    if FHandle = 0 then
      FHandle := FDIBHandle;
end;
*)
procedure TBitmap.Mask(TransparentColor: TColor);
{var
  NewHandle, NewPalette: THandle;
  DIB: TDIBSection;}
begin
{  NewHandle := 0;
  NewPalette := 0;
  try
    FreeContext;
    HandleNeeded;
    NewHandle := CopyBitmapAsMask(FImage.FHandle, FImage.FPalette,
      ColorToRGB(TransparentColor));
    FillChar(DIB, SizeOf(DIB), 0);
    GetObject(NewHandle, SizeOf(DIB), @DIB);
    if FImage.FPalette = SystemPalette16 then
      NewPalette := FImage.FPalette
    else
      NewPalette := CopyPalette(FImage.FPalette);
    NewImage(NewHandle, NewPalette, DIB, FImage.FOS2Format);
  except
    InternalDeletePalette(NewPalette);
    if NewHandle <> 0 then DeleteObject(NewHandle);
    raise;
  end;
  Changed(Self);}
end;

procedure TBitmap.MaskHandleNeeded;
begin
(*  if FMaskValid and FMaskBitsValid then Exit;
  with FImage do
  begin
    { Delete existing mask if any }
    if FMaskHandle <> 0 then
    begin
      DeselectBitmap(FMaskHandle);
      DeleteObject(FMaskHandle);
      FMaskHandle := 0;
    end;
    FreeContext;
    HandleNeeded;
    FMaskHandle := CopyBitmapAsMask(FHandle, FPalette, GetTransparentColor);
    FMaskValid := True;
    FMaskBitsValid := True;
  end; *)
end;

(*
procedure TBitmap.PaletteNeeded;
var
  DC: HDC;
begin
  with FImage do
  begin
    if FIgnorePalette or (FPalette <> 0) or (FDIBHandle = 0) then Exit;
    if FHandle = FDIBHandle then DeselectBitmap(FDIBHandle);
    FPalette := PaletteFromDIBColorTable(FDIBHandle, nil, 1 shl FDIB.dsbmih.biBitCount);
    if FPalette <> 0 then Exit;
    DC := GDICheck(GetDC(0));
    FHalftone := FHalftone or
      ((GetDeviceCaps(DC, BITSPIXEL) * GetDeviceCaps(DC, PLANES)) <
      (FDIB.dsbm.bmBitsPixel * FDIB.dsbm.bmPlanes));
    if FHalftone then FPalette := CreateHalftonePalette(DC);
    ReleaseDC(0, DC);
    if FPalette = 0 then IgnorePalette := True;
  end;
end;

procedure TBitmap.LoadFromClipboardFormat(AFormat: Word; AData: THandle;
  APalette: HPALETTE);
var
  DIB: TDIBSection;
begin
  if (AFormat <> CF_BITMAP) or (AData = 0) then
    InvalidGraphic(@SUnknownClipboardFormat);
  FreeContext;
  FillChar(DIB, sizeof(DIB), 0);
  GetObject(AData, sizeof(DIB), @DIB);
  if DIB.dsbm.bmBits = nil then DIB.dsbmih.biSize := 0;
  CopyImage(AData, APalette, DIB);
  FImage.FOS2Format := False;
  PaletteModified := Palette <> 0;
  Changed(Self);
end;
*)

procedure TBitmap.LoadFromFile(const Filename: string);
var
  Ext: String;
  F: File;
begin
  Ext := Lowercase(ExtractFileExt(Filename));
  AssignFile(F, Filename);
  FileMode := 0;  {Set file access to read only }
  Reset(F);
  if (Ext = '.png') then
    FCanvas.Handle := gdImageCreateFromPng(@F)
  else if ((Ext = '.jpg') or (Ext = '.jpg')) then
    FCanvas.Handle := gdImageCreateFromJpeg(@F)
  else
  begin
    EComponentError.CreateFmt('"%s" is not a supported format !', [Ext]);
  end;
  CloseFile(F);
end;

procedure TBitmap.LoadFromStream(Stream: TStream);
begin
  ///ReadStream(Stream, Stream.Size - Stream.Position);
end;

procedure TBitmap.LoadFromResourceName(Instance: THandle; const ResName: string);
var
  Stream: TCustomMemoryStream;
begin
  Stream := TResourceStream.Create(Instance, ResName, MakeIntResource(RT_BITMAP));
  try
    ///ReadDIB(Stream, Stream.Size);
  finally
    Stream.Free;
  end;
end;

procedure TBitmap.LoadFromResourceID(Instance: THandle; ResID: Integer);
var
  Stream: TCustomMemoryStream;
begin
  Stream := TResourceStream.CreateFromID(Instance, ResID, MakeIntResource(RT_BITMAP));
  try
    ///ReadDIB(Stream, Stream.Size);
  finally
    Stream.Free;
  end;
end;

(*
procedure TBitmap.NewImage(NewHandle: HBITMAP; NewPalette: HPALETTE;
  const NewDIB: TDIBSection; OS2Format: Boolean; RLEStream: TStream = nil);
var
  Image: TBitmapImage;
begin
  Image := TBitmapImage.Create;
  with Image do
  try
    FHandle := NewHandle;
    FPalette := NewPalette;
    FDIB := NewDIB;
    FOS2Format := OS2Format;
    if FDIB.dsbm.bmBits <> nil then FDIBHandle := FHandle;
    FSaveStream := RLEStream as TMemoryStream;
  except
    Image.Free;
    raise;
  end;
  //!! replace with InterlockedExchange()
  EnterCriticalSection(BitmapImageLock);
  try
    FImage.Release;
    FImage := Image;
    FImage.Reference;
  finally
    LeaveCriticalSection(BitmapImageLock);
  end;
  FMaskValid := False;
end;
*)
(*
procedure TBitmap.ReadData(Stream: TStream);
var
  Size: Longint;
begin
  Stream.Read(Size, SizeOf(Size));
  ReadStream(Stream, Size);
end;
*)
(*
procedure TBitmap.ReadDIB(Stream: TStream; ImageSize: LongWord);
const
  DIBPalSizes: array [Boolean] of Byte = (sizeof(TRGBQuad), sizeof(TRGBTriple));
var
  DC, MemDC: HDC;
  BitsMem: Pointer;
  OS2Header: TBitmapCoreHeader;
  BitmapInfo: PBitmapInfo;
  ColorTable: Pointer;
  HeaderSize: Integer;
  OS2Format: Boolean;
  BMHandle, OldBMP: HBITMAP;
  DIB: TDIBSection;
  Pal, OldPal: HPalette;
  RLEStream: TStream;
begin
  Pal := 0;
  BMHandle := 0;
  RLEStream := nil;
  Stream.Read(HeaderSize, sizeof(HeaderSize));
  OS2Format := HeaderSize = sizeof(OS2Header);
  if OS2Format then HeaderSize := sizeof(TBitmapInfoHeader);
  GetMem(BitmapInfo, HeaderSize + 12 + 256 * sizeof(TRGBQuad));
  with BitmapInfo^ do
  try
    try
      if OS2Format then  // convert OS2 DIB to Win DIB
      begin
        Stream.Read(Pointer(Longint(@OS2Header) + sizeof(HeaderSize))^,
          sizeof(OS2Header) - sizeof(HeaderSize));
        FillChar(bmiHeader, sizeof(bmiHeader), 0);
        with bmiHeader, OS2Header do
        begin
          biWidth := bcWidth;
          biHeight := bcHeight;
          biPlanes := bcPlanes;
          biBitCount := bcBitCount;
        end;
        Dec(ImageSize, sizeof(OS2Header));
      end
      else
      begin // support bitmap headers larger than TBitmapInfoHeader
        Stream.Read(Pointer(Longint(BitmapInfo) + sizeof(HeaderSize))^,
          HeaderSize - sizeof(HeaderSize));
        Dec(ImageSize, HeaderSize);

        if (bmiHeader.biCompression <> BI_BITFIELDS) and
          (bmiHeader.biCompression <> BI_RGB) then
        begin // Preserve funky non-DIB data (like RLE) until modified
          RLEStream := TMemoryStream.Create;
          // source stream could be unidirectional.  don't reverse seek
          RLEStream.Write(HeaderSize, sizeof(HeaderSize));
          RLEStream.Write(Pointer(Longint(BitmapInfo) + sizeof(HeaderSize))^,
            HeaderSize - sizeof(HeaderSize));
          RLEStream.CopyFrom(Stream, ImageSize);
          RLEStream.Seek(-ImageSize, soFromEnd);
          Stream := RLEStream;  // the rest of the proc reads from RLEStream
        end;
      end;

      with bmiHeader do
      begin
        biSize := HeaderSize;
        ColorTable := Pointer(Longint(BitmapInfo) + HeaderSize);

        { check number of planes. DIBs must be 1 color plane (packed pixels) }
        if biPlanes <> 1 then InvalidBitmap;

        // 3 DWORD color element bit masks (ie 888 or 565) can precede colors
        // TBitmapInfoHeader sucessors include these masks in the headersize
        if (HeaderSize = sizeof(TBitmapInfoHeader)) and
          ((biBitCount = 16) or (biBitCount = 32)) and
          (biCompression = BI_BITFIELDS) then
        begin
          Stream.ReadBuffer(ColorTable^, 3 * sizeof(DWORD));
          Inc(Longint(ColorTable), 3 * sizeof(DWORD));
          Dec(ImageSize, 3 * sizeof(DWORD));
        end;

        // Read the color palette
        if biClrUsed = 0 then
          biClrUsed := GetDInColors(biBitCount);
        Stream.ReadBuffer(ColorTable^, biClrUsed * DIBPalSizes[OS2Format]);
        Dec(ImageSize, biClrUsed * DIBPalSizes[OS2Format]);

        // biSizeImage can be zero. If zero, compute the size.
        if biSizeImage = 0 then            // top-down DIBs have negative height
          biSizeImage := BytesPerScanLine(biWidth, biBitCount, 32) * Abs(biHeight);

        if biSizeImage < ImageSize then ImageSize := biSizeImage;
      end;

      { convert OS2 color table to DIB color table }
      if OS2Format then RGBTripleToQuad(ColorTable^);

      DC := GDICheck(GetDC(0));
      try
        if ((bmiHeader.biCompression <> BI_RGB) and
          (bmiHeader.biCompression <> BI_BITFIELDS)) or DDBsOnly then
        begin
          MemDC := 0;
          GetMem(BitsMem, ImageSize);
          try
            Stream.ReadBuffer(BitsMem^, ImageSize);
            MemDC := GDICheck(CreateCompatibleDC(DC));
            OldBMP := SelectObject(MemDC, CreateCompatibleBitmap(DC, 1, 1));
            OldPal := 0;
            if bmiHeader.biClrUsed > 0 then
            begin
              Pal := PaletteFromDIBColorTable(0, ColorTable, bmiHeader.biClrUsed);
              OldPal := SelectPalette(MemDC, Pal, False);
              RealizePalette(MemDC);
            end;

            try
              BMHandle := CreateDIBitmap(MemDC, BitmapInfo^.bmiHeader, CBM_INIT, BitsMem,
                BitmapInfo^, DIB_RGB_COLORS);
              if (BMHandle = 0) then
                if GetLastError = 0 then InvalidBitmap else RaiseLastWin32Error;
            finally
              if OldPal <> 0 then
                SelectPalette(MemDC, OldPal, True);
              DeleteObject(SelectObject(MemDC, OldBMP));
            end;
          finally
            if MemDC <> 0 then DeleteDC(MemDC);
            FreeMem(BitsMem);
          end;
        end
        else
        begin
          BMHandle := CreateDIBSection(DC, BitmapInfo^, DIB_RGB_COLORS, BitsMem, 0, 0);
          if (BMHandle = 0) or (BitsMem = nil) then
            if GetLastError = 0 then InvalidBitmap else RaiseLastWin32Error;

          try
            Stream.ReadBuffer(BitsMem^, ImageSize);
          except
            DeleteObject(BMHandle);
            raise;
          end;
        end;
      finally
        ReleaseDC(0, DC);
      end;
      // Hi-color DIBs don't preserve color table, so create palette now
      if (bmiHeader.biBitCount > 8) and (bmiHeader.biClrUsed > 0) and (Pal = 0)then
        Pal := PaletteFromDIBColorTable(0, ColorTable, bmiHeader.biClrUsed);

      FillChar(DIB, sizeof(DIB), 0);
      GetObject(BMHandle, Sizeof(DIB), @DIB);
      // GetObject / CreateDIBSection don't preserve these info values
      DIB.dsBmih.biXPelsPerMeter := bmiHeader.biXPelsPerMeter;
      DIB.dsBmih.biYPelsPerMeter := bmiHeader.biYPelsPerMeter;
      DIB.dsBmih.biClrUsed := bmiHeader.biClrUsed;
      DIB.dsBmih.biClrImportant := bmiHeader.biClrImportant;
    except
      RLEStream.Free;
      raise;
    end;
  finally
    FreeMem(BitmapInfo);
  end;
  NewImage(BMHandle, Pal, DIB, OS2Format, RLEStream);
  PaletteModified := Palette <> 0;
  Changed(Self);
end;
*)


procedure TBitmap.ReadStream(Stream: TStream; Size: Longint);
{var
  Bmf: TBitmapFileHeader;
  DIB: TDIBSection;}
begin
  {FreeContext;
  if Size = 0 then
  begin
    FillChar(DIB, sizeof(DIB), 0);
    NewImage(0, 0, DIB, False);
  end
  else
  begin
    Stream.ReadBuffer(Bmf, sizeof(Bmf));
    if Bmf.bfType <> $4D42 then InvalidBitmap;
    ReadDIB(Stream, Size - sizeof(Bmf));
  end;}
end;

(*
procedure TBitmap.SetHandle(Value: HBITMAP);
var
  DIB: TDIBSection;
  APalette: HPALETTE;
begin
  with FImage do
    if FHandle <> Value then
    begin
      FreeContext;
      FillChar(DIB, sizeof(DIB), 0);
      if Value <> 0 then
        GetObject(Value, SizeOf(DIB), @DIB);
      if FRefCount = 1 then
      begin
        APalette := FPalette;
        FPalette := 0;
      end
      else
        if FPalette = SystemPalette16 then
          APalette := SystemPalette16
        else
          APalette := CopyPalette(FPalette);
      try
        NewImage(Value, APalette, DIB, False);
      except
        InternalDeletePalette(APalette);
        raise;
      end;
      Changed(Self);
    end;
end;
*)
{There are _ONLY_ DIBs under gd:}
procedure TBitmap.SetHandleType(Value: TBitmapHandleType);
begin
end;

procedure TBitmap.SetHeight(Value: Integer);
var
  NewCanvas: TCanvas;
begin
  if (FHeight <> Value) then
  begin
    NewCanvas := TCanvas.Create(Width, Value, FTransparentColor, (FPixelFormat = pf24bit) or (FPixelFormat = pf32bit));
    if (FCanvas <> nil) then
    begin
      gdImageCopy(NewCanvas.Handle, FCanvas.Handle,
        0, 0, 0, 0,
        Width, Min(Height, Value));
      FCanvas.Free;
    end;
    FCanvas := NewCanvas;
    Changed(Self);
  end;
end;

procedure TBitmap.SetMaskHandle(Value: HBITMAP);
begin
{  with FImage do
    if FMaskHandle <> Value then
    begin
      FMaskHandle := Value;
      FMaskValid := True;
      FMaskBitsValid := True;
    end;}
end;

(*
procedure TBitmap.SetMonochrome(Value: Boolean);
var
  DIB: TDIBSection;
begin
  with FImage, FDIB.dsbmih do
    if Value <> ((biPlanes = 1) and (biBitCount = 1)) then
    begin
      HandleNeeded;
      DIB := FDIB;
      with DIB.dsbmih, DIB.dsbm do
      begin
        biSize := 0;   // request DDB handle
        biPlanes := Byte(Value);  // 0 = request screen BMP format
        biBitCount := Byte(Value);
        bmPlanes := Byte(Value);
        bmBitsPixel := Byte(Value);
      end;
      CopyImage(FHandle, FPalette, DIB);
      Changed(Self);
    end;
end;
*)

(*
procedure TBitmap.SetPalette(Value: HPALETTE);
var
  AHandle: HBITMAP;
  DIB: TDIBSection;
begin
  if FImage.FPalette <> Value then
  begin
    with FImage do
      if (Value = 0) and (FRefCount = 1) then
      begin
        InternalDeletePalette(FPalette);
        FPalette := 0;
      end
      else
      begin
        FreeContext;
        HandleNeeded;
        DIB := FDIB;
        AHandle := CopyBitmap(FHandle, FPalette, Value, DIB, nil);
        try
          NewImage(AHandle, Value, DIB, FOS2Format);
        except
          DeleteObject(AHandle);
          raise;
        end;
      end;
    UpdateDIBColorTable(FImage.FDIBHandle, Value, FImage.FDIB);
    PaletteModified := True;
    Changed(Self);
  end;
end;
*)

procedure TBitmap.SetPixelFormat(Value: TPixelFormat);
begin
  case Value of
    pf8Bit: ;
    pf24Bit: ;
    pf32Bit: ;
  else
    EInvalidGraphicOperation.CreateFmt('%s is not a valid Pixel Format',
      [GetEnumName(TypeInfo(TPixelFormat), Ord(Value))]);
  end;
end;

procedure TBitmap.SetTransparentColor(Value: TColor);
begin
  if Value <> FTransparentColor then
  begin
    if Value = clDefault then
      FTransparentMode := tmAuto else
      FTransparentMode := tmFixed;
    FTransparentColor := Value;
    {if FImage.FRefCount > 1 then
    with FImage do
    begin
      HandleNeeded;
      CopyImage(FHandle, FPalette, FDIB);
    end;}
    Changed(Self);
  end;
end;

procedure TBitmap.SetTransparentMode(Value: TTransparentMode);
begin
  if Value <> FTransparentMode then
  begin
    if Value = tmAuto then
      SetTransparentColor(clDefault) else
      SetTransparentColor(GetTransparentColor);
  end;
end;

procedure TBitmap.SetWidth(Value: Integer);
var
  NewCanvas: TCanvas;
begin
  if (FHeight <> Value) then
  begin
    NewCanvas := TCanvas.Create(Value, Height, FTransparentColor, (FPixelFormat = pf24bit) or (FPixelFormat = pf32bit));
    if (FCanvas <> nil) then
    begin
      gdImageCopy(NewCanvas.Handle, FCanvas.Handle,
        0, 0, 0, 0,
        Min(Width, Value), Height);
      FCanvas.Free;
    end;
    FCanvas := NewCanvas;
    Changed(Self);
  end;
end;
(*
procedure TBitmap.WriteData(Stream: TStream);
begin
  WriteStream(Stream, True);
end;
*)
procedure TBitmap.WriteStream(Stream: TStream; WriteSize: Boolean);
{const
  PalSize: array [Boolean] of Byte = (sizeof(TRGBQuad), sizeof(TRGBTriple));
var
  Size, ColorCount: DWORD;
  HeaderSize: DWORD;
  BMF: TBitmapFileHeader;
  Save: THandle;
  BC: TBitmapCoreHeader;
  Colors: array [Byte] of TRGBQuad;}
begin
end;
(*  FillChar(BMF, sizeof(BMF), 0);
  BMF.bfType := $4D42;
  if FImage.FSaveStream <> nil then
  begin
    Size := FImage.FSaveStream.Size + sizeof(BMF);
    BMF.bfSize := Size;
    if WriteSize then
      Stream.WriteBuffer(Size, sizeof(Size));
    Stream.Write(BMF, sizeof(BMF));
    Stream.Write(FImage.FSaveStream.Memory^, FImage.FSaveStream.Size);
    Exit;
  end;
  DIBNeeded;
  with FImage do
  begin
    Size := 0;
    if FDIBHandle <> 0 then
    begin
      InternalGetDIBSizes(FDIBHandle, HeaderSize, Size, FDIB.dsbmih.biClrUsed);
      if FOS2Format then
      begin // OS2 format cannot have partial palette
        HeaderSize := sizeof(BC);
        if FDIB.dsbmih.biBitCount <= 8 then
          Inc(HeaderSize, sizeof(TRGBTriple) * (1 shl FDIB.dsbmih.biBitCount));
      end;
      Inc(Size, HeaderSize + sizeof(BMF));

      FillChar(BMF, sizeof(BMF), 0);
      BMF.bfType := $4D42;

      Canvas.RequiredState([csHandleValid]);
      Save := GDICheck(SelectObject(FCanvas.FHandle, FDIBHandle));
      ColorCount := GetDIBColorTable(FCanvas.FHandle, 0, 256, Colors);
      SelectObject(FCanvas.FHandle, Save);
      // GetDIBColorTable always reports the full palette; trim it back for partial palettes
      if (0 < FDIB.dsbmih.biClrUsed) and (FDIB.dsbmih.biClrUsed < ColorCount) then
        ColorCount := FDIB.dsbmih.biClrUsed;
      if (not FOS2Format) and (ColorCount = 0) and (FPalette <> 0) and not FHalftone then
      begin
        ColorCount := PaletteToDIBColorTable(FPalette, Colors);
        if FDIB.dsbmih.biBitCount > 8 then
        begin  // optional color palette for hicolor images (non OS2)
          Inc(Size, ColorCount * sizeof(TRGBQuad));
          Inc(HeaderSize, ColorCount * sizeof(TRGBQuad));
        end;
      end;

      BMF.bfSize := Size;
      BMF.bfOffBits := sizeof(BMF) + HeaderSize;
    end;

    if WriteSize then Stream.WriteBuffer(Size, SizeOf(Size));

    if Size <> 0 then
    begin
      FixupBitFields(FDIB);
      if (ColorCount <> 0) then
      begin
        if (FDIB.dsbmih.biClrUsed = 0) or (FDIB.dsbmih.biClrUsed <> ColorCount) then
          FDIB.dsbmih.biClrUsed := ColorCount;
        if FOS2Format then RGBQuadToTriple(Colors, Integer(ColorCount));
      end;
      if FOS2Format then
      begin
        with BC, FDIB.dsbmih do
        begin
          bcSize := sizeof(BC);
          bcWidth := biWidth;
          bcHeight := biHeight;
          bcPlanes := 1;
          bcBitCount := biBitCount;
        end;
        Stream.WriteBuffer(BMF, sizeof(BMF));
        Stream.WriteBuffer(BC, sizeof(BC));
      end
      else
      begin
        Stream.WriteBuffer(BMF, Sizeof(BMF));
        Stream.WriteBuffer(FDIB.dsbmih, Sizeof(FDIB.dsbmih));
        if (FDIB.dsbmih.biBitCount > 8) and
          ((FDIB.dsbmih.biCompression and BI_BITFIELDS) <> 0) then
          Stream.WriteBuffer(FDIB.dsBitfields, 12);
      end;
      Stream.WriteBuffer(Colors, ColorCount * PalSize[FOS2Format]);
      Stream.WriteBuffer(FDIB.dsbm.bmBits^, FDIB.dsbmih.biSizeImage);
    end;
  end;
end;
*)

(*
{ ReleaseHandle gives up ownership of the bitmap handle the TBitmap contains. }
function TBitmap.ReleaseHandle: HBITMAP;
begin
  HandleNeeded;
  Changing(Self);
  with FImage do
  begin
    Result := FHandle;
    if FHandle = FDIBHandle then
    begin
      FDIBHandle := 0;
      FDIB.dsbm.bmBits := nil;
    end;
    FHandle := 0;
  end;
end;
*)
function TBitmap.ReleaseMaskHandle: HBITMAP;
begin
  {Result := GetMaskHandle;
  FImage.FMaskHandle := 0;}
end;

(*
function TBitmap.ReleasePalette: HPALETTE;
begin
  HandleNeeded;
  Changing(Self);
  Result := FImage.FPalette;
  FImage.FPalette := 0;
end;
*)

procedure TBitmap.SaveToFile(const Filename: string);
var
  Ext: String;
  F: File;
begin
  Ext := Lowercase(ExtractFileExt(Filename));
  AssignFile(F, Filename);
  FileMode := 2;  {Set file access to read/write }
  Reset(F);
  if (Ext = '.png') then
    gdImagePng(FCanvas.Handle, @F)
  else if ((Ext = '.jpg') or (Ext = '.jpg')) then
    gdImageJpeg(FCanvas.Handle, @F, FCompressionQuality)
  else
  begin
    EComponentError.CreateFmt('"%s" is not a supported format !', [Ext]);
  end;
  CloseFile(F);
end;

procedure TBitmap.SaveToStream(Stream: TStream);
begin
  WriteStream(Stream, False);
end;

(*
procedure TBitmap.SaveToClipboardFormat(var Format: Word; var Data: THandle;
  var APalette: HPALETTE);
var
  DIB: TDIBSection;
begin
  Format := CF_BITMAP;
  HandleNeeded;
  with FImage do
  begin
    DIB := FDIB;
    DIB.dsbmih.biSize := 0;   // copy to device bitmap
    DIB.dsbm.bmBits := nil;
    Data := CopyBitmap(FHandle, FPalette, FPalette, DIB, FCanvas);
  end;
  try
    APalette := CopyPalette(FImage.FPalette);
  except
    DeleteObject(Data);
    raise;
  end;
end;
*)

function TBitmap.TransparentColorStored: Boolean;
begin
  Result := FTransparentMode = tmFixed;
end;



initialization
  RegisterIntegerConsts(TypeInfo(TColor), IdentToColor, ColorToIdent);
  RegisterIntegerConsts(TypeInfo(TFontCharset), IdentToCharset, CharsetToIdent);
finalization

end.
