unit Png;
{
  A More standard implemenation of TPng.
  Credits : Inpired by work done by Edmund Hand and Umberto Barbini
  
  I can be reached at:
  Dominique Louis (Dominique@SavageSoftware.com.au)
}

interface

uses Windows, SysUtils, Classes, Graphics, PngDef;

type TPngImage = class(TSharedImage)
  private
  protected
    procedure FreeHandle; override;
  public
    destructor Destroy; override;
  end;

  TPng = class(TGraphic)
  private
    FImage: TPngImage;
    FBitDepth:      integer;
    FBytesPerPixel: integer;
    FColorType:     integer;
    FHeight:        Cardinal; //ub used Cardinal instead of integer
    FWidth:         Cardinal; //ub used Cardinal instead of integer
    FInterlace:     integer;
    FCompression:   integer;
    FFilter:        integer;
    FBgColor :  TColor; // DL Background color Added 30/05/2000
    FTransparent : Boolean; // DL Is this Image Transparent   Added 30/05/2000
    FRowBytes : Cardinal;   //DL Added 30/05/2000
    FGamma :  double; //DL Added 07/06/2000
    FScreenGamma : double; //DL Added 07/06/2000

    FData:           PByte; // DL Changed for consistancy 30/05/2000
    FRowPtrs:        PByte; // DL Changed for consistancy 30/05/2000
    FDescription: string;
    FTitle: string;
    FAuthor: string;
    FTextChk: TStringList;
    FLastMod: TDateTime;
    FReadProgressCallback: Pointer;  //DL Added 08/06/2000
    FWriteProgressCallback: Pointer; //DL Added 08/06/2000
  protected
    procedure InitializeDemData;
    procedure SetAuthor(const Value: string);
    procedure SetDescription(const Value: string);
    procedure SetTitle(const Value: string);
    function GetTextChk: TStrings;
    //
    procedure Draw(ACanvas: TCanvas; const Rect: TRect); override;
    function  GetEmpty: Boolean; override;
    function  GetHeight: Integer; override;
    function  GetWidth: Integer; override;
    procedure SetHeight(Value: Integer); override;
    procedure SetWidth(Value: Integer); override;
  public
    constructor Create; override;
    destructor Destroy; override;
    procedure CopyToBmp( var aBmp: TBitmap );
    procedure CopyFromBmp( const aBmp: TBitmap );

    procedure Assign(Source: TPersistent); override;
    procedure LoadFromStream(Stream: TStream); override;
    procedure SaveToStream(Stream: TStream); override;
    procedure LoadFromClipboardFormat(AFormat: Word; AData: THandle;
              APalette: HPALETTE); override;
    procedure SaveToClipboardFormat(var AFormat: Word; var AData: THandle;
              var APalette: HPALETTE); override;
    procedure SetReadProgressCallback( FunctionName : Pointer ); //DL Added 08/06/2000
    procedure SetWriteProgressCallback( FunctionName : Pointer ); //DL Added 08/06/2000
  published
    property Title: string read FTitle write SetTitle;
    property Author: string read FAuthor write SetAuthor;
    property Description: string read FDescription write SetDescription;
    property BitDepth:      integer read FBitDepth;
    property BytesPerPixel: integer read FBytesPerPixel;
    property ColorType:     integer read FColorType;
    property Height:        Cardinal read FHeight;
    property Width:         Cardinal read FWidth;
    property Interlace:     integer read Finterlace;
    property Compression:   integer read FCompression;
    property Filter:        integer read FFilter;
    property TextChk: TStrings read GetTextChk;
    property LastModified: TdateTime read FLastMod;
    property Transparent : Boolean read FTransparent;
end;

implementation

{ TPngImage }
destructor TPngImage.Destroy;
begin
  // Code goes here
  FreeHandle;
  inherited;
end;

procedure TPngImage.FreeHandle;
begin
  // Code goes here
end;

procedure TPng.Draw(ACanvas: TCanvas; const Rect: TRect);
var
  bmp: TBitmap;
begin
  bmp := TBitmap.Create;
  try
    CopyToBmp( bmp );
    ACanvas.Draw( 0, 0, bmp );
  finally
    bmp.free;
  end;
end;

procedure TPng.CopyFromBmp( const aBmp: TBitmap);
var
  valuep:  PByte;
  x, y:    Integer;
  ndx:     Integer;
  sl:      PByteArray;  // Scanline of bitmap
  png:     png_structp; // PPng_Struct;
  pnginfo:  png_infop; //  PPng_Info;
  tmp:      array[0..32] of char;
begin
  aBmp.PixelFormat := pf24Bit;

  FWidth := aBmp.Width;
  FHeight := aBmp.Height;
  FBitDepth := 8; // Single channel
  FColorType := PNG_COLOR_TYPE_RGB;
  FBytesPerPixel := 3; // works with 24bits only

  InitializeDemData;
  tmp := PNG_LIBPNG_VER_STRING;
  png :=  png_create_write_struct(tmp, nil, nil, nil);
  pnginfo := png_create_info_struct( png );
  png_set_IHDR(png, pnginfo, FWidth, FHeight, FBitDepth, FColorType,
               PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT,
               PNG_FILTER_TYPE_DEFAULT);
          

  if ( FData <> nil ) and ( FRowPtrs <> nil ) then // Read the image
  begin
    valuep := FData;
    for y := 0 to FHeight - 1 do
    begin
      sl := aBmp.Scanline[ y  ];  // current scanline
      for x := 0 to FWidth - 1 do
      begin
        ndx := x * 3;    // index into current scanline

        // RGB - swap blue and red for windows format
        valuep^ := sl[ndx + 2];
        Inc(valuep);
        valuep^ := sl[ndx + 1];
        Inc(valuep);
        valuep^ := sl[ ndx ];
        Inc(valuep);
      end;
    end;
  end;

end;  // TPngImage.CopyToBmp

procedure TPng.InitializeDemData;
var
  cvaluep:  ^Cardinal; //ub
  y:        Cardinal;
begin
  // Initialize Data and RowPtrs
  if FData <> nil then
    FreeMem(FData);
  FData := nil;
  if FRowPtrs <> nil then
    FreeMem(FRowPtrs);
  FRowPtrs := nil;

  //GetMem(Data, FHeight * FWidth * Cardinal( FBytesPerPixel ) ); DL Changed 30/05/2000
  GetMem( FData, FHeight * FRowBytes ); // DL Added 30/5/2000
  GetMem( FRowPtrs, sizeof( Pointer ) * FHeight );

  if ( FData <> nil ) and ( FRowPtrs <> nil ) then
  begin
    cvaluep := Pointer( FRowPtrs );
    for y := 0 to FHeight - 1 do
    begin
      cvaluep^ := Cardinal( FData ) + ( y * FRowBytes ); //DL Added 08/07/2000
      //cvaluep^ := Cardinal(FData) + (FWidth * Cardinal( FBytesPerPixel ) * y); //DL Changed 08/07/2000
      Inc(cvaluep);
    end;
  end;  // if (Data <> nil) and (RowPtrs <> nil) then
end;  // TPngImage.InitializeDemData

var
  CurrStream : TStream;
  ioBuffer: array [ 0 .. 8192 ] of byte; //??

procedure ReadData(png_ptr: Pointer;var data: Pointer;length: png_size_t); stdcall;
begin // Callback to read from stream
  if length <= sizeof( ioBuffer ) then
    CurrStream.ReadBuffer( data, length )
  else
    raise Exception.Create( 'Buffer override: needed ' + inttostr( length ) + 'bytes for buffer !' );
end;

procedure WriteData(png_ptr: Pointer;var data: Pointer;length: png_size_t); stdcall;
begin // Callback to read from stream
  // Note that you can write also if data = nil (write 0)
  if length <= sizeof( ioBuffer ) then
    CurrStream.WriteBuffer( data, length )
  else
    raise Exception.Create( 'Buffer override: needed ' + inttostr( length ) );
end;

procedure FlushData(png_ptr: Pointer); stdcall;
begin // Callback to flush the stream
end;

procedure TPng.LoadFromStream( Stream: TStream );
var
  png:      png_structp;
  pnginfo:  png_infop;
  tmp:      array[0..31] of char;
  sig:      array[0..3] of byte;
  Txt : png_textp;
  i, nTxt: integer;
  s: string;
  Time: png_timep;
  pBackground : png_color_16p;
  RGBValue : Byte;
begin
  Stream.ReadBuffer( sig, sizeof( sig ) ); 
  CurrStream := Stream;
  if png_sig_cmp( @sig, 0, sizeof( sig )) <> 0 then
   raise Exception.Create( 'Is not a valid PNG !' );

  tmp := PNG_LIBPNG_VER_STRING;
  png := png_create_read_struct(tmp, nil, nil, nil);
  if assigned( png ) then
  begin
    pnginfo := png_create_info_struct( png );
    try
      if not assigned( pnginfo ) then
        raise Exception.Create( 'Failed to Create info struct' );

      png_set_sig_bytes(png, sizeof( sig ) );

      png_set_read_fn( png, @ioBuffer, ReadData );

      //Setup Read Callback function if one is assigned
      if assigned( FReadProgressCallBack ) then // DL Added 08/07/2000
        png_set_read_status_fn( png, FReadProgressCallBack );

      //Setup Write Callback function if one is assigned
      if assigned( FWriteProgressCallBack ) then // DL Added 08/07/2000
        png_set_write_status_fn( png, FWriteProgressCallBack );

      png_read_info( png, pnginfo );

      nTxt := 0;
      png_get_text( png, pnginfo, Txt, nTxt );

      FTextChk.Clear;
      for i := 0 to nTxt - 1 do
        begin // load all text in FTextChk
        s := txt^.key;
        s := s + '=' + Txt^.text; // better use no more than a pchar at time
        FTextChk.Add( s );
        if compareText( Txt^.key, 'Title' ) = 0 then
          FTitle := Txt^.text // load Title if present
        else if compareText( Txt^.key, 'Author' ) = 0 then
          FAuthor := Txt^.text// load Author if present
        else if compareText( Txt^.key, 'Description' ) = 0 then
          FDescription := Txt^.text; // load Description if present
        inc( Txt );
        end;

      png_get_IHDR(png, pnginfo, FWidth, FHeight,FBitDepth, FColorType, Finterlace, Fcompression, Ffilter );

      // it is not obvious from the libpng documentation, but this function
      // takes a pointer to a pointer, and it always returns valid red, green
      // and blue values, regardless of color_type: */
      png_get_bKGD(png, pnginfo, pBackground);
      // however, it always returns the raw bKGD data, regardless of any
      // bit-depth transformations, so check depth and adjust if necessary
      if (FBitDepth = 16) then
        FBgColor := RGB ( ( pBackground.red shr 8 ), ( pBackground.green shr 8 ), ( pBackground.blue shr 8 ) )
      else if (FColorType = PNG_COLOR_TYPE_GRAY) and ( FBitDepth < 8 ) then
      begin
          if ( FBitDepth = 1 ) then
          begin
            if pBackground.gray <> 0 then
              RGBValue := 255
            else
              RGBValue := 0;
            FBgColor := RGB ( RGBValue, RGBValue, RGBValue )
          end
          else if ( FBitDepth = 2 ) then
          begin
            RGBValue := ( 255 div 3 ) * pBackground.gray;
            FBgColor := RGB ( RGBValue, RGBValue, RGBValue )
          end
          else // FBitDepth = 4
          begin
            RGBValue := ( 255 div 15 ) * pBackground.gray;
            FBgColor := RGB ( RGBValue, RGBValue, RGBValue )
          end;
      end
      else
        FBgColor := RGB ( ( pBackground.red ), ( pBackground.green ), ( pBackground.blue ) );

      if ( png_get_valid( png, pnginfo, PNG_INFO_bKGD) = PNG_INFO_bKGD) then
        // Has No background color
        // Ideally this should be the color of the canvas
        FTransparent := true
      else
        FTransparent := false;
     
      // if bit depth is less than or equal 8 then expand...
      if ( FColorType = PNG_COLOR_TYPE_PALETTE ) and ( FBitDepth <= 8 ) then
        png_set_palette_to_rgb( png ); // DL Changed to be more readable

      if ( FColorType = PNG_COLOR_TYPE_GRAY ) and ( FBitDepth < 8 ) then
        png_set_gray_1_2_4_to_8( png );  // DL Changed to be more readable

      // Add alpha channel if pressent
      if png_get_valid( png, pnginfo, PNG_INFO_tRNS ) = PNG_INFO_tRNS then
        png_set_tRNS_to_alpha(png); // DL Changed to be more readable

      if ( FColorType = PNG_COLOR_TYPE_GRAY ) or ( FColorType = PNG_COLOR_TYPE_GRAY_ALPHA ) then
        png_set_gray_to_rgb( png );

      FGamma := 0;
      FScreenGamma := 2.2;
      if ( png_get_gAMA( png, pnginfo, FGamma ) <> 0 ) then
        png_set_gamma( png, FScreenGamma, FGamma )
      else
        png_set_gamma( png, FScreenGamma, 0.45455 );

      // Change to level of transparency
      png_set_invert_alpha( png ); // Moved 30/5/2000

      // expand images to 1 pixel per byte
      if FBitDepth < 8 then
        png_set_packing(png);

      // Swap 16 bit images to PC Format
      if FBitDepth = 16 then
      begin
        png_set_swap( png );
        //png_set_strip_16( png );
      end;

      // update the info structure
      png_read_update_info( png, pnginfo );
      //png_get_IHDR(png, pnginfo, FWidth, FHeight, FBitDepth, FColorType, Finterlace, Fcompression, Ffilter );

      FRowBytes := png_get_rowbytes( png, pnginfo );

      FBytesPerPixel := png_get_channels( png, pnginfo );  // DL Added 30/08/2000
      //FBytesPerPixel := rowbytes div FWidth; DL Changed 30/08/2000 in favor of above

      InitializeDemData;
      if (FData <> nil) and (FRowPtrs <> nil) then
        // Read the image
        png_read_image(png, png_bytepp(FRowPtrs));

      png_read_end(png, pnginfo); // read last information chunks

      if png_get_time( png, pnginfo, Time ) > 0 then
        begin // get time if possible
        FLastMod := EncodeDate( time.year, time.month, time.Day );
        FLastMod := FLastMod + EncodeTime( time.hour, time.minute, time.second, 0 );
        end;
        
    finally
      png_destroy_read_struct(@png, @pnginfo, nil);
    end;  // try pnginfo create
  end;
end;

procedure TPng.SaveToStream( Stream: TStream );
var
  png:      png_structp; // PPng_Struct;
  pnginfo:  png_infop; // PPng_Info;
  tmp:      array[0..32] of char;
//  costs, weights: array[ 0..4] of double;
  Txt : array[ 0 .. 2 ] of png_text;
  Time: png_time;
  yy, mm, dd, hh, mi, ss, ms: word;
begin
  CurrStream := Stream;
  tmp := PNG_LIBPNG_VER_STRING;
  png := png_create_write_struct(tmp, nil, nil, nil);
  if assigned( png ) then
  begin
    // create info struct and init io functions
    pnginfo := png_create_info_struct(png);
    try
      // set image attributes, compression, etc...
      png_set_write_fn(png, @ioBuffer, writedata, flushdata );

      png_set_IHDR(png, pnginfo, FWidth, FHeight, FBitDepth, FColorType, Finterlace, Fcompression, Ffilter );

      if ( FAuthor <> '' ) or ( FTitle <> '' ) or ( FDescription <> '' ) then
        begin // save text information only when needed
        Txt[ 0 ].key := 'Author';
        Txt[ 0 ].text := pchar( FAuthor );
        Txt[ 0 ].text_length := length( FAuthor );
        Txt[ 0 ].compression := PNG_TEXT_COMPRESSION_NONE;
        Txt[ 1 ].key := 'Title';
        Txt[ 1 ].text := pchar( FTitle );
        Txt[ 1 ].text_length := length( FTitle );
        Txt[ 1 ].compression := PNG_TEXT_COMPRESSION_NONE;
        Txt[ 2 ].key := 'Description';
        Txt[ 2 ].text := pchar( FDescription );
        Txt[ 2 ].text_length := length( FDescription );
        Txt[ 2 ].compression := PNG_TEXT_COMPRESSION_zTXt;
        png_set_text(png, pnginfo, @Txt, 3);
        end;

      png_write_info(png, pnginfo); 

      png_set_compression_level( png, 9 ); //best compression
      
{      // this'd force the DLL to calculate best filter for each row
       // but it doen't worth of. I'm not sure why.
      weights[ 0 ] := 1.0;
      weights[ 1 ] := 1.0;
      weights[ 2 ] := 1.0;
      weights[ 3 ] := 1.0;
      weights[ 4 ] := 1.0;
      costs[ 0 ] := 1.0;
      costs[ 1 ] := 1.0;
      costs[ 2 ] := 1.0;
      costs[ 3 ] := 1.0;
      costs[ 4 ] := 1.0;
      png_set_filter_heuristics( png, PNG_FILTER_HEURISTIC_WEIGHTED, 5, @weights, @costs );
}
      png_set_filter( png, PNG_FILTER_TYPE_BASE, PNG_FILTER_NONE or PNG_FILTER_SUB or PNG_FILTER_UP); // fast and good filtering
      
      if (FData <> nil) and (FRowPtrs <> nil) then
        begin
          // Swap 16 bit images from PC Format
          if FBitDepth = 16 then
            png_set_swap(png);
          // Write the image
          png_write_image(png, png_bytepp(FRowPtrs));

          // Now you can add text or time thunks to pnginfo if you want them save after image
          // I added time chunk for example but you'd use it only if you have changed the image.
          DecodeDate( Now, yy, mm, dd );
          DecodeTime( Now, hh, mi, ss, ms );
          time.year := yy;
          Time.month := mm;
          Time.day := dd;
          Time.hour := hh;
          Time.minute := mi;
          Time.second := ss;
          png_set_tIME( png, pnginfo, @time );
          
          png_write_end(png, pnginfo );             
        end;  // if buf <> nil
      finally
        png_destroy_write_struct(@png, @pnginfo);
      end;  // try pnginfo create
    end;
  end;

procedure TPng.Assign(Source: TPersistent);
begin
  if (Source = nil) or ( Source is TPng ) then
  begin
    if Assigned(FImage) then
      FImage.Release;

    if Assigned(Source) then
    begin
      FImage := TPng(Source).FImage;
    end
    else
    begin
      FImage := TPngImage.Create;
    end;
    FImage.Reference;
    Changed(Self);
  end
  else
    inherited Assign(Source);
end;

function  TPng.GetEmpty: Boolean;
begin
  if Assigned(FImage) then
    Result := False
  else
    Result := True;
end;

function TPng.GetHeight: Integer;
begin
  Result := FHeight;
end;

function TPng.GetWidth: Integer;
begin
  Result := FWidth;
end; 

procedure TPng.LoadFromClipboardFormat(AFormat: Word; AData: THandle;
          APalette: HPALETTE);
begin
  raise Exception.Create('Cannot load a TPng from the Clipboard');
end;  // TPng.LoadFromClipboardFormat

procedure TPng.SaveToClipboardFormat(var AFormat: Word;
          var AData: THandle; var APalette: HPALETTE);
begin
  raise Exception.Create('Cannot save a TPng to the Clipboard');
end;  // TPng.SaveToClipboardFormat

procedure TPng.SetHeight(Value: Integer);
begin
  raise Exception.Create('Cannot set height on a TPng');
end;  // TPng.SetHeight

procedure TPng.SetWidth(Value: Integer);
begin
  raise Exception.Create('Cannot set width on a TPng');
end;  // TPng.SetWidth

procedure TPng.SetAuthor(const Value: string);
begin
  FAuthor := Value;
end;

procedure TPng.SetDescription(const Value: string);
begin
  FDescription := Value;
end;

procedure TPng.SetTitle(const Value: string);
begin
  FTitle := Value;
end;

function TPng.GetTextChk: TStrings;
begin
  result := FTextChk;
end;

procedure TPng.SetReadProgressCallback(FunctionName: Pointer);
begin
  FReadProgressCallback := FunctionName;
end;

procedure TPng.SetWriteProgressCallback(FunctionName: Pointer);
begin
  FWriteProgressCallback := FunctionName;
end;

constructor TPng.Create;
begin
  inherited;
  FImage := TBitmapImage.Create;
  FImage.Reference;
  FData     := nil;
  FRowPtrs  := nil;
  FHeight  := 0;
  FWidth   := 0;
  // ub default values
  FColorType := PNG_COLOR_TYPE_RGB;
  Finterlace := PNG_INTERLACE_NONE;
  Fcompression := PNG_COMPRESSION_TYPE_DEFAULT;
  Ffilter := PNG_FILTER_TYPE_DEFAULT;
  FTextChk := TStringList.Create;
end;

destructor TPng.Destroy;
begin
  FImage.Release;
  FTextChk.Free;
  if FData <> nil then
    FreeMem(FData);
  if FRowPtrs <> nil then
    FreeMem(FRowPtrs);
  inherited;
end;

procedure TPng.CopyToBmp(var aBmp: TBitmap);
var
  valuep:  PByte;
  h, w, x, y:    Integer;
  ndx:     Integer;
  sl:      PByteArray;  // Scanline of bitmap
  slbpp:   Integer;     // Scanline bytes per pixel
  a, r, g, b: Byte;
begin
  if Height > Cardinal( MaxInt ) then
    raise Exception.Create( 'Image too high' );
  if Width > Cardinal( MaxInt ) then
    raise Exception.Create( 'Image too wide' );
  h := FHeight;
  w := FWidth;
  if aBmp.Height < h then
    aBmp.Height := h;
  if aBmp.Width < w then
    aBmp.Width  := w;
  case FBytesPerPixel of
    2: begin
      aBmp.PixelFormat := pf16Bit;
      slbpp := 2;
    end;
    else begin
      aBmp.PixelFormat := pf24Bit;
      slbpp := 3;
    end;
  end;

  aBmp.Transparent := Transparent;

  // point to data
  valuep := FData;
  for y := 0 to FHeight - 1 do
  begin
    sl := aBmp.Scanline[ y ];  // current scanline
    for x := 0 to FWidth - 1 do
    begin
      ndx := x * slbpp;    // index into current scanline
      if FBytesPerPixel = 2 then
      begin
        // handle 16bit grayscale images, this will display them
        // as a 16bit color image, kinda hokie but fits my needs
        // without altering the data.
        sl[ndx]     := valuep^;
        Inc(valuep);
        sl[ndx + 1] := valuep^;
        Inc(valuep);
      end
      else if FBytesPerPixel = 3 then
      begin
        // RGB - swap blue and red for windows format
        sl[ndx + 2] := valuep^;
        Inc(valuep);
        sl[ndx + 1] := valuep^;
        Inc(valuep);
        sl[ndx]     := valuep^;
        Inc(valuep);
      end
      else  // 4 bytes per pixel of image data
      begin
        // Alpha chanel present and RGB
        // this is what PNG is all about
        r := valuep^;
        Inc(valuep);
        g := valuep^;
        Inc(valuep);
        b := valuep^;
        Inc(valuep);
        a := valuep^;
        Inc(valuep);
        if a = 0 then
        begin
          // alpha is zero so no blending, just image data
          sl[ndx]     := b;
          sl[ndx + 1] := g;
          sl[ndx + 2] := r;
        end
        else if a < 255 then
        begin
          // blend with data from ACanvas as background
          {sl[ndx]     := png_composite_integer( sl[ndx], a, GetBValue( FBgColor ) ); // DL Added to centralise AlphaCompositiing
          sl[ndx + 1] := png_composite_integer( sl[ndx + 1], a, GetGValue( FBgColor ) ); // DL Added to centralise AlphaCompositiing
          sl[ndx + 2] := png_composite_integer( sl[ndx + 2], a, GetRValue( FBgColor ) ); // DL Added to centralise AlphaCompositiing}
          sl[ndx]     := ((sl[ndx] * a) + ((255 - a) * b)) div 255;
          sl[ndx + 1] := ((sl[ndx + 1] * a) + ((255 - a) * g)) div 255;
          sl[ndx + 2] := ((sl[ndx + 2] * a) + ((255 - a) * r)) div 255;
        end
        else
        begin
          // if a = 0 then do not place any color from the image at this
          // pixel, but let the background color show through instead.
          sl[ndx] := GetBValue( FBgColor );  // DL Added 08/06/2000
          sl[ndx + 1] := GetGValue( FBgColor ); // DL Added 08/06/2000
          sl[ndx + 2] := GetRValue( FBgColor ); // DL Added 08/06/2000
        end;
      end;
    end;
  end;
end;

initialization
  TPicture.RegisterFileFormat('PNG', 'Portable Network Graphics', TPng);
finalization
  TPicture.UnRegisterGraphicClass(TPng);
end.
